# -*- coding: utf-8 -*- #
# Copyright 2015 Google LLC. All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""Base template using which the apis_map.py is generated."""

from __future__ import absolute_import
from __future__ import division
from __future__ import unicode_literals


class APIDef(object):
  """Struct for info required to instantiate clients/messages for API versions.

  Attributes:
    apitools: ApitoolsClientDef for this API version.
    gapic: GapicClientDef for this API version.
    default_version: bool, Whether this API version is the default version for
      the API.
    enable_mtls: bool, Whether this API version supports mTLS.
    mtls_endpoint_override: str, The mTLS endpoint for this API version. If
      empty, the MTLS_BASE_URL in the API client will be used.
    regional_endpoints: dict[str, str], The regional endpoints for this API
      version. Dictionary maps location to endpoint URL.
  """

  def __init__(self,
               apitools=None,
               gapic=None,
               default_version=False,
               enable_mtls=True,
               mtls_endpoint_override='',
               regional_endpoints=None):
    self.apitools = apitools
    self.gapic = gapic
    self.default_version = default_version
    self.enable_mtls = enable_mtls
    self.mtls_endpoint_override = mtls_endpoint_override
    self.regional_endpoints = regional_endpoints or {}

  def __eq__(self, other):
    return (isinstance(other, self.__class__) and
            self.__dict__ == other.__dict__)

  def __ne__(self, other):
    return not self.__eq__(other)

  def get_init_source(self):
    src_fmt = 'APIDef({0}, {1}, {2}, {3}, "{4}")'
    return src_fmt.format(self.apitools, self.gapic,
                          self.default_version,
                          self.enable_mtls, self.mtls_endpoint_override)

  def __repr__(self):
    return self.get_init_source()


class ApitoolsClientDef(object):
  """Struct for info required to instantiate clients/messages for API versions.

  Attributes:
    class_path: str, Path to the package containing api related modules.
    client_classpath: str, Relative path to the client class for an API version.
    client_full_classpath: str, Full path to the client class for an API
      version.
    messages_modulepath: str, Relative path to the messages module for an API
      version.
    messages_full_modulepath: str, Full path to the messages module for an API
      version.
    base_url: str, The base_url used for the default version of the API.
  """

  def __init__(self,
               class_path,
               client_classpath,
               messages_modulepath,
               base_url):
    self.class_path = class_path
    self.client_classpath = client_classpath
    self.messages_modulepath = messages_modulepath
    self.base_url = base_url

  @property
  def client_full_classpath(self):
    return self.class_path + '.' + self.client_classpath

  @property
  def messages_full_modulepath(self):
    return self.class_path + '.' + self.messages_modulepath

  def __eq__(self, other):
    return (isinstance(other, self.__class__) and
            self.__dict__ == other.__dict__)

  def __ne__(self, other):
    return not self.__eq__(other)

  def get_init_source(self):
    src_fmt = 'ApitoolsClientDef("{0}", "{1}", "{2}", "{3}")'
    return src_fmt.format(self.class_path, self.client_classpath,
                          self.messages_modulepath, self.base_url)

  def __repr__(self):
    return self.get_init_source()


class GapicClientDef(object):
  """Struct for info required to instantiate clients/messages for API versions.

  Attributes:
    class_path: str, Path to the package containing api related modules.
    client_full_classpath: str, Full path to the client class for an API
      version.
    async_client_full_classpath: str, Full path to the async client class for an
      API version.
    rest_client_full_classpath: str, Full path to the rest client class for an
      API version.
  """

  def __init__(self,
               class_path):
    self.class_path = class_path

  @property
  def client_full_classpath(self):
    return self.class_path + '.client.GapicWrapperClient'

  @property
  def async_client_full_classpath(self):
    return self.class_path + '.async_client.GapicWrapperClient'

  @property
  def rest_client_full_classpath(self):
    return self.class_path + '.rest_client.GapicWrapperClient'

  def __eq__(self, other):
    return (isinstance(other, self.__class__) and
            self.__dict__ == other.__dict__)

  def __ne__(self, other):
    return not self.__eq__(other)

  def get_init_source(self):
    src_fmt = 'GapicClientDef("{0}")'
    return src_fmt.format(self.class_path)

  def __repr__(self):
    return self.get_init_source()


MAP = {
    'accessapproval': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.accessapproval.v1',
                    client_classpath='accessapproval_v1_client.AccessapprovalV1',
                    base_url='https://accessapproval.googleapis.com/',
                    messages_modulepath='accessapproval_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'accesscontextmanager': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.accesscontextmanager.v1',
                    client_classpath='accesscontextmanager_v1_client.AccesscontextmanagerV1',
                    base_url='https://accesscontextmanager.googleapis.com/',
                    messages_modulepath='accesscontextmanager_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.accesscontextmanager.v1alpha',
                    client_classpath='accesscontextmanager_v1alpha_client.AccesscontextmanagerV1alpha',
                    base_url='https://accesscontextmanager.googleapis.com/',
                    messages_modulepath='accesscontextmanager_v1alpha_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'admin': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.admin.v1',
                    client_classpath='admin_v1_client.AdminDirectoryV1',
                    base_url='https://www.googleapis.com/admin/directory/v1/',
                    messages_modulepath='admin_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='https://www.mtls.googleapis.com/admin/directory/v1/',
                regional_endpoints={}),
    },
    'aiplatform': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.aiplatform.v1',
                    client_classpath='aiplatform_v1_client.AiplatformV1',
                    base_url='https://aiplatform.googleapis.com/',
                    messages_modulepath='aiplatform_v1_messages'),
                gapic=GapicClientDef(
                    class_path='googlecloudsdk.generated_clients.gapic_wrappers.aiplatform.v1'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'africa-south1': 'https://africa-south1-aiplatform.googleapis.com/', 'asia-east1': 'https://asia-east1-aiplatform.googleapis.com/', 'asia-east2': 'https://asia-east2-aiplatform.googleapis.com/', 'asia-northeast1': 'https://asia-northeast1-aiplatform.googleapis.com/', 'asia-northeast2': 'https://asia-northeast2-aiplatform.googleapis.com/', 'asia-northeast3': 'https://asia-northeast3-aiplatform.googleapis.com/', 'asia-south1': 'https://asia-south1-aiplatform.googleapis.com/', 'asia-south2': 'https://asia-south2-aiplatform.googleapis.com/', 'asia-southeast1': 'https://asia-southeast1-aiplatform.googleapis.com/', 'asia-southeast2': 'https://asia-southeast2-aiplatform.googleapis.com/', 'australia-southeast1': 'https://australia-southeast1-aiplatform.googleapis.com/', 'australia-southeast2': 'https://australia-southeast2-aiplatform.googleapis.com/', 'europe-central2': 'https://europe-central2-aiplatform.googleapis.com/', 'europe-north1': 'https://europe-north1-aiplatform.googleapis.com/', 'europe-north2': 'https://europe-north2-aiplatform.googleapis.com/', 'europe-southwest1': 'https://europe-southwest1-aiplatform.googleapis.com/', 'europe-west1': 'https://europe-west1-aiplatform.googleapis.com/', 'europe-west10': 'https://europe-west10-aiplatform.googleapis.com/', 'europe-west15': 'https://europe-west15-aiplatform.googleapis.com/', 'europe-west2': 'https://europe-west2-aiplatform.googleapis.com/', 'europe-west3': 'https://europe-west3-aiplatform.googleapis.com/', 'europe-west4': 'https://europe-west4-aiplatform.googleapis.com/', 'europe-west6': 'https://europe-west6-aiplatform.googleapis.com/', 'europe-west8': 'https://europe-west8-aiplatform.googleapis.com/', 'europe-west9': 'https://europe-west9-aiplatform.googleapis.com/', 'europe-west12': 'https://europe-west12-aiplatform.googleapis.com/', 'me-central1': 'https://me-central1-aiplatform.googleapis.com/', 'me-central2': 'https://me-central2-aiplatform.googleapis.com/', 'me-west1': 'https://me-west1-aiplatform.googleapis.com/', 'northamerica-northeast1': 'https://northamerica-northeast1-aiplatform.googleapis.com/', 'northamerica-northeast2': 'https://northamerica-northeast2-aiplatform.googleapis.com/', 'southamerica-east1': 'https://southamerica-east1-aiplatform.googleapis.com/', 'southamerica-west1': 'https://southamerica-west1-aiplatform.googleapis.com/', 'us-central1': 'https://us-central1-aiplatform.googleapis.com/', 'us-central2': 'https://us-central2-aiplatform.googleapis.com/', 'us-east1': 'https://us-east1-aiplatform.googleapis.com/', 'us-east4': 'https://us-east4-aiplatform.googleapis.com/', 'us-east7': 'https://us-east7-aiplatform.googleapis.com/', 'us-south1': 'https://us-south1-aiplatform.googleapis.com/', 'us-west1': 'https://us-west1-aiplatform.googleapis.com/', 'us-west2': 'https://us-west2-aiplatform.googleapis.com/', 'us-west3': 'https://us-west3-aiplatform.googleapis.com/', 'us-west4': 'https://us-west4-aiplatform.googleapis.com/', 'us-east5': 'https://us-east5-aiplatform.googleapis.com/', 'us': 'https://aiplatform.us.rep.googleapis.com/', 'eu': 'https://aiplatform.eu.rep.googleapis.com/'}),
        'v1beta1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.aiplatform.v1beta1',
                    client_classpath='aiplatform_v1beta1_client.AiplatformV1beta1',
                    base_url='https://aiplatform.googleapis.com/',
                    messages_modulepath='aiplatform_v1beta1_messages'),
                gapic=GapicClientDef(
                    class_path='googlecloudsdk.generated_clients.gapic_wrappers.aiplatform.v1beta1'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'africa-south1': 'https://africa-south1-aiplatform.googleapis.com/', 'asia-east1': 'https://asia-east1-aiplatform.googleapis.com/', 'asia-east2': 'https://asia-east2-aiplatform.googleapis.com/', 'asia-northeast1': 'https://asia-northeast1-aiplatform.googleapis.com/', 'asia-northeast2': 'https://asia-northeast2-aiplatform.googleapis.com/', 'asia-northeast3': 'https://asia-northeast3-aiplatform.googleapis.com/', 'asia-south1': 'https://asia-south1-aiplatform.googleapis.com/', 'asia-south2': 'https://asia-south2-aiplatform.googleapis.com/', 'asia-southeast1': 'https://asia-southeast1-aiplatform.googleapis.com/', 'asia-southeast2': 'https://asia-southeast2-aiplatform.googleapis.com/', 'australia-southeast1': 'https://australia-southeast1-aiplatform.googleapis.com/', 'australia-southeast2': 'https://australia-southeast2-aiplatform.googleapis.com/', 'europe-central2': 'https://europe-central2-aiplatform.googleapis.com/', 'europe-north1': 'https://europe-north1-aiplatform.googleapis.com/', 'europe-north2': 'https://europe-north2-aiplatform.googleapis.com/', 'europe-southwest1': 'https://europe-southwest1-aiplatform.googleapis.com/', 'europe-west1': 'https://europe-west1-aiplatform.googleapis.com/', 'europe-west10': 'https://europe-west10-aiplatform.googleapis.com/', 'europe-west15': 'https://europe-west15-aiplatform.googleapis.com/', 'europe-west2': 'https://europe-west2-aiplatform.googleapis.com/', 'europe-west3': 'https://europe-west3-aiplatform.googleapis.com/', 'europe-west4': 'https://europe-west4-aiplatform.googleapis.com/', 'europe-west6': 'https://europe-west6-aiplatform.googleapis.com/', 'europe-west8': 'https://europe-west8-aiplatform.googleapis.com/', 'europe-west9': 'https://europe-west9-aiplatform.googleapis.com/', 'europe-west12': 'https://europe-west12-aiplatform.googleapis.com/', 'me-central1': 'https://me-central1-aiplatform.googleapis.com/', 'me-central2': 'https://me-central2-aiplatform.googleapis.com/', 'me-west1': 'https://me-west1-aiplatform.googleapis.com/', 'northamerica-northeast1': 'https://northamerica-northeast1-aiplatform.googleapis.com/', 'northamerica-northeast2': 'https://northamerica-northeast2-aiplatform.googleapis.com/', 'southamerica-east1': 'https://southamerica-east1-aiplatform.googleapis.com/', 'southamerica-west1': 'https://southamerica-west1-aiplatform.googleapis.com/', 'us-central1': 'https://us-central1-aiplatform.googleapis.com/', 'us-central2': 'https://us-central2-aiplatform.googleapis.com/', 'us-east1': 'https://us-east1-aiplatform.googleapis.com/', 'us-east4': 'https://us-east4-aiplatform.googleapis.com/', 'us-east7': 'https://us-east7-aiplatform.googleapis.com/', 'us-south1': 'https://us-south1-aiplatform.googleapis.com/', 'us-west1': 'https://us-west1-aiplatform.googleapis.com/', 'us-west2': 'https://us-west2-aiplatform.googleapis.com/', 'us-west3': 'https://us-west3-aiplatform.googleapis.com/', 'us-west4': 'https://us-west4-aiplatform.googleapis.com/', 'us-east5': 'https://us-east5-aiplatform.googleapis.com/', 'us-west8': 'https://us-west8-aiplatform.googleapis.com/', 'us': 'https://aiplatform.us.rep.googleapis.com/', 'eu': 'https://aiplatform.eu.rep.googleapis.com/'}),
    },
    'alloydb': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.alloydb.v1',
                    client_classpath='alloydb_v1_client.AlloydbV1',
                    base_url='https://alloydb.googleapis.com/',
                    messages_modulepath='alloydb_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'us-central2': 'https://alloydb.us-central2.rep.googleapis.com/', 'us-east7': 'https://alloydb.us-east7.rep.googleapis.com/', 'northamerica-northeast1': 'https://alloydb.northamerica-northeast1.rep.googleapis.com/', 'northamerica-northeast2': 'https://alloydb.northamerica-northeast2.rep.googleapis.com/', 'northamerica-south1': 'https://alloydb.northamerica-south1.rep.googleapis.com/', 'southamerica-east1': 'https://alloydb.southamerica-east1.rep.googleapis.com/', 'southamerica-west1': 'https://alloydb.southamerica-west1.rep.googleapis.com/', 'us-central1': 'https://alloydb.us-central1.rep.googleapis.com/', 'us-east1': 'https://alloydb.us-east1.rep.googleapis.com/', 'us-east4': 'https://alloydb.us-east4.rep.googleapis.com/', 'us-east5': 'https://alloydb.us-east5.rep.googleapis.com/', 'us-south1': 'https://alloydb.us-south1.rep.googleapis.com/', 'us-west1': 'https://alloydb.us-west1.rep.googleapis.com/', 'us-west2': 'https://alloydb.us-west2.rep.googleapis.com/', 'us-west3': 'https://alloydb.us-west3.rep.googleapis.com/', 'us-west4': 'https://alloydb.us-west4.rep.googleapis.com/', 'europe-central2': 'https://alloydb.europe-central2.rep.googleapis.com/', 'europe-southwest1': 'https://alloydb.europe-southwest1.rep.googleapis.com/', 'europe-north1': 'https://alloydb.europe-north1.rep.googleapis.com/', 'europe-north2': 'https://alloydb.europe-north2.rep.googleapis.com/', 'europe-west1': 'https://alloydb.europe-west1.rep.googleapis.com/', 'europe-west2': 'https://alloydb.europe-west2.rep.googleapis.com/', 'europe-west3': 'https://alloydb.europe-west3.rep.googleapis.com/', 'europe-west4': 'https://alloydb.europe-west4.rep.googleapis.com/', 'europe-west6': 'https://alloydb.europe-west6.rep.googleapis.com/', 'europe-west8': 'https://alloydb.europe-west8.rep.googleapis.com/', 'europe-west9': 'https://alloydb.europe-west9.rep.googleapis.com/', 'europe-west10': 'https://alloydb.europe-west10.rep.googleapis.com/', 'europe-west12': 'https://alloydb.europe-west12.rep.googleapis.com/', 'asia-east1': 'https://alloydb.asia-east1.rep.googleapis.com/', 'asia-east2': 'https://alloydb.asia-east2.rep.googleapis.com/', 'asia-northeast1': 'https://alloydb.asia-northeast1.rep.googleapis.com/', 'asia-northeast2': 'https://alloydb.asia-northeast2.rep.googleapis.com/', 'asia-northeast3': 'https://alloydb.asia-northeast3.rep.googleapis.com/', 'asia-south1': 'https://alloydb.asia-south1.rep.googleapis.com/', 'asia-south2': 'https://alloydb.asia-south2.rep.googleapis.com/', 'asia-southeast1': 'https://alloydb.asia-southeast1.rep.googleapis.com/', 'asia-southeast2': 'https://alloydb.asia-southeast2.rep.googleapis.com/', 'australia-southeast1': 'https://alloydb.australia-southeast1.rep.googleapis.com/', 'australia-southeast2': 'https://alloydb.australia-southeast2.rep.googleapis.com/', 'me-central1': 'https://alloydb.me-central1.rep.googleapis.com/', 'me-central2': 'https://alloydb.me-central2.rep.googleapis.com/', 'me-west1': 'https://alloydb.me-west1.rep.googleapis.com/', 'africa-south1': 'https://alloydb.africa-south1.rep.googleapis.com/'}),
        'v1alpha':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.alloydb.v1alpha',
                    client_classpath='alloydb_v1alpha_client.AlloydbV1alpha',
                    base_url='https://alloydb.googleapis.com/',
                    messages_modulepath='alloydb_v1alpha_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'us-central2': 'https://alloydb.us-central2.rep.googleapis.com/', 'us-east7': 'https://alloydb.us-east7.rep.googleapis.com/', 'northamerica-northeast1': 'https://alloydb.northamerica-northeast1.rep.googleapis.com/', 'northamerica-northeast2': 'https://alloydb.northamerica-northeast2.rep.googleapis.com/', 'northamerica-south1': 'https://alloydb.northamerica-south1.rep.googleapis.com/', 'southamerica-east1': 'https://alloydb.southamerica-east1.rep.googleapis.com/', 'southamerica-west1': 'https://alloydb.southamerica-west1.rep.googleapis.com/', 'us-central1': 'https://alloydb.us-central1.rep.googleapis.com/', 'us-east1': 'https://alloydb.us-east1.rep.googleapis.com/', 'us-east4': 'https://alloydb.us-east4.rep.googleapis.com/', 'us-east5': 'https://alloydb.us-east5.rep.googleapis.com/', 'us-south1': 'https://alloydb.us-south1.rep.googleapis.com/', 'us-west1': 'https://alloydb.us-west1.rep.googleapis.com/', 'us-west2': 'https://alloydb.us-west2.rep.googleapis.com/', 'us-west3': 'https://alloydb.us-west3.rep.googleapis.com/', 'us-west4': 'https://alloydb.us-west4.rep.googleapis.com/', 'europe-central2': 'https://alloydb.europe-central2.rep.googleapis.com/', 'europe-southwest1': 'https://alloydb.europe-southwest1.rep.googleapis.com/', 'europe-north1': 'https://alloydb.europe-north1.rep.googleapis.com/', 'europe-north2': 'https://alloydb.europe-north2.rep.googleapis.com/', 'europe-west1': 'https://alloydb.europe-west1.rep.googleapis.com/', 'europe-west2': 'https://alloydb.europe-west2.rep.googleapis.com/', 'europe-west3': 'https://alloydb.europe-west3.rep.googleapis.com/', 'europe-west4': 'https://alloydb.europe-west4.rep.googleapis.com/', 'europe-west6': 'https://alloydb.europe-west6.rep.googleapis.com/', 'europe-west8': 'https://alloydb.europe-west8.rep.googleapis.com/', 'europe-west9': 'https://alloydb.europe-west9.rep.googleapis.com/', 'europe-west10': 'https://alloydb.europe-west10.rep.googleapis.com/', 'europe-west12': 'https://alloydb.europe-west12.rep.googleapis.com/', 'asia-east1': 'https://alloydb.asia-east1.rep.googleapis.com/', 'asia-east2': 'https://alloydb.asia-east2.rep.googleapis.com/', 'asia-northeast1': 'https://alloydb.asia-northeast1.rep.googleapis.com/', 'asia-northeast2': 'https://alloydb.asia-northeast2.rep.googleapis.com/', 'asia-northeast3': 'https://alloydb.asia-northeast3.rep.googleapis.com/', 'asia-south1': 'https://alloydb.asia-south1.rep.googleapis.com/', 'asia-south2': 'https://alloydb.asia-south2.rep.googleapis.com/', 'asia-southeast1': 'https://alloydb.asia-southeast1.rep.googleapis.com/', 'asia-southeast2': 'https://alloydb.asia-southeast2.rep.googleapis.com/', 'australia-southeast1': 'https://alloydb.australia-southeast1.rep.googleapis.com/', 'australia-southeast2': 'https://alloydb.australia-southeast2.rep.googleapis.com/', 'me-central1': 'https://alloydb.me-central1.rep.googleapis.com/', 'me-central2': 'https://alloydb.me-central2.rep.googleapis.com/', 'me-west1': 'https://alloydb.me-west1.rep.googleapis.com/', 'africa-south1': 'https://alloydb.africa-south1.rep.googleapis.com/'}),
        'v1beta':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.alloydb.v1beta',
                    client_classpath='alloydb_v1beta_client.AlloydbV1beta',
                    base_url='https://alloydb.googleapis.com/',
                    messages_modulepath='alloydb_v1beta_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'us-central2': 'https://alloydb.us-central2.rep.googleapis.com/', 'us-east7': 'https://alloydb.us-east7.rep.googleapis.com/', 'northamerica-northeast1': 'https://alloydb.northamerica-northeast1.rep.googleapis.com/', 'northamerica-northeast2': 'https://alloydb.northamerica-northeast2.rep.googleapis.com/', 'northamerica-south1': 'https://alloydb.northamerica-south1.rep.googleapis.com/', 'southamerica-east1': 'https://alloydb.southamerica-east1.rep.googleapis.com/', 'southamerica-west1': 'https://alloydb.southamerica-west1.rep.googleapis.com/', 'us-central1': 'https://alloydb.us-central1.rep.googleapis.com/', 'us-east1': 'https://alloydb.us-east1.rep.googleapis.com/', 'us-east4': 'https://alloydb.us-east4.rep.googleapis.com/', 'us-east5': 'https://alloydb.us-east5.rep.googleapis.com/', 'us-south1': 'https://alloydb.us-south1.rep.googleapis.com/', 'us-west1': 'https://alloydb.us-west1.rep.googleapis.com/', 'us-west2': 'https://alloydb.us-west2.rep.googleapis.com/', 'us-west3': 'https://alloydb.us-west3.rep.googleapis.com/', 'us-west4': 'https://alloydb.us-west4.rep.googleapis.com/', 'europe-central2': 'https://alloydb.europe-central2.rep.googleapis.com/', 'europe-southwest1': 'https://alloydb.europe-southwest1.rep.googleapis.com/', 'europe-north1': 'https://alloydb.europe-north1.rep.googleapis.com/', 'europe-north2': 'https://alloydb.europe-north2.rep.googleapis.com/', 'europe-west1': 'https://alloydb.europe-west1.rep.googleapis.com/', 'europe-west2': 'https://alloydb.europe-west2.rep.googleapis.com/', 'europe-west3': 'https://alloydb.europe-west3.rep.googleapis.com/', 'europe-west4': 'https://alloydb.europe-west4.rep.googleapis.com/', 'europe-west6': 'https://alloydb.europe-west6.rep.googleapis.com/', 'europe-west8': 'https://alloydb.europe-west8.rep.googleapis.com/', 'europe-west9': 'https://alloydb.europe-west9.rep.googleapis.com/', 'europe-west10': 'https://alloydb.europe-west10.rep.googleapis.com/', 'europe-west12': 'https://alloydb.europe-west12.rep.googleapis.com/', 'asia-east1': 'https://alloydb.asia-east1.rep.googleapis.com/', 'asia-east2': 'https://alloydb.asia-east2.rep.googleapis.com/', 'asia-northeast1': 'https://alloydb.asia-northeast1.rep.googleapis.com/', 'asia-northeast2': 'https://alloydb.asia-northeast2.rep.googleapis.com/', 'asia-northeast3': 'https://alloydb.asia-northeast3.rep.googleapis.com/', 'asia-south1': 'https://alloydb.asia-south1.rep.googleapis.com/', 'asia-south2': 'https://alloydb.asia-south2.rep.googleapis.com/', 'asia-southeast1': 'https://alloydb.asia-southeast1.rep.googleapis.com/', 'asia-southeast2': 'https://alloydb.asia-southeast2.rep.googleapis.com/', 'australia-southeast1': 'https://alloydb.australia-southeast1.rep.googleapis.com/', 'australia-southeast2': 'https://alloydb.australia-southeast2.rep.googleapis.com/', 'me-central1': 'https://alloydb.me-central1.rep.googleapis.com/', 'me-central2': 'https://alloydb.me-central2.rep.googleapis.com/', 'me-west1': 'https://alloydb.me-west1.rep.googleapis.com/', 'africa-south1': 'https://alloydb.africa-south1.rep.googleapis.com/'}),
    },
    'anthosevents': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.anthosevents.v1',
                    client_classpath='anthosevents_v1_client.AnthoseventsV1',
                    base_url='https://anthosevents.googleapis.com/',
                    messages_modulepath='anthosevents_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.anthosevents.v1alpha1',
                    client_classpath='anthosevents_v1alpha1_client.AnthoseventsV1alpha1',
                    base_url='https://anthosevents.googleapis.com/',
                    messages_modulepath='anthosevents_v1alpha1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.anthosevents.v1beta1',
                    client_classpath='anthosevents_v1beta1_client.AnthoseventsV1beta1',
                    base_url='https://anthosevents.googleapis.com/',
                    messages_modulepath='anthosevents_v1beta1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'anthospolicycontrollerstatus_pa': {
        'v1alpha':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.anthospolicycontrollerstatus_pa.v1alpha',
                    client_classpath='anthospolicycontrollerstatus_pa_v1alpha_client.AnthospolicycontrollerstatusPaV1alpha',
                    base_url='https://anthospolicycontrollerstatus-pa.googleapis.com/',
                    messages_modulepath='anthospolicycontrollerstatus_pa_v1alpha_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'apigateway': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.apigateway.v1',
                    client_classpath='apigateway_v1_client.ApigatewayV1',
                    base_url='https://apigateway.googleapis.com/',
                    messages_modulepath='apigateway_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.apigateway.v1alpha1',
                    client_classpath='apigateway_v1alpha1_client.ApigatewayV1alpha1',
                    base_url='https://apigateway.googleapis.com/',
                    messages_modulepath='apigateway_v1alpha1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.apigateway.v1beta',
                    client_classpath='apigateway_v1beta_client.ApigatewayV1beta',
                    base_url='https://apigateway.googleapis.com/',
                    messages_modulepath='apigateway_v1beta_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'apigee': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.apigee.v1',
                    client_classpath='apigee_v1_client.ApigeeV1',
                    base_url='https://apigee.googleapis.com/',
                    messages_modulepath='apigee_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'us': 'https://apigee.us.rep.googleapis.com/', 'eu': 'https://apigee.eu.rep.googleapis.com/', 'in': 'https://apigee.in.rep.googleapis.com/'}),
        'v2alpha':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.apigee.v2alpha',
                    client_classpath='apigee_v2alpha_client.ApigeeV2alpha',
                    base_url='https://apigee.googleapis.com/',
                    messages_modulepath='apigee_v2alpha_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'apihub': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.apihub.v1',
                    client_classpath='apihub_v1_client.ApihubV1',
                    base_url='https://apihub.googleapis.com/',
                    messages_modulepath='apihub_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'apikeys': {
        'v2':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.apikeys.v2',
                    client_classpath='apikeys_v2_client.ApikeysV2',
                    base_url='https://apikeys.googleapis.com/',
                    messages_modulepath='apikeys_v2_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'appengine': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.appengine.v1',
                    client_classpath='appengine_v1_client.AppengineV1',
                    base_url='https://appengine.googleapis.com/',
                    messages_modulepath='appengine_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.appengine.v1alpha',
                    client_classpath='appengine_v1alpha_client.AppengineV1alpha',
                    base_url='https://appengine.googleapis.com/',
                    messages_modulepath='appengine_v1alpha_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.appengine.v1beta',
                    client_classpath='appengine_v1beta_client.AppengineV1beta',
                    base_url='https://appengine.googleapis.com/',
                    messages_modulepath='appengine_v1beta_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'apphub': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.apphub.v1',
                    client_classpath='apphub_v1_client.ApphubV1',
                    base_url='https://apphub.googleapis.com/',
                    messages_modulepath='apphub_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.apphub.v1alpha',
                    client_classpath='apphub_v1alpha_client.ApphubV1alpha',
                    base_url='https://apphub.googleapis.com/',
                    messages_modulepath='apphub_v1alpha_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'artifactregistry': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.artifactregistry.v1',
                    client_classpath='artifactregistry_v1_client.ArtifactregistryV1',
                    base_url='https://artifactregistry.googleapis.com/',
                    messages_modulepath='artifactregistry_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'africa-south1': 'https://artifactregistry.africa-south1.rep.googleapis.com/', 'asia-east1': 'https://artifactregistry.asia-east1.rep.googleapis.com/', 'asia-east2': 'https://artifactregistry.asia-east2.rep.googleapis.com/', 'asia-northeast1': 'https://artifactregistry.asia-northeast1.rep.googleapis.com/', 'asia-northeast2': 'https://artifactregistry.asia-northeast2.rep.googleapis.com/', 'asia-northeast3': 'https://artifactregistry.asia-northeast3.rep.googleapis.com/', 'asia-south1': 'https://artifactregistry.asia-south1.rep.googleapis.com/', 'asia-south2': 'https://artifactregistry.asia-south2.rep.googleapis.com/', 'asia-southeast1': 'https://artifactregistry.asia-southeast1.rep.googleapis.com/', 'asia-southeast2': 'https://artifactregistry.asia-southeast2.rep.googleapis.com/', 'australia-southeast1': 'https://artifactregistry.australia-southeast1.rep.googleapis.com/', 'australia-southeast2': 'https://artifactregistry.australia-southeast2.rep.googleapis.com/', 'europe-central2': 'https://artifactregistry.europe-central2.rep.googleapis.com/', 'europe-north1': 'https://artifactregistry.europe-north1.rep.googleapis.com/', 'europe-north2': 'https://artifactregistry.europe-north2.rep.googleapis.com/', 'europe-southwest1': 'https://artifactregistry.europe-southwest1.rep.googleapis.com/', 'europe-west1': 'https://artifactregistry.europe-west1.rep.googleapis.com/', 'europe-west10': 'https://artifactregistry.europe-west10.rep.googleapis.com/', 'europe-west12': 'https://artifactregistry.europe-west12.rep.googleapis.com/', 'europe-west2': 'https://artifactregistry.europe-west2.rep.googleapis.com/', 'europe-west3': 'https://artifactregistry.europe-west3.rep.googleapis.com/', 'europe-west4': 'https://artifactregistry.europe-west4.rep.googleapis.com/', 'europe-west6': 'https://artifactregistry.europe-west6.rep.googleapis.com/', 'europe-west8': 'https://artifactregistry.europe-west8.rep.googleapis.com/', 'europe-west9': 'https://artifactregistry.europe-west9.rep.googleapis.com/', 'me-central1': 'https://artifactregistry.me-central1.rep.googleapis.com/', 'me-west1': 'https://artifactregistry.me-west1.rep.googleapis.com/', 'northamerica-northeast1': 'https://artifactregistry.northamerica-northeast1.rep.googleapis.com/', 'northamerica-northeast2': 'https://artifactregistry.northamerica-northeast2.rep.googleapis.com/', 'northamerica-south1': 'https://artifactregistry.northamerica-south1.rep.googleapis.com/', 'southamerica-east1': 'https://artifactregistry.southamerica-east1.rep.googleapis.com/', 'southamerica-west1': 'https://artifactregistry.southamerica-west1.rep.googleapis.com/', 'us-central1': 'https://artifactregistry.us-central1.rep.googleapis.com/', 'us-central2': 'https://artifactregistry.us-central2.rep.googleapis.com/', 'us-east1': 'https://artifactregistry.us-east1.rep.googleapis.com/', 'us-east4': 'https://artifactregistry.us-east4.rep.googleapis.com/', 'us-east5': 'https://artifactregistry.us-east5.rep.googleapis.com/', 'us-south1': 'https://artifactregistry.us-south1.rep.googleapis.com/', 'us-west1': 'https://artifactregistry.us-west1.rep.googleapis.com/', 'us-west2': 'https://artifactregistry.us-west2.rep.googleapis.com/', 'us-west3': 'https://artifactregistry.us-west3.rep.googleapis.com/', 'us-west4': 'https://artifactregistry.us-west4.rep.googleapis.com/', 'us-west8': 'https://artifactregistry.us-west8.rep.googleapis.com/', 'me-central2': 'https://artifactregistry.me-central2.rep.googleapis.com/', 'us-east7': 'https://artifactregistry.us-east7.rep.googleapis.com/', 'us': 'https://artifactregistry.us.rep.googleapis.com/', 'eu': 'https://artifactregistry.eu.rep.googleapis.com/'}),
        'v1beta1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.artifactregistry.v1beta1',
                    client_classpath='artifactregistry_v1beta1_client.ArtifactregistryV1beta1',
                    base_url='https://artifactregistry.googleapis.com/',
                    messages_modulepath='artifactregistry_v1beta1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'africa-south1': 'https://artifactregistry.africa-south1.rep.googleapis.com/', 'asia-east1': 'https://artifactregistry.asia-east1.rep.googleapis.com/', 'asia-east2': 'https://artifactregistry.asia-east2.rep.googleapis.com/', 'asia-northeast1': 'https://artifactregistry.asia-northeast1.rep.googleapis.com/', 'asia-northeast2': 'https://artifactregistry.asia-northeast2.rep.googleapis.com/', 'asia-northeast3': 'https://artifactregistry.asia-northeast3.rep.googleapis.com/', 'asia-south1': 'https://artifactregistry.asia-south1.rep.googleapis.com/', 'asia-south2': 'https://artifactregistry.asia-south2.rep.googleapis.com/', 'asia-southeast1': 'https://artifactregistry.asia-southeast1.rep.googleapis.com/', 'asia-southeast2': 'https://artifactregistry.asia-southeast2.rep.googleapis.com/', 'australia-southeast1': 'https://artifactregistry.australia-southeast1.rep.googleapis.com/', 'australia-southeast2': 'https://artifactregistry.australia-southeast2.rep.googleapis.com/', 'europe-central2': 'https://artifactregistry.europe-central2.rep.googleapis.com/', 'europe-north1': 'https://artifactregistry.europe-north1.rep.googleapis.com/', 'europe-north2': 'https://artifactregistry.europe-north2.rep.googleapis.com/', 'europe-southwest1': 'https://artifactregistry.europe-southwest1.rep.googleapis.com/', 'europe-west1': 'https://artifactregistry.europe-west1.rep.googleapis.com/', 'europe-west10': 'https://artifactregistry.europe-west10.rep.googleapis.com/', 'europe-west12': 'https://artifactregistry.europe-west12.rep.googleapis.com/', 'europe-west2': 'https://artifactregistry.europe-west2.rep.googleapis.com/', 'europe-west3': 'https://artifactregistry.europe-west3.rep.googleapis.com/', 'europe-west4': 'https://artifactregistry.europe-west4.rep.googleapis.com/', 'europe-west6': 'https://artifactregistry.europe-west6.rep.googleapis.com/', 'europe-west8': 'https://artifactregistry.europe-west8.rep.googleapis.com/', 'europe-west9': 'https://artifactregistry.europe-west9.rep.googleapis.com/', 'me-central1': 'https://artifactregistry.me-central1.rep.googleapis.com/', 'me-west1': 'https://artifactregistry.me-west1.rep.googleapis.com/', 'northamerica-northeast1': 'https://artifactregistry.northamerica-northeast1.rep.googleapis.com/', 'northamerica-northeast2': 'https://artifactregistry.northamerica-northeast2.rep.googleapis.com/', 'northamerica-south1': 'https://artifactregistry.northamerica-south1.rep.googleapis.com/', 'southamerica-east1': 'https://artifactregistry.southamerica-east1.rep.googleapis.com/', 'southamerica-west1': 'https://artifactregistry.southamerica-west1.rep.googleapis.com/', 'us-central1': 'https://artifactregistry.us-central1.rep.googleapis.com/', 'us-central2': 'https://artifactregistry.us-central2.rep.googleapis.com/', 'us-east1': 'https://artifactregistry.us-east1.rep.googleapis.com/', 'us-east4': 'https://artifactregistry.us-east4.rep.googleapis.com/', 'us-east5': 'https://artifactregistry.us-east5.rep.googleapis.com/', 'us-south1': 'https://artifactregistry.us-south1.rep.googleapis.com/', 'us-west1': 'https://artifactregistry.us-west1.rep.googleapis.com/', 'us-west2': 'https://artifactregistry.us-west2.rep.googleapis.com/', 'us-west3': 'https://artifactregistry.us-west3.rep.googleapis.com/', 'us-west4': 'https://artifactregistry.us-west4.rep.googleapis.com/', 'us-west8': 'https://artifactregistry.us-west8.rep.googleapis.com/', 'me-central2': 'https://artifactregistry.me-central2.rep.googleapis.com/', 'us-east7': 'https://artifactregistry.us-east7.rep.googleapis.com/', 'us': 'https://artifactregistry.us.rep.googleapis.com/', 'eu': 'https://artifactregistry.eu.rep.googleapis.com/'}),
        'v1beta2':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.artifactregistry.v1beta2',
                    client_classpath='artifactregistry_v1beta2_client.ArtifactregistryV1beta2',
                    base_url='https://artifactregistry.googleapis.com/',
                    messages_modulepath='artifactregistry_v1beta2_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'africa-south1': 'https://artifactregistry.africa-south1.rep.googleapis.com/', 'asia-east1': 'https://artifactregistry.asia-east1.rep.googleapis.com/', 'asia-east2': 'https://artifactregistry.asia-east2.rep.googleapis.com/', 'asia-northeast1': 'https://artifactregistry.asia-northeast1.rep.googleapis.com/', 'asia-northeast2': 'https://artifactregistry.asia-northeast2.rep.googleapis.com/', 'asia-northeast3': 'https://artifactregistry.asia-northeast3.rep.googleapis.com/', 'asia-south1': 'https://artifactregistry.asia-south1.rep.googleapis.com/', 'asia-south2': 'https://artifactregistry.asia-south2.rep.googleapis.com/', 'asia-southeast1': 'https://artifactregistry.asia-southeast1.rep.googleapis.com/', 'asia-southeast2': 'https://artifactregistry.asia-southeast2.rep.googleapis.com/', 'australia-southeast1': 'https://artifactregistry.australia-southeast1.rep.googleapis.com/', 'australia-southeast2': 'https://artifactregistry.australia-southeast2.rep.googleapis.com/', 'europe-central2': 'https://artifactregistry.europe-central2.rep.googleapis.com/', 'europe-north1': 'https://artifactregistry.europe-north1.rep.googleapis.com/', 'europe-north2': 'https://artifactregistry.europe-north2.rep.googleapis.com/', 'europe-southwest1': 'https://artifactregistry.europe-southwest1.rep.googleapis.com/', 'europe-west1': 'https://artifactregistry.europe-west1.rep.googleapis.com/', 'europe-west10': 'https://artifactregistry.europe-west10.rep.googleapis.com/', 'europe-west12': 'https://artifactregistry.europe-west12.rep.googleapis.com/', 'europe-west2': 'https://artifactregistry.europe-west2.rep.googleapis.com/', 'europe-west3': 'https://artifactregistry.europe-west3.rep.googleapis.com/', 'europe-west4': 'https://artifactregistry.europe-west4.rep.googleapis.com/', 'europe-west6': 'https://artifactregistry.europe-west6.rep.googleapis.com/', 'europe-west8': 'https://artifactregistry.europe-west8.rep.googleapis.com/', 'europe-west9': 'https://artifactregistry.europe-west9.rep.googleapis.com/', 'me-central1': 'https://artifactregistry.me-central1.rep.googleapis.com/', 'me-west1': 'https://artifactregistry.me-west1.rep.googleapis.com/', 'northamerica-northeast1': 'https://artifactregistry.northamerica-northeast1.rep.googleapis.com/', 'northamerica-northeast2': 'https://artifactregistry.northamerica-northeast2.rep.googleapis.com/', 'northamerica-south1': 'https://artifactregistry.northamerica-south1.rep.googleapis.com/', 'southamerica-east1': 'https://artifactregistry.southamerica-east1.rep.googleapis.com/', 'southamerica-west1': 'https://artifactregistry.southamerica-west1.rep.googleapis.com/', 'us-central1': 'https://artifactregistry.us-central1.rep.googleapis.com/', 'us-central2': 'https://artifactregistry.us-central2.rep.googleapis.com/', 'us-east1': 'https://artifactregistry.us-east1.rep.googleapis.com/', 'us-east4': 'https://artifactregistry.us-east4.rep.googleapis.com/', 'us-east5': 'https://artifactregistry.us-east5.rep.googleapis.com/', 'us-south1': 'https://artifactregistry.us-south1.rep.googleapis.com/', 'us-west1': 'https://artifactregistry.us-west1.rep.googleapis.com/', 'us-west2': 'https://artifactregistry.us-west2.rep.googleapis.com/', 'us-west3': 'https://artifactregistry.us-west3.rep.googleapis.com/', 'us-west4': 'https://artifactregistry.us-west4.rep.googleapis.com/', 'us-west8': 'https://artifactregistry.us-west8.rep.googleapis.com/', 'me-central2': 'https://artifactregistry.me-central2.rep.googleapis.com/', 'us-east7': 'https://artifactregistry.us-east7.rep.googleapis.com/', 'us': 'https://artifactregistry.us.rep.googleapis.com/', 'eu': 'https://artifactregistry.eu.rep.googleapis.com/'}),
    },
    'artifactscanguard': {
        'v1alpha':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.artifactscanguard.v1alpha',
                    client_classpath='artifactscanguard_v1alpha_client.ArtifactscanguardV1alpha',
                    base_url='https://artifactscanguard.googleapis.com/',
                    messages_modulepath='artifactscanguard_v1alpha_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'assuredworkloads': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.assuredworkloads.v1',
                    client_classpath='assuredworkloads_v1_client.AssuredworkloadsV1',
                    base_url='https://assuredworkloads.googleapis.com/',
                    messages_modulepath='assuredworkloads_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.assuredworkloads.v1beta1',
                    client_classpath='assuredworkloads_v1beta1_client.AssuredworkloadsV1beta1',
                    base_url='https://assuredworkloads.googleapis.com/',
                    messages_modulepath='assuredworkloads_v1beta1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'auditmanager': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.auditmanager.v1',
                    client_classpath='auditmanager_v1_client.AuditmanagerV1',
                    base_url='https://auditmanager.googleapis.com/',
                    messages_modulepath='auditmanager_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.auditmanager.v1alpha',
                    client_classpath='auditmanager_v1alpha_client.AuditmanagerV1alpha',
                    base_url='https://auditmanager.googleapis.com/',
                    messages_modulepath='auditmanager_v1alpha_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'authztoolkit': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.authztoolkit.v1',
                    client_classpath='authztoolkit_v1_client.AuthztoolkitV1',
                    base_url='https://authztoolkit.googleapis.com/',
                    messages_modulepath='authztoolkit_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.authztoolkit.v1alpha',
                    client_classpath='authztoolkit_v1alpha_client.AuthztoolkitV1alpha',
                    base_url='https://authztoolkit.googleapis.com/',
                    messages_modulepath='authztoolkit_v1alpha_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'backupdr': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.backupdr.v1',
                    client_classpath='backupdr_v1_client.BackupdrV1',
                    base_url='https://backupdr.googleapis.com/',
                    messages_modulepath='backupdr_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'baremetalsolution': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.baremetalsolution.v1',
                    client_classpath='baremetalsolution_v1_client.BaremetalsolutionV1',
                    base_url='https://baremetalsolution.googleapis.com/',
                    messages_modulepath='baremetalsolution_v1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v2':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.baremetalsolution.v2',
                    client_classpath='baremetalsolution_v2_client.BaremetalsolutionV2',
                    base_url='https://baremetalsolution.googleapis.com/',
                    messages_modulepath='baremetalsolution_v2_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'batch': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.batch.v1',
                    client_classpath='batch_v1_client.BatchV1',
                    base_url='https://batch.googleapis.com/',
                    messages_modulepath='batch_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.batch.v1alpha',
                    client_classpath='batch_v1alpha_client.BatchV1alpha',
                    base_url='https://batch.googleapis.com/',
                    messages_modulepath='batch_v1alpha_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.batch.v1alpha1',
                    client_classpath='batch_v1alpha1_client.BatchV1alpha1',
                    base_url='https://batch.googleapis.com/',
                    messages_modulepath='batch_v1alpha1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'beyondcorp': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.beyondcorp.v1',
                    client_classpath='beyondcorp_v1_client.BeyondcorpV1',
                    base_url='https://beyondcorp.googleapis.com/',
                    messages_modulepath='beyondcorp_v1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.beyondcorp.v1alpha',
                    client_classpath='beyondcorp_v1alpha_client.BeyondcorpV1alpha',
                    base_url='https://beyondcorp.googleapis.com/',
                    messages_modulepath='beyondcorp_v1alpha_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'biglake': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.biglake.v1',
                    client_classpath='biglake_v1_client.BiglakeV1',
                    base_url='https://biglake.googleapis.com/',
                    messages_modulepath='biglake_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'bigquery': {
        'v2':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.bigquery.v2',
                    client_classpath='bigquery_v2_client.BigqueryV2',
                    base_url='https://bigquery.googleapis.com/bigquery/v2/',
                    messages_modulepath='bigquery_v2_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='https://bigquery.mtls.googleapis.com/bigquery/v2/',
                regional_endpoints={'asia-south1': 'https://bigquery.asia-south1.rep.googleapis.com/', 'asia-south2': 'https://bigquery.asia-south2.rep.googleapis.com/', 'europe-west1': 'https://bigquery.europe-west1.rep.googleapis.com/', 'europe-west2': 'https://bigquery.europe-west2.rep.googleapis.com/', 'europe-west3': 'https://bigquery.europe-west3.rep.googleapis.com/', 'europe-west4': 'https://bigquery.europe-west4.rep.googleapis.com/', 'europe-west6': 'https://bigquery.europe-west6.rep.googleapis.com/', 'europe-west8': 'https://bigquery.europe-west8.rep.googleapis.com/', 'europe-west9': 'https://bigquery.europe-west9.rep.googleapis.com/', 'me-central2': 'https://bigquery.me-central2.rep.googleapis.com/', 'northamerica-northeast1': 'https://bigquery.northamerica-northeast1.rep.googleapis.com/', 'northamerica-northeast2': 'https://bigquery.northamerica-northeast2.rep.googleapis.com/', 'us-central1': 'https://bigquery.us-central1.rep.googleapis.com/', 'us-central2': 'https://bigquery.us-central2.rep.googleapis.com/', 'us-east1': 'https://bigquery.us-east1.rep.googleapis.com/', 'us-east4': 'https://bigquery.us-east4.rep.googleapis.com/', 'us-east5': 'https://bigquery.us-east5.rep.googleapis.com/', 'us-east7': 'https://bigquery.us-east7.rep.googleapis.com/', 'us-south1': 'https://bigquery.us-south1.rep.googleapis.com/', 'us-west1': 'https://bigquery.us-west1.rep.googleapis.com/', 'us-west2': 'https://bigquery.us-west2.rep.googleapis.com/', 'us-west3': 'https://bigquery.us-west3.rep.googleapis.com/', 'us-west4': 'https://bigquery.us-west4.rep.googleapis.com/', 'us-west8': 'https://bigquery.us-west8.rep.googleapis.com/'}),
    },
    'bigquerydatatransfer': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.bigquerydatatransfer.v1',
                    client_classpath='bigquerydatatransfer_v1_client.BigquerydatatransferV1',
                    base_url='https://bigquerydatatransfer.googleapis.com/',
                    messages_modulepath='bigquerydatatransfer_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'asia-south1': 'https://bigquerydatatransfer.asia-south1.rep.googleapis.com/', 'asia-south2': 'https://bigquerydatatransfer.asia-south2.rep.googleapis.com/', 'europe-west1': 'https://bigquerydatatransfer.europe-west1.rep.googleapis.com/', 'europe-west2': 'https://bigquerydatatransfer.europe-west2.rep.googleapis.com/', 'europe-west3': 'https://bigquerydatatransfer.europe-west3.rep.googleapis.com/', 'europe-west6': 'https://bigquerydatatransfer.europe-west6.rep.googleapis.com/', 'europe-west8': 'https://bigquerydatatransfer.europe-west8.rep.googleapis.com/', 'europe-west9': 'https://bigquerydatatransfer.europe-west9.rep.googleapis.com/', 'me-central2': 'https://bigquerydatatransfer.me-central2.rep.googleapis.com/', 'us-central1': 'https://bigquerydatatransfer.us-central1.rep.googleapis.com/', 'us-central2': 'https://bigquerydatatransfer.us-central2.rep.googleapis.com/', 'us-east1': 'https://bigquerydatatransfer.us-east1.rep.googleapis.com/', 'us-east4': 'https://bigquerydatatransfer.us-east4.rep.googleapis.com/', 'us-east5': 'https://bigquerydatatransfer.us-east5.rep.googleapis.com/', 'us-east7': 'https://bigquerydatatransfer.us-east7.rep.googleapis.com/', 'us-south1': 'https://bigquerydatatransfer.us-south1.rep.googleapis.com/', 'us-west1': 'https://bigquerydatatransfer.us-west1.rep.googleapis.com/', 'us-west2': 'https://bigquerydatatransfer.us-west2.rep.googleapis.com/', 'us-west3': 'https://bigquerydatatransfer.us-west3.rep.googleapis.com/', 'us-west4': 'https://bigquerydatatransfer.us-west4.rep.googleapis.com/', 'us-west8': 'https://bigquerydatatransfer.us-west8.rep.googleapis.com/'}),
    },
    'bigquerymigration': {
        'v2':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.bigquerymigration.v2',
                    client_classpath='bigquerymigration_v2_client.BigquerymigrationV2',
                    base_url='https://bigquerymigration.googleapis.com/',
                    messages_modulepath='bigquerymigration_v2_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'asia-northeast1': 'https://bigquerymigration.asia-northeast1.rep.googleapis.com/', 'asia-south1': 'https://bigquerymigration.asia-south1.rep.googleapis.com/', 'asia-south2': 'https://bigquerymigration.asia-south2.rep.googleapis.com/', 'asia-southeast1': 'https://bigquerymigration.asia-southeast1.rep.googleapis.com/', 'australia-southeast1': 'https://bigquerymigration.australia-southeast1.rep.googleapis.com/', 'europe-west1': 'https://bigquerymigration.europe-west1.rep.googleapis.com/', 'europe-west2': 'https://bigquerymigration.europe-west2.rep.googleapis.com/', 'europe-west3': 'https://bigquerymigration.europe-west3.rep.googleapis.com/', 'europe-west6': 'https://bigquerymigration.europe-west6.rep.googleapis.com/', 'europe-west8': 'https://bigquerymigration.europe-west8.rep.googleapis.com/', 'europe-west9': 'https://bigquerymigration.europe-west9.rep.googleapis.com/', 'me-central2': 'https://bigquerymigration.me-central2.rep.googleapis.com/', 'northamerica-northeast1': 'https://bigquerymigration.northamerica-northeast1.rep.googleapis.com/', 'northamerica-northeast2': 'https://bigquerymigration.northamerica-northeast2.rep.googleapis.com/', 'southamerica-east1': 'https://bigquerymigration.southamerica-east1.rep.googleapis.com/', 'us-central1': 'https://bigquerymigration.us-central1.rep.googleapis.com/', 'us-central2': 'https://bigquerymigration.us-central2.rep.googleapis.com/', 'us-east1': 'https://bigquerymigration.us-east1.rep.googleapis.com/', 'us-east4': 'https://bigquerymigration.us-east4.rep.googleapis.com/', 'us-east5': 'https://bigquerymigration.us-east5.rep.googleapis.com/', 'us-east7': 'https://bigquerymigration.us-east7.rep.googleapis.com/', 'us-south1': 'https://bigquerymigration.us-south1.rep.googleapis.com/', 'us-west1': 'https://bigquerymigration.us-west1.rep.googleapis.com/', 'us-west2': 'https://bigquerymigration.us-west2.rep.googleapis.com/', 'us-west3': 'https://bigquerymigration.us-west3.rep.googleapis.com/', 'us-west4': 'https://bigquerymigration.us-west4.rep.googleapis.com/', 'us-west8': 'https://bigquerymigration.us-west8.rep.googleapis.com/'}),
    },
    'bigtableadmin': {
        'v2':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.bigtableadmin.v2',
                    client_classpath='bigtableadmin_v2_client.BigtableadminV2',
                    base_url='https://bigtableadmin.googleapis.com/',
                    messages_modulepath='bigtableadmin_v2_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'billingbudgets': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.billingbudgets.v1',
                    client_classpath='billingbudgets_v1_client.BillingbudgetsV1',
                    base_url='https://billingbudgets.googleapis.com/',
                    messages_modulepath='billingbudgets_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.billingbudgets.v1beta1',
                    client_classpath='billingbudgets_v1beta1_client.BillingbudgetsV1beta1',
                    base_url='https://billingbudgets.googleapis.com/',
                    messages_modulepath='billingbudgets_v1beta1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'binaryauthorization': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.binaryauthorization.v1',
                    client_classpath='binaryauthorization_v1_client.BinaryauthorizationV1',
                    base_url='https://binaryauthorization.googleapis.com/',
                    messages_modulepath='binaryauthorization_v1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha2':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.binaryauthorization.v1alpha2',
                    client_classpath='binaryauthorization_v1alpha2_client.BinaryauthorizationV1alpha2',
                    base_url='https://binaryauthorization.googleapis.com/',
                    messages_modulepath='binaryauthorization_v1alpha2_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.binaryauthorization.v1beta1',
                    client_classpath='binaryauthorization_v1beta1_client.BinaryauthorizationV1beta1',
                    base_url='https://binaryauthorization.googleapis.com/',
                    messages_modulepath='binaryauthorization_v1beta1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'blockchainnodeengine': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.blockchainnodeengine.v1',
                    client_classpath='blockchainnodeengine_v1_client.BlockchainnodeengineV1',
                    base_url='https://blockchainnodeengine.googleapis.com/',
                    messages_modulepath='blockchainnodeengine_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'blockchainvalidatormanager': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.blockchainvalidatormanager.v1',
                    client_classpath='blockchainvalidatormanager_v1_client.BlockchainvalidatormanagerV1',
                    base_url='https://blockchainvalidatormanager.googleapis.com/',
                    messages_modulepath='blockchainvalidatormanager_v1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.blockchainvalidatormanager.v1alpha',
                    client_classpath='blockchainvalidatormanager_v1alpha_client.BlockchainvalidatormanagerV1alpha',
                    base_url='https://blockchainvalidatormanager.googleapis.com/',
                    messages_modulepath='blockchainvalidatormanager_v1alpha_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'certificatemanager': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.certificatemanager.v1',
                    client_classpath='certificatemanager_v1_client.CertificatemanagerV1',
                    base_url='https://certificatemanager.googleapis.com/',
                    messages_modulepath='certificatemanager_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'cloudaicompanion': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.cloudaicompanion.v1',
                    client_classpath='cloudaicompanion_v1_client.CloudaicompanionV1',
                    base_url='https://cloudaicompanion.googleapis.com/',
                    messages_modulepath='cloudaicompanion_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.cloudaicompanion.v1alpha',
                    client_classpath='cloudaicompanion_v1alpha_client.CloudaicompanionV1alpha',
                    base_url='https://cloudaicompanion.googleapis.com/',
                    messages_modulepath='cloudaicompanion_v1alpha_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'cloudapiregistry': {
        'v1alpha':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.cloudapiregistry.v1alpha',
                    client_classpath='cloudapiregistry_v1alpha_client.CloudapiregistryV1alpha',
                    base_url='https://cloudapiregistry.googleapis.com/',
                    messages_modulepath='cloudapiregistry_v1alpha_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.cloudapiregistry.v1beta',
                    client_classpath='cloudapiregistry_v1beta_client.CloudapiregistryV1beta',
                    base_url='https://cloudapiregistry.googleapis.com/',
                    messages_modulepath='cloudapiregistry_v1beta_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'cloudasset': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.cloudasset.v1',
                    client_classpath='cloudasset_v1_client.CloudassetV1',
                    base_url='https://cloudasset.googleapis.com/',
                    messages_modulepath='cloudasset_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1p1beta1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.cloudasset.v1p1beta1',
                    client_classpath='cloudasset_v1p1beta1_client.CloudassetV1p1beta1',
                    base_url='https://cloudasset.googleapis.com/',
                    messages_modulepath='cloudasset_v1p1beta1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1p2beta1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.cloudasset.v1p2beta1',
                    client_classpath='cloudasset_v1p2beta1_client.CloudassetV1p2beta1',
                    base_url='https://cloudasset.googleapis.com/',
                    messages_modulepath='cloudasset_v1p2beta1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='https://cloudasset.mtls.googleapis.com/',
                regional_endpoints={}),
        'v1p4alpha1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.cloudasset.v1p4alpha1',
                    client_classpath='cloudasset_v1p4alpha1_client.CloudassetV1p4alpha1',
                    base_url='https://cloudasset.googleapis.com/',
                    messages_modulepath='cloudasset_v1p4alpha1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='https://cloudasset.mtls.googleapis.com/',
                regional_endpoints={}),
        'v1p5beta1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.cloudasset.v1p5beta1',
                    client_classpath='cloudasset_v1p5beta1_client.CloudassetV1p5beta1',
                    base_url='https://cloudasset.googleapis.com/',
                    messages_modulepath='cloudasset_v1p5beta1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1p7beta1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.cloudasset.v1p7beta1',
                    client_classpath='cloudasset_v1p7beta1_client.CloudassetV1p7beta1',
                    base_url='https://cloudasset.googleapis.com/',
                    messages_modulepath='cloudasset_v1p7beta1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'cloudbilling': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.cloudbilling.v1',
                    client_classpath='cloudbilling_v1_client.CloudbillingV1',
                    base_url='https://cloudbilling.googleapis.com/',
                    messages_modulepath='cloudbilling_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'cloudbuild': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.cloudbuild.v1',
                    client_classpath='cloudbuild_v1_client.CloudbuildV1',
                    base_url='https://cloudbuild.googleapis.com/',
                    messages_modulepath='cloudbuild_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'africa-south1': 'https://cloudbuild.africa-south1.rep.googleapis.com/', 'asia-east1': 'https://cloudbuild.asia-east1.rep.googleapis.com/', 'asia-east2': 'https://cloudbuild.asia-east2.rep.googleapis.com/', 'asia-northeast1': 'https://cloudbuild.asia-northeast1.rep.googleapis.com/', 'asia-northeast2': 'https://cloudbuild.asia-northeast2.rep.googleapis.com/', 'asia-northeast3': 'https://cloudbuild.asia-northeast3.rep.googleapis.com/', 'asia-south1': 'https://cloudbuild.asia-south1.rep.googleapis.com/', 'asia-south2': 'https://cloudbuild.asia-south2.rep.googleapis.com/', 'asia-southeast1': 'https://cloudbuild.asia-southeast1.rep.googleapis.com/', 'asia-southeast2': 'https://cloudbuild.asia-southeast2.rep.googleapis.com/', 'asia-southeast3': 'https://cloudbuild.asia-southeast3.rep.googleapis.com/', 'australia-southeast1': 'https://cloudbuild.australia-southeast1.rep.googleapis.com/', 'australia-southeast2': 'https://cloudbuild.australia-southeast2.rep.googleapis.com/', 'europe-central2': 'https://cloudbuild.europe-central2.rep.googleapis.com/', 'europe-north1': 'https://cloudbuild.europe-north1.rep.googleapis.com/', 'europe-north2': 'https://cloudbuild.europe-north2.rep.googleapis.com/', 'europe-southwest1': 'https://cloudbuild.europe-southwest1.rep.googleapis.com/', 'europe-west1': 'https://cloudbuild.europe-west1.rep.googleapis.com/', 'europe-west10': 'https://cloudbuild.europe-west10.rep.googleapis.com/', 'europe-west12': 'https://cloudbuild.europe-west12.rep.googleapis.com/', 'europe-west2': 'https://cloudbuild.europe-west2.rep.googleapis.com/', 'europe-west3': 'https://cloudbuild.europe-west3.rep.googleapis.com/', 'europe-west4': 'https://cloudbuild.europe-west4.rep.googleapis.com/', 'europe-west6': 'https://cloudbuild.europe-west6.rep.googleapis.com/', 'europe-west8': 'https://cloudbuild.europe-west8.rep.googleapis.com/', 'europe-west9': 'https://cloudbuild.europe-west9.rep.googleapis.com/', 'europe-west15': 'https://cloudbuild.europe-west15.rep.googleapis.com/', 'me-central1': 'https://cloudbuild.me-central1.rep.googleapis.com/', 'me-central2': 'https://cloudbuild.me-central2.rep.googleapis.com/', 'me-west1': 'https://cloudbuild.me-west1.rep.googleapis.com/', 'northamerica-northeast1': 'https://cloudbuild.northamerica-northeast1.rep.googleapis.com/', 'northamerica-northeast2': 'https://cloudbuild.northamerica-northeast2.rep.googleapis.com/', 'northamerica-south1': 'https://cloudbuild.northamerica-south1.rep.googleapis.com/', 'southamerica-east1': 'https://cloudbuild.southamerica-east1.rep.googleapis.com/', 'southamerica-west1': 'https://cloudbuild.southamerica-west1.rep.googleapis.com/', 'us-central1': 'https://cloudbuild.us-central1.rep.googleapis.com/', 'us-central2': 'https://cloudbuild.us-central2.rep.googleapis.com/', 'us-east1': 'https://cloudbuild.us-east1.rep.googleapis.com/', 'us-east4': 'https://cloudbuild.us-east4.rep.googleapis.com/', 'us-east5': 'https://cloudbuild.us-east5.rep.googleapis.com/', 'us-east7': 'https://cloudbuild.us-east7.rep.googleapis.com/', 'us-south1': 'https://cloudbuild.us-south1.rep.googleapis.com/', 'us-west1': 'https://cloudbuild.us-west1.rep.googleapis.com/', 'us-west2': 'https://cloudbuild.us-west2.rep.googleapis.com/', 'us-west3': 'https://cloudbuild.us-west3.rep.googleapis.com/', 'us-west4': 'https://cloudbuild.us-west4.rep.googleapis.com/', 'us-west8': 'https://cloudbuild.us-west8.rep.googleapis.com/'}),
        'v1alpha1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.cloudbuild.v1alpha1',
                    client_classpath='cloudbuild_v1alpha1_client.CloudbuildV1alpha1',
                    base_url='https://cloudbuild.googleapis.com/',
                    messages_modulepath='cloudbuild_v1alpha1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha2':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.cloudbuild.v1alpha2',
                    client_classpath='cloudbuild_v1alpha2_client.CloudbuildV1alpha2',
                    base_url='https://cloudbuild.googleapis.com/',
                    messages_modulepath='cloudbuild_v1alpha2_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.cloudbuild.v1beta1',
                    client_classpath='cloudbuild_v1beta1_client.CloudbuildV1beta1',
                    base_url='https://cloudbuild.googleapis.com/',
                    messages_modulepath='cloudbuild_v1beta1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v2':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.cloudbuild.v2',
                    client_classpath='cloudbuild_v2_client.CloudbuildV2',
                    base_url='https://cloudbuild.googleapis.com/',
                    messages_modulepath='cloudbuild_v2_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'africa-south1': 'https://cloudbuild.africa-south1.rep.googleapis.com/', 'asia-east1': 'https://cloudbuild.asia-east1.rep.googleapis.com/', 'asia-east2': 'https://cloudbuild.asia-east2.rep.googleapis.com/', 'asia-northeast1': 'https://cloudbuild.asia-northeast1.rep.googleapis.com/', 'asia-northeast2': 'https://cloudbuild.asia-northeast2.rep.googleapis.com/', 'asia-northeast3': 'https://cloudbuild.asia-northeast3.rep.googleapis.com/', 'asia-south1': 'https://cloudbuild.asia-south1.rep.googleapis.com/', 'asia-south2': 'https://cloudbuild.asia-south2.rep.googleapis.com/', 'asia-southeast1': 'https://cloudbuild.asia-southeast1.rep.googleapis.com/', 'asia-southeast2': 'https://cloudbuild.asia-southeast2.rep.googleapis.com/', 'asia-southeast3': 'https://cloudbuild.asia-southeast3.rep.googleapis.com/', 'australia-southeast1': 'https://cloudbuild.australia-southeast1.rep.googleapis.com/', 'australia-southeast2': 'https://cloudbuild.australia-southeast2.rep.googleapis.com/', 'europe-central2': 'https://cloudbuild.europe-central2.rep.googleapis.com/', 'europe-north1': 'https://cloudbuild.europe-north1.rep.googleapis.com/', 'europe-north2': 'https://cloudbuild.europe-north2.rep.googleapis.com/', 'europe-southwest1': 'https://cloudbuild.europe-southwest1.rep.googleapis.com/', 'europe-west1': 'https://cloudbuild.europe-west1.rep.googleapis.com/', 'europe-west10': 'https://cloudbuild.europe-west10.rep.googleapis.com/', 'europe-west12': 'https://cloudbuild.europe-west12.rep.googleapis.com/', 'europe-west2': 'https://cloudbuild.europe-west2.rep.googleapis.com/', 'europe-west3': 'https://cloudbuild.europe-west3.rep.googleapis.com/', 'europe-west4': 'https://cloudbuild.europe-west4.rep.googleapis.com/', 'europe-west6': 'https://cloudbuild.europe-west6.rep.googleapis.com/', 'europe-west8': 'https://cloudbuild.europe-west8.rep.googleapis.com/', 'europe-west9': 'https://cloudbuild.europe-west9.rep.googleapis.com/', 'europe-west15': 'https://cloudbuild.europe-west15.rep.googleapis.com/', 'me-central1': 'https://cloudbuild.me-central1.rep.googleapis.com/', 'me-central2': 'https://cloudbuild.me-central2.rep.googleapis.com/', 'me-west1': 'https://cloudbuild.me-west1.rep.googleapis.com/', 'northamerica-northeast1': 'https://cloudbuild.northamerica-northeast1.rep.googleapis.com/', 'northamerica-northeast2': 'https://cloudbuild.northamerica-northeast2.rep.googleapis.com/', 'northamerica-south1': 'https://cloudbuild.northamerica-south1.rep.googleapis.com/', 'southamerica-east1': 'https://cloudbuild.southamerica-east1.rep.googleapis.com/', 'southamerica-west1': 'https://cloudbuild.southamerica-west1.rep.googleapis.com/', 'us-central1': 'https://cloudbuild.us-central1.rep.googleapis.com/', 'us-central2': 'https://cloudbuild.us-central2.rep.googleapis.com/', 'us-east1': 'https://cloudbuild.us-east1.rep.googleapis.com/', 'us-east4': 'https://cloudbuild.us-east4.rep.googleapis.com/', 'us-east5': 'https://cloudbuild.us-east5.rep.googleapis.com/', 'us-east7': 'https://cloudbuild.us-east7.rep.googleapis.com/', 'us-south1': 'https://cloudbuild.us-south1.rep.googleapis.com/', 'us-west1': 'https://cloudbuild.us-west1.rep.googleapis.com/', 'us-west2': 'https://cloudbuild.us-west2.rep.googleapis.com/', 'us-west3': 'https://cloudbuild.us-west3.rep.googleapis.com/', 'us-west4': 'https://cloudbuild.us-west4.rep.googleapis.com/', 'us-west8': 'https://cloudbuild.us-west8.rep.googleapis.com/'}),
    },
    'cloudcommerceconsumerprocurement': {
        'v1alpha1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.cloudcommerceconsumerprocurement.v1alpha1',
                    client_classpath='cloudcommerceconsumerprocurement_v1alpha1_client.CloudcommerceconsumerprocurementV1alpha1',
                    base_url='https://cloudcommerceconsumerprocurement.googleapis.com/',
                    messages_modulepath='cloudcommerceconsumerprocurement_v1alpha1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'clouddebugger': {
        'v2':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.clouddebugger.v2',
                    client_classpath='clouddebugger_v2_client.ClouddebuggerV2',
                    base_url='https://clouddebugger.googleapis.com/',
                    messages_modulepath='clouddebugger_v2_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'clouddeploy': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.clouddeploy.v1',
                    client_classpath='clouddeploy_v1_client.ClouddeployV1',
                    base_url='https://clouddeploy.googleapis.com/',
                    messages_modulepath='clouddeploy_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'clouderrorreporting': {
        'v1beta1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.clouderrorreporting.v1beta1',
                    client_classpath='clouderrorreporting_v1beta1_client.ClouderrorreportingV1beta1',
                    base_url='https://clouderrorreporting.googleapis.com/',
                    messages_modulepath='clouderrorreporting_v1beta1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'cloudfunctions': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.cloudfunctions.v1',
                    client_classpath='cloudfunctions_v1_client.CloudfunctionsV1',
                    base_url='https://cloudfunctions.googleapis.com/',
                    messages_modulepath='cloudfunctions_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v2':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.cloudfunctions.v2',
                    client_classpath='cloudfunctions_v2_client.CloudfunctionsV2',
                    base_url='https://cloudfunctions.googleapis.com/',
                    messages_modulepath='cloudfunctions_v2_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v2alpha':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.cloudfunctions.v2alpha',
                    client_classpath='cloudfunctions_v2alpha_client.CloudfunctionsV2alpha',
                    base_url='https://cloudfunctions.googleapis.com/',
                    messages_modulepath='cloudfunctions_v2alpha_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v2beta':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.cloudfunctions.v2beta',
                    client_classpath='cloudfunctions_v2beta_client.CloudfunctionsV2beta',
                    base_url='https://cloudfunctions.googleapis.com/',
                    messages_modulepath='cloudfunctions_v2beta_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'cloudidentity': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.cloudidentity.v1',
                    client_classpath='cloudidentity_v1_client.CloudidentityV1',
                    base_url='https://cloudidentity.googleapis.com/',
                    messages_modulepath='cloudidentity_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.cloudidentity.v1alpha1',
                    client_classpath='cloudidentity_v1alpha1_client.CloudidentityV1alpha1',
                    base_url='https://cloudidentity.googleapis.com/',
                    messages_modulepath='cloudidentity_v1alpha1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.cloudidentity.v1beta1',
                    client_classpath='cloudidentity_v1beta1_client.CloudidentityV1beta1',
                    base_url='https://cloudidentity.googleapis.com/',
                    messages_modulepath='cloudidentity_v1beta1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'cloudkms': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.cloudkms.v1',
                    client_classpath='cloudkms_v1_client.CloudkmsV1',
                    base_url='https://cloudkms.googleapis.com/',
                    messages_modulepath='cloudkms_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'europe-west3': 'https://cloudkms.europe-west3.rep.googleapis.com/', 'europe-west8': 'https://cloudkms.europe-west8.rep.googleapis.com/', 'europe-west9': 'https://cloudkms.europe-west9.rep.googleapis.com/', 'me-central2': 'https://cloudkms.me-central2.rep.googleapis.com/', 'us-east1': 'https://cloudkms.us-east1.rep.googleapis.com/', 'us-east4': 'https://cloudkms.us-east4.rep.googleapis.com/', 'us-west2': 'https://cloudkms.us-west2.rep.googleapis.com/', 'us-west1': 'https://cloudkms.us-west1.rep.googleapis.com/', 'us-east7': 'https://cloudkms.us-east7.rep.googleapis.com/', 'us-central1': 'https://cloudkms.us-central1.rep.googleapis.com/', 'us-west3': 'https://cloudkms.us-west3.rep.googleapis.com/', 'us-central2': 'https://cloudkms.us-central2.rep.googleapis.com/', 'us-west4': 'https://cloudkms.us-west4.rep.googleapis.com/', 'us-west8': 'https://cloudkms.us-west8.rep.googleapis.com/', 'us-east5': 'https://cloudkms.us-east5.rep.googleapis.com/', 'us-south1': 'https://cloudkms.us-south1.rep.googleapis.com/', 'us': 'https://cloudkms.us.rep.googleapis.com/', 'in': 'https://cloudkms.in.rep.googleapis.com/'}),
    },
    'cloudlocationfinder': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.cloudlocationfinder.v1',
                    client_classpath='cloudlocationfinder_v1_client.CloudlocationfinderV1',
                    base_url='https://cloudlocationfinder.googleapis.com/',
                    messages_modulepath='cloudlocationfinder_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.cloudlocationfinder.v1alpha',
                    client_classpath='cloudlocationfinder_v1alpha_client.CloudlocationfinderV1alpha',
                    base_url='https://cloudlocationfinder.googleapis.com/',
                    messages_modulepath='cloudlocationfinder_v1alpha_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'cloudnumberregistry': {
        'v1alpha':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.cloudnumberregistry.v1alpha',
                    client_classpath='cloudnumberregistry_v1alpha_client.CloudnumberregistryV1alpha',
                    base_url='https://cloudnumberregistry.googleapis.com/',
                    messages_modulepath='cloudnumberregistry_v1alpha_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'cloudquotas': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.cloudquotas.v1',
                    client_classpath='cloudquotas_v1_client.CloudquotasV1',
                    base_url='https://cloudquotas.googleapis.com/',
                    messages_modulepath='cloudquotas_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.cloudquotas.v1alpha',
                    client_classpath='cloudquotas_v1alpha_client.CloudquotasV1alpha',
                    base_url='https://cloudquotas.googleapis.com/',
                    messages_modulepath='cloudquotas_v1alpha_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.cloudquotas.v1beta',
                    client_classpath='cloudquotas_v1beta_client.CloudquotasV1beta',
                    base_url='https://cloudquotas.googleapis.com/',
                    messages_modulepath='cloudquotas_v1beta_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'cloudresourcemanager': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.cloudresourcemanager.v1',
                    client_classpath='cloudresourcemanager_v1_client.CloudresourcemanagerV1',
                    base_url='https://cloudresourcemanager.googleapis.com/',
                    messages_modulepath='cloudresourcemanager_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.cloudresourcemanager.v1beta1',
                    client_classpath='cloudresourcemanager_v1beta1_client.CloudresourcemanagerV1beta1',
                    base_url='https://cloudresourcemanager.googleapis.com/',
                    messages_modulepath='cloudresourcemanager_v1beta1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v2':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.cloudresourcemanager.v2',
                    client_classpath='cloudresourcemanager_v2_client.CloudresourcemanagerV2',
                    base_url='https://cloudresourcemanager.googleapis.com/',
                    messages_modulepath='cloudresourcemanager_v2_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='https://cloudresourcemanager.mtls.googleapis.com/',
                regional_endpoints={}),
        'v2alpha1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.cloudresourcemanager.v2alpha1',
                    client_classpath='cloudresourcemanager_v2alpha1_client.CloudresourcemanagerV2alpha1',
                    base_url='https://cloudresourcemanager.googleapis.com/',
                    messages_modulepath='cloudresourcemanager_v2alpha1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v2beta1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.cloudresourcemanager.v2beta1',
                    client_classpath='cloudresourcemanager_v2beta1_client.CloudresourcemanagerV2beta1',
                    base_url='https://cloudresourcemanager.googleapis.com/',
                    messages_modulepath='cloudresourcemanager_v2beta1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='https://cloudresourcemanager.mtls.googleapis.com/',
                regional_endpoints={}),
        'v3':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.cloudresourcemanager.v3',
                    client_classpath='cloudresourcemanager_v3_client.CloudresourcemanagerV3',
                    base_url='https://cloudresourcemanager.googleapis.com/',
                    messages_modulepath='cloudresourcemanager_v3_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'cloudscheduler': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.cloudscheduler.v1',
                    client_classpath='cloudscheduler_v1_client.CloudschedulerV1',
                    base_url='https://cloudscheduler.googleapis.com/',
                    messages_modulepath='cloudscheduler_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='https://cloudscheduler.mtls.googleapis.com/',
                regional_endpoints={}),
        'v1alpha1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.cloudscheduler.v1alpha1',
                    client_classpath='cloudscheduler_v1alpha1_client.CloudschedulerV1alpha1',
                    base_url='https://cloudscheduler.googleapis.com/',
                    messages_modulepath='cloudscheduler_v1alpha1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.cloudscheduler.v1beta1',
                    client_classpath='cloudscheduler_v1beta1_client.CloudschedulerV1beta1',
                    base_url='https://cloudscheduler.googleapis.com/',
                    messages_modulepath='cloudscheduler_v1beta1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'cloudsecuritycompliance': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.cloudsecuritycompliance.v1',
                    client_classpath='cloudsecuritycompliance_v1_client.CloudsecuritycomplianceV1',
                    base_url='https://cloudsecuritycompliance.googleapis.com/',
                    messages_modulepath='cloudsecuritycompliance_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'cloudshell': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.cloudshell.v1',
                    client_classpath='cloudshell_v1_client.CloudshellV1',
                    base_url='https://cloudshell.googleapis.com/',
                    messages_modulepath='cloudshell_v1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.cloudshell.v1alpha1',
                    client_classpath='cloudshell_v1alpha1_client.CloudshellV1alpha1',
                    base_url='https://cloudshell.googleapis.com/',
                    messages_modulepath='cloudshell_v1alpha1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'cloudtasks': {
        'v2':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.cloudtasks.v2',
                    client_classpath='cloudtasks_v2_client.CloudtasksV2',
                    base_url='https://cloudtasks.googleapis.com/',
                    messages_modulepath='cloudtasks_v2_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v2beta2':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.cloudtasks.v2beta2',
                    client_classpath='cloudtasks_v2beta2_client.CloudtasksV2beta2',
                    base_url='https://cloudtasks.googleapis.com/',
                    messages_modulepath='cloudtasks_v2beta2_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v2beta3':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.cloudtasks.v2beta3',
                    client_classpath='cloudtasks_v2beta3_client.CloudtasksV2beta3',
                    base_url='https://cloudtasks.googleapis.com/',
                    messages_modulepath='cloudtasks_v2beta3_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'cloudtrace': {
        'v2beta1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.cloudtrace.v2beta1',
                    client_classpath='cloudtrace_v2beta1_client.CloudtraceV2beta1',
                    base_url='https://cloudtrace.googleapis.com/',
                    messages_modulepath='cloudtrace_v2beta1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'composer': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.composer.v1',
                    client_classpath='composer_v1_client.ComposerV1',
                    base_url='https://composer.googleapis.com/',
                    messages_modulepath='composer_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'africa-south1': 'https://composer.africa-south1.rep.googleapis.com/', 'asia-east1': 'https://composer.asia-east1.rep.googleapis.com/', 'asia-east2': 'https://composer.asia-east2.rep.googleapis.com/', 'asia-northeast1': 'https://composer.asia-northeast1.rep.googleapis.com/', 'asia-northeast2': 'https://composer.asia-northeast2.rep.googleapis.com/', 'asia-northeast3': 'https://composer.asia-northeast3.rep.googleapis.com/', 'asia-south1': 'https://composer.asia-south1.rep.googleapis.com/', 'asia-south2': 'https://composer.asia-south2.rep.googleapis.com/', 'asia-southeast1': 'https://composer.asia-southeast1.rep.googleapis.com/', 'asia-southeast2': 'https://composer.asia-southeast2.rep.googleapis.com/', 'asia-southeast3': 'https://composer.asia-southeast3.rep.googleapis.com/', 'australia-southeast1': 'https://composer.australia-southeast1.rep.googleapis.com/', 'australia-southeast2': 'https://composer.australia-southeast2.rep.googleapis.com/', 'europe-central2': 'https://composer.europe-central2.rep.googleapis.com/', 'europe-north1': 'https://composer.europe-north1.rep.googleapis.com/', 'europe-north2': 'https://composer.europe-north2.rep.googleapis.com/', 'europe-southwest1': 'https://composer.europe-southwest1.rep.googleapis.com/', 'europe-west1': 'https://composer.europe-west1.rep.googleapis.com/', 'europe-west10': 'https://composer.europe-west10.rep.googleapis.com/', 'europe-west12': 'https://composer.europe-west12.rep.googleapis.com/', 'europe-west15': 'https://composer.europe-west15.rep.googleapis.com/', 'europe-west2': 'https://composer.europe-west2.rep.googleapis.com/', 'europe-west3': 'https://composer.europe-west3.rep.googleapis.com/', 'europe-west4': 'https://composer.europe-west4.rep.googleapis.com/', 'europe-west6': 'https://composer.europe-west6.rep.googleapis.com/', 'europe-west8': 'https://composer.europe-west8.rep.googleapis.com/', 'europe-west9': 'https://composer.europe-west9.rep.googleapis.com/', 'me-central1': 'https://composer.me-central1.rep.googleapis.com/', 'me-central2': 'https://composer.me-central2.rep.googleapis.com/', 'me-west1': 'https://composer.me-west1.rep.googleapis.com/', 'northamerica-northeast1': 'https://composer.northamerica-northeast1.rep.googleapis.com/', 'northamerica-northeast2': 'https://composer.northamerica-northeast2.rep.googleapis.com/', 'northamerica-south1': 'https://composer.northamerica-south1.rep.googleapis.com/', 'southamerica-east1': 'https://composer.southamerica-east1.rep.googleapis.com/', 'southamerica-west1': 'https://composer.southamerica-west1.rep.googleapis.com/', 'us-central1': 'https://composer.us-central1.rep.googleapis.com/', 'us-east1': 'https://composer.us-east1.rep.googleapis.com/', 'us-east4': 'https://composer.us-east4.rep.googleapis.com/', 'us-east5': 'https://composer.us-east5.rep.googleapis.com/', 'us-east7': 'https://composer.us-east7.rep.googleapis.com/', 'us-south1': 'https://composer.us-south1.rep.googleapis.com/', 'us-west1': 'https://composer.us-west1.rep.googleapis.com/', 'us-west2': 'https://composer.us-west2.rep.googleapis.com/', 'us-west3': 'https://composer.us-west3.rep.googleapis.com/', 'us-west4': 'https://composer.us-west4.rep.googleapis.com/', 'us-west8': 'https://composer.us-west8.rep.googleapis.com/'}),
        'v1alpha2':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.composer.v1alpha2',
                    client_classpath='composer_v1alpha2_client.ComposerV1alpha2',
                    base_url='https://composer.googleapis.com/',
                    messages_modulepath='composer_v1alpha2_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'africa-south1': 'https://composer.africa-south1.rep.googleapis.com/', 'asia-east1': 'https://composer.asia-east1.rep.googleapis.com/', 'asia-east2': 'https://composer.asia-east2.rep.googleapis.com/', 'asia-northeast1': 'https://composer.asia-northeast1.rep.googleapis.com/', 'asia-northeast2': 'https://composer.asia-northeast2.rep.googleapis.com/', 'asia-northeast3': 'https://composer.asia-northeast3.rep.googleapis.com/', 'asia-south1': 'https://composer.asia-south1.rep.googleapis.com/', 'asia-south2': 'https://composer.asia-south2.rep.googleapis.com/', 'asia-southeast1': 'https://composer.asia-southeast1.rep.googleapis.com/', 'asia-southeast2': 'https://composer.asia-southeast2.rep.googleapis.com/', 'asia-southeast3': 'https://composer.asia-southeast3.rep.googleapis.com/', 'australia-southeast1': 'https://composer.australia-southeast1.rep.googleapis.com/', 'australia-southeast2': 'https://composer.australia-southeast2.rep.googleapis.com/', 'europe-central2': 'https://composer.europe-central2.rep.googleapis.com/', 'europe-north1': 'https://composer.europe-north1.rep.googleapis.com/', 'europe-north2': 'https://composer.europe-north2.rep.googleapis.com/', 'europe-southwest1': 'https://composer.europe-southwest1.rep.googleapis.com/', 'europe-west1': 'https://composer.europe-west1.rep.googleapis.com/', 'europe-west10': 'https://composer.europe-west10.rep.googleapis.com/', 'europe-west12': 'https://composer.europe-west12.rep.googleapis.com/', 'europe-west15': 'https://composer.europe-west15.rep.googleapis.com/', 'europe-west2': 'https://composer.europe-west2.rep.googleapis.com/', 'europe-west3': 'https://composer.europe-west3.rep.googleapis.com/', 'europe-west4': 'https://composer.europe-west4.rep.googleapis.com/', 'europe-west6': 'https://composer.europe-west6.rep.googleapis.com/', 'europe-west8': 'https://composer.europe-west8.rep.googleapis.com/', 'europe-west9': 'https://composer.europe-west9.rep.googleapis.com/', 'me-central1': 'https://composer.me-central1.rep.googleapis.com/', 'me-central2': 'https://composer.me-central2.rep.googleapis.com/', 'me-west1': 'https://composer.me-west1.rep.googleapis.com/', 'northamerica-northeast1': 'https://composer.northamerica-northeast1.rep.googleapis.com/', 'northamerica-northeast2': 'https://composer.northamerica-northeast2.rep.googleapis.com/', 'northamerica-south1': 'https://composer.northamerica-south1.rep.googleapis.com/', 'southamerica-east1': 'https://composer.southamerica-east1.rep.googleapis.com/', 'southamerica-west1': 'https://composer.southamerica-west1.rep.googleapis.com/', 'us-central1': 'https://composer.us-central1.rep.googleapis.com/', 'us-east1': 'https://composer.us-east1.rep.googleapis.com/', 'us-east4': 'https://composer.us-east4.rep.googleapis.com/', 'us-east5': 'https://composer.us-east5.rep.googleapis.com/', 'us-east7': 'https://composer.us-east7.rep.googleapis.com/', 'us-south1': 'https://composer.us-south1.rep.googleapis.com/', 'us-west1': 'https://composer.us-west1.rep.googleapis.com/', 'us-west2': 'https://composer.us-west2.rep.googleapis.com/', 'us-west3': 'https://composer.us-west3.rep.googleapis.com/', 'us-west4': 'https://composer.us-west4.rep.googleapis.com/', 'us-west8': 'https://composer.us-west8.rep.googleapis.com/'}),
        'v1beta1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.composer.v1beta1',
                    client_classpath='composer_v1beta1_client.ComposerV1beta1',
                    base_url='https://composer.googleapis.com/',
                    messages_modulepath='composer_v1beta1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'africa-south1': 'https://composer.africa-south1.rep.googleapis.com/', 'asia-east1': 'https://composer.asia-east1.rep.googleapis.com/', 'asia-east2': 'https://composer.asia-east2.rep.googleapis.com/', 'asia-northeast1': 'https://composer.asia-northeast1.rep.googleapis.com/', 'asia-northeast2': 'https://composer.asia-northeast2.rep.googleapis.com/', 'asia-northeast3': 'https://composer.asia-northeast3.rep.googleapis.com/', 'asia-south1': 'https://composer.asia-south1.rep.googleapis.com/', 'asia-south2': 'https://composer.asia-south2.rep.googleapis.com/', 'asia-southeast1': 'https://composer.asia-southeast1.rep.googleapis.com/', 'asia-southeast2': 'https://composer.asia-southeast2.rep.googleapis.com/', 'asia-southeast3': 'https://composer.asia-southeast3.rep.googleapis.com/', 'australia-southeast1': 'https://composer.australia-southeast1.rep.googleapis.com/', 'australia-southeast2': 'https://composer.australia-southeast2.rep.googleapis.com/', 'europe-central2': 'https://composer.europe-central2.rep.googleapis.com/', 'europe-north1': 'https://composer.europe-north1.rep.googleapis.com/', 'europe-north2': 'https://composer.europe-north2.rep.googleapis.com/', 'europe-southwest1': 'https://composer.europe-southwest1.rep.googleapis.com/', 'europe-west1': 'https://composer.europe-west1.rep.googleapis.com/', 'europe-west10': 'https://composer.europe-west10.rep.googleapis.com/', 'europe-west12': 'https://composer.europe-west12.rep.googleapis.com/', 'europe-west15': 'https://composer.europe-west15.rep.googleapis.com/', 'europe-west2': 'https://composer.europe-west2.rep.googleapis.com/', 'europe-west3': 'https://composer.europe-west3.rep.googleapis.com/', 'europe-west4': 'https://composer.europe-west4.rep.googleapis.com/', 'europe-west6': 'https://composer.europe-west6.rep.googleapis.com/', 'europe-west8': 'https://composer.europe-west8.rep.googleapis.com/', 'europe-west9': 'https://composer.europe-west9.rep.googleapis.com/', 'me-central1': 'https://composer.me-central1.rep.googleapis.com/', 'me-central2': 'https://composer.me-central2.rep.googleapis.com/', 'me-west1': 'https://composer.me-west1.rep.googleapis.com/', 'northamerica-northeast1': 'https://composer.northamerica-northeast1.rep.googleapis.com/', 'northamerica-northeast2': 'https://composer.northamerica-northeast2.rep.googleapis.com/', 'northamerica-south1': 'https://composer.northamerica-south1.rep.googleapis.com/', 'southamerica-east1': 'https://composer.southamerica-east1.rep.googleapis.com/', 'southamerica-west1': 'https://composer.southamerica-west1.rep.googleapis.com/', 'us-central1': 'https://composer.us-central1.rep.googleapis.com/', 'us-east1': 'https://composer.us-east1.rep.googleapis.com/', 'us-east4': 'https://composer.us-east4.rep.googleapis.com/', 'us-east5': 'https://composer.us-east5.rep.googleapis.com/', 'us-east7': 'https://composer.us-east7.rep.googleapis.com/', 'us-south1': 'https://composer.us-south1.rep.googleapis.com/', 'us-west1': 'https://composer.us-west1.rep.googleapis.com/', 'us-west2': 'https://composer.us-west2.rep.googleapis.com/', 'us-west3': 'https://composer.us-west3.rep.googleapis.com/', 'us-west4': 'https://composer.us-west4.rep.googleapis.com/', 'us-west8': 'https://composer.us-west8.rep.googleapis.com/'}),
    },
    'composerflex': {
        'v1alpha1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.composerflex.v1alpha1',
                    client_classpath='composerflex_v1alpha1_client.ComposerflexV1alpha1',
                    base_url='https://composerflex.googleapis.com/',
                    messages_modulepath='composerflex_v1alpha1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'compute': {
        'alpha':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.compute.alpha',
                    client_classpath='compute_alpha_client.ComputeAlpha',
                    base_url='https://compute.googleapis.com/compute/alpha/',
                    messages_modulepath='compute_alpha_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'beta':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.compute.beta',
                    client_classpath='compute_beta_client.ComputeBeta',
                    base_url='https://compute.googleapis.com/compute/beta/',
                    messages_modulepath='compute_beta_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.compute.v1',
                    client_classpath='compute_v1_client.ComputeV1',
                    base_url='https://compute.googleapis.com/compute/v1/',
                    messages_modulepath='compute_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'config': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.config.v1',
                    client_classpath='config_v1_client.ConfigV1',
                    base_url='https://config.googleapis.com/',
                    messages_modulepath='config_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha2':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.config.v1alpha2',
                    client_classpath='config_v1alpha2_client.ConfigV1alpha2',
                    base_url='https://config.googleapis.com/',
                    messages_modulepath='config_v1alpha2_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'configdelivery': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.configdelivery.v1',
                    client_classpath='configdelivery_v1_client.ConfigdeliveryV1',
                    base_url='https://configdelivery.googleapis.com/',
                    messages_modulepath='configdelivery_v1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.configdelivery.v1alpha',
                    client_classpath='configdelivery_v1alpha_client.ConfigdeliveryV1alpha',
                    base_url='https://configdelivery.googleapis.com/',
                    messages_modulepath='configdelivery_v1alpha_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.configdelivery.v1beta',
                    client_classpath='configdelivery_v1beta_client.ConfigdeliveryV1beta',
                    base_url='https://configdelivery.googleapis.com/',
                    messages_modulepath='configdelivery_v1beta_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'connectgateway': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.connectgateway.v1',
                    client_classpath='connectgateway_v1_client.ConnectgatewayV1',
                    base_url='https://connectgateway.googleapis.com/',
                    messages_modulepath='connectgateway_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.connectgateway.v1alpha1',
                    client_classpath='connectgateway_v1alpha1_client.ConnectgatewayV1alpha1',
                    base_url='https://connectgateway.googleapis.com/',
                    messages_modulepath='connectgateway_v1alpha1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.connectgateway.v1beta1',
                    client_classpath='connectgateway_v1beta1_client.ConnectgatewayV1beta1',
                    base_url='https://connectgateway.googleapis.com/',
                    messages_modulepath='connectgateway_v1beta1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'connectors': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.connectors.v1',
                    client_classpath='connectors_v1_client.ConnectorsV1',
                    base_url='https://connectors.googleapis.com/',
                    messages_modulepath='connectors_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'container': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.container.v1',
                    client_classpath='container_v1_client.ContainerV1',
                    base_url='https://container.googleapis.com/',
                    messages_modulepath='container_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.container.v1alpha1',
                    client_classpath='container_v1alpha1_client.ContainerV1alpha1',
                    base_url='https://container.googleapis.com/',
                    messages_modulepath='container_v1alpha1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.container.v1beta1',
                    client_classpath='container_v1beta1_client.ContainerV1beta1',
                    base_url='https://container.googleapis.com/',
                    messages_modulepath='container_v1beta1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'containeranalysis': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.containeranalysis.v1',
                    client_classpath='containeranalysis_v1_client.ContaineranalysisV1',
                    base_url='https://containeranalysis.googleapis.com/',
                    messages_modulepath='containeranalysis_v1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'asia-east1': 'https://containeranalysis.asia-east1.rep.googleapis.com/', 'asia-east2': 'https://containeranalysis.asia-east2.rep.googleapis.com/', 'asia-northeast1': 'https://containeranalysis.asia-northeast1.rep.googleapis.com/', 'asia-northeast3': 'https://containeranalysis.asia-northeast3.rep.googleapis.com/', 'asia-south1': 'https://containeranalysis.asia-south1.rep.googleapis.com/', 'asia-south2': 'https://containeranalysis.asia-south2.rep.googleapis.com/', 'asia-southeast1': 'https://containeranalysis.asia-southeast1.rep.googleapis.com/', 'asia-southeast2': 'https://containeranalysis.asia-southeast2.rep.googleapis.com/', 'australia-southeast1': 'https://containeranalysis.australia-southeast1.rep.googleapis.com/', 'europe-central2': 'https://containeranalysis.europe-central2.rep.googleapis.com/', 'europe-north1': 'https://containeranalysis.europe-north1.rep.googleapis.com/', 'europe-southwest1': 'https://containeranalysis.europe-southwest1.rep.googleapis.com/', 'europe-west1': 'https://containeranalysis.europe-west1.rep.googleapis.com/', 'europe-west2': 'https://containeranalysis.europe-west2.rep.googleapis.com/', 'europe-west3': 'https://containeranalysis.europe-west3.rep.googleapis.com/', 'europe-west4': 'https://containeranalysis.europe-west4.rep.googleapis.com/', 'europe-west6': 'https://containeranalysis.europe-west6.rep.googleapis.com/', 'europe-west8': 'https://containeranalysis.europe-west8.rep.googleapis.com/', 'europe-west9': 'https://containeranalysis.europe-west9.rep.googleapis.com/', 'me-central1': 'https://containeranalysis.me-central1.rep.googleapis.com/', 'me-central2': 'https://containeranalysis.me-central2.rep.googleapis.com/', 'me-west1': 'https://containeranalysis.me-west1.rep.googleapis.com/', 'northamerica-northeast1': 'https://containeranalysis.northamerica-northeast1.rep.googleapis.com/', 'northamerica-northeast2': 'https://containeranalysis.northamerica-northeast2.rep.googleapis.com/', 'southamerica-east1': 'https://containeranalysis.southamerica-east1.rep.googleapis.com/', 'southamerica-west1': 'https://containeranalysis.southamerica-west1.rep.googleapis.com/', 'us-central1': 'https://containeranalysis.us-central1.rep.googleapis.com/', 'us-central2': 'https://containeranalysis.us-central2.rep.googleapis.com/', 'us-east1': 'https://containeranalysis.us-east1.rep.googleapis.com/', 'us-east4': 'https://containeranalysis.us-east4.rep.googleapis.com/', 'us-east5': 'https://containeranalysis.us-east5.rep.googleapis.com/', 'us-east7': 'https://containeranalysis.us-east7.rep.googleapis.com/', 'us-south1': 'https://containeranalysis.us-south1.rep.googleapis.com/', 'us-west1': 'https://containeranalysis.us-west1.rep.googleapis.com/', 'us-west2': 'https://containeranalysis.us-west2.rep.googleapis.com/', 'us-west3': 'https://containeranalysis.us-west3.rep.googleapis.com/', 'us-west4': 'https://containeranalysis.us-west4.rep.googleapis.com/', 'africa-south1': 'https://containeranalysis.africa-south1.rep.googleapis.com/', 'asia-northeast2': 'https://containeranalysis.asia-northeast2.rep.googleapis.com/', 'australia-southeast2': 'https://containeranalysis.australia-southeast2.rep.googleapis.com/', 'europe-west10': 'https://containeranalysis.europe-west10.rep.googleapis.com/', 'us': 'https://containeranalysis.us.rep.googleapis.com/', 'eu': 'https://containeranalysis.eu.rep.googleapis.com/'}),
        'v1alpha1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.containeranalysis.v1alpha1',
                    client_classpath='containeranalysis_v1alpha1_client.ContaineranalysisV1alpha1',
                    base_url='https://containeranalysis.googleapis.com/',
                    messages_modulepath='containeranalysis_v1alpha1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'asia-east1': 'https://containeranalysis.asia-east1.rep.googleapis.com/', 'asia-east2': 'https://containeranalysis.asia-east2.rep.googleapis.com/', 'asia-northeast1': 'https://containeranalysis.asia-northeast1.rep.googleapis.com/', 'asia-northeast3': 'https://containeranalysis.asia-northeast3.rep.googleapis.com/', 'asia-south1': 'https://containeranalysis.asia-south1.rep.googleapis.com/', 'asia-south2': 'https://containeranalysis.asia-south2.rep.googleapis.com/', 'asia-southeast1': 'https://containeranalysis.asia-southeast1.rep.googleapis.com/', 'asia-southeast2': 'https://containeranalysis.asia-southeast2.rep.googleapis.com/', 'australia-southeast1': 'https://containeranalysis.australia-southeast1.rep.googleapis.com/', 'europe-central2': 'https://containeranalysis.europe-central2.rep.googleapis.com/', 'europe-north1': 'https://containeranalysis.europe-north1.rep.googleapis.com/', 'europe-southwest1': 'https://containeranalysis.europe-southwest1.rep.googleapis.com/', 'europe-west1': 'https://containeranalysis.europe-west1.rep.googleapis.com/', 'europe-west2': 'https://containeranalysis.europe-west2.rep.googleapis.com/', 'europe-west3': 'https://containeranalysis.europe-west3.rep.googleapis.com/', 'europe-west4': 'https://containeranalysis.europe-west4.rep.googleapis.com/', 'europe-west6': 'https://containeranalysis.europe-west6.rep.googleapis.com/', 'europe-west8': 'https://containeranalysis.europe-west8.rep.googleapis.com/', 'europe-west9': 'https://containeranalysis.europe-west9.rep.googleapis.com/', 'me-central1': 'https://containeranalysis.me-central1.rep.googleapis.com/', 'me-central2': 'https://containeranalysis.me-central2.rep.googleapis.com/', 'me-west1': 'https://containeranalysis.me-west1.rep.googleapis.com/', 'northamerica-northeast1': 'https://containeranalysis.northamerica-northeast1.rep.googleapis.com/', 'northamerica-northeast2': 'https://containeranalysis.northamerica-northeast2.rep.googleapis.com/', 'southamerica-east1': 'https://containeranalysis.southamerica-east1.rep.googleapis.com/', 'southamerica-west1': 'https://containeranalysis.southamerica-west1.rep.googleapis.com/', 'us-central1': 'https://containeranalysis.us-central1.rep.googleapis.com/', 'us-central2': 'https://containeranalysis.us-central2.rep.googleapis.com/', 'us-east1': 'https://containeranalysis.us-east1.rep.googleapis.com/', 'us-east4': 'https://containeranalysis.us-east4.rep.googleapis.com/', 'us-east5': 'https://containeranalysis.us-east5.rep.googleapis.com/', 'us-east7': 'https://containeranalysis.us-east7.rep.googleapis.com/', 'us-south1': 'https://containeranalysis.us-south1.rep.googleapis.com/', 'us-west1': 'https://containeranalysis.us-west1.rep.googleapis.com/', 'us-west2': 'https://containeranalysis.us-west2.rep.googleapis.com/', 'us-west3': 'https://containeranalysis.us-west3.rep.googleapis.com/', 'us-west4': 'https://containeranalysis.us-west4.rep.googleapis.com/', 'africa-south1': 'https://containeranalysis.africa-south1.rep.googleapis.com/', 'asia-northeast2': 'https://containeranalysis.asia-northeast2.rep.googleapis.com/', 'australia-southeast2': 'https://containeranalysis.australia-southeast2.rep.googleapis.com/', 'europe-west10': 'https://containeranalysis.europe-west10.rep.googleapis.com/', 'us': 'https://containeranalysis.us.rep.googleapis.com/', 'eu': 'https://containeranalysis.eu.rep.googleapis.com/'}),
        'v1beta1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.containeranalysis.v1beta1',
                    client_classpath='containeranalysis_v1beta1_client.ContaineranalysisV1beta1',
                    base_url='https://containeranalysis.googleapis.com/',
                    messages_modulepath='containeranalysis_v1beta1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'asia-east1': 'https://containeranalysis.asia-east1.rep.googleapis.com/', 'asia-east2': 'https://containeranalysis.asia-east2.rep.googleapis.com/', 'asia-northeast1': 'https://containeranalysis.asia-northeast1.rep.googleapis.com/', 'asia-northeast3': 'https://containeranalysis.asia-northeast3.rep.googleapis.com/', 'asia-south1': 'https://containeranalysis.asia-south1.rep.googleapis.com/', 'asia-south2': 'https://containeranalysis.asia-south2.rep.googleapis.com/', 'asia-southeast1': 'https://containeranalysis.asia-southeast1.rep.googleapis.com/', 'asia-southeast2': 'https://containeranalysis.asia-southeast2.rep.googleapis.com/', 'australia-southeast1': 'https://containeranalysis.australia-southeast1.rep.googleapis.com/', 'europe-central2': 'https://containeranalysis.europe-central2.rep.googleapis.com/', 'europe-north1': 'https://containeranalysis.europe-north1.rep.googleapis.com/', 'europe-southwest1': 'https://containeranalysis.europe-southwest1.rep.googleapis.com/', 'europe-west1': 'https://containeranalysis.europe-west1.rep.googleapis.com/', 'europe-west2': 'https://containeranalysis.europe-west2.rep.googleapis.com/', 'europe-west3': 'https://containeranalysis.europe-west3.rep.googleapis.com/', 'europe-west4': 'https://containeranalysis.europe-west4.rep.googleapis.com/', 'europe-west6': 'https://containeranalysis.europe-west6.rep.googleapis.com/', 'europe-west8': 'https://containeranalysis.europe-west8.rep.googleapis.com/', 'europe-west9': 'https://containeranalysis.europe-west9.rep.googleapis.com/', 'me-central1': 'https://containeranalysis.me-central1.rep.googleapis.com/', 'me-central2': 'https://containeranalysis.me-central2.rep.googleapis.com/', 'me-west1': 'https://containeranalysis.me-west1.rep.googleapis.com/', 'northamerica-northeast1': 'https://containeranalysis.northamerica-northeast1.rep.googleapis.com/', 'northamerica-northeast2': 'https://containeranalysis.northamerica-northeast2.rep.googleapis.com/', 'southamerica-east1': 'https://containeranalysis.southamerica-east1.rep.googleapis.com/', 'southamerica-west1': 'https://containeranalysis.southamerica-west1.rep.googleapis.com/', 'us-central1': 'https://containeranalysis.us-central1.rep.googleapis.com/', 'us-central2': 'https://containeranalysis.us-central2.rep.googleapis.com/', 'us-east1': 'https://containeranalysis.us-east1.rep.googleapis.com/', 'us-east4': 'https://containeranalysis.us-east4.rep.googleapis.com/', 'us-east5': 'https://containeranalysis.us-east5.rep.googleapis.com/', 'us-east7': 'https://containeranalysis.us-east7.rep.googleapis.com/', 'us-south1': 'https://containeranalysis.us-south1.rep.googleapis.com/', 'us-west1': 'https://containeranalysis.us-west1.rep.googleapis.com/', 'us-west2': 'https://containeranalysis.us-west2.rep.googleapis.com/', 'us-west3': 'https://containeranalysis.us-west3.rep.googleapis.com/', 'us-west4': 'https://containeranalysis.us-west4.rep.googleapis.com/', 'africa-south1': 'https://containeranalysis.africa-south1.rep.googleapis.com/', 'asia-northeast2': 'https://containeranalysis.asia-northeast2.rep.googleapis.com/', 'australia-southeast2': 'https://containeranalysis.australia-southeast2.rep.googleapis.com/', 'europe-west10': 'https://containeranalysis.europe-west10.rep.googleapis.com/', 'us': 'https://containeranalysis.us.rep.googleapis.com/', 'eu': 'https://containeranalysis.eu.rep.googleapis.com/'}),
    },
    'datacatalog': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.datacatalog.v1',
                    client_classpath='datacatalog_v1_client.DatacatalogV1',
                    base_url='https://datacatalog.googleapis.com/',
                    messages_modulepath='datacatalog_v1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha3':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.datacatalog.v1alpha3',
                    client_classpath='datacatalog_v1alpha3_client.DatacatalogV1alpha3',
                    base_url='https://datacatalog.googleapis.com/',
                    messages_modulepath='datacatalog_v1alpha3_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.datacatalog.v1beta1',
                    client_classpath='datacatalog_v1beta1_client.DatacatalogV1beta1',
                    base_url='https://datacatalog.googleapis.com/',
                    messages_modulepath='datacatalog_v1beta1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'dataflow': {
        'v1b3':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.dataflow.v1b3',
                    client_classpath='dataflow_v1b3_client.DataflowV1b3',
                    base_url='https://dataflow.googleapis.com/',
                    messages_modulepath='dataflow_v1b3_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'africa-south1': 'https://dataflow.africa-south1.rep.googleapis.com/', 'asia-east1': 'https://dataflow.asia-east1.rep.googleapis.com/', 'asia-east2': 'https://dataflow.asia-east2.rep.googleapis.com/', 'asia-northeast1': 'https://dataflow.asia-northeast1.rep.googleapis.com/', 'asia-northeast2': 'https://dataflow.asia-northeast2.rep.googleapis.com/', 'asia-northeast3': 'https://dataflow.asia-northeast3.rep.googleapis.com/', 'asia-south1': 'https://dataflow.asia-south1.rep.googleapis.com/', 'asia-south2': 'https://dataflow.asia-south2.rep.googleapis.com/', 'asia-southeast1': 'https://dataflow.asia-southeast1.rep.googleapis.com/', 'asia-southeast2': 'https://dataflow.asia-southeast2.rep.googleapis.com/', 'asia-southeast3': 'https://dataflow.asia-southeast3.rep.googleapis.com/', 'australia-southeast1': 'https://dataflow.australia-southeast1.rep.googleapis.com/', 'australia-southeast2': 'https://dataflow.australia-southeast2.rep.googleapis.com/', 'europe-central2': 'https://dataflow.europe-central2.rep.googleapis.com/', 'europe-southwest1': 'https://dataflow.europe-southwest1.rep.googleapis.com/', 'europe-west1': 'https://dataflow.europe-west1.rep.googleapis.com/', 'europe-west10': 'https://dataflow.europe-west10.rep.googleapis.com/', 'europe-west12': 'https://dataflow.europe-west12.rep.googleapis.com/', 'europe-west3': 'https://dataflow.europe-west3.rep.googleapis.com/', 'europe-west6': 'https://dataflow.europe-west6.rep.googleapis.com/', 'europe-west8': 'https://dataflow.europe-west8.rep.googleapis.com/', 'europe-west9': 'https://dataflow.europe-west9.rep.googleapis.com/', 'me-central1': 'https://dataflow.me-central1.rep.googleapis.com/', 'me-central2': 'https://dataflow.me-central2.rep.googleapis.com/', 'me-west1': 'https://dataflow.me-west1.rep.googleapis.com/', 'northamerica-northeast1': 'https://dataflow.northamerica-northeast1.rep.googleapis.com/', 'northamerica-northeast2': 'https://dataflow.northamerica-northeast2.rep.googleapis.com/', 'northamerica-south1': 'https://dataflow.northamerica-south1.rep.googleapis.com/', 'southamerica-west1': 'https://dataflow.southamerica-west1.rep.googleapis.com/', 'us-central1': 'https://dataflow.us-central1.rep.googleapis.com/', 'us-central2': 'https://dataflow.us-central2.rep.googleapis.com/', 'us-east1': 'https://dataflow.us-east1.rep.googleapis.com/', 'us-east4': 'https://dataflow.us-east4.rep.googleapis.com/', 'us-east5': 'https://dataflow.us-east5.rep.googleapis.com/', 'us-east7': 'https://dataflow.us-east7.rep.googleapis.com/', 'us-south1': 'https://dataflow.us-south1.rep.googleapis.com/', 'us-west1': 'https://dataflow.us-west1.rep.googleapis.com/', 'us-west2': 'https://dataflow.us-west2.rep.googleapis.com/', 'us-west3': 'https://dataflow.us-west3.rep.googleapis.com/', 'us-west4': 'https://dataflow.us-west4.rep.googleapis.com/', 'us-west8': 'https://dataflow.us-west8.rep.googleapis.com/'}),
    },
    'dataform': {
        'v1beta1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.dataform.v1beta1',
                    client_classpath='dataform_v1beta1_client.DataformV1beta1',
                    base_url='https://dataform.googleapis.com/',
                    messages_modulepath='dataform_v1beta1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'datafusion': {
        'v1beta1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.datafusion.v1beta1',
                    client_classpath='datafusion_v1beta1_client.DatafusionV1beta1',
                    base_url='https://datafusion.googleapis.com/',
                    messages_modulepath='datafusion_v1beta1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'africa-south1': 'https://datafusion.africa-south1.rep.googleapis.com/', 'asia-east1': 'https://datafusion.asia-east1.rep.googleapis.com/', 'asia-east2': 'https://datafusion.asia-east2.rep.googleapis.com/', 'asia-northeast1': 'https://datafusion.asia-northeast1.rep.googleapis.com/', 'asia-northeast2': 'https://datafusion.asia-northeast2.rep.googleapis.com/', 'asia-northeast3': 'https://datafusion.asia-northeast3.rep.googleapis.com/', 'asia-south1': 'https://datafusion.asia-south1.rep.googleapis.com/', 'asia-south2': 'https://datafusion.asia-south2.rep.googleapis.com/', 'asia-southeast1': 'https://datafusion.asia-southeast1.rep.googleapis.com/', 'asia-southeast2': 'https://datafusion.asia-southeast2.rep.googleapis.com/', 'australia-southeast1': 'https://datafusion.australia-southeast1.rep.googleapis.com/', 'australia-southeast2': 'https://datafusion.australia-southeast2.rep.googleapis.com/', 'europe-north1': 'https://datafusion.europe-north1.rep.googleapis.com/', 'europe-north2': 'https://datafusion.europe-north2.rep.googleapis.com/', 'europe-southwest1': 'https://datafusion.europe-southwest1.rep.googleapis.com/', 'europe-west1': 'https://datafusion.europe-west1.rep.googleapis.com/', 'europe-west12': 'https://datafusion.europe-west12.rep.googleapis.com/', 'europe-west2': 'https://datafusion.europe-west2.rep.googleapis.com/', 'europe-west3': 'https://datafusion.europe-west3.rep.googleapis.com/', 'europe-west4': 'https://datafusion.europe-west4.rep.googleapis.com/', 'europe-west6': 'https://datafusion.europe-west6.rep.googleapis.com/', 'europe-west8': 'https://datafusion.europe-west8.rep.googleapis.com/', 'europe-west9': 'https://datafusion.europe-west9.rep.googleapis.com/', 'me-central1': 'https://datafusion.me-central1.rep.googleapis.com/', 'me-central2': 'https://datafusion.me-central2.rep.googleapis.com/', 'me-west1': 'https://datafusion.me-west1.rep.googleapis.com/', 'northamerica-northeast1': 'https://datafusion.northamerica-northeast1.rep.googleapis.com/', 'northamerica-south1': 'https://datafusion.northamerica-south1.rep.googleapis.com/', 'southamerica-east1': 'https://datafusion.southamerica-east1.rep.googleapis.com/', 'southamerica-west1': 'https://datafusion.southamerica-west1.rep.googleapis.com/', 'us-central1': 'https://datafusion.us-central1.rep.googleapis.com/', 'us-east1': 'https://datafusion.us-east1.rep.googleapis.com/', 'us-east4': 'https://datafusion.us-east4.rep.googleapis.com/', 'us-east5': 'https://datafusion.us-east5.rep.googleapis.com/', 'us-east7': 'https://datafusion.us-east7.rep.googleapis.com/', 'us-south1': 'https://datafusion.us-south1.rep.googleapis.com/', 'us-west1': 'https://datafusion.us-west1.rep.googleapis.com/', 'us-west2': 'https://datafusion.us-west2.rep.googleapis.com/'}),
    },
    'datamigration': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.datamigration.v1',
                    client_classpath='datamigration_v1_client.DatamigrationV1',
                    base_url='https://datamigration.googleapis.com/',
                    messages_modulepath='datamigration_v1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha2':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.datamigration.v1alpha2',
                    client_classpath='datamigration_v1alpha2_client.DatamigrationV1alpha2',
                    base_url='https://datamigration.googleapis.com/',
                    messages_modulepath='datamigration_v1alpha2_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'datapipelines': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.datapipelines.v1',
                    client_classpath='datapipelines_v1_client.DatapipelinesV1',
                    base_url='https://datapipelines.googleapis.com/',
                    messages_modulepath='datapipelines_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'dataplex': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.dataplex.v1',
                    client_classpath='dataplex_v1_client.DataplexV1',
                    base_url='https://dataplex.googleapis.com/',
                    messages_modulepath='dataplex_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'africa-south1': 'https://dataplex.africa-south1.rep.googleapis.com/', 'asia-east1': 'https://dataplex.asia-east1.rep.googleapis.com/', 'asia-east2': 'https://dataplex.asia-east2.rep.googleapis.com/', 'asia-northeast1': 'https://dataplex.asia-northeast1.rep.googleapis.com/', 'asia-northeast2': 'https://dataplex.asia-northeast2.rep.googleapis.com/', 'asia-northeast3': 'https://dataplex.asia-northeast3.rep.googleapis.com/', 'asia-south1': 'https://dataplex.asia-south1.rep.googleapis.com/', 'asia-south2': 'https://dataplex.asia-south2.rep.googleapis.com/', 'asia-southeast1': 'https://dataplex.asia-southeast1.rep.googleapis.com/', 'asia-southeast2': 'https://dataplex.asia-southeast2.rep.googleapis.com/', 'asia-southeast3': 'https://dataplex.asia-southeast3.rep.googleapis.com/', 'australia-southeast1': 'https://dataplex.australia-southeast1.rep.googleapis.com/', 'australia-southeast2': 'https://dataplex.australia-southeast2.rep.googleapis.com/', 'europe-central2': 'https://dataplex.europe-central2.rep.googleapis.com/', 'europe-north1': 'https://dataplex.europe-north1.rep.googleapis.com/', 'europe-north2': 'https://dataplex.europe-north2.rep.googleapis.com/', 'europe-southwest1': 'https://dataplex.europe-southwest1.rep.googleapis.com/', 'europe-west1': 'https://dataplex.europe-west1.rep.googleapis.com/', 'europe-west10': 'https://dataplex.europe-west10.rep.googleapis.com/', 'europe-west12': 'https://dataplex.europe-west12.rep.googleapis.com/', 'europe-west15': 'https://dataplex.europe-west15.rep.googleapis.com/', 'europe-west2': 'https://dataplex.europe-west2.rep.googleapis.com/', 'europe-west3': 'https://dataplex.europe-west3.rep.googleapis.com/', 'europe-west4': 'https://dataplex.europe-west4.rep.googleapis.com/', 'europe-west6': 'https://dataplex.europe-west6.rep.googleapis.com/', 'europe-west8': 'https://dataplex.europe-west8.rep.googleapis.com/', 'europe-west9': 'https://dataplex.europe-west9.rep.googleapis.com/', 'me-central1': 'https://dataplex.me-central1.rep.googleapis.com/', 'me-central2': 'https://dataplex.me-central2.rep.googleapis.com/', 'me-west1': 'https://dataplex.me-west1.rep.googleapis.com/', 'northamerica-northeast1': 'https://dataplex.northamerica-northeast1.rep.googleapis.com/', 'northamerica-northeast2': 'https://dataplex.northamerica-northeast2.rep.googleapis.com/', 'northamerica-south1': 'https://dataplex.northamerica-south1.rep.googleapis.com/', 'southamerica-east1': 'https://dataplex.southamerica-east1.rep.googleapis.com/', 'southamerica-west1': 'https://dataplex.southamerica-west1.rep.googleapis.com/', 'us-central1': 'https://dataplex.us-central1.rep.googleapis.com/', 'us-central2': 'https://dataplex.us-central2.rep.googleapis.com/', 'us-east1': 'https://dataplex.us-east1.rep.googleapis.com/', 'us-east4': 'https://dataplex.us-east4.rep.googleapis.com/', 'us-east5': 'https://dataplex.us-east5.rep.googleapis.com/', 'us-east7': 'https://dataplex.us-east7.rep.googleapis.com/', 'us-south1': 'https://dataplex.us-south1.rep.googleapis.com/', 'us-west1': 'https://dataplex.us-west1.rep.googleapis.com/', 'us-west2': 'https://dataplex.us-west2.rep.googleapis.com/', 'us-west3': 'https://dataplex.us-west3.rep.googleapis.com/', 'us-west4': 'https://dataplex.us-west4.rep.googleapis.com/', 'us-west8': 'https://dataplex.us-west8.rep.googleapis.com/', 'us': 'https://dataplex.us.rep.googleapis.com/', 'eu': 'https://dataplex.eu.rep.googleapis.com/'}),
    },
    'dataproc': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.dataproc.v1',
                    client_classpath='dataproc_v1_client.DataprocV1',
                    base_url='https://dataproc.googleapis.com/',
                    messages_modulepath='dataproc_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'europe-west3': 'https://dataproc.europe-west3.rep.googleapis.com/', 'europe-west9': 'https://dataproc.europe-west9.rep.googleapis.com/', 'us-central1': 'https://dataproc.us-central1.rep.googleapis.com/', 'us-central2': 'https://dataproc.us-central2.rep.googleapis.com/', 'us-east1': 'https://dataproc.us-east1.rep.googleapis.com/', 'us-east4': 'https://dataproc.us-east4.rep.googleapis.com/', 'us-east5': 'https://dataproc.us-east5.rep.googleapis.com/', 'us-east7': 'https://dataproc.us-east7.rep.googleapis.com/', 'us-south1': 'https://dataproc.us-south1.rep.googleapis.com/', 'us-west1': 'https://dataproc.us-west1.rep.googleapis.com/', 'us-west2': 'https://dataproc.us-west2.rep.googleapis.com/', 'us-west3': 'https://dataproc.us-west3.rep.googleapis.com/', 'us-west8': 'https://dataproc.us-west8.rep.googleapis.com/', 'europe-west8': 'https://dataproc.europe-west8.rep.googleapis.com/', 'asia-south1': 'https://dataproc.asia-south1.rep.googleapis.com/', 'asia-south2': 'https://dataproc.asia-south2.rep.googleapis.com/', 'me-central2': 'https://dataproc.me-central2.rep.googleapis.com/'}),
    },
    'dataprocgdc': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.dataprocgdc.v1',
                    client_classpath='dataprocgdc_v1_client.DataprocgdcV1',
                    base_url='https://dataprocgdc.googleapis.com/',
                    messages_modulepath='dataprocgdc_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.dataprocgdc.v1alpha1',
                    client_classpath='dataprocgdc_v1alpha1_client.DataprocgdcV1alpha1',
                    base_url='https://dataprocgdc.googleapis.com/',
                    messages_modulepath='dataprocgdc_v1alpha1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'datastore': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.datastore.v1',
                    client_classpath='datastore_v1_client.DatastoreV1',
                    base_url='https://datastore.googleapis.com/',
                    messages_modulepath='datastore_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'africa-south1': 'https://batch-datastore.africa-south1.rep.googleapis.com/', 'asia-east1': 'https://batch-datastore.asia-east1.rep.googleapis.com/', 'asia-east2': 'https://batch-datastore.asia-east2.rep.googleapis.com/', 'asia-northeast1': 'https://batch-datastore.asia-northeast1.rep.googleapis.com/', 'asia-northeast2': 'https://batch-datastore.asia-northeast2.rep.googleapis.com/', 'asia-northeast3': 'https://batch-datastore.asia-northeast3.rep.googleapis.com/', 'asia-south1': 'https://batch-datastore.asia-south1.rep.googleapis.com/', 'asia-south2': 'https://batch-datastore.asia-south2.rep.googleapis.com/', 'asia-southeast1': 'https://batch-datastore.asia-southeast1.rep.googleapis.com/', 'asia-southeast2': 'https://batch-datastore.asia-southeast2.rep.googleapis.com/', 'australia-southeast1': 'https://batch-datastore.australia-southeast1.rep.googleapis.com/', 'australia-southeast2': 'https://batch-datastore.australia-southeast2.rep.googleapis.com/', 'europe-central2': 'https://batch-datastore.europe-central2.rep.googleapis.com/', 'europe-north1': 'https://batch-datastore.europe-north1.rep.googleapis.com/', 'europe-north2': 'https://batch-datastore.europe-north2.rep.googleapis.com/', 'europe-southwest1': 'https://batch-datastore.europe-southwest1.rep.googleapis.com/', 'europe-west1': 'https://batch-datastore.europe-west1.rep.googleapis.com/', 'europe-west10': 'https://batch-datastore.europe-west10.rep.googleapis.com/', 'europe-west12': 'https://batch-datastore.europe-west12.rep.googleapis.com/', 'europe-west2': 'https://batch-datastore.europe-west2.rep.googleapis.com/', 'europe-west3': 'https://batch-datastore.europe-west3.rep.googleapis.com/', 'europe-west4': 'https://batch-datastore.europe-west4.rep.googleapis.com/', 'europe-west6': 'https://batch-datastore.europe-west6.rep.googleapis.com/', 'europe-west8': 'https://batch-datastore.europe-west8.rep.googleapis.com/', 'europe-west9': 'https://batch-datastore.europe-west9.rep.googleapis.com/', 'me-central1': 'https://batch-datastore.me-central1.rep.googleapis.com/', 'me-central2': 'https://batch-datastore.me-central2.rep.googleapis.com/', 'me-west1': 'https://batch-datastore.me-west1.rep.googleapis.com/', 'northamerica-northeast1': 'https://batch-datastore.northamerica-northeast1.rep.googleapis.com/', 'northamerica-northeast2': 'https://batch-datastore.northamerica-northeast2.rep.googleapis.com/', 'northamerica-south1': 'https://batch-datastore.northamerica-south1.rep.googleapis.com/', 'southamerica-east1': 'https://batch-datastore.southamerica-east1.rep.googleapis.com/', 'southamerica-west1': 'https://batch-datastore.southamerica-west1.rep.googleapis.com/', 'us-central1': 'https://batch-datastore.us-central1.rep.googleapis.com/', 'us-east1': 'https://batch-datastore.us-east1.rep.googleapis.com/', 'us-east4': 'https://batch-datastore.us-east4.rep.googleapis.com/', 'us-east5': 'https://batch-datastore.us-east5.rep.googleapis.com/', 'us-south1': 'https://batch-datastore.us-south1.rep.googleapis.com/', 'us-west1': 'https://batch-datastore.us-west1.rep.googleapis.com/', 'us-west2': 'https://batch-datastore.us-west2.rep.googleapis.com/', 'us-west3': 'https://batch-datastore.us-west3.rep.googleapis.com/', 'us-west4': 'https://batch-datastore.us-west4.rep.googleapis.com/', 'eu': 'https://batch-datastore.eu.rep.googleapis.com/', 'us': 'https://batch-datastore.us.rep.googleapis.com/'}),
        'v1beta1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.datastore.v1beta1',
                    client_classpath='datastore_v1beta1_client.DatastoreV1beta1',
                    base_url='https://datastore.googleapis.com/',
                    messages_modulepath='datastore_v1beta1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'africa-south1': 'https://batch-datastore.africa-south1.rep.googleapis.com/', 'asia-east1': 'https://batch-datastore.asia-east1.rep.googleapis.com/', 'asia-east2': 'https://batch-datastore.asia-east2.rep.googleapis.com/', 'asia-northeast1': 'https://batch-datastore.asia-northeast1.rep.googleapis.com/', 'asia-northeast2': 'https://batch-datastore.asia-northeast2.rep.googleapis.com/', 'asia-northeast3': 'https://batch-datastore.asia-northeast3.rep.googleapis.com/', 'asia-south1': 'https://batch-datastore.asia-south1.rep.googleapis.com/', 'asia-south2': 'https://batch-datastore.asia-south2.rep.googleapis.com/', 'asia-southeast1': 'https://batch-datastore.asia-southeast1.rep.googleapis.com/', 'asia-southeast2': 'https://batch-datastore.asia-southeast2.rep.googleapis.com/', 'australia-southeast1': 'https://batch-datastore.australia-southeast1.rep.googleapis.com/', 'australia-southeast2': 'https://batch-datastore.australia-southeast2.rep.googleapis.com/', 'europe-central2': 'https://batch-datastore.europe-central2.rep.googleapis.com/', 'europe-north1': 'https://batch-datastore.europe-north1.rep.googleapis.com/', 'europe-north2': 'https://batch-datastore.europe-north2.rep.googleapis.com/', 'europe-southwest1': 'https://batch-datastore.europe-southwest1.rep.googleapis.com/', 'europe-west1': 'https://batch-datastore.europe-west1.rep.googleapis.com/', 'europe-west10': 'https://batch-datastore.europe-west10.rep.googleapis.com/', 'europe-west12': 'https://batch-datastore.europe-west12.rep.googleapis.com/', 'europe-west2': 'https://batch-datastore.europe-west2.rep.googleapis.com/', 'europe-west3': 'https://batch-datastore.europe-west3.rep.googleapis.com/', 'europe-west4': 'https://batch-datastore.europe-west4.rep.googleapis.com/', 'europe-west6': 'https://batch-datastore.europe-west6.rep.googleapis.com/', 'europe-west8': 'https://batch-datastore.europe-west8.rep.googleapis.com/', 'europe-west9': 'https://batch-datastore.europe-west9.rep.googleapis.com/', 'me-central1': 'https://batch-datastore.me-central1.rep.googleapis.com/', 'me-central2': 'https://batch-datastore.me-central2.rep.googleapis.com/', 'me-west1': 'https://batch-datastore.me-west1.rep.googleapis.com/', 'northamerica-northeast1': 'https://batch-datastore.northamerica-northeast1.rep.googleapis.com/', 'northamerica-northeast2': 'https://batch-datastore.northamerica-northeast2.rep.googleapis.com/', 'northamerica-south1': 'https://batch-datastore.northamerica-south1.rep.googleapis.com/', 'southamerica-east1': 'https://batch-datastore.southamerica-east1.rep.googleapis.com/', 'southamerica-west1': 'https://batch-datastore.southamerica-west1.rep.googleapis.com/', 'us-central1': 'https://batch-datastore.us-central1.rep.googleapis.com/', 'us-east1': 'https://batch-datastore.us-east1.rep.googleapis.com/', 'us-east4': 'https://batch-datastore.us-east4.rep.googleapis.com/', 'us-east5': 'https://batch-datastore.us-east5.rep.googleapis.com/', 'us-south1': 'https://batch-datastore.us-south1.rep.googleapis.com/', 'us-west1': 'https://batch-datastore.us-west1.rep.googleapis.com/', 'us-west2': 'https://batch-datastore.us-west2.rep.googleapis.com/', 'us-west3': 'https://batch-datastore.us-west3.rep.googleapis.com/', 'us-west4': 'https://batch-datastore.us-west4.rep.googleapis.com/', 'eu': 'https://batch-datastore.eu.rep.googleapis.com/', 'us': 'https://batch-datastore.us.rep.googleapis.com/'}),
    },
    'datastream': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.datastream.v1',
                    client_classpath='datastream_v1_client.DatastreamV1',
                    base_url='https://datastream.googleapis.com/',
                    messages_modulepath='datastream_v1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.datastream.v1alpha1',
                    client_classpath='datastream_v1alpha1_client.DatastreamV1alpha1',
                    base_url='https://datastream.googleapis.com/',
                    messages_modulepath='datastream_v1alpha1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'deploymentmanager': {
        'alpha':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.deploymentmanager.alpha',
                    client_classpath='deploymentmanager_alpha_client.DeploymentmanagerAlpha',
                    base_url='https://deploymentmanager.googleapis.com/',
                    messages_modulepath='deploymentmanager_alpha_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='https://www.mtls.googleapis.com/deploymentmanager/alpha/',
                regional_endpoints={}),
        'v2':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.deploymentmanager.v2',
                    client_classpath='deploymentmanager_v2_client.DeploymentmanagerV2',
                    base_url='https://deploymentmanager.googleapis.com/',
                    messages_modulepath='deploymentmanager_v2_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='https://www.mtls.googleapis.com/deploymentmanager/v2/',
                regional_endpoints={}),
        'v2beta':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.deploymentmanager.v2beta',
                    client_classpath='deploymentmanager_v2beta_client.DeploymentmanagerV2beta',
                    base_url='https://deploymentmanager.googleapis.com/',
                    messages_modulepath='deploymentmanager_v2beta_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='https://www.mtls.googleapis.com/deploymentmanager/v2beta/',
                regional_endpoints={}),
    },
    'designcenter': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.designcenter.v1',
                    client_classpath='designcenter_v1_client.DesigncenterV1',
                    base_url='https://designcenter.googleapis.com/',
                    messages_modulepath='designcenter_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.designcenter.v1alpha',
                    client_classpath='designcenter_v1alpha_client.DesigncenterV1alpha',
                    base_url='https://designcenter.googleapis.com/',
                    messages_modulepath='designcenter_v1alpha_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'developerconnect': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.developerconnect.v1',
                    client_classpath='developerconnect_v1_client.DeveloperconnectV1',
                    base_url='https://developerconnect.googleapis.com/',
                    messages_modulepath='developerconnect_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'us-central1': 'https://developerconnect.us-central1.rep.googleapis.com/', 'europe-west1': 'https://developerconnect.europe-west1.rep.googleapis.com/', 'asia-east1': 'https://developerconnect.asia-east1.rep.googleapis.com/', 'asia-east2': 'https://developerconnect.asia-east2.rep.googleapis.com/', 'europe-west4': 'https://developerconnect.europe-west4.rep.googleapis.com/', 'us-east4': 'https://developerconnect.us-east4.rep.googleapis.com/', 'us-east5': 'https://developerconnect.us-east5.rep.googleapis.com/', 'asia-southeast1': 'https://developerconnect.asia-southeast1.rep.googleapis.com/', 'us-west1': 'https://developerconnect.us-west1.rep.googleapis.com/', 'us-west2': 'https://developerconnect.us-west2.rep.googleapis.com/'}),
        'v1alpha':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.developerconnect.v1alpha',
                    client_classpath='developerconnect_v1alpha_client.DeveloperconnectV1alpha',
                    base_url='https://developerconnect.googleapis.com/',
                    messages_modulepath='developerconnect_v1alpha_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'us-central1': 'https://developerconnect.us-central1.rep.googleapis.com/', 'europe-west1': 'https://developerconnect.europe-west1.rep.googleapis.com/', 'asia-east1': 'https://developerconnect.asia-east1.rep.googleapis.com/', 'asia-east2': 'https://developerconnect.asia-east2.rep.googleapis.com/', 'europe-west4': 'https://developerconnect.europe-west4.rep.googleapis.com/', 'us-east4': 'https://developerconnect.us-east4.rep.googleapis.com/', 'us-east5': 'https://developerconnect.us-east5.rep.googleapis.com/', 'asia-southeast1': 'https://developerconnect.asia-southeast1.rep.googleapis.com/', 'us-west1': 'https://developerconnect.us-west1.rep.googleapis.com/', 'us-west2': 'https://developerconnect.us-west2.rep.googleapis.com/'}),
    },
    'dialogflow': {
        'v2':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.dialogflow.v2',
                    client_classpath='dialogflow_v2_client.DialogflowV2',
                    base_url='https://dialogflow.googleapis.com/',
                    messages_modulepath='dialogflow_v2_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'discovery': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.discovery.v1',
                    client_classpath='discovery_v1_client.DiscoveryV1',
                    base_url='https://www.googleapis.com/discovery/v1/',
                    messages_modulepath='discovery_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='https://www.mtls.googleapis.com/discovery/v1/',
                regional_endpoints={}),
    },
    'dlp': {
        'v2':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.dlp.v2',
                    client_classpath='dlp_v2_client.DlpV2',
                    base_url='https://dlp.googleapis.com/',
                    messages_modulepath='dlp_v2_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'africa-south1': 'https://dlp.africa-south1.rep.googleapis.com/', 'asia-east1': 'https://dlp.asia-east1.rep.googleapis.com/', 'asia-east2': 'https://dlp.asia-east2.rep.googleapis.com/', 'asia-northeast1': 'https://dlp.asia-northeast1.rep.googleapis.com/', 'asia-northeast2': 'https://dlp.asia-northeast2.rep.googleapis.com/', 'asia-northeast3': 'https://dlp.asia-northeast3.rep.googleapis.com/', 'asia-south1': 'https://dlp.asia-south1.rep.googleapis.com/', 'asia-south2': 'https://dlp.asia-south2.rep.googleapis.com/', 'asia-southeast1': 'https://dlp.asia-southeast1.rep.googleapis.com/', 'asia-southeast2': 'https://dlp.asia-southeast2.rep.googleapis.com/', 'australia-southeast1': 'https://dlp.australia-southeast1.rep.googleapis.com/', 'australia-southeast2': 'https://dlp.australia-southeast2.rep.googleapis.com/', 'europe-central2': 'https://dlp.europe-central2.rep.googleapis.com/', 'europe-north1': 'https://dlp.europe-north1.rep.googleapis.com/', 'europe-north2': 'https://dlp.europe-north2.rep.googleapis.com/', 'europe-southwest1': 'https://dlp.europe-southwest1.rep.googleapis.com/', 'europe-west1': 'https://dlp.europe-west1.rep.googleapis.com/', 'europe-west10': 'https://dlp.europe-west10.rep.googleapis.com/', 'europe-west12': 'https://dlp.europe-west12.rep.googleapis.com/', 'europe-west2': 'https://dlp.europe-west2.rep.googleapis.com/', 'europe-west3': 'https://dlp.europe-west3.rep.googleapis.com/', 'europe-west4': 'https://dlp.europe-west4.rep.googleapis.com/', 'europe-west6': 'https://dlp.europe-west6.rep.googleapis.com/', 'europe-west8': 'https://dlp.europe-west8.rep.googleapis.com/', 'europe-west9': 'https://dlp.europe-west9.rep.googleapis.com/', 'me-central1': 'https://dlp.me-central1.rep.googleapis.com/', 'me-central2': 'https://dlp.me-central2.rep.googleapis.com/', 'me-west1': 'https://dlp.me-west1.rep.googleapis.com/', 'northamerica-northeast1': 'https://dlp.northamerica-northeast1.rep.googleapis.com/', 'northamerica-northeast2': 'https://dlp.northamerica-northeast2.rep.googleapis.com/', 'northamerica-south1': 'https://dlp.northamerica-south1.rep.googleapis.com/', 'southamerica-east1': 'https://dlp.southamerica-east1.rep.googleapis.com/', 'southamerica-west1': 'https://dlp.southamerica-west1.rep.googleapis.com/', 'us-central1': 'https://dlp.us-central1.rep.googleapis.com/', 'us-central2': 'https://dlp.us-central2.rep.googleapis.com/', 'us-east1': 'https://dlp.us-east1.rep.googleapis.com/', 'us-east4': 'https://dlp.us-east4.rep.googleapis.com/', 'us-east5': 'https://dlp.us-east5.rep.googleapis.com/', 'us-south1': 'https://dlp.us-south1.rep.googleapis.com/', 'us-west1': 'https://dlp.us-west1.rep.googleapis.com/', 'us-west2': 'https://dlp.us-west2.rep.googleapis.com/', 'us-west3': 'https://dlp.us-west3.rep.googleapis.com/', 'us-west4': 'https://dlp.us-west4.rep.googleapis.com/', 'us-west8': 'https://dlp.us-west8.rep.googleapis.com/', 'us': 'https://dlp.us.rep.googleapis.com/', 'eu': 'https://dlp.eu.rep.googleapis.com/', 'in': 'https://dlp.in.rep.googleapis.com/'}),
    },
    'dns': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.dns.v1',
                    client_classpath='dns_v1_client.DnsV1',
                    base_url='https://dns.googleapis.com/dns/v1/',
                    messages_modulepath='dns_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='https://dns.mtls.googleapis.com/dns/v1/',
                regional_endpoints={}),
        'v1alpha2':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.dns.v1alpha2',
                    client_classpath='dns_v1alpha2_client.DnsV1alpha2',
                    base_url='https://dns.googleapis.com/',
                    messages_modulepath='dns_v1alpha2_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta2':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.dns.v1beta2',
                    client_classpath='dns_v1beta2_client.DnsV1beta2',
                    base_url='https://dns.googleapis.com/dns/v1beta2/',
                    messages_modulepath='dns_v1beta2_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='https://dns.mtls.googleapis.com/dns/v1beta2/',
                regional_endpoints={}),
        'v2':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.dns.v2',
                    client_classpath='dns_v2_client.DnsV2',
                    base_url='https://dns.googleapis.com/',
                    messages_modulepath='dns_v2_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='https://dns.mtls.googleapis.com/dns/v2/',
                regional_endpoints={}),
    },
    'documentai': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.documentai.v1',
                    client_classpath='documentai_v1_client.DocumentaiV1',
                    base_url='https://documentai.googleapis.com/',
                    messages_modulepath='documentai_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'domains': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.domains.v1',
                    client_classpath='domains_v1_client.DomainsV1',
                    base_url='https://domains.googleapis.com/',
                    messages_modulepath='domains_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha2':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.domains.v1alpha2',
                    client_classpath='domains_v1alpha2_client.DomainsV1alpha2',
                    base_url='https://domains.googleapis.com/',
                    messages_modulepath='domains_v1alpha2_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.domains.v1beta1',
                    client_classpath='domains_v1beta1_client.DomainsV1beta1',
                    base_url='https://domains.googleapis.com/',
                    messages_modulepath='domains_v1beta1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'edgecontainer': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.edgecontainer.v1',
                    client_classpath='edgecontainer_v1_client.EdgecontainerV1',
                    base_url='https://edgecontainer.googleapis.com/',
                    messages_modulepath='edgecontainer_v1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.edgecontainer.v1alpha',
                    client_classpath='edgecontainer_v1alpha_client.EdgecontainerV1alpha',
                    base_url='https://edgecontainer.googleapis.com/',
                    messages_modulepath='edgecontainer_v1alpha_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.edgecontainer.v1beta',
                    client_classpath='edgecontainer_v1beta_client.EdgecontainerV1beta',
                    base_url='https://edgecontainer.googleapis.com/',
                    messages_modulepath='edgecontainer_v1beta_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'edgenetwork': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.edgenetwork.v1',
                    client_classpath='edgenetwork_v1_client.EdgenetworkV1',
                    base_url='https://edgenetwork.googleapis.com/',
                    messages_modulepath='edgenetwork_v1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.edgenetwork.v1alpha1',
                    client_classpath='edgenetwork_v1alpha1_client.EdgenetworkV1alpha1',
                    base_url='https://edgenetwork.googleapis.com/',
                    messages_modulepath='edgenetwork_v1alpha1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'essentialcontacts': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.essentialcontacts.v1',
                    client_classpath='essentialcontacts_v1_client.EssentialcontactsV1',
                    base_url='https://essentialcontacts.googleapis.com/',
                    messages_modulepath='essentialcontacts_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.essentialcontacts.v1alpha1',
                    client_classpath='essentialcontacts_v1alpha1_client.EssentialcontactsV1alpha1',
                    base_url='https://essentialcontacts.googleapis.com/',
                    messages_modulepath='essentialcontacts_v1alpha1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.essentialcontacts.v1beta1',
                    client_classpath='essentialcontacts_v1beta1_client.EssentialcontactsV1beta1',
                    base_url='https://essentialcontacts.googleapis.com/',
                    messages_modulepath='essentialcontacts_v1beta1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'eventarc': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.eventarc.v1',
                    client_classpath='eventarc_v1_client.EventarcV1',
                    base_url='https://eventarc.googleapis.com/',
                    messages_modulepath='eventarc_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'eventarcpublishing': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.eventarcpublishing.v1',
                    client_classpath='eventarcpublishing_v1_client.EventarcpublishingV1',
                    base_url='https://eventarcpublishing.googleapis.com/',
                    messages_modulepath='eventarcpublishing_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'eventflow': {
        'v1beta2':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.eventflow.v1beta2',
                    client_classpath='eventflow_v1beta2_client.EventflowV1beta2',
                    base_url='https://eventflow.googleapis.com/',
                    messages_modulepath='eventflow_v1beta2_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'faultinjectiontesting': {
        'v1alpha1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.faultinjectiontesting.v1alpha1',
                    client_classpath='faultinjectiontesting_v1alpha1_client.FaultinjectiontestingV1alpha1',
                    base_url='https://faultinjectiontesting.googleapis.com/',
                    messages_modulepath='faultinjectiontesting_v1alpha1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'faulttesting': {
        'v1alpha':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.faulttesting.v1alpha',
                    client_classpath='faulttesting_v1alpha_client.FaulttestingV1alpha',
                    base_url='https://faulttesting.googleapis.com/',
                    messages_modulepath='faulttesting_v1alpha_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'file': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.file.v1',
                    client_classpath='file_v1_client.FileV1',
                    base_url='https://file.googleapis.com/',
                    messages_modulepath='file_v1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.file.v1beta1',
                    client_classpath='file_v1beta1_client.FileV1beta1',
                    base_url='https://file.googleapis.com/',
                    messages_modulepath='file_v1beta1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1p1alpha1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.file.v1p1alpha1',
                    client_classpath='file_v1p1alpha1_client.FileV1p1alpha1',
                    base_url='https://file.googleapis.com/',
                    messages_modulepath='file_v1p1alpha1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'firebasedataconnect': {
        'v1alpha':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.firebasedataconnect.v1alpha',
                    client_classpath='firebasedataconnect_v1alpha_client.FirebasedataconnectV1alpha',
                    base_url='https://firebasedataconnect.googleapis.com/',
                    messages_modulepath='firebasedataconnect_v1alpha_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.firebasedataconnect.v1beta',
                    client_classpath='firebasedataconnect_v1beta_client.FirebasedataconnectV1beta',
                    base_url='https://firebasedataconnect.googleapis.com/',
                    messages_modulepath='firebasedataconnect_v1beta_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'firestore': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.firestore.v1',
                    client_classpath='firestore_v1_client.FirestoreV1',
                    base_url='https://firestore.googleapis.com/',
                    messages_modulepath='firestore_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'africa-south1': 'https://batch-firestore.africa-south1.rep.googleapis.com/', 'asia-east1': 'https://batch-firestore.asia-east1.rep.googleapis.com/', 'asia-east2': 'https://batch-firestore.asia-east2.rep.googleapis.com/', 'asia-northeast1': 'https://batch-firestore.asia-northeast1.rep.googleapis.com/', 'asia-northeast2': 'https://batch-firestore.asia-northeast2.rep.googleapis.com/', 'asia-northeast3': 'https://batch-firestore.asia-northeast3.rep.googleapis.com/', 'asia-south1': 'https://batch-firestore.asia-south1.rep.googleapis.com/', 'asia-south2': 'https://batch-firestore.asia-south2.rep.googleapis.com/', 'asia-southeast1': 'https://batch-firestore.asia-southeast1.rep.googleapis.com/', 'asia-southeast2': 'https://batch-firestore.asia-southeast2.rep.googleapis.com/', 'australia-southeast1': 'https://batch-firestore.australia-southeast1.rep.googleapis.com/', 'australia-southeast2': 'https://batch-firestore.australia-southeast2.rep.googleapis.com/', 'europe-central2': 'https://batch-firestore.europe-central2.rep.googleapis.com/', 'europe-north1': 'https://batch-firestore.europe-north1.rep.googleapis.com/', 'europe-north2': 'https://batch-firestore.europe-north2.rep.googleapis.com/', 'europe-southwest1': 'https://batch-firestore.europe-southwest1.rep.googleapis.com/', 'europe-west1': 'https://batch-firestore.europe-west1.rep.googleapis.com/', 'europe-west10': 'https://batch-firestore.europe-west10.rep.googleapis.com/', 'europe-west12': 'https://batch-firestore.europe-west12.rep.googleapis.com/', 'europe-west2': 'https://batch-firestore.europe-west2.rep.googleapis.com/', 'europe-west3': 'https://batch-firestore.europe-west3.rep.googleapis.com/', 'europe-west4': 'https://batch-firestore.europe-west4.rep.googleapis.com/', 'europe-west6': 'https://batch-firestore.europe-west6.rep.googleapis.com/', 'europe-west8': 'https://batch-firestore.europe-west8.rep.googleapis.com/', 'europe-west9': 'https://batch-firestore.europe-west9.rep.googleapis.com/', 'me-central1': 'https://batch-firestore.me-central1.rep.googleapis.com/', 'me-central2': 'https://batch-firestore.me-central2.rep.googleapis.com/', 'me-west1': 'https://batch-firestore.me-west1.rep.googleapis.com/', 'northamerica-northeast1': 'https://batch-firestore.northamerica-northeast1.rep.googleapis.com/', 'northamerica-northeast2': 'https://batch-firestore.northamerica-northeast2.rep.googleapis.com/', 'northamerica-south1': 'https://batch-firestore.northamerica-south1.rep.googleapis.com/', 'southamerica-east1': 'https://batch-firestore.southamerica-east1.rep.googleapis.com/', 'southamerica-west1': 'https://batch-firestore.southamerica-west1.rep.googleapis.com/', 'us-central1': 'https://batch-firestore.us-central1.rep.googleapis.com/', 'us-east1': 'https://batch-firestore.us-east1.rep.googleapis.com/', 'us-east4': 'https://batch-firestore.us-east4.rep.googleapis.com/', 'us-east5': 'https://batch-firestore.us-east5.rep.googleapis.com/', 'us-south1': 'https://batch-firestore.us-south1.rep.googleapis.com/', 'us-west1': 'https://batch-firestore.us-west1.rep.googleapis.com/', 'us-west2': 'https://batch-firestore.us-west2.rep.googleapis.com/', 'us-west3': 'https://batch-firestore.us-west3.rep.googleapis.com/', 'us-west4': 'https://batch-firestore.us-west4.rep.googleapis.com/', 'eu': 'https://batch-firestore.eu.rep.googleapis.com/', 'us': 'https://batch-firestore.us.rep.googleapis.com/'}),
        'v1beta1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.firestore.v1beta1',
                    client_classpath='firestore_v1beta1_client.FirestoreV1beta1',
                    base_url='https://firestore.googleapis.com/',
                    messages_modulepath='firestore_v1beta1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'africa-south1': 'https://batch-firestore.africa-south1.rep.googleapis.com/', 'asia-east1': 'https://batch-firestore.asia-east1.rep.googleapis.com/', 'asia-east2': 'https://batch-firestore.asia-east2.rep.googleapis.com/', 'asia-northeast1': 'https://batch-firestore.asia-northeast1.rep.googleapis.com/', 'asia-northeast2': 'https://batch-firestore.asia-northeast2.rep.googleapis.com/', 'asia-northeast3': 'https://batch-firestore.asia-northeast3.rep.googleapis.com/', 'asia-south1': 'https://batch-firestore.asia-south1.rep.googleapis.com/', 'asia-south2': 'https://batch-firestore.asia-south2.rep.googleapis.com/', 'asia-southeast1': 'https://batch-firestore.asia-southeast1.rep.googleapis.com/', 'asia-southeast2': 'https://batch-firestore.asia-southeast2.rep.googleapis.com/', 'australia-southeast1': 'https://batch-firestore.australia-southeast1.rep.googleapis.com/', 'australia-southeast2': 'https://batch-firestore.australia-southeast2.rep.googleapis.com/', 'europe-central2': 'https://batch-firestore.europe-central2.rep.googleapis.com/', 'europe-north1': 'https://batch-firestore.europe-north1.rep.googleapis.com/', 'europe-north2': 'https://batch-firestore.europe-north2.rep.googleapis.com/', 'europe-southwest1': 'https://batch-firestore.europe-southwest1.rep.googleapis.com/', 'europe-west1': 'https://batch-firestore.europe-west1.rep.googleapis.com/', 'europe-west10': 'https://batch-firestore.europe-west10.rep.googleapis.com/', 'europe-west12': 'https://batch-firestore.europe-west12.rep.googleapis.com/', 'europe-west2': 'https://batch-firestore.europe-west2.rep.googleapis.com/', 'europe-west3': 'https://batch-firestore.europe-west3.rep.googleapis.com/', 'europe-west4': 'https://batch-firestore.europe-west4.rep.googleapis.com/', 'europe-west6': 'https://batch-firestore.europe-west6.rep.googleapis.com/', 'europe-west8': 'https://batch-firestore.europe-west8.rep.googleapis.com/', 'europe-west9': 'https://batch-firestore.europe-west9.rep.googleapis.com/', 'me-central1': 'https://batch-firestore.me-central1.rep.googleapis.com/', 'me-central2': 'https://batch-firestore.me-central2.rep.googleapis.com/', 'me-west1': 'https://batch-firestore.me-west1.rep.googleapis.com/', 'northamerica-northeast1': 'https://batch-firestore.northamerica-northeast1.rep.googleapis.com/', 'northamerica-northeast2': 'https://batch-firestore.northamerica-northeast2.rep.googleapis.com/', 'northamerica-south1': 'https://batch-firestore.northamerica-south1.rep.googleapis.com/', 'southamerica-east1': 'https://batch-firestore.southamerica-east1.rep.googleapis.com/', 'southamerica-west1': 'https://batch-firestore.southamerica-west1.rep.googleapis.com/', 'us-central1': 'https://batch-firestore.us-central1.rep.googleapis.com/', 'us-east1': 'https://batch-firestore.us-east1.rep.googleapis.com/', 'us-east4': 'https://batch-firestore.us-east4.rep.googleapis.com/', 'us-east5': 'https://batch-firestore.us-east5.rep.googleapis.com/', 'us-south1': 'https://batch-firestore.us-south1.rep.googleapis.com/', 'us-west1': 'https://batch-firestore.us-west1.rep.googleapis.com/', 'us-west2': 'https://batch-firestore.us-west2.rep.googleapis.com/', 'us-west3': 'https://batch-firestore.us-west3.rep.googleapis.com/', 'us-west4': 'https://batch-firestore.us-west4.rep.googleapis.com/', 'eu': 'https://batch-firestore.eu.rep.googleapis.com/', 'us': 'https://batch-firestore.us.rep.googleapis.com/'}),
        'v1beta2':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.firestore.v1beta2',
                    client_classpath='firestore_v1beta2_client.FirestoreV1beta2',
                    base_url='https://firestore.googleapis.com/',
                    messages_modulepath='firestore_v1beta2_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'africa-south1': 'https://batch-firestore.africa-south1.rep.googleapis.com/', 'asia-east1': 'https://batch-firestore.asia-east1.rep.googleapis.com/', 'asia-east2': 'https://batch-firestore.asia-east2.rep.googleapis.com/', 'asia-northeast1': 'https://batch-firestore.asia-northeast1.rep.googleapis.com/', 'asia-northeast2': 'https://batch-firestore.asia-northeast2.rep.googleapis.com/', 'asia-northeast3': 'https://batch-firestore.asia-northeast3.rep.googleapis.com/', 'asia-south1': 'https://batch-firestore.asia-south1.rep.googleapis.com/', 'asia-south2': 'https://batch-firestore.asia-south2.rep.googleapis.com/', 'asia-southeast1': 'https://batch-firestore.asia-southeast1.rep.googleapis.com/', 'asia-southeast2': 'https://batch-firestore.asia-southeast2.rep.googleapis.com/', 'australia-southeast1': 'https://batch-firestore.australia-southeast1.rep.googleapis.com/', 'australia-southeast2': 'https://batch-firestore.australia-southeast2.rep.googleapis.com/', 'europe-central2': 'https://batch-firestore.europe-central2.rep.googleapis.com/', 'europe-north1': 'https://batch-firestore.europe-north1.rep.googleapis.com/', 'europe-north2': 'https://batch-firestore.europe-north2.rep.googleapis.com/', 'europe-southwest1': 'https://batch-firestore.europe-southwest1.rep.googleapis.com/', 'europe-west1': 'https://batch-firestore.europe-west1.rep.googleapis.com/', 'europe-west10': 'https://batch-firestore.europe-west10.rep.googleapis.com/', 'europe-west12': 'https://batch-firestore.europe-west12.rep.googleapis.com/', 'europe-west2': 'https://batch-firestore.europe-west2.rep.googleapis.com/', 'europe-west3': 'https://batch-firestore.europe-west3.rep.googleapis.com/', 'europe-west4': 'https://batch-firestore.europe-west4.rep.googleapis.com/', 'europe-west6': 'https://batch-firestore.europe-west6.rep.googleapis.com/', 'europe-west8': 'https://batch-firestore.europe-west8.rep.googleapis.com/', 'europe-west9': 'https://batch-firestore.europe-west9.rep.googleapis.com/', 'me-central1': 'https://batch-firestore.me-central1.rep.googleapis.com/', 'me-central2': 'https://batch-firestore.me-central2.rep.googleapis.com/', 'me-west1': 'https://batch-firestore.me-west1.rep.googleapis.com/', 'northamerica-northeast1': 'https://batch-firestore.northamerica-northeast1.rep.googleapis.com/', 'northamerica-northeast2': 'https://batch-firestore.northamerica-northeast2.rep.googleapis.com/', 'northamerica-south1': 'https://batch-firestore.northamerica-south1.rep.googleapis.com/', 'southamerica-east1': 'https://batch-firestore.southamerica-east1.rep.googleapis.com/', 'southamerica-west1': 'https://batch-firestore.southamerica-west1.rep.googleapis.com/', 'us-central1': 'https://batch-firestore.us-central1.rep.googleapis.com/', 'us-east1': 'https://batch-firestore.us-east1.rep.googleapis.com/', 'us-east4': 'https://batch-firestore.us-east4.rep.googleapis.com/', 'us-east5': 'https://batch-firestore.us-east5.rep.googleapis.com/', 'us-south1': 'https://batch-firestore.us-south1.rep.googleapis.com/', 'us-west1': 'https://batch-firestore.us-west1.rep.googleapis.com/', 'us-west2': 'https://batch-firestore.us-west2.rep.googleapis.com/', 'us-west3': 'https://batch-firestore.us-west3.rep.googleapis.com/', 'us-west4': 'https://batch-firestore.us-west4.rep.googleapis.com/', 'eu': 'https://batch-firestore.eu.rep.googleapis.com/', 'us': 'https://batch-firestore.us.rep.googleapis.com/'}),
    },
    'geminicloudassist': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.geminicloudassist.v1',
                    client_classpath='geminicloudassist_v1_client.GeminicloudassistV1',
                    base_url='https://geminicloudassist.googleapis.com/',
                    messages_modulepath='geminicloudassist_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.geminicloudassist.v1alpha',
                    client_classpath='geminicloudassist_v1alpha_client.GeminicloudassistV1alpha',
                    base_url='https://geminicloudassist.googleapis.com/',
                    messages_modulepath='geminicloudassist_v1alpha_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'genomics': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.genomics.v1',
                    client_classpath='genomics_v1_client.GenomicsV1',
                    base_url='https://genomics.googleapis.com/',
                    messages_modulepath='genomics_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha2':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.genomics.v1alpha2',
                    client_classpath='genomics_v1alpha2_client.GenomicsV1alpha2',
                    base_url='https://genomics.googleapis.com/',
                    messages_modulepath='genomics_v1alpha2_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v2alpha1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.genomics.v2alpha1',
                    client_classpath='genomics_v2alpha1_client.GenomicsV2alpha1',
                    base_url='https://genomics.googleapis.com/',
                    messages_modulepath='genomics_v2alpha1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'gkebackup': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.gkebackup.v1',
                    client_classpath='gkebackup_v1_client.GkebackupV1',
                    base_url='https://gkebackup.googleapis.com/',
                    messages_modulepath='gkebackup_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'gkehub': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.gkehub.v1',
                    client_classpath='gkehub_v1_client.GkehubV1',
                    base_url='https://gkehub.googleapis.com/',
                    messages_modulepath='gkehub_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.gkehub.v1alpha',
                    client_classpath='gkehub_v1alpha_client.GkehubV1alpha',
                    base_url='https://gkehub.googleapis.com/',
                    messages_modulepath='gkehub_v1alpha_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.gkehub.v1alpha1',
                    client_classpath='gkehub_v1alpha1_client.GkehubV1alpha1',
                    base_url='https://gkehub.googleapis.com/',
                    messages_modulepath='gkehub_v1alpha1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha2':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.gkehub.v1alpha2',
                    client_classpath='gkehub_v1alpha2_client.GkehubV1alpha2',
                    base_url='https://gkehub.googleapis.com/',
                    messages_modulepath='gkehub_v1alpha2_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.gkehub.v1beta',
                    client_classpath='gkehub_v1beta_client.GkehubV1beta',
                    base_url='https://gkehub.googleapis.com/',
                    messages_modulepath='gkehub_v1beta_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.gkehub.v1beta1',
                    client_classpath='gkehub_v1beta1_client.GkehubV1beta1',
                    base_url='https://gkehub.googleapis.com/',
                    messages_modulepath='gkehub_v1beta1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v2':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.gkehub.v2',
                    client_classpath='gkehub_v2_client.GkehubV2',
                    base_url='https://gkehub.googleapis.com/',
                    messages_modulepath='gkehub_v2_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v2alpha':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.gkehub.v2alpha',
                    client_classpath='gkehub_v2alpha_client.GkehubV2alpha',
                    base_url='https://gkehub.googleapis.com/',
                    messages_modulepath='gkehub_v2alpha_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v2beta':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.gkehub.v2beta',
                    client_classpath='gkehub_v2beta_client.GkehubV2beta',
                    base_url='https://gkehub.googleapis.com/',
                    messages_modulepath='gkehub_v2beta_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'gkemulticloud': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.gkemulticloud.v1',
                    client_classpath='gkemulticloud_v1_client.GkemulticloudV1',
                    base_url='https://gkemulticloud.googleapis.com/',
                    messages_modulepath='gkemulticloud_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'gkeonprem': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.gkeonprem.v1',
                    client_classpath='gkeonprem_v1_client.GkeonpremV1',
                    base_url='https://gkeonprem.googleapis.com/',
                    messages_modulepath='gkeonprem_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'gkerecommender': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.gkerecommender.v1',
                    client_classpath='gkerecommender_v1_client.GkerecommenderV1',
                    base_url='https://gkerecommender.googleapis.com/',
                    messages_modulepath='gkerecommender_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.gkerecommender.v1alpha1',
                    client_classpath='gkerecommender_v1alpha1_client.GkerecommenderV1alpha1',
                    base_url='https://gkerecommender.googleapis.com/',
                    messages_modulepath='gkerecommender_v1alpha1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'gsuiteaddons': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.gsuiteaddons.v1',
                    client_classpath='gsuiteaddons_v1_client.GsuiteaddonsV1',
                    base_url='https://gsuiteaddons.googleapis.com/',
                    messages_modulepath='gsuiteaddons_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'healthcare': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.healthcare.v1',
                    client_classpath='healthcare_v1_client.HealthcareV1',
                    base_url='https://healthcare.googleapis.com/',
                    messages_modulepath='healthcare_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha2':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.healthcare.v1alpha2',
                    client_classpath='healthcare_v1alpha2_client.HealthcareV1alpha2',
                    base_url='https://healthcare.googleapis.com/',
                    messages_modulepath='healthcare_v1alpha2_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.healthcare.v1beta1',
                    client_classpath='healthcare_v1beta1_client.HealthcareV1beta1',
                    base_url='https://healthcare.googleapis.com/',
                    messages_modulepath='healthcare_v1beta1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'hypercomputecluster': {
        'v1alpha':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.hypercomputecluster.v1alpha',
                    client_classpath='hypercomputecluster_v1alpha_client.HypercomputeclusterV1alpha',
                    base_url='https://hypercomputecluster.googleapis.com/',
                    messages_modulepath='hypercomputecluster_v1alpha_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'iam': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.iam.v1',
                    client_classpath='iam_v1_client.IamV1',
                    base_url='https://iam.googleapis.com/',
                    messages_modulepath='iam_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.iam.v1beta',
                    client_classpath='iam_v1beta_client.IamV1beta',
                    base_url='https://iam.googleapis.com/',
                    messages_modulepath='iam_v1beta_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v2':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.iam.v2',
                    client_classpath='iam_v2_client.IamV2',
                    base_url='https://iam.googleapis.com/',
                    messages_modulepath='iam_v2_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v2alpha':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.iam.v2alpha',
                    client_classpath='iam_v2alpha_client.IamV2alpha',
                    base_url='https://iam.googleapis.com/',
                    messages_modulepath='iam_v2alpha_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v2beta':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.iam.v2beta',
                    client_classpath='iam_v2beta_client.IamV2beta',
                    base_url='https://iam.googleapis.com/',
                    messages_modulepath='iam_v2beta_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v3':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.iam.v3',
                    client_classpath='iam_v3_client.IamV3',
                    base_url='https://iam.googleapis.com/',
                    messages_modulepath='iam_v3_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v3alpha':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.iam.v3alpha',
                    client_classpath='iam_v3alpha_client.IamV3alpha',
                    base_url='https://iam.googleapis.com/',
                    messages_modulepath='iam_v3alpha_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v3beta':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.iam.v3beta',
                    client_classpath='iam_v3beta_client.IamV3beta',
                    base_url='https://iam.googleapis.com/',
                    messages_modulepath='iam_v3beta_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'iamcredentials': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.iamcredentials.v1',
                    client_classpath='iamcredentials_v1_client.IamcredentialsV1',
                    base_url='https://iamcredentials.googleapis.com/',
                    messages_modulepath='iamcredentials_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'asia-east1': 'https://iamcredentials.asia-east1.rep.googleapis.com/', 'europe-west1': 'https://iamcredentials.europe-west1.rep.googleapis.com/', 'us-central1': 'https://iamcredentials.us-central1.rep.googleapis.com/', 'us-east1': 'https://iamcredentials.us-east1.rep.googleapis.com/', 'us-east7': 'https://iamcredentials.us-east7.rep.googleapis.com/', 'us-west1': 'https://iamcredentials.us-west1.rep.googleapis.com/'}),
    },
    'iap': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.iap.v1',
                    client_classpath='iap_v1_client.IapV1',
                    base_url='https://iap.googleapis.com/',
                    messages_modulepath='iap_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.iap.v1beta1',
                    client_classpath='iap_v1beta1_client.IapV1beta1',
                    base_url='https://iap.googleapis.com/',
                    messages_modulepath='iap_v1beta1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'ids': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.ids.v1',
                    client_classpath='ids_v1_client.IdsV1',
                    base_url='https://ids.googleapis.com/',
                    messages_modulepath='ids_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'kmsinventory': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.kmsinventory.v1',
                    client_classpath='kmsinventory_v1_client.KmsinventoryV1',
                    base_url='https://kmsinventory.googleapis.com/',
                    messages_modulepath='kmsinventory_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'krmapihosting': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.krmapihosting.v1',
                    client_classpath='krmapihosting_v1_client.KrmapihostingV1',
                    base_url='https://krmapihosting.googleapis.com/',
                    messages_modulepath='krmapihosting_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.krmapihosting.v1alpha1',
                    client_classpath='krmapihosting_v1alpha1_client.KrmapihostingV1alpha1',
                    base_url='https://krmapihosting.googleapis.com/',
                    messages_modulepath='krmapihosting_v1alpha1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'language': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.language.v1',
                    client_classpath='language_v1_client.LanguageV1',
                    base_url='https://language.googleapis.com/',
                    messages_modulepath='language_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta2':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.language.v1beta2',
                    client_classpath='language_v1beta2_client.LanguageV1beta2',
                    base_url='https://language.googleapis.com/',
                    messages_modulepath='language_v1beta2_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='https://language.mtls.googleapis.com/',
                regional_endpoints={}),
    },
    'lifesciences': {
        'v2beta':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.lifesciences.v2beta',
                    client_classpath='lifesciences_v2beta_client.LifesciencesV2beta',
                    base_url='https://lifesciences.googleapis.com/',
                    messages_modulepath='lifesciences_v2beta_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'logging': {
        'v2':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.logging.v2',
                    client_classpath='logging_v2_client.LoggingV2',
                    base_url='https://logging.googleapis.com/',
                    messages_modulepath='logging_v2_messages'),
                gapic=GapicClientDef(
                    class_path='googlecloudsdk.generated_clients.gapic_wrappers.logging.v2'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'africa-south1': 'https://logging.africa-south1.rep.googleapis.com/', 'asia-east1': 'https://logging.asia-east1.rep.googleapis.com/', 'asia-east2': 'https://logging.asia-east2.rep.googleapis.com/', 'asia-northeast1': 'https://logging.asia-northeast1.rep.googleapis.com/', 'asia-northeast2': 'https://logging.asia-northeast2.rep.googleapis.com/', 'asia-northeast3': 'https://logging.asia-northeast3.rep.googleapis.com/', 'asia-south1': 'https://logging.asia-south1.rep.googleapis.com/', 'asia-south2': 'https://logging.asia-south2.rep.googleapis.com/', 'asia-southeast1': 'https://logging.asia-southeast1.rep.googleapis.com/', 'asia-southeast2': 'https://logging.asia-southeast2.rep.googleapis.com/', 'asia-southeast3': 'https://logging.asia-southeast3.rep.googleapis.com/', 'australia-southeast1': 'https://logging.australia-southeast1.rep.googleapis.com/', 'australia-southeast2': 'https://logging.australia-southeast2.rep.googleapis.com/', 'europe-central2': 'https://logging.europe-central2.rep.googleapis.com/', 'europe-north1': 'https://logging.europe-north1.rep.googleapis.com/', 'europe-north2': 'https://logging.europe-north2.rep.googleapis.com/', 'europe-north3': 'https://logging.europe-north3.rep.googleapis.com/', 'europe-southwest1': 'https://logging.europe-southwest1.rep.googleapis.com/', 'europe-west1': 'https://logging.europe-west1.rep.googleapis.com/', 'europe-west10': 'https://logging.europe-west10.rep.googleapis.com/', 'europe-west12': 'https://logging.europe-west12.rep.googleapis.com/', 'europe-west15': 'https://logging.europe-west15.rep.googleapis.com/', 'europe-west2': 'https://logging.europe-west2.rep.googleapis.com/', 'europe-west3': 'https://logging.europe-west3.rep.googleapis.com/', 'europe-west4': 'https://logging.europe-west4.rep.googleapis.com/', 'europe-west6': 'https://logging.europe-west6.rep.googleapis.com/', 'europe-west8': 'https://logging.europe-west8.rep.googleapis.com/', 'europe-west9': 'https://logging.europe-west9.rep.googleapis.com/', 'me-central1': 'https://logging.me-central1.rep.googleapis.com/', 'me-central2': 'https://logging.me-central2.rep.googleapis.com/', 'me-west1': 'https://logging.me-west1.rep.googleapis.com/', 'northamerica-northeast1': 'https://logging.northamerica-northeast1.rep.googleapis.com/', 'northamerica-northeast2': 'https://logging.northamerica-northeast2.rep.googleapis.com/', 'northamerica-south1': 'https://logging.northamerica-south1.rep.googleapis.com/', 'southamerica-east1': 'https://logging.southamerica-east1.rep.googleapis.com/', 'southamerica-west1': 'https://logging.southamerica-west1.rep.googleapis.com/', 'us-central1': 'https://logging.us-central1.rep.googleapis.com/', 'us-central2': 'https://logging.us-central2.rep.googleapis.com/', 'us-east1': 'https://logging.us-east1.rep.googleapis.com/', 'us-east4': 'https://logging.us-east4.rep.googleapis.com/', 'us-east5': 'https://logging.us-east5.rep.googleapis.com/', 'us-east7': 'https://logging.us-east7.rep.googleapis.com/', 'us-south1': 'https://logging.us-south1.rep.googleapis.com/', 'us-west1': 'https://logging.us-west1.rep.googleapis.com/', 'us-west2': 'https://logging.us-west2.rep.googleapis.com/', 'us-west3': 'https://logging.us-west3.rep.googleapis.com/', 'us-west4': 'https://logging.us-west4.rep.googleapis.com/', 'us-west8': 'https://logging.us-west8.rep.googleapis.com/', 'eu': 'https://logging.eu.rep.googleapis.com/', 'in': 'https://logging.in.rep.googleapis.com/', 'us': 'https://logging.us.rep.googleapis.com/'}),
    },
    'looker': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.looker.v1',
                    client_classpath='looker_v1_client.LookerV1',
                    base_url='https://looker.googleapis.com/',
                    messages_modulepath='looker_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha2':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.looker.v1alpha2',
                    client_classpath='looker_v1alpha2_client.LookerV1alpha2',
                    base_url='https://looker.googleapis.com/',
                    messages_modulepath='looker_v1alpha2_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'lustre': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.lustre.v1',
                    client_classpath='lustre_v1_client.LustreV1',
                    base_url='https://lustre.googleapis.com/',
                    messages_modulepath='lustre_v1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.lustre.v1alpha',
                    client_classpath='lustre_v1alpha_client.LustreV1alpha',
                    base_url='https://lustre.googleapis.com/',
                    messages_modulepath='lustre_v1alpha_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'managedflink': {
        'v1alpha':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.managedflink.v1alpha',
                    client_classpath='managedflink_v1alpha_client.ManagedflinkV1alpha',
                    base_url='https://managedflink.googleapis.com/',
                    messages_modulepath='managedflink_v1alpha_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'managedidentities': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.managedidentities.v1',
                    client_classpath='managedidentities_v1_client.ManagedidentitiesV1',
                    base_url='https://managedidentities.googleapis.com/',
                    messages_modulepath='managedidentities_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.managedidentities.v1alpha1',
                    client_classpath='managedidentities_v1alpha1_client.ManagedidentitiesV1alpha1',
                    base_url='https://managedidentities.googleapis.com/',
                    messages_modulepath='managedidentities_v1alpha1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.managedidentities.v1beta1',
                    client_classpath='managedidentities_v1beta1_client.ManagedidentitiesV1beta1',
                    base_url='https://managedidentities.googleapis.com/',
                    messages_modulepath='managedidentities_v1beta1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'managedkafka': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.managedkafka.v1',
                    client_classpath='managedkafka_v1_client.ManagedkafkaV1',
                    base_url='https://managedkafka.googleapis.com/',
                    messages_modulepath='managedkafka_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'africa-south1': 'https://managedkafka.africa-south1.rep.googleapis.com/', 'asia-east1': 'https://managedkafka.asia-east1.rep.googleapis.com/', 'asia-east2': 'https://managedkafka.asia-east2.rep.googleapis.com/', 'asia-northeast1': 'https://managedkafka.asia-northeast1.rep.googleapis.com/', 'asia-northeast2': 'https://managedkafka.asia-northeast2.rep.googleapis.com/', 'asia-northeast3': 'https://managedkafka.asia-northeast3.rep.googleapis.com/', 'asia-south1': 'https://managedkafka.asia-south1.rep.googleapis.com/', 'asia-south2': 'https://managedkafka.asia-south2.rep.googleapis.com/', 'asia-southeast1': 'https://managedkafka.asia-southeast1.rep.googleapis.com/', 'asia-southeast2': 'https://managedkafka.asia-southeast2.rep.googleapis.com/', 'australia-southeast1': 'https://managedkafka.australia-southeast1.rep.googleapis.com/', 'australia-southeast2': 'https://managedkafka.australia-southeast2.rep.googleapis.com/', 'europe-central2': 'https://managedkafka.europe-central2.rep.googleapis.com/', 'europe-north1': 'https://managedkafka.europe-north1.rep.googleapis.com/', 'europe-north2': 'https://managedkafka.europe-north2.rep.googleapis.com/', 'europe-southwest1': 'https://managedkafka.europe-southwest1.rep.googleapis.com/', 'europe-west1': 'https://managedkafka.europe-west1.rep.googleapis.com/', 'europe-west2': 'https://managedkafka.europe-west2.rep.googleapis.com/', 'europe-west3': 'https://managedkafka.europe-west3.rep.googleapis.com/', 'europe-west4': 'https://managedkafka.europe-west4.rep.googleapis.com/', 'europe-west6': 'https://managedkafka.europe-west6.rep.googleapis.com/', 'europe-west8': 'https://managedkafka.europe-west8.rep.googleapis.com/', 'europe-west9': 'https://managedkafka.europe-west9.rep.googleapis.com/', 'europe-west10': 'https://managedkafka.europe-west10.rep.googleapis.com/', 'europe-west12': 'https://managedkafka.europe-west12.rep.googleapis.com/', 'me-west1': 'https://managedkafka.me-west1.rep.googleapis.com/', 'me-central1': 'https://managedkafka.me-central1.rep.googleapis.com/', 'me-central2': 'https://managedkafka.me-central2.rep.googleapis.com/', 'northamerica-northeast1': 'https://managedkafka.northamerica-northeast1.rep.googleapis.com/', 'northamerica-northeast2': 'https://managedkafka.northamerica-northeast2.rep.googleapis.com/', 'southamerica-east1': 'https://managedkafka.southamerica-east1.rep.googleapis.com/', 'southamerica-west1': 'https://managedkafka.southamerica-west1.rep.googleapis.com/', 'us-central1': 'https://managedkafka.us-central1.rep.googleapis.com/', 'us-east1': 'https://managedkafka.us-east1.rep.googleapis.com/', 'us-east4': 'https://managedkafka.us-east4.rep.googleapis.com/', 'us-east5': 'https://managedkafka.us-east5.rep.googleapis.com/', 'us-east7': 'https://managedkafka.us-east7.rep.googleapis.com/', 'us-south1': 'https://managedkafka.us-south1.rep.googleapis.com/', 'us-west1': 'https://managedkafka.us-west1.rep.googleapis.com/', 'us-west2': 'https://managedkafka.us-west2.rep.googleapis.com/', 'us-west3': 'https://managedkafka.us-west3.rep.googleapis.com/', 'us-west4': 'https://managedkafka.us-west4.rep.googleapis.com/'}),
    },
    'marketplacesolutions': {
        'v1alpha1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.marketplacesolutions.v1alpha1',
                    client_classpath='marketplacesolutions_v1alpha1_client.MarketplacesolutionsV1alpha1',
                    base_url='https://marketplacesolutions.googleapis.com/',
                    messages_modulepath='marketplacesolutions_v1alpha1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'mediaasset': {
        'v1alpha':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.mediaasset.v1alpha',
                    client_classpath='mediaasset_v1alpha_client.MediaassetV1alpha',
                    base_url='https://mediaasset.googleapis.com/',
                    messages_modulepath='mediaasset_v1alpha_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'memcache': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.memcache.v1',
                    client_classpath='memcache_v1_client.MemcacheV1',
                    base_url='https://memcache.googleapis.com/',
                    messages_modulepath='memcache_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta2':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.memcache.v1beta2',
                    client_classpath='memcache_v1beta2_client.MemcacheV1beta2',
                    base_url='https://memcache.googleapis.com/',
                    messages_modulepath='memcache_v1beta2_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'memorystore': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.memorystore.v1',
                    client_classpath='memorystore_v1_client.MemorystoreV1',
                    base_url='https://memorystore.googleapis.com/',
                    messages_modulepath='memorystore_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.memorystore.v1alpha',
                    client_classpath='memorystore_v1alpha_client.MemorystoreV1alpha',
                    base_url='https://memorystore.googleapis.com/',
                    messages_modulepath='memorystore_v1alpha_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.memorystore.v1beta',
                    client_classpath='memorystore_v1beta_client.MemorystoreV1beta',
                    base_url='https://memorystore.googleapis.com/',
                    messages_modulepath='memorystore_v1beta_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'messagestreams': {
        'v1alpha':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.messagestreams.v1alpha',
                    client_classpath='messagestreams_v1alpha_client.MessagestreamsV1alpha',
                    base_url='https://messagestreams.googleapis.com/',
                    messages_modulepath='messagestreams_v1alpha_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'metastore': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.metastore.v1',
                    client_classpath='metastore_v1_client.MetastoreV1',
                    base_url='https://metastore.googleapis.com/',
                    messages_modulepath='metastore_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.metastore.v1alpha',
                    client_classpath='metastore_v1alpha_client.MetastoreV1alpha',
                    base_url='https://metastore.googleapis.com/',
                    messages_modulepath='metastore_v1alpha_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.metastore.v1beta',
                    client_classpath='metastore_v1beta_client.MetastoreV1beta',
                    base_url='https://metastore.googleapis.com/',
                    messages_modulepath='metastore_v1beta_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'microservices': {
        'v1alpha1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.microservices.v1alpha1',
                    client_classpath='microservices_v1alpha1_client.MicroservicesV1alpha1',
                    base_url='https://microservices.googleapis.com/',
                    messages_modulepath='microservices_v1alpha1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'ml': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.ml.v1',
                    client_classpath='ml_v1_client.MlV1',
                    base_url='https://ml.googleapis.com/',
                    messages_modulepath='ml_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'modelarmor': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.modelarmor.v1',
                    client_classpath='modelarmor_v1_client.ModelarmorV1',
                    base_url='https://modelarmor.us.rep.googleapis.com/',
                    messages_modulepath='modelarmor_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'us-central1': 'https://modelarmor.us-central1.rep.googleapis.com/', 'us-east1': 'https://modelarmor.us-east1.rep.googleapis.com/', 'us-east7': 'https://modelarmor.us-east7.rep.googleapis.com/', 'us-west1': 'https://modelarmor.us-west1.rep.googleapis.com/', 'europe-west1': 'https://modelarmor.europe-west1.rep.googleapis.com/', 'europe-west4': 'https://modelarmor.europe-west4.rep.googleapis.com/', 'us-east4': 'https://modelarmor.us-east4.rep.googleapis.com/', 'asia-southeast1': 'https://modelarmor.asia-southeast1.rep.googleapis.com/', 'europe-west2': 'https://modelarmor.europe-west2.rep.googleapis.com/', 'europe-west3': 'https://modelarmor.europe-west3.rep.googleapis.com/', 'asia-south1': 'https://modelarmor.asia-south1.rep.googleapis.com/', 'europe-southwest1': 'https://modelarmor.europe-southwest1.rep.googleapis.com/', 'asia-northeast1': 'https://modelarmor.asia-northeast1.rep.googleapis.com/', 'asia-northeast3': 'https://modelarmor.asia-northeast3.rep.googleapis.com/', 'australia-southeast1': 'https://modelarmor.australia-southeast1.rep.googleapis.com/', 'australia-southeast2': 'https://modelarmor.australia-southeast2.rep.googleapis.com/', 'northamerica-northeast1': 'https://modelarmor.northamerica-northeast1.rep.googleapis.com/', 'northamerica-northeast2': 'https://modelarmor.northamerica-northeast2.rep.googleapis.com/', 'europe-west9': 'https://modelarmor.europe-west9.rep.googleapis.com/', 'us': 'https://modelarmor.us.rep.googleapis.com/', 'eu': 'https://modelarmor.eu.rep.googleapis.com/'}),
        'v1alpha':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.modelarmor.v1alpha',
                    client_classpath='modelarmor_v1alpha_client.ModelarmorV1alpha',
                    base_url='https://modelarmor.us.rep.googleapis.com/',
                    messages_modulepath='modelarmor_v1alpha_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'us-central1': 'https://modelarmor.us-central1.rep.googleapis.com/', 'us-east1': 'https://modelarmor.us-east1.rep.googleapis.com/', 'us-east7': 'https://modelarmor.us-east7.rep.googleapis.com/', 'us-west1': 'https://modelarmor.us-west1.rep.googleapis.com/', 'europe-west1': 'https://modelarmor.europe-west1.rep.googleapis.com/', 'europe-west4': 'https://modelarmor.europe-west4.rep.googleapis.com/', 'us-east4': 'https://modelarmor.us-east4.rep.googleapis.com/', 'asia-southeast1': 'https://modelarmor.asia-southeast1.rep.googleapis.com/', 'europe-west2': 'https://modelarmor.europe-west2.rep.googleapis.com/', 'europe-west3': 'https://modelarmor.europe-west3.rep.googleapis.com/', 'asia-south1': 'https://modelarmor.asia-south1.rep.googleapis.com/', 'europe-southwest1': 'https://modelarmor.europe-southwest1.rep.googleapis.com/', 'asia-northeast1': 'https://modelarmor.asia-northeast1.rep.googleapis.com/', 'asia-northeast3': 'https://modelarmor.asia-northeast3.rep.googleapis.com/', 'australia-southeast1': 'https://modelarmor.australia-southeast1.rep.googleapis.com/', 'australia-southeast2': 'https://modelarmor.australia-southeast2.rep.googleapis.com/', 'northamerica-northeast1': 'https://modelarmor.northamerica-northeast1.rep.googleapis.com/', 'northamerica-northeast2': 'https://modelarmor.northamerica-northeast2.rep.googleapis.com/', 'europe-west9': 'https://modelarmor.europe-west9.rep.googleapis.com/', 'us': 'https://modelarmor.us.rep.googleapis.com/', 'eu': 'https://modelarmor.eu.rep.googleapis.com/'}),
        'v1beta':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.modelarmor.v1beta',
                    client_classpath='modelarmor_v1beta_client.ModelarmorV1beta',
                    base_url='https://modelarmor.us.rep.googleapis.com/',
                    messages_modulepath='modelarmor_v1beta_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'us-central1': 'https://modelarmor.us-central1.rep.googleapis.com/', 'us-east1': 'https://modelarmor.us-east1.rep.googleapis.com/', 'us-east7': 'https://modelarmor.us-east7.rep.googleapis.com/', 'us-west1': 'https://modelarmor.us-west1.rep.googleapis.com/', 'europe-west1': 'https://modelarmor.europe-west1.rep.googleapis.com/', 'europe-west4': 'https://modelarmor.europe-west4.rep.googleapis.com/', 'us-east4': 'https://modelarmor.us-east4.rep.googleapis.com/', 'asia-southeast1': 'https://modelarmor.asia-southeast1.rep.googleapis.com/', 'europe-west2': 'https://modelarmor.europe-west2.rep.googleapis.com/', 'europe-west3': 'https://modelarmor.europe-west3.rep.googleapis.com/', 'asia-south1': 'https://modelarmor.asia-south1.rep.googleapis.com/', 'europe-southwest1': 'https://modelarmor.europe-southwest1.rep.googleapis.com/', 'asia-northeast1': 'https://modelarmor.asia-northeast1.rep.googleapis.com/', 'asia-northeast3': 'https://modelarmor.asia-northeast3.rep.googleapis.com/', 'australia-southeast1': 'https://modelarmor.australia-southeast1.rep.googleapis.com/', 'australia-southeast2': 'https://modelarmor.australia-southeast2.rep.googleapis.com/', 'northamerica-northeast1': 'https://modelarmor.northamerica-northeast1.rep.googleapis.com/', 'northamerica-northeast2': 'https://modelarmor.northamerica-northeast2.rep.googleapis.com/', 'europe-west9': 'https://modelarmor.europe-west9.rep.googleapis.com/', 'us': 'https://modelarmor.us.rep.googleapis.com/', 'eu': 'https://modelarmor.eu.rep.googleapis.com/'}),
    },
    'monitoring': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.monitoring.v1',
                    client_classpath='monitoring_v1_client.MonitoringV1',
                    base_url='https://monitoring.googleapis.com/',
                    messages_modulepath='monitoring_v1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v3':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.monitoring.v3',
                    client_classpath='monitoring_v3_client.MonitoringV3',
                    base_url='https://monitoring.googleapis.com/',
                    messages_modulepath='monitoring_v3_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='https://monitoring.mtls.googleapis.com/',
                regional_endpoints={}),
    },
    'netapp': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.netapp.v1',
                    client_classpath='netapp_v1_client.NetappV1',
                    base_url='https://netapp.googleapis.com/',
                    messages_modulepath='netapp_v1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'africa-south1': 'https://netapp.africa-south1.rep.googleapis.com/', 'asia-east1': 'https://netapp.asia-east1.rep.googleapis.com/', 'asia-east2': 'https://netapp.asia-east2.rep.googleapis.com/', 'asia-northeast1': 'https://netapp.asia-northeast1.rep.googleapis.com/', 'asia-northeast2': 'https://netapp.asia-northeast2.rep.googleapis.com/', 'asia-northeast3': 'https://netapp.asia-northeast3.rep.googleapis.com/', 'asia-south1': 'https://netapp.asia-south1.rep.googleapis.com/', 'asia-south2': 'https://netapp.asia-south2.rep.googleapis.com/', 'asia-southeast1': 'https://netapp.asia-southeast1.rep.googleapis.com/', 'asia-southeast2': 'https://netapp.asia-southeast2.rep.googleapis.com/', 'australia-southeast1': 'https://netapp.australia-southeast1.rep.googleapis.com/', 'australia-southeast2': 'https://netapp.australia-southeast2.rep.googleapis.com/', 'europe-central2': 'https://netapp.europe-central2.rep.googleapis.com/', 'europe-north1': 'https://netapp.europe-north1.rep.googleapis.com/', 'europe-north2': 'https://netapp.europe-north2.rep.googleapis.com/', 'europe-southwest1': 'https://netapp.europe-southwest1.rep.googleapis.com/', 'europe-west1': 'https://netapp.europe-west1.rep.googleapis.com/', 'europe-west10': 'https://netapp.europe-west10.rep.googleapis.com/', 'europe-west12': 'https://netapp.europe-west12.rep.googleapis.com/', 'europe-west2': 'https://netapp.europe-west2.rep.googleapis.com/', 'europe-west3': 'https://netapp.europe-west3.rep.googleapis.com/', 'europe-west4': 'https://netapp.europe-west4.rep.googleapis.com/', 'europe-west6': 'https://netapp.europe-west6.rep.googleapis.com/', 'europe-west8': 'https://netapp.europe-west8.rep.googleapis.com/', 'europe-west9': 'https://netapp.europe-west9.rep.googleapis.com/', 'me-central1': 'https://netapp.me-central1.rep.googleapis.com/', 'me-central2': 'https://netapp.me-central2.rep.googleapis.com/', 'me-west1': 'https://netapp.me-west1.rep.googleapis.com/', 'northamerica-northeast1': 'https://netapp.northamerica-northeast1.rep.googleapis.com/', 'northamerica-northeast2': 'https://netapp.northamerica-northeast2.rep.googleapis.com/', 'northamerica-south1': 'https://netapp.northamerica-south1.rep.googleapis.com/', 'southamerica-east1': 'https://netapp.southamerica-east1.rep.googleapis.com/', 'southamerica-west1': 'https://netapp.southamerica-west1.rep.googleapis.com/', 'us-central1': 'https://netapp.us-central1.rep.googleapis.com/', 'us-east1': 'https://netapp.us-east1.rep.googleapis.com/', 'us-east4': 'https://netapp.us-east4.rep.googleapis.com/', 'us-east5': 'https://netapp.us-east5.rep.googleapis.com/', 'us-east7': 'https://netapp.us-east7.rep.googleapis.com/', 'us-south1': 'https://netapp.us-south1.rep.googleapis.com/', 'us-west1': 'https://netapp.us-west1.rep.googleapis.com/', 'us-west2': 'https://netapp.us-west2.rep.googleapis.com/', 'us-west3': 'https://netapp.us-west3.rep.googleapis.com/', 'us-west4': 'https://netapp.us-west4.rep.googleapis.com/'}),
        'v1alpha1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.netapp.v1alpha1',
                    client_classpath='netapp_v1alpha1_client.NetappV1alpha1',
                    base_url='https://netapp.googleapis.com/',
                    messages_modulepath='netapp_v1alpha1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'africa-south1': 'https://netapp.africa-south1.rep.googleapis.com/', 'asia-east1': 'https://netapp.asia-east1.rep.googleapis.com/', 'asia-east2': 'https://netapp.asia-east2.rep.googleapis.com/', 'asia-northeast1': 'https://netapp.asia-northeast1.rep.googleapis.com/', 'asia-northeast2': 'https://netapp.asia-northeast2.rep.googleapis.com/', 'asia-northeast3': 'https://netapp.asia-northeast3.rep.googleapis.com/', 'asia-south1': 'https://netapp.asia-south1.rep.googleapis.com/', 'asia-south2': 'https://netapp.asia-south2.rep.googleapis.com/', 'asia-southeast1': 'https://netapp.asia-southeast1.rep.googleapis.com/', 'asia-southeast2': 'https://netapp.asia-southeast2.rep.googleapis.com/', 'australia-southeast1': 'https://netapp.australia-southeast1.rep.googleapis.com/', 'australia-southeast2': 'https://netapp.australia-southeast2.rep.googleapis.com/', 'europe-central2': 'https://netapp.europe-central2.rep.googleapis.com/', 'europe-north1': 'https://netapp.europe-north1.rep.googleapis.com/', 'europe-north2': 'https://netapp.europe-north2.rep.googleapis.com/', 'europe-southwest1': 'https://netapp.europe-southwest1.rep.googleapis.com/', 'europe-west1': 'https://netapp.europe-west1.rep.googleapis.com/', 'europe-west10': 'https://netapp.europe-west10.rep.googleapis.com/', 'europe-west12': 'https://netapp.europe-west12.rep.googleapis.com/', 'europe-west2': 'https://netapp.europe-west2.rep.googleapis.com/', 'europe-west3': 'https://netapp.europe-west3.rep.googleapis.com/', 'europe-west4': 'https://netapp.europe-west4.rep.googleapis.com/', 'europe-west6': 'https://netapp.europe-west6.rep.googleapis.com/', 'europe-west8': 'https://netapp.europe-west8.rep.googleapis.com/', 'europe-west9': 'https://netapp.europe-west9.rep.googleapis.com/', 'me-central1': 'https://netapp.me-central1.rep.googleapis.com/', 'me-central2': 'https://netapp.me-central2.rep.googleapis.com/', 'me-west1': 'https://netapp.me-west1.rep.googleapis.com/', 'northamerica-northeast1': 'https://netapp.northamerica-northeast1.rep.googleapis.com/', 'northamerica-northeast2': 'https://netapp.northamerica-northeast2.rep.googleapis.com/', 'northamerica-south1': 'https://netapp.northamerica-south1.rep.googleapis.com/', 'southamerica-east1': 'https://netapp.southamerica-east1.rep.googleapis.com/', 'southamerica-west1': 'https://netapp.southamerica-west1.rep.googleapis.com/', 'us-central1': 'https://netapp.us-central1.rep.googleapis.com/', 'us-east1': 'https://netapp.us-east1.rep.googleapis.com/', 'us-east4': 'https://netapp.us-east4.rep.googleapis.com/', 'us-east5': 'https://netapp.us-east5.rep.googleapis.com/', 'us-east7': 'https://netapp.us-east7.rep.googleapis.com/', 'us-south1': 'https://netapp.us-south1.rep.googleapis.com/', 'us-west1': 'https://netapp.us-west1.rep.googleapis.com/', 'us-west2': 'https://netapp.us-west2.rep.googleapis.com/', 'us-west3': 'https://netapp.us-west3.rep.googleapis.com/', 'us-west4': 'https://netapp.us-west4.rep.googleapis.com/'}),
        'v1beta1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.netapp.v1beta1',
                    client_classpath='netapp_v1beta1_client.NetappV1beta1',
                    base_url='https://netapp.googleapis.com/',
                    messages_modulepath='netapp_v1beta1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'africa-south1': 'https://netapp.africa-south1.rep.googleapis.com/', 'asia-east1': 'https://netapp.asia-east1.rep.googleapis.com/', 'asia-east2': 'https://netapp.asia-east2.rep.googleapis.com/', 'asia-northeast1': 'https://netapp.asia-northeast1.rep.googleapis.com/', 'asia-northeast2': 'https://netapp.asia-northeast2.rep.googleapis.com/', 'asia-northeast3': 'https://netapp.asia-northeast3.rep.googleapis.com/', 'asia-south1': 'https://netapp.asia-south1.rep.googleapis.com/', 'asia-south2': 'https://netapp.asia-south2.rep.googleapis.com/', 'asia-southeast1': 'https://netapp.asia-southeast1.rep.googleapis.com/', 'asia-southeast2': 'https://netapp.asia-southeast2.rep.googleapis.com/', 'australia-southeast1': 'https://netapp.australia-southeast1.rep.googleapis.com/', 'australia-southeast2': 'https://netapp.australia-southeast2.rep.googleapis.com/', 'europe-central2': 'https://netapp.europe-central2.rep.googleapis.com/', 'europe-north1': 'https://netapp.europe-north1.rep.googleapis.com/', 'europe-north2': 'https://netapp.europe-north2.rep.googleapis.com/', 'europe-southwest1': 'https://netapp.europe-southwest1.rep.googleapis.com/', 'europe-west1': 'https://netapp.europe-west1.rep.googleapis.com/', 'europe-west10': 'https://netapp.europe-west10.rep.googleapis.com/', 'europe-west12': 'https://netapp.europe-west12.rep.googleapis.com/', 'europe-west2': 'https://netapp.europe-west2.rep.googleapis.com/', 'europe-west3': 'https://netapp.europe-west3.rep.googleapis.com/', 'europe-west4': 'https://netapp.europe-west4.rep.googleapis.com/', 'europe-west6': 'https://netapp.europe-west6.rep.googleapis.com/', 'europe-west8': 'https://netapp.europe-west8.rep.googleapis.com/', 'europe-west9': 'https://netapp.europe-west9.rep.googleapis.com/', 'me-central1': 'https://netapp.me-central1.rep.googleapis.com/', 'me-central2': 'https://netapp.me-central2.rep.googleapis.com/', 'me-west1': 'https://netapp.me-west1.rep.googleapis.com/', 'northamerica-northeast1': 'https://netapp.northamerica-northeast1.rep.googleapis.com/', 'northamerica-northeast2': 'https://netapp.northamerica-northeast2.rep.googleapis.com/', 'northamerica-south1': 'https://netapp.northamerica-south1.rep.googleapis.com/', 'southamerica-east1': 'https://netapp.southamerica-east1.rep.googleapis.com/', 'southamerica-west1': 'https://netapp.southamerica-west1.rep.googleapis.com/', 'us-central1': 'https://netapp.us-central1.rep.googleapis.com/', 'us-east1': 'https://netapp.us-east1.rep.googleapis.com/', 'us-east4': 'https://netapp.us-east4.rep.googleapis.com/', 'us-east5': 'https://netapp.us-east5.rep.googleapis.com/', 'us-east7': 'https://netapp.us-east7.rep.googleapis.com/', 'us-south1': 'https://netapp.us-south1.rep.googleapis.com/', 'us-west1': 'https://netapp.us-west1.rep.googleapis.com/', 'us-west2': 'https://netapp.us-west2.rep.googleapis.com/', 'us-west3': 'https://netapp.us-west3.rep.googleapis.com/', 'us-west4': 'https://netapp.us-west4.rep.googleapis.com/'}),
    },
    'networkconnectivity': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.networkconnectivity.v1',
                    client_classpath='networkconnectivity_v1_client.NetworkconnectivityV1',
                    base_url='https://networkconnectivity.googleapis.com/',
                    messages_modulepath='networkconnectivity_v1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.networkconnectivity.v1alpha1',
                    client_classpath='networkconnectivity_v1alpha1_client.NetworkconnectivityV1alpha1',
                    base_url='https://networkconnectivity.googleapis.com/',
                    messages_modulepath='networkconnectivity_v1alpha1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.networkconnectivity.v1beta',
                    client_classpath='networkconnectivity_v1beta_client.NetworkconnectivityV1beta',
                    base_url='https://networkconnectivity.googleapis.com/',
                    messages_modulepath='networkconnectivity_v1beta_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'networkmanagement': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.networkmanagement.v1',
                    client_classpath='networkmanagement_v1_client.NetworkmanagementV1',
                    base_url='https://networkmanagement.googleapis.com/',
                    messages_modulepath='networkmanagement_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.networkmanagement.v1alpha1',
                    client_classpath='networkmanagement_v1alpha1_client.NetworkmanagementV1alpha1',
                    base_url='https://networkmanagement.googleapis.com/',
                    messages_modulepath='networkmanagement_v1alpha1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.networkmanagement.v1beta1',
                    client_classpath='networkmanagement_v1beta1_client.NetworkmanagementV1beta1',
                    base_url='https://networkmanagement.googleapis.com/',
                    messages_modulepath='networkmanagement_v1beta1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'networksecurity': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.networksecurity.v1',
                    client_classpath='networksecurity_v1_client.NetworksecurityV1',
                    base_url='https://networksecurity.googleapis.com/',
                    messages_modulepath='networksecurity_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.networksecurity.v1alpha1',
                    client_classpath='networksecurity_v1alpha1_client.NetworksecurityV1alpha1',
                    base_url='https://networksecurity.googleapis.com/',
                    messages_modulepath='networksecurity_v1alpha1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.networksecurity.v1beta1',
                    client_classpath='networksecurity_v1beta1_client.NetworksecurityV1beta1',
                    base_url='https://networksecurity.googleapis.com/',
                    messages_modulepath='networksecurity_v1beta1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'networkservices': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.networkservices.v1',
                    client_classpath='networkservices_v1_client.NetworkservicesV1',
                    base_url='https://networkservices.googleapis.com/',
                    messages_modulepath='networkservices_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.networkservices.v1alpha1',
                    client_classpath='networkservices_v1alpha1_client.NetworkservicesV1alpha1',
                    base_url='https://networkservices.googleapis.com/',
                    messages_modulepath='networkservices_v1alpha1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.networkservices.v1beta1',
                    client_classpath='networkservices_v1beta1_client.NetworkservicesV1beta1',
                    base_url='https://networkservices.googleapis.com/',
                    messages_modulepath='networkservices_v1beta1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'notebooks': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.notebooks.v1',
                    client_classpath='notebooks_v1_client.NotebooksV1',
                    base_url='https://notebooks.googleapis.com/',
                    messages_modulepath='notebooks_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.notebooks.v1beta1',
                    client_classpath='notebooks_v1beta1_client.NotebooksV1beta1',
                    base_url='https://notebooks.googleapis.com/',
                    messages_modulepath='notebooks_v1beta1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v2':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.notebooks.v2',
                    client_classpath='notebooks_v2_client.NotebooksV2',
                    base_url='https://notebooks.googleapis.com/',
                    messages_modulepath='notebooks_v2_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'observability': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.observability.v1',
                    client_classpath='observability_v1_client.ObservabilityV1',
                    base_url='https://observability.googleapis.com/',
                    messages_modulepath='observability_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'ondemandscanning': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.ondemandscanning.v1',
                    client_classpath='ondemandscanning_v1_client.OndemandscanningV1',
                    base_url='https://ondemandscanning.googleapis.com/',
                    messages_modulepath='ondemandscanning_v1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.ondemandscanning.v1beta1',
                    client_classpath='ondemandscanning_v1beta1_client.OndemandscanningV1beta1',
                    base_url='https://ondemandscanning.googleapis.com/',
                    messages_modulepath='ondemandscanning_v1beta1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'oracledatabase': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.oracledatabase.v1',
                    client_classpath='oracledatabase_v1_client.OracledatabaseV1',
                    base_url='https://oracledatabase.googleapis.com/',
                    messages_modulepath='oracledatabase_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.oracledatabase.v1alpha',
                    client_classpath='oracledatabase_v1alpha_client.OracledatabaseV1alpha',
                    base_url='https://oracledatabase.googleapis.com/',
                    messages_modulepath='oracledatabase_v1alpha_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'orglifecycle': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.orglifecycle.v1',
                    client_classpath='orglifecycle_v1_client.OrglifecycleV1',
                    base_url='https://orglifecycle.googleapis.com/',
                    messages_modulepath='orglifecycle_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'orgpolicy': {
        'v2':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.orgpolicy.v2',
                    client_classpath='orgpolicy_v2_client.OrgpolicyV2',
                    base_url='https://orgpolicy.googleapis.com/',
                    messages_modulepath='orgpolicy_v2_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'osconfig': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.osconfig.v1',
                    client_classpath='osconfig_v1_client.OsconfigV1',
                    base_url='https://osconfig.googleapis.com/',
                    messages_modulepath='osconfig_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.osconfig.v1alpha',
                    client_classpath='osconfig_v1alpha_client.OsconfigV1alpha',
                    base_url='https://osconfig.googleapis.com/',
                    messages_modulepath='osconfig_v1alpha_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.osconfig.v1beta',
                    client_classpath='osconfig_v1beta_client.OsconfigV1beta',
                    base_url='https://osconfig.googleapis.com/',
                    messages_modulepath='osconfig_v1beta_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v2':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.osconfig.v2',
                    client_classpath='osconfig_v2_client.OsconfigV2',
                    base_url='https://osconfig.googleapis.com/',
                    messages_modulepath='osconfig_v2_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v2alpha':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.osconfig.v2alpha',
                    client_classpath='osconfig_v2alpha_client.OsconfigV2alpha',
                    base_url='https://osconfig.googleapis.com/',
                    messages_modulepath='osconfig_v2alpha_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v2beta':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.osconfig.v2beta',
                    client_classpath='osconfig_v2beta_client.OsconfigV2beta',
                    base_url='https://osconfig.googleapis.com/',
                    messages_modulepath='osconfig_v2beta_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'oslogin': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.oslogin.v1',
                    client_classpath='oslogin_v1_client.OsloginV1',
                    base_url='https://oslogin.googleapis.com/',
                    messages_modulepath='oslogin_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.oslogin.v1alpha',
                    client_classpath='oslogin_v1alpha_client.OsloginV1alpha',
                    base_url='https://oslogin.googleapis.com/',
                    messages_modulepath='oslogin_v1alpha_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.oslogin.v1beta',
                    client_classpath='oslogin_v1beta_client.OsloginV1beta',
                    base_url='https://oslogin.googleapis.com/',
                    messages_modulepath='oslogin_v1beta_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'parallelstore': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.parallelstore.v1',
                    client_classpath='parallelstore_v1_client.ParallelstoreV1',
                    base_url='https://parallelstore.googleapis.com/',
                    messages_modulepath='parallelstore_v1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.parallelstore.v1alpha',
                    client_classpath='parallelstore_v1alpha_client.ParallelstoreV1alpha',
                    base_url='https://parallelstore.googleapis.com/',
                    messages_modulepath='parallelstore_v1alpha_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.parallelstore.v1beta',
                    client_classpath='parallelstore_v1beta_client.ParallelstoreV1beta',
                    base_url='https://parallelstore.googleapis.com/',
                    messages_modulepath='parallelstore_v1beta_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'parametermanager': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.parametermanager.v1',
                    client_classpath='parametermanager_v1_client.ParametermanagerV1',
                    base_url='https://parametermanager.googleapis.com/',
                    messages_modulepath='parametermanager_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'us-central1': 'https://parametermanager.us-central1.rep.googleapis.com/', 'europe-west1': 'https://parametermanager.europe-west1.rep.googleapis.com/', 'europe-west4': 'https://parametermanager.europe-west4.rep.googleapis.com/', 'us-east4': 'https://parametermanager.us-east4.rep.googleapis.com/', 'europe-west2': 'https://parametermanager.europe-west2.rep.googleapis.com/', 'europe-west3': 'https://parametermanager.europe-west3.rep.googleapis.com/', 'us-east7': 'https://parametermanager.us-east7.rep.googleapis.com/', 'us-central2': 'https://parametermanager.us-central2.rep.googleapis.com/', 'us-east1': 'https://parametermanager.us-east1.rep.googleapis.com/', 'us-east5': 'https://parametermanager.us-east5.rep.googleapis.com/', 'us-south1': 'https://parametermanager.us-south1.rep.googleapis.com/', 'us-west1': 'https://parametermanager.us-west1.rep.googleapis.com/', 'us-west2': 'https://parametermanager.us-west2.rep.googleapis.com/', 'us-west3': 'https://parametermanager.us-west3.rep.googleapis.com/', 'us-west4': 'https://parametermanager.us-west4.rep.googleapis.com/', 'asia-northeast1': 'https://parametermanager.asia-northeast1.rep.googleapis.com/', 'australia-southeast1': 'https://parametermanager.australia-southeast1.rep.googleapis.com/', 'australia-southeast2': 'https://parametermanager.australia-southeast2.rep.googleapis.com/', 'europe-west6': 'https://parametermanager.europe-west6.rep.googleapis.com/', 'europe-west8': 'https://parametermanager.europe-west8.rep.googleapis.com/', 'europe-west9': 'https://parametermanager.europe-west9.rep.googleapis.com/', 'me-central2': 'https://parametermanager.me-central2.rep.googleapis.com/', 'me-west1': 'https://parametermanager.me-west1.rep.googleapis.com/', 'northamerica-northeast1': 'https://parametermanager.northamerica-northeast1.rep.googleapis.com/', 'northamerica-northeast2': 'https://parametermanager.northamerica-northeast2.rep.googleapis.com/', 'europe-west12': 'https://parametermanager.europe-west12.rep.googleapis.com/', 'africa-south1': 'https://parametermanager.africa-south1.rep.googleapis.com/', 'asia-southeast1': 'https://parametermanager.asia-southeast1.rep.googleapis.com/', 'asia-southeast2': 'https://parametermanager.asia-southeast2.rep.googleapis.com/', 'me-central1': 'https://parametermanager.me-central1.rep.googleapis.com/', 'southamerica-east1': 'https://parametermanager.southamerica-east1.rep.googleapis.com/', 'asia-south1': 'https://parametermanager.asia-south1.rep.googleapis.com/', 'europe-west10': 'https://parametermanager.europe-west10.rep.googleapis.com/', 'europe-north1': 'https://parametermanager.europe-north1.rep.googleapis.com/', 'europe-central2': 'https://parametermanager.europe-central2.rep.googleapis.com/', 'europe-southwest1': 'https://parametermanager.europe-southwest1.rep.googleapis.com/', 'asia-south2': 'https://parametermanager.asia-south2.rep.googleapis.com/', 'asia-east1': 'https://parametermanager.asia-east1.rep.googleapis.com/', 'asia-east2': 'https://parametermanager.asia-east2.rep.googleapis.com/', 'asia-northeast2': 'https://parametermanager.asia-northeast2.rep.googleapis.com/', 'asia-northeast3': 'https://parametermanager.asia-northeast3.rep.googleapis.com/', 'northamerica-south1': 'https://parametermanager.northamerica-south1.rep.googleapis.com/', 'southamerica-west1': 'https://parametermanager.southamerica-west1.rep.googleapis.com/', 'europe-north2': 'https://parametermanager.europe-north2.rep.googleapis.com/', 'europe-west15': 'https://parametermanager.europe-west15.rep.googleapis.com/', 'us': 'https://parametermanager.us.rep.googleapis.com/', 'eu': 'https://parametermanager.eu.rep.googleapis.com/', 'ca': 'https://parametermanager.ca.rep.googleapis.com/', 'in': 'https://parametermanager.in.rep.googleapis.com/'}),
    },
    'policyanalyzer': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.policyanalyzer.v1',
                    client_classpath='policyanalyzer_v1_client.PolicyanalyzerV1',
                    base_url='https://policyanalyzer.googleapis.com/',
                    messages_modulepath='policyanalyzer_v1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.policyanalyzer.v1beta1',
                    client_classpath='policyanalyzer_v1beta1_client.PolicyanalyzerV1beta1',
                    base_url='https://policyanalyzer.googleapis.com/',
                    messages_modulepath='policyanalyzer_v1beta1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'policysimulator': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.policysimulator.v1',
                    client_classpath='policysimulator_v1_client.PolicysimulatorV1',
                    base_url='https://policysimulator.googleapis.com/',
                    messages_modulepath='policysimulator_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.policysimulator.v1alpha',
                    client_classpath='policysimulator_v1alpha_client.PolicysimulatorV1alpha',
                    base_url='https://policysimulator.googleapis.com/',
                    messages_modulepath='policysimulator_v1alpha_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.policysimulator.v1beta',
                    client_classpath='policysimulator_v1beta_client.PolicysimulatorV1beta',
                    base_url='https://policysimulator.googleapis.com/',
                    messages_modulepath='policysimulator_v1beta_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'policytroubleshooter': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.policytroubleshooter.v1',
                    client_classpath='policytroubleshooter_v1_client.PolicytroubleshooterV1',
                    base_url='https://policytroubleshooter.googleapis.com/',
                    messages_modulepath='policytroubleshooter_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.policytroubleshooter.v1beta',
                    client_classpath='policytroubleshooter_v1beta_client.PolicytroubleshooterV1beta',
                    base_url='https://policytroubleshooter.googleapis.com/',
                    messages_modulepath='policytroubleshooter_v1beta_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v2alpha1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.policytroubleshooter.v2alpha1',
                    client_classpath='policytroubleshooter_v2alpha1_client.PolicytroubleshooterV2alpha1',
                    base_url='https://policytroubleshooter.googleapis.com/',
                    messages_modulepath='policytroubleshooter_v2alpha1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v3':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.policytroubleshooter.v3',
                    client_classpath='policytroubleshooter_v3_client.PolicytroubleshooterV3',
                    base_url='https://policytroubleshooter.googleapis.com/',
                    messages_modulepath='policytroubleshooter_v3_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v3alpha':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.policytroubleshooter.v3alpha',
                    client_classpath='policytroubleshooter_v3alpha_client.PolicytroubleshooterV3alpha',
                    base_url='https://policytroubleshooter.googleapis.com/',
                    messages_modulepath='policytroubleshooter_v3alpha_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v3beta':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.policytroubleshooter.v3beta',
                    client_classpath='policytroubleshooter_v3beta_client.PolicytroubleshooterV3beta',
                    base_url='https://policytroubleshooter.googleapis.com/',
                    messages_modulepath='policytroubleshooter_v3beta_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'privateca': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.privateca.v1',
                    client_classpath='privateca_v1_client.PrivatecaV1',
                    base_url='https://privateca.googleapis.com/',
                    messages_modulepath='privateca_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'privilegedaccessmanager': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.privilegedaccessmanager.v1',
                    client_classpath='privilegedaccessmanager_v1_client.PrivilegedaccessmanagerV1',
                    base_url='https://privilegedaccessmanager.googleapis.com/',
                    messages_modulepath='privilegedaccessmanager_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.privilegedaccessmanager.v1alpha',
                    client_classpath='privilegedaccessmanager_v1alpha_client.PrivilegedaccessmanagerV1alpha',
                    base_url='https://privilegedaccessmanager.googleapis.com/',
                    messages_modulepath='privilegedaccessmanager_v1alpha_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.privilegedaccessmanager.v1beta',
                    client_classpath='privilegedaccessmanager_v1beta_client.PrivilegedaccessmanagerV1beta',
                    base_url='https://privilegedaccessmanager.googleapis.com/',
                    messages_modulepath='privilegedaccessmanager_v1beta_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'publicca': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.publicca.v1',
                    client_classpath='publicca_v1_client.PubliccaV1',
                    base_url='https://publicca.googleapis.com/',
                    messages_modulepath='publicca_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.publicca.v1alpha1',
                    client_classpath='publicca_v1alpha1_client.PubliccaV1alpha1',
                    base_url='https://publicca.googleapis.com/',
                    messages_modulepath='publicca_v1alpha1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.publicca.v1beta1',
                    client_classpath='publicca_v1beta1_client.PubliccaV1beta1',
                    base_url='https://publicca.googleapis.com/',
                    messages_modulepath='publicca_v1beta1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'pubsub': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.pubsub.v1',
                    client_classpath='pubsub_v1_client.PubsubV1',
                    base_url='https://pubsub.googleapis.com/',
                    messages_modulepath='pubsub_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'africa-south1': 'https://pubsub.africa-south1.rep.googleapis.com/', 'asia-east1': 'https://pubsub.asia-east1.rep.googleapis.com/', 'asia-east2': 'https://pubsub.asia-east2.rep.googleapis.com/', 'asia-northeast1': 'https://pubsub.asia-northeast1.rep.googleapis.com/', 'asia-northeast2': 'https://pubsub.asia-northeast2.rep.googleapis.com/', 'asia-northeast3': 'https://pubsub.asia-northeast3.rep.googleapis.com/', 'asia-south1': 'https://pubsub.asia-south1.rep.googleapis.com/', 'asia-south2': 'https://pubsub.asia-south2.rep.googleapis.com/', 'asia-southeast1': 'https://pubsub.asia-southeast1.rep.googleapis.com/', 'asia-southeast2': 'https://pubsub.asia-southeast2.rep.googleapis.com/', 'asia-southeast3': 'https://pubsub.asia-southeast3.rep.googleapis.com/', 'australia-southeast1': 'https://pubsub.australia-southeast1.rep.googleapis.com/', 'australia-southeast2': 'https://pubsub.australia-southeast2.rep.googleapis.com/', 'europe-central2': 'https://pubsub.europe-central2.rep.googleapis.com/', 'europe-north1': 'https://pubsub.europe-north1.rep.googleapis.com/', 'europe-north2': 'https://pubsub.europe-north2.rep.googleapis.com/', 'europe-southwest1': 'https://pubsub.europe-southwest1.rep.googleapis.com/', 'europe-west1': 'https://pubsub.europe-west1.rep.googleapis.com/', 'europe-west2': 'https://pubsub.europe-west2.rep.googleapis.com/', 'europe-west3': 'https://pubsub.europe-west3.rep.googleapis.com/', 'europe-west4': 'https://pubsub.europe-west4.rep.googleapis.com/', 'europe-west6': 'https://pubsub.europe-west6.rep.googleapis.com/', 'europe-west8': 'https://pubsub.europe-west8.rep.googleapis.com/', 'europe-west9': 'https://pubsub.europe-west9.rep.googleapis.com/', 'europe-west10': 'https://pubsub.europe-west10.rep.googleapis.com/', 'europe-west12': 'https://pubsub.europe-west12.rep.googleapis.com/', 'me-central1': 'https://pubsub.me-central1.rep.googleapis.com/', 'me-central2': 'https://pubsub.me-central2.rep.googleapis.com/', 'me-west1': 'https://pubsub.me-west1.rep.googleapis.com/', 'northamerica-northeast1': 'https://pubsub.northamerica-northeast1.rep.googleapis.com/', 'northamerica-northeast2': 'https://pubsub.northamerica-northeast2.rep.googleapis.com/', 'northamerica-south1': 'https://pubsub.northamerica-south1.rep.googleapis.com/', 'southamerica-east1': 'https://pubsub.southamerica-east1.rep.googleapis.com/', 'southamerica-west1': 'https://pubsub.southamerica-west1.rep.googleapis.com/', 'us-central1': 'https://pubsub.us-central1.rep.googleapis.com/', 'us-central2': 'https://pubsub.us-central2.rep.googleapis.com/', 'us-east1': 'https://pubsub.us-east1.rep.googleapis.com/', 'us-east4': 'https://pubsub.us-east4.rep.googleapis.com/', 'us-east5': 'https://pubsub.us-east5.rep.googleapis.com/', 'us-east7': 'https://pubsub.us-east7.rep.googleapis.com/', 'us-south1': 'https://pubsub.us-south1.rep.googleapis.com/', 'us-west1': 'https://pubsub.us-west1.rep.googleapis.com/', 'us-west2': 'https://pubsub.us-west2.rep.googleapis.com/', 'us-west3': 'https://pubsub.us-west3.rep.googleapis.com/', 'us-west4': 'https://pubsub.us-west4.rep.googleapis.com/', 'us-west8': 'https://pubsub.us-west8.rep.googleapis.com/'}),
    },
    'pubsublite': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.pubsublite.v1',
                    client_classpath='pubsublite_v1_client.PubsubliteV1',
                    base_url='https://pubsublite.googleapis.com/',
                    messages_modulepath='pubsublite_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'recaptchaenterprise': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.recaptchaenterprise.v1',
                    client_classpath='recaptchaenterprise_v1_client.RecaptchaenterpriseV1',
                    base_url='https://recaptchaenterprise.googleapis.com/',
                    messages_modulepath='recaptchaenterprise_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'recommender': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.recommender.v1',
                    client_classpath='recommender_v1_client.RecommenderV1',
                    base_url='https://recommender.googleapis.com/',
                    messages_modulepath='recommender_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha2':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.recommender.v1alpha2',
                    client_classpath='recommender_v1alpha2_client.RecommenderV1alpha2',
                    base_url='https://recommender.googleapis.com/',
                    messages_modulepath='recommender_v1alpha2_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.recommender.v1beta1',
                    client_classpath='recommender_v1beta1_client.RecommenderV1beta1',
                    base_url='https://recommender.googleapis.com/',
                    messages_modulepath='recommender_v1beta1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'redis': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.redis.v1',
                    client_classpath='redis_v1_client.RedisV1',
                    base_url='https://redis.googleapis.com/',
                    messages_modulepath='redis_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.redis.v1alpha1',
                    client_classpath='redis_v1alpha1_client.RedisV1alpha1',
                    base_url='https://redis.googleapis.com/',
                    messages_modulepath='redis_v1alpha1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.redis.v1beta1',
                    client_classpath='redis_v1beta1_client.RedisV1beta1',
                    base_url='https://redis.googleapis.com/',
                    messages_modulepath='redis_v1beta1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'remotebuildexecution': {
        'v1alpha':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.remotebuildexecution.v1alpha',
                    client_classpath='remotebuildexecution_v1alpha_client.RemotebuildexecutionV1alpha',
                    base_url='https://admin-remotebuildexecution.googleapis.com/',
                    messages_modulepath='remotebuildexecution_v1alpha_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'run': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.run.v1',
                    client_classpath='run_v1_client.RunV1',
                    base_url='https://run.googleapis.com/',
                    messages_modulepath='run_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'africa-south1': 'https://run.africa-south1.rep.googleapis.com/', 'asia-east1': 'https://run.asia-east1.rep.googleapis.com/', 'asia-east2': 'https://run.asia-east2.rep.googleapis.com/', 'asia-northeast1': 'https://run.asia-northeast1.rep.googleapis.com/', 'asia-northeast2': 'https://run.asia-northeast2.rep.googleapis.com/', 'asia-northeast3': 'https://run.asia-northeast3.rep.googleapis.com/', 'asia-south1': 'https://run.asia-south1.rep.googleapis.com/', 'asia-south2': 'https://run.asia-south2.rep.googleapis.com/', 'asia-southeast1': 'https://run.asia-southeast1.rep.googleapis.com/', 'asia-southeast2': 'https://run.asia-southeast2.rep.googleapis.com/', 'australia-southeast1': 'https://run.australia-southeast1.rep.googleapis.com/', 'australia-southeast2': 'https://run.australia-southeast2.rep.googleapis.com/', 'europe-central2': 'https://run.europe-central2.rep.googleapis.com/', 'europe-north1': 'https://run.europe-north1.rep.googleapis.com/', 'europe-north2': 'https://run.europe-north2.rep.googleapis.com/', 'europe-southwest1': 'https://run.europe-southwest1.rep.googleapis.com/', 'europe-west1': 'https://run.europe-west1.rep.googleapis.com/', 'europe-west10': 'https://run.europe-west10.rep.googleapis.com/', 'europe-west12': 'https://run.europe-west12.rep.googleapis.com/', 'europe-west15': 'https://run.europe-west15.rep.googleapis.com/', 'europe-west2': 'https://run.europe-west2.rep.googleapis.com/', 'europe-west3': 'https://run.europe-west3.rep.googleapis.com/', 'europe-west4': 'https://run.europe-west4.rep.googleapis.com/', 'europe-west6': 'https://run.europe-west6.rep.googleapis.com/', 'europe-west8': 'https://run.europe-west8.rep.googleapis.com/', 'europe-west9': 'https://run.europe-west9.rep.googleapis.com/', 'me-central1': 'https://run.me-central1.rep.googleapis.com/', 'me-central2': 'https://run.me-central2.rep.googleapis.com/', 'me-west1': 'https://run.me-west1.rep.googleapis.com/', 'northamerica-northeast1': 'https://run.northamerica-northeast1.rep.googleapis.com/', 'northamerica-northeast2': 'https://run.northamerica-northeast2.rep.googleapis.com/', 'northamerica-south1': 'https://run.northamerica-south1.rep.googleapis.com/', 'southamerica-east1': 'https://run.southamerica-east1.rep.googleapis.com/', 'southamerica-west1': 'https://run.southamerica-west1.rep.googleapis.com/', 'us-central1': 'https://run.us-central1.rep.googleapis.com/', 'us-central2': 'https://run.us-central2.rep.googleapis.com/', 'us-east1': 'https://run.us-east1.rep.googleapis.com/', 'us-east4': 'https://run.us-east4.rep.googleapis.com/', 'us-east5': 'https://run.us-east5.rep.googleapis.com/', 'us-east7': 'https://run.us-east7.rep.googleapis.com/', 'us-south1': 'https://run.us-south1.rep.googleapis.com/', 'us-west1': 'https://run.us-west1.rep.googleapis.com/', 'us-west2': 'https://run.us-west2.rep.googleapis.com/', 'us-west3': 'https://run.us-west3.rep.googleapis.com/', 'us-west4': 'https://run.us-west4.rep.googleapis.com/', 'us-west8': 'https://run.us-west8.rep.googleapis.com/'}),
        'v1alpha1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.run.v1alpha1',
                    client_classpath='run_v1alpha1_client.RunV1alpha1',
                    base_url='https://run.googleapis.com/',
                    messages_modulepath='run_v1alpha1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v2':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.run.v2',
                    client_classpath='run_v2_client.RunV2',
                    base_url='https://run.googleapis.com/',
                    messages_modulepath='run_v2_messages'),
                gapic=GapicClientDef(
                    class_path='googlecloudsdk.generated_clients.gapic_wrappers.run.v2'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'africa-south1': 'https://run.africa-south1.rep.googleapis.com/', 'asia-east1': 'https://run.asia-east1.rep.googleapis.com/', 'asia-east2': 'https://run.asia-east2.rep.googleapis.com/', 'asia-northeast1': 'https://run.asia-northeast1.rep.googleapis.com/', 'asia-northeast2': 'https://run.asia-northeast2.rep.googleapis.com/', 'asia-northeast3': 'https://run.asia-northeast3.rep.googleapis.com/', 'asia-south1': 'https://run.asia-south1.rep.googleapis.com/', 'asia-south2': 'https://run.asia-south2.rep.googleapis.com/', 'asia-southeast1': 'https://run.asia-southeast1.rep.googleapis.com/', 'asia-southeast2': 'https://run.asia-southeast2.rep.googleapis.com/', 'australia-southeast1': 'https://run.australia-southeast1.rep.googleapis.com/', 'australia-southeast2': 'https://run.australia-southeast2.rep.googleapis.com/', 'europe-central2': 'https://run.europe-central2.rep.googleapis.com/', 'europe-north1': 'https://run.europe-north1.rep.googleapis.com/', 'europe-north2': 'https://run.europe-north2.rep.googleapis.com/', 'europe-southwest1': 'https://run.europe-southwest1.rep.googleapis.com/', 'europe-west1': 'https://run.europe-west1.rep.googleapis.com/', 'europe-west10': 'https://run.europe-west10.rep.googleapis.com/', 'europe-west12': 'https://run.europe-west12.rep.googleapis.com/', 'europe-west15': 'https://run.europe-west15.rep.googleapis.com/', 'europe-west2': 'https://run.europe-west2.rep.googleapis.com/', 'europe-west3': 'https://run.europe-west3.rep.googleapis.com/', 'europe-west4': 'https://run.europe-west4.rep.googleapis.com/', 'europe-west6': 'https://run.europe-west6.rep.googleapis.com/', 'europe-west8': 'https://run.europe-west8.rep.googleapis.com/', 'europe-west9': 'https://run.europe-west9.rep.googleapis.com/', 'me-central1': 'https://run.me-central1.rep.googleapis.com/', 'me-central2': 'https://run.me-central2.rep.googleapis.com/', 'me-west1': 'https://run.me-west1.rep.googleapis.com/', 'northamerica-northeast1': 'https://run.northamerica-northeast1.rep.googleapis.com/', 'northamerica-northeast2': 'https://run.northamerica-northeast2.rep.googleapis.com/', 'northamerica-south1': 'https://run.northamerica-south1.rep.googleapis.com/', 'southamerica-east1': 'https://run.southamerica-east1.rep.googleapis.com/', 'southamerica-west1': 'https://run.southamerica-west1.rep.googleapis.com/', 'us-central1': 'https://run.us-central1.rep.googleapis.com/', 'us-central2': 'https://run.us-central2.rep.googleapis.com/', 'us-east1': 'https://run.us-east1.rep.googleapis.com/', 'us-east4': 'https://run.us-east4.rep.googleapis.com/', 'us-east5': 'https://run.us-east5.rep.googleapis.com/', 'us-east7': 'https://run.us-east7.rep.googleapis.com/', 'us-south1': 'https://run.us-south1.rep.googleapis.com/', 'us-west1': 'https://run.us-west1.rep.googleapis.com/', 'us-west2': 'https://run.us-west2.rep.googleapis.com/', 'us-west3': 'https://run.us-west3.rep.googleapis.com/', 'us-west4': 'https://run.us-west4.rep.googleapis.com/', 'us-west8': 'https://run.us-west8.rep.googleapis.com/'}),
    },
    'runapps': {
        'v1alpha1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.runapps.v1alpha1',
                    client_classpath='runapps_v1alpha1_client.RunappsV1alpha1',
                    base_url='https://runapps.googleapis.com/',
                    messages_modulepath='runapps_v1alpha1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'runtimeconfig': {
        'v1beta1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.runtimeconfig.v1beta1',
                    client_classpath='runtimeconfig_v1beta1_client.RuntimeconfigV1beta1',
                    base_url='https://runtimeconfig.googleapis.com/',
                    messages_modulepath='runtimeconfig_v1beta1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'saasservicemgmt': {
        'v1alpha1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.saasservicemgmt.v1alpha1',
                    client_classpath='saasservicemgmt_v1alpha1_client.SaasservicemgmtV1alpha1',
                    base_url='https://saasservicemgmt.googleapis.com/',
                    messages_modulepath='saasservicemgmt_v1alpha1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.saasservicemgmt.v1beta1',
                    client_classpath='saasservicemgmt_v1beta1_client.SaasservicemgmtV1beta1',
                    base_url='https://saasservicemgmt.googleapis.com/',
                    messages_modulepath='saasservicemgmt_v1beta1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'sasportal': {
        'v1alpha1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.sasportal.v1alpha1',
                    client_classpath='sasportal_v1alpha1_client.SasportalV1alpha1',
                    base_url='https://sasportal.googleapis.com/',
                    messages_modulepath='sasportal_v1alpha1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'sddc': {
        'v1alpha1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.sddc.v1alpha1',
                    client_classpath='sddc_v1alpha1_client.SddcV1alpha1',
                    base_url='https://sddc.googleapis.com/',
                    messages_modulepath='sddc_v1alpha1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'seclm': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.seclm.v1',
                    client_classpath='seclm_v1_client.SeclmV1',
                    base_url='https://seclm.googleapis.com/',
                    messages_modulepath='seclm_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.seclm.v1alpha',
                    client_classpath='seclm_v1alpha_client.SeclmV1alpha',
                    base_url='https://seclm.googleapis.com/',
                    messages_modulepath='seclm_v1alpha_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'secretmanager': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.secretmanager.v1',
                    client_classpath='secretmanager_v1_client.SecretmanagerV1',
                    base_url='https://secretmanager.googleapis.com/',
                    messages_modulepath='secretmanager_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'me-central2': 'https://secretmanager.me-central2.rep.googleapis.com/', 'africa-south1': 'https://secretmanager.africa-south1.rep.googleapis.com/', 'asia-east1': 'https://secretmanager.asia-east1.rep.googleapis.com/', 'asia-east2': 'https://secretmanager.asia-east2.rep.googleapis.com/', 'asia-northeast1': 'https://secretmanager.asia-northeast1.rep.googleapis.com/', 'asia-northeast2': 'https://secretmanager.asia-northeast2.rep.googleapis.com/', 'asia-northeast3': 'https://secretmanager.asia-northeast3.rep.googleapis.com/', 'asia-south1': 'https://secretmanager.asia-south1.rep.googleapis.com/', 'asia-south2': 'https://secretmanager.asia-south2.rep.googleapis.com/', 'asia-southeast1': 'https://secretmanager.asia-southeast1.rep.googleapis.com/', 'asia-southeast2': 'https://secretmanager.asia-southeast2.rep.googleapis.com/', 'asia-southeast3': 'https://secretmanager.asia-southeast3.rep.googleapis.com/', 'australia-southeast1': 'https://secretmanager.australia-southeast1.rep.googleapis.com/', 'australia-southeast2': 'https://secretmanager.australia-southeast2.rep.googleapis.com/', 'europe-central2': 'https://secretmanager.europe-central2.rep.googleapis.com/', 'europe-north1': 'https://secretmanager.europe-north1.rep.googleapis.com/', 'europe-north2': 'https://secretmanager.europe-north2.rep.googleapis.com/', 'europe-southwest1': 'https://secretmanager.europe-southwest1.rep.googleapis.com/', 'europe-west1': 'https://secretmanager.europe-west1.rep.googleapis.com/', 'europe-west10': 'https://secretmanager.europe-west10.rep.googleapis.com/', 'europe-west12': 'https://secretmanager.europe-west12.rep.googleapis.com/', 'europe-west15': 'https://secretmanager.europe-west15.rep.googleapis.com/', 'europe-west2': 'https://secretmanager.europe-west2.rep.googleapis.com/', 'europe-west3': 'https://secretmanager.europe-west3.rep.googleapis.com/', 'europe-west4': 'https://secretmanager.europe-west4.rep.googleapis.com/', 'europe-west6': 'https://secretmanager.europe-west6.rep.googleapis.com/', 'europe-west8': 'https://secretmanager.europe-west8.rep.googleapis.com/', 'europe-west9': 'https://secretmanager.europe-west9.rep.googleapis.com/', 'me-central1': 'https://secretmanager.me-central1.rep.googleapis.com/', 'me-west1': 'https://secretmanager.me-west1.rep.googleapis.com/', 'northamerica-northeast1': 'https://secretmanager.northamerica-northeast1.rep.googleapis.com/', 'northamerica-northeast2': 'https://secretmanager.northamerica-northeast2.rep.googleapis.com/', 'northamerica-south1': 'https://secretmanager.northamerica-south1.rep.googleapis.com/', 'southamerica-east1': 'https://secretmanager.southamerica-east1.rep.googleapis.com/', 'southamerica-west1': 'https://secretmanager.southamerica-west1.rep.googleapis.com/', 'us-central1': 'https://secretmanager.us-central1.rep.googleapis.com/', 'us-central2': 'https://secretmanager.us-central2.rep.googleapis.com/', 'us-east1': 'https://secretmanager.us-east1.rep.googleapis.com/', 'us-east4': 'https://secretmanager.us-east4.rep.googleapis.com/', 'us-east5': 'https://secretmanager.us-east5.rep.googleapis.com/', 'us-east7': 'https://secretmanager.us-east7.rep.googleapis.com/', 'us-south1': 'https://secretmanager.us-south1.rep.googleapis.com/', 'us-west1': 'https://secretmanager.us-west1.rep.googleapis.com/', 'us-west2': 'https://secretmanager.us-west2.rep.googleapis.com/', 'us-west3': 'https://secretmanager.us-west3.rep.googleapis.com/', 'us-west4': 'https://secretmanager.us-west4.rep.googleapis.com/', 'us': 'https://secretmanager.us.rep.googleapis.com/', 'eu': 'https://secretmanager.eu.rep.googleapis.com/', 'ca': 'https://secretmanager.ca.rep.googleapis.com/', 'in': 'https://secretmanager.in.rep.googleapis.com/'}),
        'v1beta2':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.secretmanager.v1beta2',
                    client_classpath='secretmanager_v1beta2_client.SecretmanagerV1beta2',
                    base_url='https://secretmanager.googleapis.com/',
                    messages_modulepath='secretmanager_v1beta2_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'me-central2': 'https://secretmanager.me-central2.rep.googleapis.com/', 'africa-south1': 'https://secretmanager.africa-south1.rep.googleapis.com/', 'asia-east1': 'https://secretmanager.asia-east1.rep.googleapis.com/', 'asia-east2': 'https://secretmanager.asia-east2.rep.googleapis.com/', 'asia-northeast1': 'https://secretmanager.asia-northeast1.rep.googleapis.com/', 'asia-northeast2': 'https://secretmanager.asia-northeast2.rep.googleapis.com/', 'asia-northeast3': 'https://secretmanager.asia-northeast3.rep.googleapis.com/', 'asia-south1': 'https://secretmanager.asia-south1.rep.googleapis.com/', 'asia-south2': 'https://secretmanager.asia-south2.rep.googleapis.com/', 'asia-southeast1': 'https://secretmanager.asia-southeast1.rep.googleapis.com/', 'asia-southeast2': 'https://secretmanager.asia-southeast2.rep.googleapis.com/', 'asia-southeast3': 'https://secretmanager.asia-southeast3.rep.googleapis.com/', 'australia-southeast1': 'https://secretmanager.australia-southeast1.rep.googleapis.com/', 'australia-southeast2': 'https://secretmanager.australia-southeast2.rep.googleapis.com/', 'europe-central2': 'https://secretmanager.europe-central2.rep.googleapis.com/', 'europe-north1': 'https://secretmanager.europe-north1.rep.googleapis.com/', 'europe-north2': 'https://secretmanager.europe-north2.rep.googleapis.com/', 'europe-southwest1': 'https://secretmanager.europe-southwest1.rep.googleapis.com/', 'europe-west1': 'https://secretmanager.europe-west1.rep.googleapis.com/', 'europe-west10': 'https://secretmanager.europe-west10.rep.googleapis.com/', 'europe-west12': 'https://secretmanager.europe-west12.rep.googleapis.com/', 'europe-west15': 'https://secretmanager.europe-west15.rep.googleapis.com/', 'europe-west2': 'https://secretmanager.europe-west2.rep.googleapis.com/', 'europe-west3': 'https://secretmanager.europe-west3.rep.googleapis.com/', 'europe-west4': 'https://secretmanager.europe-west4.rep.googleapis.com/', 'europe-west6': 'https://secretmanager.europe-west6.rep.googleapis.com/', 'europe-west8': 'https://secretmanager.europe-west8.rep.googleapis.com/', 'europe-west9': 'https://secretmanager.europe-west9.rep.googleapis.com/', 'me-central1': 'https://secretmanager.me-central1.rep.googleapis.com/', 'me-west1': 'https://secretmanager.me-west1.rep.googleapis.com/', 'northamerica-northeast1': 'https://secretmanager.northamerica-northeast1.rep.googleapis.com/', 'northamerica-northeast2': 'https://secretmanager.northamerica-northeast2.rep.googleapis.com/', 'northamerica-south1': 'https://secretmanager.northamerica-south1.rep.googleapis.com/', 'southamerica-east1': 'https://secretmanager.southamerica-east1.rep.googleapis.com/', 'southamerica-west1': 'https://secretmanager.southamerica-west1.rep.googleapis.com/', 'us-central1': 'https://secretmanager.us-central1.rep.googleapis.com/', 'us-central2': 'https://secretmanager.us-central2.rep.googleapis.com/', 'us-east1': 'https://secretmanager.us-east1.rep.googleapis.com/', 'us-east4': 'https://secretmanager.us-east4.rep.googleapis.com/', 'us-east5': 'https://secretmanager.us-east5.rep.googleapis.com/', 'us-east7': 'https://secretmanager.us-east7.rep.googleapis.com/', 'us-south1': 'https://secretmanager.us-south1.rep.googleapis.com/', 'us-west1': 'https://secretmanager.us-west1.rep.googleapis.com/', 'us-west2': 'https://secretmanager.us-west2.rep.googleapis.com/', 'us-west3': 'https://secretmanager.us-west3.rep.googleapis.com/', 'us-west4': 'https://secretmanager.us-west4.rep.googleapis.com/', 'us': 'https://secretmanager.us.rep.googleapis.com/', 'eu': 'https://secretmanager.eu.rep.googleapis.com/', 'ca': 'https://secretmanager.ca.rep.googleapis.com/', 'in': 'https://secretmanager.in.rep.googleapis.com/'}),
    },
    'securesourcemanager': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.securesourcemanager.v1',
                    client_classpath='securesourcemanager_v1_client.SecuresourcemanagerV1',
                    base_url='https://securesourcemanager.googleapis.com/',
                    messages_modulepath='securesourcemanager_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'asia-east1': 'https://securesourcemanager.asia-east1.rep.googleapis.com/', 'asia-northeast1': 'https://securesourcemanager.asia-northeast1.rep.googleapis.com/', 'asia-northeast3': 'https://securesourcemanager.asia-northeast3.rep.googleapis.com/', 'australia-southeast1': 'https://securesourcemanager.australia-southeast1.rep.googleapis.com/', 'europe-west2': 'https://securesourcemanager.europe-west2.rep.googleapis.com/', 'europe-west4': 'https://securesourcemanager.europe-west4.rep.googleapis.com/', 'me-central2': 'https://securesourcemanager.me-central2.rep.googleapis.com/', 'me-west1': 'https://securesourcemanager.me-west1.rep.googleapis.com/', 'northamerica-northeast1': 'https://securesourcemanager.northamerica-northeast1.rep.googleapis.com/', 'us-central1': 'https://securesourcemanager.us-central1.rep.googleapis.com/', 'us-east1': 'https://securesourcemanager.us-east1.rep.googleapis.com/'}),
    },
    'securitycenter': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.securitycenter.v1',
                    client_classpath='securitycenter_v1_client.SecuritycenterV1',
                    base_url='https://securitycenter.googleapis.com/',
                    messages_modulepath='securitycenter_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'me-central2': 'https://securitycenter.me-central2.rep.googleapis.com/', 'us': 'https://securitycenter.us.rep.googleapis.com/', 'eu': 'https://securitycenter.eu.rep.googleapis.com/'}),
        'v1beta2':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.securitycenter.v1beta2',
                    client_classpath='securitycenter_v1beta2_client.SecuritycenterV1beta2',
                    base_url='https://securitycenter.googleapis.com/',
                    messages_modulepath='securitycenter_v1beta2_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'me-central2': 'https://securitycenter.me-central2.rep.googleapis.com/', 'us': 'https://securitycenter.us.rep.googleapis.com/', 'eu': 'https://securitycenter.eu.rep.googleapis.com/'}),
        'v2':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.securitycenter.v2',
                    client_classpath='securitycenter_v2_client.SecuritycenterV2',
                    base_url='https://securitycenter.googleapis.com/',
                    messages_modulepath='securitycenter_v2_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'me-central2': 'https://securitycenter.me-central2.rep.googleapis.com/', 'us': 'https://securitycenter.us.rep.googleapis.com/', 'eu': 'https://securitycenter.eu.rep.googleapis.com/'}),
    },
    'securitycentermanagement': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.securitycentermanagement.v1',
                    client_classpath='securitycentermanagement_v1_client.SecuritycentermanagementV1',
                    base_url='https://securitycentermanagement.googleapis.com/',
                    messages_modulepath='securitycentermanagement_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'securityposture': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.securityposture.v1',
                    client_classpath='securityposture_v1_client.SecuritypostureV1',
                    base_url='https://securityposture.googleapis.com/',
                    messages_modulepath='securityposture_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.securityposture.v1alpha',
                    client_classpath='securityposture_v1alpha_client.SecuritypostureV1alpha',
                    base_url='https://securityposture.googleapis.com/',
                    messages_modulepath='securityposture_v1alpha_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'serviceconsumermanagement': {
        'v1beta1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.serviceconsumermanagement.v1beta1',
                    client_classpath='serviceconsumermanagement_v1beta1_client.ServiceconsumermanagementV1beta1',
                    base_url='https://serviceconsumermanagement.googleapis.com/',
                    messages_modulepath='serviceconsumermanagement_v1beta1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'servicedirectory': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.servicedirectory.v1',
                    client_classpath='servicedirectory_v1_client.ServicedirectoryV1',
                    base_url='https://servicedirectory.googleapis.com/',
                    messages_modulepath='servicedirectory_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.servicedirectory.v1beta1',
                    client_classpath='servicedirectory_v1beta1_client.ServicedirectoryV1beta1',
                    base_url='https://servicedirectory.googleapis.com/',
                    messages_modulepath='servicedirectory_v1beta1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'servicemanagement': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.servicemanagement.v1',
                    client_classpath='servicemanagement_v1_client.ServicemanagementV1',
                    base_url='https://servicemanagement.googleapis.com/',
                    messages_modulepath='servicemanagement_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'servicenetworking': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.servicenetworking.v1',
                    client_classpath='servicenetworking_v1_client.ServicenetworkingV1',
                    base_url='https://servicenetworking.googleapis.com/',
                    messages_modulepath='servicenetworking_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.servicenetworking.v1beta',
                    client_classpath='servicenetworking_v1beta_client.ServicenetworkingV1beta',
                    base_url='https://servicenetworking.googleapis.com/',
                    messages_modulepath='servicenetworking_v1beta_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='https://servicenetworking.mtls.googleapis.com/',
                regional_endpoints={}),
    },
    'serviceusage': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.serviceusage.v1',
                    client_classpath='serviceusage_v1_client.ServiceusageV1',
                    base_url='https://serviceusage.googleapis.com/',
                    messages_modulepath='serviceusage_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.serviceusage.v1alpha',
                    client_classpath='serviceusage_v1alpha_client.ServiceusageV1alpha',
                    base_url='https://serviceusage.googleapis.com/',
                    messages_modulepath='serviceusage_v1alpha_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='https://serviceusage.mtls.googleapis.com/',
                regional_endpoints={}),
        'v1beta1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.serviceusage.v1beta1',
                    client_classpath='serviceusage_v1beta1_client.ServiceusageV1beta1',
                    base_url='https://serviceusage.googleapis.com/',
                    messages_modulepath='serviceusage_v1beta1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v2alpha':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.serviceusage.v2alpha',
                    client_classpath='serviceusage_v2alpha_client.ServiceusageV2alpha',
                    base_url='https://serviceusage.googleapis.com/',
                    messages_modulepath='serviceusage_v2alpha_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v2beta':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.serviceusage.v2beta',
                    client_classpath='serviceusage_v2beta_client.ServiceusageV2beta',
                    base_url='https://serviceusage.googleapis.com/',
                    messages_modulepath='serviceusage_v2beta_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'source': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.source.v1',
                    client_classpath='source_v1_client.SourceV1',
                    base_url='https://source.googleapis.com/',
                    messages_modulepath='source_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='https://source.mtls.googleapis.com/',
                regional_endpoints={}),
    },
    'sourcerepo': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.sourcerepo.v1',
                    client_classpath='sourcerepo_v1_client.SourcerepoV1',
                    base_url='https://sourcerepo.googleapis.com/',
                    messages_modulepath='sourcerepo_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'spanner': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.spanner.v1',
                    client_classpath='spanner_v1_client.SpannerV1',
                    base_url='https://spanner.googleapis.com/',
                    messages_modulepath='spanner_v1_messages'),
                gapic=GapicClientDef(
                    class_path='googlecloudsdk.generated_clients.gapic_wrappers.spanner.v1'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'europe-west8': 'https://spanner.europe-west8.rep.googleapis.com/', 'me-central2': 'https://spanner.me-central2.rep.googleapis.com/', 'us-central1': 'https://spanner.us-central1.rep.googleapis.com/', 'us-central2': 'https://spanner.us-central2.rep.googleapis.com/', 'us-east1': 'https://spanner.us-east1.rep.googleapis.com/', 'us-east4': 'https://spanner.us-east4.rep.googleapis.com/', 'us-east5': 'https://spanner.us-east5.rep.googleapis.com/', 'us-south1': 'https://spanner.us-south1.rep.googleapis.com/', 'us-west1': 'https://spanner.us-west1.rep.googleapis.com/', 'us-west2': 'https://spanner.us-west2.rep.googleapis.com/', 'us-west3': 'https://spanner.us-west3.rep.googleapis.com/', 'us-west4': 'https://spanner.us-west4.rep.googleapis.com/', 'us-west8': 'https://spanner.us-west8.rep.googleapis.com/', 'us-east7': 'https://spanner.us-east7.rep.googleapis.com/'}),
    },
    'speech': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.speech.v1',
                    client_classpath='speech_v1_client.SpeechV1',
                    base_url='https://speech.googleapis.com/',
                    messages_modulepath='speech_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'us-central1': 'https://speech.us-central1.rep.googleapis.com/', 'us-west1': 'https://speech.us-west1.rep.googleapis.com/', 'me-west1': 'https://speech.me-west1.rep.googleapis.com/', 'europe-west1': 'https://speech.europe-west1.rep.googleapis.com/', 'europe-west2': 'https://speech.europe-west2.rep.googleapis.com/', 'europe-west3': 'https://speech.europe-west3.rep.googleapis.com/', 'europe-west4': 'https://speech.europe-west4.rep.googleapis.com/'}),
        'v1p1beta1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.speech.v1p1beta1',
                    client_classpath='speech_v1p1beta1_client.SpeechV1p1beta1',
                    base_url='https://speech.googleapis.com/',
                    messages_modulepath='speech_v1p1beta1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'us-central1': 'https://speech.us-central1.rep.googleapis.com/', 'us-west1': 'https://speech.us-west1.rep.googleapis.com/', 'me-west1': 'https://speech.me-west1.rep.googleapis.com/', 'europe-west1': 'https://speech.europe-west1.rep.googleapis.com/', 'europe-west2': 'https://speech.europe-west2.rep.googleapis.com/', 'europe-west3': 'https://speech.europe-west3.rep.googleapis.com/', 'europe-west4': 'https://speech.europe-west4.rep.googleapis.com/'}),
        'v2':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.speech.v2',
                    client_classpath='speech_v2_client.SpeechV2',
                    base_url='https://speech.googleapis.com/',
                    messages_modulepath='speech_v2_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'us-central1': 'https://speech.us-central1.rep.googleapis.com/', 'us-west1': 'https://speech.us-west1.rep.googleapis.com/', 'me-west1': 'https://speech.me-west1.rep.googleapis.com/', 'europe-west1': 'https://speech.europe-west1.rep.googleapis.com/', 'europe-west2': 'https://speech.europe-west2.rep.googleapis.com/', 'europe-west3': 'https://speech.europe-west3.rep.googleapis.com/', 'europe-west4': 'https://speech.europe-west4.rep.googleapis.com/'}),
    },
    'sqladmin': {
        'v1beta4':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.sqladmin.v1beta4',
                    client_classpath='sqladmin_v1beta4_client.SqladminV1beta4',
                    base_url='https://sqladmin.googleapis.com/',
                    messages_modulepath='sqladmin_v1beta4_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'storage': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.storage.v1',
                    client_classpath='storage_v1_client.StorageV1',
                    base_url='https://storage.googleapis.com/storage/v1/',
                    messages_modulepath='storage_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'africa-south1': 'https://storage.africa-south1.rep.googleapis.com/', 'asia-east1': 'https://storage.asia-east1.rep.googleapis.com/', 'asia-east2': 'https://storage.asia-east2.rep.googleapis.com/', 'asia-northeast1': 'https://storage.asia-northeast1.rep.googleapis.com/', 'asia-northeast2': 'https://storage.asia-northeast2.rep.googleapis.com/', 'asia-northeast3': 'https://storage.asia-northeast3.rep.googleapis.com/', 'asia-south1': 'https://storage.asia-south1.rep.googleapis.com/', 'asia-south2': 'https://storage.asia-south2.rep.googleapis.com/', 'asia-southeast1': 'https://storage.asia-southeast1.rep.googleapis.com/', 'asia-southeast2': 'https://storage.asia-southeast2.rep.googleapis.com/', 'australia-southeast1': 'https://storage.australia-southeast1.rep.googleapis.com/', 'australia-southeast2': 'https://storage.australia-southeast2.rep.googleapis.com/', 'eu': 'https://storage.eu.rep.googleapis.com/', 'europe-central2': 'https://storage.europe-central2.rep.googleapis.com/', 'europe-north1': 'https://storage.europe-north1.rep.googleapis.com/', 'europe-southwest1': 'https://storage.europe-southwest1.rep.googleapis.com/', 'europe-west1': 'https://storage.europe-west1.rep.googleapis.com/', 'europe-west2': 'https://storage.europe-west2.rep.googleapis.com/', 'europe-west3': 'https://storage.europe-west3.rep.googleapis.com/', 'europe-west4': 'https://storage.europe-west4.rep.googleapis.com/', 'europe-west6': 'https://storage.europe-west6.rep.googleapis.com/', 'europe-west8': 'https://storage.europe-west8.rep.googleapis.com/', 'europe-west9': 'https://storage.europe-west9.rep.googleapis.com/', 'europe-west10': 'https://storage.europe-west10.rep.googleapis.com/', 'europe-west12': 'https://storage.europe-west12.rep.googleapis.com/', 'me-central2': 'https://storage.me-central2.rep.googleapis.com/', 'me-west1': 'https://storage.me-west1.rep.googleapis.com/', 'us': 'https://storage.us.rep.googleapis.com/', 'us-central1': 'https://storage.us-central1.rep.googleapis.com/', 'us-east1': 'https://storage.us-east1.rep.googleapis.com/', 'us-east4': 'https://storage.us-east4.rep.googleapis.com/', 'us-east5': 'https://storage.us-east5.rep.googleapis.com/', 'us-south1': 'https://storage.us-south1.rep.googleapis.com/', 'us-west1': 'https://storage.us-west1.rep.googleapis.com/', 'us-west2': 'https://storage.us-west2.rep.googleapis.com/', 'us-west3': 'https://storage.us-west3.rep.googleapis.com/', 'us-west4': 'https://storage.us-west4.rep.googleapis.com/', 'northamerica-northeast2': 'https://storage.northamerica-northeast2.rep.googleapis.com/', 'southamerica-east1': 'https://storage.southamerica-east1.rep.googleapis.com/', 'southamerica-west1': 'https://storage.southamerica-west1.rep.googleapis.com/', 'me-central1': 'https://storage.me-central1.rep.googleapis.com/', 'northamerica-northeast1': 'https://storage.northamerica-northeast1.rep.googleapis.com/', 'europe-north2': 'https://storage.europe-north2.rep.googleapis.com/', 'us-west8': 'https://storage.us-west8.rep.googleapis.com/', 'northamerica-south1': 'https://storage.northamerica-south1.rep.googleapis.com/'}),
        'v2':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.storage.v2',
                    client_classpath='storage_v2_client.StorageV2',
                    base_url='https://storage.googleapis.com/',
                    messages_modulepath='storage_v2_messages'),
                gapic=GapicClientDef(
                    class_path='googlecloudsdk.generated_clients.gapic_wrappers.storage.v2'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={'africa-south1': 'https://storage.africa-south1.rep.googleapis.com/', 'asia-east1': 'https://storage.asia-east1.rep.googleapis.com/', 'asia-east2': 'https://storage.asia-east2.rep.googleapis.com/', 'asia-northeast1': 'https://storage.asia-northeast1.rep.googleapis.com/', 'asia-northeast2': 'https://storage.asia-northeast2.rep.googleapis.com/', 'asia-northeast3': 'https://storage.asia-northeast3.rep.googleapis.com/', 'asia-south1': 'https://storage.asia-south1.rep.googleapis.com/', 'asia-south2': 'https://storage.asia-south2.rep.googleapis.com/', 'asia-southeast1': 'https://storage.asia-southeast1.rep.googleapis.com/', 'asia-southeast2': 'https://storage.asia-southeast2.rep.googleapis.com/', 'australia-southeast1': 'https://storage.australia-southeast1.rep.googleapis.com/', 'australia-southeast2': 'https://storage.australia-southeast2.rep.googleapis.com/', 'europe-central2': 'https://storage.europe-central2.rep.googleapis.com/', 'europe-north1': 'https://storage.europe-north1.rep.googleapis.com/', 'europe-north2': 'https://storage.europe-north2.rep.googleapis.com/', 'europe-southwest1': 'https://storage.europe-southwest1.rep.googleapis.com/', 'europe-west1': 'https://storage.europe-west1.rep.googleapis.com/', 'europe-west10': 'https://storage.europe-west10.rep.googleapis.com/', 'europe-west12': 'https://storage.europe-west12.rep.googleapis.com/', 'europe-west2': 'https://storage.europe-west2.rep.googleapis.com/', 'europe-west3': 'https://storage.europe-west3.rep.googleapis.com/', 'europe-west4': 'https://storage.europe-west4.rep.googleapis.com/', 'europe-west6': 'https://storage.europe-west6.rep.googleapis.com/', 'europe-west8': 'https://storage.europe-west8.rep.googleapis.com/', 'europe-west9': 'https://storage.europe-west9.rep.googleapis.com/', 'me-central1': 'https://storage.me-central1.rep.googleapis.com/', 'me-central2': 'https://storage.me-central2.rep.googleapis.com/', 'me-west1': 'https://storage.me-west1.rep.googleapis.com/', 'northamerica-northeast1': 'https://storage.northamerica-northeast1.rep.googleapis.com/', 'northamerica-northeast2': 'https://storage.northamerica-northeast2.rep.googleapis.com/', 'northamerica-south1': 'https://storage.northamerica-south1.rep.googleapis.com/', 'southamerica-east1': 'https://storage.southamerica-east1.rep.googleapis.com/', 'southamerica-west1': 'https://storage.southamerica-west1.rep.googleapis.com/', 'us-central1': 'https://storage.us-central1.rep.googleapis.com/', 'us-central2': 'https://storage.us-central2.rep.googleapis.com/', 'us-east1': 'https://storage.us-east1.rep.googleapis.com/', 'us-east4': 'https://storage.us-east4.rep.googleapis.com/', 'us-east5': 'https://storage.us-east5.rep.googleapis.com/', 'us-east7': 'https://storage.us-east7.rep.googleapis.com/', 'us-south1': 'https://storage.us-south1.rep.googleapis.com/', 'us-west1': 'https://storage.us-west1.rep.googleapis.com/', 'us-west2': 'https://storage.us-west2.rep.googleapis.com/', 'us-west3': 'https://storage.us-west3.rep.googleapis.com/', 'us-west4': 'https://storage.us-west4.rep.googleapis.com/', 'us-west8': 'https://storage.us-west8.rep.googleapis.com/', 'eu': 'https://storage.eu.rep.googleapis.com/', 'us': 'https://storage.us.rep.googleapis.com/'}),
    },
    'storagebatchoperations': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.storagebatchoperations.v1',
                    client_classpath='storagebatchoperations_v1_client.StoragebatchoperationsV1',
                    base_url='https://storagebatchoperations.googleapis.com/',
                    messages_modulepath='storagebatchoperations_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'storageinsights': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.storageinsights.v1',
                    client_classpath='storageinsights_v1_client.StorageinsightsV1',
                    base_url='https://storageinsights.googleapis.com/',
                    messages_modulepath='storageinsights_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'storagetransfer': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.storagetransfer.v1',
                    client_classpath='storagetransfer_v1_client.StoragetransferV1',
                    base_url='https://storagetransfer.googleapis.com/',
                    messages_modulepath='storagetransfer_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'stream': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.stream.v1',
                    client_classpath='stream_v1_client.StreamV1',
                    base_url='https://stream.googleapis.com/',
                    messages_modulepath='stream_v1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.stream.v1alpha1',
                    client_classpath='stream_v1alpha1_client.StreamV1alpha1',
                    base_url='https://stream.googleapis.com/',
                    messages_modulepath='stream_v1alpha1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'telcoautomation': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.telcoautomation.v1',
                    client_classpath='telcoautomation_v1_client.TelcoautomationV1',
                    base_url='https://telcoautomation.googleapis.com/',
                    messages_modulepath='telcoautomation_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.telcoautomation.v1alpha1',
                    client_classpath='telcoautomation_v1alpha1_client.TelcoautomationV1alpha1',
                    base_url='https://telcoautomation.googleapis.com/',
                    messages_modulepath='telcoautomation_v1alpha1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'testing': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.testing.v1',
                    client_classpath='testing_v1_client.TestingV1',
                    base_url='https://testing.googleapis.com/',
                    messages_modulepath='testing_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'toolresults': {
        'v1beta3':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.toolresults.v1beta3',
                    client_classpath='toolresults_v1beta3_client.ToolresultsV1beta3',
                    base_url='https://toolresults.googleapis.com/',
                    messages_modulepath='toolresults_v1beta3_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'tpu': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.tpu.v1',
                    client_classpath='tpu_v1_client.TpuV1',
                    base_url='https://tpu.googleapis.com/',
                    messages_modulepath='tpu_v1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.tpu.v1alpha1',
                    client_classpath='tpu_v1alpha1_client.TpuV1alpha1',
                    base_url='https://tpu.googleapis.com/',
                    messages_modulepath='tpu_v1alpha1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v2':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.tpu.v2',
                    client_classpath='tpu_v2_client.TpuV2',
                    base_url='https://tpu.googleapis.com/',
                    messages_modulepath='tpu_v2_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v2alpha1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.tpu.v2alpha1',
                    client_classpath='tpu_v2alpha1_client.TpuV2alpha1',
                    base_url='https://tpu.googleapis.com/',
                    messages_modulepath='tpu_v2alpha1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'transcoder': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.transcoder.v1',
                    client_classpath='transcoder_v1_client.TranscoderV1',
                    base_url='https://transcoder.googleapis.com/',
                    messages_modulepath='transcoder_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'transferappliance': {
        'v1alpha1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.transferappliance.v1alpha1',
                    client_classpath='transferappliance_v1alpha1_client.TransferapplianceV1alpha1',
                    base_url='https://transferappliance.googleapis.com/',
                    messages_modulepath='transferappliance_v1alpha1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'translate': {
        'v3':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.translate.v3',
                    client_classpath='translate_v3_client.TranslateV3',
                    base_url='https://translation.googleapis.com/',
                    messages_modulepath='translate_v3_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v3beta1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.translate.v3beta1',
                    client_classpath='translate_v3beta1_client.TranslateV3beta1',
                    base_url='https://translation.googleapis.com/',
                    messages_modulepath='translate_v3beta1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'vectorsearch': {
        'v1beta':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.vectorsearch.v1beta',
                    client_classpath='vectorsearch_v1beta_client.VectorsearchV1beta',
                    base_url='https://vectorsearch.googleapis.com/',
                    messages_modulepath='vectorsearch_v1beta_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'videointelligence': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.videointelligence.v1',
                    client_classpath='videointelligence_v1_client.VideointelligenceV1',
                    base_url='https://videointelligence.googleapis.com/',
                    messages_modulepath='videointelligence_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'vision': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.vision.v1',
                    client_classpath='vision_v1_client.VisionV1',
                    base_url='https://vision.googleapis.com/',
                    messages_modulepath='vision_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'vmmigration': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.vmmigration.v1',
                    client_classpath='vmmigration_v1_client.VmmigrationV1',
                    base_url='https://vmmigration.googleapis.com/',
                    messages_modulepath='vmmigration_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.vmmigration.v1alpha1',
                    client_classpath='vmmigration_v1alpha1_client.VmmigrationV1alpha1',
                    base_url='https://vmmigration.googleapis.com/',
                    messages_modulepath='vmmigration_v1alpha1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'vmwareengine': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.vmwareengine.v1',
                    client_classpath='vmwareengine_v1_client.VmwareengineV1',
                    base_url='https://vmwareengine.googleapis.com/',
                    messages_modulepath='vmwareengine_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'vpcaccess': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.vpcaccess.v1',
                    client_classpath='vpcaccess_v1_client.VpcaccessV1',
                    base_url='https://vpcaccess.googleapis.com/',
                    messages_modulepath='vpcaccess_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.vpcaccess.v1alpha1',
                    client_classpath='vpcaccess_v1alpha1_client.VpcaccessV1alpha1',
                    base_url='https://vpcaccess.googleapis.com/',
                    messages_modulepath='vpcaccess_v1alpha1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.vpcaccess.v1beta1',
                    client_classpath='vpcaccess_v1beta1_client.VpcaccessV1beta1',
                    base_url='https://vpcaccess.googleapis.com/',
                    messages_modulepath='vpcaccess_v1beta1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'websecurityscanner': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.websecurityscanner.v1',
                    client_classpath='websecurityscanner_v1_client.WebsecurityscannerV1',
                    base_url='https://websecurityscanner.googleapis.com/',
                    messages_modulepath='websecurityscanner_v1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.websecurityscanner.v1beta',
                    client_classpath='websecurityscanner_v1beta_client.WebsecurityscannerV1beta',
                    base_url='https://websecurityscanner.googleapis.com/',
                    messages_modulepath='websecurityscanner_v1beta_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'workflowexecutions': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.workflowexecutions.v1',
                    client_classpath='workflowexecutions_v1_client.WorkflowexecutionsV1',
                    base_url='https://workflowexecutions.googleapis.com/',
                    messages_modulepath='workflowexecutions_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.workflowexecutions.v1alpha1',
                    client_classpath='workflowexecutions_v1alpha1_client.WorkflowexecutionsV1alpha1',
                    base_url='https://workflowexecutions.googleapis.com/',
                    messages_modulepath='workflowexecutions_v1alpha1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.workflowexecutions.v1beta',
                    client_classpath='workflowexecutions_v1beta_client.WorkflowexecutionsV1beta',
                    base_url='https://workflowexecutions.googleapis.com/',
                    messages_modulepath='workflowexecutions_v1beta_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'workflows': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.workflows.v1',
                    client_classpath='workflows_v1_client.WorkflowsV1',
                    base_url='https://workflows.googleapis.com/',
                    messages_modulepath='workflows_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1alpha1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.workflows.v1alpha1',
                    client_classpath='workflows_v1alpha1_client.WorkflowsV1alpha1',
                    base_url='https://workflows.googleapis.com/',
                    messages_modulepath='workflows_v1alpha1_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.workflows.v1beta',
                    client_classpath='workflows_v1beta_client.WorkflowsV1beta',
                    base_url='https://workflows.googleapis.com/',
                    messages_modulepath='workflows_v1beta_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
    'workstations': {
        'v1':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.workstations.v1',
                    client_classpath='workstations_v1_client.WorkstationsV1',
                    base_url='https://workstations.googleapis.com/',
                    messages_modulepath='workstations_v1_messages'),
                default_version=True,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
        'v1beta':
            APIDef(
                apitools=ApitoolsClientDef(
                    class_path='googlecloudsdk.generated_clients.apis.workstations.v1beta',
                    client_classpath='workstations_v1beta_client.WorkstationsV1beta',
                    base_url='https://workstations.googleapis.com/',
                    messages_modulepath='workstations_v1beta_messages'),
                default_version=False,
                enable_mtls=True,
                mtls_endpoint_override='',
                regional_endpoints={}),
    },
}
