/*
 * Decompiled with CFR 0.152.
 */
package org.logstash.common.io;

import java.io.Closeable;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedByInterruptException;
import java.nio.channels.FileChannel;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.Arrays;
import java.util.Comparator;
import java.util.function.Function;
import java.util.zip.CRC32;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.logstash.common.io.RecordHeader;
import org.logstash.common.io.RecordType;

public final class RecordIOReader
implements Closeable {
    private static final Logger logger = LogManager.getLogger(RecordIOReader.class);
    private static final int UNSET = -1;
    private final FileChannel channel;
    private final Path path;
    private ByteBuffer currentBlock;
    private int currentBlockSizeReadFromChannel;
    private long streamPosition;

    public RecordIOReader(Path path) throws IOException {
        this.path = path;
        this.channel = FileChannel.open(path, StandardOpenOption.READ);
        this.currentBlock = ByteBuffer.allocate(32768);
        this.currentBlockSizeReadFromChannel = 0;
        ByteBuffer versionBuffer = ByteBuffer.allocate(1);
        this.channel.read(versionBuffer);
        versionBuffer.rewind();
        byte versionInFile = versionBuffer.get();
        if (versionInFile != 49) {
            this.channel.close();
            throw new IllegalStateException(String.format("Invalid version on DLQ data file %s. Expected version: %c. Version found on file: %c", path, Character.valueOf('1'), versionInFile));
        }
        this.streamPosition = this.channel.position();
    }

    public Path getPath() {
        return this.path;
    }

    public void seekToBlock(int bid) throws IOException {
        this.seekToOffset((long)bid * 32768L + 1L);
    }

    public void seekToOffset(long channelOffset) throws IOException {
        int readBytes;
        this.currentBlock.rewind();
        long segmentOffset = channelOffset - 1L;
        long blockIndex = segmentOffset / 32768L;
        long blockStartOffset = blockIndex * 32768L;
        this.channel.position(blockStartOffset + 1L);
        this.currentBlockSizeReadFromChannel = readBytes = this.channel.read(this.currentBlock);
        this.currentBlock.position((int)segmentOffset % 32768);
        this.streamPosition = channelOffset;
    }

    public <T> byte[] seekToNextEventPosition(T target, Function<byte[], T> keyExtractor, Comparator<T> keyComparator) throws IOException {
        int matchingBlock = -1;
        int lowBlock = 0;
        int highBlock = (int)(this.channel.size() - 1L) / 32768 - 1;
        while (lowBlock < highBlock) {
            int middle = (int)Math.ceil((double)(highBlock + lowBlock) / 2.0);
            this.seekToBlock(middle);
            byte[] readEvent = this.readEvent();
            if (readEvent == null) {
                matchingBlock = lowBlock;
                break;
            }
            T found = keyExtractor.apply(readEvent);
            int compare = keyComparator.compare(found, target);
            if (compare > 0) {
                highBlock = middle - 1;
                continue;
            }
            if (compare < 0) {
                lowBlock = middle;
                continue;
            }
            matchingBlock = middle;
            break;
        }
        if (matchingBlock == -1) {
            matchingBlock = lowBlock;
        }
        this.seekToBlock(matchingBlock);
        int compare = -1;
        byte[] event = null;
        BufferState restorePoint = null;
        while (compare < 0) {
            restorePoint = this.saveBufferState();
            event = this.readEvent();
            if (event == null) {
                return null;
            }
            compare = keyComparator.compare(keyExtractor.apply(event), target);
        }
        if (restorePoint != null) {
            this.restoreFrom(restorePoint);
        }
        return event;
    }

    public long getChannelPosition() {
        return this.streamPosition;
    }

    void consumeBlock(boolean rewind) throws IOException {
        if (!rewind && this.currentBlockSizeReadFromChannel == 32768) {
            return;
        }
        if (rewind) {
            this.currentBlockSizeReadFromChannel = 0;
            this.currentBlock.rewind();
        }
        this.currentBlock.mark();
        try {
            this.currentBlock.position(this.currentBlockSizeReadFromChannel);
            this.channel.read(this.currentBlock);
            this.currentBlockSizeReadFromChannel = this.currentBlock.position();
        }
        finally {
            this.currentBlock.reset();
        }
    }

    public boolean isEndOfStream() {
        return this.currentBlockSizeReadFromChannel < 32768;
    }

    int seekToStartOfEventInBlock() {
        if (this.currentBlock.position() >= this.currentBlockSizeReadFromChannel) {
            return -1;
        }
        RecordType type;
        while ((type = RecordType.fromByte(this.currentBlock.array()[this.currentBlock.arrayOffset() + this.currentBlock.position()])) != null) {
            switch (type) {
                case COMPLETE: 
                case START: {
                    return this.currentBlock.position();
                }
                case MIDDLE: {
                    return -1;
                }
                case END: {
                    RecordHeader header = RecordHeader.get(this.currentBlock);
                    this.currentBlock.position(this.currentBlock.position() + header.getSize());
                    if (!this.isEndOfStream()) break;
                    return -1;
                }
            }
        }
        return -1;
    }

    boolean consumeToStartOfEvent() throws IOException {
        this.consumeBlock(false);
        int eventStartPosition;
        while ((eventStartPosition = this.seekToStartOfEventInBlock()) < 0) {
            if (this.isEndOfStream()) {
                return false;
            }
            this.consumeBlock(true);
        }
        return true;
    }

    private void maybeRollToNextBlock() throws IOException {
        if (this.currentBlock.remaining() < 14) {
            this.streamPosition = this.channel.position();
            this.consumeBlock(true);
        }
    }

    private void getRecord(ByteBuffer buffer, RecordHeader header) throws IOException {
        CRC32 computedChecksum = new CRC32();
        computedChecksum.update(this.currentBlock.array(), this.currentBlock.position(), header.getSize());
        if ((int)computedChecksum.getValue() != header.getChecksum()) {
            throw new IllegalStateException("invalid checksum of record");
        }
        buffer.put(this.currentBlock.array(), this.currentBlock.position(), header.getSize());
        this.currentBlock.position(this.currentBlock.position() + header.getSize());
        this.streamPosition = this.currentBlock.remaining() < 14 ? this.channel.position() : (this.streamPosition += (long)header.getSize());
    }

    public byte[] readEvent() throws IOException {
        try {
            if (!this.channel.isOpen() || !this.consumeToStartOfEvent()) {
                return null;
            }
            RecordHeader header = RecordHeader.get(this.currentBlock);
            this.streamPosition += 13L;
            int cumReadSize = 0;
            int bufferSize = header.getTotalEventSize().orElseGet(header::getSize);
            ByteBuffer buffer = ByteBuffer.allocate(bufferSize);
            this.getRecord(buffer, header);
            cumReadSize += header.getSize();
            while (cumReadSize < bufferSize) {
                this.maybeRollToNextBlock();
                RecordHeader nextHeader = RecordHeader.get(this.currentBlock);
                this.streamPosition += 13L;
                this.getRecord(buffer, nextHeader);
                cumReadSize += nextHeader.getSize();
            }
            return buffer.array();
        }
        catch (ClosedByInterruptException e) {
            return null;
        }
    }

    @Override
    public void close() throws IOException {
        this.channel.close();
    }

    private BufferState saveBufferState() throws IOException {
        return new BufferState.Builder().channelPosition(this.channel.position()).blockContents(Arrays.copyOf(this.currentBlock.array(), this.currentBlock.array().length)).currentBlockPosition(this.currentBlock.position()).currentBlockSizeReadFromChannel(this.currentBlockSizeReadFromChannel).build();
    }

    private void restoreFrom(BufferState bufferState) throws IOException {
        this.currentBlock = ByteBuffer.wrap(bufferState.blockContents);
        this.currentBlock.position(bufferState.currentBlockPosition);
        this.channel.position(bufferState.channelPosition);
        this.streamPosition = this.channel.position();
        this.currentBlockSizeReadFromChannel = bufferState.currentBlockSizeReadFromChannel;
    }

    static SegmentStatus getSegmentStatus(Path path) {
        SegmentStatus segmentStatus;
        RecordIOReader ioReader = new RecordIOReader(path);
        try {
            boolean moreEvents = true;
            SegmentStatus segmentStatus2 = SegmentStatus.EMPTY;
            while (moreEvents) {
                moreEvents = ioReader.readEvent() != null;
                if (!moreEvents) continue;
                segmentStatus2 = SegmentStatus.VALID;
            }
            segmentStatus = segmentStatus2;
        }
        catch (Throwable throwable) {
            try {
                try {
                    ioReader.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException | IllegalStateException e) {
                logger.warn("Error reading segment file {}", (Object)path, (Object)e);
                return SegmentStatus.INVALID;
            }
        }
        ioReader.close();
        return segmentStatus;
    }

    static final class BufferState {
        private int currentBlockPosition;
        private int currentBlockSizeReadFromChannel;
        private long channelPosition;
        private byte[] blockContents;

        BufferState(Builder builder) {
            this.currentBlockPosition = builder.currentBlockPosition;
            this.currentBlockSizeReadFromChannel = builder.currentBlockSizeReadFromChannel;
            this.channelPosition = builder.channelPosition;
            this.blockContents = builder.blockContents;
        }

        public String toString() {
            return String.format("CurrentBlockPosition:%d, currentBlockSizeReadFromChannel: %d, channelPosition: %d", this.currentBlockPosition, this.currentBlockSizeReadFromChannel, this.channelPosition);
        }

        static final class Builder {
            private int currentBlockPosition;
            private int currentBlockSizeReadFromChannel;
            private long channelPosition;
            private byte[] blockContents;

            Builder() {
            }

            Builder currentBlockPosition(int currentBlockPosition) {
                this.currentBlockPosition = currentBlockPosition;
                return this;
            }

            Builder currentBlockSizeReadFromChannel(int currentBlockSizeReadFromChannel) {
                this.currentBlockSizeReadFromChannel = currentBlockSizeReadFromChannel;
                return this;
            }

            Builder channelPosition(long channelPosition) {
                this.channelPosition = channelPosition;
                return this;
            }

            Builder blockContents(byte[] blockContents) {
                this.blockContents = blockContents;
                return this;
            }

            BufferState build() {
                return new BufferState(this);
            }
        }
    }

    static enum SegmentStatus {
        EMPTY,
        VALID,
        INVALID;

    }
}

