//----------------------------------------------------------------------------
//
// TSDuck - The MPEG Transport Stream Toolkit
// Copyright (c) 2019-2020, Anthony Delannoy
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
// 1. Redistributions of source code must retain the above copyright notice,
//    this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
// LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
// CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
// SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
// CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
// THE POSSIBILITY OF SUCH DAMAGE.
//
//----------------------------------------------------------------------------
//!
//!  @file
//!  Representation of a sky_logical_channel_number_descriptor.
//!  This is a private descriptor, must be preceded by the BskyB PDS.
//!
//----------------------------------------------------------------------------

#pragma once
#include "tsAbstractDescriptor.h"

namespace ts {
    //!
    //! Representation of a sky_logical_channel_number_descriptor.
    //!
    //! This is a private descriptor, must be preceded by the BskyB PDS.
    //! @ingroup descriptor
    //!
    class TSDUCKDLL SkyLogicalChannelNumberDescriptor : public AbstractDescriptor
    {
    public:
        //!
        //! Service entry.
        //!
        struct TSDUCKDLL Entry
        {
            // Public members
            uint16_t service_id;   //!< Service id.
            uint8_t  service_type; //!< Service type.
            uint16_t channel_id;   //!< Channel id
            uint16_t lcn;          //!< Logical channel number.
            uint16_t sky_id;       //!< Sky channel number.

            //!
            //! Constructor
            //! @param [in] id_ Service id.
            //! @param [in] type_ Service type.
            //! @param [in] cid_ Channel id.
            //! @param [in] lcn_ Logical channel number.
            //! @param [in] skyid_ Sky id.
            //!
            Entry(uint16_t id_ = 0, uint8_t type_ = 0, uint16_t cid_ = 0,
                  uint16_t lcn_ = 0, uint16_t skyid_ = 0):
                service_id(id_),
                service_type(type_),
                channel_id(cid_),
                lcn(lcn_),
                sky_id(skyid_)
            {
            }
        };

        //!
        //! List of service entries.
        //!
        typedef std::list<Entry> EntryList;

        //!
        //! Maximum number of services entries to fit in 255 bytes.
        //!
        static const size_t MAX_ENTRIES = 28;

        // SkyLogicalChannelNumberDescriptor public members:
        EntryList entries;  //!< List of service entries.
        uint16_t region_id; //!< Region id (maybe in the UK?, 0xFFFF for all country).

        //!
        //! Default constructor.
        //!
        SkyLogicalChannelNumberDescriptor();

        //!
        //! Constructor from a binary descriptor
        //! @param [in,out] duck TSDuck execution context.
        //! @param [in] bin A binary descriptor to deserialize.
        //!
        SkyLogicalChannelNumberDescriptor(DuckContext& duck, const Descriptor& bin);

        // Inherited methods
        virtual void serialize(DuckContext&, Descriptor&) const override;
        virtual void deserialize(DuckContext&, const Descriptor&) override;
        virtual void clearContent() override;
        virtual void buildXML(DuckContext&, xml::Element*) const override;
        virtual bool analyzeXML(DuckContext& duck, const xml::Element* element) override;
        DeclareLegacyDisplayDescriptor();
    };
}
