=head1 NAME

iPE::SequenceReader::NoLoad::Raw - NoLoad version of the Raw Sequence reader

=head1 DESCRIPTION

This SequenceReader expects a file with a single line which contains the sequence with no additional characters.

=head1 FUNCTIONS

=over 8

=cut
package iPE::SequenceReader::NoLoad::Raw;

use Carp;
use iPE::Util::DNATools;
use base ("iPE::SequenceReader::NoLoad");
use strict;

=item new (memberHash)

Constructs a new Raw sequence reader.  This doesn't read in the sequences, it simply make calculations which will enable it to later read portions of the sequence in from the file.

=cut
sub new {
    my $class = shift;
    # no superclass call; there is no header in the raw format.
    my $this = new iPE::SequenceReader(@_);
    $this = bless $this, $class;

    my $fh = $this->fh;
    $this->{fileSize_} = (-s $this->filename);
    $this->{length_} = $this->{fileSize_} - 1;
    $this->{header_} = "";
    $this->{headerLength_} = 0;

    seek $fh, 0, 0;
    
    return $this;
}

=item length ()

Returns length of the sequence.

=cut
sub length      { shift->{length_}          }

=item getSeq (start, length), getRevSeq (start, length)

This gets sequence directly from the file.  This works similarly to substr in that it takes in a start and 'offset' or length of the substring to retrieve.  Note that getRevSeq also reverse-complements the sequence.

=cut
sub getSeq {
    my ($this, $start, $length) = @_;

    my $fh = $this->{fh_};
    my $seq;
    sysseek($fh, $start, 0);
    sysread($fh, $seq,   $length);

    $seq =~ s/\n//g;
    
    #debugging, we should be able to nail this.
    warn "Length $length does not equal actual length ".length($seq)." \n"
        if(length($seq) != $length);

    return $seq;
}

sub getRevSeq {
    my ($this, $start, $length, $seqNum) = @_;
    
    $start = $this->length-$start-$length;
    my $forwSeq = $this->getSeq($start, $length, $seqNum);
    my $revSeqRef = reverseComplement(\$forwSeq);
    return $$revSeqRef;
}

=back

=head1 SEE ALSO

L<iPE::SequenceReader::NoLoad>

=head1 AUTHOR

Bob Zimmermann (rpz@cse.wustl.edu).
(With much acknowledgement to Sam Gross's code).

=cut
1;
