(*
 * Builtin.m
 * 
 * This file is part of the Oxford Oberon-2 compiler
 * Copyright (c) 2006--2016 J. M. Spivey
 * All rights reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *)

MODULE Builtin;

IMPORT SYSTEM;

CONST version =
  "Oxford Oberon-2 library version 3.3.0 [build unknown]";

PROCEDURE HALT(n: INTEGER) IS "xmain_exit";

PROCEDURE NEW(desc: SYSTEM.PTR; size: INTEGER): SYSTEM.PTR IS "*NEW";

PROCEDURE NEWFLEX(elmap, elsize, k: INTEGER): SYSTEM.PTR IS "*NEWFLEX";

PROCEDURE COMPARE(s, t: ARRAY OF CHAR): INTEGER IS "*COMPARE";

PROCEDURE CAP(c: CHAR): CHAR IS "toupper";

PROCEDURE COPY(VAR s, t: ARRAY OF CHAR) IS "*COPY";

PROCEDURE FLEXASSIGN(elsize, dim: INTEGER (* ... *)) IS "*FLEXASSIGN";

PROCEDURE ABSINT(x: INTEGER): INTEGER;
BEGIN
  IF x >= 0 THEN RETURN x ELSE RETURN -x END
END ABSINT;

PROCEDURE ABSLONG(x: LONGINT): LONGINT;
BEGIN
  IF x >= 0 THEN RETURN x ELSE RETURN -x END
END ABSLONG;

PROCEDURE ABSFLOAT(x: REAL): REAL;
BEGIN
  IF x >= 0 THEN RETURN x ELSE RETURN -x END
END ABSFLOAT;

PROCEDURE ABSDOUBLE(x: LONGREAL): LONGREAL;
BEGIN
  IF x >= 0 THEN RETURN x ELSE RETURN -x END
END ABSDOUBLE;

PROCEDURE INCLONG(VAR x: LONGINT; n: LONGINT);
BEGIN
  x := x + n
END INCLONG;

PROCEDURE DECLONG(VAR x: LONGINT; n: LONGINT);
BEGIN
  x := x - n
END DECLONG;

PROCEDURE ASH(a, b: INTEGER): INTEGER;
BEGIN
  IF b >= 0 THEN RETURN LSL(a, b) ELSE RETURN ASR(a, -b) END
END ASH;

(* These are very flaky, but they are what Wirth describes. *)
 
PROCEDURE PACK(VAR x: REAL; n: INTEGER);
BEGIN
  x := SYSTEM.VAL(REAL, SYSTEM.VAL(INTEGER, x) + LSL(n, 23))
END PACK;
 
PROCEDURE UNPK(VAR x: REAL; VAR n: INTEGER);
  VAR xx: INTEGER;
BEGIN
  xx := SYSTEM.VAL(INTEGER, x);
  n := LSR(xx, 23) - 127;
  x := SYSTEM.VAL(REAL, xx - LSL(n, 23))
END UNPK;
 
CONST two52 = 10000000000000H;
 
PROCEDURE PACKLONG(VAR x: LONGREAL; n: INTEGER);
BEGIN
  x := SYSTEM.VAL(LONGREAL, SYSTEM.VAL(LONGINT, x) + LONG(n) * two52)
END PACKLONG;
 
PROCEDURE UNPKLONG(VAR x: LONGREAL; VAR n: INTEGER);
  VAR xx: LONGINT;
BEGIN
  xx := SYSTEM.VAL(LONGINT, x);
  n := SHORT(xx DIV two52 - 1023);
  x := SYSTEM.VAL(LONGREAL, xx - LONG(n) * two52)
END UNPKLONG;

PROCEDURE EASSERT(line, arg: INTEGER) IS "*EASSERT";

PROCEDURE SysMOVE(src, dest: INTEGER; n: INTEGER) IS "*sys_move";
PROCEDURE SysLIBERROR(CONST msg: ARRAY OF CHAR) IS "*sys_liberror";
PROCEDURE SysLOADLIB(name: ARRAY OF CHAR) IS "load_lib";
PROCEDURE SysGC IS "*gc_collect";

(* These are declared here so the primitives appear in primtab.c,
   but they must be redeclared in any module that uses them *)
PROCEDURE GcDebug(flags: ARRAY OF CHAR) IS "gc_debug";
PROCEDURE GcHeapSize(): INTEGER IS "gc_heap_size";

END Builtin.

--CODE--

#include "obx.h"
#include <ctype.h>
#include <math.h>
#include <string.h>

char *lib_version = 
  "Oxford Oberon-2 library version 3.3.0 [build unknown]";

value *NEW(value *bp) {
     word desc = bp[HEAD+0].a;
     int size = bp[HEAD+1].i+4;
     value *sp = bp;
     word p = gc_alloc(size, bp);
     value *q = ptrcast(value, p);
     q[0].a = desc;
     (*--sp).a = p+4;
     return sp;
}

/*
Layout for flexible arrays:

q:	desc

	element 0
	...
	element n-1

desc:	map			desc = q + align(n*elsize) + 4
	dim 1
        ...
	dim k

map:	-4
	desc-(q+4)
[	GC_REPEAT		if elmap != 0
	0
	n
	elsize
	GC_MAP
	elmap
	GC_END          ]		
	GC_END

size = align(n*elsize) + 4*k + 20 + (28 if elmap != 0)

Parameters NEWFLEX(elmap, elsize, k: INTEGER;
			dim_0, ..., dim_{k-1}: INTEGER): SYSTEM.PTR
*/

value *NEWFLEX(value *bp) {
     word elmap = bp[HEAD+0].a;
     int elsize = bp[HEAD+1].i;
     int k = bp[HEAD+2].i;
     value *dim = &bp[HEAD+3];		/* Array of bounds */
     value *sp = bp;

     int size, arsize, i, n;
     word p, desc, map;
     value *q, *pdesc, *pmap;

     /* Compute no. of elements */
     n = 1;
     for (i = 0; i < k; i++) n *= dim[i].i;
     if (n < 0) liberror("allocating negative size");
     arsize = align(n * elsize, 4); 
     if (n == 0) elmap = 0;

     /* Allocate the space */
     size = arsize + 4*k + 20;
     if (elmap != 0) size += 28;

     p = gc_alloc(size, bp);
     desc = p + arsize + 4;
     map = desc + 4*k + 4;

     q = ptrcast(value, p);
     pdesc = ptrcast(value, desc);
     pmap = ptrcast(value, map);
     q[0].a = desc;

     /* Fill in the descriptor */
     pdesc[DESC_MAP].a = map;
     for (i = 0; i < k; i++) pdesc[DESC_BOUND+i].i = dim[i].i;

     /* Fill in the map */
     pmap[0].i = -4;
     pmap[1].i = desc-(p+4);
     if (elmap == 0) 
          pmap[2].i = GC_END;
     else {
          pmap[2].i = GC_REPEAT;
          pmap[3].i = 0;
          pmap[4].i = n;
          pmap[5].i = elsize;
          pmap[6].i = GC_MAP;
          pmap[7].i = elmap;
          pmap[8].i = GC_END;
          pmap[9].i = GC_END;
     }

     (*--sp).a = p+4; 
     return sp;
}

value *COMPARE(value *bp) {
     uchar *s1 = pointer(bp[HEAD+0]), *s2 = pointer(bp[HEAD+2]);
     int i = 0, n = min(bp[HEAD+1].i, bp[HEAD+3].i);
     value *sp = bp;

     while (i < n && s1[i] != '\0' && s1[i] == s2[i]) i++;
     if (i >= n) liberror("string is not null-terminated");
     (*--sp).i = s1[i] - s2[i]; 
     return sp;
}

value *COPY(value *bp) {
     obcopy((char * ) pointer(bp[HEAD+2]), bp[HEAD+3].i,
            (char * ) pointer(bp[HEAD+0]), bp[HEAD+1].i, bp);
     return bp;                                           
}

value *FLEXASSIGN(value *bp) {
     int size = bp[HEAD+0].i;
     int dim = bp[HEAD+1].i;
     uchar *src = pointer(bp[HEAD+2]), *dst = pointer(bp[HEAD+dim+3]);
     value *sbound = &bp[HEAD+3], *dbound = &bp[HEAD+dim+4];
     int i;

     for (i = 0; i < dim; i++) {
	  int sb = sbound[i].i, db = dbound[i].i;
	  if (sb > db || (i > 0 && sb < db)) 
	       liberror("bound mismatch in open array assignment");
	  size *= sb;
     }

     memmove(dst, src, size);
     return bp;
}

value *sys_move(value *bp) {
     memmove(pointer(bp[HEAD+1]), pointer(bp[HEAD+0]), bp[HEAD+2].i);
     return bp;
}

value *sys_liberror(value *bp) {
     value *bp1 = valptr(bp[BP]);
     error_stop((char * ) pointer(bp[HEAD+0]), 0, 0, bp1, NULL);
     return bp;
}

value *EASSERT(value *bp) {
     int line = bp[HEAD+0].i;
     int arg = bp[HEAD+1].i;
     value *bp1 = valptr(bp[BP]);
     uchar *pc = pointer(bp[PC]);
     error_stop("assertion failed (%d)", arg, line, bp1, pc);
     return bp;
}
