/* LIBDS
 * =====
 * This software is Copyright (c) 2002-03 Malcolm Smith.
 * No warranty is provided, including but not limited to
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * This code is licenced subject to the GNU General
 * Public Licence (GPL).  See the COPYING file for more.
 */
#ifndef DS_TREE_H
#define DS_TREE_H

#include <DSCore.h>
#include <DSTreeElement.h>

/*!
 * @header DSTree
 * @discussion Tree implements a simple tree for both string and integer keys and values.
 */

/*!
 * @class DSTree
 * @abstract A single-threaded simple implementation of a Tree.
 * @discussion Tree implements a simple tree for both string and integer keys and values.
 */
class EXPSYM DSTree {
	private:
	protected:
		int nNodes;
		DSTree ** nodes;
		DSTreeElement * value;
	public:
		/*!
		 * @function DSTree
		 * @discussion Creates a new empty Tree.
		 */
		DSTree(int nodes=1);

		/*!
		 * @function ~DSTree
		 * @discussion Destroys a Tree.
		 */
		~DSTree();

		/*!
		 * @function close
		 * @discussion Purges all data and nodes from the tree, resulting in an empty tree.  Data is cleaned according to the flags used when it was inserted.
		 */
		void close();

		/*!
		 * @function insert
		 * @discussion Inserts a new TreeElement into the Tree.
		 * @param path Specifies an array of ints corresponding to the path to use to locate this element in the tree.
		 * @param npathelements Specifies the number of nodes in path.
		 * @param le Specifies the TreeElement to be added to the tree.
		 * @result Returns TRUE if the addition succeeded, FALSE if an element already exists at the desired location.
		 */
		BOOL insert(int * path, int npathelements, DSTreeElement * te);

		/*!
		 * @function insert
		 * @discussion Inserts the data into the Tree.
		 * @param path Specifies an array of ints corresponding to the path to use to locate this element in the tree.
		 * @param npathelements Specifies the number of nodes in path.
		 * @param data Specifies the data to be added to the tree.
		 * @param cleanup Specifies the flags controlling how the data should be cleaned up.  See the TreeElement class for more information.
		 * @result Returns TRUE if the addition succeeded, FALSE if an element already exists at the desired location.
		 */
		BOOL insert(int * path, int npathelements, void * data, int cleanup=0);

		/*!
		 * @function insert
		 * @discussion Inserts the data into the Tree.
		 * @param path Specifies an array of ints corresponding to the path to use to locate this element in the tree.
		 * @param npathelements Specifies the number of nodes in path.
		 * @param data Specifies the data to be added to the tree.
		 * @param cleanup Specifies the flags controlling how the data should be cleaned up.  See the TreeElement class for more information.
		 * @result Returns TRUE if the addition succeeded, FALSE if an element already exists at the desired location.
		 */
		BOOL insert(int * path, int npathelements, unsigned int data, int cleanup=0);

		/*!
		 * @function getElement
		 * @discussion Returns a pointer to the TreeElement at the specified path, or NULL if no such element exists.  Remember, this is a pointer to internal data in the tree, so modifying its contents is modifying the tree.
		 * @param path Specifies an array of ints corresponding to the path to use to locate this element in the tree.
		 * @param npathelements Specifies the number of nodes in path.
		 * @result A pointer to the TreeElement at the specified path.
		 */
		DSTreeElement * getElement(int * path, int npathelements);

		/*!
		 * @function getValue
		 * @discussion Returns the pointer data value at the specified path, or NULL if no such element exists.  Remember, this is a pointer to internal data in the tree, so modifying its contents is modifying the tree.
		 * @param path Specifies an array of ints corresponding to the path to use to locate this element in the tree.
		 * @param npathelements Specifies the number of nodes in path.
		 * @result A pointer to the data at the specified path.
		 */
		void * getValue(int * path, int npathelements);

		/*!
		 * @function getInt
		 * @discussion Returns the numeric data value at the specified path, or 0 if no such element exists.
		 * @param path Specifies an array of ints corresponding to the path to use to locate this element in the tree.
		 * @param npathelements Specifies the number of nodes in path.
		 * @result The data at the specified path.
		 */
		unsigned int getInt(int * path, int npathelements);


		DSTreeElement * getClosestElement(int * path, int npathelements);
		void * getClosestValue(int * path, int npathelements);
		unsigned int getClosestInt(int * path, int npathelements);

};

#endif
