/* LIBDS
 * =====
 * This software is Copyright (c) 2002-03 Malcolm Smith.
 * No warranty is provided, including but not limited to
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * This code is licenced subject to the GNU General
 * Public Licence (GPL).  See the COPYING file for more.
 */
#ifndef DS_LISTELEMENT_H
#define DS_LISTELEMENT_H

#include <DSCore.h>
#include <stdlib.h>
/*!
 * @header DSListElement
 * @discussion A ListElement contains a key, value, and details about how to clean up after itself.  It can only have one key, numeric or string, and one value. 
 */

/*!
 * @class DSListElement
 * @abstract A unit capable of being stored in a Linked List.
 * @discussion A ListElement contains a key, value, and details about how to clean up after itself.  It can only have one key, numeric or string, and one value. 
 */
class EXPSYM DSListElement
{
	private:
		int flags;
		union {
			unsigned int nKey;
			char* strKey;
		};
		union {
			unsigned int nData;
			void* pData;
		};
		DSListElement* next;
		enum {
			KEYTYPE_STRING=1,
			VALUETYPE_STRING=2
		};
		void cleanupKey();
		void cleanupData();
	public:
		/*!
		 * @enum Cleanup Cleanup flags
		 * @constant CLEANUP_KEY_FREE Specifies that the key should be cleaned up using free() when it's disposed.
		 * @constant CLEANUP_KEY_DELETE Specifies that the key should be cleaned up using delete when it's disposed.
		 * @constant CLEANUP_VALUE_FREE Specifies that the value should be cleaned up using free() when it's disposed.
		 */
		enum {
			CLEANUP_KEY_FREE=16,
			CLEANUP_KEY_DELETE=32,
			CLEANUP_VALUE_FREE=64,
			CLEANUP_ALL=112
		};

		/*!
		 * @function DSListElement
		 * @discussion Creates a new ListElement.
		 * @param cleanup Specifies the cleanup flags that apply to this ListElement.
		 */
		DSListElement(int cleanup=0);

		/*!
		 * @function DSListElement
		 * @discussion Copies a ListElement; does not copy the data contained inside.
		 * @param le Specifies the ListElement to copy.
		 */
		DSListElement(DSListElement * le);

		/*!
		 * @function DSListElement
		 * @discussion Creates a new ListElement.
		 * @param Key Specifies the string key of the new element.  This pointer is used in the element, copy data yourself.
		 * @param Data Specifies the value of the new element.  This pointer is used in the element, copy data yourself.
		 * @param Next Points to the next ListElement in the List.
		 * @param Cleanup Specifies the cleanup flags that apply to this ListElement.
		 */
		DSListElement(char* Key,void* Data,DSListElement* Next,
				int Cleanup=0);

		/*!
		 * @function DSListElement
		 * @discussion Creates a new ListElement.
		 * @param Key Specifies the numeric key of the new element.
		 * @param Data Specifies the value of the new element.  This pointer is used in the element, copy data yourself.
		 * @param Next Points to the next ListElement in the List.
		 * @param Cleanup Specifies the cleanup flags that apply to this ListElement.
		 */
		DSListElement(unsigned int Key,void* Data,DSListElement* Next,
				int Cleanup=0);
		/*!
		 * @function DSListElement
		 * @discussion Creates a new ListElement.
		 * @param Key Specifies the string key of the new element.  This pointer is used in the element, copy data yourself.
		 * @param Data Specifies the numeric value of the new element.
		 * @param Next Points to the next ListElement in the List.
		 * @param Cleanup Specifies the cleanup flags that apply to this ListElement.
		 */
		DSListElement(char* Key,unsigned int Data,DSListElement* Next,
				int Cleanup=0);

		/*!
		 * @function DSListElement
		 * @discussion Creates a new ListElement.
		 * @param Key Specifies the numeric key of the new element.
		 * @param Data Specifies the numeric value of the new element.
		 * @param Next Points to the next ListElement in the List.
		 * @param Cleanup Specifies the cleanup flags that apply to this ListElement.
		 */
		DSListElement(unsigned int Key,unsigned int Data,
				DSListElement* Next, int Cleanup=0);

		/*!
		 * @function ~ListElement
		 * @discussion Destroys the ListElement.
		 */
		~DSListElement();

		/*!
		 * @function getDataInt
		 * @discussion Retrieves the numeric value of the ListElement.
		 * @result The numeric value of the ListElement, or 0, if the ListElement contains a pointer.
		 */
		unsigned int getDataInt();

		/*!
		 * @function getDataPtr
		 * @discussion Retrieves the pointer value of the ListElement.
		 * @result The pointer value of the ListElement, or NULL, if the ListElement contains a number.
		 */
		void*        getDataPtr();

		/*!
		 * @function getKeyString
		 * @discussion Retrieves the string key of the ListElement.
		 * @result The string key of the ListElement, or NULL, if the key is numeric.
		 */
		char*        getKeyString();

		/*!
		 * @function getKeyInt
		 * @discussion Retrieves the numeric key of the ListElement.
		 * @result The numeric key of the ListElement, or 0, if the key is a string.
		 */
		unsigned int getKeyInt();

		/*!
		 * @function getNext
		 * @discussion Retrieves the next ListElement.
		 * @result A pointer to the next ListElement, or NULL, if there is no following element.
		 */
		DSListElement* getNext();

		/*!
		 * @function setCleanup
		 * @discussion Modifies the cleanup flags.
		 * @param Cleanup The new cleanup flags.
		 */
		void         setCleanup(int Cleanup);

		/*!
		 * @function setDataInt
		 * @discussion Modifies the value in the ListElement.  Any previous value is destroyed.
		 * @param Data The new numeric value.
		 */
		void         setDataInt(unsigned int Data);

		/*!
		 * @function setDataPtr
		 * @discussion Modifies the value in the ListElement.  Any previous value is destroyed.
		 * @param Data The new pointer value.  This pointer is inserted into the ListElement and is not copied.
		 */
		void         setDataPtr(void * Data);

		/*!
		 * @function setKeyInt
		 * @discussion Modifies the key in the ListElement.  Any previous key is destroyed.
		 * @param Key The new numeric value.
		 */
		void         setKeyInt(unsigned int Key);

		/*!
		 * @function setKeyInt
		 * @discussion Modifies the key in the ListElement.  Any previous key is destroyed.
		 * @param Key The new string value.  This pointer is used directly, not copied; make sure you have copied it yourself if you'd like the ListElement to keep it.
		 */
		void         setKeyString(char * Key);

		/*!
		 * @function setNext
		 * @discussion Modifies the next the ListElement.
		 * @param Next The new next ListElement.
		 */
		void         setNext(DSListElement* Next);
};

#endif



