/* LIBDS
 * =====
 * This software is Copyright (c) 2002-03 Malcolm Smith.
 * No warranty is provided, including but not limited to
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * This code is licenced subject to the GNU General
 * Public Licence (GPL).  See the COPYING file for more.
 */

#ifndef DS_CONFIGFILE_H
#define DS_CONFIGFILE_H
#include <DSCore.h>
#include <DSHashTable.h>
#ifdef HAVE_STDIO_H
#include <stdio.h>
#endif
/*!
 * @header DSConfigFile
 * @discussion ConfigFile implements the necessary mechanisms to have in-memory configuration files.  These configuration files are similar to Windows .INI files, having a section name in square braces, followed by key=value pairs.  This class makes dealing with files computationally efficient.
 */

/*!
 * @class DSConfigFile
 * @abstract Single-threaded ConfigFile implementation.
 * @discussion ConfigFile implements the necessary mechanisms to have in-memory configuration files.  These configuration files are similar to Windows .INI files, having a section name in square braces, followed by key=value pairs.  This class makes dealing with files computationally efficient.
 */
class EXPSYM DSConfigFile
{
	private:
	DSHashTable* sections;
	DSHashTable * getSection(char * section);
	void printkeys(FILE * fp, DSHashTable * section);

	public:
	/*!
	 * @function DSConfigFile
	 * @discussion Create an empty ConfigFile.
	 */
	DSConfigFile();

	/*!
	 * @function ~DSConfigFile
	 * @discussion Destroy a ConfigFile.
	 */
	~DSConfigFile();

	/*!
	 * @function close
	 * @discussion Purge all the data from the ConfigFile, but keep the ConfigFile itself available for subsequent uses.
	 */
	void close();

	/*!
	 * @function loadFile
	 * @discussion Load data from a file into this ConfigFile.
	 * @param filename Path to a file in which the data is located.
	 * @result Returns nonzero to indicate success, zero indicates failure.
	 */
	int loadFile(const char * filename);

	/*!
	 * @function saveFile
	 * @discussion Save data from this ConfigFile into a file.
	 * @param filename Path to a file in which the data is to be stored.
	 * @result Returns nonzero to indicate success, zero indicates failure.
	 */
	int saveFile(const char * filename);

	/*!
	 * @function forEach
	 * @discussion Iterate through a section on the ConfigFile, and call a function for each entry found.
	 * @param section Specifies the section to be iterated through.  This can be NULL, in which case the sections themselves are iterated through.
	 * @param callback Specifies the function to be called when an entry is found.  This function will be passed the key and value, plus any extra data, described below
	 * @param extra A pointer to additional data which will be passed to the function specified above for each time an item is found.
	 */
	void forEach(char * section,void (*callback)(char *, char *, void *), void * extra);

	/*!
	 * @function getValue
	 * @discussion Retrieve a pointer to the data inside the ConfigFile.
	 * @param section Specifies the section that the data is located in.
	 * @param key Specifies the key that is being searched for.
	 * @param def Specifies the default value to return, if no match is found.
	 * @result A pointer to the data, or def.  Remember, this is a pointer, and can be used to tamper with the internals of the ConfigFile.  Please be careful.
	 */
	char * getValue(const char * section, const char * key, char * def=NULL);

	/*!
	 * @function getValueInt
	 * @discussion Retrieve the data inside the ConfigFile.
	 * @param section Specifies the section that the data is located in.
	 * @param key Specifies the key that is being searched for.
	 * @param def Specifies the default value to return, if no match is found.
	 * @result The numeric data, or def.
	 */
	unsigned int getValueInt(const char * section, const char * key, unsigned int def=0);

	/*!
	 * @function setValue
	 * @discussion Set a value in the ConfigFile.
	 * @param section Specify the section the value is located in.
	 * @param key Specify the key corresponding to the value.  This parameter is used directly in the ConfigFile, not copied.  It will be deleted when the ConfigFile is closed or destroyed.  You should copy the data yourself prior to calling this function.
	 * @param value Specify the new value.  This parameter is used directly in the ConfigFile, not copied.  It will be deleted when the ConfigFile is closed or destroyed.  You should copy the data yourself prior to calling this function.
	 */
	void setValue(const char * section, char * key, char * value);

	/*!
	 * @function delValue
	 * @discussion Delete a value from the ConfigFile
	 * @param section Specify the section the value is located in.
	 * @param key Specify the key corresponding to the value.
	 */
	void delValue(const char * section, const char * key);

	/*!
	 * @function isValue
	 * @discussion Determine if a value is in the ConfigFile.
	 * @param section Specify the section the value is located in.
	 * @param key Specify the key corresponding to the value.
	 * @result Returns the number of instances of the value in the ConfigFile.
	 */
	int isValue(const char * section, const char * key);
};

#endif

