/**
 * @file polarity.h
 *
 * Polarity Events format definition and handling functions.
 * This event contains change information, with an X/Y address
 * and an ON/OFF polarity.
 * The (0, 0) address is in the upper left corner of the screen,
 * like in OpenCV/computer graphics.
 */

#ifndef LIBCAER_EVENTS_POLARITY_H_
#define LIBCAER_EVENTS_POLARITY_H_

#include "common.h"

#ifdef __cplusplus
extern "C" {
#endif

/**
 * Shift and mask values for the polarity, X and Y addresses
 * of a polarity event.
 * Addresses up to 15 bit are supported. Polarity is ON(=1) or OFF(=0).
 * Bit 0 is the valid mark, see 'common.h' for more details.
 */
//@{
#define POLARITY_SHIFT 1
#define POLARITY_MASK 0x00000001
#define POLARITY_Y_ADDR_SHIFT 2
#define POLARITY_Y_ADDR_MASK 0x00007FFF
#define POLARITY_X_ADDR_SHIFT 17
#define POLARITY_X_ADDR_MASK 0x00007FFF
//@}

/**
 * Polarity event data structure definition.
 * This contains the actual X/Y addresses, the polarity,
 * as well as the 32 bit event timestamp.
 * The (0, 0) address is in the upper left corner of the screen,
 * like in OpenCV/computer graphics.
 * Signed integers are used for fields that are to be interpreted
 * directly, for compatibility with languages that do not have
 * unsigned integer types, such as Java.
 */
PACKED_STRUCT(
struct caer_polarity_event {
	/// Event data. First because of valid mark.
	uint32_t data;
	/// Event timestamp.
	int32_t timestamp;
});

/**
 * Type for pointer to polarity event data structure.
 */
typedef struct caer_polarity_event *caerPolarityEvent;
typedef const struct caer_polarity_event *caerPolarityEventConst;

/**
 * Polarity event packet data structure definition.
 * EventPackets are always made up of the common packet header,
 * followed by 'eventCapacity' events. Everything has to
 * be in one contiguous memory block.
 */
PACKED_STRUCT(
struct caer_polarity_event_packet {
	/// The common event packet header.
	struct caer_event_packet_header packetHeader;
	/// The events array.
	struct caer_polarity_event events[];
});

/**
 * Type for pointer to polarity event packet data structure.
 */
typedef struct caer_polarity_event_packet *caerPolarityEventPacket;
typedef const struct caer_polarity_event_packet *caerPolarityEventPacketConst;

/**
 * Allocate a new polarity events packet.
 * Use free() to reclaim this memory.
 *
 * @param eventCapacity the maximum number of events this packet will hold.
 * @param eventSource the unique ID representing the source/generator of this packet.
 * @param tsOverflow the current timestamp overflow counter value for this packet.
 *
 * @return a valid PolarityEventPacket handle or NULL on error.
 */
caerPolarityEventPacket caerPolarityEventPacketAllocate(int32_t eventCapacity, int16_t eventSource, int32_t tsOverflow);

/**
 * Transform a generic event packet header into a Polarity event packet.
 * This takes care of proper casting and checks that the packet type really matches
 * the intended conversion type.
 *
 * @param header a valid event packet header pointer. Cannot be NULL.
 * @return a properly converted, typed event packet pointer.
 */
static inline caerPolarityEventPacket caerPolarityEventPacketFromPacketHeader(caerEventPacketHeader header) {
	if (caerEventPacketHeaderGetEventType(header) != POLARITY_EVENT) {
		return (NULL);
	}

	return ((caerPolarityEventPacket) header);
}

/**
 * Transform a generic read-only event packet header into a read-only Polarity event packet.
 * This takes care of proper casting and checks that the packet type really matches
 * the intended conversion type.
 *
 * @param header a valid read-only event packet header pointer. Cannot be NULL.
 * @return a properly converted, read-only typed event packet pointer.
 */
static inline caerPolarityEventPacketConst caerPolarityEventPacketFromPacketHeaderConst(caerEventPacketHeaderConst header) {
	if (caerEventPacketHeaderGetEventType(header) != POLARITY_EVENT) {
		return (NULL);
	}

	return ((caerPolarityEventPacketConst) header);
}

/**
 * Get the polarity event at the given index from the event packet.
 *
 * @param packet a valid PolarityEventPacket pointer. Cannot be NULL.
 * @param n the index of the returned event. Must be within [0,eventCapacity[ bounds.
 *
 * @return the requested polarity event. NULL on error.
 */
static inline caerPolarityEvent caerPolarityEventPacketGetEvent(caerPolarityEventPacket packet, int32_t n) {
	// Check that we're not out of bounds.
	if (n < 0 || n >= caerEventPacketHeaderGetEventCapacity(&packet->packetHeader)) {
		caerLog(CAER_LOG_CRITICAL, "Polarity Event",
			"Called caerPolarityEventPacketGetEvent() with invalid event offset %" PRIi32 ", while maximum allowed value is %" PRIi32 ".",
			n, caerEventPacketHeaderGetEventCapacity(&packet->packetHeader) - 1);
		return (NULL);
	}

	// Return a pointer to the specified event.
	return (packet->events + n);
}

/**
 * Get the polarity event at the given index from the event packet.
 * This is a read-only event, do not change its contents in any way!
 *
 * @param packet a valid PolarityEventPacket pointer. Cannot be NULL.
 * @param n the index of the returned event. Must be within [0,eventCapacity[ bounds.
 *
 * @return the requested read-only polarity event. NULL on error.
 */
static inline caerPolarityEventConst caerPolarityEventPacketGetEventConst(caerPolarityEventPacketConst packet, int32_t n) {
	// Check that we're not out of bounds.
	if (n < 0 || n >= caerEventPacketHeaderGetEventCapacity(&packet->packetHeader)) {
		caerLog(CAER_LOG_CRITICAL, "Polarity Event",
			"Called caerPolarityEventPacketGetEventConst() with invalid event offset %" PRIi32 ", while maximum allowed value is %" PRIi32 ".",
			n, caerEventPacketHeaderGetEventCapacity(&packet->packetHeader) - 1);
		return (NULL);
	}

	// Return a pointer to the specified event.
	return (packet->events + n);
}

/**
 * Get the 32bit event timestamp, in microseconds.
 * Be aware that this wraps around! You can either ignore this fact,
 * or handle the special 'TIMESTAMP_WRAP' event that is generated when
 * this happens, or use the 64bit timestamp which never wraps around.
 * See 'caerEventPacketHeaderGetEventTSOverflow()' documentation
 * for more details on the 64bit timestamp.
 *
 * @param event a valid PolarityEvent pointer. Cannot be NULL.
 *
 * @return this event's 32bit microsecond timestamp.
 */
static inline int32_t caerPolarityEventGetTimestamp(caerPolarityEventConst event) {
	return (le32toh(event->timestamp));
}

/**
 * Get the 64bit event timestamp, in microseconds.
 * See 'caerEventPacketHeaderGetEventTSOverflow()' documentation
 * for more details on the 64bit timestamp.
 *
 * @param event a valid PolarityEvent pointer. Cannot be NULL.
 * @param packet the PolarityEventPacket pointer for the packet containing this event. Cannot be NULL.
 *
 * @return this event's 64bit microsecond timestamp.
 */
static inline int64_t caerPolarityEventGetTimestamp64(caerPolarityEventConst event, caerPolarityEventPacketConst packet) {
	return (I64T(
		(U64T(caerEventPacketHeaderGetEventTSOverflow(&packet->packetHeader)) << TS_OVERFLOW_SHIFT) | U64T(caerPolarityEventGetTimestamp(event))));
}

/**
 * Set the 32bit event timestamp, the value has to be in microseconds.
 *
 * @param event a valid PolarityEvent pointer. Cannot be NULL.
 * @param timestamp a positive 32bit microsecond timestamp.
 */
static inline void caerPolarityEventSetTimestamp(caerPolarityEvent event, int32_t timestamp) {
	if (timestamp < 0) {
		// Negative means using the 31st bit!
		caerLog(CAER_LOG_CRITICAL, "Polarity Event", "Called caerPolarityEventSetTimestamp() with negative value!");
		return;
	}

	event->timestamp = htole32(timestamp);
}

/**
 * Check if this polarity event is valid.
 *
 * @param event a valid PolarityEvent pointer. Cannot be NULL.
 *
 * @return true if valid, false if not.
 */
static inline bool caerPolarityEventIsValid(caerPolarityEventConst event) {
	return (GET_NUMBITS32(event->data, VALID_MARK_SHIFT, VALID_MARK_MASK));
}

/**
 * Validate the current event by setting its valid bit to true
 * and increasing the event packet's event count and valid
 * event count. Only works on events that are invalid.
 * DO NOT CALL THIS AFTER HAVING PREVIOUSLY ALREADY
 * INVALIDATED THIS EVENT, the total count will be incorrect.
 *
 * @param event a valid PolarityEvent pointer. Cannot be NULL.
 * @param packet the PolarityEventPacket pointer for the packet containing this event. Cannot be NULL.
 */
static inline void caerPolarityEventValidate(caerPolarityEvent event, caerPolarityEventPacket packet) {
	if (!caerPolarityEventIsValid(event)) {
		SET_NUMBITS32(event->data, VALID_MARK_SHIFT, VALID_MARK_MASK, 1);

		// Also increase number of events and valid events.
		// Only call this on (still) invalid events!
		caerEventPacketHeaderSetEventNumber(&packet->packetHeader,
			caerEventPacketHeaderGetEventNumber(&packet->packetHeader) + 1);
		caerEventPacketHeaderSetEventValid(&packet->packetHeader,
			caerEventPacketHeaderGetEventValid(&packet->packetHeader) + 1);
	}
	else {
		caerLog(CAER_LOG_CRITICAL, "Polarity Event", "Called caerPolarityEventValidate() on already valid event.");
	}
}

/**
 * Invalidate the current event by setting its valid bit
 * to false and decreasing the number of valid events held
 * in the packet. Only works with events that are already
 * valid!
 *
 * @param event a valid PolarityEvent pointer. Cannot be NULL.
 * @param packet the PolarityEventPacket pointer for the packet containing this event. Cannot be NULL.
 */
static inline void caerPolarityEventInvalidate(caerPolarityEvent event, caerPolarityEventPacket packet) {
	if (caerPolarityEventIsValid(event)) {
		CLEAR_NUMBITS32(event->data, VALID_MARK_SHIFT, VALID_MARK_MASK);

		// Also decrease number of valid events. Number of total events doesn't change.
		// Only call this on valid events!
		caerEventPacketHeaderSetEventValid(&packet->packetHeader,
			caerEventPacketHeaderGetEventValid(&packet->packetHeader) - 1);
	}
	else {
		caerLog(CAER_LOG_CRITICAL, "Polarity Event", "Called caerPolarityEventInvalidate() on already invalid event.");
	}
}

/**
 * Get the change event polarity. 1 is ON, 0 is OFF.
 *
 * @param event a valid PolarityEvent pointer. Cannot be NULL.
 *
 * @return event polarity value.
 */
static inline bool caerPolarityEventGetPolarity(caerPolarityEventConst event) {
	return (GET_NUMBITS32(event->data, POLARITY_SHIFT, POLARITY_MASK));
}

/**
 * Set the change event polarity. 1 is ON, 0 is OFF.
 *
 * @param event a valid PolarityEvent pointer. Cannot be NULL.
 * @param polarity event polarity value.
 */
static inline void caerPolarityEventSetPolarity(caerPolarityEvent event, bool polarity) {
	CLEAR_NUMBITS32(event->data, POLARITY_SHIFT, POLARITY_MASK);
	SET_NUMBITS32(event->data, POLARITY_SHIFT, POLARITY_MASK, polarity);
}

/**
 * Get the Y (row) address for a change event, in pixels.
 * The (0, 0) address is in the upper left corner, like in OpenCV/computer graphics.
 *
 * @param event a valid PolarityEvent pointer. Cannot be NULL.
 *
 * @return the event Y address.
 */
static inline uint16_t caerPolarityEventGetY(caerPolarityEventConst event) {
	return U16T(GET_NUMBITS32(event->data, POLARITY_Y_ADDR_SHIFT, POLARITY_Y_ADDR_MASK));
}

/**
 * Set the Y (row) address for a change event, in pixels.
 * The (0, 0) address is in the upper left corner, like in OpenCV/computer graphics.
 *
 * @param event a valid PolarityEvent pointer. Cannot be NULL.
 * @param yAddress the event Y address.
 */
static inline void caerPolarityEventSetY(caerPolarityEvent event, uint16_t yAddress) {
	CLEAR_NUMBITS32(event->data, POLARITY_Y_ADDR_SHIFT, POLARITY_Y_ADDR_MASK);
	SET_NUMBITS32(event->data, POLARITY_Y_ADDR_SHIFT, POLARITY_Y_ADDR_MASK, yAddress);
}

/**
 * Get the X (column) address for a change event, in pixels.
 * The (0, 0) address is in the upper left corner, like in OpenCV/computer graphics.
 *
 * @param event a valid PolarityEvent pointer. Cannot be NULL.
 *
 * @return the event X address.
 */
static inline uint16_t caerPolarityEventGetX(caerPolarityEventConst event) {
	return U16T(GET_NUMBITS32(event->data, POLARITY_X_ADDR_SHIFT, POLARITY_X_ADDR_MASK));
}

/**
 * Set the X (column) address for a change event, in pixels.
 * The (0, 0) address is in the upper left corner, like in OpenCV/computer graphics.
 *
 * @param event a valid PolarityEvent pointer. Cannot be NULL.
 * @param xAddress the event X address.
 */
static inline void caerPolarityEventSetX(caerPolarityEvent event, uint16_t xAddress) {
	CLEAR_NUMBITS32(event->data, POLARITY_X_ADDR_SHIFT, POLARITY_X_ADDR_MASK);
	SET_NUMBITS32(event->data, POLARITY_X_ADDR_SHIFT, POLARITY_X_ADDR_MASK, xAddress);
}

/**
 * Iterator over all polarity events in a packet.
 * Returns the current index in the 'caerPolarityIteratorCounter' variable of type
 * 'int32_t' and the current event in the 'caerPolarityIteratorElement' variable
 * of type caerPolarityEvent.
 *
 * POLARITY_PACKET: a valid PolarityEventPacket pointer. Cannot be NULL.
 */
#define CAER_POLARITY_ITERATOR_ALL_START(POLARITY_PACKET) \
	for (int32_t caerPolarityIteratorCounter = 0; \
		caerPolarityIteratorCounter < caerEventPacketHeaderGetEventNumber(&(POLARITY_PACKET)->packetHeader); \
		caerPolarityIteratorCounter++) { \
		caerPolarityEvent caerPolarityIteratorElement = caerPolarityEventPacketGetEvent(POLARITY_PACKET, caerPolarityIteratorCounter);

/**
 * Const-Iterator over all polarity events in a packet.
 * Returns the current index in the 'caerPolarityIteratorCounter' variable of type
 * 'int32_t' and the current read-only event in the 'caerPolarityIteratorElement' variable
 * of type caerPolarityEventConst.
 *
 * POLARITY_PACKET: a valid PolarityEventPacket pointer. Cannot be NULL.
 */
#define CAER_POLARITY_CONST_ITERATOR_ALL_START(POLARITY_PACKET) \
	for (int32_t caerPolarityIteratorCounter = 0; \
		caerPolarityIteratorCounter < caerEventPacketHeaderGetEventNumber(&(POLARITY_PACKET)->packetHeader); \
		caerPolarityIteratorCounter++) { \
		caerPolarityEventConst caerPolarityIteratorElement = caerPolarityEventPacketGetEventConst(POLARITY_PACKET, caerPolarityIteratorCounter);

/**
 * Iterator close statement.
 */
#define CAER_POLARITY_ITERATOR_ALL_END }

/**
 * Iterator over only the valid polarity events in a packet.
 * Returns the current index in the 'caerPolarityIteratorCounter' variable of type
 * 'int32_t' and the current event in the 'caerPolarityIteratorElement' variable
 * of type caerPolarityEvent.
 *
 * POLARITY_PACKET: a valid PolarityEventPacket pointer. Cannot be NULL.
 */
#define CAER_POLARITY_ITERATOR_VALID_START(POLARITY_PACKET) \
	for (int32_t caerPolarityIteratorCounter = 0; \
		caerPolarityIteratorCounter < caerEventPacketHeaderGetEventNumber(&(POLARITY_PACKET)->packetHeader); \
		caerPolarityIteratorCounter++) { \
		caerPolarityEvent caerPolarityIteratorElement = caerPolarityEventPacketGetEvent(POLARITY_PACKET, caerPolarityIteratorCounter); \
		if (!caerPolarityEventIsValid(caerPolarityIteratorElement)) { continue; } // Skip invalid polarity events.

/**
 * Const-Iterator over only the valid polarity events in a packet.
 * Returns the current index in the 'caerPolarityIteratorCounter' variable of type
 * 'int32_t' and the current read-only event in the 'caerPolarityIteratorElement' variable
 * of type caerPolarityEventConst.
 *
 * POLARITY_PACKET: a valid PolarityEventPacket pointer. Cannot be NULL.
 */
#define CAER_POLARITY_CONST_ITERATOR_VALID_START(POLARITY_PACKET) \
	for (int32_t caerPolarityIteratorCounter = 0; \
		caerPolarityIteratorCounter < caerEventPacketHeaderGetEventNumber(&(POLARITY_PACKET)->packetHeader); \
		caerPolarityIteratorCounter++) { \
		caerPolarityEventConst caerPolarityIteratorElement = caerPolarityEventPacketGetEventConst(POLARITY_PACKET, caerPolarityIteratorCounter); \
		if (!caerPolarityEventIsValid(caerPolarityIteratorElement)) { continue; } // Skip invalid polarity events.

/**
 * Iterator close statement.
 */
#define CAER_POLARITY_ITERATOR_VALID_END }

/**
 * Reverse iterator over all polarity events in a packet.
 * Returns the current index in the 'caerPolarityIteratorCounter' variable of type
 * 'int32_t' and the current event in the 'caerPolarityIteratorElement' variable
 * of type caerPolarityEvent.
 *
 * POLARITY_PACKET: a valid PolarityEventPacket pointer. Cannot be NULL.
 */
#define CAER_POLARITY_REVERSE_ITERATOR_ALL_START(POLARITY_PACKET) \
	for (int32_t caerPolarityIteratorCounter = caerEventPacketHeaderGetEventNumber(&(POLARITY_PACKET)->packetHeader) - 1; \
		caerPolarityIteratorCounter >= 0; \
		caerPolarityIteratorCounter--) { \
		caerPolarityEvent caerPolarityIteratorElement = caerPolarityEventPacketGetEvent(POLARITY_PACKET, caerPolarityIteratorCounter);

/**
 * Const-Reverse iterator over all polarity events in a packet.
 * Returns the current index in the 'caerPolarityIteratorCounter' variable of type
 * 'int32_t' and the current read-only event in the 'caerPolarityIteratorElement' variable
 * of type caerPolarityEventConst.
 *
 * POLARITY_PACKET: a valid PolarityEventPacket pointer. Cannot be NULL.
 */
#define CAER_POLARITY_CONST_REVERSE_ITERATOR_ALL_START(POLARITY_PACKET) \
	for (int32_t caerPolarityIteratorCounter = caerEventPacketHeaderGetEventNumber(&(POLARITY_PACKET)->packetHeader) - 1; \
		caerPolarityIteratorCounter >= 0; \
		caerPolarityIteratorCounter--) { \
		caerPolarityEventConst caerPolarityIteratorElement = caerPolarityEventPacketGetEventConst(POLARITY_PACKET, caerPolarityIteratorCounter);

/**
 * Reverse iterator close statement.
 */
#define CAER_POLARITY_REVERSE_ITERATOR_ALL_END }

/**
 * Reverse iterator over only the valid polarity events in a packet.
 * Returns the current index in the 'caerPolarityIteratorCounter' variable of type
 * 'int32_t' and the current event in the 'caerPolarityIteratorElement' variable
 * of type caerPolarityEvent.
 *
 * POLARITY_PACKET: a valid PolarityEventPacket pointer. Cannot be NULL.
 */
#define CAER_POLARITY_REVERSE_ITERATOR_VALID_START(POLARITY_PACKET) \
	for (int32_t caerPolarityIteratorCounter = caerEventPacketHeaderGetEventNumber(&(POLARITY_PACKET)->packetHeader) - 1; \
		caerPolarityIteratorCounter >= 0; \
		caerPolarityIteratorCounter--) { \
		caerPolarityEvent caerPolarityIteratorElement = caerPolarityEventPacketGetEvent(POLARITY_PACKET, caerPolarityIteratorCounter); \
		if (!caerPolarityEventIsValid(caerPolarityIteratorElement)) { continue; } // Skip invalid polarity events.

/**
 * Const-Reverse iterator over only the valid polarity events in a packet.
 * Returns the current index in the 'caerPolarityIteratorCounter' variable of type
 * 'int32_t' and the current read-only event in the 'caerPolarityIteratorElement' variable
 * of type caerPolarityEventConst.
 *
 * POLARITY_PACKET: a valid PolarityEventPacket pointer. Cannot be NULL.
 */
#define CAER_POLARITY_CONST_REVERSE_ITERATOR_VALID_START(POLARITY_PACKET) \
	for (int32_t caerPolarityIteratorCounter = caerEventPacketHeaderGetEventNumber(&(POLARITY_PACKET)->packetHeader) - 1; \
		caerPolarityIteratorCounter >= 0; \
		caerPolarityIteratorCounter--) { \
		caerPolarityEventConst caerPolarityIteratorElement = caerPolarityEventPacketGetEventConst(POLARITY_PACKET, caerPolarityIteratorCounter); \
		if (!caerPolarityEventIsValid(caerPolarityIteratorElement)) { continue; } // Skip invalid polarity events.

/**
 * Reverse iterator close statement.
 */
#define CAER_POLARITY_REVERSE_ITERATOR_VALID_END }

#ifdef __cplusplus
}
#endif

#endif /* LIBCAER_EVENTS_POLARITY_H_ */
