
# Copyright 2016 The Kubernetes Authors.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
# bash completion for kubectl                              -*- shell-script -*-

__kubectl_debug()
{
    if [[ -n ${BASH_COMP_DEBUG_FILE} ]]; then
        echo "$*" >> "${BASH_COMP_DEBUG_FILE}"
    fi
}

# Homebrew on Macs have version 1.3 of bash-completion which doesn't include
# _init_completion. This is a very minimal version of that function.
__kubectl_init_completion()
{
    COMPREPLY=()
    _get_comp_words_by_ref "$@" cur prev words cword
}

__kubectl_index_of_word()
{
    local w word=$1
    shift
    index=0
    for w in "$@"; do
        [[ $w = "$word" ]] && return
        index=$((index+1))
    done
    index=-1
}

__kubectl_contains_word()
{
    local w word=$1; shift
    for w in "$@"; do
        [[ $w = "$word" ]] && return
    done
    return 1
}

__kubectl_handle_go_custom_completion()
{
    __kubectl_debug "${FUNCNAME[0]}: cur is ${cur}, words[*] is ${words[*]}, #words[@] is ${#words[@]}"

    local shellCompDirectiveError=1
    local shellCompDirectiveNoSpace=2
    local shellCompDirectiveNoFileComp=4
    local shellCompDirectiveFilterFileExt=8
    local shellCompDirectiveFilterDirs=16

    local out requestComp lastParam lastChar comp directive args

    # Prepare the command to request completions for the program.
    # Calling ${words[0]} instead of directly kubectl allows to handle aliases
    args=("${words[@]:1}")
    requestComp="${words[0]} __completeNoDesc ${args[*]}"

    lastParam=${words[$((${#words[@]}-1))]}
    lastChar=${lastParam:$((${#lastParam}-1)):1}
    __kubectl_debug "${FUNCNAME[0]}: lastParam ${lastParam}, lastChar ${lastChar}"

    if [ -z "${cur}" ] && [ "${lastChar}" != "=" ]; then
        # If the last parameter is complete (there is a space following it)
        # We add an extra empty parameter so we can indicate this to the go method.
        __kubectl_debug "${FUNCNAME[0]}: Adding extra empty parameter"
        requestComp="${requestComp} \"\""
    fi

    __kubectl_debug "${FUNCNAME[0]}: calling ${requestComp}"
    # Use eval to handle any environment variables and such
    out=$(eval "${requestComp}" 2>/dev/null)

    # Extract the directive integer at the very end of the output following a colon (:)
    directive=${out##*:}
    # Remove the directive
    out=${out%:*}
    if [ "${directive}" = "${out}" ]; then
        # There is not directive specified
        directive=0
    fi
    __kubectl_debug "${FUNCNAME[0]}: the completion directive is: ${directive}"
    __kubectl_debug "${FUNCNAME[0]}: the completions are: ${out[*]}"

    if [ $((directive & shellCompDirectiveError)) -ne 0 ]; then
        # Error code.  No completion.
        __kubectl_debug "${FUNCNAME[0]}: received error from custom completion go code"
        return
    else
        if [ $((directive & shellCompDirectiveNoSpace)) -ne 0 ]; then
            if [[ $(type -t compopt) = "builtin" ]]; then
                __kubectl_debug "${FUNCNAME[0]}: activating no space"
                compopt -o nospace
            fi
        fi
        if [ $((directive & shellCompDirectiveNoFileComp)) -ne 0 ]; then
            if [[ $(type -t compopt) = "builtin" ]]; then
                __kubectl_debug "${FUNCNAME[0]}: activating no file completion"
                compopt +o default
            fi
        fi
    fi

    if [ $((directive & shellCompDirectiveFilterFileExt)) -ne 0 ]; then
        # File extension filtering
        local fullFilter filter filteringCmd
        # Do not use quotes around the $out variable or else newline
        # characters will be kept.
        for filter in ${out[*]}; do
            fullFilter+="$filter|"
        done

        filteringCmd="_filedir $fullFilter"
        __kubectl_debug "File filtering command: $filteringCmd"
        $filteringCmd
    elif [ $((directive & shellCompDirectiveFilterDirs)) -ne 0 ]; then
        # File completion for directories only
        local subDir
        # Use printf to strip any trailing newline
        subdir=$(printf "%s" "${out[0]}")
        if [ -n "$subdir" ]; then
            __kubectl_debug "Listing directories in $subdir"
            __kubectl_handle_subdirs_in_dir_flag "$subdir"
        else
            __kubectl_debug "Listing directories in ."
            _filedir -d
        fi
    else
        while IFS='' read -r comp; do
            COMPREPLY+=("$comp")
        done < <(compgen -W "${out[*]}" -- "$cur")
    fi
}

__kubectl_handle_reply()
{
    __kubectl_debug "${FUNCNAME[0]}"
    local comp
    case $cur in
        -*)
            if [[ $(type -t compopt) = "builtin" ]]; then
                compopt -o nospace
            fi
            local allflags
            if [ ${#must_have_one_flag[@]} -ne 0 ]; then
                allflags=("${must_have_one_flag[@]}")
            else
                allflags=("${flags[*]} ${two_word_flags[*]}")
            fi
            while IFS='' read -r comp; do
                COMPREPLY+=("$comp")
            done < <(compgen -W "${allflags[*]}" -- "$cur")
            if [[ $(type -t compopt) = "builtin" ]]; then
                [[ "${COMPREPLY[0]}" == *= ]] || compopt +o nospace
            fi

            # complete after --flag=abc
            if [[ $cur == *=* ]]; then
                if [[ $(type -t compopt) = "builtin" ]]; then
                    compopt +o nospace
                fi

                local index flag
                flag="${cur%=*}"
                __kubectl_index_of_word "${flag}" "${flags_with_completion[@]}"
                COMPREPLY=()
                if [[ ${index} -ge 0 ]]; then
                    PREFIX=""
                    cur="${cur#*=}"
                    ${flags_completion[${index}]}
                    if [ -n "${ZSH_VERSION}" ]; then
                        # zsh completion needs --flag= prefix
                        eval "COMPREPLY=( \"\${COMPREPLY[@]/#/${flag}=}\" )"
                    fi
                fi
            fi
            return 0;
            ;;
    esac

    # check if we are handling a flag with special work handling
    local index
    __kubectl_index_of_word "${prev}" "${flags_with_completion[@]}"
    if [[ ${index} -ge 0 ]]; then
        ${flags_completion[${index}]}
        return
    fi

    # we are parsing a flag and don't have a special handler, no completion
    if [[ ${cur} != "${words[cword]}" ]]; then
        return
    fi

    local completions
    completions=("${commands[@]}")
    if [[ ${#must_have_one_noun[@]} -ne 0 ]]; then
        completions+=("${must_have_one_noun[@]}")
    elif [[ -n "${has_completion_function}" ]]; then
        # if a go completion function is provided, defer to that function
        __kubectl_handle_go_custom_completion
    fi
    if [[ ${#must_have_one_flag[@]} -ne 0 ]]; then
        completions+=("${must_have_one_flag[@]}")
    fi
    while IFS='' read -r comp; do
        COMPREPLY+=("$comp")
    done < <(compgen -W "${completions[*]}" -- "$cur")

    if [[ ${#COMPREPLY[@]} -eq 0 && ${#noun_aliases[@]} -gt 0 && ${#must_have_one_noun[@]} -ne 0 ]]; then
        while IFS='' read -r comp; do
            COMPREPLY+=("$comp")
        done < <(compgen -W "${noun_aliases[*]}" -- "$cur")
    fi

    if [[ ${#COMPREPLY[@]} -eq 0 ]]; then
		if declare -F __kubectl_custom_func >/dev/null; then
			# try command name qualified custom func
			__kubectl_custom_func
		else
			# otherwise fall back to unqualified for compatibility
			declare -F __custom_func >/dev/null && __custom_func
		fi
    fi

    # available in bash-completion >= 2, not always present on macOS
    if declare -F __ltrim_colon_completions >/dev/null; then
        __ltrim_colon_completions "$cur"
    fi

    # If there is only 1 completion and it is a flag with an = it will be completed
    # but we don't want a space after the =
    if [[ "${#COMPREPLY[@]}" -eq "1" ]] && [[ $(type -t compopt) = "builtin" ]] && [[ "${COMPREPLY[0]}" == --*= ]]; then
       compopt -o nospace
    fi
}

# The arguments should be in the form "ext1|ext2|extn"
__kubectl_handle_filename_extension_flag()
{
    local ext="$1"
    _filedir "@(${ext})"
}

__kubectl_handle_subdirs_in_dir_flag()
{
    local dir="$1"
    pushd "${dir}" >/dev/null 2>&1 && _filedir -d && popd >/dev/null 2>&1 || return
}

__kubectl_handle_flag()
{
    __kubectl_debug "${FUNCNAME[0]}: c is $c words[c] is ${words[c]}"

    # if a command required a flag, and we found it, unset must_have_one_flag()
    local flagname=${words[c]}
    local flagvalue
    # if the word contained an =
    if [[ ${words[c]} == *"="* ]]; then
        flagvalue=${flagname#*=} # take in as flagvalue after the =
        flagname=${flagname%=*} # strip everything after the =
        flagname="${flagname}=" # but put the = back
    fi
    __kubectl_debug "${FUNCNAME[0]}: looking for ${flagname}"
    if __kubectl_contains_word "${flagname}" "${must_have_one_flag[@]}"; then
        must_have_one_flag=()
    fi

    # if you set a flag which only applies to this command, don't show subcommands
    if __kubectl_contains_word "${flagname}" "${local_nonpersistent_flags[@]}"; then
      commands=()
    fi

    # keep flag value with flagname as flaghash
    # flaghash variable is an associative array which is only supported in bash > 3.
    if [[ -z "${BASH_VERSION}" || "${BASH_VERSINFO[0]}" -gt 3 ]]; then
        if [ -n "${flagvalue}" ] ; then
            flaghash[${flagname}]=${flagvalue}
        elif [ -n "${words[ $((c+1)) ]}" ] ; then
            flaghash[${flagname}]=${words[ $((c+1)) ]}
        else
            flaghash[${flagname}]="true" # pad "true" for bool flag
        fi
    fi

    # skip the argument to a two word flag
    if [[ ${words[c]} != *"="* ]] && __kubectl_contains_word "${words[c]}" "${two_word_flags[@]}"; then
			  __kubectl_debug "${FUNCNAME[0]}: found a flag ${words[c]}, skip the next argument"
        c=$((c+1))
        # if we are looking for a flags value, don't show commands
        if [[ $c -eq $cword ]]; then
            commands=()
        fi
    fi

    c=$((c+1))

}

__kubectl_handle_noun()
{
    __kubectl_debug "${FUNCNAME[0]}: c is $c words[c] is ${words[c]}"

    if __kubectl_contains_word "${words[c]}" "${must_have_one_noun[@]}"; then
        must_have_one_noun=()
    elif __kubectl_contains_word "${words[c]}" "${noun_aliases[@]}"; then
        must_have_one_noun=()
    fi

    nouns+=("${words[c]}")
    c=$((c+1))
}

__kubectl_handle_command()
{
    __kubectl_debug "${FUNCNAME[0]}: c is $c words[c] is ${words[c]}"

    local next_command
    if [[ -n ${last_command} ]]; then
        next_command="_${last_command}_${words[c]//:/__}"
    else
        if [[ $c -eq 0 ]]; then
            next_command="_kubectl_root_command"
        else
            next_command="_${words[c]//:/__}"
        fi
    fi
    c=$((c+1))
    __kubectl_debug "${FUNCNAME[0]}: looking for ${next_command}"
    declare -F "$next_command" >/dev/null && $next_command
}

__kubectl_handle_word()
{
    if [[ $c -ge $cword ]]; then
        __kubectl_handle_reply
        return
    fi
    __kubectl_debug "${FUNCNAME[0]}: c is $c words[c] is ${words[c]}"
    if [[ "${words[c]}" == -* ]]; then
        __kubectl_handle_flag
    elif __kubectl_contains_word "${words[c]}" "${commands[@]}"; then
        __kubectl_handle_command
    elif [[ $c -eq 0 ]]; then
        __kubectl_handle_command
    elif __kubectl_contains_word "${words[c]}" "${command_aliases[@]}"; then
        # aliashash variable is an associative array which is only supported in bash > 3.
        if [[ -z "${BASH_VERSION}" || "${BASH_VERSINFO[0]}" -gt 3 ]]; then
            words[c]=${aliashash[${words[c]}]}
            __kubectl_handle_command
        else
            __kubectl_handle_noun
        fi
    else
        __kubectl_handle_noun
    fi
    __kubectl_handle_word
}

_kubectl_alpha_events()
{
    last_command="kubectl_alpha_events"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--all-namespaces")
    flags+=("-A")
    local_nonpersistent_flags+=("--all-namespaces")
    local_nonpersistent_flags+=("-A")
    flags+=("--chunk-size=")
    two_word_flags+=("--chunk-size")
    local_nonpersistent_flags+=("--chunk-size")
    local_nonpersistent_flags+=("--chunk-size=")
    flags+=("--for=")
    two_word_flags+=("--for")
    local_nonpersistent_flags+=("--for")
    local_nonpersistent_flags+=("--for=")
    flags+=("--watch")
    flags+=("-w")
    local_nonpersistent_flags+=("--watch")
    local_nonpersistent_flags+=("-w")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_alpha()
{
    last_command="kubectl_alpha"

    command_aliases=()

    commands=()
    commands+=("events")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_annotate()
{
    last_command="kubectl_annotate"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--all")
    local_nonpersistent_flags+=("--all")
    flags+=("--all-namespaces")
    flags+=("-A")
    local_nonpersistent_flags+=("--all-namespaces")
    local_nonpersistent_flags+=("-A")
    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--field-manager=")
    two_word_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager=")
    flags+=("--field-selector=")
    two_word_flags+=("--field-selector")
    local_nonpersistent_flags+=("--field-selector")
    local_nonpersistent_flags+=("--field-selector=")
    flags+=("--filename=")
    two_word_flags+=("--filename")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename")
    local_nonpersistent_flags+=("--filename=")
    local_nonpersistent_flags+=("-f")
    flags+=("--kustomize=")
    two_word_flags+=("--kustomize")
    two_word_flags+=("-k")
    local_nonpersistent_flags+=("--kustomize")
    local_nonpersistent_flags+=("--kustomize=")
    local_nonpersistent_flags+=("-k")
    flags+=("--list")
    local_nonpersistent_flags+=("--list")
    flags+=("--local")
    local_nonpersistent_flags+=("--local")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--overwrite")
    local_nonpersistent_flags+=("--overwrite")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    local_nonpersistent_flags+=("-R")
    flags+=("--resource-version=")
    two_word_flags+=("--resource-version")
    local_nonpersistent_flags+=("--resource-version")
    local_nonpersistent_flags+=("--resource-version=")
    flags+=("--selector=")
    two_word_flags+=("--selector")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--selector")
    local_nonpersistent_flags+=("--selector=")
    local_nonpersistent_flags+=("-l")
    flags+=("--show-managed-fields")
    local_nonpersistent_flags+=("--show-managed-fields")
    flags+=("--template=")
    two_word_flags+=("--template")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template")
    local_nonpersistent_flags+=("--template=")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    has_completion_function=1
    noun_aliases=()
}

_kubectl_api-resources()
{
    last_command="kubectl_api-resources"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-group=")
    two_word_flags+=("--api-group")
    local_nonpersistent_flags+=("--api-group")
    local_nonpersistent_flags+=("--api-group=")
    flags+=("--cached")
    local_nonpersistent_flags+=("--cached")
    flags+=("--namespaced")
    local_nonpersistent_flags+=("--namespaced")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--sort-by=")
    two_word_flags+=("--sort-by")
    local_nonpersistent_flags+=("--sort-by")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--verbs=")
    two_word_flags+=("--verbs")
    local_nonpersistent_flags+=("--verbs")
    local_nonpersistent_flags+=("--verbs=")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_api-versions()
{
    last_command="kubectl_api-versions"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_apply_edit-last-applied()
{
    last_command="kubectl_apply_edit-last-applied"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--field-manager=")
    two_word_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager=")
    flags+=("--filename=")
    two_word_flags+=("--filename")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename")
    local_nonpersistent_flags+=("--filename=")
    local_nonpersistent_flags+=("-f")
    flags+=("--kustomize=")
    two_word_flags+=("--kustomize")
    two_word_flags+=("-k")
    local_nonpersistent_flags+=("--kustomize")
    local_nonpersistent_flags+=("--kustomize=")
    local_nonpersistent_flags+=("-k")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    local_nonpersistent_flags+=("-R")
    flags+=("--show-managed-fields")
    local_nonpersistent_flags+=("--show-managed-fields")
    flags+=("--template=")
    two_word_flags+=("--template")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template")
    local_nonpersistent_flags+=("--template=")
    flags+=("--windows-line-endings")
    local_nonpersistent_flags+=("--windows-line-endings")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    has_completion_function=1
    noun_aliases=()
}

_kubectl_apply_set-last-applied()
{
    last_command="kubectl_apply_set-last-applied"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--create-annotation")
    local_nonpersistent_flags+=("--create-annotation")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--filename=")
    two_word_flags+=("--filename")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename")
    local_nonpersistent_flags+=("--filename=")
    local_nonpersistent_flags+=("-f")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--show-managed-fields")
    local_nonpersistent_flags+=("--show-managed-fields")
    flags+=("--template=")
    two_word_flags+=("--template")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template")
    local_nonpersistent_flags+=("--template=")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_apply_view-last-applied()
{
    last_command="kubectl_apply_view-last-applied"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--all")
    local_nonpersistent_flags+=("--all")
    flags+=("--filename=")
    two_word_flags+=("--filename")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename")
    local_nonpersistent_flags+=("--filename=")
    local_nonpersistent_flags+=("-f")
    flags+=("--kustomize=")
    two_word_flags+=("--kustomize")
    two_word_flags+=("-k")
    local_nonpersistent_flags+=("--kustomize")
    local_nonpersistent_flags+=("--kustomize=")
    local_nonpersistent_flags+=("-k")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    local_nonpersistent_flags+=("-R")
    flags+=("--selector=")
    two_word_flags+=("--selector")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--selector")
    local_nonpersistent_flags+=("--selector=")
    local_nonpersistent_flags+=("-l")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    has_completion_function=1
    noun_aliases=()
}

_kubectl_apply()
{
    last_command="kubectl_apply"

    command_aliases=()

    commands=()
    commands+=("edit-last-applied")
    commands+=("set-last-applied")
    commands+=("view-last-applied")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--all")
    local_nonpersistent_flags+=("--all")
    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--cascade")
    local_nonpersistent_flags+=("--cascade")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--field-manager=")
    two_word_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager=")
    flags+=("--filename=")
    two_word_flags+=("--filename")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename")
    local_nonpersistent_flags+=("--filename=")
    local_nonpersistent_flags+=("-f")
    flags+=("--force")
    local_nonpersistent_flags+=("--force")
    flags+=("--force-conflicts")
    local_nonpersistent_flags+=("--force-conflicts")
    flags+=("--grace-period=")
    two_word_flags+=("--grace-period")
    local_nonpersistent_flags+=("--grace-period")
    local_nonpersistent_flags+=("--grace-period=")
    flags+=("--kustomize=")
    two_word_flags+=("--kustomize")
    two_word_flags+=("-k")
    local_nonpersistent_flags+=("--kustomize")
    local_nonpersistent_flags+=("--kustomize=")
    local_nonpersistent_flags+=("-k")
    flags+=("--openapi-patch")
    local_nonpersistent_flags+=("--openapi-patch")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--overwrite")
    local_nonpersistent_flags+=("--overwrite")
    flags+=("--prune")
    local_nonpersistent_flags+=("--prune")
    flags+=("--prune-whitelist=")
    two_word_flags+=("--prune-whitelist")
    local_nonpersistent_flags+=("--prune-whitelist")
    local_nonpersistent_flags+=("--prune-whitelist=")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    local_nonpersistent_flags+=("-R")
    flags+=("--selector=")
    two_word_flags+=("--selector")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--selector")
    local_nonpersistent_flags+=("--selector=")
    local_nonpersistent_flags+=("-l")
    flags+=("--server-side")
    local_nonpersistent_flags+=("--server-side")
    flags+=("--show-managed-fields")
    local_nonpersistent_flags+=("--show-managed-fields")
    flags+=("--template=")
    two_word_flags+=("--template")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template")
    local_nonpersistent_flags+=("--template=")
    flags+=("--timeout=")
    two_word_flags+=("--timeout")
    local_nonpersistent_flags+=("--timeout")
    local_nonpersistent_flags+=("--timeout=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--wait")
    local_nonpersistent_flags+=("--wait")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_attach()
{
    last_command="kubectl_attach"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--container=")
    two_word_flags+=("--container")
    two_word_flags+=("-c")
    local_nonpersistent_flags+=("--container")
    local_nonpersistent_flags+=("--container=")
    local_nonpersistent_flags+=("-c")
    flags+=("--pod-running-timeout=")
    two_word_flags+=("--pod-running-timeout")
    local_nonpersistent_flags+=("--pod-running-timeout")
    local_nonpersistent_flags+=("--pod-running-timeout=")
    flags+=("--quiet")
    flags+=("-q")
    local_nonpersistent_flags+=("--quiet")
    local_nonpersistent_flags+=("-q")
    flags+=("--stdin")
    flags+=("-i")
    local_nonpersistent_flags+=("--stdin")
    local_nonpersistent_flags+=("-i")
    flags+=("--tty")
    flags+=("-t")
    local_nonpersistent_flags+=("--tty")
    local_nonpersistent_flags+=("-t")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    has_completion_function=1
    noun_aliases=()
}

_kubectl_auth_can-i()
{
    last_command="kubectl_auth_can-i"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--all-namespaces")
    flags+=("-A")
    local_nonpersistent_flags+=("--all-namespaces")
    local_nonpersistent_flags+=("-A")
    flags+=("--list")
    local_nonpersistent_flags+=("--list")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--quiet")
    flags+=("-q")
    local_nonpersistent_flags+=("--quiet")
    local_nonpersistent_flags+=("-q")
    flags+=("--subresource=")
    two_word_flags+=("--subresource")
    local_nonpersistent_flags+=("--subresource")
    local_nonpersistent_flags+=("--subresource=")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_auth_reconcile()
{
    last_command="kubectl_auth_reconcile"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--filename=")
    two_word_flags+=("--filename")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename")
    local_nonpersistent_flags+=("--filename=")
    local_nonpersistent_flags+=("-f")
    flags+=("--kustomize=")
    two_word_flags+=("--kustomize")
    two_word_flags+=("-k")
    local_nonpersistent_flags+=("--kustomize")
    local_nonpersistent_flags+=("--kustomize=")
    local_nonpersistent_flags+=("-k")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    local_nonpersistent_flags+=("-R")
    flags+=("--remove-extra-permissions")
    local_nonpersistent_flags+=("--remove-extra-permissions")
    flags+=("--remove-extra-subjects")
    local_nonpersistent_flags+=("--remove-extra-subjects")
    flags+=("--show-managed-fields")
    local_nonpersistent_flags+=("--show-managed-fields")
    flags+=("--template=")
    two_word_flags+=("--template")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template")
    local_nonpersistent_flags+=("--template=")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_auth()
{
    last_command="kubectl_auth"

    command_aliases=()

    commands=()
    commands+=("can-i")
    commands+=("reconcile")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_autoscale()
{
    last_command="kubectl_autoscale"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--cpu-percent=")
    two_word_flags+=("--cpu-percent")
    local_nonpersistent_flags+=("--cpu-percent")
    local_nonpersistent_flags+=("--cpu-percent=")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--field-manager=")
    two_word_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager=")
    flags+=("--filename=")
    two_word_flags+=("--filename")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename")
    local_nonpersistent_flags+=("--filename=")
    local_nonpersistent_flags+=("-f")
    flags+=("--kustomize=")
    two_word_flags+=("--kustomize")
    two_word_flags+=("-k")
    local_nonpersistent_flags+=("--kustomize")
    local_nonpersistent_flags+=("--kustomize=")
    local_nonpersistent_flags+=("-k")
    flags+=("--max=")
    two_word_flags+=("--max")
    local_nonpersistent_flags+=("--max")
    local_nonpersistent_flags+=("--max=")
    flags+=("--min=")
    two_word_flags+=("--min")
    local_nonpersistent_flags+=("--min")
    local_nonpersistent_flags+=("--min=")
    flags+=("--name=")
    two_word_flags+=("--name")
    local_nonpersistent_flags+=("--name")
    local_nonpersistent_flags+=("--name=")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    local_nonpersistent_flags+=("-R")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--show-managed-fields")
    local_nonpersistent_flags+=("--show-managed-fields")
    flags+=("--template=")
    two_word_flags+=("--template")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template")
    local_nonpersistent_flags+=("--template=")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_flag+=("--max=")
    must_have_one_noun=()
    has_completion_function=1
    noun_aliases=()
}

_kubectl_certificate_approve()
{
    last_command="kubectl_certificate_approve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--filename=")
    two_word_flags+=("--filename")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename")
    local_nonpersistent_flags+=("--filename=")
    local_nonpersistent_flags+=("-f")
    flags+=("--force")
    local_nonpersistent_flags+=("--force")
    flags+=("--kustomize=")
    two_word_flags+=("--kustomize")
    two_word_flags+=("-k")
    local_nonpersistent_flags+=("--kustomize")
    local_nonpersistent_flags+=("--kustomize=")
    local_nonpersistent_flags+=("-k")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    local_nonpersistent_flags+=("-R")
    flags+=("--show-managed-fields")
    local_nonpersistent_flags+=("--show-managed-fields")
    flags+=("--template=")
    two_word_flags+=("--template")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template")
    local_nonpersistent_flags+=("--template=")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_certificate_deny()
{
    last_command="kubectl_certificate_deny"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--filename=")
    two_word_flags+=("--filename")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename")
    local_nonpersistent_flags+=("--filename=")
    local_nonpersistent_flags+=("-f")
    flags+=("--force")
    local_nonpersistent_flags+=("--force")
    flags+=("--kustomize=")
    two_word_flags+=("--kustomize")
    two_word_flags+=("-k")
    local_nonpersistent_flags+=("--kustomize")
    local_nonpersistent_flags+=("--kustomize=")
    local_nonpersistent_flags+=("-k")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    local_nonpersistent_flags+=("-R")
    flags+=("--show-managed-fields")
    local_nonpersistent_flags+=("--show-managed-fields")
    flags+=("--template=")
    two_word_flags+=("--template")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template")
    local_nonpersistent_flags+=("--template=")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_certificate()
{
    last_command="kubectl_certificate"

    command_aliases=()

    commands=()
    commands+=("approve")
    commands+=("deny")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_cluster-info_dump()
{
    last_command="kubectl_cluster-info_dump"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--all-namespaces")
    flags+=("-A")
    local_nonpersistent_flags+=("--all-namespaces")
    local_nonpersistent_flags+=("-A")
    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--namespaces=")
    two_word_flags+=("--namespaces")
    local_nonpersistent_flags+=("--namespaces")
    local_nonpersistent_flags+=("--namespaces=")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--output-directory=")
    two_word_flags+=("--output-directory")
    local_nonpersistent_flags+=("--output-directory")
    local_nonpersistent_flags+=("--output-directory=")
    flags+=("--pod-running-timeout=")
    two_word_flags+=("--pod-running-timeout")
    local_nonpersistent_flags+=("--pod-running-timeout")
    local_nonpersistent_flags+=("--pod-running-timeout=")
    flags+=("--show-managed-fields")
    local_nonpersistent_flags+=("--show-managed-fields")
    flags+=("--template=")
    two_word_flags+=("--template")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template")
    local_nonpersistent_flags+=("--template=")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_cluster-info()
{
    last_command="kubectl_cluster-info"

    command_aliases=()

    commands=()
    commands+=("dump")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_completion()
{
    last_command="kubectl_completion"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--help")
    flags+=("-h")
    local_nonpersistent_flags+=("--help")
    local_nonpersistent_flags+=("-h")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    must_have_one_noun+=("bash")
    must_have_one_noun+=("fish")
    must_have_one_noun+=("powershell")
    must_have_one_noun+=("zsh")
    noun_aliases=()
}

_kubectl_config_current-context()
{
    last_command="kubectl_config_current-context"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_config_delete-cluster()
{
    last_command="kubectl_config_delete-cluster"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    has_completion_function=1
    noun_aliases=()
}

_kubectl_config_delete-context()
{
    last_command="kubectl_config_delete-context"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    has_completion_function=1
    noun_aliases=()
}

_kubectl_config_delete-user()
{
    last_command="kubectl_config_delete-user"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_config_get-clusters()
{
    last_command="kubectl_config_get-clusters"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_config_get-contexts()
{
    last_command="kubectl_config_get-contexts"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_config_get-users()
{
    last_command="kubectl_config_get-users"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_config_rename-context()
{
    last_command="kubectl_config_rename-context"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    has_completion_function=1
    noun_aliases=()
}

_kubectl_config_set()
{
    last_command="kubectl_config_set"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--set-raw-bytes")
    local_nonpersistent_flags+=("--set-raw-bytes")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_config_set-cluster()
{
    last_command="kubectl_config_set-cluster"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--embed-certs")
    local_nonpersistent_flags+=("--embed-certs")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_config_set-context()
{
    last_command="kubectl_config_set-context"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--current")
    local_nonpersistent_flags+=("--current")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_config_set-credentials()
{
    last_command="kubectl_config_set-credentials"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--auth-provider=")
    two_word_flags+=("--auth-provider")
    local_nonpersistent_flags+=("--auth-provider")
    local_nonpersistent_flags+=("--auth-provider=")
    flags+=("--auth-provider-arg=")
    two_word_flags+=("--auth-provider-arg")
    local_nonpersistent_flags+=("--auth-provider-arg")
    local_nonpersistent_flags+=("--auth-provider-arg=")
    flags+=("--embed-certs")
    local_nonpersistent_flags+=("--embed-certs")
    flags+=("--exec-api-version=")
    two_word_flags+=("--exec-api-version")
    local_nonpersistent_flags+=("--exec-api-version")
    local_nonpersistent_flags+=("--exec-api-version=")
    flags+=("--exec-arg=")
    two_word_flags+=("--exec-arg")
    local_nonpersistent_flags+=("--exec-arg")
    local_nonpersistent_flags+=("--exec-arg=")
    flags+=("--exec-command=")
    two_word_flags+=("--exec-command")
    local_nonpersistent_flags+=("--exec-command")
    local_nonpersistent_flags+=("--exec-command=")
    flags+=("--exec-env=")
    two_word_flags+=("--exec-env")
    local_nonpersistent_flags+=("--exec-env")
    local_nonpersistent_flags+=("--exec-env=")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_config_unset()
{
    last_command="kubectl_config_unset"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_config_use-context()
{
    last_command="kubectl_config_use-context"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    has_completion_function=1
    noun_aliases=()
}

_kubectl_config_view()
{
    last_command="kubectl_config_view"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--flatten")
    local_nonpersistent_flags+=("--flatten")
    flags+=("--merge")
    local_nonpersistent_flags+=("--merge")
    flags+=("--minify")
    local_nonpersistent_flags+=("--minify")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--raw")
    local_nonpersistent_flags+=("--raw")
    flags+=("--show-managed-fields")
    local_nonpersistent_flags+=("--show-managed-fields")
    flags+=("--template=")
    two_word_flags+=("--template")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template")
    local_nonpersistent_flags+=("--template=")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_config()
{
    last_command="kubectl_config"

    command_aliases=()

    commands=()
    commands+=("current-context")
    commands+=("delete-cluster")
    commands+=("delete-context")
    commands+=("delete-user")
    commands+=("get-clusters")
    commands+=("get-contexts")
    commands+=("get-users")
    commands+=("rename-context")
    commands+=("set")
    commands+=("set-cluster")
    commands+=("set-context")
    commands+=("set-credentials")
    commands+=("unset")
    commands+=("use-context")
    if [[ -z "${BASH_VERSION}" || "${BASH_VERSINFO[0]}" -gt 3 ]]; then
        command_aliases+=("use")
        aliashash["use"]="use-context"
    fi
    commands+=("view")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_cordon()
{
    last_command="kubectl_cordon"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--selector=")
    two_word_flags+=("--selector")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--selector")
    local_nonpersistent_flags+=("--selector=")
    local_nonpersistent_flags+=("-l")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    has_completion_function=1
    noun_aliases=()
}

_kubectl_cp()
{
    last_command="kubectl_cp"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--container=")
    two_word_flags+=("--container")
    two_word_flags+=("-c")
    local_nonpersistent_flags+=("--container")
    local_nonpersistent_flags+=("--container=")
    local_nonpersistent_flags+=("-c")
    flags+=("--no-preserve")
    local_nonpersistent_flags+=("--no-preserve")
    flags+=("--retries=")
    two_word_flags+=("--retries")
    local_nonpersistent_flags+=("--retries")
    local_nonpersistent_flags+=("--retries=")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    has_completion_function=1
    noun_aliases=()
}

_kubectl_create_clusterrole()
{
    last_command="kubectl_create_clusterrole"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--aggregation-rule=")
    two_word_flags+=("--aggregation-rule")
    local_nonpersistent_flags+=("--aggregation-rule")
    local_nonpersistent_flags+=("--aggregation-rule=")
    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--field-manager=")
    two_word_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager=")
    flags+=("--non-resource-url=")
    two_word_flags+=("--non-resource-url")
    local_nonpersistent_flags+=("--non-resource-url")
    local_nonpersistent_flags+=("--non-resource-url=")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--resource=")
    two_word_flags+=("--resource")
    local_nonpersistent_flags+=("--resource")
    local_nonpersistent_flags+=("--resource=")
    flags+=("--resource-name=")
    two_word_flags+=("--resource-name")
    local_nonpersistent_flags+=("--resource-name")
    local_nonpersistent_flags+=("--resource-name=")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--show-managed-fields")
    local_nonpersistent_flags+=("--show-managed-fields")
    flags+=("--template=")
    two_word_flags+=("--template")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--verb=")
    two_word_flags+=("--verb")
    local_nonpersistent_flags+=("--verb")
    local_nonpersistent_flags+=("--verb=")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_clusterrolebinding()
{
    last_command="kubectl_create_clusterrolebinding"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--clusterrole=")
    two_word_flags+=("--clusterrole")
    flags_with_completion+=("--clusterrole")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    local_nonpersistent_flags+=("--clusterrole")
    local_nonpersistent_flags+=("--clusterrole=")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--field-manager=")
    two_word_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager=")
    flags+=("--group=")
    two_word_flags+=("--group")
    local_nonpersistent_flags+=("--group")
    local_nonpersistent_flags+=("--group=")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--serviceaccount=")
    two_word_flags+=("--serviceaccount")
    local_nonpersistent_flags+=("--serviceaccount")
    local_nonpersistent_flags+=("--serviceaccount=")
    flags+=("--show-managed-fields")
    local_nonpersistent_flags+=("--show-managed-fields")
    flags+=("--template=")
    two_word_flags+=("--template")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_flag+=("--clusterrole=")
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_configmap()
{
    last_command="kubectl_create_configmap"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--append-hash")
    local_nonpersistent_flags+=("--append-hash")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--field-manager=")
    two_word_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager=")
    flags+=("--from-env-file=")
    two_word_flags+=("--from-env-file")
    local_nonpersistent_flags+=("--from-env-file")
    local_nonpersistent_flags+=("--from-env-file=")
    flags+=("--from-file=")
    two_word_flags+=("--from-file")
    local_nonpersistent_flags+=("--from-file")
    local_nonpersistent_flags+=("--from-file=")
    flags+=("--from-literal=")
    two_word_flags+=("--from-literal")
    local_nonpersistent_flags+=("--from-literal")
    local_nonpersistent_flags+=("--from-literal=")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--show-managed-fields")
    local_nonpersistent_flags+=("--show-managed-fields")
    flags+=("--template=")
    two_word_flags+=("--template")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_cronjob()
{
    last_command="kubectl_create_cronjob"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--field-manager=")
    two_word_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager=")
    flags+=("--image=")
    two_word_flags+=("--image")
    local_nonpersistent_flags+=("--image")
    local_nonpersistent_flags+=("--image=")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--restart=")
    two_word_flags+=("--restart")
    local_nonpersistent_flags+=("--restart")
    local_nonpersistent_flags+=("--restart=")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--schedule=")
    two_word_flags+=("--schedule")
    local_nonpersistent_flags+=("--schedule")
    local_nonpersistent_flags+=("--schedule=")
    flags+=("--show-managed-fields")
    local_nonpersistent_flags+=("--show-managed-fields")
    flags+=("--template=")
    two_word_flags+=("--template")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_flag+=("--image=")
    must_have_one_flag+=("--schedule=")
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_deployment()
{
    last_command="kubectl_create_deployment"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--field-manager=")
    two_word_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager=")
    flags+=("--image=")
    two_word_flags+=("--image")
    local_nonpersistent_flags+=("--image")
    local_nonpersistent_flags+=("--image=")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--port=")
    two_word_flags+=("--port")
    local_nonpersistent_flags+=("--port")
    local_nonpersistent_flags+=("--port=")
    flags+=("--replicas=")
    two_word_flags+=("--replicas")
    two_word_flags+=("-r")
    local_nonpersistent_flags+=("--replicas")
    local_nonpersistent_flags+=("--replicas=")
    local_nonpersistent_flags+=("-r")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--show-managed-fields")
    local_nonpersistent_flags+=("--show-managed-fields")
    flags+=("--template=")
    two_word_flags+=("--template")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_flag+=("--image=")
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_ingress()
{
    last_command="kubectl_create_ingress"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--annotation=")
    two_word_flags+=("--annotation")
    local_nonpersistent_flags+=("--annotation")
    local_nonpersistent_flags+=("--annotation=")
    flags+=("--class=")
    two_word_flags+=("--class")
    local_nonpersistent_flags+=("--class")
    local_nonpersistent_flags+=("--class=")
    flags+=("--default-backend=")
    two_word_flags+=("--default-backend")
    local_nonpersistent_flags+=("--default-backend")
    local_nonpersistent_flags+=("--default-backend=")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--field-manager=")
    two_word_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager=")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--rule=")
    two_word_flags+=("--rule")
    local_nonpersistent_flags+=("--rule")
    local_nonpersistent_flags+=("--rule=")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--show-managed-fields")
    local_nonpersistent_flags+=("--show-managed-fields")
    flags+=("--template=")
    two_word_flags+=("--template")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_job()
{
    last_command="kubectl_create_job"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--field-manager=")
    two_word_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager=")
    flags+=("--from=")
    two_word_flags+=("--from")
    local_nonpersistent_flags+=("--from")
    local_nonpersistent_flags+=("--from=")
    flags+=("--image=")
    two_word_flags+=("--image")
    local_nonpersistent_flags+=("--image")
    local_nonpersistent_flags+=("--image=")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--show-managed-fields")
    local_nonpersistent_flags+=("--show-managed-fields")
    flags+=("--template=")
    two_word_flags+=("--template")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_namespace()
{
    last_command="kubectl_create_namespace"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--field-manager=")
    two_word_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager=")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--show-managed-fields")
    local_nonpersistent_flags+=("--show-managed-fields")
    flags+=("--template=")
    two_word_flags+=("--template")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_poddisruptionbudget()
{
    last_command="kubectl_create_poddisruptionbudget"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--field-manager=")
    two_word_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager=")
    flags+=("--max-unavailable=")
    two_word_flags+=("--max-unavailable")
    local_nonpersistent_flags+=("--max-unavailable")
    local_nonpersistent_flags+=("--max-unavailable=")
    flags+=("--min-available=")
    two_word_flags+=("--min-available")
    local_nonpersistent_flags+=("--min-available")
    local_nonpersistent_flags+=("--min-available=")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--selector=")
    two_word_flags+=("--selector")
    local_nonpersistent_flags+=("--selector")
    local_nonpersistent_flags+=("--selector=")
    flags+=("--show-managed-fields")
    local_nonpersistent_flags+=("--show-managed-fields")
    flags+=("--template=")
    two_word_flags+=("--template")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_priorityclass()
{
    last_command="kubectl_create_priorityclass"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--description=")
    two_word_flags+=("--description")
    local_nonpersistent_flags+=("--description")
    local_nonpersistent_flags+=("--description=")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--field-manager=")
    two_word_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager=")
    flags+=("--global-default")
    local_nonpersistent_flags+=("--global-default")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--preemption-policy=")
    two_word_flags+=("--preemption-policy")
    local_nonpersistent_flags+=("--preemption-policy")
    local_nonpersistent_flags+=("--preemption-policy=")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--show-managed-fields")
    local_nonpersistent_flags+=("--show-managed-fields")
    flags+=("--template=")
    two_word_flags+=("--template")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--value=")
    two_word_flags+=("--value")
    local_nonpersistent_flags+=("--value")
    local_nonpersistent_flags+=("--value=")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_quota()
{
    last_command="kubectl_create_quota"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--field-manager=")
    two_word_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager=")
    flags+=("--hard=")
    two_word_flags+=("--hard")
    local_nonpersistent_flags+=("--hard")
    local_nonpersistent_flags+=("--hard=")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--scopes=")
    two_word_flags+=("--scopes")
    local_nonpersistent_flags+=("--scopes")
    local_nonpersistent_flags+=("--scopes=")
    flags+=("--show-managed-fields")
    local_nonpersistent_flags+=("--show-managed-fields")
    flags+=("--template=")
    two_word_flags+=("--template")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_role()
{
    last_command="kubectl_create_role"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--field-manager=")
    two_word_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager=")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--resource=")
    two_word_flags+=("--resource")
    local_nonpersistent_flags+=("--resource")
    local_nonpersistent_flags+=("--resource=")
    flags+=("--resource-name=")
    two_word_flags+=("--resource-name")
    local_nonpersistent_flags+=("--resource-name")
    local_nonpersistent_flags+=("--resource-name=")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--show-managed-fields")
    local_nonpersistent_flags+=("--show-managed-fields")
    flags+=("--template=")
    two_word_flags+=("--template")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--verb=")
    two_word_flags+=("--verb")
    local_nonpersistent_flags+=("--verb")
    local_nonpersistent_flags+=("--verb=")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_rolebinding()
{
    last_command="kubectl_create_rolebinding"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--clusterrole=")
    two_word_flags+=("--clusterrole")
    local_nonpersistent_flags+=("--clusterrole")
    local_nonpersistent_flags+=("--clusterrole=")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--field-manager=")
    two_word_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager=")
    flags+=("--group=")
    two_word_flags+=("--group")
    local_nonpersistent_flags+=("--group")
    local_nonpersistent_flags+=("--group=")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--role=")
    two_word_flags+=("--role")
    local_nonpersistent_flags+=("--role")
    local_nonpersistent_flags+=("--role=")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--serviceaccount=")
    two_word_flags+=("--serviceaccount")
    local_nonpersistent_flags+=("--serviceaccount")
    local_nonpersistent_flags+=("--serviceaccount=")
    flags+=("--show-managed-fields")
    local_nonpersistent_flags+=("--show-managed-fields")
    flags+=("--template=")
    two_word_flags+=("--template")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_secret_docker-registry()
{
    last_command="kubectl_create_secret_docker-registry"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--append-hash")
    local_nonpersistent_flags+=("--append-hash")
    flags+=("--docker-email=")
    two_word_flags+=("--docker-email")
    local_nonpersistent_flags+=("--docker-email")
    local_nonpersistent_flags+=("--docker-email=")
    flags+=("--docker-password=")
    two_word_flags+=("--docker-password")
    local_nonpersistent_flags+=("--docker-password")
    local_nonpersistent_flags+=("--docker-password=")
    flags+=("--docker-server=")
    two_word_flags+=("--docker-server")
    local_nonpersistent_flags+=("--docker-server")
    local_nonpersistent_flags+=("--docker-server=")
    flags+=("--docker-username=")
    two_word_flags+=("--docker-username")
    local_nonpersistent_flags+=("--docker-username")
    local_nonpersistent_flags+=("--docker-username=")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--field-manager=")
    two_word_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager=")
    flags+=("--from-file=")
    two_word_flags+=("--from-file")
    local_nonpersistent_flags+=("--from-file")
    local_nonpersistent_flags+=("--from-file=")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--show-managed-fields")
    local_nonpersistent_flags+=("--show-managed-fields")
    flags+=("--template=")
    two_word_flags+=("--template")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_secret_generic()
{
    last_command="kubectl_create_secret_generic"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--append-hash")
    local_nonpersistent_flags+=("--append-hash")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--field-manager=")
    two_word_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager=")
    flags+=("--from-env-file=")
    two_word_flags+=("--from-env-file")
    local_nonpersistent_flags+=("--from-env-file")
    local_nonpersistent_flags+=("--from-env-file=")
    flags+=("--from-file=")
    two_word_flags+=("--from-file")
    local_nonpersistent_flags+=("--from-file")
    local_nonpersistent_flags+=("--from-file=")
    flags+=("--from-literal=")
    two_word_flags+=("--from-literal")
    local_nonpersistent_flags+=("--from-literal")
    local_nonpersistent_flags+=("--from-literal=")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--show-managed-fields")
    local_nonpersistent_flags+=("--show-managed-fields")
    flags+=("--template=")
    two_word_flags+=("--template")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template")
    local_nonpersistent_flags+=("--template=")
    flags+=("--type=")
    two_word_flags+=("--type")
    local_nonpersistent_flags+=("--type")
    local_nonpersistent_flags+=("--type=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_secret_tls()
{
    last_command="kubectl_create_secret_tls"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--append-hash")
    local_nonpersistent_flags+=("--append-hash")
    flags+=("--cert=")
    two_word_flags+=("--cert")
    local_nonpersistent_flags+=("--cert")
    local_nonpersistent_flags+=("--cert=")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--field-manager=")
    two_word_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager=")
    flags+=("--key=")
    two_word_flags+=("--key")
    local_nonpersistent_flags+=("--key")
    local_nonpersistent_flags+=("--key=")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--show-managed-fields")
    local_nonpersistent_flags+=("--show-managed-fields")
    flags+=("--template=")
    two_word_flags+=("--template")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_secret()
{
    last_command="kubectl_create_secret"

    command_aliases=()

    commands=()
    commands+=("docker-registry")
    commands+=("generic")
    commands+=("tls")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_service_clusterip()
{
    last_command="kubectl_create_service_clusterip"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--clusterip=")
    two_word_flags+=("--clusterip")
    local_nonpersistent_flags+=("--clusterip")
    local_nonpersistent_flags+=("--clusterip=")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--field-manager=")
    two_word_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager=")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--show-managed-fields")
    local_nonpersistent_flags+=("--show-managed-fields")
    flags+=("--tcp=")
    two_word_flags+=("--tcp")
    local_nonpersistent_flags+=("--tcp")
    local_nonpersistent_flags+=("--tcp=")
    flags+=("--template=")
    two_word_flags+=("--template")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_service_externalname()
{
    last_command="kubectl_create_service_externalname"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--external-name=")
    two_word_flags+=("--external-name")
    local_nonpersistent_flags+=("--external-name")
    local_nonpersistent_flags+=("--external-name=")
    flags+=("--field-manager=")
    two_word_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager=")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--show-managed-fields")
    local_nonpersistent_flags+=("--show-managed-fields")
    flags+=("--tcp=")
    two_word_flags+=("--tcp")
    local_nonpersistent_flags+=("--tcp")
    local_nonpersistent_flags+=("--tcp=")
    flags+=("--template=")
    two_word_flags+=("--template")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_flag+=("--external-name=")
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_service_loadbalancer()
{
    last_command="kubectl_create_service_loadbalancer"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--field-manager=")
    two_word_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager=")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--show-managed-fields")
    local_nonpersistent_flags+=("--show-managed-fields")
    flags+=("--tcp=")
    two_word_flags+=("--tcp")
    local_nonpersistent_flags+=("--tcp")
    local_nonpersistent_flags+=("--tcp=")
    flags+=("--template=")
    two_word_flags+=("--template")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_service_nodeport()
{
    last_command="kubectl_create_service_nodeport"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--field-manager=")
    two_word_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager=")
    flags+=("--node-port=")
    two_word_flags+=("--node-port")
    local_nonpersistent_flags+=("--node-port")
    local_nonpersistent_flags+=("--node-port=")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--show-managed-fields")
    local_nonpersistent_flags+=("--show-managed-fields")
    flags+=("--tcp=")
    two_word_flags+=("--tcp")
    local_nonpersistent_flags+=("--tcp")
    local_nonpersistent_flags+=("--tcp=")
    flags+=("--template=")
    two_word_flags+=("--template")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_service()
{
    last_command="kubectl_create_service"

    command_aliases=()

    commands=()
    commands+=("clusterip")
    commands+=("externalname")
    commands+=("loadbalancer")
    commands+=("nodeport")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_serviceaccount()
{
    last_command="kubectl_create_serviceaccount"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--field-manager=")
    two_word_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager=")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--show-managed-fields")
    local_nonpersistent_flags+=("--show-managed-fields")
    flags+=("--template=")
    two_word_flags+=("--template")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create()
{
    last_command="kubectl_create"

    command_aliases=()

    commands=()
    commands+=("clusterrole")
    commands+=("clusterrolebinding")
    commands+=("configmap")
    if [[ -z "${BASH_VERSION}" || "${BASH_VERSINFO[0]}" -gt 3 ]]; then
        command_aliases+=("cm")
        aliashash["cm"]="configmap"
    fi
    commands+=("cronjob")
    if [[ -z "${BASH_VERSION}" || "${BASH_VERSINFO[0]}" -gt 3 ]]; then
        command_aliases+=("cj")
        aliashash["cj"]="cronjob"
    fi
    commands+=("deployment")
    if [[ -z "${BASH_VERSION}" || "${BASH_VERSINFO[0]}" -gt 3 ]]; then
        command_aliases+=("deploy")
        aliashash["deploy"]="deployment"
    fi
    commands+=("ingress")
    if [[ -z "${BASH_VERSION}" || "${BASH_VERSINFO[0]}" -gt 3 ]]; then
        command_aliases+=("ing")
        aliashash["ing"]="ingress"
    fi
    commands+=("job")
    commands+=("namespace")
    if [[ -z "${BASH_VERSION}" || "${BASH_VERSINFO[0]}" -gt 3 ]]; then
        command_aliases+=("ns")
        aliashash["ns"]="namespace"
    fi
    commands+=("poddisruptionbudget")
    if [[ -z "${BASH_VERSION}" || "${BASH_VERSINFO[0]}" -gt 3 ]]; then
        command_aliases+=("pdb")
        aliashash["pdb"]="poddisruptionbudget"
    fi
    commands+=("priorityclass")
    if [[ -z "${BASH_VERSION}" || "${BASH_VERSINFO[0]}" -gt 3 ]]; then
        command_aliases+=("pc")
        aliashash["pc"]="priorityclass"
    fi
    commands+=("quota")
    if [[ -z "${BASH_VERSION}" || "${BASH_VERSINFO[0]}" -gt 3 ]]; then
        command_aliases+=("resourcequota")
        aliashash["resourcequota"]="quota"
    fi
    commands+=("role")
    commands+=("rolebinding")
    commands+=("secret")
    commands+=("service")
    if [[ -z "${BASH_VERSION}" || "${BASH_VERSINFO[0]}" -gt 3 ]]; then
        command_aliases+=("svc")
        aliashash["svc"]="service"
    fi
    commands+=("serviceaccount")
    if [[ -z "${BASH_VERSION}" || "${BASH_VERSINFO[0]}" -gt 3 ]]; then
        command_aliases+=("sa")
        aliashash["sa"]="serviceaccount"
    fi

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--edit")
    local_nonpersistent_flags+=("--edit")
    flags+=("--field-manager=")
    two_word_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager=")
    flags+=("--filename=")
    two_word_flags+=("--filename")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename")
    local_nonpersistent_flags+=("--filename=")
    local_nonpersistent_flags+=("-f")
    flags+=("--kustomize=")
    two_word_flags+=("--kustomize")
    two_word_flags+=("-k")
    local_nonpersistent_flags+=("--kustomize")
    local_nonpersistent_flags+=("--kustomize=")
    local_nonpersistent_flags+=("-k")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--raw=")
    two_word_flags+=("--raw")
    local_nonpersistent_flags+=("--raw")
    local_nonpersistent_flags+=("--raw=")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    local_nonpersistent_flags+=("-R")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--selector=")
    two_word_flags+=("--selector")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--selector")
    local_nonpersistent_flags+=("--selector=")
    local_nonpersistent_flags+=("-l")
    flags+=("--show-managed-fields")
    local_nonpersistent_flags+=("--show-managed-fields")
    flags+=("--template=")
    two_word_flags+=("--template")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--windows-line-endings")
    local_nonpersistent_flags+=("--windows-line-endings")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_debug()
{
    last_command="kubectl_debug"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--arguments-only")
    local_nonpersistent_flags+=("--arguments-only")
    flags+=("--attach")
    local_nonpersistent_flags+=("--attach")
    flags+=("--container=")
    two_word_flags+=("--container")
    two_word_flags+=("-c")
    local_nonpersistent_flags+=("--container")
    local_nonpersistent_flags+=("--container=")
    local_nonpersistent_flags+=("-c")
    flags+=("--copy-to=")
    two_word_flags+=("--copy-to")
    local_nonpersistent_flags+=("--copy-to")
    local_nonpersistent_flags+=("--copy-to=")
    flags+=("--env=")
    two_word_flags+=("--env")
    local_nonpersistent_flags+=("--env")
    local_nonpersistent_flags+=("--env=")
    flags+=("--image=")
    two_word_flags+=("--image")
    local_nonpersistent_flags+=("--image")
    local_nonpersistent_flags+=("--image=")
    flags+=("--image-pull-policy=")
    two_word_flags+=("--image-pull-policy")
    local_nonpersistent_flags+=("--image-pull-policy")
    local_nonpersistent_flags+=("--image-pull-policy=")
    flags+=("--quiet")
    flags+=("-q")
    local_nonpersistent_flags+=("--quiet")
    local_nonpersistent_flags+=("-q")
    flags+=("--replace")
    local_nonpersistent_flags+=("--replace")
    flags+=("--same-node")
    local_nonpersistent_flags+=("--same-node")
    flags+=("--set-image=")
    two_word_flags+=("--set-image")
    local_nonpersistent_flags+=("--set-image")
    local_nonpersistent_flags+=("--set-image=")
    flags+=("--share-processes")
    local_nonpersistent_flags+=("--share-processes")
    flags+=("--stdin")
    flags+=("-i")
    local_nonpersistent_flags+=("--stdin")
    local_nonpersistent_flags+=("-i")
    flags+=("--target=")
    two_word_flags+=("--target")
    local_nonpersistent_flags+=("--target")
    local_nonpersistent_flags+=("--target=")
    flags+=("--tty")
    flags+=("-t")
    local_nonpersistent_flags+=("--tty")
    local_nonpersistent_flags+=("-t")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_delete()
{
    last_command="kubectl_delete"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--all")
    local_nonpersistent_flags+=("--all")
    flags+=("--all-namespaces")
    flags+=("-A")
    local_nonpersistent_flags+=("--all-namespaces")
    local_nonpersistent_flags+=("-A")
    flags+=("--cascade")
    local_nonpersistent_flags+=("--cascade")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--field-selector=")
    two_word_flags+=("--field-selector")
    local_nonpersistent_flags+=("--field-selector")
    local_nonpersistent_flags+=("--field-selector=")
    flags+=("--filename=")
    two_word_flags+=("--filename")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename")
    local_nonpersistent_flags+=("--filename=")
    local_nonpersistent_flags+=("-f")
    flags+=("--force")
    local_nonpersistent_flags+=("--force")
    flags+=("--grace-period=")
    two_word_flags+=("--grace-period")
    local_nonpersistent_flags+=("--grace-period")
    local_nonpersistent_flags+=("--grace-period=")
    flags+=("--ignore-not-found")
    local_nonpersistent_flags+=("--ignore-not-found")
    flags+=("--kustomize=")
    two_word_flags+=("--kustomize")
    two_word_flags+=("-k")
    local_nonpersistent_flags+=("--kustomize")
    local_nonpersistent_flags+=("--kustomize=")
    local_nonpersistent_flags+=("-k")
    flags+=("--now")
    local_nonpersistent_flags+=("--now")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--raw=")
    two_word_flags+=("--raw")
    local_nonpersistent_flags+=("--raw")
    local_nonpersistent_flags+=("--raw=")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    local_nonpersistent_flags+=("-R")
    flags+=("--selector=")
    two_word_flags+=("--selector")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--selector")
    local_nonpersistent_flags+=("--selector=")
    local_nonpersistent_flags+=("-l")
    flags+=("--timeout=")
    two_word_flags+=("--timeout")
    local_nonpersistent_flags+=("--timeout")
    local_nonpersistent_flags+=("--timeout=")
    flags+=("--wait")
    local_nonpersistent_flags+=("--wait")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    has_completion_function=1
    noun_aliases=()
}

_kubectl_describe()
{
    last_command="kubectl_describe"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--all-namespaces")
    flags+=("-A")
    local_nonpersistent_flags+=("--all-namespaces")
    local_nonpersistent_flags+=("-A")
    flags+=("--chunk-size=")
    two_word_flags+=("--chunk-size")
    local_nonpersistent_flags+=("--chunk-size")
    local_nonpersistent_flags+=("--chunk-size=")
    flags+=("--filename=")
    two_word_flags+=("--filename")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename")
    local_nonpersistent_flags+=("--filename=")
    local_nonpersistent_flags+=("-f")
    flags+=("--kustomize=")
    two_word_flags+=("--kustomize")
    two_word_flags+=("-k")
    local_nonpersistent_flags+=("--kustomize")
    local_nonpersistent_flags+=("--kustomize=")
    local_nonpersistent_flags+=("-k")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    local_nonpersistent_flags+=("-R")
    flags+=("--selector=")
    two_word_flags+=("--selector")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--selector")
    local_nonpersistent_flags+=("--selector=")
    local_nonpersistent_flags+=("-l")
    flags+=("--show-events")
    local_nonpersistent_flags+=("--show-events")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    has_completion_function=1
    noun_aliases=()
}

_kubectl_diff()
{
    last_command="kubectl_diff"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--field-manager=")
    two_word_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager=")
    flags+=("--filename=")
    two_word_flags+=("--filename")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename")
    local_nonpersistent_flags+=("--filename=")
    local_nonpersistent_flags+=("-f")
    flags+=("--force-conflicts")
    local_nonpersistent_flags+=("--force-conflicts")
    flags+=("--kustomize=")
    two_word_flags+=("--kustomize")
    two_word_flags+=("-k")
    local_nonpersistent_flags+=("--kustomize")
    local_nonpersistent_flags+=("--kustomize=")
    local_nonpersistent_flags+=("-k")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    local_nonpersistent_flags+=("-R")
    flags+=("--selector=")
    two_word_flags+=("--selector")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--selector")
    local_nonpersistent_flags+=("--selector=")
    local_nonpersistent_flags+=("-l")
    flags+=("--server-side")
    local_nonpersistent_flags+=("--server-side")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_drain()
{
    last_command="kubectl_drain"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--chunk-size=")
    two_word_flags+=("--chunk-size")
    local_nonpersistent_flags+=("--chunk-size")
    local_nonpersistent_flags+=("--chunk-size=")
    flags+=("--delete-emptydir-data")
    local_nonpersistent_flags+=("--delete-emptydir-data")
    flags+=("--disable-eviction")
    local_nonpersistent_flags+=("--disable-eviction")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--force")
    local_nonpersistent_flags+=("--force")
    flags+=("--grace-period=")
    two_word_flags+=("--grace-period")
    local_nonpersistent_flags+=("--grace-period")
    local_nonpersistent_flags+=("--grace-period=")
    flags+=("--ignore-daemonsets")
    local_nonpersistent_flags+=("--ignore-daemonsets")
    flags+=("--pod-selector=")
    two_word_flags+=("--pod-selector")
    local_nonpersistent_flags+=("--pod-selector")
    local_nonpersistent_flags+=("--pod-selector=")
    flags+=("--selector=")
    two_word_flags+=("--selector")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--selector")
    local_nonpersistent_flags+=("--selector=")
    local_nonpersistent_flags+=("-l")
    flags+=("--skip-wait-for-delete-timeout=")
    two_word_flags+=("--skip-wait-for-delete-timeout")
    local_nonpersistent_flags+=("--skip-wait-for-delete-timeout")
    local_nonpersistent_flags+=("--skip-wait-for-delete-timeout=")
    flags+=("--timeout=")
    two_word_flags+=("--timeout")
    local_nonpersistent_flags+=("--timeout")
    local_nonpersistent_flags+=("--timeout=")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    has_completion_function=1
    noun_aliases=()
}

_kubectl_edit()
{
    last_command="kubectl_edit"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--field-manager=")
    two_word_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager=")
    flags+=("--filename=")
    two_word_flags+=("--filename")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename")
    local_nonpersistent_flags+=("--filename=")
    local_nonpersistent_flags+=("-f")
    flags+=("--kustomize=")
    two_word_flags+=("--kustomize")
    two_word_flags+=("-k")
    local_nonpersistent_flags+=("--kustomize")
    local_nonpersistent_flags+=("--kustomize=")
    local_nonpersistent_flags+=("-k")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--output-patch")
    local_nonpersistent_flags+=("--output-patch")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    local_nonpersistent_flags+=("-R")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--show-managed-fields")
    local_nonpersistent_flags+=("--show-managed-fields")
    flags+=("--template=")
    two_word_flags+=("--template")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--windows-line-endings")
    local_nonpersistent_flags+=("--windows-line-endings")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    has_completion_function=1
    noun_aliases=()
}

_kubectl_exec()
{
    last_command="kubectl_exec"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--container=")
    two_word_flags+=("--container")
    two_word_flags+=("-c")
    local_nonpersistent_flags+=("--container")
    local_nonpersistent_flags+=("--container=")
    local_nonpersistent_flags+=("-c")
    flags+=("--filename=")
    two_word_flags+=("--filename")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename")
    local_nonpersistent_flags+=("--filename=")
    local_nonpersistent_flags+=("-f")
    flags+=("--pod-running-timeout=")
    two_word_flags+=("--pod-running-timeout")
    local_nonpersistent_flags+=("--pod-running-timeout")
    local_nonpersistent_flags+=("--pod-running-timeout=")
    flags+=("--quiet")
    flags+=("-q")
    local_nonpersistent_flags+=("--quiet")
    local_nonpersistent_flags+=("-q")
    flags+=("--stdin")
    flags+=("-i")
    local_nonpersistent_flags+=("--stdin")
    local_nonpersistent_flags+=("-i")
    flags+=("--tty")
    flags+=("-t")
    local_nonpersistent_flags+=("--tty")
    local_nonpersistent_flags+=("-t")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    has_completion_function=1
    noun_aliases=()
}

_kubectl_explain()
{
    last_command="kubectl_explain"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-version=")
    two_word_flags+=("--api-version")
    local_nonpersistent_flags+=("--api-version")
    local_nonpersistent_flags+=("--api-version=")
    flags+=("--recursive")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_expose()
{
    last_command="kubectl_expose"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--cluster-ip=")
    two_word_flags+=("--cluster-ip")
    local_nonpersistent_flags+=("--cluster-ip")
    local_nonpersistent_flags+=("--cluster-ip=")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--external-ip=")
    two_word_flags+=("--external-ip")
    local_nonpersistent_flags+=("--external-ip")
    local_nonpersistent_flags+=("--external-ip=")
    flags+=("--field-manager=")
    two_word_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager=")
    flags+=("--filename=")
    two_word_flags+=("--filename")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename")
    local_nonpersistent_flags+=("--filename=")
    local_nonpersistent_flags+=("-f")
    flags+=("--generator=")
    two_word_flags+=("--generator")
    local_nonpersistent_flags+=("--generator")
    local_nonpersistent_flags+=("--generator=")
    flags+=("--kustomize=")
    two_word_flags+=("--kustomize")
    two_word_flags+=("-k")
    local_nonpersistent_flags+=("--kustomize")
    local_nonpersistent_flags+=("--kustomize=")
    local_nonpersistent_flags+=("-k")
    flags+=("--labels=")
    two_word_flags+=("--labels")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--labels")
    local_nonpersistent_flags+=("--labels=")
    local_nonpersistent_flags+=("-l")
    flags+=("--load-balancer-ip=")
    two_word_flags+=("--load-balancer-ip")
    local_nonpersistent_flags+=("--load-balancer-ip")
    local_nonpersistent_flags+=("--load-balancer-ip=")
    flags+=("--name=")
    two_word_flags+=("--name")
    local_nonpersistent_flags+=("--name")
    local_nonpersistent_flags+=("--name=")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--override-type=")
    two_word_flags+=("--override-type")
    local_nonpersistent_flags+=("--override-type")
    local_nonpersistent_flags+=("--override-type=")
    flags+=("--overrides=")
    two_word_flags+=("--overrides")
    local_nonpersistent_flags+=("--overrides")
    local_nonpersistent_flags+=("--overrides=")
    flags+=("--port=")
    two_word_flags+=("--port")
    local_nonpersistent_flags+=("--port")
    local_nonpersistent_flags+=("--port=")
    flags+=("--protocol=")
    two_word_flags+=("--protocol")
    local_nonpersistent_flags+=("--protocol")
    local_nonpersistent_flags+=("--protocol=")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    local_nonpersistent_flags+=("-R")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--selector=")
    two_word_flags+=("--selector")
    local_nonpersistent_flags+=("--selector")
    local_nonpersistent_flags+=("--selector=")
    flags+=("--session-affinity=")
    two_word_flags+=("--session-affinity")
    local_nonpersistent_flags+=("--session-affinity")
    local_nonpersistent_flags+=("--session-affinity=")
    flags+=("--show-managed-fields")
    local_nonpersistent_flags+=("--show-managed-fields")
    flags+=("--target-port=")
    two_word_flags+=("--target-port")
    local_nonpersistent_flags+=("--target-port")
    local_nonpersistent_flags+=("--target-port=")
    flags+=("--template=")
    two_word_flags+=("--template")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template")
    local_nonpersistent_flags+=("--template=")
    flags+=("--type=")
    two_word_flags+=("--type")
    local_nonpersistent_flags+=("--type")
    local_nonpersistent_flags+=("--type=")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    has_completion_function=1
    noun_aliases=()
}

_kubectl_get()
{
    last_command="kubectl_get"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--all-namespaces")
    flags+=("-A")
    local_nonpersistent_flags+=("--all-namespaces")
    local_nonpersistent_flags+=("-A")
    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--chunk-size=")
    two_word_flags+=("--chunk-size")
    local_nonpersistent_flags+=("--chunk-size")
    local_nonpersistent_flags+=("--chunk-size=")
    flags+=("--field-selector=")
    two_word_flags+=("--field-selector")
    local_nonpersistent_flags+=("--field-selector")
    local_nonpersistent_flags+=("--field-selector=")
    flags+=("--filename=")
    two_word_flags+=("--filename")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename")
    local_nonpersistent_flags+=("--filename=")
    local_nonpersistent_flags+=("-f")
    flags+=("--ignore-not-found")
    local_nonpersistent_flags+=("--ignore-not-found")
    flags+=("--kustomize=")
    two_word_flags+=("--kustomize")
    two_word_flags+=("-k")
    local_nonpersistent_flags+=("--kustomize")
    local_nonpersistent_flags+=("--kustomize=")
    local_nonpersistent_flags+=("-k")
    flags+=("--label-columns=")
    two_word_flags+=("--label-columns")
    two_word_flags+=("-L")
    local_nonpersistent_flags+=("--label-columns")
    local_nonpersistent_flags+=("--label-columns=")
    local_nonpersistent_flags+=("-L")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("--output")
    flags_with_completion+=("--output")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-o")
    flags_with_completion+=("-o")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--output-watch-events")
    local_nonpersistent_flags+=("--output-watch-events")
    flags+=("--raw=")
    two_word_flags+=("--raw")
    local_nonpersistent_flags+=("--raw")
    local_nonpersistent_flags+=("--raw=")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    local_nonpersistent_flags+=("-R")
    flags+=("--selector=")
    two_word_flags+=("--selector")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--selector")
    local_nonpersistent_flags+=("--selector=")
    local_nonpersistent_flags+=("-l")
    flags+=("--server-print")
    local_nonpersistent_flags+=("--server-print")
    flags+=("--show-kind")
    local_nonpersistent_flags+=("--show-kind")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--show-managed-fields")
    local_nonpersistent_flags+=("--show-managed-fields")
    flags+=("--sort-by=")
    two_word_flags+=("--sort-by")
    local_nonpersistent_flags+=("--sort-by")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--template=")
    two_word_flags+=("--template")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template")
    local_nonpersistent_flags+=("--template=")
    flags+=("--watch")
    flags+=("-w")
    local_nonpersistent_flags+=("--watch")
    local_nonpersistent_flags+=("-w")
    flags+=("--watch-only")
    local_nonpersistent_flags+=("--watch-only")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    has_completion_function=1
    noun_aliases=()
}

_kubectl_help()
{
    last_command="kubectl_help"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    has_completion_function=1
    noun_aliases=()
}

_kubectl_kustomize()
{
    last_command="kubectl_kustomize"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--as-current-user")
    local_nonpersistent_flags+=("--as-current-user")
    flags+=("--enable-alpha-plugins")
    local_nonpersistent_flags+=("--enable-alpha-plugins")
    flags+=("--enable-helm")
    local_nonpersistent_flags+=("--enable-helm")
    flags+=("--enable-managedby-label")
    local_nonpersistent_flags+=("--enable-managedby-label")
    flags+=("--env=")
    two_word_flags+=("--env")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--env")
    local_nonpersistent_flags+=("--env=")
    local_nonpersistent_flags+=("-e")
    flags+=("--helm-command=")
    two_word_flags+=("--helm-command")
    local_nonpersistent_flags+=("--helm-command")
    local_nonpersistent_flags+=("--helm-command=")
    flags+=("--load-restrictor=")
    two_word_flags+=("--load-restrictor")
    local_nonpersistent_flags+=("--load-restrictor")
    local_nonpersistent_flags+=("--load-restrictor=")
    flags+=("--mount=")
    two_word_flags+=("--mount")
    local_nonpersistent_flags+=("--mount")
    local_nonpersistent_flags+=("--mount=")
    flags+=("--network")
    local_nonpersistent_flags+=("--network")
    flags+=("--network-name=")
    two_word_flags+=("--network-name")
    local_nonpersistent_flags+=("--network-name")
    local_nonpersistent_flags+=("--network-name=")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--reorder=")
    two_word_flags+=("--reorder")
    local_nonpersistent_flags+=("--reorder")
    local_nonpersistent_flags+=("--reorder=")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_label()
{
    last_command="kubectl_label"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--all")
    local_nonpersistent_flags+=("--all")
    flags+=("--all-namespaces")
    flags+=("-A")
    local_nonpersistent_flags+=("--all-namespaces")
    local_nonpersistent_flags+=("-A")
    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--field-manager=")
    two_word_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager=")
    flags+=("--field-selector=")
    two_word_flags+=("--field-selector")
    local_nonpersistent_flags+=("--field-selector")
    local_nonpersistent_flags+=("--field-selector=")
    flags+=("--filename=")
    two_word_flags+=("--filename")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename")
    local_nonpersistent_flags+=("--filename=")
    local_nonpersistent_flags+=("-f")
    flags+=("--kustomize=")
    two_word_flags+=("--kustomize")
    two_word_flags+=("-k")
    local_nonpersistent_flags+=("--kustomize")
    local_nonpersistent_flags+=("--kustomize=")
    local_nonpersistent_flags+=("-k")
    flags+=("--list")
    local_nonpersistent_flags+=("--list")
    flags+=("--local")
    local_nonpersistent_flags+=("--local")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--overwrite")
    local_nonpersistent_flags+=("--overwrite")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    local_nonpersistent_flags+=("-R")
    flags+=("--resource-version=")
    two_word_flags+=("--resource-version")
    local_nonpersistent_flags+=("--resource-version")
    local_nonpersistent_flags+=("--resource-version=")
    flags+=("--selector=")
    two_word_flags+=("--selector")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--selector")
    local_nonpersistent_flags+=("--selector=")
    local_nonpersistent_flags+=("-l")
    flags+=("--show-managed-fields")
    local_nonpersistent_flags+=("--show-managed-fields")
    flags+=("--template=")
    two_word_flags+=("--template")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template")
    local_nonpersistent_flags+=("--template=")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    has_completion_function=1
    noun_aliases=()
}

_kubectl_logs()
{
    last_command="kubectl_logs"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--all-containers")
    local_nonpersistent_flags+=("--all-containers")
    flags+=("--container=")
    two_word_flags+=("--container")
    two_word_flags+=("-c")
    local_nonpersistent_flags+=("--container")
    local_nonpersistent_flags+=("--container=")
    local_nonpersistent_flags+=("-c")
    flags+=("--follow")
    flags+=("-f")
    local_nonpersistent_flags+=("--follow")
    local_nonpersistent_flags+=("-f")
    flags+=("--ignore-errors")
    local_nonpersistent_flags+=("--ignore-errors")
    flags+=("--insecure-skip-tls-verify-backend")
    local_nonpersistent_flags+=("--insecure-skip-tls-verify-backend")
    flags+=("--limit-bytes=")
    two_word_flags+=("--limit-bytes")
    local_nonpersistent_flags+=("--limit-bytes")
    local_nonpersistent_flags+=("--limit-bytes=")
    flags+=("--max-log-requests=")
    two_word_flags+=("--max-log-requests")
    local_nonpersistent_flags+=("--max-log-requests")
    local_nonpersistent_flags+=("--max-log-requests=")
    flags+=("--pod-running-timeout=")
    two_word_flags+=("--pod-running-timeout")
    local_nonpersistent_flags+=("--pod-running-timeout")
    local_nonpersistent_flags+=("--pod-running-timeout=")
    flags+=("--prefix")
    local_nonpersistent_flags+=("--prefix")
    flags+=("--previous")
    flags+=("-p")
    local_nonpersistent_flags+=("--previous")
    local_nonpersistent_flags+=("-p")
    flags+=("--selector=")
    two_word_flags+=("--selector")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--selector")
    local_nonpersistent_flags+=("--selector=")
    local_nonpersistent_flags+=("-l")
    flags+=("--since=")
    two_word_flags+=("--since")
    local_nonpersistent_flags+=("--since")
    local_nonpersistent_flags+=("--since=")
    flags+=("--since-time=")
    two_word_flags+=("--since-time")
    local_nonpersistent_flags+=("--since-time")
    local_nonpersistent_flags+=("--since-time=")
    flags+=("--tail=")
    two_word_flags+=("--tail")
    local_nonpersistent_flags+=("--tail")
    local_nonpersistent_flags+=("--tail=")
    flags+=("--timestamps")
    local_nonpersistent_flags+=("--timestamps")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    has_completion_function=1
    noun_aliases=()
}

_kubectl_options()
{
    last_command="kubectl_options"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_patch()
{
    last_command="kubectl_patch"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--field-manager=")
    two_word_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager=")
    flags+=("--filename=")
    two_word_flags+=("--filename")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename")
    local_nonpersistent_flags+=("--filename=")
    local_nonpersistent_flags+=("-f")
    flags+=("--kustomize=")
    two_word_flags+=("--kustomize")
    two_word_flags+=("-k")
    local_nonpersistent_flags+=("--kustomize")
    local_nonpersistent_flags+=("--kustomize=")
    local_nonpersistent_flags+=("-k")
    flags+=("--local")
    local_nonpersistent_flags+=("--local")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--patch=")
    two_word_flags+=("--patch")
    two_word_flags+=("-p")
    local_nonpersistent_flags+=("--patch")
    local_nonpersistent_flags+=("--patch=")
    local_nonpersistent_flags+=("-p")
    flags+=("--patch-file=")
    two_word_flags+=("--patch-file")
    local_nonpersistent_flags+=("--patch-file")
    local_nonpersistent_flags+=("--patch-file=")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    local_nonpersistent_flags+=("-R")
    flags+=("--show-managed-fields")
    local_nonpersistent_flags+=("--show-managed-fields")
    flags+=("--template=")
    two_word_flags+=("--template")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template")
    local_nonpersistent_flags+=("--template=")
    flags+=("--type=")
    two_word_flags+=("--type")
    local_nonpersistent_flags+=("--type")
    local_nonpersistent_flags+=("--type=")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    has_completion_function=1
    noun_aliases=()
}

_kubectl_plugin_list()
{
    last_command="kubectl_plugin_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--name-only")
    local_nonpersistent_flags+=("--name-only")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_plugin()
{
    last_command="kubectl_plugin"

    command_aliases=()

    commands=()
    commands+=("list")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_port-forward()
{
    last_command="kubectl_port-forward"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--address=")
    two_word_flags+=("--address")
    local_nonpersistent_flags+=("--address")
    local_nonpersistent_flags+=("--address=")
    flags+=("--pod-running-timeout=")
    two_word_flags+=("--pod-running-timeout")
    local_nonpersistent_flags+=("--pod-running-timeout")
    local_nonpersistent_flags+=("--pod-running-timeout=")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    has_completion_function=1
    noun_aliases=()
}

_kubectl_proxy()
{
    last_command="kubectl_proxy"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--accept-hosts=")
    two_word_flags+=("--accept-hosts")
    local_nonpersistent_flags+=("--accept-hosts")
    local_nonpersistent_flags+=("--accept-hosts=")
    flags+=("--accept-paths=")
    two_word_flags+=("--accept-paths")
    local_nonpersistent_flags+=("--accept-paths")
    local_nonpersistent_flags+=("--accept-paths=")
    flags+=("--address=")
    two_word_flags+=("--address")
    local_nonpersistent_flags+=("--address")
    local_nonpersistent_flags+=("--address=")
    flags+=("--api-prefix=")
    two_word_flags+=("--api-prefix")
    local_nonpersistent_flags+=("--api-prefix")
    local_nonpersistent_flags+=("--api-prefix=")
    flags+=("--append-server-path")
    local_nonpersistent_flags+=("--append-server-path")
    flags+=("--disable-filter")
    local_nonpersistent_flags+=("--disable-filter")
    flags+=("--keepalive=")
    two_word_flags+=("--keepalive")
    local_nonpersistent_flags+=("--keepalive")
    local_nonpersistent_flags+=("--keepalive=")
    flags+=("--port=")
    two_word_flags+=("--port")
    two_word_flags+=("-p")
    local_nonpersistent_flags+=("--port")
    local_nonpersistent_flags+=("--port=")
    local_nonpersistent_flags+=("-p")
    flags+=("--reject-methods=")
    two_word_flags+=("--reject-methods")
    local_nonpersistent_flags+=("--reject-methods")
    local_nonpersistent_flags+=("--reject-methods=")
    flags+=("--reject-paths=")
    two_word_flags+=("--reject-paths")
    local_nonpersistent_flags+=("--reject-paths")
    local_nonpersistent_flags+=("--reject-paths=")
    flags+=("--unix-socket=")
    two_word_flags+=("--unix-socket")
    two_word_flags+=("-u")
    local_nonpersistent_flags+=("--unix-socket")
    local_nonpersistent_flags+=("--unix-socket=")
    local_nonpersistent_flags+=("-u")
    flags+=("--www=")
    two_word_flags+=("--www")
    two_word_flags+=("-w")
    local_nonpersistent_flags+=("--www")
    local_nonpersistent_flags+=("--www=")
    local_nonpersistent_flags+=("-w")
    flags+=("--www-prefix=")
    two_word_flags+=("--www-prefix")
    two_word_flags+=("-P")
    local_nonpersistent_flags+=("--www-prefix")
    local_nonpersistent_flags+=("--www-prefix=")
    local_nonpersistent_flags+=("-P")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_replace()
{
    last_command="kubectl_replace"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--cascade")
    local_nonpersistent_flags+=("--cascade")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--field-manager=")
    two_word_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager=")
    flags+=("--filename=")
    two_word_flags+=("--filename")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename")
    local_nonpersistent_flags+=("--filename=")
    local_nonpersistent_flags+=("-f")
    flags+=("--force")
    local_nonpersistent_flags+=("--force")
    flags+=("--grace-period=")
    two_word_flags+=("--grace-period")
    local_nonpersistent_flags+=("--grace-period")
    local_nonpersistent_flags+=("--grace-period=")
    flags+=("--kustomize=")
    two_word_flags+=("--kustomize")
    two_word_flags+=("-k")
    local_nonpersistent_flags+=("--kustomize")
    local_nonpersistent_flags+=("--kustomize=")
    local_nonpersistent_flags+=("-k")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--raw=")
    two_word_flags+=("--raw")
    local_nonpersistent_flags+=("--raw")
    local_nonpersistent_flags+=("--raw=")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    local_nonpersistent_flags+=("-R")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--show-managed-fields")
    local_nonpersistent_flags+=("--show-managed-fields")
    flags+=("--template=")
    two_word_flags+=("--template")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template")
    local_nonpersistent_flags+=("--template=")
    flags+=("--timeout=")
    two_word_flags+=("--timeout")
    local_nonpersistent_flags+=("--timeout")
    local_nonpersistent_flags+=("--timeout=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--wait")
    local_nonpersistent_flags+=("--wait")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_rollout_history()
{
    last_command="kubectl_rollout_history"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--filename=")
    two_word_flags+=("--filename")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename")
    local_nonpersistent_flags+=("--filename=")
    local_nonpersistent_flags+=("-f")
    flags+=("--kustomize=")
    two_word_flags+=("--kustomize")
    two_word_flags+=("-k")
    local_nonpersistent_flags+=("--kustomize")
    local_nonpersistent_flags+=("--kustomize=")
    local_nonpersistent_flags+=("-k")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    local_nonpersistent_flags+=("-R")
    flags+=("--revision=")
    two_word_flags+=("--revision")
    local_nonpersistent_flags+=("--revision")
    local_nonpersistent_flags+=("--revision=")
    flags+=("--show-managed-fields")
    local_nonpersistent_flags+=("--show-managed-fields")
    flags+=("--template=")
    two_word_flags+=("--template")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template")
    local_nonpersistent_flags+=("--template=")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    has_completion_function=1
    noun_aliases=()
}

_kubectl_rollout_pause()
{
    last_command="kubectl_rollout_pause"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--field-manager=")
    two_word_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager=")
    flags+=("--filename=")
    two_word_flags+=("--filename")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename")
    local_nonpersistent_flags+=("--filename=")
    local_nonpersistent_flags+=("-f")
    flags+=("--kustomize=")
    two_word_flags+=("--kustomize")
    two_word_flags+=("-k")
    local_nonpersistent_flags+=("--kustomize")
    local_nonpersistent_flags+=("--kustomize=")
    local_nonpersistent_flags+=("-k")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    local_nonpersistent_flags+=("-R")
    flags+=("--show-managed-fields")
    local_nonpersistent_flags+=("--show-managed-fields")
    flags+=("--template=")
    two_word_flags+=("--template")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template")
    local_nonpersistent_flags+=("--template=")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    has_completion_function=1
    noun_aliases=()
}

_kubectl_rollout_restart()
{
    last_command="kubectl_rollout_restart"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--field-manager=")
    two_word_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager=")
    flags+=("--filename=")
    two_word_flags+=("--filename")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename")
    local_nonpersistent_flags+=("--filename=")
    local_nonpersistent_flags+=("-f")
    flags+=("--kustomize=")
    two_word_flags+=("--kustomize")
    two_word_flags+=("-k")
    local_nonpersistent_flags+=("--kustomize")
    local_nonpersistent_flags+=("--kustomize=")
    local_nonpersistent_flags+=("-k")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    local_nonpersistent_flags+=("-R")
    flags+=("--show-managed-fields")
    local_nonpersistent_flags+=("--show-managed-fields")
    flags+=("--template=")
    two_word_flags+=("--template")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template")
    local_nonpersistent_flags+=("--template=")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    has_completion_function=1
    noun_aliases=()
}

_kubectl_rollout_resume()
{
    last_command="kubectl_rollout_resume"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--field-manager=")
    two_word_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager=")
    flags+=("--filename=")
    two_word_flags+=("--filename")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename")
    local_nonpersistent_flags+=("--filename=")
    local_nonpersistent_flags+=("-f")
    flags+=("--kustomize=")
    two_word_flags+=("--kustomize")
    two_word_flags+=("-k")
    local_nonpersistent_flags+=("--kustomize")
    local_nonpersistent_flags+=("--kustomize=")
    local_nonpersistent_flags+=("-k")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    local_nonpersistent_flags+=("-R")
    flags+=("--show-managed-fields")
    local_nonpersistent_flags+=("--show-managed-fields")
    flags+=("--template=")
    two_word_flags+=("--template")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template")
    local_nonpersistent_flags+=("--template=")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    has_completion_function=1
    noun_aliases=()
}

_kubectl_rollout_status()
{
    last_command="kubectl_rollout_status"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--filename=")
    two_word_flags+=("--filename")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename")
    local_nonpersistent_flags+=("--filename=")
    local_nonpersistent_flags+=("-f")
    flags+=("--kustomize=")
    two_word_flags+=("--kustomize")
    two_word_flags+=("-k")
    local_nonpersistent_flags+=("--kustomize")
    local_nonpersistent_flags+=("--kustomize=")
    local_nonpersistent_flags+=("-k")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    local_nonpersistent_flags+=("-R")
    flags+=("--revision=")
    two_word_flags+=("--revision")
    local_nonpersistent_flags+=("--revision")
    local_nonpersistent_flags+=("--revision=")
    flags+=("--timeout=")
    two_word_flags+=("--timeout")
    local_nonpersistent_flags+=("--timeout")
    local_nonpersistent_flags+=("--timeout=")
    flags+=("--watch")
    flags+=("-w")
    local_nonpersistent_flags+=("--watch")
    local_nonpersistent_flags+=("-w")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    has_completion_function=1
    noun_aliases=()
}

_kubectl_rollout_undo()
{
    last_command="kubectl_rollout_undo"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--filename=")
    two_word_flags+=("--filename")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename")
    local_nonpersistent_flags+=("--filename=")
    local_nonpersistent_flags+=("-f")
    flags+=("--kustomize=")
    two_word_flags+=("--kustomize")
    two_word_flags+=("-k")
    local_nonpersistent_flags+=("--kustomize")
    local_nonpersistent_flags+=("--kustomize=")
    local_nonpersistent_flags+=("-k")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    local_nonpersistent_flags+=("-R")
    flags+=("--show-managed-fields")
    local_nonpersistent_flags+=("--show-managed-fields")
    flags+=("--template=")
    two_word_flags+=("--template")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template")
    local_nonpersistent_flags+=("--template=")
    flags+=("--to-revision=")
    two_word_flags+=("--to-revision")
    local_nonpersistent_flags+=("--to-revision")
    local_nonpersistent_flags+=("--to-revision=")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    has_completion_function=1
    noun_aliases=()
}

_kubectl_rollout()
{
    last_command="kubectl_rollout"

    command_aliases=()

    commands=()
    commands+=("history")
    commands+=("pause")
    commands+=("restart")
    commands+=("resume")
    commands+=("status")
    commands+=("undo")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_run()
{
    last_command="kubectl_run"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--annotations=")
    two_word_flags+=("--annotations")
    local_nonpersistent_flags+=("--annotations")
    local_nonpersistent_flags+=("--annotations=")
    flags+=("--attach")
    local_nonpersistent_flags+=("--attach")
    flags+=("--cascade")
    local_nonpersistent_flags+=("--cascade")
    flags+=("--command")
    local_nonpersistent_flags+=("--command")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--env=")
    two_word_flags+=("--env")
    local_nonpersistent_flags+=("--env")
    local_nonpersistent_flags+=("--env=")
    flags+=("--expose")
    local_nonpersistent_flags+=("--expose")
    flags+=("--field-manager=")
    two_word_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager=")
    flags+=("--filename=")
    two_word_flags+=("--filename")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename")
    local_nonpersistent_flags+=("--filename=")
    local_nonpersistent_flags+=("-f")
    flags+=("--force")
    local_nonpersistent_flags+=("--force")
    flags+=("--grace-period=")
    two_word_flags+=("--grace-period")
    local_nonpersistent_flags+=("--grace-period")
    local_nonpersistent_flags+=("--grace-period=")
    flags+=("--image=")
    two_word_flags+=("--image")
    local_nonpersistent_flags+=("--image")
    local_nonpersistent_flags+=("--image=")
    flags+=("--image-pull-policy=")
    two_word_flags+=("--image-pull-policy")
    local_nonpersistent_flags+=("--image-pull-policy")
    local_nonpersistent_flags+=("--image-pull-policy=")
    flags+=("--kustomize=")
    two_word_flags+=("--kustomize")
    two_word_flags+=("-k")
    local_nonpersistent_flags+=("--kustomize")
    local_nonpersistent_flags+=("--kustomize=")
    local_nonpersistent_flags+=("-k")
    flags+=("--labels=")
    two_word_flags+=("--labels")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--labels")
    local_nonpersistent_flags+=("--labels=")
    local_nonpersistent_flags+=("-l")
    flags+=("--leave-stdin-open")
    local_nonpersistent_flags+=("--leave-stdin-open")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--override-type=")
    two_word_flags+=("--override-type")
    local_nonpersistent_flags+=("--override-type")
    local_nonpersistent_flags+=("--override-type=")
    flags+=("--overrides=")
    two_word_flags+=("--overrides")
    local_nonpersistent_flags+=("--overrides")
    local_nonpersistent_flags+=("--overrides=")
    flags+=("--pod-running-timeout=")
    two_word_flags+=("--pod-running-timeout")
    local_nonpersistent_flags+=("--pod-running-timeout")
    local_nonpersistent_flags+=("--pod-running-timeout=")
    flags+=("--port=")
    two_word_flags+=("--port")
    local_nonpersistent_flags+=("--port")
    local_nonpersistent_flags+=("--port=")
    flags+=("--privileged")
    local_nonpersistent_flags+=("--privileged")
    flags+=("--quiet")
    flags+=("-q")
    local_nonpersistent_flags+=("--quiet")
    local_nonpersistent_flags+=("-q")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    local_nonpersistent_flags+=("-R")
    flags+=("--restart=")
    two_word_flags+=("--restart")
    local_nonpersistent_flags+=("--restart")
    local_nonpersistent_flags+=("--restart=")
    flags+=("--rm")
    local_nonpersistent_flags+=("--rm")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--show-managed-fields")
    local_nonpersistent_flags+=("--show-managed-fields")
    flags+=("--stdin")
    flags+=("-i")
    local_nonpersistent_flags+=("--stdin")
    local_nonpersistent_flags+=("-i")
    flags+=("--template=")
    two_word_flags+=("--template")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template")
    local_nonpersistent_flags+=("--template=")
    flags+=("--timeout=")
    two_word_flags+=("--timeout")
    local_nonpersistent_flags+=("--timeout")
    local_nonpersistent_flags+=("--timeout=")
    flags+=("--tty")
    flags+=("-t")
    local_nonpersistent_flags+=("--tty")
    local_nonpersistent_flags+=("-t")
    flags+=("--wait")
    local_nonpersistent_flags+=("--wait")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_flag+=("--image=")
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_scale()
{
    last_command="kubectl_scale"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--all")
    local_nonpersistent_flags+=("--all")
    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--current-replicas=")
    two_word_flags+=("--current-replicas")
    local_nonpersistent_flags+=("--current-replicas")
    local_nonpersistent_flags+=("--current-replicas=")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--filename=")
    two_word_flags+=("--filename")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename")
    local_nonpersistent_flags+=("--filename=")
    local_nonpersistent_flags+=("-f")
    flags+=("--kustomize=")
    two_word_flags+=("--kustomize")
    two_word_flags+=("-k")
    local_nonpersistent_flags+=("--kustomize")
    local_nonpersistent_flags+=("--kustomize=")
    local_nonpersistent_flags+=("-k")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    local_nonpersistent_flags+=("-R")
    flags+=("--replicas=")
    two_word_flags+=("--replicas")
    local_nonpersistent_flags+=("--replicas")
    local_nonpersistent_flags+=("--replicas=")
    flags+=("--resource-version=")
    two_word_flags+=("--resource-version")
    local_nonpersistent_flags+=("--resource-version")
    local_nonpersistent_flags+=("--resource-version=")
    flags+=("--selector=")
    two_word_flags+=("--selector")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--selector")
    local_nonpersistent_flags+=("--selector=")
    local_nonpersistent_flags+=("-l")
    flags+=("--show-managed-fields")
    local_nonpersistent_flags+=("--show-managed-fields")
    flags+=("--template=")
    two_word_flags+=("--template")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template")
    local_nonpersistent_flags+=("--template=")
    flags+=("--timeout=")
    two_word_flags+=("--timeout")
    local_nonpersistent_flags+=("--timeout")
    local_nonpersistent_flags+=("--timeout=")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_flag+=("--replicas=")
    must_have_one_noun=()
    has_completion_function=1
    noun_aliases=()
}

_kubectl_set_env()
{
    last_command="kubectl_set_env"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--all")
    local_nonpersistent_flags+=("--all")
    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--containers=")
    two_word_flags+=("--containers")
    two_word_flags+=("-c")
    local_nonpersistent_flags+=("--containers")
    local_nonpersistent_flags+=("--containers=")
    local_nonpersistent_flags+=("-c")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--env=")
    two_word_flags+=("--env")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--env")
    local_nonpersistent_flags+=("--env=")
    local_nonpersistent_flags+=("-e")
    flags+=("--field-manager=")
    two_word_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager=")
    flags+=("--filename=")
    two_word_flags+=("--filename")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename")
    local_nonpersistent_flags+=("--filename=")
    local_nonpersistent_flags+=("-f")
    flags+=("--from=")
    two_word_flags+=("--from")
    local_nonpersistent_flags+=("--from")
    local_nonpersistent_flags+=("--from=")
    flags+=("--keys=")
    two_word_flags+=("--keys")
    local_nonpersistent_flags+=("--keys")
    local_nonpersistent_flags+=("--keys=")
    flags+=("--kustomize=")
    two_word_flags+=("--kustomize")
    two_word_flags+=("-k")
    local_nonpersistent_flags+=("--kustomize")
    local_nonpersistent_flags+=("--kustomize=")
    local_nonpersistent_flags+=("-k")
    flags+=("--list")
    local_nonpersistent_flags+=("--list")
    flags+=("--local")
    local_nonpersistent_flags+=("--local")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--overwrite")
    local_nonpersistent_flags+=("--overwrite")
    flags+=("--prefix=")
    two_word_flags+=("--prefix")
    local_nonpersistent_flags+=("--prefix")
    local_nonpersistent_flags+=("--prefix=")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    local_nonpersistent_flags+=("-R")
    flags+=("--resolve")
    local_nonpersistent_flags+=("--resolve")
    flags+=("--selector=")
    two_word_flags+=("--selector")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--selector")
    local_nonpersistent_flags+=("--selector=")
    local_nonpersistent_flags+=("-l")
    flags+=("--show-managed-fields")
    local_nonpersistent_flags+=("--show-managed-fields")
    flags+=("--template=")
    two_word_flags+=("--template")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template")
    local_nonpersistent_flags+=("--template=")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_set_image()
{
    last_command="kubectl_set_image"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--all")
    local_nonpersistent_flags+=("--all")
    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--field-manager=")
    two_word_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager=")
    flags+=("--filename=")
    two_word_flags+=("--filename")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename")
    local_nonpersistent_flags+=("--filename=")
    local_nonpersistent_flags+=("-f")
    flags+=("--kustomize=")
    two_word_flags+=("--kustomize")
    two_word_flags+=("-k")
    local_nonpersistent_flags+=("--kustomize")
    local_nonpersistent_flags+=("--kustomize=")
    local_nonpersistent_flags+=("-k")
    flags+=("--local")
    local_nonpersistent_flags+=("--local")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    local_nonpersistent_flags+=("-R")
    flags+=("--selector=")
    two_word_flags+=("--selector")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--selector")
    local_nonpersistent_flags+=("--selector=")
    local_nonpersistent_flags+=("-l")
    flags+=("--show-managed-fields")
    local_nonpersistent_flags+=("--show-managed-fields")
    flags+=("--template=")
    two_word_flags+=("--template")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template")
    local_nonpersistent_flags+=("--template=")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_set_resources()
{
    last_command="kubectl_set_resources"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--all")
    local_nonpersistent_flags+=("--all")
    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--containers=")
    two_word_flags+=("--containers")
    two_word_flags+=("-c")
    local_nonpersistent_flags+=("--containers")
    local_nonpersistent_flags+=("--containers=")
    local_nonpersistent_flags+=("-c")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--field-manager=")
    two_word_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager=")
    flags+=("--filename=")
    two_word_flags+=("--filename")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename")
    local_nonpersistent_flags+=("--filename=")
    local_nonpersistent_flags+=("-f")
    flags+=("--kustomize=")
    two_word_flags+=("--kustomize")
    two_word_flags+=("-k")
    local_nonpersistent_flags+=("--kustomize")
    local_nonpersistent_flags+=("--kustomize=")
    local_nonpersistent_flags+=("-k")
    flags+=("--limits=")
    two_word_flags+=("--limits")
    local_nonpersistent_flags+=("--limits")
    local_nonpersistent_flags+=("--limits=")
    flags+=("--local")
    local_nonpersistent_flags+=("--local")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    local_nonpersistent_flags+=("-R")
    flags+=("--requests=")
    two_word_flags+=("--requests")
    local_nonpersistent_flags+=("--requests")
    local_nonpersistent_flags+=("--requests=")
    flags+=("--selector=")
    two_word_flags+=("--selector")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--selector")
    local_nonpersistent_flags+=("--selector=")
    local_nonpersistent_flags+=("-l")
    flags+=("--show-managed-fields")
    local_nonpersistent_flags+=("--show-managed-fields")
    flags+=("--template=")
    two_word_flags+=("--template")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template")
    local_nonpersistent_flags+=("--template=")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_set_selector()
{
    last_command="kubectl_set_selector"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--all")
    local_nonpersistent_flags+=("--all")
    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--field-manager=")
    two_word_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager=")
    flags+=("--filename=")
    two_word_flags+=("--filename")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename")
    local_nonpersistent_flags+=("--filename=")
    local_nonpersistent_flags+=("-f")
    flags+=("--local")
    local_nonpersistent_flags+=("--local")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    local_nonpersistent_flags+=("-R")
    flags+=("--resource-version=")
    two_word_flags+=("--resource-version")
    local_nonpersistent_flags+=("--resource-version")
    local_nonpersistent_flags+=("--resource-version=")
    flags+=("--show-managed-fields")
    local_nonpersistent_flags+=("--show-managed-fields")
    flags+=("--template=")
    two_word_flags+=("--template")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template")
    local_nonpersistent_flags+=("--template=")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_set_serviceaccount()
{
    last_command="kubectl_set_serviceaccount"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--all")
    local_nonpersistent_flags+=("--all")
    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--field-manager=")
    two_word_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager=")
    flags+=("--filename=")
    two_word_flags+=("--filename")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename")
    local_nonpersistent_flags+=("--filename=")
    local_nonpersistent_flags+=("-f")
    flags+=("--kustomize=")
    two_word_flags+=("--kustomize")
    two_word_flags+=("-k")
    local_nonpersistent_flags+=("--kustomize")
    local_nonpersistent_flags+=("--kustomize=")
    local_nonpersistent_flags+=("-k")
    flags+=("--local")
    local_nonpersistent_flags+=("--local")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    local_nonpersistent_flags+=("-R")
    flags+=("--show-managed-fields")
    local_nonpersistent_flags+=("--show-managed-fields")
    flags+=("--template=")
    two_word_flags+=("--template")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template")
    local_nonpersistent_flags+=("--template=")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_set_subject()
{
    last_command="kubectl_set_subject"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--all")
    local_nonpersistent_flags+=("--all")
    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--field-manager=")
    two_word_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager=")
    flags+=("--filename=")
    two_word_flags+=("--filename")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename")
    local_nonpersistent_flags+=("--filename=")
    local_nonpersistent_flags+=("-f")
    flags+=("--group=")
    two_word_flags+=("--group")
    local_nonpersistent_flags+=("--group")
    local_nonpersistent_flags+=("--group=")
    flags+=("--kustomize=")
    two_word_flags+=("--kustomize")
    two_word_flags+=("-k")
    local_nonpersistent_flags+=("--kustomize")
    local_nonpersistent_flags+=("--kustomize=")
    local_nonpersistent_flags+=("-k")
    flags+=("--local")
    local_nonpersistent_flags+=("--local")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    local_nonpersistent_flags+=("-R")
    flags+=("--selector=")
    two_word_flags+=("--selector")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--selector")
    local_nonpersistent_flags+=("--selector=")
    local_nonpersistent_flags+=("-l")
    flags+=("--serviceaccount=")
    two_word_flags+=("--serviceaccount")
    local_nonpersistent_flags+=("--serviceaccount")
    local_nonpersistent_flags+=("--serviceaccount=")
    flags+=("--show-managed-fields")
    local_nonpersistent_flags+=("--show-managed-fields")
    flags+=("--template=")
    two_word_flags+=("--template")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template")
    local_nonpersistent_flags+=("--template=")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_set()
{
    last_command="kubectl_set"

    command_aliases=()

    commands=()
    commands+=("env")
    commands+=("image")
    commands+=("resources")
    commands+=("selector")
    commands+=("serviceaccount")
    if [[ -z "${BASH_VERSION}" || "${BASH_VERSINFO[0]}" -gt 3 ]]; then
        command_aliases+=("sa")
        aliashash["sa"]="serviceaccount"
    fi
    commands+=("subject")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_taint()
{
    last_command="kubectl_taint"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--all")
    local_nonpersistent_flags+=("--all")
    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--field-manager=")
    two_word_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager")
    local_nonpersistent_flags+=("--field-manager=")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--overwrite")
    local_nonpersistent_flags+=("--overwrite")
    flags+=("--selector=")
    two_word_flags+=("--selector")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--selector")
    local_nonpersistent_flags+=("--selector=")
    local_nonpersistent_flags+=("-l")
    flags+=("--show-managed-fields")
    local_nonpersistent_flags+=("--show-managed-fields")
    flags+=("--template=")
    two_word_flags+=("--template")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    has_completion_function=1
    noun_aliases=()
}

_kubectl_top_node()
{
    last_command="kubectl_top_node"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--selector=")
    two_word_flags+=("--selector")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--selector")
    local_nonpersistent_flags+=("--selector=")
    local_nonpersistent_flags+=("-l")
    flags+=("--show-capacity")
    local_nonpersistent_flags+=("--show-capacity")
    flags+=("--sort-by=")
    two_word_flags+=("--sort-by")
    local_nonpersistent_flags+=("--sort-by")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--use-protocol-buffers")
    local_nonpersistent_flags+=("--use-protocol-buffers")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    has_completion_function=1
    noun_aliases=()
}

_kubectl_top_pod()
{
    last_command="kubectl_top_pod"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--all-namespaces")
    flags+=("-A")
    local_nonpersistent_flags+=("--all-namespaces")
    local_nonpersistent_flags+=("-A")
    flags+=("--containers")
    local_nonpersistent_flags+=("--containers")
    flags+=("--field-selector=")
    two_word_flags+=("--field-selector")
    local_nonpersistent_flags+=("--field-selector")
    local_nonpersistent_flags+=("--field-selector=")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--selector=")
    two_word_flags+=("--selector")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--selector")
    local_nonpersistent_flags+=("--selector=")
    local_nonpersistent_flags+=("-l")
    flags+=("--sort-by=")
    two_word_flags+=("--sort-by")
    local_nonpersistent_flags+=("--sort-by")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--use-protocol-buffers")
    local_nonpersistent_flags+=("--use-protocol-buffers")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    has_completion_function=1
    noun_aliases=()
}

_kubectl_top()
{
    last_command="kubectl_top"

    command_aliases=()

    commands=()
    commands+=("node")
    if [[ -z "${BASH_VERSION}" || "${BASH_VERSINFO[0]}" -gt 3 ]]; then
        command_aliases+=("no")
        aliashash["no"]="node"
        command_aliases+=("nodes")
        aliashash["nodes"]="node"
    fi
    commands+=("pod")
    if [[ -z "${BASH_VERSION}" || "${BASH_VERSINFO[0]}" -gt 3 ]]; then
        command_aliases+=("po")
        aliashash["po"]="pod"
        command_aliases+=("pods")
        aliashash["pods"]="pod"
    fi

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_uncordon()
{
    last_command="kubectl_uncordon"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--selector=")
    two_word_flags+=("--selector")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--selector")
    local_nonpersistent_flags+=("--selector=")
    local_nonpersistent_flags+=("-l")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    has_completion_function=1
    noun_aliases=()
}

_kubectl_version()
{
    last_command="kubectl_version"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--client")
    local_nonpersistent_flags+=("--client")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--short")
    local_nonpersistent_flags+=("--short")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_wait()
{
    last_command="kubectl_wait"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--all")
    local_nonpersistent_flags+=("--all")
    flags+=("--all-namespaces")
    flags+=("-A")
    local_nonpersistent_flags+=("--all-namespaces")
    local_nonpersistent_flags+=("-A")
    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--field-selector=")
    two_word_flags+=("--field-selector")
    local_nonpersistent_flags+=("--field-selector")
    local_nonpersistent_flags+=("--field-selector=")
    flags+=("--filename=")
    two_word_flags+=("--filename")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename")
    local_nonpersistent_flags+=("--filename=")
    local_nonpersistent_flags+=("-f")
    flags+=("--for=")
    two_word_flags+=("--for")
    local_nonpersistent_flags+=("--for")
    local_nonpersistent_flags+=("--for=")
    flags+=("--local")
    local_nonpersistent_flags+=("--local")
    flags+=("--output=")
    two_word_flags+=("--output")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output")
    local_nonpersistent_flags+=("--output=")
    local_nonpersistent_flags+=("-o")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    local_nonpersistent_flags+=("-R")
    flags+=("--selector=")
    two_word_flags+=("--selector")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--selector")
    local_nonpersistent_flags+=("--selector=")
    local_nonpersistent_flags+=("-l")
    flags+=("--show-managed-fields")
    local_nonpersistent_flags+=("--show-managed-fields")
    flags+=("--template=")
    two_word_flags+=("--template")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template")
    local_nonpersistent_flags+=("--template=")
    flags+=("--timeout=")
    two_word_flags+=("--timeout")
    local_nonpersistent_flags+=("--timeout")
    local_nonpersistent_flags+=("--timeout=")
    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_root_command()
{
    last_command="kubectl"

    command_aliases=()

    commands=()
    commands+=("alpha")
    commands+=("annotate")
    commands+=("api-resources")
    commands+=("api-versions")
    commands+=("apply")
    commands+=("attach")
    commands+=("auth")
    commands+=("autoscale")
    commands+=("certificate")
    commands+=("cluster-info")
    commands+=("completion")
    commands+=("config")
    commands+=("cordon")
    commands+=("cp")
    commands+=("create")
    commands+=("debug")
    commands+=("delete")
    commands+=("describe")
    commands+=("diff")
    commands+=("drain")
    commands+=("edit")
    commands+=("exec")
    commands+=("explain")
    commands+=("expose")
    commands+=("get")
    commands+=("help")
    commands+=("kustomize")
    commands+=("label")
    commands+=("logs")
    commands+=("options")
    commands+=("patch")
    commands+=("plugin")
    commands+=("port-forward")
    commands+=("proxy")
    commands+=("replace")
    commands+=("rollout")
    commands+=("run")
    commands+=("scale")
    commands+=("set")
    commands+=("taint")
    commands+=("top")
    commands+=("uncordon")
    commands+=("version")
    commands+=("wait")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--as=")
    two_word_flags+=("--as")
    flags+=("--as-group=")
    two_word_flags+=("--as-group")
    flags+=("--as-uid=")
    two_word_flags+=("--as-uid")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--certificate-authority=")
    two_word_flags+=("--certificate-authority")
    flags+=("--client-certificate=")
    two_word_flags+=("--client-certificate")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--cluster=")
    two_word_flags+=("--cluster")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--context=")
    two_word_flags+=("--context")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    two_word_flags+=("--kubeconfig")
    flags+=("--log-flush-frequency=")
    two_word_flags+=("--log-flush-frequency")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    two_word_flags+=("--namespace")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--password=")
    two_word_flags+=("--password")
    flags+=("--profile=")
    two_word_flags+=("--profile")
    flags+=("--profile-output=")
    two_word_flags+=("--profile-output")
    flags+=("--request-timeout=")
    two_word_flags+=("--request-timeout")
    flags+=("--server=")
    two_word_flags+=("--server")
    two_word_flags+=("-s")
    flags+=("--tls-server-name=")
    two_word_flags+=("--tls-server-name")
    flags+=("--token=")
    two_word_flags+=("--token")
    flags+=("--user=")
    two_word_flags+=("--user")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_handle_go_custom_completion")
    flags+=("--username=")
    two_word_flags+=("--username")
    flags+=("--v=")
    two_word_flags+=("--v")
    two_word_flags+=("-v")
    flags+=("--vmodule=")
    two_word_flags+=("--vmodule")
    flags+=("--warnings-as-errors")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

__start_kubectl()
{
    local cur prev words cword split
    declare -A flaghash 2>/dev/null || :
    declare -A aliashash 2>/dev/null || :
    if declare -F _init_completion >/dev/null 2>&1; then
        _init_completion -s || return
    else
        __kubectl_init_completion -n "=" || return
    fi

    local c=0
    local flags=()
    local two_word_flags=()
    local local_nonpersistent_flags=()
    local flags_with_completion=()
    local flags_completion=()
    local commands=("kubectl")
    local command_aliases=()
    local must_have_one_flag=()
    local must_have_one_noun=()
    local has_completion_function
    local last_command
    local nouns=()
    local noun_aliases=()

    __kubectl_handle_word
}

if [[ $(type -t compopt) = "builtin" ]]; then
    complete -o default -F __start_kubectl kubectl
else
    complete -o default -o nospace -F __start_kubectl kubectl
fi

# ex: ts=4 sw=4 et filetype=sh
