
# Copyright 2016 The Kubernetes Authors.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
# bash completion for kubectl                              -*- shell-script -*-

__kubectl_debug()
{
    if [[ -n ${BASH_COMP_DEBUG_FILE} ]]; then
        echo "$*" >> "${BASH_COMP_DEBUG_FILE}"
    fi
}

# Homebrew on Macs have version 1.3 of bash-completion which doesn't include
# _init_completion. This is a very minimal version of that function.
__kubectl_init_completion()
{
    COMPREPLY=()
    _get_comp_words_by_ref "$@" cur prev words cword
}

__kubectl_index_of_word()
{
    local w word=$1
    shift
    index=0
    for w in "$@"; do
        [[ $w = "$word" ]] && return
        index=$((index+1))
    done
    index=-1
}

__kubectl_contains_word()
{
    local w word=$1; shift
    for w in "$@"; do
        [[ $w = "$word" ]] && return
    done
    return 1
}

__kubectl_handle_reply()
{
    __kubectl_debug "${FUNCNAME[0]}"
    case $cur in
        -*)
            if [[ $(type -t compopt) = "builtin" ]]; then
                compopt -o nospace
            fi
            local allflags
            if [ ${#must_have_one_flag[@]} -ne 0 ]; then
                allflags=("${must_have_one_flag[@]}")
            else
                allflags=("${flags[*]} ${two_word_flags[*]}")
            fi
            COMPREPLY=( $(compgen -W "${allflags[*]}" -- "$cur") )
            if [[ $(type -t compopt) = "builtin" ]]; then
                [[ "${COMPREPLY[0]}" == *= ]] || compopt +o nospace
            fi

            # complete after --flag=abc
            if [[ $cur == *=* ]]; then
                if [[ $(type -t compopt) = "builtin" ]]; then
                    compopt +o nospace
                fi

                local index flag
                flag="${cur%=*}"
                __kubectl_index_of_word "${flag}" "${flags_with_completion[@]}"
                COMPREPLY=()
                if [[ ${index} -ge 0 ]]; then
                    PREFIX=""
                    cur="${cur#*=}"
                    ${flags_completion[${index}]}
                    if [ -n "${ZSH_VERSION}" ]; then
                        # zsh completion needs --flag= prefix
                        eval "COMPREPLY=( \"\${COMPREPLY[@]/#/${flag}=}\" )"
                    fi
                fi
            fi
            return 0;
            ;;
    esac

    # check if we are handling a flag with special work handling
    local index
    __kubectl_index_of_word "${prev}" "${flags_with_completion[@]}"
    if [[ ${index} -ge 0 ]]; then
        ${flags_completion[${index}]}
        return
    fi

    # we are parsing a flag and don't have a special handler, no completion
    if [[ ${cur} != "${words[cword]}" ]]; then
        return
    fi

    local completions
    completions=("${commands[@]}")
    if [[ ${#must_have_one_noun[@]} -ne 0 ]]; then
        completions=("${must_have_one_noun[@]}")
    fi
    if [[ ${#must_have_one_flag[@]} -ne 0 ]]; then
        completions+=("${must_have_one_flag[@]}")
    fi
    COMPREPLY=( $(compgen -W "${completions[*]}" -- "$cur") )

    if [[ ${#COMPREPLY[@]} -eq 0 && ${#noun_aliases[@]} -gt 0 && ${#must_have_one_noun[@]} -ne 0 ]]; then
        COMPREPLY=( $(compgen -W "${noun_aliases[*]}" -- "$cur") )
    fi

    if [[ ${#COMPREPLY[@]} -eq 0 ]]; then
        declare -F __custom_func >/dev/null && __custom_func
    fi

    # available in bash-completion >= 2, not always present on macOS
    if declare -F __ltrim_colon_completions >/dev/null; then
        __ltrim_colon_completions "$cur"
    fi

    # If there is only 1 completion and it is a flag with an = it will be completed
    # but we don't want a space after the =
    if [[ "${#COMPREPLY[@]}" -eq "1" ]] && [[ $(type -t compopt) = "builtin" ]] && [[ "${COMPREPLY[0]}" == --*= ]]; then
       compopt -o nospace
    fi
}

# The arguments should be in the form "ext1|ext2|extn"
__kubectl_handle_filename_extension_flag()
{
    local ext="$1"
    _filedir "@(${ext})"
}

__kubectl_handle_subdirs_in_dir_flag()
{
    local dir="$1"
    pushd "${dir}" >/dev/null 2>&1 && _filedir -d && popd >/dev/null 2>&1
}

__kubectl_handle_flag()
{
    __kubectl_debug "${FUNCNAME[0]}: c is $c words[c] is ${words[c]}"

    # if a command required a flag, and we found it, unset must_have_one_flag()
    local flagname=${words[c]}
    local flagvalue
    # if the word contained an =
    if [[ ${words[c]} == *"="* ]]; then
        flagvalue=${flagname#*=} # take in as flagvalue after the =
        flagname=${flagname%=*} # strip everything after the =
        flagname="${flagname}=" # but put the = back
    fi
    __kubectl_debug "${FUNCNAME[0]}: looking for ${flagname}"
    if __kubectl_contains_word "${flagname}" "${must_have_one_flag[@]}"; then
        must_have_one_flag=()
    fi

    # if you set a flag which only applies to this command, don't show subcommands
    if __kubectl_contains_word "${flagname}" "${local_nonpersistent_flags[@]}"; then
      commands=()
    fi

    # keep flag value with flagname as flaghash
    # flaghash variable is an associative array which is only supported in bash > 3.
    if [[ -z "${BASH_VERSION}" || "${BASH_VERSINFO[0]}" -gt 3 ]]; then
        if [ -n "${flagvalue}" ] ; then
            flaghash[${flagname}]=${flagvalue}
        elif [ -n "${words[ $((c+1)) ]}" ] ; then
            flaghash[${flagname}]=${words[ $((c+1)) ]}
        else
            flaghash[${flagname}]="true" # pad "true" for bool flag
        fi
    fi

    # skip the argument to a two word flag
    if __kubectl_contains_word "${words[c]}" "${two_word_flags[@]}"; then
        c=$((c+1))
        # if we are looking for a flags value, don't show commands
        if [[ $c -eq $cword ]]; then
            commands=()
        fi
    fi

    c=$((c+1))

}

__kubectl_handle_noun()
{
    __kubectl_debug "${FUNCNAME[0]}: c is $c words[c] is ${words[c]}"

    if __kubectl_contains_word "${words[c]}" "${must_have_one_noun[@]}"; then
        must_have_one_noun=()
    elif __kubectl_contains_word "${words[c]}" "${noun_aliases[@]}"; then
        must_have_one_noun=()
    fi

    nouns+=("${words[c]}")
    c=$((c+1))
}

__kubectl_handle_command()
{
    __kubectl_debug "${FUNCNAME[0]}: c is $c words[c] is ${words[c]}"

    local next_command
    if [[ -n ${last_command} ]]; then
        next_command="_${last_command}_${words[c]//:/__}"
    else
        if [[ $c -eq 0 ]]; then
            next_command="_kubectl_root_command"
        else
            next_command="_${words[c]//:/__}"
        fi
    fi
    c=$((c+1))
    __kubectl_debug "${FUNCNAME[0]}: looking for ${next_command}"
    declare -F "$next_command" >/dev/null && $next_command
}

__kubectl_handle_word()
{
    if [[ $c -ge $cword ]]; then
        __kubectl_handle_reply
        return
    fi
    __kubectl_debug "${FUNCNAME[0]}: c is $c words[c] is ${words[c]}"
    if [[ "${words[c]}" == -* ]]; then
        __kubectl_handle_flag
    elif __kubectl_contains_word "${words[c]}" "${commands[@]}"; then
        __kubectl_handle_command
    elif [[ $c -eq 0 ]]; then
        __kubectl_handle_command
    else
        __kubectl_handle_noun
    fi
    __kubectl_handle_word
}

# call kubectl get $1,
__kubectl_override_flag_list=(--kubeconfig --cluster --user --context --namespace --server -n -s)
__kubectl_override_flags()
{
    local ${__kubectl_override_flag_list[*]##*-} two_word_of of var
    for w in "${words[@]}"; do
        if [ -n "${two_word_of}" ]; then
            eval "${two_word_of##*-}=\"${two_word_of}=\${w}\""
            two_word_of=
            continue
        fi
        for of in "${__kubectl_override_flag_list[@]}"; do
            case "${w}" in
                ${of}=*)
                    eval "${of##*-}=\"${w}\""
                    ;;
                ${of})
                    two_word_of="${of}"
                    ;;
            esac
        done
        if [ "${w}" == "--all-namespaces" ]; then
            namespace="--all-namespaces"
        fi
    done
    for var in "${__kubectl_override_flag_list[@]##*-}"; do
        if eval "test -n \"\$${var}\""; then
            eval "echo \${${var}}"
        fi
    done
}

__kubectl_config_get_contexts()
{
    __kubectl_parse_config "contexts"
}

__kubectl_config_get_clusters()
{
    __kubectl_parse_config "clusters"
}

__kubectl_config_get_users()
{
    __kubectl_parse_config "users"
}

# $1 has to be "contexts", "clusters" or "users"
__kubectl_parse_config()
{
    local template kubectl_out
    template="{{ range .$1  }}{{ .name }} {{ end }}"
    if kubectl_out=$(kubectl config $(__kubectl_override_flags) -o template --template="${template}" view 2>/dev/null); then
        COMPREPLY=( $( compgen -W "${kubectl_out[*]}" -- "$cur" ) )
    fi
}

__kubectl_parse_get()
{
    local template
    template="{{ range .items  }}{{ .metadata.name }} {{ end }}"
    local kubectl_out
    if kubectl_out=$(kubectl get $(__kubectl_override_flags) -o template --template="${template}" "$1" 2>/dev/null); then
        COMPREPLY=( $( compgen -W "${kubectl_out[*]}" -- "$cur" ) )
    fi
}

__kubectl_get_resource()
{
    if [[ ${#nouns[@]} -eq 0 ]]; then
        return 1
    fi
    __kubectl_parse_get "${nouns[${#nouns[@]} -1]}"
}

__kubectl_get_resource_namespace()
{
    __kubectl_parse_get "namespace"
}

__kubectl_get_resource_pod()
{
    __kubectl_parse_get "pod"
}

__kubectl_get_resource_rc()
{
    __kubectl_parse_get "rc"
}

__kubectl_get_resource_node()
{
    __kubectl_parse_get "node"
}

__kubectl_get_resource_clusterrole()
{
    __kubectl_parse_get "clusterrole"
}

# $1 is the name of the pod we want to get the list of containers inside
__kubectl_get_containers()
{
    local template
    template="{{ range .spec.containers  }}{{ .name }} {{ end }}"
    __kubectl_debug "${FUNCNAME} nouns are ${nouns[*]}"

    local len="${#nouns[@]}"
    if [[ ${len} -ne 1 ]]; then
        return
    fi
    local last=${nouns[${len} -1]}
    local kubectl_out
    if kubectl_out=$(kubectl get $(__kubectl_override_flags) -o template --template="${template}" pods "${last}" 2>/dev/null); then
        COMPREPLY=( $( compgen -W "${kubectl_out[*]}" -- "$cur" ) )
    fi
}

# Require both a pod and a container to be specified
__kubectl_require_pod_and_container()
{
    if [[ ${#nouns[@]} -eq 0 ]]; then
        __kubectl_parse_get pods
        return 0
    fi;
    __kubectl_get_containers
    return 0
}

__custom_func() {
    case ${last_command} in
        kubectl_get | kubectl_describe | kubectl_delete | kubectl_label | kubectl_edit | kubectl_patch |\
        kubectl_annotate | kubectl_expose | kubectl_scale | kubectl_autoscale | kubectl_taint | kubectl_rollout_*)
            __kubectl_get_resource
            return
            ;;
        kubectl_logs | kubectl_attach)
            __kubectl_require_pod_and_container
            return
            ;;
        kubectl_exec | kubectl_port-forward | kubectl_top_pod)
            __kubectl_get_resource_pod
            return
            ;;
        kubectl_rolling-update)
            __kubectl_get_resource_rc
            return
            ;;
        kubectl_cordon | kubectl_uncordon | kubectl_drain | kubectl_top_node)
            __kubectl_get_resource_node
            return
            ;;
        kubectl_config_use-context | kubectl_config_rename-context)
            __kubectl_config_get_contexts
            return
            ;;
        kubectl_config_delete-cluster)
            __kubectl_config_get_clusters
            return
            ;;
        *)
            ;;
    esac
}

_kubectl_alpha_diff()
{
    last_command="kubectl_alpha_diff"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--filename=")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename=")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_flag+=("--filename=")
    must_have_one_flag+=("-f")
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_alpha()
{
    last_command="kubectl_alpha"
    commands=()
    commands+=("diff")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_annotate()
{
    last_command="kubectl_annotate"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--all")
    local_nonpersistent_flags+=("--all")
    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--filename=")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename=")
    flags+=("--include-uninitialized")
    local_nonpersistent_flags+=("--include-uninitialized")
    flags+=("--local")
    local_nonpersistent_flags+=("--local")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--overwrite")
    local_nonpersistent_flags+=("--overwrite")
    flags+=("--record")
    local_nonpersistent_flags+=("--record")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--resource-version=")
    local_nonpersistent_flags+=("--resource-version=")
    flags+=("--selector=")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--selector=")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    must_have_one_noun+=("certificatesigningrequest")
    must_have_one_noun+=("clusterrolebinding")
    must_have_one_noun+=("componentstatus")
    must_have_one_noun+=("configmap")
    must_have_one_noun+=("controllerrevision")
    must_have_one_noun+=("cronjob")
    must_have_one_noun+=("daemonset")
    must_have_one_noun+=("deployment")
    must_have_one_noun+=("endpoints")
    must_have_one_noun+=("event")
    must_have_one_noun+=("horizontalpodautoscaler")
    must_have_one_noun+=("ingress")
    must_have_one_noun+=("job")
    must_have_one_noun+=("namespace")
    must_have_one_noun+=("networkpolicy")
    must_have_one_noun+=("node")
    must_have_one_noun+=("persistentvolume")
    must_have_one_noun+=("persistentvolumeclaim")
    must_have_one_noun+=("pod")
    must_have_one_noun+=("poddisruptionbudget")
    must_have_one_noun+=("podsecuritypolicy")
    must_have_one_noun+=("podtemplate")
    must_have_one_noun+=("replicaset")
    must_have_one_noun+=("replicationcontroller")
    must_have_one_noun+=("rolebinding")
    must_have_one_noun+=("secret")
    must_have_one_noun+=("service")
    must_have_one_noun+=("serviceaccount")
    must_have_one_noun+=("statefulset")
    must_have_one_noun+=("status")
    must_have_one_noun+=("storageclass")
    noun_aliases=()
    noun_aliases+=("certificatesigningrequests")
    noun_aliases+=("clusterrolebindings")
    noun_aliases+=("cm")
    noun_aliases+=("componentstatuses")
    noun_aliases+=("configmaps")
    noun_aliases+=("controllerrevisions")
    noun_aliases+=("cronjobs")
    noun_aliases+=("cs")
    noun_aliases+=("csr")
    noun_aliases+=("daemonsets")
    noun_aliases+=("deploy")
    noun_aliases+=("deployments")
    noun_aliases+=("ds")
    noun_aliases+=("endpoints")
    noun_aliases+=("ep")
    noun_aliases+=("ev")
    noun_aliases+=("events")
    noun_aliases+=("horizontalpodautoscalers")
    noun_aliases+=("hpa")
    noun_aliases+=("hpa")
    noun_aliases+=("ing")
    noun_aliases+=("ingresses")
    noun_aliases+=("jobs")
    noun_aliases+=("namespaces")
    noun_aliases+=("netpol")
    noun_aliases+=("networkpolicies")
    noun_aliases+=("no")
    noun_aliases+=("nodes")
    noun_aliases+=("ns")
    noun_aliases+=("pdb")
    noun_aliases+=("persistentvolumeclaims")
    noun_aliases+=("persistentvolumes")
    noun_aliases+=("po")
    noun_aliases+=("poddisruptionbudgets")
    noun_aliases+=("pods")
    noun_aliases+=("podsecuritypolicies")
    noun_aliases+=("podtemplates")
    noun_aliases+=("pv")
    noun_aliases+=("pvc")
    noun_aliases+=("rc")
    noun_aliases+=("replicasets")
    noun_aliases+=("replicationcontrollers")
    noun_aliases+=("rolebindings")
    noun_aliases+=("rs")
    noun_aliases+=("sa")
    noun_aliases+=("secrets")
    noun_aliases+=("serviceaccounts")
    noun_aliases+=("services")
    noun_aliases+=("statefulsets")
    noun_aliases+=("statuses")
    noun_aliases+=("storageclasses")
    noun_aliases+=("svc")
}

_kubectl_api-versions()
{
    last_command="kubectl_api-versions"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_apply_edit-last-applied()
{
    last_command="kubectl_apply_edit-last-applied"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--filename=")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename=")
    flags+=("--include-uninitialized")
    local_nonpersistent_flags+=("--include-uninitialized")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--record")
    local_nonpersistent_flags+=("--record")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--windows-line-endings")
    local_nonpersistent_flags+=("--windows-line-endings")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    must_have_one_noun+=("certificatesigningrequest")
    must_have_one_noun+=("clusterrolebinding")
    must_have_one_noun+=("componentstatus")
    must_have_one_noun+=("configmap")
    must_have_one_noun+=("controllerrevision")
    must_have_one_noun+=("cronjob")
    must_have_one_noun+=("daemonset")
    must_have_one_noun+=("deployment")
    must_have_one_noun+=("endpoints")
    must_have_one_noun+=("event")
    must_have_one_noun+=("horizontalpodautoscaler")
    must_have_one_noun+=("ingress")
    must_have_one_noun+=("job")
    must_have_one_noun+=("namespace")
    must_have_one_noun+=("networkpolicy")
    must_have_one_noun+=("node")
    must_have_one_noun+=("persistentvolume")
    must_have_one_noun+=("persistentvolumeclaim")
    must_have_one_noun+=("pod")
    must_have_one_noun+=("poddisruptionbudget")
    must_have_one_noun+=("podsecuritypolicy")
    must_have_one_noun+=("podtemplate")
    must_have_one_noun+=("replicaset")
    must_have_one_noun+=("replicationcontroller")
    must_have_one_noun+=("rolebinding")
    must_have_one_noun+=("secret")
    must_have_one_noun+=("service")
    must_have_one_noun+=("serviceaccount")
    must_have_one_noun+=("statefulset")
    must_have_one_noun+=("status")
    must_have_one_noun+=("storageclass")
    noun_aliases=()
    noun_aliases+=("certificatesigningrequests")
    noun_aliases+=("clusterrolebindings")
    noun_aliases+=("cm")
    noun_aliases+=("componentstatuses")
    noun_aliases+=("configmaps")
    noun_aliases+=("controllerrevisions")
    noun_aliases+=("cronjobs")
    noun_aliases+=("cs")
    noun_aliases+=("csr")
    noun_aliases+=("daemonsets")
    noun_aliases+=("deploy")
    noun_aliases+=("deployments")
    noun_aliases+=("ds")
    noun_aliases+=("endpoints")
    noun_aliases+=("ep")
    noun_aliases+=("ev")
    noun_aliases+=("events")
    noun_aliases+=("horizontalpodautoscalers")
    noun_aliases+=("hpa")
    noun_aliases+=("hpa")
    noun_aliases+=("ing")
    noun_aliases+=("ingresses")
    noun_aliases+=("jobs")
    noun_aliases+=("namespaces")
    noun_aliases+=("netpol")
    noun_aliases+=("networkpolicies")
    noun_aliases+=("no")
    noun_aliases+=("nodes")
    noun_aliases+=("ns")
    noun_aliases+=("pdb")
    noun_aliases+=("persistentvolumeclaims")
    noun_aliases+=("persistentvolumes")
    noun_aliases+=("po")
    noun_aliases+=("poddisruptionbudgets")
    noun_aliases+=("pods")
    noun_aliases+=("podsecuritypolicies")
    noun_aliases+=("podtemplates")
    noun_aliases+=("pv")
    noun_aliases+=("pvc")
    noun_aliases+=("rc")
    noun_aliases+=("replicasets")
    noun_aliases+=("replicationcontrollers")
    noun_aliases+=("rolebindings")
    noun_aliases+=("rs")
    noun_aliases+=("sa")
    noun_aliases+=("secrets")
    noun_aliases+=("serviceaccounts")
    noun_aliases+=("services")
    noun_aliases+=("statefulsets")
    noun_aliases+=("statuses")
    noun_aliases+=("storageclasses")
    noun_aliases+=("svc")
}

_kubectl_apply_set-last-applied()
{
    last_command="kubectl_apply_set-last-applied"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--create-annotation")
    local_nonpersistent_flags+=("--create-annotation")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--filename=")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename=")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--record")
    local_nonpersistent_flags+=("--record")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_apply_view-last-applied()
{
    last_command="kubectl_apply_view-last-applied"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--all")
    local_nonpersistent_flags+=("--all")
    flags+=("--filename=")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename=")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--selector=")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--selector=")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_apply()
{
    last_command="kubectl_apply"
    commands=()
    commands+=("edit-last-applied")
    commands+=("set-last-applied")
    commands+=("view-last-applied")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--all")
    local_nonpersistent_flags+=("--all")
    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--cascade")
    local_nonpersistent_flags+=("--cascade")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--filename=")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename=")
    flags+=("--force")
    local_nonpersistent_flags+=("--force")
    flags+=("--grace-period=")
    local_nonpersistent_flags+=("--grace-period=")
    flags+=("--include-uninitialized")
    local_nonpersistent_flags+=("--include-uninitialized")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--openapi-patch")
    local_nonpersistent_flags+=("--openapi-patch")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--overwrite")
    local_nonpersistent_flags+=("--overwrite")
    flags+=("--prune")
    local_nonpersistent_flags+=("--prune")
    flags+=("--prune-whitelist=")
    local_nonpersistent_flags+=("--prune-whitelist=")
    flags+=("--record")
    local_nonpersistent_flags+=("--record")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--selector=")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--selector=")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--timeout=")
    local_nonpersistent_flags+=("--timeout=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_flag+=("--filename=")
    must_have_one_flag+=("-f")
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_attach()
{
    last_command="kubectl_attach"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--container=")
    two_word_flags+=("-c")
    local_nonpersistent_flags+=("--container=")
    flags+=("--pod-running-timeout=")
    local_nonpersistent_flags+=("--pod-running-timeout=")
    flags+=("--stdin")
    flags+=("-i")
    local_nonpersistent_flags+=("--stdin")
    flags+=("--tty")
    flags+=("-t")
    local_nonpersistent_flags+=("--tty")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_auth_can-i()
{
    last_command="kubectl_auth_can-i"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--all-namespaces")
    local_nonpersistent_flags+=("--all-namespaces")
    flags+=("--quiet")
    flags+=("-q")
    local_nonpersistent_flags+=("--quiet")
    flags+=("--subresource=")
    local_nonpersistent_flags+=("--subresource=")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_auth_reconcile()
{
    last_command="kubectl_auth_reconcile"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--filename=")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename=")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_flag+=("--filename=")
    must_have_one_flag+=("-f")
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_auth()
{
    last_command="kubectl_auth"
    commands=()
    commands+=("can-i")
    commands+=("reconcile")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_autoscale()
{
    last_command="kubectl_autoscale"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--cpu-percent=")
    local_nonpersistent_flags+=("--cpu-percent=")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--filename=")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename=")
    flags+=("--generator=")
    local_nonpersistent_flags+=("--generator=")
    flags+=("--max=")
    local_nonpersistent_flags+=("--max=")
    flags+=("--min=")
    local_nonpersistent_flags+=("--min=")
    flags+=("--name=")
    local_nonpersistent_flags+=("--name=")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--record")
    local_nonpersistent_flags+=("--record")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_flag+=("--max=")
    must_have_one_noun=()
    must_have_one_noun+=("deployment")
    must_have_one_noun+=("replicaset")
    must_have_one_noun+=("replicationcontroller")
    noun_aliases=()
    noun_aliases+=("deploy")
    noun_aliases+=("deployments")
    noun_aliases+=("rc")
    noun_aliases+=("replicasets")
    noun_aliases+=("replicationcontrollers")
    noun_aliases+=("rs")
}

_kubectl_certificate_approve()
{
    last_command="kubectl_certificate_approve"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--filename=")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename=")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_certificate_deny()
{
    last_command="kubectl_certificate_deny"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--filename=")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename=")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_certificate()
{
    last_command="kubectl_certificate"
    commands=()
    commands+=("approve")
    commands+=("deny")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_cluster-info_dump()
{
    last_command="kubectl_cluster-info_dump"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--all-namespaces")
    local_nonpersistent_flags+=("--all-namespaces")
    flags+=("--namespaces=")
    local_nonpersistent_flags+=("--namespaces=")
    flags+=("--output-directory=")
    local_nonpersistent_flags+=("--output-directory=")
    flags+=("--pod-running-timeout=")
    local_nonpersistent_flags+=("--pod-running-timeout=")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_cluster-info()
{
    last_command="kubectl_cluster-info"
    commands=()
    commands+=("dump")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_completion()
{
    last_command="kubectl_completion"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--help")
    flags+=("-h")
    local_nonpersistent_flags+=("--help")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    must_have_one_noun+=("bash")
    must_have_one_noun+=("zsh")
    noun_aliases=()
}

_kubectl_config_current-context()
{
    last_command="kubectl_config_current-context"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_config_delete-cluster()
{
    last_command="kubectl_config_delete-cluster"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_config_delete-context()
{
    last_command="kubectl_config_delete-context"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_config_get-clusters()
{
    last_command="kubectl_config_get-clusters"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_config_get-contexts()
{
    last_command="kubectl_config_get-contexts"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_config_rename-context()
{
    last_command="kubectl_config_rename-context"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_config_set()
{
    last_command="kubectl_config_set"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--set-raw-bytes")
    local_nonpersistent_flags+=("--set-raw-bytes")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_config_set-cluster()
{
    last_command="kubectl_config_set-cluster"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--embed-certs")
    local_nonpersistent_flags+=("--embed-certs")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_config_set-context()
{
    last_command="kubectl_config_set-context"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_config_set-credentials()
{
    last_command="kubectl_config_set-credentials"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--auth-provider=")
    local_nonpersistent_flags+=("--auth-provider=")
    flags+=("--auth-provider-arg=")
    local_nonpersistent_flags+=("--auth-provider-arg=")
    flags+=("--embed-certs")
    local_nonpersistent_flags+=("--embed-certs")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_config_unset()
{
    last_command="kubectl_config_unset"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_config_use-context()
{
    last_command="kubectl_config_use-context"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_config_view()
{
    last_command="kubectl_config_view"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--flatten")
    local_nonpersistent_flags+=("--flatten")
    flags+=("--merge")
    local_nonpersistent_flags+=("--merge")
    flags+=("--minify")
    local_nonpersistent_flags+=("--minify")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--raw")
    local_nonpersistent_flags+=("--raw")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_config()
{
    last_command="kubectl_config"
    commands=()
    commands+=("current-context")
    commands+=("delete-cluster")
    commands+=("delete-context")
    commands+=("get-clusters")
    commands+=("get-contexts")
    commands+=("rename-context")
    commands+=("set")
    commands+=("set-cluster")
    commands+=("set-context")
    commands+=("set-credentials")
    commands+=("unset")
    commands+=("use-context")
    commands+=("view")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_convert()
{
    last_command="kubectl_convert"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--filename=")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename=")
    flags+=("--local")
    local_nonpersistent_flags+=("--local")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--output-version=")
    local_nonpersistent_flags+=("--output-version=")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_flag+=("--filename=")
    must_have_one_flag+=("-f")
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_cordon()
{
    last_command="kubectl_cordon"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--selector=")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--selector=")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_cp()
{
    last_command="kubectl_cp"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--container=")
    two_word_flags+=("-c")
    local_nonpersistent_flags+=("--container=")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_clusterrole()
{
    last_command="kubectl_create_clusterrole"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--non-resource-url=")
    local_nonpersistent_flags+=("--non-resource-url=")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--resource=")
    local_nonpersistent_flags+=("--resource=")
    flags+=("--resource-name=")
    local_nonpersistent_flags+=("--resource-name=")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--verb=")
    local_nonpersistent_flags+=("--verb=")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_clusterrolebinding()
{
    last_command="kubectl_create_clusterrolebinding"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--clusterrole=")
    flags_with_completion+=("--clusterrole")
    flags_completion+=("__kubectl_get_resource_clusterrole")
    local_nonpersistent_flags+=("--clusterrole=")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--generator=")
    local_nonpersistent_flags+=("--generator=")
    flags+=("--group=")
    local_nonpersistent_flags+=("--group=")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--serviceaccount=")
    local_nonpersistent_flags+=("--serviceaccount=")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_configmap()
{
    last_command="kubectl_create_configmap"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--append-hash")
    local_nonpersistent_flags+=("--append-hash")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--from-env-file=")
    local_nonpersistent_flags+=("--from-env-file=")
    flags+=("--from-file=")
    local_nonpersistent_flags+=("--from-file=")
    flags+=("--from-literal=")
    local_nonpersistent_flags+=("--from-literal=")
    flags+=("--generator=")
    local_nonpersistent_flags+=("--generator=")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_deployment()
{
    last_command="kubectl_create_deployment"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--generator=")
    local_nonpersistent_flags+=("--generator=")
    flags+=("--image=")
    local_nonpersistent_flags+=("--image=")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_flag+=("--image=")
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_job()
{
    last_command="kubectl_create_job"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--from=")
    local_nonpersistent_flags+=("--from=")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_namespace()
{
    last_command="kubectl_create_namespace"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--generator=")
    local_nonpersistent_flags+=("--generator=")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_poddisruptionbudget()
{
    last_command="kubectl_create_poddisruptionbudget"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--generator=")
    local_nonpersistent_flags+=("--generator=")
    flags+=("--max-unavailable=")
    local_nonpersistent_flags+=("--max-unavailable=")
    flags+=("--min-available=")
    local_nonpersistent_flags+=("--min-available=")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--selector=")
    local_nonpersistent_flags+=("--selector=")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_priorityclass()
{
    last_command="kubectl_create_priorityclass"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--description=")
    local_nonpersistent_flags+=("--description=")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--generator=")
    local_nonpersistent_flags+=("--generator=")
    flags+=("--global-default")
    local_nonpersistent_flags+=("--global-default")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--value=")
    local_nonpersistent_flags+=("--value=")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_quota()
{
    last_command="kubectl_create_quota"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--generator=")
    local_nonpersistent_flags+=("--generator=")
    flags+=("--hard=")
    local_nonpersistent_flags+=("--hard=")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--scopes=")
    local_nonpersistent_flags+=("--scopes=")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_role()
{
    last_command="kubectl_create_role"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--resource=")
    local_nonpersistent_flags+=("--resource=")
    flags+=("--resource-name=")
    local_nonpersistent_flags+=("--resource-name=")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--verb=")
    local_nonpersistent_flags+=("--verb=")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_rolebinding()
{
    last_command="kubectl_create_rolebinding"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--clusterrole=")
    local_nonpersistent_flags+=("--clusterrole=")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--generator=")
    local_nonpersistent_flags+=("--generator=")
    flags+=("--group=")
    local_nonpersistent_flags+=("--group=")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--role=")
    local_nonpersistent_flags+=("--role=")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--serviceaccount=")
    local_nonpersistent_flags+=("--serviceaccount=")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_secret_docker-registry()
{
    last_command="kubectl_create_secret_docker-registry"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--append-hash")
    local_nonpersistent_flags+=("--append-hash")
    flags+=("--docker-email=")
    local_nonpersistent_flags+=("--docker-email=")
    flags+=("--docker-password=")
    local_nonpersistent_flags+=("--docker-password=")
    flags+=("--docker-server=")
    local_nonpersistent_flags+=("--docker-server=")
    flags+=("--docker-username=")
    local_nonpersistent_flags+=("--docker-username=")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--generator=")
    local_nonpersistent_flags+=("--generator=")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_flag+=("--docker-password=")
    must_have_one_flag+=("--docker-username=")
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_secret_generic()
{
    last_command="kubectl_create_secret_generic"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--append-hash")
    local_nonpersistent_flags+=("--append-hash")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--from-env-file=")
    local_nonpersistent_flags+=("--from-env-file=")
    flags+=("--from-file=")
    local_nonpersistent_flags+=("--from-file=")
    flags+=("--from-literal=")
    local_nonpersistent_flags+=("--from-literal=")
    flags+=("--generator=")
    local_nonpersistent_flags+=("--generator=")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--type=")
    local_nonpersistent_flags+=("--type=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_secret_tls()
{
    last_command="kubectl_create_secret_tls"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--append-hash")
    local_nonpersistent_flags+=("--append-hash")
    flags+=("--cert=")
    local_nonpersistent_flags+=("--cert=")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--generator=")
    local_nonpersistent_flags+=("--generator=")
    flags+=("--key=")
    local_nonpersistent_flags+=("--key=")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_secret()
{
    last_command="kubectl_create_secret"
    commands=()
    commands+=("docker-registry")
    commands+=("generic")
    commands+=("tls")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_service_clusterip()
{
    last_command="kubectl_create_service_clusterip"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--clusterip=")
    local_nonpersistent_flags+=("--clusterip=")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--generator=")
    local_nonpersistent_flags+=("--generator=")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--tcp=")
    local_nonpersistent_flags+=("--tcp=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_service_externalname()
{
    last_command="kubectl_create_service_externalname"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--external-name=")
    local_nonpersistent_flags+=("--external-name=")
    flags+=("--generator=")
    local_nonpersistent_flags+=("--generator=")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--tcp=")
    local_nonpersistent_flags+=("--tcp=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_flag+=("--external-name=")
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_service_loadbalancer()
{
    last_command="kubectl_create_service_loadbalancer"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--generator=")
    local_nonpersistent_flags+=("--generator=")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--tcp=")
    local_nonpersistent_flags+=("--tcp=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_service_nodeport()
{
    last_command="kubectl_create_service_nodeport"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--generator=")
    local_nonpersistent_flags+=("--generator=")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--node-port=")
    local_nonpersistent_flags+=("--node-port=")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--tcp=")
    local_nonpersistent_flags+=("--tcp=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_service()
{
    last_command="kubectl_create_service"
    commands=()
    commands+=("clusterip")
    commands+=("externalname")
    commands+=("loadbalancer")
    commands+=("nodeport")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_serviceaccount()
{
    last_command="kubectl_create_serviceaccount"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--generator=")
    local_nonpersistent_flags+=("--generator=")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create()
{
    last_command="kubectl_create"
    commands=()
    commands+=("clusterrole")
    commands+=("clusterrolebinding")
    commands+=("configmap")
    commands+=("deployment")
    commands+=("job")
    commands+=("namespace")
    commands+=("poddisruptionbudget")
    commands+=("priorityclass")
    commands+=("quota")
    commands+=("role")
    commands+=("rolebinding")
    commands+=("secret")
    commands+=("service")
    commands+=("serviceaccount")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--edit")
    local_nonpersistent_flags+=("--edit")
    flags+=("--filename=")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename=")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--raw=")
    local_nonpersistent_flags+=("--raw=")
    flags+=("--record")
    local_nonpersistent_flags+=("--record")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--selector=")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--selector=")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--windows-line-endings")
    local_nonpersistent_flags+=("--windows-line-endings")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_flag+=("--filename=")
    must_have_one_flag+=("-f")
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_delete()
{
    last_command="kubectl_delete"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--all")
    local_nonpersistent_flags+=("--all")
    flags+=("--cascade")
    local_nonpersistent_flags+=("--cascade")
    flags+=("--filename=")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename=")
    flags+=("--force")
    local_nonpersistent_flags+=("--force")
    flags+=("--grace-period=")
    local_nonpersistent_flags+=("--grace-period=")
    flags+=("--ignore-not-found")
    local_nonpersistent_flags+=("--ignore-not-found")
    flags+=("--include-uninitialized")
    local_nonpersistent_flags+=("--include-uninitialized")
    flags+=("--now")
    local_nonpersistent_flags+=("--now")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--selector=")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--selector=")
    flags+=("--timeout=")
    local_nonpersistent_flags+=("--timeout=")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    must_have_one_noun+=("certificatesigningrequest")
    must_have_one_noun+=("clusterrolebinding")
    must_have_one_noun+=("componentstatus")
    must_have_one_noun+=("configmap")
    must_have_one_noun+=("controllerrevision")
    must_have_one_noun+=("cronjob")
    must_have_one_noun+=("daemonset")
    must_have_one_noun+=("deployment")
    must_have_one_noun+=("endpoints")
    must_have_one_noun+=("event")
    must_have_one_noun+=("horizontalpodautoscaler")
    must_have_one_noun+=("ingress")
    must_have_one_noun+=("job")
    must_have_one_noun+=("namespace")
    must_have_one_noun+=("networkpolicy")
    must_have_one_noun+=("node")
    must_have_one_noun+=("persistentvolume")
    must_have_one_noun+=("persistentvolumeclaim")
    must_have_one_noun+=("pod")
    must_have_one_noun+=("poddisruptionbudget")
    must_have_one_noun+=("podsecuritypolicy")
    must_have_one_noun+=("podtemplate")
    must_have_one_noun+=("replicaset")
    must_have_one_noun+=("replicationcontroller")
    must_have_one_noun+=("rolebinding")
    must_have_one_noun+=("secret")
    must_have_one_noun+=("service")
    must_have_one_noun+=("serviceaccount")
    must_have_one_noun+=("statefulset")
    must_have_one_noun+=("status")
    must_have_one_noun+=("storageclass")
    noun_aliases=()
    noun_aliases+=("certificatesigningrequests")
    noun_aliases+=("clusterrolebindings")
    noun_aliases+=("cm")
    noun_aliases+=("componentstatuses")
    noun_aliases+=("configmaps")
    noun_aliases+=("controllerrevisions")
    noun_aliases+=("cronjobs")
    noun_aliases+=("cs")
    noun_aliases+=("csr")
    noun_aliases+=("daemonsets")
    noun_aliases+=("deploy")
    noun_aliases+=("deployments")
    noun_aliases+=("ds")
    noun_aliases+=("endpoints")
    noun_aliases+=("ep")
    noun_aliases+=("ev")
    noun_aliases+=("events")
    noun_aliases+=("horizontalpodautoscalers")
    noun_aliases+=("hpa")
    noun_aliases+=("hpa")
    noun_aliases+=("ing")
    noun_aliases+=("ingresses")
    noun_aliases+=("jobs")
    noun_aliases+=("namespaces")
    noun_aliases+=("netpol")
    noun_aliases+=("networkpolicies")
    noun_aliases+=("no")
    noun_aliases+=("nodes")
    noun_aliases+=("ns")
    noun_aliases+=("pdb")
    noun_aliases+=("persistentvolumeclaims")
    noun_aliases+=("persistentvolumes")
    noun_aliases+=("po")
    noun_aliases+=("poddisruptionbudgets")
    noun_aliases+=("pods")
    noun_aliases+=("podsecuritypolicies")
    noun_aliases+=("podtemplates")
    noun_aliases+=("pv")
    noun_aliases+=("pvc")
    noun_aliases+=("rc")
    noun_aliases+=("replicasets")
    noun_aliases+=("replicationcontrollers")
    noun_aliases+=("rolebindings")
    noun_aliases+=("rs")
    noun_aliases+=("sa")
    noun_aliases+=("secrets")
    noun_aliases+=("serviceaccounts")
    noun_aliases+=("services")
    noun_aliases+=("statefulsets")
    noun_aliases+=("statuses")
    noun_aliases+=("storageclasses")
    noun_aliases+=("svc")
}

_kubectl_describe()
{
    last_command="kubectl_describe"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--all-namespaces")
    local_nonpersistent_flags+=("--all-namespaces")
    flags+=("--filename=")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename=")
    flags+=("--include-uninitialized")
    local_nonpersistent_flags+=("--include-uninitialized")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--selector=")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--selector=")
    flags+=("--show-events")
    local_nonpersistent_flags+=("--show-events")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    must_have_one_noun+=("certificatesigningrequest")
    must_have_one_noun+=("clusterrole")
    must_have_one_noun+=("clusterrolebinding")
    must_have_one_noun+=("configmap")
    must_have_one_noun+=("cronjob")
    must_have_one_noun+=("daemonset")
    must_have_one_noun+=("daemonset")
    must_have_one_noun+=("deployment")
    must_have_one_noun+=("deployment")
    must_have_one_noun+=("endpoints")
    must_have_one_noun+=("horizontalpodautoscaler")
    must_have_one_noun+=("ingress")
    must_have_one_noun+=("job")
    must_have_one_noun+=("limitrange")
    must_have_one_noun+=("namespace")
    must_have_one_noun+=("networkpolicy")
    must_have_one_noun+=("networkpolicy")
    must_have_one_noun+=("node")
    must_have_one_noun+=("persistentvolume")
    must_have_one_noun+=("persistentvolumeclaim")
    must_have_one_noun+=("pod")
    must_have_one_noun+=("poddisruptionbudget")
    must_have_one_noun+=("podsecuritypolicy")
    must_have_one_noun+=("priorityclass")
    must_have_one_noun+=("priorityclass")
    must_have_one_noun+=("replicaset")
    must_have_one_noun+=("replicaset")
    must_have_one_noun+=("replicationcontroller")
    must_have_one_noun+=("resourcequota")
    must_have_one_noun+=("role")
    must_have_one_noun+=("rolebinding")
    must_have_one_noun+=("secret")
    must_have_one_noun+=("service")
    must_have_one_noun+=("serviceaccount")
    must_have_one_noun+=("statefulset")
    must_have_one_noun+=("storageclass")
    noun_aliases=()
    noun_aliases+=("certificatesigningrequests")
    noun_aliases+=("clusterrolebindings")
    noun_aliases+=("clusterroles")
    noun_aliases+=("cm")
    noun_aliases+=("configmaps")
    noun_aliases+=("cronjobs")
    noun_aliases+=("csr")
    noun_aliases+=("daemonsets")
    noun_aliases+=("daemonsets")
    noun_aliases+=("deploy")
    noun_aliases+=("deployments")
    noun_aliases+=("deployments")
    noun_aliases+=("ds")
    noun_aliases+=("endpoints")
    noun_aliases+=("ep")
    noun_aliases+=("horizontalpodautoscalers")
    noun_aliases+=("hpa")
    noun_aliases+=("hpa")
    noun_aliases+=("ing")
    noun_aliases+=("ingresses")
    noun_aliases+=("jobs")
    noun_aliases+=("limitranges")
    noun_aliases+=("limits")
    noun_aliases+=("namespaces")
    noun_aliases+=("netpol")
    noun_aliases+=("networkpolicies")
    noun_aliases+=("networkpolicies")
    noun_aliases+=("no")
    noun_aliases+=("nodes")
    noun_aliases+=("ns")
    noun_aliases+=("pdb")
    noun_aliases+=("persistentvolumeclaims")
    noun_aliases+=("persistentvolumes")
    noun_aliases+=("po")
    noun_aliases+=("poddisruptionbudgets")
    noun_aliases+=("pods")
    noun_aliases+=("podsecuritypolicies")
    noun_aliases+=("priorityclasses")
    noun_aliases+=("priorityclasses")
    noun_aliases+=("pv")
    noun_aliases+=("pvc")
    noun_aliases+=("quota")
    noun_aliases+=("rc")
    noun_aliases+=("replicasets")
    noun_aliases+=("replicasets")
    noun_aliases+=("replicationcontrollers")
    noun_aliases+=("resourcequotas")
    noun_aliases+=("rolebindings")
    noun_aliases+=("roles")
    noun_aliases+=("rs")
    noun_aliases+=("sa")
    noun_aliases+=("secrets")
    noun_aliases+=("serviceaccounts")
    noun_aliases+=("services")
    noun_aliases+=("statefulsets")
    noun_aliases+=("storageclasses")
    noun_aliases+=("svc")
}

_kubectl_drain()
{
    last_command="kubectl_drain"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--delete-local-data")
    local_nonpersistent_flags+=("--delete-local-data")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--force")
    local_nonpersistent_flags+=("--force")
    flags+=("--grace-period=")
    local_nonpersistent_flags+=("--grace-period=")
    flags+=("--ignore-daemonsets")
    local_nonpersistent_flags+=("--ignore-daemonsets")
    flags+=("--pod-selector=")
    local_nonpersistent_flags+=("--pod-selector=")
    flags+=("--selector=")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--selector=")
    flags+=("--timeout=")
    local_nonpersistent_flags+=("--timeout=")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_edit()
{
    last_command="kubectl_edit"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--filename=")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename=")
    flags+=("--include-uninitialized")
    local_nonpersistent_flags+=("--include-uninitialized")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--output-patch")
    local_nonpersistent_flags+=("--output-patch")
    flags+=("--record")
    local_nonpersistent_flags+=("--record")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--windows-line-endings")
    local_nonpersistent_flags+=("--windows-line-endings")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    must_have_one_noun+=("certificatesigningrequest")
    must_have_one_noun+=("clusterrolebinding")
    must_have_one_noun+=("componentstatus")
    must_have_one_noun+=("configmap")
    must_have_one_noun+=("controllerrevision")
    must_have_one_noun+=("cronjob")
    must_have_one_noun+=("daemonset")
    must_have_one_noun+=("deployment")
    must_have_one_noun+=("endpoints")
    must_have_one_noun+=("event")
    must_have_one_noun+=("horizontalpodautoscaler")
    must_have_one_noun+=("ingress")
    must_have_one_noun+=("job")
    must_have_one_noun+=("namespace")
    must_have_one_noun+=("networkpolicy")
    must_have_one_noun+=("node")
    must_have_one_noun+=("persistentvolume")
    must_have_one_noun+=("persistentvolumeclaim")
    must_have_one_noun+=("pod")
    must_have_one_noun+=("poddisruptionbudget")
    must_have_one_noun+=("podsecuritypolicy")
    must_have_one_noun+=("podtemplate")
    must_have_one_noun+=("replicaset")
    must_have_one_noun+=("replicationcontroller")
    must_have_one_noun+=("rolebinding")
    must_have_one_noun+=("secret")
    must_have_one_noun+=("service")
    must_have_one_noun+=("serviceaccount")
    must_have_one_noun+=("statefulset")
    must_have_one_noun+=("status")
    must_have_one_noun+=("storageclass")
    noun_aliases=()
    noun_aliases+=("certificatesigningrequests")
    noun_aliases+=("clusterrolebindings")
    noun_aliases+=("cm")
    noun_aliases+=("componentstatuses")
    noun_aliases+=("configmaps")
    noun_aliases+=("controllerrevisions")
    noun_aliases+=("cronjobs")
    noun_aliases+=("cs")
    noun_aliases+=("csr")
    noun_aliases+=("daemonsets")
    noun_aliases+=("deploy")
    noun_aliases+=("deployments")
    noun_aliases+=("ds")
    noun_aliases+=("endpoints")
    noun_aliases+=("ep")
    noun_aliases+=("ev")
    noun_aliases+=("events")
    noun_aliases+=("horizontalpodautoscalers")
    noun_aliases+=("hpa")
    noun_aliases+=("hpa")
    noun_aliases+=("ing")
    noun_aliases+=("ingresses")
    noun_aliases+=("jobs")
    noun_aliases+=("namespaces")
    noun_aliases+=("netpol")
    noun_aliases+=("networkpolicies")
    noun_aliases+=("no")
    noun_aliases+=("nodes")
    noun_aliases+=("ns")
    noun_aliases+=("pdb")
    noun_aliases+=("persistentvolumeclaims")
    noun_aliases+=("persistentvolumes")
    noun_aliases+=("po")
    noun_aliases+=("poddisruptionbudgets")
    noun_aliases+=("pods")
    noun_aliases+=("podsecuritypolicies")
    noun_aliases+=("podtemplates")
    noun_aliases+=("pv")
    noun_aliases+=("pvc")
    noun_aliases+=("rc")
    noun_aliases+=("replicasets")
    noun_aliases+=("replicationcontrollers")
    noun_aliases+=("rolebindings")
    noun_aliases+=("rs")
    noun_aliases+=("sa")
    noun_aliases+=("secrets")
    noun_aliases+=("serviceaccounts")
    noun_aliases+=("services")
    noun_aliases+=("statefulsets")
    noun_aliases+=("statuses")
    noun_aliases+=("storageclasses")
    noun_aliases+=("svc")
}

_kubectl_exec()
{
    last_command="kubectl_exec"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--container=")
    two_word_flags+=("-c")
    local_nonpersistent_flags+=("--container=")
    flags+=("--pod=")
    two_word_flags+=("-p")
    local_nonpersistent_flags+=("--pod=")
    flags+=("--stdin")
    flags+=("-i")
    local_nonpersistent_flags+=("--stdin")
    flags+=("--tty")
    flags+=("-t")
    local_nonpersistent_flags+=("--tty")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_explain()
{
    last_command="kubectl_explain"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-version=")
    local_nonpersistent_flags+=("--api-version=")
    flags+=("--recursive")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_expose()
{
    last_command="kubectl_expose"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--cluster-ip=")
    local_nonpersistent_flags+=("--cluster-ip=")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--external-ip=")
    local_nonpersistent_flags+=("--external-ip=")
    flags+=("--filename=")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename=")
    flags+=("--generator=")
    local_nonpersistent_flags+=("--generator=")
    flags+=("--labels=")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--labels=")
    flags+=("--load-balancer-ip=")
    local_nonpersistent_flags+=("--load-balancer-ip=")
    flags+=("--name=")
    local_nonpersistent_flags+=("--name=")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--overrides=")
    local_nonpersistent_flags+=("--overrides=")
    flags+=("--port=")
    local_nonpersistent_flags+=("--port=")
    flags+=("--protocol=")
    local_nonpersistent_flags+=("--protocol=")
    flags+=("--record")
    local_nonpersistent_flags+=("--record")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--selector=")
    local_nonpersistent_flags+=("--selector=")
    flags+=("--session-affinity=")
    local_nonpersistent_flags+=("--session-affinity=")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--target-port=")
    local_nonpersistent_flags+=("--target-port=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--type=")
    local_nonpersistent_flags+=("--type=")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    must_have_one_noun+=("deployment")
    must_have_one_noun+=("pod")
    must_have_one_noun+=("replicaset")
    must_have_one_noun+=("replicationcontroller")
    must_have_one_noun+=("service")
    noun_aliases=()
    noun_aliases+=("deploy")
    noun_aliases+=("deployments")
    noun_aliases+=("po")
    noun_aliases+=("pods")
    noun_aliases+=("rc")
    noun_aliases+=("replicasets")
    noun_aliases+=("replicationcontrollers")
    noun_aliases+=("rs")
    noun_aliases+=("services")
    noun_aliases+=("svc")
}

_kubectl_get()
{
    last_command="kubectl_get"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--all-namespaces")
    local_nonpersistent_flags+=("--all-namespaces")
    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--chunk-size=")
    local_nonpersistent_flags+=("--chunk-size=")
    flags+=("--experimental-server-print")
    local_nonpersistent_flags+=("--experimental-server-print")
    flags+=("--export")
    local_nonpersistent_flags+=("--export")
    flags+=("--field-selector=")
    local_nonpersistent_flags+=("--field-selector=")
    flags+=("--filename=")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename=")
    flags+=("--ignore-not-found")
    local_nonpersistent_flags+=("--ignore-not-found")
    flags+=("--include-uninitialized")
    local_nonpersistent_flags+=("--include-uninitialized")
    flags+=("--label-columns=")
    two_word_flags+=("-L")
    local_nonpersistent_flags+=("--label-columns=")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--raw=")
    local_nonpersistent_flags+=("--raw=")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--selector=")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--selector=")
    flags+=("--show-kind")
    local_nonpersistent_flags+=("--show-kind")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--use-openapi-print-columns")
    local_nonpersistent_flags+=("--use-openapi-print-columns")
    flags+=("--watch")
    flags+=("-w")
    local_nonpersistent_flags+=("--watch")
    flags+=("--watch-only")
    local_nonpersistent_flags+=("--watch-only")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    must_have_one_noun+=("certificatesigningrequest")
    must_have_one_noun+=("clusterrolebinding")
    must_have_one_noun+=("componentstatus")
    must_have_one_noun+=("configmap")
    must_have_one_noun+=("controllerrevision")
    must_have_one_noun+=("cronjob")
    must_have_one_noun+=("daemonset")
    must_have_one_noun+=("deployment")
    must_have_one_noun+=("endpoints")
    must_have_one_noun+=("event")
    must_have_one_noun+=("horizontalpodautoscaler")
    must_have_one_noun+=("ingress")
    must_have_one_noun+=("job")
    must_have_one_noun+=("namespace")
    must_have_one_noun+=("networkpolicy")
    must_have_one_noun+=("node")
    must_have_one_noun+=("persistentvolume")
    must_have_one_noun+=("persistentvolumeclaim")
    must_have_one_noun+=("pod")
    must_have_one_noun+=("poddisruptionbudget")
    must_have_one_noun+=("podsecuritypolicy")
    must_have_one_noun+=("podtemplate")
    must_have_one_noun+=("replicaset")
    must_have_one_noun+=("replicationcontroller")
    must_have_one_noun+=("rolebinding")
    must_have_one_noun+=("secret")
    must_have_one_noun+=("service")
    must_have_one_noun+=("serviceaccount")
    must_have_one_noun+=("statefulset")
    must_have_one_noun+=("status")
    must_have_one_noun+=("storageclass")
    noun_aliases=()
    noun_aliases+=("certificatesigningrequests")
    noun_aliases+=("clusterrolebindings")
    noun_aliases+=("cm")
    noun_aliases+=("componentstatuses")
    noun_aliases+=("configmaps")
    noun_aliases+=("controllerrevisions")
    noun_aliases+=("cronjobs")
    noun_aliases+=("cs")
    noun_aliases+=("csr")
    noun_aliases+=("daemonsets")
    noun_aliases+=("deploy")
    noun_aliases+=("deployments")
    noun_aliases+=("ds")
    noun_aliases+=("endpoints")
    noun_aliases+=("ep")
    noun_aliases+=("ev")
    noun_aliases+=("events")
    noun_aliases+=("horizontalpodautoscalers")
    noun_aliases+=("hpa")
    noun_aliases+=("hpa")
    noun_aliases+=("ing")
    noun_aliases+=("ingresses")
    noun_aliases+=("jobs")
    noun_aliases+=("namespaces")
    noun_aliases+=("netpol")
    noun_aliases+=("networkpolicies")
    noun_aliases+=("no")
    noun_aliases+=("nodes")
    noun_aliases+=("ns")
    noun_aliases+=("pdb")
    noun_aliases+=("persistentvolumeclaims")
    noun_aliases+=("persistentvolumes")
    noun_aliases+=("po")
    noun_aliases+=("poddisruptionbudgets")
    noun_aliases+=("pods")
    noun_aliases+=("podsecuritypolicies")
    noun_aliases+=("podtemplates")
    noun_aliases+=("pv")
    noun_aliases+=("pvc")
    noun_aliases+=("rc")
    noun_aliases+=("replicasets")
    noun_aliases+=("replicationcontrollers")
    noun_aliases+=("rolebindings")
    noun_aliases+=("rs")
    noun_aliases+=("sa")
    noun_aliases+=("secrets")
    noun_aliases+=("serviceaccounts")
    noun_aliases+=("services")
    noun_aliases+=("statefulsets")
    noun_aliases+=("statuses")
    noun_aliases+=("storageclasses")
    noun_aliases+=("svc")
}

_kubectl_label()
{
    last_command="kubectl_label"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--all")
    local_nonpersistent_flags+=("--all")
    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--filename=")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename=")
    flags+=("--include-uninitialized")
    local_nonpersistent_flags+=("--include-uninitialized")
    flags+=("--list")
    local_nonpersistent_flags+=("--list")
    flags+=("--local")
    local_nonpersistent_flags+=("--local")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--overwrite")
    local_nonpersistent_flags+=("--overwrite")
    flags+=("--record")
    local_nonpersistent_flags+=("--record")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--resource-version=")
    local_nonpersistent_flags+=("--resource-version=")
    flags+=("--selector=")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--selector=")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    must_have_one_noun+=("certificatesigningrequest")
    must_have_one_noun+=("clusterrolebinding")
    must_have_one_noun+=("componentstatus")
    must_have_one_noun+=("configmap")
    must_have_one_noun+=("controllerrevision")
    must_have_one_noun+=("cronjob")
    must_have_one_noun+=("daemonset")
    must_have_one_noun+=("deployment")
    must_have_one_noun+=("endpoints")
    must_have_one_noun+=("event")
    must_have_one_noun+=("horizontalpodautoscaler")
    must_have_one_noun+=("ingress")
    must_have_one_noun+=("job")
    must_have_one_noun+=("namespace")
    must_have_one_noun+=("networkpolicy")
    must_have_one_noun+=("node")
    must_have_one_noun+=("persistentvolume")
    must_have_one_noun+=("persistentvolumeclaim")
    must_have_one_noun+=("pod")
    must_have_one_noun+=("poddisruptionbudget")
    must_have_one_noun+=("podsecuritypolicy")
    must_have_one_noun+=("podtemplate")
    must_have_one_noun+=("replicaset")
    must_have_one_noun+=("replicationcontroller")
    must_have_one_noun+=("rolebinding")
    must_have_one_noun+=("secret")
    must_have_one_noun+=("service")
    must_have_one_noun+=("serviceaccount")
    must_have_one_noun+=("statefulset")
    must_have_one_noun+=("status")
    must_have_one_noun+=("storageclass")
    noun_aliases=()
    noun_aliases+=("certificatesigningrequests")
    noun_aliases+=("clusterrolebindings")
    noun_aliases+=("cm")
    noun_aliases+=("componentstatuses")
    noun_aliases+=("configmaps")
    noun_aliases+=("controllerrevisions")
    noun_aliases+=("cronjobs")
    noun_aliases+=("cs")
    noun_aliases+=("csr")
    noun_aliases+=("daemonsets")
    noun_aliases+=("deploy")
    noun_aliases+=("deployments")
    noun_aliases+=("ds")
    noun_aliases+=("endpoints")
    noun_aliases+=("ep")
    noun_aliases+=("ev")
    noun_aliases+=("events")
    noun_aliases+=("horizontalpodautoscalers")
    noun_aliases+=("hpa")
    noun_aliases+=("hpa")
    noun_aliases+=("ing")
    noun_aliases+=("ingresses")
    noun_aliases+=("jobs")
    noun_aliases+=("namespaces")
    noun_aliases+=("netpol")
    noun_aliases+=("networkpolicies")
    noun_aliases+=("no")
    noun_aliases+=("nodes")
    noun_aliases+=("ns")
    noun_aliases+=("pdb")
    noun_aliases+=("persistentvolumeclaims")
    noun_aliases+=("persistentvolumes")
    noun_aliases+=("po")
    noun_aliases+=("poddisruptionbudgets")
    noun_aliases+=("pods")
    noun_aliases+=("podsecuritypolicies")
    noun_aliases+=("podtemplates")
    noun_aliases+=("pv")
    noun_aliases+=("pvc")
    noun_aliases+=("rc")
    noun_aliases+=("replicasets")
    noun_aliases+=("replicationcontrollers")
    noun_aliases+=("rolebindings")
    noun_aliases+=("rs")
    noun_aliases+=("sa")
    noun_aliases+=("secrets")
    noun_aliases+=("serviceaccounts")
    noun_aliases+=("services")
    noun_aliases+=("statefulsets")
    noun_aliases+=("statuses")
    noun_aliases+=("storageclasses")
    noun_aliases+=("svc")
}

_kubectl_logs()
{
    last_command="kubectl_logs"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--container=")
    two_word_flags+=("-c")
    local_nonpersistent_flags+=("--container=")
    flags+=("--follow")
    flags+=("-f")
    local_nonpersistent_flags+=("--follow")
    flags+=("--limit-bytes=")
    local_nonpersistent_flags+=("--limit-bytes=")
    flags+=("--pod-running-timeout=")
    local_nonpersistent_flags+=("--pod-running-timeout=")
    flags+=("--previous")
    flags+=("-p")
    local_nonpersistent_flags+=("--previous")
    flags+=("--selector=")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--selector=")
    flags+=("--since=")
    local_nonpersistent_flags+=("--since=")
    flags+=("--since-time=")
    local_nonpersistent_flags+=("--since-time=")
    flags+=("--tail=")
    local_nonpersistent_flags+=("--tail=")
    flags+=("--timestamps")
    local_nonpersistent_flags+=("--timestamps")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_options()
{
    last_command="kubectl_options"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_patch()
{
    last_command="kubectl_patch"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--filename=")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename=")
    flags+=("--local")
    local_nonpersistent_flags+=("--local")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--patch=")
    two_word_flags+=("-p")
    local_nonpersistent_flags+=("--patch=")
    flags+=("--record")
    local_nonpersistent_flags+=("--record")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--type=")
    local_nonpersistent_flags+=("--type=")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_flag+=("--patch=")
    must_have_one_flag+=("-p")
    must_have_one_noun=()
    must_have_one_noun+=("certificatesigningrequest")
    must_have_one_noun+=("clusterrolebinding")
    must_have_one_noun+=("componentstatus")
    must_have_one_noun+=("configmap")
    must_have_one_noun+=("controllerrevision")
    must_have_one_noun+=("cronjob")
    must_have_one_noun+=("daemonset")
    must_have_one_noun+=("deployment")
    must_have_one_noun+=("endpoints")
    must_have_one_noun+=("event")
    must_have_one_noun+=("horizontalpodautoscaler")
    must_have_one_noun+=("ingress")
    must_have_one_noun+=("job")
    must_have_one_noun+=("namespace")
    must_have_one_noun+=("networkpolicy")
    must_have_one_noun+=("node")
    must_have_one_noun+=("persistentvolume")
    must_have_one_noun+=("persistentvolumeclaim")
    must_have_one_noun+=("pod")
    must_have_one_noun+=("poddisruptionbudget")
    must_have_one_noun+=("podsecuritypolicy")
    must_have_one_noun+=("podtemplate")
    must_have_one_noun+=("replicaset")
    must_have_one_noun+=("replicationcontroller")
    must_have_one_noun+=("rolebinding")
    must_have_one_noun+=("secret")
    must_have_one_noun+=("service")
    must_have_one_noun+=("serviceaccount")
    must_have_one_noun+=("statefulset")
    must_have_one_noun+=("status")
    must_have_one_noun+=("storageclass")
    noun_aliases=()
    noun_aliases+=("certificatesigningrequests")
    noun_aliases+=("clusterrolebindings")
    noun_aliases+=("cm")
    noun_aliases+=("componentstatuses")
    noun_aliases+=("configmaps")
    noun_aliases+=("controllerrevisions")
    noun_aliases+=("cronjobs")
    noun_aliases+=("cs")
    noun_aliases+=("csr")
    noun_aliases+=("daemonsets")
    noun_aliases+=("deploy")
    noun_aliases+=("deployments")
    noun_aliases+=("ds")
    noun_aliases+=("endpoints")
    noun_aliases+=("ep")
    noun_aliases+=("ev")
    noun_aliases+=("events")
    noun_aliases+=("horizontalpodautoscalers")
    noun_aliases+=("hpa")
    noun_aliases+=("hpa")
    noun_aliases+=("ing")
    noun_aliases+=("ingresses")
    noun_aliases+=("jobs")
    noun_aliases+=("namespaces")
    noun_aliases+=("netpol")
    noun_aliases+=("networkpolicies")
    noun_aliases+=("no")
    noun_aliases+=("nodes")
    noun_aliases+=("ns")
    noun_aliases+=("pdb")
    noun_aliases+=("persistentvolumeclaims")
    noun_aliases+=("persistentvolumes")
    noun_aliases+=("po")
    noun_aliases+=("poddisruptionbudgets")
    noun_aliases+=("pods")
    noun_aliases+=("podsecuritypolicies")
    noun_aliases+=("podtemplates")
    noun_aliases+=("pv")
    noun_aliases+=("pvc")
    noun_aliases+=("rc")
    noun_aliases+=("replicasets")
    noun_aliases+=("replicationcontrollers")
    noun_aliases+=("rolebindings")
    noun_aliases+=("rs")
    noun_aliases+=("sa")
    noun_aliases+=("secrets")
    noun_aliases+=("serviceaccounts")
    noun_aliases+=("services")
    noun_aliases+=("statefulsets")
    noun_aliases+=("statuses")
    noun_aliases+=("storageclasses")
    noun_aliases+=("svc")
}

_kubectl_plugin()
{
    last_command="kubectl_plugin"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_port-forward()
{
    last_command="kubectl_port-forward"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--pod-running-timeout=")
    local_nonpersistent_flags+=("--pod-running-timeout=")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_proxy()
{
    last_command="kubectl_proxy"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--accept-hosts=")
    local_nonpersistent_flags+=("--accept-hosts=")
    flags+=("--accept-paths=")
    local_nonpersistent_flags+=("--accept-paths=")
    flags+=("--address=")
    local_nonpersistent_flags+=("--address=")
    flags+=("--api-prefix=")
    local_nonpersistent_flags+=("--api-prefix=")
    flags+=("--disable-filter")
    local_nonpersistent_flags+=("--disable-filter")
    flags+=("--port=")
    two_word_flags+=("-p")
    local_nonpersistent_flags+=("--port=")
    flags+=("--reject-methods=")
    local_nonpersistent_flags+=("--reject-methods=")
    flags+=("--reject-paths=")
    local_nonpersistent_flags+=("--reject-paths=")
    flags+=("--unix-socket=")
    two_word_flags+=("-u")
    local_nonpersistent_flags+=("--unix-socket=")
    flags+=("--www=")
    two_word_flags+=("-w")
    local_nonpersistent_flags+=("--www=")
    flags+=("--www-prefix=")
    two_word_flags+=("-P")
    local_nonpersistent_flags+=("--www-prefix=")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_replace()
{
    last_command="kubectl_replace"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--cascade")
    local_nonpersistent_flags+=("--cascade")
    flags+=("--filename=")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename=")
    flags+=("--force")
    local_nonpersistent_flags+=("--force")
    flags+=("--grace-period=")
    local_nonpersistent_flags+=("--grace-period=")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--record")
    local_nonpersistent_flags+=("--record")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--timeout=")
    local_nonpersistent_flags+=("--timeout=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_flag+=("--filename=")
    must_have_one_flag+=("-f")
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_rolling-update()
{
    last_command="kubectl_rolling-update"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--container=")
    local_nonpersistent_flags+=("--container=")
    flags+=("--deployment-label-key=")
    local_nonpersistent_flags+=("--deployment-label-key=")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--filename=")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename=")
    flags+=("--image=")
    local_nonpersistent_flags+=("--image=")
    flags+=("--image-pull-policy=")
    local_nonpersistent_flags+=("--image-pull-policy=")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--poll-interval=")
    local_nonpersistent_flags+=("--poll-interval=")
    flags+=("--rollback")
    local_nonpersistent_flags+=("--rollback")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--timeout=")
    local_nonpersistent_flags+=("--timeout=")
    flags+=("--update-period=")
    local_nonpersistent_flags+=("--update-period=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_rollout_history()
{
    last_command="kubectl_rollout_history"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--filename=")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename=")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--revision=")
    local_nonpersistent_flags+=("--revision=")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    must_have_one_noun+=("daemonset")
    must_have_one_noun+=("deployment")
    must_have_one_noun+=("statefulset")
    noun_aliases=()
    noun_aliases+=("daemonsets")
    noun_aliases+=("deploy")
    noun_aliases+=("deployments")
    noun_aliases+=("ds")
    noun_aliases+=("statefulsets")
}

_kubectl_rollout_pause()
{
    last_command="kubectl_rollout_pause"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--filename=")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename=")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    must_have_one_noun+=("deployment")
    noun_aliases=()
    noun_aliases+=("deploy")
    noun_aliases+=("deployments")
}

_kubectl_rollout_resume()
{
    last_command="kubectl_rollout_resume"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--filename=")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename=")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    must_have_one_noun+=("deployment")
    noun_aliases=()
    noun_aliases+=("deploy")
    noun_aliases+=("deployments")
}

_kubectl_rollout_status()
{
    last_command="kubectl_rollout_status"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--filename=")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename=")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--revision=")
    local_nonpersistent_flags+=("--revision=")
    flags+=("--watch")
    flags+=("-w")
    local_nonpersistent_flags+=("--watch")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    must_have_one_noun+=("daemonset")
    must_have_one_noun+=("deployment")
    must_have_one_noun+=("statefulset")
    noun_aliases=()
    noun_aliases+=("daemonsets")
    noun_aliases+=("deploy")
    noun_aliases+=("deployments")
    noun_aliases+=("ds")
    noun_aliases+=("statefulsets")
}

_kubectl_rollout_undo()
{
    last_command="kubectl_rollout_undo"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--filename=")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename=")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--to-revision=")
    local_nonpersistent_flags+=("--to-revision=")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    must_have_one_noun+=("daemonset")
    must_have_one_noun+=("deployment")
    must_have_one_noun+=("statefulset")
    noun_aliases=()
    noun_aliases+=("daemonsets")
    noun_aliases+=("deploy")
    noun_aliases+=("deployments")
    noun_aliases+=("ds")
    noun_aliases+=("statefulsets")
}

_kubectl_rollout()
{
    last_command="kubectl_rollout"
    commands=()
    commands+=("history")
    commands+=("pause")
    commands+=("resume")
    commands+=("status")
    commands+=("undo")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_run()
{
    last_command="kubectl_run"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--attach")
    local_nonpersistent_flags+=("--attach")
    flags+=("--command")
    local_nonpersistent_flags+=("--command")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--env=")
    local_nonpersistent_flags+=("--env=")
    flags+=("--expose")
    local_nonpersistent_flags+=("--expose")
    flags+=("--generator=")
    local_nonpersistent_flags+=("--generator=")
    flags+=("--hostport=")
    local_nonpersistent_flags+=("--hostport=")
    flags+=("--image=")
    local_nonpersistent_flags+=("--image=")
    flags+=("--image-pull-policy=")
    local_nonpersistent_flags+=("--image-pull-policy=")
    flags+=("--labels=")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--labels=")
    flags+=("--leave-stdin-open")
    local_nonpersistent_flags+=("--leave-stdin-open")
    flags+=("--limits=")
    local_nonpersistent_flags+=("--limits=")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--overrides=")
    local_nonpersistent_flags+=("--overrides=")
    flags+=("--pod-running-timeout=")
    local_nonpersistent_flags+=("--pod-running-timeout=")
    flags+=("--port=")
    local_nonpersistent_flags+=("--port=")
    flags+=("--quiet")
    local_nonpersistent_flags+=("--quiet")
    flags+=("--record")
    local_nonpersistent_flags+=("--record")
    flags+=("--replicas=")
    two_word_flags+=("-r")
    local_nonpersistent_flags+=("--replicas=")
    flags+=("--requests=")
    local_nonpersistent_flags+=("--requests=")
    flags+=("--restart=")
    local_nonpersistent_flags+=("--restart=")
    flags+=("--rm")
    local_nonpersistent_flags+=("--rm")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--schedule=")
    local_nonpersistent_flags+=("--schedule=")
    flags+=("--service-generator=")
    local_nonpersistent_flags+=("--service-generator=")
    flags+=("--service-overrides=")
    local_nonpersistent_flags+=("--service-overrides=")
    flags+=("--serviceaccount=")
    local_nonpersistent_flags+=("--serviceaccount=")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--stdin")
    flags+=("-i")
    local_nonpersistent_flags+=("--stdin")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--tty")
    flags+=("-t")
    local_nonpersistent_flags+=("--tty")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_flag+=("--image=")
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_scale()
{
    last_command="kubectl_scale"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--all")
    local_nonpersistent_flags+=("--all")
    flags+=("--current-replicas=")
    local_nonpersistent_flags+=("--current-replicas=")
    flags+=("--filename=")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename=")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--record")
    local_nonpersistent_flags+=("--record")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--replicas=")
    local_nonpersistent_flags+=("--replicas=")
    flags+=("--resource-version=")
    local_nonpersistent_flags+=("--resource-version=")
    flags+=("--selector=")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--selector=")
    flags+=("--timeout=")
    local_nonpersistent_flags+=("--timeout=")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_flag+=("--replicas=")
    must_have_one_noun=()
    must_have_one_noun+=("deployment")
    must_have_one_noun+=("job")
    must_have_one_noun+=("replicaset")
    must_have_one_noun+=("replicationcontroller")
    must_have_one_noun+=("statefulset")
    noun_aliases=()
    noun_aliases+=("deploy")
    noun_aliases+=("deployments")
    noun_aliases+=("jobs")
    noun_aliases+=("rc")
    noun_aliases+=("replicasets")
    noun_aliases+=("replicationcontrollers")
    noun_aliases+=("rs")
    noun_aliases+=("statefulsets")
}

_kubectl_set_env()
{
    last_command="kubectl_set_env"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--all")
    local_nonpersistent_flags+=("--all")
    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--containers=")
    two_word_flags+=("-c")
    local_nonpersistent_flags+=("--containers=")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--env=")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--env=")
    flags+=("--filename=")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename=")
    flags+=("--from=")
    local_nonpersistent_flags+=("--from=")
    flags+=("--list")
    local_nonpersistent_flags+=("--list")
    flags+=("--local")
    local_nonpersistent_flags+=("--local")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--overwrite")
    local_nonpersistent_flags+=("--overwrite")
    flags+=("--prefix=")
    local_nonpersistent_flags+=("--prefix=")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--resolve")
    local_nonpersistent_flags+=("--resolve")
    flags+=("--selector=")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--selector=")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_set_image()
{
    last_command="kubectl_set_image"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--all")
    local_nonpersistent_flags+=("--all")
    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--filename=")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename=")
    flags+=("--include-uninitialized")
    local_nonpersistent_flags+=("--include-uninitialized")
    flags+=("--local")
    local_nonpersistent_flags+=("--local")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--record")
    local_nonpersistent_flags+=("--record")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--selector=")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--selector=")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_set_resources()
{
    last_command="kubectl_set_resources"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--all")
    local_nonpersistent_flags+=("--all")
    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--containers=")
    two_word_flags+=("-c")
    local_nonpersistent_flags+=("--containers=")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--filename=")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename=")
    flags+=("--include-uninitialized")
    local_nonpersistent_flags+=("--include-uninitialized")
    flags+=("--limits=")
    local_nonpersistent_flags+=("--limits=")
    flags+=("--local")
    local_nonpersistent_flags+=("--local")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--record")
    local_nonpersistent_flags+=("--record")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--requests=")
    local_nonpersistent_flags+=("--requests=")
    flags+=("--selector=")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--selector=")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_set_selector()
{
    last_command="kubectl_set_selector"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--all")
    local_nonpersistent_flags+=("--all")
    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--filename=")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename=")
    flags+=("--include-uninitialized")
    local_nonpersistent_flags+=("--include-uninitialized")
    flags+=("--local")
    local_nonpersistent_flags+=("--local")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--record")
    local_nonpersistent_flags+=("--record")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--resource-version=")
    local_nonpersistent_flags+=("--resource-version=")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_set_serviceaccount()
{
    last_command="kubectl_set_serviceaccount"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--all")
    local_nonpersistent_flags+=("--all")
    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--filename=")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename=")
    flags+=("--include-uninitialized")
    local_nonpersistent_flags+=("--include-uninitialized")
    flags+=("--local")
    local_nonpersistent_flags+=("--local")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--record")
    local_nonpersistent_flags+=("--record")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_set_subject()
{
    last_command="kubectl_set_subject"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--all")
    local_nonpersistent_flags+=("--all")
    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--filename=")
    flags_with_completion+=("--filename")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__kubectl_handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename=")
    flags+=("--group=")
    local_nonpersistent_flags+=("--group=")
    flags+=("--include-uninitialized")
    local_nonpersistent_flags+=("--include-uninitialized")
    flags+=("--local")
    local_nonpersistent_flags+=("--local")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--selector=")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--selector=")
    flags+=("--serviceaccount=")
    local_nonpersistent_flags+=("--serviceaccount=")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_set()
{
    last_command="kubectl_set"
    commands=()
    commands+=("env")
    commands+=("image")
    commands+=("resources")
    commands+=("selector")
    commands+=("serviceaccount")
    commands+=("subject")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_taint()
{
    last_command="kubectl_taint"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--all")
    local_nonpersistent_flags+=("--all")
    flags+=("--allow-missing-template-keys")
    local_nonpersistent_flags+=("--allow-missing-template-keys")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--overwrite")
    local_nonpersistent_flags+=("--overwrite")
    flags+=("--selector=")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--selector=")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    must_have_one_noun+=("node")
    noun_aliases=()
    noun_aliases+=("no")
    noun_aliases+=("nodes")
}

_kubectl_top_node()
{
    last_command="kubectl_top_node"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--heapster-namespace=")
    local_nonpersistent_flags+=("--heapster-namespace=")
    flags+=("--heapster-port=")
    local_nonpersistent_flags+=("--heapster-port=")
    flags+=("--heapster-scheme=")
    local_nonpersistent_flags+=("--heapster-scheme=")
    flags+=("--heapster-service=")
    local_nonpersistent_flags+=("--heapster-service=")
    flags+=("--selector=")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--selector=")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_top_pod()
{
    last_command="kubectl_top_pod"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--all-namespaces")
    local_nonpersistent_flags+=("--all-namespaces")
    flags+=("--containers")
    local_nonpersistent_flags+=("--containers")
    flags+=("--heapster-namespace=")
    local_nonpersistent_flags+=("--heapster-namespace=")
    flags+=("--heapster-port=")
    local_nonpersistent_flags+=("--heapster-port=")
    flags+=("--heapster-scheme=")
    local_nonpersistent_flags+=("--heapster-scheme=")
    flags+=("--heapster-service=")
    local_nonpersistent_flags+=("--heapster-service=")
    flags+=("--selector=")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--selector=")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_top()
{
    last_command="kubectl_top"
    commands=()
    commands+=("node")
    commands+=("pod")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_uncordon()
{
    last_command="kubectl_uncordon"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--selector=")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--selector=")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_version()
{
    last_command="kubectl_version"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--client")
    flags+=("-c")
    local_nonpersistent_flags+=("--client")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--short")
    local_nonpersistent_flags+=("--short")
    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_root_command()
{
    last_command="kubectl"
    commands=()
    commands+=("alpha")
    commands+=("annotate")
    commands+=("api-versions")
    commands+=("apply")
    commands+=("attach")
    commands+=("auth")
    commands+=("autoscale")
    commands+=("certificate")
    commands+=("cluster-info")
    commands+=("completion")
    commands+=("config")
    commands+=("convert")
    commands+=("cordon")
    commands+=("cp")
    commands+=("create")
    commands+=("delete")
    commands+=("describe")
    commands+=("drain")
    commands+=("edit")
    commands+=("exec")
    commands+=("explain")
    commands+=("expose")
    commands+=("get")
    commands+=("label")
    commands+=("logs")
    commands+=("options")
    commands+=("patch")
    commands+=("plugin")
    commands+=("port-forward")
    commands+=("proxy")
    commands+=("replace")
    commands+=("rolling-update")
    commands+=("rollout")
    commands+=("run")
    commands+=("scale")
    commands+=("set")
    commands+=("taint")
    commands+=("top")
    commands+=("uncordon")
    commands+=("version")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--alsologtostderr")
    flags+=("--as=")
    flags+=("--as-group=")
    flags+=("--cache-dir=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags_with_completion+=("--cluster")
    flags_completion+=("__kubectl_config_get_clusters")
    flags+=("--context=")
    flags_with_completion+=("--context")
    flags_completion+=("__kubectl_config_get_contexts")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--logtostderr")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_resource_namespace")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_resource_namespace")
    flags+=("--password=")
    flags+=("--request-timeout=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--token=")
    flags+=("--user=")
    flags_with_completion+=("--user")
    flags_completion+=("__kubectl_config_get_users")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

__start_kubectl()
{
    local cur prev words cword
    declare -A flaghash 2>/dev/null || :
    if declare -F _init_completion >/dev/null 2>&1; then
        _init_completion -s || return
    else
        __kubectl_init_completion -n "=" || return
    fi

    local c=0
    local flags=()
    local two_word_flags=()
    local local_nonpersistent_flags=()
    local flags_with_completion=()
    local flags_completion=()
    local commands=("kubectl")
    local must_have_one_flag=()
    local must_have_one_noun=()
    local last_command
    local nouns=()

    __kubectl_handle_word
}

if [[ $(type -t compopt) = "builtin" ]]; then
    complete -o default -F __start_kubectl kubectl
else
    complete -o default -o nospace -F __start_kubectl kubectl
fi

# ex: ts=4 sw=4 et filetype=sh
