######################################################################
# The Falcon Programming Language.
# CMake module for find_package( falcon )
######################################################################
#
# Generates the following variables.
# - Falcon_INCLUDE_DIRS: List of directories for inclusion
# - Falcon_LIBRARY_DIRS: List of directories for library search
# - Falcon_BINARY_DIR: Where the falcon binary resides
# - Falcon_MODULES_DIR: Default installation directory for modules
#
# - Falcon_LIBRARIES: Libraries needed for link.
#
######################################################################
# Notes: set -DMOD_INSTALL=<target> to have modules installed in a
#        non-default directory.
######################################################################

get_filename_component(SELF_DIR ${CMAKE_CURRENT_LIST_FILE} PATH)

# OPTIONAL for the case this project is built as a subproject.
if(NOT Falcon_IN_CORE_SOURCETREE)
  include(${SELF_DIR}/falcon-core-targets.cmake OPTIONAL)
endif()

set(Falcon_APP_DIR "lib/falcon/apps")
set(Falcon_MOD_DIR "lib/falcon")
set(Falcon_BIN_DIR "bin")
set(Falcon_LIB_DIR "lib")
set(Falcon_MAN_DIR "share/man/man1")
set(Falcon_INC_DIR "include/falcon0.9.6")
set(Falcon_SHARE_DIR "share/falcon0.9.6")
set(Falcon_DOC_DIR "share/doc/falcon0.9.6")
set(Falcon_CMAKE_DIR "share/falcon0.9.6/cmake")

# Allow flexible installation location to perform out-of-install module builds
# Also, on WIN32, the target installation directory is decided during package install.
string(REGEX REPLACE "[^/]+" ".." _ups "${Falcon_CMAKE_DIR}")
get_filename_component(Falcon_PREFIX ${SELF_DIR}/${_ups} ABSOLUTE)

set(Falcon_INCLUDE_DIR "${Falcon_PREFIX}/include/falcon0.9.6")
set(Falcon_BINARY_DIR "${Falcon_PREFIX}/bin")
set(Falcon_LIBRARY_DIR "${Falcon_PREFIX}/lib")
set(Falcon_MODULES_DIR  "${Falcon_PREFIX}/lib/falcon")

# If defined, prefer the source directory include files over the installed ones
list(APPEND Falcon_INCLUDE_DIRS "${Falcon_INCLUDE_DIR}")

#Versioning
set( Falcon_VERSION "0.9.6.8" )
set( Falcon_VERSION_NAME "Chimera" )
#
set( Falcon_MAJOR_VERSION 0 )
set( Falcon_MINOR_VERSION 9 )
set( Falcon_SUBMINOR_VERSION 6 )
set( Falcon_PATCH_VERSION 8 )

if(NOT Falcon_IN_CORE_SOURCETREE)
  #Check if everything is in place.
  find_program(Falcon_EXECUTABLE falcon "${Falcon_BINARY_DIR}")
  find_program(Falcon_falpack_EXECUTABLE falpack "${Falcon_BINARY_DIR}")
  find_library(Falcon_ENGINE falcon_engine "${Falcon_LIBRARY_DIR}")
  find_file(Falcon_ENGINE_INC falcon/engine.h "${Falcon_INCLUDE_DIR}" )
endif()

# Prepare Falcon in development environment.
list(APPEND Falcon_INCLUDE_DIRS  ${Falcon_INCLUDE_DIR})
list(APPEND Falcon_LIBRARY_DIRS  ${Falcon_LIBRARY_DIR})

set(Falcon_LIBRARIES falcon_engine)

# for install(TARGETS .. ${FALCON_INSTALL_DESTINATIONS})
set(FALCON_INSTALL_DESTINATIONS
  RUNTIME DESTINATION ${Falcon_BIN_DIR}
  LIBRARY DESTINATION ${Falcon_LIB_DIR}
  ARCHIVE DESTINATION ${Falcon_ARCH_DIR}
)
# NOTE modules are installed via
#   install(FILES .. DESTINATION ${FALCON_MOD_DIR})
# since they are neither RUNTIME, LIBRARY nor ARCHIVE.

if(WIN32)
   #mingw requires -mthreads global option
   if(CMAKE_GENERATOR STREQUAL "MinGW Makefiles")
      list(APPEND CMAKE_EXE_LINKER_FLAGS -mthreads )
      list(APPEND CMAKE_SHARED_LINKER_FLAGS -mthreads )
      list(APPEND CMAKE_MODULE_LINKER_FLAGS -mthreads )
   endif()
endif(WIN32)


macro( falcon_define_module varname modname )
      set( ${varname} "${modname}_fm" )
endmacro()

function(falcon_install_module2 tgt dir )

   if(NOT CMAKE_INSTALL_PREFIX STREQUAL "/opt/local")
     message("WARNING: falcon was built with the with CMAKE_INSTALL_PREFIX /opt/local."
             "The current prefix is ${CMAKE_INSTALL_PREFIX}.")
   endif()

   if(APPLE)
      set_target_properties(${tgt} PROPERTIES
         PREFIX ""
         SUFFIX ".dylib" )
   else()
      set_target_properties(${tgt} PROPERTIES
         PREFIX "" )
   endif()

   if( DEFINED MOD_INSTALL )
      set( dest "${MOD_INSTALL}/${dir}" )
   else()
      set( dest "${Falcon_MOD_DIR}/${dir}" )
   endif()

  set_target_properties(${tgt} PROPERTIES
    LIBRARY_OUTPUT_DIRECTORY ${CMAKE_BINARY_DIR}/${Falcon_MOD_DIR}
  )

   install( TARGETS ${tgt}
            DESTINATION ${dest} )
endfunction()

function(falcon_install_module tgt )
   falcon_install_module2( "${tgt}" .)
endfunction()

function(falcon_finalize_module2 tgt libs)
      target_link_libraries(${tgt} ${Falcon_LIBRARIES} ${libs} )
      falcon_install_module( ${tgt} )
endfunction()

function(falcon_finalize_module tgt )
      target_link_libraries(${tgt} ${Falcon_LIBRARIES} )
      falcon_install_module( ${tgt} )
endfunction()
