# ===========================================================================
# File: convert.tcl
# Target: markdown
#                        Created: 2010-10-24 12:37:25
#              Last modification: 2013-12-23 17:34:13
# Author: Bernard Desgraupes
# e-mail: <bdesgraupes@users.sourceforge.net>
# (c) Copyright: Bernard Desgraupes 2010-2013
# All rights reserved.
# Description: Aida callbacks for target markdown
# ===========================================================================

namespace eval markdown {

	# This line is necessary to ensure a fallback on the base commands
	namespace path ::base

	if {[catch {aida::loadTarget html convert} res]} {
		error "can't load html target definition required by the markdown target: $res"
	} 

}


# Hooks
# -----

proc markdown::preConvertHook {} {}
proc markdown::postConvertHook {} {}

proc markdown::splitHook {file} {}

# Callbacks
# ---------

## 
 # ------------------------------------------------------------------------
 # 
 # "markdown::anchorProc" --
 # 
 # Markdown engine seems to be confused by auto-closing tags
 # 
 # ------------------------------------------------------------------------
 ##
proc markdown::anchorProc {label} {
	set label [string trim $label "\"' "]
	return "<A NAME=\"$label\"></A>"
}


## 
 # ------------------------------------------------------------------------
 # 
 # "markdown::commentProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc markdown::commentProc {str} {
	return [::html::commentProc $str]
}


## 
 # ------------------------------------------------------------------------
 # 
 # "markdown::horizRuleProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc markdown::horizRuleProc {} {
	return "\n[aida::getParam HorizRule]\n"
}


##
 # ------------------------------------------------------------------------
 # 
 # "markdown::imageProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc markdown::imageProc {str attr} {
	set attrDict [aida::getAttr img $attr]
	set str [string trim $str]
	if {[catch {dict get $attrDict alt} txt]} {
		set txt $str 
	} 
	if {![catch {dict get $attrDict title} res]} {
		set addtitle " \"$res\"" 
	} else {
		set addtitle ""
	}
	
	return "!\[$txt\]($str$addtitle)"
}


##
 # ------------------------------------------------------------------------
 # 
 # "markdown::linkProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc markdown::linkProc {str url} {
	set url [string trim $url "\"' "]
	set str [string trim $str]
	if {$str eq ""} {
		set str $url
	} 
	return "\[$str\]($url)"
}


## 
 # ------------------------------------------------------------------------
 # 
 # "markdown::listProc" --
 # 
 # The 'start' and 'type' attributes are not supported by the markdown
 # syntax.
 # 
 # ------------------------------------------------------------------------
 ##
proc markdown::listProc {kind depth attr itemList} {
	set block [list "\n"]
	set indent [string repeat "\t" [expr {$depth - 1}]]
	
	foreach itm $itemList {
		if {$kind eq "dl"} {
			lappend block "${indent}* _[lindex $itm 0]_: [string trimleft [lindex $itm 1]]"
		} elseif {$kind eq "ul"} {
			set symbs [list "+" "*" "-"]
			set prfx [lindex $symbs [expr {$depth % 3}]]
			lappend block "$indent$prfx $itm"
		} else {
			set prfx "1.  "
			lappend block "$indent$prfx$itm"
			incr cnt
		} 
	} 
	
	return [join $block "\n"]
}


## 
 # ------------------------------------------------------------------------
 # 
 # "markdown::navBarProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc markdown::navBarProc {curr prev next top} {
	if {[aida::getParam NavBar]} {
		set ext [aida::getParam NavExtension]
		set result [list]
		if {$curr ne $top} {
			set top "[markdown::_pageName $top]$ext"
			lappend result "\[[aida::getParam NavTop]\]($top)"		
		} 
		if {$prev ne ""} {
			set prev "[markdown::_pageName $prev]$ext"
			lappend result "\[[aida::getParam NavPrev]\]($prev)"		
		} 
		if {$next ne ""} {
			set next "[markdown::_pageName $next]$ext"
			lappend result "\[[aida::getParam NavNext]\]($next)"		
		} 

		return "[join $result " | "]\n\n"		
	} else {
		return ""
	} 
}


## 
 # ------------------------------------------------------------------------
 # 
 # "markdown::newLineProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc markdown::newLineProc {} {
	return "\n\n"
}


## 
 # ------------------------------------------------------------------------
 # 
 # "markdown::postambleProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc markdown::postambleProc {} {}


## 
 # ------------------------------------------------------------------------
 # 
 # "markdown::preambleProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc markdown::preambleProc {} {
	return [aida::addPreamble]
}


##
 # ------------------------------------------------------------------------
 # 
 # "markdown::printIndexProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc markdown::printIndexProc {} {
	variable aida_head

	for {set idx 0} {$idx < [aida::countIndexMarks]} {incr idx} {
		lassign [aida::getIndexMark $idx] fl str
		incr countArr($str)
		set tag "\[$countArr($str)\](${fl}#[aida::getParam IndexMark]${idx})"
		lappend indexArr($str) $tag			
	} 
	if {[info exists indexArr]} {
		foreach word [lsort -dict [array names indexArr]] {
			lappend result "* $word\t[join $indexArr($word) ", "]"
		} 
	} 
	return [join $result "\n"]
}


##
 # ------------------------------------------------------------------------
 # 
 # "markdown::refProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc markdown::refProc {str label {file ""}} {
	if {$file eq ""} {
		set file "."
	} 
	set page [markdown::_pageName $file]
	return "\[$str\](${page}#${label})"
}


## 
 # ------------------------------------------------------------------------
 # 
 # "markdown::sectionProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc markdown::sectionProc {str level {file ""}} {
	variable aida_head

	if {$file eq ""} {
		set file "."
	} 
	set title [aida::newSectionNumber $level]
	append title [aida::unwrapText [string trim $str]]
	set sc [aida::setSectionMark $level [markdown::_pageName $file] $title]
	set mark "<A name=\"[aida::getParam SectionMark]$sc\"></A>"
	
	if {$level == 1 || $level == 2} {
		set len [string length $title]
		set underline [string repeat [expr {$level==1? "=":"-"}] $len]
		set result "$mark\n\n$title\n$underline\n"
	} else {
		set result "$mark\n\n[string repeat "#" $level] $title\n"
	} 
	return $result
}


## 
 # ------------------------------------------------------------------------
 # 
 # "markdown::setIndexProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc markdown::setIndexProc {str {file ""}} {
	variable aida_head
	
	if {$file eq ""} {
		set file "."
	} 
	set idx [aida::setIndexMark [markdown::_pageName $file] $str]
	set mark "<A name=\"[aida::getParam IndexMark]$idx\"></A>"
	
	return $mark
}


## 
 # ------------------------------------------------------------------------
 # 
 # "markdown::styleProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc markdown::styleProc {style begin} {
	switch -- $style {
		"i" {
			set macro [aida::getParam EmphMark]
		}
		"b" {
			set macro [aida::getParam StrongMark]
		}
		"u" {
			set macro "_" 
		}
		"y" {
			set macro "`" 
		}
	}
	return $macro
}


## 
 # ------------------------------------------------------------------------
 # 
 # "markdown::tableProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc markdown::tableProc {attr rowList} {
	return "\n[::html::tableProc $attr $rowList]\n"
}


##
 # ------------------------------------------------------------------------
 # 
 # "markdown::tocProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc markdown::tocProc {} {
	variable aida_head
	
	set result [list]
	set depth [aida::getParam TocDepth]
	
	for {set sc 0} {$sc < [aida::countSectionMarks]} {incr sc} {
		lassign [aida::getSectionMark $sc] lv fl title
		if {$lv <= $depth} {
			lappend result "[string repeat "\t" [expr {$lv - 1}]]* \[$title\](${fl}#[aida::getParam SectionMark]$sc)"
		} 
	} 
	return [join $result "\n"]
}


##
 # ------------------------------------------------------------------------
 # 
 # "markdown::verbProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc markdown::verbProc {str} {
	return "`[aida::unwrapText $str]`"
}


## 
 # ------------------------------------------------------------------------
 # 
 # "markdown::verbatimProc" --
 # 
 # ------------------------------------------------------------------------
 ##

# TODO: to put a code block within a list item, the code block needs to be
# indented twice  8 spaces or two tabs.

proc markdown::verbatimProc {str} {
	set txt [split $str "\r\n"]
	set result [list]
	foreach line $txt {
		lappend result "    $line" 
	} 
	return [join $result "\n"]
}


## 
 # ------------------------------------------------------------------------
 # 
 # "markdown::_pageName" --
 # 
 # ------------------------------------------------------------------------
 ##
proc markdown::_pageName {file} {
	set ext [aida::getParam NavExtension]
	set result "[file root $file]$ext"
	
	return $result
}

