# ===========================================================================
# File: convert.tcl
# Target: hyperref
#                        Created: 2010-10-01 16:29:24
#              Last modification: 2012-04-11 09:50:44
# Author: Bernard Desgraupes
# e-mail: <bdesgraupes@users.sourceforge.net>
# (c) Copyright: Bernard Desgraupes 2010-2012
# All rights reserved.
# Description: default Aida settings
# ===========================================================================

namespace eval hyperref {
	variable split_file ""

	# Ensure fallback on base commands
	namespace path ::base 

	if {[catch {aida::loadTarget latex convert} res]} {
		error "can't load latex target definition required by the hyperref target: $res"
	} 

}


# Hooks
# -----

proc hyperref::preConvertHook {} {}
proc hyperref::postConvertHook {} {}

proc hyperref::splitHook {file} {}


# Callbacks
# ---------

## 
 # ------------------------------------------------------------------------
 # 
 # "hyperref::anchorProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc hyperref::anchorProc {label} {
	set label [string trim $label "\"' "]
	if {[aida::splitting]} {
		return "\\hyperdef{label}{$label}{}"
	} else {
		return "\\hypertarget{$label}{}"
	} 	
}


## 
 # ------------------------------------------------------------------------
 # 
 # "hyperref::commentProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc hyperref::commentProc {str} {
	return [::latex::commentProc $str]
}


## 
 # ------------------------------------------------------------------------
 # 
 # "hyperref::horizRuleProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc hyperref::horizRuleProc {} {
	return [::latex::horizRuleProc]
}


##
 # ------------------------------------------------------------------------
 # 
 # "hyperref::imageProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc hyperref::imageProc {str attr} {
	return [::latex::imageProc $str $attr]
}


##
 # ------------------------------------------------------------------------
 # 
 # "hyperref::linkProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc hyperref::linkProc {str url} {
	set url [string trim $url "\"' "]
	if {$str eq ""} {
		set str $url
	} 
	return "\\href{$url}{[string trim $str]}"
}


## 
 # ------------------------------------------------------------------------
 # 
 # "hyperref::listProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc hyperref::listProc {kind depth attr itemList} {
	return [::latex::listProc $kind $depth $attr $itemList]
}


## 
 # ------------------------------------------------------------------------
 # 
 # "hyperref::navBarProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc hyperref::navBarProc {curr prev next top} {
	if {[aida::getParam NavBar]} {
		set ext [aida::getParam NavExtension]
		set result [list]
		if {$curr ne $top} {
			set top "[file root $top]$ext"
			lappend result "\\href{file:$top}{[aida::getParam NavTop]}"		
		} 
		if {$prev ne ""} {
			set prev "[file root $prev]$ext"
			lappend result "\\href{file:$prev}{[aida::getParam NavPrev]}"
		} 
		if {$next ne ""} {
			set next "[file root $next]$ext"
			lappend result "\\href{file:$next}{[aida::getParam NavNext]}"
		} 

		return "[join $result "\\quad|\\quad"]\\vspace{10mm}"
	} else {
		return ""
	}
}


## 
 # ------------------------------------------------------------------------
 # 
 # "hyperref::newLineProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc hyperref::newLineProc {} {
	return [::latex::newLineProc]
}


## 
 # ------------------------------------------------------------------------
 # 
 # "hyperref::postambleProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc hyperref::postambleProc {} {
	return "\n\\end{document}\n"
}


## 
 # ------------------------------------------------------------------------
 # 
 # "hyperref::preambleProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc hyperref::preambleProc {} {	
	set cls [aida::getParam DocClass]
	set opt [aida::getParam DocOptions]
	regsub -all {[ ,]+} $opt "," opt

	set result [list "\\documentclass\[$opt\]{$cls}"]
	
	# Build the \usepackage commands
	set pkgDict [dict merge [aida::getParam "Packages"] [aida::getParam "AddPackages"]]
	set result [concat $result [::latex::_loadPackages $pkgDict]]
	
	# Ensure these packages
	foreach p [list "graphicx" "hyperref" "makeidx"] {
		if {![dict exists $pkgDict $p] == -1} {
			lappend result "\\usepackage{$p}"
		} 
	} 
	lappend result "\\makeindex"
	
	# Build a \hyperbaseurl macro if necessary
	set base [aida::getParam BaseUrl]
	if {$base ne ""} {
		lappend result "\\hyperbaseurl{$base}"
	} 
	
	# Build the \hypersetup macro
	set setup [list]
	foreach itm [list Author Creator Keywords Producer Subject Title] {
		if {![catch {aida::getParam $itm} res] && $res ne ""} {
			lappend setup "pdf[string tolower $itm]=\{$res\}"
		} 
	} 
	foreach itm [list Bookmarks BookmarksNumbered BookmarksOpen] {
		if {![catch {aida::getParam $itm} res]} {
			lappend setup "[string tolower $itm]=$res"
		} 	
	} 
	if {[llength $setup] > 0} {
		lappend result "\\hypersetup{[join $setup ",\n"]}"
	} 
	
	set tocdp [aida::getParam TocDepth]
	lappend result "\\setcounter{tocdepth}{$tocdp}"
	set secdp [aida::getParam SectionDepth]
	lappend result "\\setcounter{secnumdepth}{$secdp}"
	
	# Finally look for preamble data
	set result [concat $result [aida::addPreamble]] 

	lappend result "\n\\begin{document}\n"
	
	# Build the \maketitle macro
	set result [concat $result [::latex::_addTitle hyperref]] 
	
	return [join $result "\n"]
}


##
 # ------------------------------------------------------------------------
 # 
 # "hyperref::printIndexProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc hyperref::printIndexProc {} {
	return [::latex::printIndexProc]
}


##
 # ------------------------------------------------------------------------
 # 
 # "hyperref::refProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc hyperref::refProc {str label {file ""}} {
	set label [string trim $label "\"' "]
	if {[aida::splitting]} {
		set ext [aida::getParam NavExtension]
		set file "[file root $file]$ext"
		return "\\hyperref{file:$file}{label}{$label}{[string trim $str]}"
	} else {
		return "\\hyperlink{$label}{[string trim $str]}"
	}
}


## 
 # ------------------------------------------------------------------------
 # 
 # "hyperref::sectionProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc hyperref::sectionProc {str level {file ""}} {
	set numsecs [aida::getParam NumberSections]
	if {$numsecs} {
		set star ""
	} else {
		set star "*"
	}
	set secName [latex::_sectionName $level "long"]
	set title [string trim $str]
	set sc [aida::setSectionMark $level $file $title]
	set label [hyperref::anchorProc [hyperref::_sectionPrefix $level $sc]]
	
	return "$label\n\\$secName$star\{$title\}"
}


## 
 # ------------------------------------------------------------------------
 # 
 # "hyperref::setIndexProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc hyperref::setIndexProc {str {file ""}} {
	return [::latex::setIndexProc $str $file]
}


## 
 # ------------------------------------------------------------------------
 # 
 # "hyperref::styleProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc hyperref::styleProc {style begin} {
	return [::latex::styleProc $style $begin]
}


## 
 # ------------------------------------------------------------------------
 # 
 # "hyperref::tableProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc hyperref::tableProc {attr rowList} {
	return [::latex::tableProc $attr $rowList]
}


##
 # ------------------------------------------------------------------------
 # 
 # "hyperref::tocProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc hyperref::tocProc {} {
	if {![aida::splitting]} {
		return "\\tableofcontents"
	} else {
		variable aida_head
		variable split_file
		
		set result [list]
		set depth [aida::getParam TocDepth]
		set count 1

		if {[aida::countSectionMarks] > 0} {
			lappend result "\\begin{itemize}"
			for {set sc 0} {$sc < [aida::countSectionMarks]} {incr sc} {
				lassign [aida::getSectionMark $sc] lv fl title
				if {$lv <= $depth} {
					set ext [aida::getParam NavExtension]
					set fl "[file root $fl]$ext"
					if {$fl ne $split_file} {
						if {$count > 1} {
							lappend result "\\end{itemize}"
						} 
						lappend result "\\item\[\] "
						lappend result "\\begin{itemize}"
						set split_file $fl
					} 
					
					set label [hyperref::_sectionPrefix $lv $sc]
					set link "\\hyperref{file:$fl}{label}{$label}{[string trim $title]}"
					lappend result "\\item\[[aida::newSectionNumber $lv]\] $link"
					incr count
				} 
			} 
			lappend result "\\end{itemize}" "\\end{itemize}"
		} 
		return [join $result "\n"]
	} 
}


##
 # ------------------------------------------------------------------------
 # 
 # "hyperref::verbProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc hyperref::verbProc {str} {
	return [::latex::verbProc $str]
}


## 
 # ------------------------------------------------------------------------
 # 
 # "hyperref::verbatimProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc hyperref::verbatimProc {str} {
	return [::latex::verbatimProc $str]
}


# Target specific utility procs
# =============================

##
 # ------------------------------------------------------------------------
 # 
 # "hyperref::defaultExtension" --
 # 
 # ------------------------------------------------------------------------
 ##
proc hyperref::defaultExtension {} {
	if {[catch {aida::getParam Extension hyperref} result]} {
		set result ".tex"
	} 
	return $result
}


##
 # ------------------------------------------------------------------------
 # 
 # "hyperref::_sectionPrefix" --
 # 
 # Build a label for the different kinds of sections.
 # 
 # ------------------------------------------------------------------------
 ##
proc hyperref::_sectionPrefix {level secnum} {
	variable aida_head
	set abbrev [latex::_sectionName $level "short"]
	return "${abbrev}-[aida::getParam SectionMark]$secnum"
}


