// This file is part of CAF, the C++ Actor Framework. See the file LICENSE in
// the main distribution directory for license terms and copyright or visit
// https://github.com/actor-framework/actor-framework/blob/master/LICENSE.

#pragma once

#include <vector>

#include "caf/config_option.hpp"
#include "caf/detail/core_export.hpp"
#include "caf/fwd.hpp"

namespace caf {

/// Adds config options of the same category to a `config_option_set`.
class CAF_CORE_EXPORT config_option_adder {
public:
  // -- constructors, destructors, and assignment operators --------------------

  config_option_adder(config_option_set& target, string_view category);

  // -- properties -------------------------------------------------------------

  /// Adds a config option to the category that synchronizes with `ref`.
  template <class T>
  config_option_adder& add(T& ref, string_view name, string_view description) {
    return add_impl(make_config_option(ref, category_, name, description));
  }

  /// Adds a config option to the category.
  template <class T>
  config_option_adder& add(string_view name, string_view description) {
    return add_impl(make_config_option<T>(category_, name, description));
  }

  /// For backward compatibility only. Do not use for new code!
  /// @private
  config_option_adder&
  add_neg(bool& ref, string_view name, string_view description);

private:
  // -- properties -------------------------------------------------------------

  config_option_adder& add_impl(config_option&& opt);

  // -- member variables -------------------------------------------------------

  /// Reference to the target set.
  config_option_set& xs_;

  /// Category for all options generated by this adder.
  string_view category_;
};

} // namespace caf
