/*
 * Copyright (C) 2017, 2018 TypeFox and others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at http://www.apache.org/licenses/LICENSE-2.0
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
import { platform } from 'node:os';
import * as path from 'node:path';
import * as fs from 'node:fs';
import { fileURLToPath } from 'node:url';
import * as lsp from 'vscode-languageserver';
import { normalizePath, pathToUri } from './protocol-translation.js';
import { LspServer } from './lsp-server.js';
import { ConsoleLogger } from './logger.js';
import { TextDocument } from 'vscode-languageserver-textdocument';
import { TypeScriptVersionProvider } from './utils/versionProvider.js';
const CONSOLE_LOG_LEVEL = ConsoleLogger.toMessageTypeLevel(process.env.CONSOLE_LOG_LEVEL);
export const PACKAGE_ROOT = fileURLToPath(new URL('..', import.meta.url));
export function getDefaultClientCapabilities() {
    return {
        textDocument: {
            completion: {
                completionItem: {
                    labelDetailsSupport: true
                }
            },
            documentSymbol: {
                hierarchicalDocumentSymbolSupport: true
            },
            publishDiagnostics: {
                tagSupport: {
                    valueSet: [
                        lsp.DiagnosticTag.Unnecessary,
                        lsp.DiagnosticTag.Deprecated
                    ]
                }
            },
            moniker: {}
        }
    };
}
export function uri(...components) {
    const resolved = filePath(...components);
    return pathToUri(resolved, undefined);
}
export function filePath(...components) {
    return normalizePath(path.resolve(PACKAGE_ROOT, 'test-data', ...components));
}
export function readContents(path) {
    return fs.readFileSync(path, 'utf-8').toString();
}
export function positionAt(document, idx) {
    const doc = TextDocument.create(document.uri, document.languageId, document.version, document.text);
    const pos = doc.positionAt(idx);
    return {
        line: pos.line,
        character: pos.character
    };
}
export function position(document, match) {
    return positionAt(document, document.text.indexOf(match));
}
export function positionAfter(document, match) {
    return positionAt(document, document.text.indexOf(match) + match.length);
}
export function lastPosition(document, match) {
    return positionAt(document, document.text.lastIndexOf(match));
}
export function toPlatformEOL(text) {
    if (platform() === 'win32') {
        return text.replace(/(?!\r)\n/g, '\r\n');
    }
    return text;
}
export class TestLspServer extends LspServer {
    constructor() {
        super(...arguments);
        this.workspaceEdits = [];
    }
}
export function createServer(options) {
    return __awaiter(this, void 0, void 0, function* () {
        const typescriptVersionProvider = new TypeScriptVersionProvider();
        const bundled = typescriptVersionProvider.bundledVersion();
        const logger = new ConsoleLogger(CONSOLE_LOG_LEVEL);
        const server = new TestLspServer({
            logger,
            tsserverPath: bundled.tsServerPath,
            tsserverLogVerbosity: options.tsserverLogVerbosity,
            tsserverLogFile: path.resolve(PACKAGE_ROOT, 'tsserver.log'),
            lspClient: {
                setClientCapabilites() { },
                createProgressReporter() {
                    return {
                        begin() { },
                        report() { },
                        end() { }
                    };
                },
                publishDiagnostics: options.publishDiagnostics,
                showMessage(args) {
                    throw args; // should not be called.
                },
                logMessage(args) {
                    logger.log('logMessage', JSON.stringify(args));
                },
                telemetry(args) {
                    logger.log('telemetry', JSON.stringify(args));
                },
                applyWorkspaceEdit(args) {
                    return __awaiter(this, void 0, void 0, function* () {
                        server.workspaceEdits.push(args);
                        return { applied: true };
                    });
                },
                rename: () => Promise.reject(new Error('unsupported'))
            }
        });
        yield server.initialize({
            rootPath: undefined,
            rootUri: options.rootUri,
            processId: 42,
            capabilities: options.clientCapabilitiesOverride || getDefaultClientCapabilities(),
            workspaceFolders: null
        });
        return server;
    });
}
//# sourceMappingURL=test-utils.js.map