/* 
   sand - helps you keep your diary

   sand.c - main program file

   Copyright (C) 2001 Per Jonsson

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

#include <termios.h>
#include <grp.h>
#include <pwd.h>
*/

#define __SAND_C__SAND__

#include <stdio.h>
#include <sys/types.h>
#include <getopt.h>
#include <string.h>
#include "system.h"
#include "ddecode.h"
#include "output.h"
#include "sand.h"

static void usage (int status);

/* The name the program was run with, stripped of any leading path. */
char *program_name = (char *)NULL;
/* The rc directory for the program */
char *program_rc_dir = (char *)NULL;
/* The name of the diary file */
char *program_diary_file = (char *)NULL;
/* The program home dir */
char *program_home = (char *)NULL;

/* getopt_long return codes */
enum {DUMMY_CODE=129
};

/* Option flags and variables */

int want_verbose;		/* --verbose */
int want_output_text;           /* --output-text */
int want_dump_text;             /* --dump-text */
int want_list;                  /* --list */
int want_note;                  /* --note */
int note_number = 0;

static struct option const long_options[] =
{
  {"verbose", no_argument, 0, 'v'},
  {"help", no_argument, 0, 'h'},
  {"version", no_argument, 0, 'V'},
  {"output-text", no_argument, 0, 'o'},
  {"dump-text", no_argument, 0, 'd'},
  {"note", required_argument, 0, 'n'},
  {"list",no_argument,0,'l'},
  {NULL, 0, NULL, 0}
};

static int decode_switches (int argc, char **argv);

int
main (int argc, char **argv)
{
  int i,j;
  unsigned char status;

  program_name = argv[0];

  i = decode_switches (argc, argv);

  /* Set up some configuration parsing */
  status = sand_config();

  if (want_output_text){
    cli_output();
  } else if (want_dump_text){
    cli_dump();
  } else {
    /* Normal input loop */
    cli_input(status);
  }

  exit (0);
}

/* Set all the option flags according to the switches specified.
   Return the index of the first non-option argument.  */

static int
decode_switches (int argc, char **argv)
{
  int c;


  while ((c = getopt_long (argc, argv, 
			   "v"	/* verbose */
			   "h"	/* help */
			   "V"	/* version */
			   "o"  /* output-text */
			   "d"  /* dump-text */
			   "n:" /* note */
			   "l", /* list */
			   long_options, (int *) 0)) != EOF)
    {
      switch (c)
	{
	case 'v':		/* --verbose */
	  want_verbose = 1;
	  break;
	case 'o':
	  want_output_text = 1; /* --output-text */
	  break;
	case 'd':
	  want_dump_text = 1;   /* --dump-text */
	  break;
	case 'n':
	  want_note = 1;        /* --note */
	  if ((optarg != NULL) && (isdigit(optarg[0])))
	      note_number = atoi(optarg);
	  break;
	case 'l':
	  want_list = 1;        /* --list */
	  break;
	case 'V':
	  printf ("sand %s\n", VERSION);
	  exit (0);
	  break;
	case 'h':
	  usage (0);
	  break;
	default:
	  usage (EXIT_FAILURE);
	}
    }

  return optind;
}


static void
usage (int status)
{
  printf (_("%s - \
helps you keep your diary\n"), program_name);
  printf (_("Usage: %s [OPTION]...\n"), program_name);
  printf (_("\
Options:\n\
  -o, --output-text          output a diary note as plain text\n\
  -d, --dump-text            output the entire diary as plain text\n\
\n\
  -n ID, --note=ID           choose note ID as note to work with\n\
                             only affects output modes\n\
                             zero based\n\
\n\
  -v, --verbose              print more information\n\
  -h, --help                 display this help and exit\n\
  -V, --version              output version information and exit\n\
"));
  exit (status);
}

/* parse configuration */
unsigned char
sand_config()
{
  int dir_status;
  
  get_create_rc_dir();
  get_diary_file_name();
  return 0;
}


/* input_diary - reads an entire diary into memory */
dinfo * input_diary(dinfo * diary_ret)
{
  FILE * diary = (FILE *) NULL;  /* FILE * to diary */
  char * diary_p = (char *)NULL; /* PATH to diary */
  char * temp = (char *)NULL;
  int diarylen = 0;              /* Misc uses concerning length of diary */
  ssize_t nc = (ssize_t) 0;
  size_t length = 0;
  size_t diary_size = (size_t) 0;
 
  diary_ret->text = (char*) NULL;
  diary_ret->length = 0;
  diary_ret->position = 0;

  /* Open the file for reading */
  diary = fopen(get_diary_file_name(), "r");

  /* Do some error checking on the FILE * */ 
  if (diary == NULL)
    {

      if (errno == EINVAL){
	/* The system isn't handling my mode, exiting */
	perror("sand");
	exit (EXIT_FAILURE);
      }
      
      /* Something else is wrong, exit */
      perror("sand");
      exit(EXIT_FAILURE);
    }

  /* Loop until the whole diary is read */
  do {

    /* Read the line */
    nc = getline( &diary_p, &diary_size, diary);
    
    /* If char_read == -1 then we probabably have read until EOF */
    if (nc == -1)
      {
	if (!feof(diary))         /* Everything that is error except */
	  {                       /* EOF is evil */
	    message_err("Evil input from diary file\nExiting....\n");
	    fclose (diary);
	    exit (EXIT_FAILURE); /* Escape from the program */
	  } 
      } else {
	temp = xrealloc(temp, length + nc + 1);
	diary_ret->text = temp;

	temp = temp + length;
	memcpy(temp, diary_p, nc);

	length += nc;
	temp = diary_ret->text;
      }
  } while ( nc != -1 );
  
  diary_ret->length = length;

  fclose(diary);
  return diary_ret;
}


/* get_diary_file_name - returns the filename of the diary */
char * get_diary_file_name()
{
  int diarylen;
    
  if (program_diary_file == NULL)
    {
      /* get the directory */
      program_diary_file = xstrdup(get_home_dir());
      diarylen = strlen(program_diary_file);
      
      program_diary_file = xrealloc(program_diary_file, 
				    diarylen + strlen(DIARY_NAME)+ 2);
      
      if (program_diary_file[diarylen] != '/')
	{
	  program_diary_file[diarylen] = '/';
	  program_diary_file[diarylen+1] = '\000';
	}
      
      strncat((program_diary_file + diarylen),
	      DIARY_NAME, strlen(DIARY_NAME)+1);
    }
  return program_diary_file;
}

/* open_diary - open, create, and init the diary as needed */
FILE * open_diary(int mode)
{
  FILE * fs;
  char * s = get_diary_file_name();
  int i = 0;


  fs = fopen(s,"r");
  switch (mode)
    {
    case O_RDONLY:
      return fs;
      break;
    case O_RDWR:
    default:
      if (fs != NULL)
	{
	  i = 1;
	  fclose(fs);
	}
      if (i == 0)
	{
	  fs = fopen(s,"w+");
	  if (fs == NULL)
	    return fs;
	  fputs("<?sand>\n", fs);
	} else {
	  fs = fopen(s, "a+");
	}
      break;
    }

  return fs;
}

/* dump_file - takes the contents of from and writes it to to */
int dump_file(FILE * from, FILE * to)
{
  int i;

  for (i = getc(from);i != EOF;i=getc(from))
      i = putc(i, to);

  if (i==EOF)
    return EOF;
  else
    return 0;
}





