/* 
   sand - helps you keep your diary
   
   ddecode.c - decoder for diary format 

   Copyright (C) 2001 Per Jonsson

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

#include <termios.h>
#include <grp.h>
#include <pwd.h>
*/

#include <stdio.h>
#include <sys/types.h>
#include <getopt.h>
#include <string.h>
#include "system.h"
#include "repl_func.h"

#include "ddecode.h"

char *xmalloc ();
char *xrealloc ();
char *xstrdup ();
char *xcalloc ();

/* ddecode_init checks the first characters of the stream to see that
   it is a diary */
int ddecode_init(dinfo * diary)
{
  int ret_val;

  if (strncasecmp (diary->text, "<?sand>",7)== 0)
    {
      ret_val = DIARY_IS;
      diary->position += 7;
    } else {
      ret_val = DIARY_IS_NOT;
    }
    
  return ret_val;
}

/* ddecode_find_tag searches for a tag in the diary */
int ddecode_find_tag(const dinfo d, const char * t)
{
  int i,j,f;

  f = strlen(t);

  /* Loop to find <note> */
  for (i = 0; i < (d.length - d.position - f); i++)
    {
      j=-1;
      if (strncasecmp ((d.text + (d.position+i)), t ,f ) == 0)
	{
	  j = i;
	  i = d.length;
	}
    }
  return j;
}
    
/* ddecode_note decodes a note */
dnote ddecode_note(dinfo * d, int note)
{
  int i, j, k;
  int date_pos;
  int pos;
  int title_start = 0;
  int title_end = 0;
  dnote d_note;
  memset(&d_note, '\0', sizeof(d_note));

  if (note == DIARY_NEXT_NOTE)
    {
      note = 0;
    } else {
      d->position = 0;
    }
  pos = d->position;

  /* Search for the note:th <note> tag */
  for (i=0; i <= note; i++)
    {
      j=ddecode_find_tag(*d, "<note>");
      d->position += ((j>0)?(j+6):(0));
    }

  /* Check to se if there is a note:th tag */
  if (j == -1)
    {
      d_note.length = ERR_NO_SUCH_NOTE;
      d->position = pos;
      return d_note;
    }
  
  /* Find the date */
  date_pos = ddecode_find_tag(*d,"<date>");
  if ( date_pos == -1)
    {
      d_note.length = ERR_INV_FORMAT;
      d->position = pos;
      return d_note;
    }

  /* Find end of date information */
  j = ddecode_find_tag(*d,"</date>");
  if ( j == -1)
    {
      d_note.length = ERR_INV_FORMAT;
      d->position = pos;
      return d_note;
    }

  /* Decode the date */
  if (ddecode_date(*d, &d_note, date_pos, j - date_pos)== -1)
    {
      d_note.length = ERR_INV_FORMAT;
      d->position = pos;
      return d_note;
    }

  /* Locate the title of the not if there is one */
  title_start = ddecode_find_tag(*d,"<title>");
  if ( title_start != -1)
    {
      /* If the title is opened, an end is required. */
      title_end = ddecode_find_tag(*d,"</title>");
      if ( title_end == -1)
	{
	  d_note.length = ERR_INV_FORMAT;
	  d->position = pos;
	  return d_note;
	} 

      title_end -= (title_start + strlen("<title>"));
      title_start += d->position;
    }

  /* Find the position of the body */
  j = ddecode_find_tag(*d,"<body>");
  if ( j == -1)
    {
      d_note.length = ERR_INV_FORMAT;
      d->position = pos;
      return d_note;
    }
  d->position += j + 6;

  /* Find the end of the body */
  d_note.length = ddecode_find_tag(*d,"</body>");
  if ( d_note.length == -1)
    {
      d_note.length == ERR_INV_FORMAT;
      d->position = pos;
      return d_note;
    }

  /* Find the end of note tag */
  j = ddecode_find_tag(*d,"</note>");
  if ( j == -1)
    {
      d_note.length = ERR_INV_FORMAT;
      d->position = pos;
      return d_note;
    }

  d_note.text = d->text + d->position;
  d->position += j;


  if ((d->position > title_start) && (title_end > 0))
    {
      /* Now, Lets take care of the title */
      d_note.title = strndup((d->text + title_start + strlen("<title>")), 
			     title_end);
    } else {
      d_note.title = (char *) NULL;
    }

  return d_note;
}


/* ddecode_date decodes the date */
int ddecode_date(const dinfo d, dnote * n, int p, int e)
{
  char bck[5];
  char * temp;
  int p_org = p;
  int i = 0;

  while (isspace (d.text[p + d.position + 6]))
      p++;

  bck[0] = d.text[p_org + d.position + e];
  d.text[p_org + d.position + e] = '\0';
  bck[1] = d.text[p_org + d.position + e - 1];
  if (!isdigit(bck[1]))
    d.text[p_org + d.position + e - 1] = '\0';

  temp = d.text + p + d.position + 6;

  /* YEAR */
  if (isdigit(temp[0]) && isdigit(temp[1]) &&
      isdigit(temp[2]) && isdigit(temp[3]))
    {
      bck[2] = temp[4];
      temp[4] = '\0';
    } else {
      d.text[p_org + d.position + e] = bck[0];
      d.text[p_org + d.position + e - 1] = bck[1];
      return ERR_INV_FORMAT;
    }

  n->time.year = atoi(temp);
  /* MISSING: Error checking on date */
  
  temp[4]=bck[2];
  temp += 5;  /* Increase pointer do read next */
  
  /* MONTH */
  if (isdigit(temp[0]) && isdigit(temp[1]))
    {
      bck[2] = temp[2];
      temp[2] = '\0';
    } else {
      d.text[p_org + d.position + e] = bck[0];
      d.text[p_org + d.position + e - 1] = bck[1];
      return ERR_INV_FORMAT;
    }
  
  if (temp[0] == '0') 
    n->time.month = atoi(temp+1); 
  else
    n->time.month = atoi(temp);
  /* MISSING: Error checking on date */
  
  temp[2]=bck[2];
  temp += 3;  /* Increase pointer do read next */

  /* DAY */
  if (isdigit(temp[0]) && isdigit(temp[1]))
    {
      bck[2] = temp[2];
      temp[2] = '\0';
    } else {
      d.text[p_org + d.position + e] = bck[0];
      d.text[p_org + d.position + e - 1] = bck[1];
      return ERR_INV_FORMAT;
    }
  
  if (temp[0] == '0') 
    n->time.day = atoi(temp+1); 
  else
    n->time.day = atoi(temp);
  /* MISSING: Error checking on date */
  
  temp[2]=bck[2];
  temp += 3;  /* Increase pointer do read next */


  /* HOUR : MIN : SECOND */
  while (!isdigit(temp[i]))
    i++;

  temp += i;

  if (isdigit(temp[0]) && isdigit(temp[1]) &&
      isdigit(temp[3]) && isdigit(temp[4]) &&
      isdigit(temp[6]) && isdigit(temp[7]))
    {
      bck[2] = temp[2];
      bck[3] = temp[5];
      bck[4] = temp[8];
      temp[2] = '\0';
      temp[5] = '\0';
      temp[8] = '\0';
    } else {
      d.text[p_org + d.position + e] = bck[0];
      d.text[p_org + d.position + e - 1] = bck[1];
      return ERR_INV_FORMAT;
    }
  
  if (temp[0] == '0') 
    n->time.hour = atoi(temp + 1); 
  else
    n->time.hour = atoi(temp);

  if (temp[3] == '0') 
    n->time.minute = atoi(temp + 3 + 1); 
  else
    n->time.minute = atoi(temp + 3);
  
  if (temp[6] == '0') 
    n->time.seconds = atoi(temp + 6 + 1); 
  else
    n->time.seconds = atoi(temp + 6);

  /* MISSING: Error checking on date */

  temp[2] = bck[2];
  temp[5] = bck[3];
  temp[8] = bck[4];

  d.text[p_org + d.position + e] = bck[0];
  d.text[p_org + d.position + e - 1] = bck[1];

  return DATE_OK;
}

/* time_t2dtime converts a time_t to a dtime */
dtime time_t2dtime(time_t t)
{
  dtime d;
  struct tm * tm;

  tm = gmtime(&t);

  d.year = tm->tm_year + 1900;
  d.month = tm->tm_mon + 1;
  d.day = tm->tm_mday;
  d.hour = tm->tm_hour;
  d.minute = tm->tm_min;
  d.seconds = tm->tm_sec;

  return d;
}

dnote ddecode_note_next(dinfo * d)
{
  return ddecode_note(d, DIARY_NEXT_NOTE);
}

