class TestRelationship < Test::Unit::TestCase
	include TapKit

	def setup
		@app = Application.new MODELS
		@ec  = @app.create_editing_context
	end

	def test_default_parameter
		relation = Relationship.new
		assert_equal(Relationship::INNER_JOIN, relation.join_semantic)
		assert_equal(Relationship::DELETE_RULE_NULLIFY, relation.delete_rule)
		assert_equal(false, relation.own?)
		assert_equal(false, relation.propagate?)
	end

	def test_beautify_name
		rel = Relationship.new

		rel.name = 'RELATIONSHIP_NAME'
		rel.beautify_name
		assert_equal('relationship_name', rel.name)
	end

	def test_relation
		list = { 'name' => 'publisher',
		         'destination' => 'Publisher',
		         'to_many' => false,
		         'joins' => [{'source'=>'publisher_id', 'destination'=>'publisher_id'}] }
		source_entity      = @app.entity 'Book'
		destination_entity = @app.entity 'Publisher'
		relation    = Relationship.new(list, source_entity)
		source      = source_entity.attribute('publisher_id')
		destination = destination_entity.attribute('publisher_id')

		assert_equal(source_entity, relation.entity)
		assert_equal([source], relation.source_attributes)
		assert_equal(destination_entity, relation.destination_entity)
		assert_equal([destination], relation.destination_attributes)

		join = Join.new(source, destination)
		assert_equal([join], relation.joins)
	end

	def test_set
		source      = @app.entity('Book').attribute('publisher_id')
		destination = @app.entity('Publisher').attribute('publisher_id')
		join        = Join.new(source, destination)
		relation    = Relationship.new
		relation.add join
		relation.remove join
		assert relation.joins.empty?
	end

	# empty to-one, size 0 to-many
	def test_validate_value
		entity                 = @app.model_group.entity 'Book'
		re_publisher           = entity.relationship 'publisher'
		re_publisher.mandatory = true

		# type check (fault)
		fault = @ec.create 'Publisher'
		fault.turn_into_fault FaultHandler.new
		assert_equal(fault, re_publisher.validate_value(fault))

		# type check (other class)
		assert_raises(ValidationError) { re_publisher.validate_value 'test' }

		# to one
		assert_raises(ValidationError) { re_publisher.validate_value nil }

		# to many
		re_publisher.to_many = true
		assert_raises(ValidationError) { re_publisher.validate_value [] }

		re_publisher.mandatory = false
	end

	def test_validate_required_attributes
		relation = Relationship.new
		assert_raises(ValidationError) { relation.validate_required_attributes }
	end

	def test_to_h
		list = {'name'=>'publisher', 'destination'=>'Publisher', 'to_many'=>false,
			'mandatory'=>false, 'delete_rule'=>'cascade', 'join_semantic'=>'inner',
			'joins'=>[{'source'=>'publisher_id', 'destination'=>'publisher_id'}]}
		entity = @app.entity 'Book'
		re = Relationship.new(list, entity)

		assert_equal(list, re.to_h)
	end

	def test_inverse_relationship
		book      = @app.entity 'Book'
		publisher = @app.entity 'Publisher'
		rel       = publisher.relationship 'books'
		inverse   = book.relationship('publisher').inverse_relationship
		assert_equal(rel, inverse)
	end
end
