class TestEditingContextDeletion < Test::Unit::TestCase
	include TapKit

	def setup
		@app = Application.new MODELS
		@ec  = @app.create_editing_context
	end

	def setup_for_single
		test = "test_delete<#{Time.now}>"
		object = @ec.create 'cd'
		object['title'] = test
		object['artist'] = test
		object['release'] = 2000
		@ec.save
		object
	end

	def setup_for_relationship( name = 'delete_relationship' )
		book1 = @ec.create 'Book'
		book2 = @ec.create 'Book'
		publisher = @ec.create 'Publisher'

		book1['title'] = "test_#{name}1<#{Time.now}>"
		book2['title'] = "test_#{name}2<#{Time.now}>"
		publisher['name'] = "test_#{name}3<#{Time.now}>"

		book1['publisher'] = publisher
		book2['publisher'] = publisher
		publisher['books'] << book1
		publisher['books'] << book2

		@ec.save

		[book1, book2, publisher]
	end

	def test_01_delete_object_without_relationships
		# delete the test data
		object = setup_for_single
		title = object['title'].dup
		@ec.delete object
		@ec.save

		# check the deleted data
		q = Qualifier.format("title == %@", [title])
		fs = FetchSpec.new('cd', q)
		fs.refresh = true
		objects = @ec.fetch fs
		assert objects.empty?
	end

	def test_02_delete_object_with_to_one_relationship_by_nullify
		book1, book2, publisher = setup_for_relationship 'delete_to_one_nullify'
		@ec.delete book1
		@ec.delete book2
		@ec.save

		# check the publisher object
		assert publisher['books'].empty?

		# check refetched objects
		name       = publisher['name'].dup
		q          = Qualifier.format("name == %@", [name])
		fs         = FetchSpec.new('Publisher', q)
		fs.refresh = true
		objects    = @ec.fetch fs
		object     = objects.first
		assert object['books'].empty?
	end

	def test_03_delete_object_with_to_many_relationship_by_nullify
		book1, book2, publisher = setup_for_relationship 'delete_to_many_nullify'
		@ec.delete publisher
		@ec.save

		# check the book objects
		assert_nil book1['publisher']
		assert_nil book2['publisher']

		# check refetched objects
		title1     = book1['title'].dup
		title2     = book2['title'].dup
		q          = Qualifier.format("(title == %@) or (title == %@)", [title1, title2])
		fs         = FetchSpec.new('Book', q)
		fs.refresh = true
		objects    = @ec.fetch fs
		assert_nil objects[0]['publisher']
		assert_nil objects[1]['publisher']
	end

	def test_04_delete_object_with_to_one_relationship_by_cascade
		# set cascade for delete rule
		entity = @app.entity 'Book'
		rel = entity.relationship 'publisher'
		rel.delete_rule = Relationship::DELETE_RULE_CASCADE

		# delete objects
		book1, book2, publisher = setup_for_relationship 'delete_to_one_cascade'
		@ec.delete book1
		@ec.delete book2
		@ec.save

		# check
		name       = publisher['name'].dup
		q          = Qualifier.format("name == %@", [name])
		fs         = FetchSpec.new('Publisher', q)
		fs.refresh = true
		objects    = @ec.fetch fs
		assert objects.empty?
	end

	def test_05_delete_object_with_to_many_relationship_by_cascade
		# set cascade for delete rule
		entity = @app.entity 'Publisher'
		rel = entity.relationship 'books'
		rel.delete_rule = Relationship::DELETE_RULE_CASCADE

		# delete objects
		book1, book2, publisher = setup_for_relationship 'delete_to_many_cascade'
		@ec.delete publisher
		@ec.save

		# check
		title1     = book1['title'].dup
		title2     = book2['title'].dup
		q          = Qualifier.format("(title == %@) or (title == %@)", [title1, title2])
		fs         = FetchSpec.new('Book', q)
		fs.refresh = true
		objects    = @ec.fetch fs
		assert objects.empty?
	end

	def test_06_delete_object_with_to_one_relationship_by_deny
		# set cascade for delete rule
		entity = @app.entity 'Book'
		rel = entity.relationship 'publisher'
		rel.delete_rule = Relationship::DELETE_RULE_DENY

		# delete objects
		book1, book2, publisher = setup_for_relationship 'delete_to_one_deny'
		@ec.delete book1
		@ec.delete book2
		assert_raises(ValidationError) { @ec.save }
	end

	def test_07_delete_object_with_to_many_relationship_by_deny
		# set cascade for delete rule
		entity = @app.entity 'Publisher'
		rel = entity.relationship 'books'
		rel.delete_rule = Relationship::DELETE_RULE_DENY

		# delete objects
		book1, book2, publisher = setup_for_relationship 'delete_to_many_deny'
		@ec.delete publisher
		assert_raises(ValidationError) { @ec.save }
	end

end
