module TapKit
	class Attribute
		ADAPTER_CHARACTERS_TYPE = 0
		ADAPTER_NUMBER_TYPE     = 1
		ADAPTER_BYTES_TYPE      = 2
		ADAPTER_DATE_TYPE       = 3

		attr_accessor :allow_null, :class_name, :column_name, :definition, \
			:external_type, :name, :read_only, :value_type, :width, :precision, \
			:read_format, :write_format, :factory_method, :conversion_method
		attr_reader :entity

		def initialize( list = {}, entity = nil )
			@entity            = entity
			@allow_null        = list['allow_null'] 
			@name              = list['name']
			@class_name        = list['class_name']
			@column_name       = list['column_name']
			@external_type     = list['external_type']
			@width             = list['width']
			@precision         = list['precision']
			@read_only         = list['read_only'] || false
			@factory_method    = list['factory_method']
			@conversion_method = list['conversion_method']
			@read_format       = nil
			@write_format      = nil

			if @allow_null.nil? then
				@allow_null = false	
			end
		end

		def beautify_name
			if column_name then
				@name = column_name.downcase
			end
		end

		def allow_null?
			@allow_null
		end

		def read_only?
			@read_only
		end


		#
		# Validation
		#

		def validate_required_attributes
			msg = "Attribute requires attributes: 'name', 'column_name', " +
				"'external_type', 'class_name'"

			if @name.nil? or @column_name.nil? or @external_type.nil? or \
				@class_name.nil? then
				key = @name || :attribute
				error = ValidationError.new(msg, key)
				raise error
			end
		end

		def validate_value( value )
			unless _read_only_primary_key? then
				errors = []
				_validate_allow_null(errors, value)
				_validate_read_only(errors, value)
				_validate_width(errors, value)

				unless errors.empty? then
					raise ValidationError.aggregate(errors)
				end
			end

			value
		end

		private

		def _read_only_primary_key?
			if read_only? and \
				@entity.primary_key_attribute?(self) and \
				(@entity.primary_key_attributes.size == 1) then
				true
			else
				false
			end
		end

		def _property
			"The '#@name' property of '#{@entity.name}'"
		end

		def _validate_allow_null( errors, value )
			if (allow_null? != true) and value.nil? then
				errors << ValidationError.new("#{_property} is not allow null", @name)
			end
		end

		def _validate_read_only( errors, value )
			if read_only? then
				errors << ValidationError.new("#{_property} is read only", @name)
			end
		end

		def _validate_width( errors, value )
			if @width and value and (String === value) and (@class_name == 'String') then
				if value.size > @width then
					errors << ValidationError.new(
						"#{_property} exceeds maximum length of #@width characters",
						@name)
				end
			end
		end

		public

		def new_value( value, encoding = nil )
			if value.nil? then return value end

			if @class_name and @factory_method then
				klass = eval @class_name
				converted = klass.__send__(@factory_method, value)
				return converted
			end

			begin
				converted = value

				case @class_name
				when 'String'
					converted = value.to_s.dup

					if encoding then
						converted = Utilities.encode(converted, encoding)
					end
				when 'Integer'
					converted = value.to_i
				when 'Float'
					converted = value.to_f
				when 'BigDecimal'
					converted = value.to_s.dup
				when 'Date'
					unless Date === value then
						converted = Date.new(value.year, value.mon, value.mday)
					end
				when 'Time'
					unless Time === value then
						converted = Time.new(value.hour, value.min, value.sec)
					end
				when 'Timestamp'
					if String === value then
						value = DateTime.parse value
					end

					unless Timestamp === value then
						converted = Timestamp.new(value.year, value.mon, value.mday, \
							value.hour, value.min, value.sec)
					end
				when 'Boolean'
					if (value == 1) or (value == true) then
						converted = true
					elsif (value == 0) or (value == false) then
						converted = false
					else
						converted = nil
					end
				else
					# (not yet implemented) custom class
					raise ArgumentError, "uninitialized constant #@class_name"
				end
				value = converted
			rescue
				raise ArgumentError, "can't cast to #@class_name - #{value.class}:#{value}"
			end

			value
		end

		def adapter_value( value )
			if @conversion_method and (value.class.to_s == @class_name) then
				value.__send__ @conversion_method
			elsif @class_name == 'Boolean' then
				_convert_bool value
			else
				value
			end
		end

		private

		def _convert_bool( value )
			case adapter_value_type
			when ADAPTER_NUMBER_TYPE
				case value.to_s
				when 'true'  then 1
				when 'false' then 0
				else              nil
				end
			else
				raise "can't cast '#@external_type' to boolean"
			end
		end

		public

		def adapter_value_type
			adapter = Adapter.adapter_class(@entity.model.adapter_name)
			case adapter.internal_type(@external_type).to_s
			when 'String'            then ADAPTER_CHARACTERS_TYPE
			when 'Integer'           then ADAPTER_NUMBER_TYPE
			when 'Float'             then ADAPTER_NUMBER_TYPE
			when 'TapKit::Date'      then ADAPTER_DATE_TYPE
			when 'TapKit::Time'      then ADAPTER_DATE_TYPE
			when 'TapKit::Timestamp' then ADAPTER_DATE_TYPE
			end
		end

		def to_h
			property = {
				'name'          => @name,
				'class_name'    => @class_name,
				'column_name'   => @column_name,
				'allow_null'    => @allow_null,
				'external_type' => @external_type,
				'read_only'     => @read_only
			}
			property['width'] = @width if @width

			property
		end
	end
end
