/**
 * @file src/capstone2llvmir/powerpc/powerpc_init.cpp
 * @brief Initializations for PowerPC implementation of @c Capstone2LlvmIrTranslator.
 * @copyright (c) 2017 Avast Software, licensed under the MIT license
 */

#include "capstone2llvmir/powerpc/powerpc_impl.h"

namespace retdec {
namespace capstone2llvmir {

//
//==============================================================================
// Pure virtual methods from Capstone2LlvmIrTranslator_impl
//==============================================================================
//

void Capstone2LlvmIrTranslatorPowerpc_impl::initializeArchSpecific()
{
	std::map<uint32_t, std::string> r2n =
	{
			{PPC_REG_CR0_LT, "cr0_lt"},
			{PPC_REG_CR0_GT, "cr0_gt"},
			{PPC_REG_CR0_EQ, "cr0_eq"},
			{PPC_REG_CR0_SO, "cr0_so"},

			{PPC_REG_CR1_LT, "cr1_lt"},
			{PPC_REG_CR1_GT, "cr1_gt"},
			{PPC_REG_CR1_EQ, "cr1_eq"},
			{PPC_REG_CR1_SO, "cr1_so"},

			{PPC_REG_CR2_LT, "cr2_lt"},
			{PPC_REG_CR2_GT, "cr2_gt"},
			{PPC_REG_CR2_EQ, "cr2_eq"},
			{PPC_REG_CR2_SO, "cr2_so"},

			{PPC_REG_CR3_LT, "cr3_lt"},
			{PPC_REG_CR3_GT, "cr3_gt"},
			{PPC_REG_CR3_EQ, "cr3_eq"},
			{PPC_REG_CR3_SO, "cr3_so"},

			{PPC_REG_CR4_LT, "cr4_lt"},
			{PPC_REG_CR4_GT, "cr4_gt"},
			{PPC_REG_CR4_EQ, "cr4_eq"},
			{PPC_REG_CR4_SO, "cr4_so"},

			{PPC_REG_CR5_LT, "cr5_lt"},
			{PPC_REG_CR5_GT, "cr5_gt"},
			{PPC_REG_CR5_EQ, "cr5_eq"},
			{PPC_REG_CR5_SO, "cr5_so"},

			{PPC_REG_CR6_LT, "cr6_lt"},
			{PPC_REG_CR6_GT, "cr6_gt"},
			{PPC_REG_CR6_EQ, "cr6_eq"},
			{PPC_REG_CR6_SO, "cr6_so"},

			{PPC_REG_CR7_LT, "cr7_lt"},
			{PPC_REG_CR7_GT, "cr7_gt"},
			{PPC_REG_CR7_EQ, "cr7_eq"},
			{PPC_REG_CR7_SO, "cr7_so"},
	};

	_reg2name = std::move(r2n);
}

void Capstone2LlvmIrTranslatorPowerpc_impl::initializeRegNameMap()
{
	// Nothing.
}

void Capstone2LlvmIrTranslatorPowerpc_impl::initializeRegTypeMap()
{
	auto* i1 = llvm::IntegerType::getInt1Ty(_module->getContext());
	auto* i4 = llvm::IntegerType::getIntNTy(_module->getContext(), 4);
	auto* i32 = llvm::IntegerType::getInt32Ty(_module->getContext());
	auto* i64 = llvm::IntegerType::getInt64Ty(_module->getContext());
	auto* i128 = llvm::IntegerType::getInt128Ty(_module->getContext());
	auto* i256 = llvm::IntegerType::getIntNTy(_module->getContext(), 256);
	auto* f64 = llvm::Type::getDoubleTy(_module->getContext());

	auto* defTy = getArchBitSize() == 64 ? i64 : i32;

	std::map<uint32_t, llvm::Type*> r2t =
	{
			{PPC_REG_CARRY, i1},

			// Condition registers.
			// 4-bit x 8 = 32-bit CR register.
			//
			{PPC_REG_CR0_LT, i1},
			{PPC_REG_CR0_GT, i1},
			{PPC_REG_CR0_EQ, i1},
			{PPC_REG_CR0_SO, i1},

			{PPC_REG_CR1_LT, i1},
			{PPC_REG_CR1_GT, i1},
			{PPC_REG_CR1_EQ, i1},
			{PPC_REG_CR1_SO, i1},

			{PPC_REG_CR2_LT, i1},
			{PPC_REG_CR2_GT, i1},
			{PPC_REG_CR2_EQ, i1},
			{PPC_REG_CR2_SO, i1},

			{PPC_REG_CR3_LT, i1},
			{PPC_REG_CR3_GT, i1},
			{PPC_REG_CR3_EQ, i1},
			{PPC_REG_CR3_SO, i1},

			{PPC_REG_CR4_LT, i1},
			{PPC_REG_CR4_GT, i1},
			{PPC_REG_CR4_EQ, i1},
			{PPC_REG_CR4_SO, i1},

			{PPC_REG_CR5_LT, i1},
			{PPC_REG_CR5_GT, i1},
			{PPC_REG_CR5_EQ, i1},
			{PPC_REG_CR5_SO, i1},

			{PPC_REG_CR6_LT, i1},
			{PPC_REG_CR6_GT, i1},
			{PPC_REG_CR6_EQ, i1},
			{PPC_REG_CR6_SO, i1},

			{PPC_REG_CR7_LT, i1},
			{PPC_REG_CR7_GT, i1},
			{PPC_REG_CR7_EQ, i1},
			{PPC_REG_CR7_SO, i1},

			// We also define these registers, because they can unexpectedly be
			// operands of some instructions, and if we do not have it, operand
			// loading/storing would fail.
			// Do not intentionally use it, use PPC_REG_CRx_LT, etc. instead.
			//
			// TODO: Remove these and modify register load/store methods to
			// translate loading/storing of these into PPC_REG_CRx_LT, etc.
			//
			{PPC_REG_CR0, i4},
			{PPC_REG_CR1, i4},
			{PPC_REG_CR2, i4},
			{PPC_REG_CR3, i4},
			{PPC_REG_CR4, i4},
			{PPC_REG_CR5, i4},
			{PPC_REG_CR6, i4},
			{PPC_REG_CR7, i4},

			// Count register.
			// 32-bit in 32-bit implementation, 64-bit in 64-bit.
			//
			{PPC_REG_CTR, defTy},

			// Floating-point registers.
			// Always 64-bit.
			//
			{PPC_REG_F0, f64},
			{PPC_REG_F1, f64},
			{PPC_REG_F2, f64},
			{PPC_REG_F3, f64},
			{PPC_REG_F4, f64},
			{PPC_REG_F5, f64},
			{PPC_REG_F6, f64},
			{PPC_REG_F7, f64},
			{PPC_REG_F8, f64},
			{PPC_REG_F9, f64},
			{PPC_REG_F10, f64},
			{PPC_REG_F11, f64},
			{PPC_REG_F12, f64},
			{PPC_REG_F13, f64},
			{PPC_REG_F14, f64},
			{PPC_REG_F15, f64},
			{PPC_REG_F16, f64},
			{PPC_REG_F17, f64},
			{PPC_REG_F18, f64},
			{PPC_REG_F19, f64},
			{PPC_REG_F20, f64},
			{PPC_REG_F21, f64},
			{PPC_REG_F22, f64},
			{PPC_REG_F23, f64},
			{PPC_REG_F24, f64},
			{PPC_REG_F25, f64},
			{PPC_REG_F26, f64},
			{PPC_REG_F27, f64},
			{PPC_REG_F28, f64},
			{PPC_REG_F29, f64},
			{PPC_REG_F30, f64},
			{PPC_REG_F31, f64},

			// Link register.
			// 32-bit in 32-bit implementation, 64-bit in 64-bit.
			//
			{PPC_REG_LR, defTy},

			// General-purpose registers.
			// 32-bit in 32-bit implementation, 64-bit in 64-bit.
			//
			{PPC_REG_R0, defTy},
			{PPC_REG_R1, defTy},
			{PPC_REG_R2, defTy},
			{PPC_REG_R3, defTy},
			{PPC_REG_R4, defTy},
			{PPC_REG_R5, defTy},
			{PPC_REG_R6, defTy},
			{PPC_REG_R7, defTy},
			{PPC_REG_R8, defTy},
			{PPC_REG_R9, defTy},
			{PPC_REG_R10, defTy},
			{PPC_REG_R11, defTy},
			{PPC_REG_R12, defTy},
			{PPC_REG_R13, defTy},
			{PPC_REG_R14, defTy},
			{PPC_REG_R15, defTy},
			{PPC_REG_R16, defTy},
			{PPC_REG_R17, defTy},
			{PPC_REG_R18, defTy},
			{PPC_REG_R19, defTy},
			{PPC_REG_R20, defTy},
			{PPC_REG_R21, defTy},
			{PPC_REG_R22, defTy},
			{PPC_REG_R23, defTy},
			{PPC_REG_R24, defTy},
			{PPC_REG_R25, defTy},
			{PPC_REG_R26, defTy},
			{PPC_REG_R27, defTy},
			{PPC_REG_R28, defTy},
			{PPC_REG_R29, defTy},
			{PPC_REG_R30, defTy},
			{PPC_REG_R31, defTy},

			// Vector registers.
			//
			{PPC_REG_V0, i128},
			{PPC_REG_V1, i128},
			{PPC_REG_V2, i128},
			{PPC_REG_V3, i128},
			{PPC_REG_V4, i128},
			{PPC_REG_V5, i128},
			{PPC_REG_V6, i128},
			{PPC_REG_V7, i128},
			{PPC_REG_V8, i128},
			{PPC_REG_V9, i128},
			{PPC_REG_V10, i128},
			{PPC_REG_V11, i128},
			{PPC_REG_V12, i128},
			{PPC_REG_V13, i128},
			{PPC_REG_V14, i128},
			{PPC_REG_V15, i128},
			{PPC_REG_V16, i128},
			{PPC_REG_V17, i128},
			{PPC_REG_V18, i128},
			{PPC_REG_V19, i128},
			{PPC_REG_V20, i128},
			{PPC_REG_V21, i128},
			{PPC_REG_V22, i128},
			{PPC_REG_V23, i128},
			{PPC_REG_V24, i128},
			{PPC_REG_V25, i128},
			{PPC_REG_V26, i128},
			{PPC_REG_V27, i128},
			{PPC_REG_V28, i128},
			{PPC_REG_V29, i128},
			{PPC_REG_V30, i128},
			{PPC_REG_V31, i128},
			{PPC_REG_VRSAVE, defTy},

			// More vector registers.
			//
			{PPC_REG_VS0, i128},
			{PPC_REG_VS1, i128},
			{PPC_REG_VS2, i128},
			{PPC_REG_VS3, i128},
			{PPC_REG_VS4, i128},
			{PPC_REG_VS5, i128},
			{PPC_REG_VS6, i128},
			{PPC_REG_VS7, i128},
			{PPC_REG_VS8, i128},
			{PPC_REG_VS9, i128},
			{PPC_REG_VS10, i128},
			{PPC_REG_VS11, i128},
			{PPC_REG_VS12, i128},
			{PPC_REG_VS13, i128},
			{PPC_REG_VS14, i128},
			{PPC_REG_VS15, i128},
			{PPC_REG_VS16, i128},
			{PPC_REG_VS17, i128},
			{PPC_REG_VS18, i128},
			{PPC_REG_VS19, i128},
			{PPC_REG_VS20, i128},
			{PPC_REG_VS21, i128},
			{PPC_REG_VS22, i128},
			{PPC_REG_VS23, i128},
			{PPC_REG_VS24, i128},
			{PPC_REG_VS25, i128},
			{PPC_REG_VS26, i128},
			{PPC_REG_VS27, i128},
			{PPC_REG_VS28, i128},
			{PPC_REG_VS29, i128},
			{PPC_REG_VS30, i128},
			{PPC_REG_VS31, i128},
			{PPC_REG_VS32, i128},
			{PPC_REG_VS33, i128},
			{PPC_REG_VS34, i128},
			{PPC_REG_VS35, i128},
			{PPC_REG_VS36, i128},
			{PPC_REG_VS37, i128},
			{PPC_REG_VS38, i128},
			{PPC_REG_VS39, i128},
			{PPC_REG_VS40, i128},
			{PPC_REG_VS41, i128},
			{PPC_REG_VS42, i128},
			{PPC_REG_VS43, i128},
			{PPC_REG_VS44, i128},
			{PPC_REG_VS45, i128},
			{PPC_REG_VS46, i128},
			{PPC_REG_VS47, i128},
			{PPC_REG_VS48, i128},
			{PPC_REG_VS49, i128},
			{PPC_REG_VS50, i128},
			{PPC_REG_VS51, i128},
			{PPC_REG_VS52, i128},
			{PPC_REG_VS53, i128},
			{PPC_REG_VS54, i128},
			{PPC_REG_VS55, i128},
			{PPC_REG_VS56, i128},
			{PPC_REG_VS57, i128},
			{PPC_REG_VS58, i128},
			{PPC_REG_VS59, i128},
			{PPC_REG_VS60, i128},
			{PPC_REG_VS61, i128},
			{PPC_REG_VS62, i128},
			{PPC_REG_VS63, i128},

			// Quad floating point registers.
			// Each 256-bits = 4 x 64-bit double.
			// We can not create such huge floating point data types in LLVM,
			// so we use 256-bit integers instead.
			//
			{PPC_REG_Q0, i256},
			{PPC_REG_Q1, i256},
			{PPC_REG_Q2, i256},
			{PPC_REG_Q3, i256},
			{PPC_REG_Q4, i256},
			{PPC_REG_Q5, i256},
			{PPC_REG_Q6, i256},
			{PPC_REG_Q7, i256},
			{PPC_REG_Q8, i256},
			{PPC_REG_Q9, i256},
			{PPC_REG_Q10, i256},
			{PPC_REG_Q11, i256},
			{PPC_REG_Q12, i256},
			{PPC_REG_Q13, i256},
			{PPC_REG_Q14, i256},
			{PPC_REG_Q15, i256},
			{PPC_REG_Q16, i256},
			{PPC_REG_Q17, i256},
			{PPC_REG_Q18, i256},
			{PPC_REG_Q19, i256},
			{PPC_REG_Q20, i256},
			{PPC_REG_Q21, i256},
			{PPC_REG_Q22, i256},
			{PPC_REG_Q23, i256},
			{PPC_REG_Q24, i256},
			{PPC_REG_Q25, i256},
			{PPC_REG_Q26, i256},
			{PPC_REG_Q27, i256},
			{PPC_REG_Q28, i256},
			{PPC_REG_Q29, i256},
			{PPC_REG_Q30, i256},
			{PPC_REG_Q31, i256},

			// extra registers for PPCMapping.c
			{PPC_REG_RM, defTy},
			{PPC_REG_CTR8, defTy},
			{PPC_REG_LR8, defTy},
			{PPC_REG_CR1EQ, defTy},
			{PPC_REG_X2, defTy},
	};

	_reg2type = std::move(r2t);
}

void Capstone2LlvmIrTranslatorPowerpc_impl::initializePseudoCallInstructionIDs()
{
	_callInsnIds =
	{
			// On PowerPC, it is hard to tell what the branch instruction
			// actually do without very complex analysis. See:
			// Capstone2LlvmIrTranslatorPowerpc_impl::translateB
	};

	_returnInsnIds =
	{
			// On PowerPC, it is hard to tell what the branch instruction
			// actually do without very complex analysis. See:
			// Capstone2LlvmIrTranslatorPowerpc_impl::translateB
	};

	_branchInsnIds =
	{
			// On PowerPC, it is hard to tell what the branch instruction
			// actually do without very complex analysis. See:
			// Capstone2LlvmIrTranslatorPowerpc_impl::translateB
	};

	_condBranchInsnIds =
	{
			// On PowerPC, it is hard to tell what the branch instruction
			// actually do without very complex analysis. See:
			// Capstone2LlvmIrTranslatorPowerpc_impl::translateB
	};

	_controlFlowInsnIds =
	{
			 PPC_INS_B,
			 PPC_INS_BA,
			 PPC_INS_BL,
			 PPC_INS_BLA,

			 PPC_INS_BC,
			 PPC_INS_BCA,
			 PPC_INS_BCLR,
			 PPC_INS_BCCTR,
			 PPC_INS_BCL,
			 PPC_INS_BCLA,
			 PPC_INS_BCLRL,
			 PPC_INS_BCCTRL,

			 PPC_INS_BLR,
			 PPC_INS_BCTR,
			 PPC_INS_BLRL,
			 PPC_INS_BCTRL,

			 PPC_INS_BT,
			 PPC_INS_BTA,
			 PPC_INS_BTLR,
			 PPC_INS_BTCTR,
			 PPC_INS_BTL,
			 PPC_INS_BTLA,
			 PPC_INS_BTLRL,
			 PPC_INS_BTCTRL,

			 PPC_INS_BF,
			 PPC_INS_BFA,
			 PPC_INS_BFLR,
			 PPC_INS_BFCTR,
			 PPC_INS_BFL,
			 PPC_INS_BFLA,
			 PPC_INS_BFLRL,
			 PPC_INS_BFCTRL,

			 PPC_INS_BDNZ,
			 PPC_INS_BDNZA,
			 PPC_INS_BDNZLR,
			 PPC_INS_BDNZL,
			 PPC_INS_BDNZLA,
			 PPC_INS_BDNZLRL,

			 PPC_INS_BDNZT,
			 PPC_INS_BDNZTA,
			 PPC_INS_BDNZTLR,
			 PPC_INS_BDNZTL,
			 PPC_INS_BDNZTLA,
			 PPC_INS_BDNZTLRL,

			 PPC_INS_BDNZF,
			 PPC_INS_BDNZFA,
//			 PPC_INS_BDNZFLR,
			 PPC_INS_BDNZFL,
			 PPC_INS_BDNZFLA,
			 PPC_INS_BDNZFLRL,

			 PPC_INS_BDZ,
			 PPC_INS_BDZA,
			 PPC_INS_BDZLR,
			 PPC_INS_BDZL,
			 PPC_INS_BDZLA,
			 PPC_INS_BDZLRL,

			 PPC_INS_BDZT,
			 PPC_INS_BDZTA,
			 PPC_INS_BDZTLR,
			 PPC_INS_BDZTL,
			 PPC_INS_BDZTLA,
			 PPC_INS_BDZTLRL,

			 PPC_INS_BDZF,
			 PPC_INS_BDZFA,
			 PPC_INS_BDZFLR,
			 PPC_INS_BDZFL,
			 PPC_INS_BDZFLA,
			 PPC_INS_BDZFLRL,
	};
}

//
//==============================================================================
// Instruction translation map initialization.
//==============================================================================
//

std::map<
	std::size_t,
	void (Capstone2LlvmIrTranslatorPowerpc_impl::*)(
			cs_insn* i,
			cs_ppc*,
			llvm::IRBuilder<>&)>
Capstone2LlvmIrTranslatorPowerpc_impl::_i2fm =
{
		{PPC_INS_INVALID, nullptr},

		{PPC_INS_ADD, &Capstone2LlvmIrTranslatorPowerpc_impl::translateAdd},
		{PPC_INS_ADDC, &Capstone2LlvmIrTranslatorPowerpc_impl::translateAddc},
		{PPC_INS_ADDE, &Capstone2LlvmIrTranslatorPowerpc_impl::translateAdde},
		{PPC_INS_ADDI, &Capstone2LlvmIrTranslatorPowerpc_impl::translateAdd},
		{PPC_INS_ADDIC, &Capstone2LlvmIrTranslatorPowerpc_impl::translateAddc},
		{PPC_INS_ADDIS, &Capstone2LlvmIrTranslatorPowerpc_impl::translateAddis},
		{PPC_INS_ADDME, &Capstone2LlvmIrTranslatorPowerpc_impl::translateAddme},
		{PPC_INS_ADDZE, &Capstone2LlvmIrTranslatorPowerpc_impl::translateAddze},
		{PPC_INS_AND, &Capstone2LlvmIrTranslatorPowerpc_impl::translateAnd},
		{PPC_INS_ANDC, &Capstone2LlvmIrTranslatorPowerpc_impl::translateAndc},
		{PPC_INS_ANDIS, &Capstone2LlvmIrTranslatorPowerpc_impl::translateAndis},
		{PPC_INS_ANDI, &Capstone2LlvmIrTranslatorPowerpc_impl::translateAnd},
		{PPC_INS_ATTN, nullptr},
		{PPC_INS_BRINC, nullptr},
		{PPC_INS_CMPB, nullptr},
		{PPC_INS_CMPD, &Capstone2LlvmIrTranslatorPowerpc_impl::translateCmp},
		{PPC_INS_CMPDI, &Capstone2LlvmIrTranslatorPowerpc_impl::translateCmp},
		{PPC_INS_CMPLD, &Capstone2LlvmIrTranslatorPowerpc_impl::translateCmp},
		{PPC_INS_CMPLDI, &Capstone2LlvmIrTranslatorPowerpc_impl::translateCmp},
		{PPC_INS_CMPLW, &Capstone2LlvmIrTranslatorPowerpc_impl::translateCmp},
		{PPC_INS_CMPLWI, &Capstone2LlvmIrTranslatorPowerpc_impl::translateCmp},
		{PPC_INS_CMPW, &Capstone2LlvmIrTranslatorPowerpc_impl::translateCmp},
		{PPC_INS_CMPWI, &Capstone2LlvmIrTranslatorPowerpc_impl::translateCmp},
		{PPC_INS_CNTLZD, nullptr},
		{PPC_INS_CNTLZW, &Capstone2LlvmIrTranslatorPowerpc_impl::translateCntlzw},
		{PPC_INS_CREQV, &Capstone2LlvmIrTranslatorPowerpc_impl::translateCrModifTernary},
		{PPC_INS_CRXOR, &Capstone2LlvmIrTranslatorPowerpc_impl::translateCrModifTernary},
		{PPC_INS_CRAND, &Capstone2LlvmIrTranslatorPowerpc_impl::translateCrModifTernary},
		{PPC_INS_CRANDC, &Capstone2LlvmIrTranslatorPowerpc_impl::translateCrModifTernary},
		{PPC_INS_CRNAND, &Capstone2LlvmIrTranslatorPowerpc_impl::translateCrModifTernary},
		{PPC_INS_CRNOR, &Capstone2LlvmIrTranslatorPowerpc_impl::translateCrModifTernary},
		{PPC_INS_CROR, &Capstone2LlvmIrTranslatorPowerpc_impl::translateCrModifTernary},
		{PPC_INS_CRORC, &Capstone2LlvmIrTranslatorPowerpc_impl::translateCrModifTernary},
		{PPC_INS_DCBA, nullptr},
		{PPC_INS_DCBF, nullptr},
		{PPC_INS_DCBI, nullptr},
		{PPC_INS_DCBST, nullptr},
		{PPC_INS_DCBT, nullptr},
		{PPC_INS_DCBTST, nullptr},
		{PPC_INS_DCBZ, nullptr},
		{PPC_INS_DCBZL, nullptr},
		{PPC_INS_DCCCI, nullptr},
		{PPC_INS_DIVD, nullptr},
		{PPC_INS_DIVDU, nullptr},
		{PPC_INS_DIVW, &Capstone2LlvmIrTranslatorPowerpc_impl::translateDivw},
		{PPC_INS_DIVWU, &Capstone2LlvmIrTranslatorPowerpc_impl::translateDivw},
		{PPC_INS_DSS, nullptr},
		{PPC_INS_DSSALL, nullptr},
		{PPC_INS_DST, nullptr},
		{PPC_INS_DSTST, nullptr},
		{PPC_INS_DSTSTT, nullptr},
		{PPC_INS_DSTT, nullptr},
		{PPC_INS_EQV, &Capstone2LlvmIrTranslatorPowerpc_impl::translateEqv},
		{PPC_INS_EVABS, nullptr},
		{PPC_INS_EVADDIW, nullptr},
		{PPC_INS_EVADDSMIAAW, nullptr},
		{PPC_INS_EVADDSSIAAW, nullptr},
		{PPC_INS_EVADDUMIAAW, nullptr},
		{PPC_INS_EVADDUSIAAW, nullptr},
		{PPC_INS_EVADDW, nullptr},
		{PPC_INS_EVAND, nullptr},
		{PPC_INS_EVANDC, nullptr},
		{PPC_INS_EVCMPEQ, nullptr},
		{PPC_INS_EVCMPGTS, nullptr},
		{PPC_INS_EVCMPGTU, nullptr},
		{PPC_INS_EVCMPLTS, nullptr},
		{PPC_INS_EVCMPLTU, nullptr},
		{PPC_INS_EVCNTLSW, nullptr},
		{PPC_INS_EVCNTLZW, nullptr},
		{PPC_INS_EVDIVWS, nullptr},
		{PPC_INS_EVDIVWU, nullptr},
		{PPC_INS_EVEQV, nullptr},
		{PPC_INS_EVEXTSB, nullptr},
		{PPC_INS_EVEXTSH, nullptr},
		{PPC_INS_EVLDD, nullptr},
		{PPC_INS_EVLDDX, nullptr},
		{PPC_INS_EVLDH, nullptr},
		{PPC_INS_EVLDHX, nullptr},
		{PPC_INS_EVLDW, nullptr},
		{PPC_INS_EVLDWX, nullptr},
		{PPC_INS_EVLHHESPLAT, nullptr},
		{PPC_INS_EVLHHESPLATX, nullptr},
		{PPC_INS_EVLHHOSSPLAT, nullptr},
		{PPC_INS_EVLHHOSSPLATX, nullptr},
		{PPC_INS_EVLHHOUSPLAT, nullptr},
		{PPC_INS_EVLHHOUSPLATX, nullptr},
		{PPC_INS_EVLWHE, nullptr},
		{PPC_INS_EVLWHEX, nullptr},
		{PPC_INS_EVLWHOS, nullptr},
		{PPC_INS_EVLWHOSX, nullptr},
		{PPC_INS_EVLWHOU, nullptr},
		{PPC_INS_EVLWHOUX, nullptr},
		{PPC_INS_EVLWHSPLAT, nullptr},
		{PPC_INS_EVLWHSPLATX, nullptr},
		{PPC_INS_EVLWWSPLAT, nullptr},
		{PPC_INS_EVLWWSPLATX, nullptr},
		{PPC_INS_EVMERGEHI, nullptr},
		{PPC_INS_EVMERGEHILO, nullptr},
		{PPC_INS_EVMERGELO, nullptr},
		{PPC_INS_EVMERGELOHI, nullptr},
		{PPC_INS_EVMHEGSMFAA, nullptr},
		{PPC_INS_EVMHEGSMFAN, nullptr},
		{PPC_INS_EVMHEGSMIAA, nullptr},
		{PPC_INS_EVMHEGSMIAN, nullptr},
		{PPC_INS_EVMHEGUMIAA, nullptr},
		{PPC_INS_EVMHEGUMIAN, nullptr},
		{PPC_INS_EVMHESMF, nullptr},
		{PPC_INS_EVMHESMFA, nullptr},
		{PPC_INS_EVMHESMFAAW, nullptr},
		{PPC_INS_EVMHESMFANW, nullptr},
		{PPC_INS_EVMHESMI, nullptr},
		{PPC_INS_EVMHESMIA, nullptr},
		{PPC_INS_EVMHESMIAAW, nullptr},
		{PPC_INS_EVMHESMIANW, nullptr},
		{PPC_INS_EVMHESSF, nullptr},
		{PPC_INS_EVMHESSFA, nullptr},
		{PPC_INS_EVMHESSFAAW, nullptr},
		{PPC_INS_EVMHESSFANW, nullptr},
		{PPC_INS_EVMHESSIAAW, nullptr},
		{PPC_INS_EVMHESSIANW, nullptr},
		{PPC_INS_EVMHEUMI, nullptr},
		{PPC_INS_EVMHEUMIA, nullptr},
		{PPC_INS_EVMHEUMIAAW, nullptr},
		{PPC_INS_EVMHEUMIANW, nullptr},
		{PPC_INS_EVMHEUSIAAW, nullptr},
		{PPC_INS_EVMHEUSIANW, nullptr},
		{PPC_INS_EVMHOGSMFAA, nullptr},
		{PPC_INS_EVMHOGSMFAN, nullptr},
		{PPC_INS_EVMHOGSMIAA, nullptr},
		{PPC_INS_EVMHOGSMIAN, nullptr},
		{PPC_INS_EVMHOGUMIAA, nullptr},
		{PPC_INS_EVMHOGUMIAN, nullptr},
		{PPC_INS_EVMHOSMF, nullptr},
		{PPC_INS_EVMHOSMFA, nullptr},
		{PPC_INS_EVMHOSMFAAW, nullptr},
		{PPC_INS_EVMHOSMFANW, nullptr},
		{PPC_INS_EVMHOSMI, nullptr},
		{PPC_INS_EVMHOSMIA, nullptr},
		{PPC_INS_EVMHOSMIAAW, nullptr},
		{PPC_INS_EVMHOSMIANW, nullptr},
		{PPC_INS_EVMHOSSF, nullptr},
		{PPC_INS_EVMHOSSFA, nullptr},
		{PPC_INS_EVMHOSSFAAW, nullptr},
		{PPC_INS_EVMHOSSFANW, nullptr},
		{PPC_INS_EVMHOSSIAAW, nullptr},
		{PPC_INS_EVMHOSSIANW, nullptr},
		{PPC_INS_EVMHOUMI, nullptr},
		{PPC_INS_EVMHOUMIA, nullptr},
		{PPC_INS_EVMHOUMIAAW, nullptr},
		{PPC_INS_EVMHOUMIANW, nullptr},
		{PPC_INS_EVMHOUSIAAW, nullptr},
		{PPC_INS_EVMHOUSIANW, nullptr},
		{PPC_INS_EVMRA, nullptr},
		{PPC_INS_EVMWHSMF, nullptr},
		{PPC_INS_EVMWHSMFA, nullptr},
		{PPC_INS_EVMWHSMI, nullptr},
		{PPC_INS_EVMWHSMIA, nullptr},
		{PPC_INS_EVMWHSSF, nullptr},
		{PPC_INS_EVMWHSSFA, nullptr},
		{PPC_INS_EVMWHUMI, nullptr},
		{PPC_INS_EVMWHUMIA, nullptr},
		{PPC_INS_EVMWLSMIAAW, nullptr},
		{PPC_INS_EVMWLSMIANW, nullptr},
		{PPC_INS_EVMWLSSIAAW, nullptr},
		{PPC_INS_EVMWLSSIANW, nullptr},
		{PPC_INS_EVMWLUMI, nullptr},
		{PPC_INS_EVMWLUMIA, nullptr},
		{PPC_INS_EVMWLUMIAAW, nullptr},
		{PPC_INS_EVMWLUMIANW, nullptr},
		{PPC_INS_EVMWLUSIAAW, nullptr},
		{PPC_INS_EVMWLUSIANW, nullptr},
		{PPC_INS_EVMWSMF, nullptr},
		{PPC_INS_EVMWSMFA, nullptr},
		{PPC_INS_EVMWSMFAA, nullptr},
		{PPC_INS_EVMWSMFAN, nullptr},
		{PPC_INS_EVMWSMI, nullptr},
		{PPC_INS_EVMWSMIA, nullptr},
		{PPC_INS_EVMWSMIAA, nullptr},
		{PPC_INS_EVMWSMIAN, nullptr},
		{PPC_INS_EVMWSSF, nullptr},
		{PPC_INS_EVMWSSFA, nullptr},
		{PPC_INS_EVMWSSFAA, nullptr},
		{PPC_INS_EVMWSSFAN, nullptr},
		{PPC_INS_EVMWUMI, nullptr},
		{PPC_INS_EVMWUMIA, nullptr},
		{PPC_INS_EVMWUMIAA, nullptr},
		{PPC_INS_EVMWUMIAN, nullptr},
		{PPC_INS_EVNAND, nullptr},
		{PPC_INS_EVNEG, nullptr},
		{PPC_INS_EVNOR, nullptr},
		{PPC_INS_EVOR, nullptr},
		{PPC_INS_EVORC, nullptr},
		{PPC_INS_EVRLW, nullptr},
		{PPC_INS_EVRLWI, nullptr},
		{PPC_INS_EVRNDW, nullptr},
		{PPC_INS_EVSLW, nullptr},
		{PPC_INS_EVSLWI, nullptr},
		{PPC_INS_EVSPLATFI, nullptr},
		{PPC_INS_EVSPLATI, nullptr},
		{PPC_INS_EVSRWIS, nullptr},
		{PPC_INS_EVSRWIU, nullptr},
		{PPC_INS_EVSRWS, nullptr},
		{PPC_INS_EVSRWU, nullptr},
		{PPC_INS_EVSTDD, nullptr},
		{PPC_INS_EVSTDDX, nullptr},
		{PPC_INS_EVSTDH, nullptr},
		{PPC_INS_EVSTDHX, nullptr},
		{PPC_INS_EVSTDW, nullptr},
		{PPC_INS_EVSTDWX, nullptr},
		{PPC_INS_EVSTWHE, nullptr},
		{PPC_INS_EVSTWHEX, nullptr},
		{PPC_INS_EVSTWHO, nullptr},
		{PPC_INS_EVSTWHOX, nullptr},
		{PPC_INS_EVSTWWE, nullptr},
		{PPC_INS_EVSTWWEX, nullptr},
		{PPC_INS_EVSTWWO, nullptr},
		{PPC_INS_EVSTWWOX, nullptr},
		{PPC_INS_EVSUBFSMIAAW, nullptr},
		{PPC_INS_EVSUBFSSIAAW, nullptr},
		{PPC_INS_EVSUBFUMIAAW, nullptr},
		{PPC_INS_EVSUBFUSIAAW, nullptr},
		{PPC_INS_EVSUBFW, nullptr},
		{PPC_INS_EVSUBIFW, nullptr},
		{PPC_INS_EVXOR, nullptr},
		{PPC_INS_EXTSB, &Capstone2LlvmIrTranslatorPowerpc_impl::translateExtendSign},
		{PPC_INS_EXTSH, &Capstone2LlvmIrTranslatorPowerpc_impl::translateExtendSign},
		{PPC_INS_EXTSW, &Capstone2LlvmIrTranslatorPowerpc_impl::translateExtendSign},
		{PPC_INS_EIEIO, nullptr},
		{PPC_INS_FABS, nullptr},
		{PPC_INS_FADD, nullptr},
		{PPC_INS_FADDS, nullptr},
		{PPC_INS_FCFID, nullptr},
		{PPC_INS_FCFIDS, nullptr},
		{PPC_INS_FCFIDU, nullptr},
		{PPC_INS_FCFIDUS, nullptr},
		{PPC_INS_FCMPU, nullptr},
		{PPC_INS_FCPSGN, nullptr},
		{PPC_INS_FCTID, nullptr},
		{PPC_INS_FCTIDUZ, nullptr},
		{PPC_INS_FCTIDZ, nullptr},
		{PPC_INS_FCTIW, nullptr},
		{PPC_INS_FCTIWUZ, nullptr},
		{PPC_INS_FCTIWZ, nullptr},
		{PPC_INS_FDIV, nullptr},
		{PPC_INS_FDIVS, nullptr},
		{PPC_INS_FMADD, nullptr},
		{PPC_INS_FMADDS, nullptr},
		{PPC_INS_FMR, nullptr},
		{PPC_INS_FMSUB, nullptr},
		{PPC_INS_FMSUBS, nullptr},
		{PPC_INS_FMUL, nullptr},
		{PPC_INS_FMULS, nullptr},
		{PPC_INS_FNABS, nullptr},
		{PPC_INS_FNEG, nullptr},
		{PPC_INS_FNMADD, nullptr},
		{PPC_INS_FNMADDS, nullptr},
		{PPC_INS_FNMSUB, nullptr},
		{PPC_INS_FNMSUBS, nullptr},
		{PPC_INS_FRE, nullptr},
		{PPC_INS_FRES, nullptr},
		{PPC_INS_FRIM, nullptr},
		{PPC_INS_FRIN, nullptr},
		{PPC_INS_FRIP, nullptr},
		{PPC_INS_FRIZ, nullptr},
		{PPC_INS_FRSP, nullptr},
		{PPC_INS_FRSQRTE, nullptr},
		{PPC_INS_FRSQRTES, nullptr},
		{PPC_INS_FSEL, nullptr},
		{PPC_INS_FSQRT, nullptr},
		{PPC_INS_FSQRTS, nullptr},
		{PPC_INS_FSUB, nullptr},
		{PPC_INS_FSUBS, nullptr},
		{PPC_INS_ICBI, nullptr},
		{PPC_INS_ICBT, nullptr},
		{PPC_INS_ICCCI, nullptr},
		{PPC_INS_ISEL, nullptr},
		{PPC_INS_ISYNC, nullptr},

		{PPC_INS_LA, &Capstone2LlvmIrTranslatorPowerpc_impl::translateAdd},
		{PPC_INS_LBZ, &Capstone2LlvmIrTranslatorPowerpc_impl::translateLoad},
		{PPC_INS_LBZCIX, nullptr},
		{PPC_INS_LBZU, &Capstone2LlvmIrTranslatorPowerpc_impl::translateLoad},
		{PPC_INS_LBZUX, &Capstone2LlvmIrTranslatorPowerpc_impl::translateLoadIndexed},
		{PPC_INS_LBZX, &Capstone2LlvmIrTranslatorPowerpc_impl::translateLoadIndexed},
		{PPC_INS_LD, nullptr},
		{PPC_INS_LDARX, nullptr},
		{PPC_INS_LDBRX, nullptr},
		{PPC_INS_LDCIX, nullptr},
		{PPC_INS_LDU, nullptr},
		{PPC_INS_LDUX, nullptr},
		{PPC_INS_LDX, nullptr},
		{PPC_INS_LFD, nullptr},
		{PPC_INS_LFDU, nullptr},
		{PPC_INS_LFDUX, nullptr},
		{PPC_INS_LFDX, nullptr},
		{PPC_INS_LFIWAX, nullptr},
		{PPC_INS_LFIWZX, nullptr},
		{PPC_INS_LFS, nullptr},
		{PPC_INS_LFSU, nullptr},
		{PPC_INS_LFSUX, nullptr},
		{PPC_INS_LFSX, nullptr},
		{PPC_INS_LHA, &Capstone2LlvmIrTranslatorPowerpc_impl::translateLoad},
		{PPC_INS_LHAU, &Capstone2LlvmIrTranslatorPowerpc_impl::translateLoad},
		{PPC_INS_LHAUX, &Capstone2LlvmIrTranslatorPowerpc_impl::translateLoadIndexed},
		{PPC_INS_LHAX, &Capstone2LlvmIrTranslatorPowerpc_impl::translateLoadIndexed},
		{PPC_INS_LHBRX, &Capstone2LlvmIrTranslatorPowerpc_impl::translateLhbrx},
		{PPC_INS_LHZ, &Capstone2LlvmIrTranslatorPowerpc_impl::translateLoad},
		{PPC_INS_LHZCIX, nullptr},
		{PPC_INS_LHZU, &Capstone2LlvmIrTranslatorPowerpc_impl::translateLoad},
		{PPC_INS_LHZUX, &Capstone2LlvmIrTranslatorPowerpc_impl::translateLoadIndexed},
		{PPC_INS_LHZX, &Capstone2LlvmIrTranslatorPowerpc_impl::translateLoadIndexed},
		{PPC_INS_LI, &Capstone2LlvmIrTranslatorPowerpc_impl::translateLi},
		{PPC_INS_LIS, &Capstone2LlvmIrTranslatorPowerpc_impl::translateLis},
		{PPC_INS_LMW, nullptr},
		{PPC_INS_LSWI, nullptr},
		{PPC_INS_LVEBX, nullptr},
		{PPC_INS_LVEHX, nullptr},
		{PPC_INS_LVEWX, nullptr},
		{PPC_INS_LVSL, nullptr},
		{PPC_INS_LVSR, nullptr},
		{PPC_INS_LVX, nullptr},
		{PPC_INS_LVXL, nullptr},
		{PPC_INS_LWA, nullptr},
		{PPC_INS_LWARX, nullptr},
		{PPC_INS_LWAUX, nullptr},
		{PPC_INS_LWAX, nullptr},
		{PPC_INS_LWBRX, &Capstone2LlvmIrTranslatorPowerpc_impl::translatePseudoAsmOp0FncOp1Op2},
		{PPC_INS_LWZ, &Capstone2LlvmIrTranslatorPowerpc_impl::translateLoad},
		{PPC_INS_LWZCIX, nullptr},
		{PPC_INS_LWZU, &Capstone2LlvmIrTranslatorPowerpc_impl::translateLoad},
		{PPC_INS_LWZUX, &Capstone2LlvmIrTranslatorPowerpc_impl::translateLoadIndexed},
		{PPC_INS_LWZX, &Capstone2LlvmIrTranslatorPowerpc_impl::translateLoadIndexed},

		{PPC_INS_LXSDX, nullptr},
		{PPC_INS_LXVD2X, nullptr},
		{PPC_INS_LXVDSX, nullptr},
		{PPC_INS_LXVW4X, nullptr},
		{PPC_INS_MBAR, nullptr},
		{PPC_INS_MCRF, &Capstone2LlvmIrTranslatorPowerpc_impl::translateMcrf},
		{PPC_INS_MCRFS, nullptr},
		{PPC_INS_MFCR, &Capstone2LlvmIrTranslatorPowerpc_impl::translatePseudoAsmOp0Fnc},
		{PPC_INS_MFCTR, &Capstone2LlvmIrTranslatorPowerpc_impl::translateMfctr},
		{PPC_INS_MFDCR, nullptr},
		{PPC_INS_MFFS, nullptr},
		{PPC_INS_MFLR, &Capstone2LlvmIrTranslatorPowerpc_impl::translateMflr},
		{PPC_INS_MFMSR, nullptr},
		{PPC_INS_MFOCRF, nullptr},
		{PPC_INS_MFSPR, &Capstone2LlvmIrTranslatorPowerpc_impl::translatePseudoAsmOp0FncOp1},
		{PPC_INS_MFSR, nullptr},
		{PPC_INS_MFSRIN, nullptr},
		{PPC_INS_MFTB, nullptr},
		{PPC_INS_MFVSCR, nullptr},
		{PPC_INS_MSYNC, nullptr},
		{PPC_INS_MTCRF, &Capstone2LlvmIrTranslatorPowerpc_impl::translateMtcrf},
		{PPC_INS_MTCTR, &Capstone2LlvmIrTranslatorPowerpc_impl::translateMtctr},
		{PPC_INS_MTDCR, nullptr},
		{PPC_INS_MTFSB0, nullptr},
		{PPC_INS_MTFSB1, nullptr},
		{PPC_INS_MTFSF, nullptr},
		{PPC_INS_MTFSFI, nullptr},
		{PPC_INS_MTLR, &Capstone2LlvmIrTranslatorPowerpc_impl::translateMtlr},
		{PPC_INS_MTMSR, nullptr},
		{PPC_INS_MTMSRD, nullptr},
		{PPC_INS_MTOCRF, nullptr},
		{PPC_INS_MTSPR, &Capstone2LlvmIrTranslatorPowerpc_impl::translatePseudoAsmFncOp0Op1},
		{PPC_INS_MTSR, nullptr},
		{PPC_INS_MTSRIN, nullptr},
		{PPC_INS_MTVSCR, nullptr},
		{PPC_INS_MULHD, nullptr},
		{PPC_INS_MULHDU, nullptr},
		{PPC_INS_MULHW, &Capstone2LlvmIrTranslatorPowerpc_impl::translateMulhw},
		{PPC_INS_MULHWU, &Capstone2LlvmIrTranslatorPowerpc_impl::translateMulhw},
		{PPC_INS_MULLD, nullptr},
		{PPC_INS_MULLI, &Capstone2LlvmIrTranslatorPowerpc_impl::translateMullw},
		{PPC_INS_MULLW, &Capstone2LlvmIrTranslatorPowerpc_impl::translateMullw},
		{PPC_INS_NAND, &Capstone2LlvmIrTranslatorPowerpc_impl::translateNand},
		{PPC_INS_NEG, &Capstone2LlvmIrTranslatorPowerpc_impl::translateNeg},
		{PPC_INS_NOP, &Capstone2LlvmIrTranslatorPowerpc_impl::translateNop},
		{PPC_INS_ORI, &Capstone2LlvmIrTranslatorPowerpc_impl::translateOr},
		{PPC_INS_NOR, &Capstone2LlvmIrTranslatorPowerpc_impl::translateNor},
		{PPC_INS_OR, &Capstone2LlvmIrTranslatorPowerpc_impl::translateOr},
		{PPC_INS_ORC, &Capstone2LlvmIrTranslatorPowerpc_impl::translateOrc},
		{PPC_INS_ORIS, &Capstone2LlvmIrTranslatorPowerpc_impl::translateOris},
		{PPC_INS_POPCNTD, nullptr},
		{PPC_INS_POPCNTW, nullptr},
		{PPC_INS_QVALIGNI, nullptr},
		{PPC_INS_QVESPLATI, nullptr},
		{PPC_INS_QVFABS, nullptr},
		{PPC_INS_QVFADD, nullptr},
		{PPC_INS_QVFADDS, nullptr},
		{PPC_INS_QVFCFID, nullptr},
		{PPC_INS_QVFCFIDS, nullptr},
		{PPC_INS_QVFCFIDU, nullptr},
		{PPC_INS_QVFCFIDUS, nullptr},
		{PPC_INS_QVFCMPEQ, nullptr},
		{PPC_INS_QVFCMPGT, nullptr},
		{PPC_INS_QVFCMPLT, nullptr},
		{PPC_INS_QVFCPSGN, nullptr},
		{PPC_INS_QVFCTID, nullptr},
		{PPC_INS_QVFCTIDU, nullptr},
		{PPC_INS_QVFCTIDUZ, nullptr},
		{PPC_INS_QVFCTIDZ, nullptr},
		{PPC_INS_QVFCTIW, nullptr},
		{PPC_INS_QVFCTIWU, nullptr},
		{PPC_INS_QVFCTIWUZ, nullptr},
		{PPC_INS_QVFCTIWZ, nullptr},
		{PPC_INS_QVFLOGICAL, nullptr},
		{PPC_INS_QVFMADD, nullptr},
		{PPC_INS_QVFMADDS, nullptr},
		{PPC_INS_QVFMR, nullptr},
		{PPC_INS_QVFMSUB, nullptr},
		{PPC_INS_QVFMSUBS, nullptr},
		{PPC_INS_QVFMUL, nullptr},
		{PPC_INS_QVFMULS, nullptr},
		{PPC_INS_QVFNABS, nullptr},
		{PPC_INS_QVFNEG, nullptr},
		{PPC_INS_QVFNMADD, nullptr},
		{PPC_INS_QVFNMADDS, nullptr},
		{PPC_INS_QVFNMSUB, nullptr},
		{PPC_INS_QVFNMSUBS, nullptr},
		{PPC_INS_QVFPERM, nullptr},
		{PPC_INS_QVFRE, nullptr},
		{PPC_INS_QVFRES, nullptr},
		{PPC_INS_QVFRIM, nullptr},
		{PPC_INS_QVFRIN, nullptr},
		{PPC_INS_QVFRIP, nullptr},
		{PPC_INS_QVFRIZ, nullptr},
		{PPC_INS_QVFRSP, nullptr},
		{PPC_INS_QVFRSQRTE, nullptr},
		{PPC_INS_QVFRSQRTES, nullptr},
		{PPC_INS_QVFSEL, nullptr},
		{PPC_INS_QVFSUB, nullptr},
		{PPC_INS_QVFSUBS, nullptr},
		{PPC_INS_QVFTSTNAN, nullptr},
		{PPC_INS_QVFXMADD, nullptr},
		{PPC_INS_QVFXMADDS, nullptr},
		{PPC_INS_QVFXMUL, nullptr},
		{PPC_INS_QVFXMULS, nullptr},
		{PPC_INS_QVFXXCPNMADD, nullptr},
		{PPC_INS_QVFXXCPNMADDS, nullptr},
		{PPC_INS_QVFXXMADD, nullptr},
		{PPC_INS_QVFXXMADDS, nullptr},
		{PPC_INS_QVFXXNPMADD, nullptr},
		{PPC_INS_QVFXXNPMADDS, nullptr},
		{PPC_INS_QVGPCI, nullptr},
		{PPC_INS_QVLFCDUX, nullptr},
		{PPC_INS_QVLFCDUXA, nullptr},
		{PPC_INS_QVLFCDX, nullptr},
		{PPC_INS_QVLFCDXA, nullptr},
		{PPC_INS_QVLFCSUX, nullptr},
		{PPC_INS_QVLFCSUXA, nullptr},
		{PPC_INS_QVLFCSX, nullptr},
		{PPC_INS_QVLFCSXA, nullptr},
		{PPC_INS_QVLFDUX, nullptr},
		{PPC_INS_QVLFDUXA, nullptr},
		{PPC_INS_QVLFDX, nullptr},
		{PPC_INS_QVLFDXA, nullptr},
		{PPC_INS_QVLFIWAX, nullptr},
		{PPC_INS_QVLFIWAXA, nullptr},
		{PPC_INS_QVLFIWZX, nullptr},
		{PPC_INS_QVLFIWZXA, nullptr},
		{PPC_INS_QVLFSUX, nullptr},
		{PPC_INS_QVLFSUXA, nullptr},
		{PPC_INS_QVLFSX, nullptr},
		{PPC_INS_QVLFSXA, nullptr},
		{PPC_INS_QVLPCLDX, nullptr},
		{PPC_INS_QVLPCLSX, nullptr},
		{PPC_INS_QVLPCRDX, nullptr},
		{PPC_INS_QVLPCRSX, nullptr},
		{PPC_INS_QVSTFCDUX, nullptr},
		{PPC_INS_QVSTFCDUXA, nullptr},
		{PPC_INS_QVSTFCDUXI, nullptr},
		{PPC_INS_QVSTFCDUXIA, nullptr},
		{PPC_INS_QVSTFCDX, nullptr},
		{PPC_INS_QVSTFCDXA, nullptr},
		{PPC_INS_QVSTFCDXI, nullptr},
		{PPC_INS_QVSTFCDXIA, nullptr},
		{PPC_INS_QVSTFCSUX, nullptr},
		{PPC_INS_QVSTFCSUXA, nullptr},
		{PPC_INS_QVSTFCSUXI, nullptr},
		{PPC_INS_QVSTFCSUXIA, nullptr},
		{PPC_INS_QVSTFCSX, nullptr},
		{PPC_INS_QVSTFCSXA, nullptr},
		{PPC_INS_QVSTFCSXI, nullptr},
		{PPC_INS_QVSTFCSXIA, nullptr},
		{PPC_INS_QVSTFDUX, nullptr},
		{PPC_INS_QVSTFDUXA, nullptr},
		{PPC_INS_QVSTFDUXI, nullptr},
		{PPC_INS_QVSTFDUXIA, nullptr},
		{PPC_INS_QVSTFDX, nullptr},
		{PPC_INS_QVSTFDXA, nullptr},
		{PPC_INS_QVSTFDXI, nullptr},
		{PPC_INS_QVSTFDXIA, nullptr},
		{PPC_INS_QVSTFIWX, nullptr},
		{PPC_INS_QVSTFIWXA, nullptr},
		{PPC_INS_QVSTFSUX, nullptr},
		{PPC_INS_QVSTFSUXA, nullptr},
		{PPC_INS_QVSTFSUXI, nullptr},
		{PPC_INS_QVSTFSUXIA, nullptr},
		{PPC_INS_QVSTFSX, nullptr},
		{PPC_INS_QVSTFSXA, nullptr},
		{PPC_INS_QVSTFSXI, nullptr},
		{PPC_INS_QVSTFSXIA, nullptr},
		{PPC_INS_RFCI, nullptr},
		{PPC_INS_RFDI, nullptr},
		{PPC_INS_RFI, nullptr},
		{PPC_INS_RFID, nullptr},
		{PPC_INS_RFMCI, nullptr},
		{PPC_INS_RLDCL, nullptr},
		{PPC_INS_RLDCR, nullptr},
		{PPC_INS_RLDIC, nullptr},
		{PPC_INS_RLDICL, nullptr},
		{PPC_INS_RLDICR, nullptr},
		{PPC_INS_RLDIMI, nullptr},
		{PPC_INS_RLWIMI, &Capstone2LlvmIrTranslatorPowerpc_impl::translateRotateComplex5op},
		{PPC_INS_RLWINM, &Capstone2LlvmIrTranslatorPowerpc_impl::translateRotateComplex5op},
		{PPC_INS_RLWNM, &Capstone2LlvmIrTranslatorPowerpc_impl::translateRotateComplex5op},
		{PPC_INS_SC, nullptr},
		{PPC_INS_SLBIA, nullptr},
		{PPC_INS_SLBIE, nullptr},
		{PPC_INS_SLBMFEE, nullptr},
		{PPC_INS_SLBMTE, nullptr},
		{PPC_INS_SLD, nullptr},
		{PPC_INS_SLW, &Capstone2LlvmIrTranslatorPowerpc_impl::translateShiftLeft},
		{PPC_INS_SRAD, nullptr},
		{PPC_INS_SRADI, nullptr},
		{PPC_INS_SRAW, &Capstone2LlvmIrTranslatorPowerpc_impl::translateSraw},
		{PPC_INS_SRAWI, &Capstone2LlvmIrTranslatorPowerpc_impl::translateSraw},
		{PPC_INS_SRD, nullptr},
		{PPC_INS_SRW, &Capstone2LlvmIrTranslatorPowerpc_impl::translateShiftRight},
		{PPC_INS_STB, &Capstone2LlvmIrTranslatorPowerpc_impl::translateStore},
		{PPC_INS_STBCIX, nullptr},
		{PPC_INS_STBU, &Capstone2LlvmIrTranslatorPowerpc_impl::translateStore},
		{PPC_INS_STBUX, &Capstone2LlvmIrTranslatorPowerpc_impl::translateStoreIndexed},
		{PPC_INS_STBX, &Capstone2LlvmIrTranslatorPowerpc_impl::translateStoreIndexed},
		{PPC_INS_STD, nullptr},
		{PPC_INS_STDBRX, nullptr},
		{PPC_INS_STDCIX, nullptr},
		{PPC_INS_STDCX, nullptr},
		{PPC_INS_STDU, nullptr},
		{PPC_INS_STDUX, nullptr},
		{PPC_INS_STDX, nullptr},
		{PPC_INS_STFD, nullptr},
		{PPC_INS_STFDU, nullptr},
		{PPC_INS_STFDUX, nullptr},
		{PPC_INS_STFDX, nullptr},
		{PPC_INS_STFIWX, nullptr},
		{PPC_INS_STFS, nullptr},
		{PPC_INS_STFSU, nullptr},
		{PPC_INS_STFSUX, nullptr},
		{PPC_INS_STFSX, nullptr},
		{PPC_INS_STH, &Capstone2LlvmIrTranslatorPowerpc_impl::translateStore},
		{PPC_INS_STHBRX, &Capstone2LlvmIrTranslatorPowerpc_impl::translatePseudoAsmFncOp0Op1Op2},
		{PPC_INS_STHCIX, nullptr},
		{PPC_INS_STHU, &Capstone2LlvmIrTranslatorPowerpc_impl::translateStore},
		{PPC_INS_STHUX, &Capstone2LlvmIrTranslatorPowerpc_impl::translateStoreIndexed},
		{PPC_INS_STHX, &Capstone2LlvmIrTranslatorPowerpc_impl::translateStoreIndexed},
		{PPC_INS_STMW, nullptr},
		{PPC_INS_STSWI, nullptr},
		{PPC_INS_STVEBX, nullptr},
		{PPC_INS_STVEHX, nullptr},
		{PPC_INS_STVEWX, nullptr},
		{PPC_INS_STVX, nullptr},
		{PPC_INS_STVXL, nullptr},
		{PPC_INS_STW, &Capstone2LlvmIrTranslatorPowerpc_impl::translateStore},
		{PPC_INS_STWBRX, &Capstone2LlvmIrTranslatorPowerpc_impl::translatePseudoAsmFncOp0Op1Op2},
		{PPC_INS_STWCIX, nullptr},
		{PPC_INS_STWCX, nullptr},
		{PPC_INS_STWU, &Capstone2LlvmIrTranslatorPowerpc_impl::translateStore},
		{PPC_INS_STWUX, &Capstone2LlvmIrTranslatorPowerpc_impl::translateStoreIndexed},
		{PPC_INS_STWX, &Capstone2LlvmIrTranslatorPowerpc_impl::translateStoreIndexed},
		{PPC_INS_STXSDX, nullptr},
		{PPC_INS_STXVD2X, nullptr},
		{PPC_INS_STXVW4X, nullptr},
		{PPC_INS_SUBF, &Capstone2LlvmIrTranslatorPowerpc_impl::translateSubf},
		{PPC_INS_SUBFC, &Capstone2LlvmIrTranslatorPowerpc_impl::translateSubfc},
		{PPC_INS_SUBFE, &Capstone2LlvmIrTranslatorPowerpc_impl::translateSubfe},
		{PPC_INS_SUBFIC, &Capstone2LlvmIrTranslatorPowerpc_impl::translateSubfc},
		{PPC_INS_SUBFME, &Capstone2LlvmIrTranslatorPowerpc_impl::translateSubfme},
		{PPC_INS_SUBFZE, &Capstone2LlvmIrTranslatorPowerpc_impl::translateSubfze},
		{PPC_INS_SYNC, nullptr},
		{PPC_INS_TD, nullptr},
		{PPC_INS_TDI, nullptr},
		{PPC_INS_TLBIA, nullptr},
		{PPC_INS_TLBIE, nullptr},
		{PPC_INS_TLBIEL, nullptr},
		{PPC_INS_TLBIVAX, nullptr},
		{PPC_INS_TLBLD, nullptr},
		{PPC_INS_TLBLI, nullptr},
		{PPC_INS_TLBRE, nullptr},
		{PPC_INS_TLBSX, nullptr},
		{PPC_INS_TLBSYNC, nullptr},
		{PPC_INS_TLBWE, nullptr},
		{PPC_INS_TRAP, nullptr},
		{PPC_INS_TW, nullptr},
		{PPC_INS_TWI, nullptr},
		{PPC_INS_VADDCUW, nullptr},
		{PPC_INS_VADDFP, nullptr},
		{PPC_INS_VADDSBS, nullptr},
		{PPC_INS_VADDSHS, nullptr},
		{PPC_INS_VADDSWS, nullptr},
		{PPC_INS_VADDUBM, nullptr},
		{PPC_INS_VADDUBS, nullptr},
		{PPC_INS_VADDUDM, nullptr},
		{PPC_INS_VADDUHM, nullptr},
		{PPC_INS_VADDUHS, nullptr},
		{PPC_INS_VADDUWM, nullptr},
		{PPC_INS_VADDUWS, nullptr},
		{PPC_INS_VAND, nullptr},
		{PPC_INS_VANDC, nullptr},
		{PPC_INS_VAVGSB, nullptr},
		{PPC_INS_VAVGSH, nullptr},
		{PPC_INS_VAVGSW, nullptr},
		{PPC_INS_VAVGUB, nullptr},
		{PPC_INS_VAVGUH, nullptr},
		{PPC_INS_VAVGUW, nullptr},
		{PPC_INS_VCFSX, nullptr},
		{PPC_INS_VCFUX, nullptr},
		{PPC_INS_VCLZB, nullptr},
		{PPC_INS_VCLZD, nullptr},
		{PPC_INS_VCLZH, nullptr},
		{PPC_INS_VCLZW, nullptr},
		{PPC_INS_VCMPBFP, nullptr},
		{PPC_INS_VCMPEQFP, nullptr},
		{PPC_INS_VCMPEQUB, nullptr},
		{PPC_INS_VCMPEQUD, nullptr},
		{PPC_INS_VCMPEQUH, nullptr},
		{PPC_INS_VCMPEQUW, nullptr},
		{PPC_INS_VCMPGEFP, nullptr},
		{PPC_INS_VCMPGTFP, nullptr},
		{PPC_INS_VCMPGTSB, nullptr},
		{PPC_INS_VCMPGTSD, nullptr},
		{PPC_INS_VCMPGTSH, nullptr},
		{PPC_INS_VCMPGTSW, nullptr},
		{PPC_INS_VCMPGTUB, nullptr},
		{PPC_INS_VCMPGTUD, nullptr},
		{PPC_INS_VCMPGTUH, nullptr},
		{PPC_INS_VCMPGTUW, nullptr},
		{PPC_INS_VCTSXS, nullptr},
		{PPC_INS_VCTUXS, nullptr},
		{PPC_INS_VEQV, nullptr},
		{PPC_INS_VEXPTEFP, nullptr},
		{PPC_INS_VLOGEFP, nullptr},
		{PPC_INS_VMADDFP, nullptr},
		{PPC_INS_VMAXFP, nullptr},
		{PPC_INS_VMAXSB, nullptr},
		{PPC_INS_VMAXSD, nullptr},
		{PPC_INS_VMAXSH, nullptr},
		{PPC_INS_VMAXSW, nullptr},
		{PPC_INS_VMAXUB, nullptr},
		{PPC_INS_VMAXUD, nullptr},
		{PPC_INS_VMAXUH, nullptr},
		{PPC_INS_VMAXUW, nullptr},
		{PPC_INS_VMHADDSHS, nullptr},
		{PPC_INS_VMHRADDSHS, nullptr},
		{PPC_INS_VMINUD, nullptr},
		{PPC_INS_VMINFP, nullptr},
		{PPC_INS_VMINSB, nullptr},
		{PPC_INS_VMINSD, nullptr},
		{PPC_INS_VMINSH, nullptr},
		{PPC_INS_VMINSW, nullptr},
		{PPC_INS_VMINUB, nullptr},
		{PPC_INS_VMINUH, nullptr},
		{PPC_INS_VMINUW, nullptr},
		{PPC_INS_VMLADDUHM, nullptr},
		{PPC_INS_VMRGHB, nullptr},
		{PPC_INS_VMRGHH, nullptr},
		{PPC_INS_VMRGHW, nullptr},
		{PPC_INS_VMRGLB, nullptr},
		{PPC_INS_VMRGLH, nullptr},
		{PPC_INS_VMRGLW, nullptr},
		{PPC_INS_VMSUMMBM, nullptr},
		{PPC_INS_VMSUMSHM, nullptr},
		{PPC_INS_VMSUMSHS, nullptr},
		{PPC_INS_VMSUMUBM, nullptr},
		{PPC_INS_VMSUMUHM, nullptr},
		{PPC_INS_VMSUMUHS, nullptr},
		{PPC_INS_VMULESB, nullptr},
		{PPC_INS_VMULESH, nullptr},
		{PPC_INS_VMULESW, nullptr},
		{PPC_INS_VMULEUB, nullptr},
		{PPC_INS_VMULEUH, nullptr},
		{PPC_INS_VMULEUW, nullptr},
		{PPC_INS_VMULOSB, nullptr},
		{PPC_INS_VMULOSH, nullptr},
		{PPC_INS_VMULOSW, nullptr},
		{PPC_INS_VMULOUB, nullptr},
		{PPC_INS_VMULOUH, nullptr},
		{PPC_INS_VMULOUW, nullptr},
		{PPC_INS_VMULUWM, nullptr},
		{PPC_INS_VNAND, nullptr},
		{PPC_INS_VNMSUBFP, nullptr},
		{PPC_INS_VNOR, nullptr},
		{PPC_INS_VOR, nullptr},
		{PPC_INS_VORC, nullptr},
		{PPC_INS_VPERM, nullptr},
		{PPC_INS_VPKPX, nullptr},
		{PPC_INS_VPKSHSS, nullptr},
		{PPC_INS_VPKSHUS, nullptr},
		{PPC_INS_VPKSWSS, nullptr},
		{PPC_INS_VPKSWUS, nullptr},
		{PPC_INS_VPKUHUM, nullptr},
		{PPC_INS_VPKUHUS, nullptr},
		{PPC_INS_VPKUWUM, nullptr},
		{PPC_INS_VPKUWUS, nullptr},
		{PPC_INS_VPOPCNTB, nullptr},
		{PPC_INS_VPOPCNTD, nullptr},
		{PPC_INS_VPOPCNTH, nullptr},
		{PPC_INS_VPOPCNTW, nullptr},
		{PPC_INS_VREFP, nullptr},
		{PPC_INS_VRFIM, nullptr},
		{PPC_INS_VRFIN, nullptr},
		{PPC_INS_VRFIP, nullptr},
		{PPC_INS_VRFIZ, nullptr},
		{PPC_INS_VRLB, nullptr},
		{PPC_INS_VRLD, nullptr},
		{PPC_INS_VRLH, nullptr},
		{PPC_INS_VRLW, nullptr},
		{PPC_INS_VRSQRTEFP, nullptr},
		{PPC_INS_VSEL, nullptr},
		{PPC_INS_VSL, nullptr},
		{PPC_INS_VSLB, nullptr},
		{PPC_INS_VSLD, nullptr},
		{PPC_INS_VSLDOI, nullptr},
		{PPC_INS_VSLH, nullptr},
		{PPC_INS_VSLO, nullptr},
		{PPC_INS_VSLW, nullptr},
		{PPC_INS_VSPLTB, nullptr},
		{PPC_INS_VSPLTH, nullptr},
		{PPC_INS_VSPLTISB, nullptr},
		{PPC_INS_VSPLTISH, nullptr},
		{PPC_INS_VSPLTISW, nullptr},
		{PPC_INS_VSPLTW, nullptr},
		{PPC_INS_VSR, nullptr},
		{PPC_INS_VSRAB, nullptr},
		{PPC_INS_VSRAD, nullptr},
		{PPC_INS_VSRAH, nullptr},
		{PPC_INS_VSRAW, nullptr},
		{PPC_INS_VSRB, nullptr},
		{PPC_INS_VSRD, nullptr},
		{PPC_INS_VSRH, nullptr},
		{PPC_INS_VSRO, nullptr},
		{PPC_INS_VSRW, nullptr},
		{PPC_INS_VSUBCUW, nullptr},
		{PPC_INS_VSUBFP, nullptr},
		{PPC_INS_VSUBSBS, nullptr},
		{PPC_INS_VSUBSHS, nullptr},
		{PPC_INS_VSUBSWS, nullptr},
		{PPC_INS_VSUBUBM, nullptr},
		{PPC_INS_VSUBUBS, nullptr},
		{PPC_INS_VSUBUDM, nullptr},
		{PPC_INS_VSUBUHM, nullptr},
		{PPC_INS_VSUBUHS, nullptr},
		{PPC_INS_VSUBUWM, nullptr},
		{PPC_INS_VSUBUWS, nullptr},
		{PPC_INS_VSUM2SWS, nullptr},
		{PPC_INS_VSUM4SBS, nullptr},
		{PPC_INS_VSUM4SHS, nullptr},
		{PPC_INS_VSUM4UBS, nullptr},
		{PPC_INS_VSUMSWS, nullptr},
		{PPC_INS_VUPKHPX, nullptr},
		{PPC_INS_VUPKHSB, nullptr},
		{PPC_INS_VUPKHSH, nullptr},
		{PPC_INS_VUPKLPX, nullptr},
		{PPC_INS_VUPKLSB, nullptr},
		{PPC_INS_VUPKLSH, nullptr},
		{PPC_INS_VXOR, nullptr},
		{PPC_INS_WAIT, nullptr},
		{PPC_INS_WRTEE, nullptr},
		{PPC_INS_WRTEEI, nullptr},
		{PPC_INS_XOR, &Capstone2LlvmIrTranslatorPowerpc_impl::translateXor},
		{PPC_INS_XORI, &Capstone2LlvmIrTranslatorPowerpc_impl::translateXor},
		{PPC_INS_XORIS, &Capstone2LlvmIrTranslatorPowerpc_impl::translateXoris},
		{PPC_INS_XSABSDP, nullptr},
		{PPC_INS_XSADDDP, nullptr},
		{PPC_INS_XSCMPODP, nullptr},
		{PPC_INS_XSCMPUDP, nullptr},
		{PPC_INS_XSCPSGNDP, nullptr},
		{PPC_INS_XSCVDPSP, nullptr},
		{PPC_INS_XSCVDPSXDS, nullptr},
		{PPC_INS_XSCVDPSXWS, nullptr},
		{PPC_INS_XSCVDPUXDS, nullptr},
		{PPC_INS_XSCVDPUXWS, nullptr},
		{PPC_INS_XSCVSPDP, nullptr},
		{PPC_INS_XSCVSXDDP, nullptr},
		{PPC_INS_XSCVUXDDP, nullptr},
		{PPC_INS_XSDIVDP, nullptr},
		{PPC_INS_XSMADDADP, nullptr},
		{PPC_INS_XSMADDMDP, nullptr},
		{PPC_INS_XSMAXDP, nullptr},
		{PPC_INS_XSMINDP, nullptr},
		{PPC_INS_XSMSUBADP, nullptr},
		{PPC_INS_XSMSUBMDP, nullptr},
		{PPC_INS_XSMULDP, nullptr},
		{PPC_INS_XSNABSDP, nullptr},
		{PPC_INS_XSNEGDP, nullptr},
		{PPC_INS_XSNMADDADP, nullptr},
		{PPC_INS_XSNMADDMDP, nullptr},
		{PPC_INS_XSNMSUBADP, nullptr},
		{PPC_INS_XSNMSUBMDP, nullptr},
		{PPC_INS_XSRDPI, nullptr},
		{PPC_INS_XSRDPIC, nullptr},
		{PPC_INS_XSRDPIM, nullptr},
		{PPC_INS_XSRDPIP, nullptr},
		{PPC_INS_XSRDPIZ, nullptr},
		{PPC_INS_XSREDP, nullptr},
		{PPC_INS_XSRSQRTEDP, nullptr},
		{PPC_INS_XSSQRTDP, nullptr},
		{PPC_INS_XSSUBDP, nullptr},
		{PPC_INS_XSTDIVDP, nullptr},
		{PPC_INS_XSTSQRTDP, nullptr},
		{PPC_INS_XVABSDP, nullptr},
		{PPC_INS_XVABSSP, nullptr},
		{PPC_INS_XVADDDP, nullptr},
		{PPC_INS_XVADDSP, nullptr},
		{PPC_INS_XVCMPEQDP, nullptr},
		{PPC_INS_XVCMPEQSP, nullptr},
		{PPC_INS_XVCMPGEDP, nullptr},
		{PPC_INS_XVCMPGESP, nullptr},
		{PPC_INS_XVCMPGTDP, nullptr},
		{PPC_INS_XVCMPGTSP, nullptr},
		{PPC_INS_XVCPSGNDP, nullptr},
		{PPC_INS_XVCPSGNSP, nullptr},
		{PPC_INS_XVCVDPSP, nullptr},
		{PPC_INS_XVCVDPSXDS, nullptr},
		{PPC_INS_XVCVDPSXWS, nullptr},
		{PPC_INS_XVCVDPUXDS, nullptr},
		{PPC_INS_XVCVDPUXWS, nullptr},
		{PPC_INS_XVCVSPDP, nullptr},
		{PPC_INS_XVCVSPSXDS, nullptr},
		{PPC_INS_XVCVSPSXWS, nullptr},
		{PPC_INS_XVCVSPUXDS, nullptr},
		{PPC_INS_XVCVSPUXWS, nullptr},
		{PPC_INS_XVCVSXDDP, nullptr},
		{PPC_INS_XVCVSXDSP, nullptr},
		{PPC_INS_XVCVSXWDP, nullptr},
		{PPC_INS_XVCVSXWSP, nullptr},
		{PPC_INS_XVCVUXDDP, nullptr},
		{PPC_INS_XVCVUXDSP, nullptr},
		{PPC_INS_XVCVUXWDP, nullptr},
		{PPC_INS_XVCVUXWSP, nullptr},
		{PPC_INS_XVDIVDP, nullptr},
		{PPC_INS_XVDIVSP, nullptr},
		{PPC_INS_XVMADDADP, nullptr},
		{PPC_INS_XVMADDASP, nullptr},
		{PPC_INS_XVMADDMDP, nullptr},
		{PPC_INS_XVMADDMSP, nullptr},
		{PPC_INS_XVMAXDP, nullptr},
		{PPC_INS_XVMAXSP, nullptr},
		{PPC_INS_XVMINDP, nullptr},
		{PPC_INS_XVMINSP, nullptr},
		{PPC_INS_XVMSUBADP, nullptr},
		{PPC_INS_XVMSUBASP, nullptr},
		{PPC_INS_XVMSUBMDP, nullptr},
		{PPC_INS_XVMSUBMSP, nullptr},
		{PPC_INS_XVMULDP, nullptr},
		{PPC_INS_XVMULSP, nullptr},
		{PPC_INS_XVNABSDP, nullptr},
		{PPC_INS_XVNABSSP, nullptr},
		{PPC_INS_XVNEGDP, nullptr},
		{PPC_INS_XVNEGSP, nullptr},
		{PPC_INS_XVNMADDADP, nullptr},
		{PPC_INS_XVNMADDASP, nullptr},
		{PPC_INS_XVNMADDMDP, nullptr},
		{PPC_INS_XVNMADDMSP, nullptr},
		{PPC_INS_XVNMSUBADP, nullptr},
		{PPC_INS_XVNMSUBASP, nullptr},
		{PPC_INS_XVNMSUBMDP, nullptr},
		{PPC_INS_XVNMSUBMSP, nullptr},
		{PPC_INS_XVRDPI, nullptr},
		{PPC_INS_XVRDPIC, nullptr},
		{PPC_INS_XVRDPIM, nullptr},
		{PPC_INS_XVRDPIP, nullptr},
		{PPC_INS_XVRDPIZ, nullptr},
		{PPC_INS_XVREDP, nullptr},
		{PPC_INS_XVRESP, nullptr},
		{PPC_INS_XVRSPI, nullptr},
		{PPC_INS_XVRSPIC, nullptr},
		{PPC_INS_XVRSPIM, nullptr},
		{PPC_INS_XVRSPIP, nullptr},
		{PPC_INS_XVRSPIZ, nullptr},
		{PPC_INS_XVRSQRTEDP, nullptr},
		{PPC_INS_XVRSQRTESP, nullptr},
		{PPC_INS_XVSQRTDP, nullptr},
		{PPC_INS_XVSQRTSP, nullptr},
		{PPC_INS_XVSUBDP, nullptr},
		{PPC_INS_XVSUBSP, nullptr},
		{PPC_INS_XVTDIVDP, nullptr},
		{PPC_INS_XVTDIVSP, nullptr},
		{PPC_INS_XVTSQRTDP, nullptr},
		{PPC_INS_XVTSQRTSP, nullptr},
		{PPC_INS_XXLAND, nullptr},
		{PPC_INS_XXLANDC, nullptr},
		{PPC_INS_XXLEQV, nullptr},
		{PPC_INS_XXLNAND, nullptr},
		{PPC_INS_XXLNOR, nullptr},
		{PPC_INS_XXLOR, nullptr},
		{PPC_INS_XXLORC, nullptr},
		{PPC_INS_XXLXOR, nullptr},
		{PPC_INS_XXMRGHW, nullptr},
		{PPC_INS_XXMRGLW, nullptr},
		{PPC_INS_XXPERMDI, nullptr},
		{PPC_INS_XXSEL, nullptr},
		{PPC_INS_XXSLDWI, nullptr},
		{PPC_INS_XXSPLTW, nullptr},

		// extra & alias instructions
		{PPC_INS_SLWI, &Capstone2LlvmIrTranslatorPowerpc_impl::translateSlwi},
		{PPC_INS_SRWI, &Capstone2LlvmIrTranslatorPowerpc_impl::translateSrwi},
		{PPC_INS_SLDI, nullptr},

		{PPC_INS_CRSET, &Capstone2LlvmIrTranslatorPowerpc_impl::translateCrSetClr},
		{PPC_INS_CRCLR, &Capstone2LlvmIrTranslatorPowerpc_impl::translateCrSetClr},
		{PPC_INS_CRNOT, &Capstone2LlvmIrTranslatorPowerpc_impl::translateCrNotMove},
		{PPC_INS_CRMOVE, &Capstone2LlvmIrTranslatorPowerpc_impl::translateCrNotMove},

		{PPC_INS_MFBR0, nullptr},
		{PPC_INS_MFBR1, nullptr},
		{PPC_INS_MFBR2, nullptr},
		{PPC_INS_MFBR3, nullptr},
		{PPC_INS_MFBR4, nullptr},
		{PPC_INS_MFBR5, nullptr},
		{PPC_INS_MFBR6, nullptr},
		{PPC_INS_MFBR7, nullptr},
		{PPC_INS_MFXER, nullptr},
		{PPC_INS_MFRTCU, nullptr},
		{PPC_INS_MFRTCL, nullptr},
		{PPC_INS_MFDSCR, nullptr},
		{PPC_INS_MFDSISR, nullptr},
		{PPC_INS_MFDAR, nullptr},
		{PPC_INS_MFSRR2, nullptr},
		{PPC_INS_MFSRR3, nullptr},
		{PPC_INS_MFCFAR, nullptr},
		{PPC_INS_MFAMR, nullptr},
		{PPC_INS_MFPID, nullptr},
		{PPC_INS_MFTBLO, nullptr},
		{PPC_INS_MFTBHI, nullptr},
		{PPC_INS_MFDBATU, nullptr},
		{PPC_INS_MFDBATL, nullptr},
		{PPC_INS_MFIBATU, nullptr},
		{PPC_INS_MFIBATL, nullptr},
		{PPC_INS_MFDCCR, nullptr},
		{PPC_INS_MFICCR, nullptr},
		{PPC_INS_MFDEAR, nullptr},
		{PPC_INS_MFESR, nullptr},
		{PPC_INS_MFSPEFSCR, nullptr},
		{PPC_INS_MFTCR, nullptr},
		{PPC_INS_MFASR, nullptr},
		{PPC_INS_MFPVR, nullptr},
		{PPC_INS_MFTBU, nullptr},
		{PPC_INS_MTCR, &Capstone2LlvmIrTranslatorPowerpc_impl::translateMtcr},
		{PPC_INS_MTBR0, nullptr},
		{PPC_INS_MTBR1, nullptr},
		{PPC_INS_MTBR2, nullptr},
		{PPC_INS_MTBR3, nullptr},
		{PPC_INS_MTBR4, nullptr},
		{PPC_INS_MTBR5, nullptr},
		{PPC_INS_MTBR6, nullptr},
		{PPC_INS_MTBR7, nullptr},
		{PPC_INS_MTXER, nullptr},
		{PPC_INS_MTDSCR, nullptr},
		{PPC_INS_MTDSISR, nullptr},
		{PPC_INS_MTDAR, nullptr},
		{PPC_INS_MTSRR2, nullptr},
		{PPC_INS_MTSRR3, nullptr},
		{PPC_INS_MTCFAR, nullptr},
		{PPC_INS_MTAMR, nullptr},
		{PPC_INS_MTPID, nullptr},
		{PPC_INS_MTTBL, nullptr},
		{PPC_INS_MTTBU, nullptr},
		{PPC_INS_MTTBLO, nullptr},
		{PPC_INS_MTTBHI, nullptr},
		{PPC_INS_MTDBATU, nullptr},
		{PPC_INS_MTDBATL, nullptr},
		{PPC_INS_MTIBATU, nullptr},
		{PPC_INS_MTIBATL, nullptr},
		{PPC_INS_MTDCCR, nullptr},
		{PPC_INS_MTICCR, nullptr},
		{PPC_INS_MTDEAR, nullptr},
		{PPC_INS_MTESR, nullptr},
		{PPC_INS_MTSPEFSCR, nullptr},
		{PPC_INS_MTTCR, nullptr},
		{PPC_INS_NOT, &Capstone2LlvmIrTranslatorPowerpc_impl::translateNot},
		{PPC_INS_MR, &Capstone2LlvmIrTranslatorPowerpc_impl::translateMr},
		{PPC_INS_ROTLD, nullptr},
		{PPC_INS_ROTLDI, nullptr},
		{PPC_INS_CLRLDI, nullptr},
		{PPC_INS_ROTLWI, &Capstone2LlvmIrTranslatorPowerpc_impl::translateRotlw},
		{PPC_INS_CLRLWI, &Capstone2LlvmIrTranslatorPowerpc_impl::translateClrlwi},
		{PPC_INS_ROTLW, &Capstone2LlvmIrTranslatorPowerpc_impl::translateRotlw},
		{PPC_INS_SUB, &Capstone2LlvmIrTranslatorPowerpc_impl::translateSubf},
		{PPC_INS_SUBC, &Capstone2LlvmIrTranslatorPowerpc_impl::translateSubfc},
		{PPC_INS_LWSYNC, nullptr},
		{PPC_INS_PTESYNC, nullptr},
		{PPC_INS_TDLT, nullptr},
		{PPC_INS_TDEQ, nullptr},
		{PPC_INS_TDGT, nullptr},
		{PPC_INS_TDNE, nullptr},
		{PPC_INS_TDLLT, nullptr},
		{PPC_INS_TDLGT, nullptr},
		{PPC_INS_TDU, nullptr},
		{PPC_INS_TDLTI, nullptr},
		{PPC_INS_TDEQI, nullptr},
		{PPC_INS_TDGTI, nullptr},
		{PPC_INS_TDNEI, nullptr},
		{PPC_INS_TDLLTI, nullptr},
		{PPC_INS_TDLGTI, nullptr},
		{PPC_INS_TDUI, nullptr},
		{PPC_INS_TLBREHI, nullptr},
		{PPC_INS_TLBRELO, nullptr},
		{PPC_INS_TLBWEHI, nullptr},
		{PPC_INS_TLBWELO, nullptr},
		{PPC_INS_TWLT, nullptr},
		{PPC_INS_TWEQ, nullptr},
		{PPC_INS_TWGT, nullptr},
		{PPC_INS_TWNE, nullptr},
		{PPC_INS_TWLLT, nullptr},
		{PPC_INS_TWLGT, nullptr},
		{PPC_INS_TWU, nullptr},
		{PPC_INS_TWLTI, nullptr},
		{PPC_INS_TWEQI, nullptr},
		{PPC_INS_TWGTI, nullptr},
		{PPC_INS_TWNEI, nullptr},
		{PPC_INS_TWLLTI, nullptr},
		{PPC_INS_TWLGTI, nullptr},
		{PPC_INS_TWUI, nullptr},
		{PPC_INS_WAITRSV, nullptr},
		{PPC_INS_WAITIMPL, nullptr},
		{PPC_INS_XNOP, nullptr},
		{PPC_INS_XVMOVDP, nullptr},
		{PPC_INS_XVMOVSP, nullptr},
		{PPC_INS_XXSPLTD, nullptr},
		{PPC_INS_XXMRGHD, nullptr},
		{PPC_INS_XXMRGLD, nullptr},
		{PPC_INS_XXSWAPD, nullptr},

		// QPX
		{PPC_INS_QVFAND, nullptr},
		{PPC_INS_QVFCLR, nullptr},
		{PPC_INS_QVFANDC, nullptr},
		{PPC_INS_QVFCTFB, nullptr},
		{PPC_INS_QVFXOR, nullptr},
		{PPC_INS_QVFOR, nullptr},
		{PPC_INS_QVFNOR, nullptr},
		{PPC_INS_QVFEQU, nullptr},
		{PPC_INS_QVFNOT, nullptr},
		{PPC_INS_QVFORC, nullptr},
		{PPC_INS_QVFNAND, nullptr},
		{PPC_INS_QVFSET, nullptr},

		{PPC_INS_ENDING, nullptr},

		// Branches.
		//

		// Basic unconditional.
		{PPC_INS_B, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		{PPC_INS_BA, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		{PPC_INS_BL, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		{PPC_INS_BLA, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},

		// Basic conditional.
		{PPC_INS_BC, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		{PPC_INS_BCA, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		{PPC_INS_BCLR, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		{PPC_INS_BCCTR, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		{PPC_INS_BCL, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		{PPC_INS_BCLA, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		{PPC_INS_BCLRL, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		{PPC_INS_BCCTRL, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},

		// Branch unconditionally.
		{PPC_INS_BLR, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		{PPC_INS_BCTR, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		{PPC_INS_BLRL, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		{PPC_INS_BCTRL, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},

		// Branch if condition true.
		{PPC_INS_BT, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		{PPC_INS_BTA, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		{PPC_INS_BTLR, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		{PPC_INS_BTCTR, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		{PPC_INS_BTL, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		{PPC_INS_BTLA, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		{PPC_INS_BTLRL, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		{PPC_INS_BTCTRL, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},

		// Branch if condition false.
		{PPC_INS_BF, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		{PPC_INS_BFA, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		{PPC_INS_BFLR, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		{PPC_INS_BFCTR, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		{PPC_INS_BFL, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		{PPC_INS_BFLA, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		{PPC_INS_BFLRL, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		{PPC_INS_BFCTRL, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},

		// Decrement CTR, branch if CTR != 0
		{PPC_INS_BDNZ, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		{PPC_INS_BDNZA, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		{PPC_INS_BDNZL, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		{PPC_INS_BDNZLA, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		{PPC_INS_BDNZLR, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		{PPC_INS_BDNZLRL, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		// Decrement CTR, branch if CTR != 0 & cond true
		{PPC_INS_BDNZT, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		{PPC_INS_BDNZTA, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		{PPC_INS_BDNZTLR, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		{PPC_INS_BDNZTL, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		{PPC_INS_BDNZTLA, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		{PPC_INS_BDNZTLRL, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		// Decrement CTR, branch if CTR != 0 & cond false
		{PPC_INS_BDNZF, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		{PPC_INS_BDNZFA, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		// TODO: Where is bdnzflr? Report Capstone bug?
		{PPC_INS_BDNZFL, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		{PPC_INS_BDNZFLA, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		{PPC_INS_BDNZFLRL, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},

		// Decrement CTR, branch if CTR == 0
		//
		{PPC_INS_BDZ, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		{PPC_INS_BDZA, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		{PPC_INS_BDZLR, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		{PPC_INS_BDZL, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		{PPC_INS_BDZLA, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		{PPC_INS_BDZLRL, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		// Decrement CTR, branch if CTR == 0 & cond true
		{PPC_INS_BDZT, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		{PPC_INS_BDZTA, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		{PPC_INS_BDZTLR, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		{PPC_INS_BDZTL, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		{PPC_INS_BDZTLA, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		{PPC_INS_BDZTLRL, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		// Decrement CTR, branch if CTR == 0 & cond false
		{PPC_INS_BDZF, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		{PPC_INS_BDZFA, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		{PPC_INS_BDZFLR, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		{PPC_INS_BDZFL, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		{PPC_INS_BDZFLA, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},
		{PPC_INS_BDZFLRL, &Capstone2LlvmIrTranslatorPowerpc_impl::translateB},

		{PPC_INS_BCT, nullptr},
};

} // namespace capstone2llvmir
} // namespace retdec
