//! Qore namespace
namespace Qore {
//! Json namespace
namespace Json {
//! The JsonRpcClient class provides easy access to JSON-RPC web services
/** This class inherits all public methods of the HTTPClient class. The inherited HTTPClient methods are not listed in this section, see the documentation for the HTTPClient class for more information on methods provided by the parent class.  For a list of low-level JSON-RPC functions, see @ref JSONRPC.

    Thie object supports the following JSON-RPC protocol versions:
    - \c "2.0": the default; requests and responses will include the \c "jsonrpc" attribute assigned to the version number (this is true for any declared JSON-RPC protocol version >= 2)
    - \c "1.0": can be explicitly set in the @ref Qore::Json::JsonRpcClient::constructor() "JsonRpcClient::constructor()" or by calling @ref Qore::Json::JsonRpcClient::setVersion() "JsonRpcClient::setVersion()"

    @note other JSON-RPC protocol versions can be used; for example the unpublished \c "1.1" version can be used as well, but support for JSON-RPC 1.1 is deprecated; for any JSON-RPC protocol version > 1.0 and < 2.0, the \c "version" attribute is serialized with the declared protocol version number (unlike with 2.0 or above, where the \c "jsonrpc" attribute contains this value)

    The JsonRpcClient class understands the following protocol schemes in addition to the protocol schemes supported by the HTTPClient class:

    <b>JsonRpcClient Class Protocols</b>
    @htmlonly <style><!-- td.qore { background-color: #5b9409; color: white; } --></style> @endhtmlonly
    <table>
    <tr>
        <td class="qore"><b>Protocol</b></td>
        <td class="qore"><b>Default Port</b></td>
        <td class="qore"><b>SSL?</b></td>
        <td class="qore"><b>Description</b></td>
    </tr>
    <tr>
        <td>\c jsonrpc</td>
        <td>\c 80</td>
        <td>No</td>
        <td>Unencrypted JSON-RPC protocol over HTTP</td>
    </tr>
    <tr>
        <td>\c jsonrpcs</td>
        <td>\c 443</td>
        <td>Yes</td>
        <td>JSON-RPC protocol over HTTP with SSL/TLS encryption</td>
    </tr>
    </table>

    The JsonRpcClient supplies default values for HTTP headers as follows:

    <b>JsonRpcClient Default, but Overridable Headers</b>
    @htmlonly <style><!-- td.qore { background-color: #5b9409; color: white; } --></style> @endhtmlonly
    <table>
    <tr>
        <td class="qore"><b>Header</b></td>
        <td class="qore"><b>Default Value</b></td>
    </tr>
    <tr>
        <td>\c Accept</td>
        <td>\c text/json</td>
    </tr>
    <tr>
        <td>\c Content-Type</td>
        <td>\c application/json;charset=utf-8</td>
    </tr>
    <tr>
        <td>\c User-Agent</td>
        <td>\c Qore-JSON-RPC-Client/1.7</td>
    </tr>
    <tr>
        <td>\c Connection</td>
        <td>\c Keep-Alive</td>
    </tr>
    </table>

    @note
    - if no \c default_path is given in the constructor(), then \c "JSON" is used
    - This class is not available with the \c PO_NO_NETWORK parse option.
*/
class JsonRpcClient : public HTTPClient {

public:
//! Calls a remote method taking all arguments after the method name for the method arguments and returns the response as qore data structure
/** @param method The JSON-RPC method name to call
    @param ... the arguments to the call

    @return a data structure corresponding to the JSON data returned by the JSON-RPC server

    @throw JSON-SERIALIZATION-ERROR cannot serialize value passed (ex: binary, object)
    @throw JSON-PARSE-ERROR syntax error parsing JSON string
    @throw HTTP-CLIENT-TIMEOUT timeout on response from HTTP server
    @throw HTTP-CLIENT-RECEIVE-ERROR error communicating with HTTP server

    @note other exceptions may be thrown related to communication errors (ex: SSL errors, etc)

    @par Example:
    @code{.py}
hash result = jrc.call("method.name", arg1, arg2);
    @endcode

    @since json 1.7 this method calls with a hardcoded \c id value of 1
*/
hash call(string method,  ...);

public:
//! Calls a remote method using a single value after the method name for the method arguments and returns the response as qore data structure
/** @param method The JSON-RPC method name to call
    @param args An optional list of arguments (or single argument) for the method

    @return a data structure corresponding to the JSON data returned by the JSON-RPC server

    @throw JSON-SERIALIZATION-ERROR cannot serialize value passed (ex: binary, object)
    @throw JSON-PARSE-ERROR syntax error parsing JSON string
    @throw HTTP-CLIENT-TIMEOUT timeout on response from HTTP server
    @throw HTTP-CLIENT-RECEIVE-ERROR error communicating with HTTP server

    @note other exceptions may be thrown related to communication errors (ex: SSL errors, etc)

    @par Example:
    @code{.py}
hash result = jrc.callArgs("method.name", arg_list);
    @endcode

    @since json 1.7 this method calls with a hardcoded \c id value of 1
*/
hash callArgs(string method, auto args);

public:
//! Calls a remote method using a single value after the method name for the method arguments and returns the response as qore data structure, accepts a reference to a hash as the first argument to give technical information about the call
/** @param info a reference to a hash that provides the following keys on output giving technical information about the HTTP call:
    - \c request: the literal outgoing request body sent
    - \c request-uri: the first line of the HTTP request
    - \c headers: a hash of HTTP headers in the outgoing request
    - \c response-uri: the first line of the HTTP response
    - \c response: the literal response body received from the server
    - \c response_headers: a hash of headers received in the response
    @param method The JSON-RPC method name to call
    @param args An optional list of arguments (or single argument) for the method

    @return a data structure corresponding to the JSON data returned by the JSON-RPC server

    @throw JSON-SERIALIZATION-ERROR cannot serialize value passed (ex: binary, object)
    @throw JSON-PARSE-ERROR syntax error parsing JSON string
    @throw HTTP-CLIENT-TIMEOUT timeout on response from HTTP server
    @throw HTTP-CLIENT-RECEIVE-ERROR error communicating with HTTP server

    @note other exceptions may be thrown related to communication errors (ex: SSL errors, etc)

    @par Example:
    @code{.py}
hash info;
hash result = jrc.callArgsWithInfo(\info, "method.name", arg_list, 2);
    @endcode

    @since json 1.7 this method calls with a hardcoded \c id value of 1
*/
hash callArgsWithInfo(reference info, string method, any args);

public:
//! Calls a remote method with a call ID, taking all arguments after the method name for the method arguments and returns the response as qore data structure
/** @param id the \c id value for the call
    @param method The JSON-RPC method name to call
    @param ... the arguments to the call

    @return a data structure corresponding to the JSON data returned by the JSON-RPC server

    @throw JSON-SERIALIZATION-ERROR cannot serialize value passed (ex: binary, object)
    @throw JSON-PARSE-ERROR syntax error parsing JSON string
    @throw HTTP-CLIENT-TIMEOUT timeout on response from HTTP server
    @throw HTTP-CLIENT-RECEIVE-ERROR error communicating with HTTP server

    @note other exceptions may be thrown related to communication errors (ex: SSL errors, etc)

    @par Example:
    @code{.py}
hash result = jrc.callId(1, "method.name", arg1, arg2);
    @endcode

    @since json 1.7
*/
hash callId(any id, string method,  ...);

public:
//! Calls a remote method using a single value after the method name for the method arguments and returns the response as qore data structure, accepts a reference to a hash as the first argument to give technical information about the call
/** @param info a reference to a hash that provides the following keys on output giving technical information about the HTTP call:
    - \c request: the literal outgoing request body sent
    - \c request-uri: the first line of the HTTP request
    - \c headers: a hash of HTTP headers in the outgoing request
    - \c response-uri: the first line of the HTTP response
    - \c response: the literal response body received from the server
    - \c response_headers: a hash of headers received in the response
    @param id the \c id value for the call
    @param method The JSON-RPC method name to call
    @param args An optional list of arguments (or single argument) for the method

    @return a data structure corresponding to the JSON data returned by the JSON-RPC server

    @throw JSON-SERIALIZATION-ERROR cannot serialize value passed (ex: binary, object)
    @throw JSON-PARSE-ERROR syntax error parsing JSON string
    @throw HTTP-CLIENT-TIMEOUT timeout on response from HTTP server
    @throw HTTP-CLIENT-RECEIVE-ERROR error communicating with HTTP server

    @note other exceptions may be thrown related to communication errors (ex: SSL errors, etc)

    @par Example:
    @code{.py}
hash info;
hash result = jrc.callIdArgsWithInfo(\info, 4, "method.name", arg_list, 2);
    @endcode

    @since json 1.7
*/
hash callIdArgsWithInfo(reference info, any id, string method, any args);

public:
//! Calls a remote method taking all arguments after the method name for the method arguments and returns the response as qore data structure, accepts a reference to a hash as the first argument to give technical information about the call
/** @param info a reference to a hash that provides the following keys on output giving technical information about the HTTP call:
    - \c request: the literal outgoing request body sent
    - \c request-uri: the first line of the HTTP request
    - \c headers: a hash of HTTP headers in the outgoing request
    - \c response-uri: the first line of the HTTP response
    - \c response: the literal response body received from the server
    - \c response_headers: a hash of headers received in the response
    @param id the \c id value for the call
    @param method The JSON-RPC method name to call

    @return a data structure corresponding to the JSON data returned by the JSON-RPC server

    @throw JSON-SERIALIZATION-ERROR cannot serialize value passed (ex: binary, object)
    @throw JSON-PARSE-ERROR syntax error parsing JSON string
    @throw HTTP-CLIENT-TIMEOUT timeout on response from HTTP server
    @throw HTTP-CLIENT-RECEIVE-ERROR error communicating with HTTP server

    @note other exceptions may be thrown related to communication errors (ex: SSL errors, etc)

    @par Example:
    @code{.py}
hash info;
hash result = jrc.callIdWithInfo(\info, 3, "method.name", arg1, arg2);
    @endcode

    @since the \c id parameter was added in json 1.7
*/
hash callIdWithInfo(reference info, any id, string method,  ...);

public:
//! Calls a remote method taking all arguments after the method name for the method arguments and returns the response as qore data structure, accepts a reference to a hash as the first argument to give technical information about the call
/** @param info a reference to a hash that provides the following keys on output giving technical information about the HTTP call:
    - \c request: the literal outgoing request body sent
    - \c request-uri: the first line of the HTTP request
    - \c headers: a hash of HTTP headers in the outgoing request
    - \c response-uri: the first line of the HTTP response
    - \c response: the literal response body received from the server
    - \c response_headers: a hash of headers received in the response
    @param method The JSON-RPC method name to call

    @return a data structure corresponding to the JSON data returned by the JSON-RPC server

    @throw JSON-SERIALIZATION-ERROR cannot serialize value passed (ex: binary, object)
    @throw JSON-PARSE-ERROR syntax error parsing JSON string
    @throw HTTP-CLIENT-TIMEOUT timeout on response from HTTP server
    @throw HTTP-CLIENT-RECEIVE-ERROR error communicating with HTTP server

    @note other exceptions may be thrown related to communication errors (ex: SSL errors, etc)

    @par Example:
    @code{.py}
hash info;
hash result = jrc.callWithInfo(\info, "method.name", arg1, arg2);
    @endcode

    @since json 1.7 this method calls with a hardcoded \c id value of 1
*/
hash callWithInfo(reference info, string method,  ...);

public:
//! Creates the JsonRpcClient object with no arguments
/** No connection is made because no connection parameters are set with this call; connection parameters must be set afterwards using the appropriate HTTPClient methods.

    The JSON-RPC version is set to \c "2.0" by default with this variant of the constructor()

    @par Example:
    @code{.py}
JsonRpcClient jrc();
jrc.setURL("http://localhost:8080");@endcode
*/
 constructor();

public:
//! Creates the JsonRpcClient object based on the parameters passed
/** By default the object will immediately attempt to establish a connection to the server
    @param opts HTTPClient constructor options:
    - \c connect_timeout: The timeout value in milliseconds for establishing a new socket connection (also can be a relative date-time value for clarity, ex: \c 30s)
    - \c default_path: The default path to use for new connections if a path is not otherwise specified in the connection URL; if this option is not given and no path appears in the URL, then \c "JSON" is used instead
    - \c default_port: The default port number to connect to if none is given in the URL
    - \c http_version: Either \c '1.0' or \c '1.1' for the claimed HTTP protocol version compliancy in outgoing message headers
    - \c protocols: A hash describing new protocols, the key is the protocol name and the value is either an integer giving the default port number or a hash with \c 'port' and \c 'ssl' keys giving the default port number and a boolean value to indicate that an SSL connection should be established
    - \c max_redirects: The maximum number of redirects before throwing an exception (the default is 5)
    - \c proxy: The proxy URL for connecting through a proxy
    - \c timeout: The timeout value in milliseconds (also can be a relative date-time value for clarity, ex: \c 5m)
    - \c url: A string giving the URL to connect to; if this option is not given and no path appears in the URL, then \c "JSON" is used instead
    - \c version: A string giving the JSON-RPC protocol version; default: \c "2.0"
    @param no_connect pass a boolean True value argument to suppress the automatic connection and establish a connection on demand with the first request

    @see HTTPClient::constructor() and HTTPClient::connect() for information on possible exceptions

    @par Example:
    @code{.py}
JsonRpcClient jrc(("url": "http://authuser:authpass@otherhost:8080/JSONRPC"));
    @endcode

    @since json 1.7 added support for the \c "version" option
*/
 constructor(hash opts, softbool no_connect = False);

public:
//! Throws an exception; copying JsonRpcClient objects is currently not supported
/** @throw JSONRPCCLIENT-COPY-ERROR copying JsonRpcClient objects is currently not supported
*/
 copy();

public:
//! Returns the JSON-RPC protocol version
/** @par Code Flags:
    @ref CONSTANT

@return the JSON-RPC protocol version

    @since json 1.7
 */
string getVersion();

public:
//! clears the event queue for the JsonRpcClient object
/** @par Example:
    @code{.py}
jrc.setEventQueue();
    @endcode
 */
nothing setEventQueue();

public:
//! sets the event queue for the JsonRpcClient object
/** @param queue the Queue object to receive network events from the JsonRpcClient object
    @par Example:
    @code{.py}
Queue queue();
jrc.setEventQueue(queue);
    @endcode
 */
nothing setEventQueue(Queue queue);

public:
//! Sets the JSON-RPC protocol version
/** @param version the JSON-RPC protocol version

    @since json 1.7
 */
nothing setVersion(string version);
};
}
}
