
/*  Q eQuational Programming System
    Copyright (c) 1991-2002 by Albert Graef
    <ag@muwiinfa.geschichte.uni-mainz.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "qdefs.h"

/*

Interface to the built-in functions:

The built-in functions are implemented as parameterless functions;
argument expressions are placed in the global args vector (module qm).

It is supposed that each function returns zero if the function
application failed (due to mismatch of arguments, or other error
conditions in which case qmstat should be set accordingly), and
nonzero in case of a successful application. In the latter case, the
result is pushed on top of the stack.

List of built-in functions:

qmdef()
- process variable def
qmundef()
- process variable undef

qmconcat()
- string/list/tuple concatenation
qmadd()
- integer/float addition
qmmin()
- integer/float subtraction
qmmul()
- integer/float multiplication
qmfdiv()
- integer/float division
qmdiv()
- integer division
qmmod()
- remainder of integer division
qmpow()
- integer/float exponentiation
qmidx()
- indexing
qmumin()
- unary minus
qmhash()
- size operator
qmunquote()
- unquote operator
qmforce()
- force operator
qmmem()
- memo operator
qmor()
- logical or (special)
qmand()
- logical and (special)
qmorelse()
- logical or (short-circuit)
qmandthen()
- logical and (short-circuit)
qmnot()
- logical not
qmle()
- check for "less than"
qmgr()
- check for "greater than"
qmeq()
- check for "equal"
qmleq()
- check for "less than or equal"
qmgeq()
- check for "greater than or equal"
qmneq()
- check for "not equal"
qmid()
- check for "identical" (special)

qmrapp()
- right-associative apply operator
qmseq()
- sequence operator

qmshl()
- integer bit shift left
qmshr()
- integer bit shift right
qmpred()
- predecessor function
qmsucc()
- successor function
qmenum()
- enumerate members of an enumeration type

qmexp()
- exponential function
qmln()
- natural logarithm
qmsqrt()
- square root
qmsin()
- sine
qmcos()
- cosine
qmatan()
- arc tan
qmatan2()
- arc tan of 2 args
qmrandom()
- random number
qmseed()
- initialize random number generator

qmsub()
- return subsequence of string, list or tuple
qmsubstr()
- return substring of a string
qmpos()
- determine position of a substring in a string

qmint()
- return integral part of a floating point number
qmfrac()
- return fractional part of a floating point number
qmtrunc()
- truncate a float to an integer value
qmround()
- round a float to an integer value
qmfloat()
- convert an integer to a floating point number
qmhashnum()
- return a 32 bit hash code for any expression
qmord()
- convert a character or enumeration constant to an integer
qmchr()
- convert an integer to a character
qmlist()
- convert a tuple to a list
qmtuple()
- convert a list to a tuple
qmstr()
- convert expression to string
qmval()
- convert string to expression
qmstrq()
- convert quoted expression to string
qmvalq()
- convert string to quoted expression

qmisspecial()
- verify that argument is a special form
qmisconst()
- verify that argument is a constant
qmisfun()
- verify that argument is other function symbol
qmisdef()
- check whether variable has been assigned a value
qmflip()
- flip args of binary function

qmread()
- read an expression from the terminal
qmreadq()
- read a quoted expression from the terminal
qmreadc()
- read a character from the terminal
qmreads()
- read a string from the terminal
qmwrite()
- write an expression to the terminal
qmwriteq()
- write a quoted expression to the terminal
qmwritec()
- write a character to the terminal
qmwrites()
- write a string to the terminal

qmfread()
- read an expression from a file
qmfreadq()
- read a quoted expression from a file
qmfreadc()
- read a character from a file
qmfreads()
- read a string from a file
qmfwrite()
- write an expression to a file
qmfwriteq()
- write a quoted expression to a file
qmfwritec()
- write a character to a file
qmfwrites()
- write a string to a file

qmfopen()
- open a file
qmpopen()
- open a pipe
qmfclose()
- close a file
qmeof()
- check for end-of-file on terminal
qmfeof()
- check for end-of-file on file
qmflush()
- flush output buffer of terminal
qmfflush()
- flush output buffer of a file

qmversion()
- return interpreter version number
qmsysinfo
- return string describing the host system (hardware-vendor-os)
qmwhich
- return absolute path of given script
qmhalt()
- halt execution
qmquit()
- quit the interpreter
qmbreak()
- interrupt evaluation and invoke the Q code debugger
qmcatch()
- handle an exception
qmthrow()
- throw an exception
qmtrap()
- catch signals
qmfail()
- let current rule fail
qmfail2()
- let current reduction fail
qmtime()
- return the current time
qmsleep()
- sleep for given number of seconds

*/

#ifdef HAVE_UNICODE

/* Helper functions to deal with UTF-8 encoded unicode strings. */

static inline size_t
u8strlen(char *s)
{
  size_t n = 0;
  unsigned p = 0, q = 0;
 start:
  for (; *s; s++) {
    unsigned char uc = (unsigned char)*s;
    if (q == 0) {
      if (((signed char)uc) < 0) {
	switch (uc & 0xf0) {
	case 0xc0: case 0xd0:
	  q = 1;
	  break;
	case 0xe0:
	  q = 2;
	  break;
	case 0xf0:
	  if ((uc & 0x8) == 0)
	    q = 3;
	  break;
	}
      }
      p = 0; n++;
    } else if ((uc & 0xc0) == 0x80) {
      /* continuation byte */
      if (--q == 0)
	p = 0;
      else
	p++;
    } else {
      /* malformed char */
      s -= p+1; p = q = 0;
    }
  }
  if (q > 0) {
    /* unterminated char */
    s -= p; p = q = 0;
    goto start;
  }
  return n;
}

static inline char *
u8strind(char *s, size_t i)
{
  unsigned p = 0, q = 0;
 start:
  for (; *s && i > 0; s++) {
    unsigned char uc = (unsigned char)*s;
    if (q == 0) {
      if (((signed char)uc) < 0) {
	switch (uc & 0xf0) {
	case 0xc0: case 0xd0:
	  q = 1;
	  break;
	case 0xe0:
	  q = 2;
	  break;
	case 0xf0:
	  if ((uc & 0x8) == 0)
	    q = 3;
	  break;
	}
      }
      p = 0; if (q == 0) i--;
    } else if ((uc & 0xc0) == 0x80) {
      /* continuation byte */
      if (--q == 0) {
	p = 0; i--;
      } else
	p++;
    } else {
      /* malformed char */
      i--; s -= p+1; p = q = 0;
    }
  }
  if (q > 0) {
    /* unterminated char */
    i--; s -= p; p = q = 0;
    goto start;
  }
  return s;
}

static inline long
u8strpos(char *s, size_t i)
{
  size_t n = 0;
  unsigned p = 0, q = 0;
  char *t = s, *s0 = s;
  for (; i > 0 && *s; i--, s++) {
    unsigned char uc = (unsigned char)*s;
    if (q == 0) {
      if (((signed char)uc) < 0) {
	switch (uc & 0xf0) {
	case 0xc0: case 0xd0:
	  q = 1;
	  break;
	case 0xe0:
	  q = 2;
	  break;
	case 0xf0:
	  if ((uc & 0x8) == 0)
	    q = 3;
	  break;
	}
      }
      p = 0; if (q == 0) n++;
    } else if ((uc & 0xc0) == 0x80) {
      /* continuation byte */
      if (--q == 0) {
	n++; p = 0;
      } else
	p++;
    } else {
      /* malformed char */
      n++; i += p+1; s -= p+1; p = q = 0;
    }
    if (q == 0) t = s+1;
  }
  if (i != 0) return -1;
  while (t < s) {
    t = u8strind(t, 1);
    if (t <= s) n++;
  }
  return n;
}

static inline long
u8decode(char *s)
{
  size_t n = 0;
  unsigned p = 0, q = 0;
  unsigned long c = 0;
  if (s[0] == 0)
    return -1;
  else if (s[1] == 0)
    return (unsigned char)s[0];
  for (; n == 0 && *s; s++) {
    unsigned char uc = (unsigned char)*s;
    if (q == 0) {
      if (((signed char)uc) < 0) {
	switch (uc & 0xf0) {
	case 0xc0: case 0xd0:
	  q = 1;
	  c = uc & 0x1f;
	  break;
	case 0xe0:
	  q = 2;
	  c = uc & 0xf;
	  break;
	case 0xf0:
	  if ((uc & 0x8) == 0) {
	    q = 3;
	    c = uc & 0x7;
	  } else
	    c = uc;
	  break;
	default:
	  c = uc;
	  break;
	}
      } else
	c = uc;
      p = 0; if (q == 0) n++;
    } else if ((uc & 0xc0) == 0x80) {
      /* continuation byte */
      c = c << 6 | (uc & 0x3f);
      if (--q == 0)
	n++;
      else
	p++;
    } else {
      /* malformed char */
      return -1;
    }
  }
  if (n == 1 && *s == 0)
    return c;
  else
    return -1;
}

static inline char *
u8encode(char *t, unsigned long c)
{
  unsigned char *uc = (unsigned char*)t;
  if (c < 0x80) {
    uc[1] = 0;
    uc[0] = c;
  } else if (c < 0x800) {
    uc[2] = 0;
    uc[1] = 0x80 | c&0x3f;
    c = c >> 6;
    uc[0] = 0xc0 | c;
  } else if (c < 0x10000) {
    uc[3] = 0;
    uc[2] = 0x80 | c&0x3f;
    c = c >> 6;
    uc[1] = 0x80 | c&0x3f;
    c = c >> 6;
    uc[0] = 0xe0 | c;
  } else {
    uc[4] = 0;
    uc[3] = 0x80 | c&0x3f;
    c = c >> 6;
    uc[2] = 0x80 | c&0x3f;
    c = c >> 6;
    uc[1] = 0x80 | c&0x3f;
    c = c >> 6;
    uc[0] = 0xf0 | c;
  }
  return t;
}

static inline char *
u8getc(FILE *fp, char *t)
{
  size_t n;
  unsigned p = 0, q = 0;
  unsigned long c = 0;
  int ch;
  for (n = 0; n == 0 && (ch = fgetc(fp)) != EOF; ) {
    unsigned char uc = (unsigned char)ch;
    if (q == 0) {
      if (((signed char)uc) < 0) {
	switch (uc & 0xf0) {
	case 0xc0: case 0xd0:
	  q = 1;
	  c = uc & 0x1f;
	  break;
	case 0xe0:
	  q = 2;
	  c = uc & 0xf;
	  break;
	case 0xf0:
	  if ((uc & 0x8) == 0) {
	    q = 3;
	    c = uc & 0x7;
	  } else
	    c = uc;
	  break;
	default:
	  c = uc;
	  break;
	}
      } else
	c = uc;
      p = 0; if (q == 0) n++;
    } else if ((uc & 0xc0) == 0x80) {
      /* continuation byte */
      c = c << 6 | (uc & 0x3f);
      if (--q == 0)
	n++;
      else
	p++;
    } else {
      /* malformed char */
      return NULL;
    }
  }
  if (n == 1)
    return u8encode(t, c);
  else
    return NULL;
}

#endif

#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)

#define CHUNKSZ 128

/* Convert strings from/to utf-8. The converted string is allocated
   dynamically and must be freed by the caller. The desired source/target
   encoding may be given as the second argument (if NULL then the system
   encoding is assumed). If no suitable conversion can be found then a copy of
   the original string is returned. */

static inline char *
toutf8(char *s, char *codeset)
{
  iconv_t ic;
  if (!codeset || !*codeset)
    codeset = default_encoding();
  if (codeset && strcmp(codeset, "UTF-8"))
    ic = iconv_open("UTF-8", codeset);
  else
    ic = (iconv_t)-1;

  if (ic == (iconv_t)-1)
    return strdup(s);

  else {
    size_t l = strlen(s);
    char *t = malloc(l+1), *t1;
    char *inbuf = s, *outbuf = t;
    size_t inbytes = l, outbytes = l;

    while (iconv(ic, &inbuf, &inbytes, &outbuf, &outbytes) ==
	   (size_t)-1)
      if (errno == E2BIG) {
	/* try to enlarge the output buffer */
	size_t k = outbuf-t;
	if ((t1 = realloc(t, l+CHUNKSZ+1))) {
	  t = t1;
	  outbuf = t+k;
	  l += CHUNKSZ;
	  outbytes += CHUNKSZ;
	} else {
	  /* memory overflow */
	  free(t);
	  return NULL;
	}
      } else {
	/* conversion error */
	free(t);
	return strdup(s);
      }

    /* terminate the output string */
    *outbuf = 0;
    iconv_close(ic);

    if (!(t1 = realloc(t, strlen(t)+1)))
      /* this shouldn't happen */
      return t;
    else
      return t1;
  }
}

static inline char *
fromutf8(char *s, char *codeset)
{
  iconv_t ic;
  if (!codeset || !*codeset)
    codeset = default_encoding();
  if (codeset && strcmp(codeset, "UTF-8"))
    ic = iconv_open(codeset, "UTF-8");
  else
    ic = (iconv_t)-1;

  if (ic == (iconv_t)-1)
    return strdup(s);

  else {
    size_t l = strlen(s);
    char *t = malloc(l+1), *t1;
    char *inbuf = s, *outbuf = t;
    size_t inbytes = l, outbytes = l;

    while (iconv(ic, &inbuf, &inbytes, &outbuf, &outbytes) ==
	   (size_t)-1)
      if (errno == E2BIG) {
	/* try to enlarge the output buffer */
	size_t k = outbuf-t;
	if ((t1 = realloc(t, l+CHUNKSZ+1))) {
	  t = t1;
	  outbuf = t+k;
	  l += CHUNKSZ;
	  outbytes += CHUNKSZ;
	} else {
	  /* memory overflow */
	  free(t);
	  return NULL;
	}
      } else {
	/* conversion error */
	free(t);
	return strdup(s);
      }

    /* here we might have to deal with a stateful encoding, so make sure that
       we emit the closing shift sequence */

    while (iconv(ic, NULL, NULL, &outbuf, &outbytes) ==
	   (size_t)-1)
      if (errno == E2BIG) {
	/* try to enlarge the output buffer */
	size_t k = outbuf-t;
	if ((t1 = realloc(t, l+CHUNKSZ+1))) {
	  t = t1;
	  outbuf = t+k;
	  l += CHUNKSZ;
	  outbytes += CHUNKSZ;
	} else {
	  /* memory overflow */
	  free(t);
	  return NULL;
	}
      } else {
	/* conversion error */
	free(t);
	return strdup(s);
      }

    /* terminate the output string */
    *outbuf = 0;
    iconv_close(ic);

    if (!(t1 = realloc(t, strlen(t)+1)))
      /* this shouldn't happen */
      return t;
    else
      return t1;
  }
}

/* Convert from/to utf-8 strings using a preallocated pair of iconv
   descriptors (ic[0] is always used for input/ictoutf8, ic[1] for
   output/icfromutf8 conversions). These functions are used for conversions in
   file operations where the conversion state must be kept between different
   invokations. */

static inline char *
icfromutf8(iconv_t ic[2], char *s)
{
  if (ic[1] == (iconv_t)-2) {
    char *codeset = default_encoding();
    if (codeset && strcmp(codeset, "UTF-8"))
      ic[1] = iconv_open(codeset, "UTF-8");
    else
      ic[1] = (iconv_t)-1;
  }
  if (ic[1] == (iconv_t)-1)
    return s?strdup(s):NULL;
  else {
    /* Here the input buffer may be NULL, to emit a terminating shift
       sequence. In this case we initialize an output buffer of size
       CHUNKSZ. */
    size_t l = s?strlen(s):0, m = s?l:CHUNKSZ;
    char *t = malloc(m+1), *t1;
    char *inbuf = s, *outbuf = t;
    size_t inbytes = l, outbytes = m;

    while (iconv(ic[1], &inbuf, &inbytes, &outbuf, &outbytes) ==
	   (size_t)-1)
      if (errno == E2BIG) {
	/* try to enlarge the output buffer */
	size_t k = outbuf-t;
	if ((t1 = realloc(t, m+CHUNKSZ+1))) {
	  t = t1;
	  outbuf = t+k;
	  m += CHUNKSZ;
	  outbytes += CHUNKSZ;
	} else {
	  /* memory overflow */
	  free(t);
	  return NULL;
	}
      } else {
	/* conversion error */
	free(t);
	return s?strdup(s):NULL;
      }
    /* terminate the output string */
    *outbuf = 0;
    if (!(t1 = realloc(t, strlen(t)+1)))
      /* this shouldn't happen */
      return t;
    else
      return t1;
  }
}

static inline char *
ictoutf8(iconv_t ic[2], char *s)
{
  if (ic[0] == (iconv_t)-2) {
    char *codeset = default_encoding();
    if (codeset && strcmp(codeset, "UTF-8"))
      ic[0] = iconv_open("UTF-8", codeset);
    else
      ic[0] = (iconv_t)-1;
  }
  if (ic[0] == (iconv_t)-1)
    return strdup(s);
  else {
    size_t l = strlen(s);
    char *t = malloc(l+1), *t1;
    char *inbuf = s, *outbuf = t;
    size_t inbytes = l, outbytes = l;

    while (iconv(ic[0], &inbuf, &inbytes, &outbuf, &outbytes) ==
	   (size_t)-1)
      if (errno == E2BIG) {
	/* try to enlarge the output buffer */
	size_t k = outbuf-t;
	if ((t1 = realloc(t, l+CHUNKSZ+1))) {
	  t = t1;
	  outbuf = t+k;
	  l += CHUNKSZ;
	  outbytes += CHUNKSZ;
	} else {
	  /* memory overflow */
	  free(t);
	  return NULL;
	}
      } else {
	/* conversion error */
	free(t);
	return strdup(s);
      }
    /* terminate the output string */
    *outbuf = 0;
    if (!(t1 = realloc(t, strlen(t)+1)))
      /* this shouldn't happen */
      return t;
    else
      return t1;
  }
}

/* Convert a single multibyte character in a file. */

static inline char *
utf8_getc(iconv_t ic[2], FILE *fp, char *t)
{
  if (ic[0] == (iconv_t)-2) {
    char *codeset = default_encoding();
    if (codeset && strcmp(codeset, "UTF-8"))
      ic[0] = iconv_open("UTF-8", codeset);
    else
      ic[0] = (iconv_t)-1;
  }
  if (ic[0] == (iconv_t)-1)
    return u8getc(fp, t);
  else {
    size_t res = (size_t)-1;
    int ch;
    char s[MB_LEN_MAX];
    char *inbuf = s, *outbuf = t;
    size_t inbytes = 0, outbytes = 4;

    while (inbytes < MB_LEN_MAX && (ch = fgetc(fp)) != EOF) {
      s[inbytes++] = (char)ch;
      res = iconv(ic[0], &inbuf, &inbytes, &outbuf, &outbytes);
      if (res != (size_t)-1)
	/* done */
	break;
      else if (errno != EINVAL)
	/* conversion error */
	return NULL;
    }
    if (res == (size_t)-1)
      /* conversion error */
      return NULL;
    else {
      /* terminate the output string */
      *outbuf = 0;
      return t;
    }
  }
}

#else

/* dummy versions of toutf8/fromutf8 to be used when iconv is not available */

static inline char *
toutf8(char *s, char *codeset)
{
  return strdup(s);
}

static inline char *
fromutf8(char *s, char *codeset)
{
  return strdup(s);
}

#endif

static inline int is_nan(double f) {
  return !(f == f);
}

/* This operation should be "failsafe", so we implement it (mostly)
   non-recursively. */

static struct xstk_entry {
  EXPR *x, *y;
} *xstk = NULL;

static int xstkp = 0, xstka = 0;

static inline EXPR* make_view(THREAD *thr, EXPR *x)
{
  int _mode = thr->mode, res;
  EXPR *f = funexpr(thr, UNPARSEOP), *y = NULL;
  thr->mode = 1;
  if (!f) {
    thr->mode = _mode;
    return NULL;
  } else if (!(y = qmnew(consexpr(thr, APPOP, f, x)))) {
    qmfree(thr, qmnew(f));
    thr->mode = _mode;
    return NULL;
  }
  thr->mode = _mode;
  res = eval(thr, y);
  qmfree(thr, y);
  if (!res) return NULL;
  y = *--thr->xsp;
  if (y->fno == APPOP && y->data.args.x1->fno == QUOTEOP) {
    EXPR *z = qmnew(y->data.args.x2);
    qmfree(thr, y);
    return z;
  } else {
    qmfree(thr, y);
    return NULL;
  }
}

static
xeq(THREAD *thr, EXPR *x, EXPR *y)
/* check two terms for syntactic equality */
{
  int mark = xstkp;
  char test;
  extern int stack_dir;
  if (cstackmax > 0 && stack_dir*(&test - thr->baseptr) >= cstackmax) {
    /* C stack overflow -- bail out */
    thr->qmstat = AST_OVF;
    return 0;
  }
 loop:
  if (x == y)
    goto pop;
  else if (!x || !y)
    goto exit;
  else if (x->fno != y->fno || x->type != y->type)
    goto exit;
  else
    switch (x->fno) {
    case INTVALOP:
      if (mpz_cmp(x->data.z, y->data.z) == 0)
	goto pop;
      else
	goto exit;
    case FLOATVALOP:
      if (x->data.f == y->data.f || is_nan(x->data.f) && is_nan(y->data.f))
	goto pop;
      else
	goto exit;
    case STRVALOP:
      if (strcmp(x->data.s, y->data.s) == 0)
	goto pop;
      else
	goto exit;
    case BADFILEVALOP:
    case FILEVALOP:
      if (x->data.fp == y->data.fp)
	goto pop;
      else
	goto exit;
    case USRVALOP:
      if (x->data.vp == y->data.vp)
	goto pop;
      if (x->type && (symtb[x->type].flags&VIRT)) {
	/* external objects with a view; generate the views and compare them
	   in place of the original expressions */
	EXPR *u = NULL, *v = NULL;
	int res = 0;
	if ((u = make_view(thr, x)) && (v = make_view(thr, y)))
	  res = xeq(thr, u, v);
	if (u) qmfree(thr, u);
	if (v) qmfree(thr, v);
	if (res) goto pop;
      }
      goto exit;
    case CONSOP:
    case PAIROP:
    case APPOP:
      if (xstkp >= xstka) {
	if (xstka >= INT_MAX ||
	    !(xstk = xstka?
	      realloc(xstk, (xstka+10240)*sizeof(struct xstk_entry)):
	      malloc(10240*sizeof(struct xstk_entry))))
	  fatal("memory overflow");
	else
	  xstka += 10240;
      }
      xstk[xstkp].x = x;
      xstk[xstkp++].y = y;
      x = x->data.args.x1;
      y = y->data.args.x1;
      goto loop;
    case VECTOP: {
      int i, n = x->data.vect.n, m = y->data.vect.n;
      if (n != m)
	goto exit;
      else {
	for (i = 0; i < n; i++)
	  if (!xeq(thr, x->data.vect.xv[i], y->data.vect.xv[i]))
	    goto exit;
	goto pop;
      }
    }
    default:
    pop:
      while (xstkp > mark && x == xstk[xstkp-1].x->data.args.x2) {
	x = xstk[--xstkp].x;
	y = xstk[xstkp].y;
      }
      if (xstkp > mark) {
	x = xstk[xstkp-1].x->data.args.x2;
	y = xstk[xstkp-1].y->data.args.x2;
	goto loop;
      }
    }
  return 1;
 exit:
  xstkp = mark;
  return 0;
}

static inline
xeqchk(THREAD *thr, EXPR *x, EXPR *y)
{
  int res;
  char base, *baseptr = thr->baseptr;
  extern int stack_dir;
  if (!baseptr)
    thr->baseptr = &base;
  if (cstackmax > 0 && stack_dir*(&base - thr->baseptr) >= cstackmax) {
    /* C stack overflow -- bail out */
    thr->qmstat = AST_OVF;
    return 0;
  }
  res = xeq(thr, x, y);
  if (!baseptr)
    thr->baseptr = NULL;
  return res;
}

static int avtbsz = 0, vtbsz = 0, *vtb = NULL;

static
add_vtb(THREAD *thr, int fno, EXPR *x)
{
  if (fno == DEFVAROP)
    return 1;
  else if (symtb[fno].xp)
    return xeqchk(thr, x, symtb[fno].xp);
  else {
    if (vtbsz >= avtbsz) {
      int *vtb1;
      if ((vtb1 = (int*)arealloc(vtb, avtbsz, 64, sizeof(int)))) {
	vtb = vtb1;
	avtbsz += 64;
      } else {
	thr0->qmstat = MEM_OVF;
	return 0;
      }
    }
    vtb[vtbsz++] = fno;
    symtb[fno].xp = qmnew(x);
    return 1;
  }
}

static void clear_vtb(THREAD *thr)
{
  int i;
  for (i = 0; i < vtbsz; i++) {
    qmfree(thr, symtb[vtb[i]].xp);
    symtb[vtb[i]].xp = NULL;
  }
  vtbsz = 0;
}

/* get the argv vector of the given expression */
static inline unsigned long get_argv(EXPR *x)
{
  if (x->fno==APPOP)
    return x->data.args.argv;
  else
    return symtb[(x)->fno].argv;
}

static
xmatch(THREAD *thr, EXPR *x, EXPR *y, int mode, int vmode)
{
  if ((symtb[x->fno].flags & VSYM))
    return add_vtb(thr, x->fno, y);
  else if (mode && y->red) {
    /* reducible special subterm matched against a non-variable, evaluate
       recursively */
    int res;
    EXPR *v;
    if (!eval(thr, y)) return 0;
    v = *--thr->xsp;
    res = xmatch(thr, x, v, 0, vmode);
    qmfree(thr, v);
    return res;
  } else if (!vmode && y->argc==0 && y->type && (symtb[y->type].flags&VIRT) &&
	     x->virt && x->argc==0 && x->type==y->type) {
    /* the target object type has a view and the pattern has a virtual
       constructor, so generate the view and match against it in place of the
       original expression */
    int _mode = thr->mode, res;
    EXPR *f, *u, *v;
    thr->mode = 1;
    f = funexpr(thr, UNPARSEOP);
    if (!f) {
      thr->mode = _mode;
      return 0;
    } else if (!(u = qmnew(consexpr(thr, APPOP, f, y)))) {
      qmfree(thr, qmnew(f));
      thr->mode = _mode;
      return 0;
    }
    thr->mode = _mode;
    res = eval(thr, u);
    y->refc++;
    qmfree(thr, u);
    y->refc--;
    if (!res) return 0;
    u = *--thr->xsp;
    if (u->fno == APPOP && u->data.args.x1->fno == QUOTEOP) {
      v = qmnew(u->data.args.x2);
      qmfree(thr, u);
      res = xmatch(thr, x, v, 0, 1);
      qmfree(thr, v);
      return res;
    } else {
      qmfree(thr, u);
    }
  }
  if (x->fno == PAIROP && y->fno == VECTOP) {
    int i = 0, n = y->data.vect.n;
    while (x->fno == PAIROP && i < n)
      if (!xmatch(thr, x->data.args.x1, y->data.vect.xv[i], 0, init_mode))
	return 0;
      else {
	x = x->data.args.x2;
	i++;
      }
    if (x->fno == PAIROP)
      return 0;
    if ((symtb[x->fno].flags & VSYM)) {
      if (i >= n)
	/* end of the vector has been reached */
	y = funexpr(thr, VOIDOP);
      else {
	/* copy vector of remaining elements (assert: n>i>0) */
	EXPR **yv = malloc((n-i)*sizeof(EXPR*));
	int j;
	if (!yv) {
	  thr->qmstat = MEM_OVF;
	  return 0;
	}
	for (j = i; j < n; j++)
	  yv[j-i] = qmnew(y->data.vect.xv[j]);
	y = vectexpr(thr, n-i, yv);
      }
      if (y)
	return add_vtb(thr, x->fno, y);
      else
	return 0;
    } else
      return 0;
  } else if (x->fno != y->fno)
    return 0;
  else
    switch (x->fno) {
    case INTVALOP:
      return mpz_cmp(x->data.z, y->data.z) == 0;
    case FLOATVALOP:
      return x->data.f == y->data.f || is_nan(x->data.f) && is_nan(y->data.f);
    case STRVALOP:
      return strcmp(x->data.s, y->data.s) == 0;
    case BADFILEVALOP:
    case FILEVALOP:
      return x->data.fp == y->data.fp;
    case USRVALOP:
      return x->data.vp == y->data.vp;
    case CONSOP:
    case PAIROP:
      return xmatch(thr, x->data.args.x1, y->data.args.x1, 0, init_mode) &&
	xmatch(thr, x->data.args.x2, y->data.args.x2, 0, init_mode);
    case APPOP:
      return xmatch(thr, x->data.args.x1, y->data.args.x1, 0, init_mode) &&
	xmatch(thr, x->data.args.x2, y->data.args.x2, get_argv(x->data.args.x1)&1, init_mode);
    case VECTOP: {
      int i, n = x->data.vect.n, m = y->data.vect.n;
      if (n!=m)
	return 0;
      else {
	for (i = 0; i < n; i++)
	  if (!xmatch(thr, x->data.vect.xv[i], y->data.vect.xv[i], 0,
		      init_mode))
	    return 0;
	return 1;
      }
    }
    default:
      return 1;
    }
}

static
qmdef(THREAD* thr)
{
  EXPR **args = thr->args;
  if (xmatch(thr, args[0], args[1], 0, init_mode)) {
    int i, res = 1;
    for (i = 0; i < vtbsz; i++)
      if (!setvar(vtb[i], symtb[vtb[i]].xp)) {
	res = 0;
	break;
      } else if (init_mode)
	symtb[vtb[i]].flags &= ~MODIF;
    clear_vtb(thr);
    if (res && pushfun(thr, VOIDOP)) {
      thr->nredns--;
      return 1;
    } else
      return 0;
  } else {
    clear_vtb(thr);
    thr->qmstat = MATCH_ERR;
    return 0;
  }
}

static
qmundef(THREAD* thr)
{
  EXPR **args = thr->args;
  if (!setvar(args[0]->fno, NULL))
    return 0;
  else {
    if (init_mode)
      symtb[args[0]->fno].flags &= ~MODIF;
    if (pushfun(thr, VOIDOP)) {
      thr->nredns--;
      return 1;
    } else
      return 0;
  }
}

typedef struct _env {
  int vno;
  EXPR *xvar;
  struct _env *next;
} env_t;

static int add_env(env_t **env, int vno, EXPR *xvar)
{
  if (vno != DEFVAROP) {
    env_t *new = malloc(sizeof(env_t));
    if (!new) return 0;
    new->vno = vno; new->xvar = xvar;
    new->next = *env;
    *env = new;
  }
  return 1;
}

static EXPR *get_env(env_t *env, int vno)
{
  if (env)
    if (vno == env->vno)
      return env->xvar;
    else
      return get_env(env->next, vno);
  else
    return NULL;
}

static void free_env(env_t *env)
{
  if (env) {
    free_env(env->next);
    free(env);
  }
}

static EXPR *make_xvar(THREAD *thr)
{
  EXPR **ref = malloc(sizeof(EXPR*));
  if (ref) {
    *ref = NULL;
    return usrexpr(thr, LAMBDAVARTYPE, ref);
  } else {
    thr->qmstat = MEM_OVF;
    return NULL;
  }
}

static EXPR *
xlpat(THREAD *thr, EXPR *x, env_t **env)
{
  char test;
  extern int stack_dir;
  if (cstackmax > 0 && stack_dir*(&test - thr->baseptr) >= cstackmax) {
    /* C stack overflow -- bail out */
    thr->qmstat = AST_OVF;
    return 0;
  }
  if ((symtb[x->fno].flags & VSYM)) {
    EXPR *xvar;
    if (x->fno != DEFVAROP && (xvar = get_env(*env, x->fno)))
      return xvar;
    else {
      if (!(xvar = make_xvar(thr))) return NULL;
      if (add_env(env, x->fno, xvar))
	return xvar;
      else {
	qmfree(thr, qmnew(xvar));
	thr->qmstat = MEM_OVF;
	return NULL;
      }
    }
  }
  switch (x->fno) {
  case CONSOP: case PAIROP: case APPOP: {
    EXPR *x1 = x->data.args.x1, *x2 = x->data.args.x2,
      *y1 = NULL, *y2 = NULL;
    if (y1 = xlpat(thr, x1, env))
      y2 = xlpat(thr, x2, env);
    if (!y1 || !y2) {
      if (y1 && y1 != x1) qmfree(thr, qmnew(y1));
      if (y2 && y2 != x2) qmfree(thr, qmnew(y2));
      return NULL;
    } else if (y1 == x1 && y2 == x2)
      return x;
    else
      return consexpr(thr, x->fno, y1, y2);
  }
  case VECTOP: {
    int i, n = x->data.vect.n;
    EXPR **xv = x->data.vect.xv, **yv = NULL;
    for (i = 0; i < n; i++) {
      EXPR *y = xlpat(thr, xv[i], env);
      if (!y) {
	if (yv) {
	  int j;
	  for (j = 0; j < i; j++) qmfree(thr, yv[j]);
	  free(yv);
	  return NULL;
	}
      } else {
	if (y != xv[i] && !yv) {
	  int j;
	  yv = malloc(n*sizeof(EXPR*));
	  if (!yv) {
	    qmfree(thr, qmnew(y));
	    return NULL;
	  }
	  for (j = 0; j < i; j++) yv[j] = qmnew(xv[j]);
	}
	if (yv) yv[i] = qmnew(y);
      }
    }
    if (yv)
      return vectexpr(thr, n, yv);
    else
      return x;
  }
  default:
    return x;
  }
}

static inline int is_lambda(EXPR *x)
{
  int type = x->type;
  if (type == 0 || x->argc > 0 || x->virt)
    return 0;
  while (type > LAMBDATYPE)
    type = symtb[type].type;
  return (type == LAMBDATYPE);
}

#define get_argv(x) (((x)->fno==APPOP)?(x)->data.args.argv:symtb[(x)->fno].argv)

typedef struct {
  EXPR *pat, *body;
  unsigned *key;
} clos_t;

static inline EXPR *cleanup(THREAD *thr, EXPR *t, EXPR *x, EXPR *y)
{
  if (x != t)
    if (y != t)
      qmfree(thr, t);
    else
      t->refc--;
  return y;
}

static EXPR *
xlbody(THREAD *thr, EXPR *x, unsigned *key, env_t *env)
{
  EXPR *xvar, *_x = x;
  char test;
  extern int stack_dir;
  if (cstackmax > 0 && stack_dir*(&test - thr->baseptr) >= cstackmax) {
    /* C stack overflow -- bail out */
    thr->qmstat = AST_OVF;
    return 0;
  }
  qmnew(x);
 retry:
  if (x->fno == APPOP && x->data.args.x1->fno == APPOP &&
      x->data.args.x1->data.args.x1->fno == LAMBDAOP) {
    /* evaluate nested lambda recursively */
    int res = eval(thr, x);
    if (res) {
      qmfree(thr, x);
      x = *--thr->xsp;
      /* falls through */
    } else {
      qmfree(thr, x);
      return NULL;
    }
  } else if (is_lambda(x)) {
    EXPR *z;
    /* expand embedded Lambda object */
    EXPR *f = funexpr(thr, LAMBDAXOP), *y;
    if (!f) {
      qmfree(thr, x);
      return NULL;
    }
    if ((y = qmnew(consexpr(thr, APPOP, f, x)))) {
      if (eval(thr, y)) {
	qmfree(thr, y);
	y = *--thr->xsp;
      } else {
	qmfree(thr, y);
	qmfree(thr, x);
	return NULL;
      }
    } else {
      qmfree(thr, qmnew(f));
      qmfree(thr, x);
      return NULL;
    }
    if (y->fno == APPOP && y->data.args.x1->fno == QUOTEOP) {
      z = qmnew(y->data.args.x2);
      qmfree(thr, y);
    } else {
      qmfree(thr, y);
      qmfree(thr, x);
      return NULL;
    }
    /* collect temporaries: */
    if (x != z) qmfree(thr, x);
    /* try again: */
    x = z;
    goto retry;
  }
  if (x->fno == USRVALOP && x->type == FUNCTIONTYPE &&
      ((clos_t*)x->data.vp)->key == key) {
    /* substitute free variables in body of embedded lambda closure */
    clos_t *clos = malloc(sizeof(clos_t)), *clos1 = (clos_t*)x->data.vp;
    if (clos) {
      EXPR *body = xlbody(thr, clos1->body, key, env);
      if (body == clos1->body) {
	free(clos);
	return cleanup(thr, x, _x, x);
      } else if (qmnew(body)) {
	clos->pat = qmnew(clos1->pat);
	clos->body = body;
	clos->key = clos1->key;
	(*(clos->key))++;
	return cleanup(thr, x, _x, usrexpr(thr, FUNCTIONTYPE, clos));
      } else {
	free(clos);
	return cleanup(thr, x, _x, NULL);
      }
    } else
      return cleanup(thr, x, _x, NULL);
  }
  if ((symtb[x->fno].flags & VSYM) && (xvar = get_env(env, x->fno)))
    return cleanup(thr, x, _x, xvar);
  switch (x->fno) {
  case CONSOP: case PAIROP: case APPOP: {
    EXPR *x1 = x->data.args.x1, *x2 = x->data.args.x2,
      *y1 = NULL, *y2 = NULL;
    if (y1 = xlbody(thr, x1, key, env))
      y2 = xlbody(thr, x2, key, env);
    if (!y1 || !y2) {
      if (y1 && y1 != x1) qmfree(thr, qmnew(y1));
      if (y2 && y2 != x2) qmfree(thr, qmnew(y2));
      return cleanup(thr, x, _x, NULL);
    } else if (y1 == x1 && y2 == x2)
      return cleanup(thr, x, _x, x);
    else
      return cleanup(thr, x, _x, consexpr(thr, x->fno, y1, y2));
  }
  case VECTOP: {
    int i, n = x->data.vect.n;
    EXPR **xv = x->data.vect.xv, **yv = NULL;
    for (i = 0; i < n; i++) {
      EXPR *y = xlbody(thr, xv[i], key, env);
      if (!y) {
	if (yv) {
	  int j;
	  for (j = 0; j < i; j++) qmfree(thr, yv[j]);
	  free(yv);
	  return cleanup(thr, x, _x, NULL);
	}
      } else {
	if (y != xv[i] && !yv) {
	  int j;
	  yv = malloc(n*sizeof(EXPR*));
	  if (!yv) {
	    qmfree(thr, qmnew(y));
	    return cleanup(thr, x, _x, NULL);
	  }
	  for (j = 0; j < i; j++) yv[j] = qmnew(xv[j]);
	}
	if (yv) yv[i] = qmnew(y);
      }
    }
    if (yv)
      return cleanup(thr, x, _x, vectexpr(thr, n, yv));
    else
      return cleanup(thr, x, _x, x);
  }
  default:
    return cleanup(thr, x, _x, x);
  }
}

static inline void add_xvar(EXPRL **xvars, EXPR *x)
{
  EXPRL *new = malloc(sizeof(EXPRL));
  if (!new) return;
  new->x = x;
  new->next = *xvars;
  *xvars = new;
}

static inline void clear_xvars(THREAD *thr, EXPRL **xvars)
{
  while (*xvars) {
    EXPRL *next = (*xvars)->next;
    EXPR **ref = (EXPR**)(*xvars)->x->data.vp;
    qmfree(thr, *ref); *ref = NULL;
    free(*xvars); *xvars = next;
  }
}

static inline int is_xvar(EXPR *x)
{
  return x->fno == USRVALOP && x->type == LAMBDAVARTYPE;
}

static inline int set_xvar(THREAD *thr, EXPRL **xvars, EXPR *x, EXPR *y)
{
  EXPR **ref = (EXPR**)x->data.vp;
  if (*ref) {
    int res = xeqchk(thr, y, *ref);
    if (y->refc == 0) qmfree(thr, qmnew(y));
    return res;
  } else {
    *ref = qmnew(y);
    add_xvar(xvars, x);
    return 1;
  }
}

static
xlmatch(THREAD *thr, EXPRL **xvars, int mode, int vmode,
	EXPR *x, EXPR *y, EXPR **z)
{
  int res;
  char test;
  extern int stack_dir;
  if (cstackmax > 0 && stack_dir*(&test - thr->baseptr) >= cstackmax) {
    /* C stack overflow -- bail out */
    thr->qmstat = AST_OVF;
    return 0;
  }
  *z = y;
  if (is_xvar(x))
    return set_xvar(thr, xvars, x, y);
  if (mode && y->red) {
    /* reducible special subterm, evaluate recursively */
    EXPR *u, *v;
    if (!eval(thr, y)) return 0;
    u = *--thr->xsp; u->refc--;
    res = xlmatch(thr, xvars, 0, vmode, x, u, &v);
    if (u != v) qmfree(thr, qmnew(u));
    *z = v;
    return res;
  } else if (!vmode && y->argc==0 && y->type && (symtb[y->type].flags&VIRT) &&
	     x->virt && x->argc==0 && x->type==y->type) {
    /* the target object type has a view and the pattern has a virtual
       constructor, so generate the view and match against it in place of the
       original expression */
    int _mode = thr->mode, res;
    EXPR *f, *u, *v;
    thr->mode = 1;
    f = funexpr(thr, UNPARSEOP);
    if (!f) {
      thr->mode = _mode;
      return 0;
    } else if (!(u = qmnew(consexpr(thr, APPOP, f, y)))) {
      qmfree(thr, qmnew(f));
      thr->mode = _mode;
      return 0;
    }
    thr->mode = _mode;
    res = eval(thr, u);
    y->refc++;
    qmfree(thr, u);
    y->refc--;
    if (!res) return 0;
    u = *--thr->xsp;
    if (u->fno == APPOP && u->data.args.x1->fno == QUOTEOP) {
      v = qmnew(u->data.args.x2);
      qmfree(thr, u);
      u = v; u->refc--;
      res = xlmatch(thr, xvars, 0, 1, x, u, &v);
      if (u != v) qmfree(thr, qmnew(u));
      *z = v;
      return res;
    } else {
      qmfree(thr, u);
    }
  }
  if (x->fno == PAIROP && y->fno == VECTOP) {
    int i = 0, n = y->data.vect.n;
    EXPR **yv = y->data.vect.xv, **zv = NULL, *z;
    res = 0;
    while (x->fno == PAIROP && i < n)
      if (!xlmatch(thr, xvars, 0, 0, x->data.args.x1, yv[i], &z)) {
	if (yv[i] != z) qmfree(thr, qmnew(z));
	goto cleanup;
      } else {
	if (yv[i] != z) {
	  if (!zv) {
	    int j;
	    if (!(zv = malloc(n*sizeof(EXPR*)))) {
	      qmfree(thr, qmnew(z));
	      goto done;
	    }
	    for (j = 0; j < i; j++) zv[j] = qmnew(yv[j]);
	  }
	  zv[i] = qmnew(z);
	} else if (zv)
	  zv[i] = qmnew(yv[i]);
	x = x->data.args.x2;
	i++;
      }
    if (x->fno == PAIROP)
      goto cleanup;
    if (is_xvar(x)) {
      if (i >= n)
	/* end of the vector has been reached */
	y = funexpr(thr, VOIDOP);
      else {
	/* copy vector of remaining elements (assert: n>i>0) */
	EXPR **yv = malloc((n-i)*sizeof(EXPR*));
	int j;
	if (!yv) {
	  thr->qmstat = MEM_OVF;
	  return 0;
	}
	for (j = i; j < n; j++) {
	  yv[j-i] = qmnew(y->data.vect.xv[j]);
	  if (zv) zv[j] = qmnew(y->data.vect.xv[j]);
	}
	y = vectexpr(thr, n-i, yv);
	i = n;
      }
      if (y)
	res = set_xvar(thr, xvars, x, y);
    }
    if (res) goto done;
  cleanup:
    if (zv) {
      int j;
      for (j = 0; j < i; j++) qmfree(thr, zv[j]);
      free(zv);
      zv = NULL;
    }
  done:
    if (zv && !(y = vectexpr(thr, n, zv)))
      res = 0;
  } else if (x->fno != y->fno)
    res = 0;
  else
    switch (x->fno) {
    case INTVALOP:
      res = mpz_cmp(x->data.z, y->data.z) == 0;
      break;
    case FLOATVALOP:
      res = x->data.f == y->data.f || is_nan(x->data.f) && is_nan(y->data.f);
      break;
    case STRVALOP:
      res = strcmp(x->data.s, y->data.s) == 0;
      break;
    case BADFILEVALOP:
    case FILEVALOP:
      res = x->data.fp == y->data.fp;
      break;
    case USRVALOP:
      res = x->data.vp == y->data.vp;
      break;
    case CONSOP:
    case PAIROP:
    case APPOP: {
      EXPR *y1 = y->data.args.x1, *y2 = y->data.args.x2, *z1 = y1, *z2 = y2;
      if (x->fno == APPOP)
	mode = (get_argv(y1)&1) != 0;
      else
	mode = 0;
      res = xlmatch(thr, xvars, 0, 0, x->data.args.x1, y1, &z1) &&
	xlmatch(thr, xvars, mode, 0, x->data.args.x2, y2, &z2);
      if (res) {
	if ((y1 != z1 || y2 != z2) && !(y = consexpr(thr, x->fno, z1, z2))) {
	  if (y1 != z1) qmfree(thr, qmnew(z1));
	  if (y2 != z2) qmfree(thr, qmnew(z2));
	  res = 0;
	}
      } else {
	if (y1 != z1) qmfree(thr, qmnew(z1));
	if (y2 != z2) qmfree(thr, qmnew(z2));
      }
      break;
    }
    case VECTOP: {
      int i, n = x->data.vect.n, m = y->data.vect.n;
      EXPR **xv = x->data.vect.xv, **yv = y->data.vect.xv, **zv = NULL, *z;
      res = 0;
      if (n!=m) goto done2;
      for (i = 0; i < n; i++) {
	if (!xlmatch(thr, xvars, 0, 0, xv[i], yv[i], &z)) {
	  if (yv[i] != z) qmfree(thr, qmnew(z));
	  if (zv) {
	    int j;
	    for (j = 0; j < i; j++) qmfree(thr, zv[j]);
	    free(zv);
	    zv = NULL;
	  }
	  goto done2;
	} else {
	  if (yv[i] != z) {
	    if (!zv) {
	      int j;
	      if (!(zv = malloc(n*sizeof(EXPR*)))) {
		qmfree(thr, qmnew(z));
		goto done2;
	      }
	      for (j = 0; j < i; j++) zv[j] = qmnew(yv[j]);
	    }
	    zv[i] = qmnew(z);
	  } else if (zv)
	    zv[i] = qmnew(yv[i]);
	}
      }
      res = 1;
      done2:
      if (zv && !(y = vectexpr(thr, n, zv)))
	res = 0;
      break;
    }
    default:
      res = 1;
      break;
    }
  *z = y;
  return res;
}

typedef struct _xenv {
  EXPR **ref;
  EXPR *xvar;
  struct _xenv *next;
} xenv_t;

static int add_xenv(xenv_t **xenv, EXPR **ref, EXPR *xvar)
{
  xenv_t *new = malloc(sizeof(xenv_t));
  if (!new) return 0;
  new->ref = ref; new->xvar = xvar;
  new->next = *xenv;
  *xenv = new;
  return 1;
}

static EXPR *get_xenv(xenv_t *xenv, EXPR **ref)
{
  if (xenv)
    if (ref == xenv->ref)
      return xenv->xvar;
    else
      return get_xenv(xenv->next, ref);
  else
    return NULL;
}

static void free_xenv(xenv_t *xenv)
{
  if (xenv) {
    free_xenv(xenv->next);
    free(xenv);
  }
}

static EXPR *
xlcppat(THREAD *thr, EXPR *x, xenv_t *xenv)
{
  char test;
  extern int stack_dir;
  if (cstackmax > 0 && stack_dir*(&test - thr->baseptr) >= cstackmax) {
    /* C stack overflow -- bail out */
    thr->qmstat = AST_OVF;
    return 0;
  }
  if (is_xvar(x)) {
    EXPR **ref = (EXPR**)x->data.vp;
    EXPR *xvar = get_xenv(xenv, ref);
    if (xvar)
      return xvar;
    else
      return x;
  }
  switch (x->fno) {
  case CONSOP: case PAIROP: case APPOP: {
    EXPR *x1 = x->data.args.x1, *x2 = x->data.args.x2,
      *y1 = NULL, *y2 = NULL;
    if (y1 = xlcppat(thr, x1, xenv))
      y2 = xlcppat(thr, x2, xenv);
    if (!y1 || !y2) {
      if (y1 && y1 != x1) qmfree(thr, qmnew(y1));
      if (y2 && y2 != x2) qmfree(thr, qmnew(y2));
      return NULL;
    } else if (y1 == x1 && y2 == x2)
      return x;
    else
      return consexpr(thr, x->fno, y1, y2);
  }
  case VECTOP: {
    int i, n = x->data.vect.n;
    EXPR **xv = x->data.vect.xv, **yv = NULL;
    for (i = 0; i < n; i++) {
      EXPR *y = xlcppat(thr, xv[i], xenv);
      if (!y) {
	if (yv) {
	  int j;
	  for (j = 0; j < i; j++) qmfree(thr, yv[j]);
	  free(yv);
	  return NULL;
	}
      } else {
	if (y != xv[i] && !yv) {
	  int j;
	  yv = malloc(n*sizeof(EXPR*));
	  if (!yv) {
	    qmfree(thr, qmnew(y));
	    return NULL;
	  }
	  for (j = 0; j < i; j++) yv[j] = qmnew(xv[j]);
	}
	if (yv) yv[i] = qmnew(y);
      }
    }
    if (yv)
      return vectexpr(thr, n, yv);
    else
      return x;
  }
  default:
    return x;
  }
}

static EXPR *
xlrepl(THREAD *thr, EXPR *x, unsigned *key, xenv_t **xenv)
{
  char test;
  extern int stack_dir;
  if (cstackmax > 0 && stack_dir*(&test - thr->baseptr) >= cstackmax) {
    /* C stack overflow -- bail out */
    thr->qmstat = AST_OVF;
    return 0;
  }
  if (is_xvar(x)) {
    EXPR **ref = (EXPR**)x->data.vp;
    EXPR *xvar;
    if (*ref) return *ref;
    /* Unbound variable in nested closure. Force a copy to prevent cycles when
       a closure happens to be applied to itself. */
    if ((xvar = get_xenv(*xenv, ref)))
      return xvar;
    else if (!(xvar = make_xvar(thr)))
      return NULL;
    if (add_xenv(xenv, ref, xvar))
      return xvar;
    else {
      qmfree(thr, qmnew(xvar));
      thr->qmstat = MEM_OVF;
      return NULL;
    }
  }
  if (x->fno == USRVALOP && x->type == FUNCTIONTYPE &&
      ((clos_t*)x->data.vp)->key == key) {
    /* nested closure, copy if necessary */
    clos_t *clos = malloc(sizeof(clos_t)), *clos1 = (clos_t*)x->data.vp;
    if (clos) {
      EXPR *body = xlrepl(thr, clos1->body, key, xenv);
      if (body == clos1->body) {
	free(clos);
	return x;
      } else if (qmnew(body)) {
	EXPR *pat = qmnew(xlcppat(thr, clos1->pat, *xenv));
	if (!pat) {
	  qmfree(thr, body);
	  free(clos);
	  return NULL;
	}
	clos->pat = pat;
	clos->body = body;
	clos->key = clos1->key;
	(*(clos->key))++;
	return usrexpr(thr, FUNCTIONTYPE, clos);
      } else {
	free(clos);
	return NULL;
      }
    } else
      return NULL;
  }
  switch (x->fno) {
  case CONSOP: case PAIROP: case APPOP: {
    EXPR *x1 = x->data.args.x1, *x2 = x->data.args.x2,
      *y1 = NULL, *y2 = NULL;
    if (y1 = xlrepl(thr, x1, key, xenv))
	y2 = xlrepl(thr, x2, key, xenv);
    if (!y1 || !y2) {
      if (y1 && y1 != x1) qmfree(thr, qmnew(y1));
      if (y2 && y2 != x2) qmfree(thr, qmnew(y2));
      return NULL;
    } else if (y1 == x1 && y2 == x2)
      return x;
    else
      return consexpr(thr, x->fno, y1, y2);
  }
  case VECTOP: {
    int i, n = x->data.vect.n;
    EXPR **xv = x->data.vect.xv, **yv = NULL;
    for (i = 0; i < n; i++) {
      EXPR *y = xlrepl(thr, xv[i], key, xenv);
      if (!y) {
	if (yv) {
	  int j;
	  for (j = 0; j < i; j++) qmfree(thr, yv[j]);
	  free(yv);
	  return NULL;
	}
      } else {
	if (y != xv[i] && !yv) {
	  int j;
	  yv = malloc(n*sizeof(EXPR*));
	  if (!yv) {
	    qmfree(thr, qmnew(y));
	    return NULL;
	  }
	  for (j = 0; j < i; j++) yv[j] = qmnew(xv[j]);
	}
	if (yv) yv[i] = qmnew(y);
      }
    }
    if (yv)
      return vectexpr(thr, n, yv);
    else
      return x;
  }
  default:
    return x;
  }
}

static qmlambda(THREAD* thr)
{
  static unsigned *key = NULL;
  EXPR **args = thr->args, *pat = args[0], *body = args[1];
  env_t *env = NULL;
  int _mode = thr->mode;
  char base, *baseptr = thr->baseptr;
  extern int stack_dir;
  if (!baseptr)
    thr->baseptr = &base;
  if (cstackmax > 0 && stack_dir*(&base - thr->baseptr) >= cstackmax) {
    /* C stack overflow -- bail out */
    thr->qmstat = AST_OVF;
    return 0;
  }
  thr->mode = 1;
  pat = qmnew(xlpat(thr, pat, &env));
  if (pat) {
    unsigned *_key = key, *actkey = key;
    if (!key) {
      /* toplevel invokation, generate a new key for this closure */
      actkey = key = malloc(sizeof(unsigned));
      *key = 0;
    }
    (*actkey)++;
    body = qmnew(xlbody(thr, body, key, env));
    thr->mode = _mode;
    free_env(env);
    key = _key;
    if (!baseptr)
      thr->baseptr = NULL;
    if (body) {
      clos_t *clos = malloc(sizeof(clos_t));
      if (clos) {
	EXPR *x;
	clos->pat = pat;
	clos->body = body;
	clos->key = actkey;
	x = usrexpr(thr, FUNCTIONTYPE, clos);
	if (x)
	  if (push(thr, x))
	    return 1;
	  else
	    qmfree(thr, x);
      } else {
	if (actkey != key) free(actkey);
	qmfree(thr, pat);
	qmfree(thr, body);
      }
    } else
      qmfree(thr, pat);
  } else {
    thr->mode = _mode;
    free_env(env);
    if (!baseptr)
      thr->baseptr = NULL;
  }
  return 0;
}

static qmlambda_app(THREAD* thr)
{
  EXPR *self = thr->self, **args = thr->args, *x = NULL, *y = args[0], *z = y;
  clos_t *clos = (clos_t*)self->data.vp;
  EXPRL *xvars = NULL;
  char base, *baseptr = thr->baseptr;
  extern int stack_dir;
  if (!baseptr)
    thr->baseptr = &base;
  if (cstackmax > 0 && stack_dir*(&base - thr->baseptr) >= cstackmax) {
    /* C stack overflow -- bail out */
    thr->qmstat = AST_OVF;
    return 0;
  }
  if (xlmatch(thr, &xvars, 0, 0, clos->pat, y, &z)) {
    xenv_t *xenv = NULL;
    int _mode = thr->mode;
    thr->mode = 1;
    x = qmnew(xlrepl(thr, clos->body, clos->key, &xenv));
    free_xenv(xenv);
    thr->mode = _mode;
  }
  if (z != y) qmfree(thr, qmnew(z));
  clear_xvars(thr, &xvars);
  if (!baseptr)
    thr->baseptr = NULL;
  if (x) {
    int res;
    res = eval(thr, x);
    qmfree(thr, x);
    return res;
  } else
    return 0;
}

void qmlambda_dtor(void *vp)
{
  THREAD *thr = get_thr();
  clos_t *clos = (clos_t*)vp;
  qmfree(thr, clos->pat);
  qmfree(thr, clos->body);
  if (--(*(clos->key)) == 0) free(clos->key);
  free(clos);
}

/* Create a view for a closure. */

typedef struct {
  int asz, sz;
  int *item;
} xxvartb_t;

static void
xxinitvartb(xxvartb_t *vtb)
{
  vtb->asz = vtb->sz = 0;
  vtb->item = NULL;
}

static inline void
xxaddvartb(xxvartb_t *vtb, int fno)
{
  if (fno != DEFVAROP) {
    if (vtb->sz >= vtb->asz) {
      int *item = (int*)arealloc(vtb->item, vtb->asz, 64, sizeof(int));
      if (item) {
	vtb->item = item;
	avtbsz += 64;
      } else {
	thr0->qmstat = MEM_OVF;
      }
    }
    vtb->item[vtb->sz++] = fno;
  }
}

static void xxclearvartb(xxvartb_t *vtb)
{
  free(vtb->item);
  vtb->asz = vtb->sz = 0;
  vtb->item = NULL;
}

static void
xscanvars(xxvartb_t *vtb, EXPR *x)
{
  if (symtb[x->fno].flags&VSYM)
    xxaddvartb(vtb, x->fno);
  else if (x->fno == USRVALOP && x->type == FUNCTIONTYPE) {
    clos_t *clos = (clos_t*)x->data.vp;
    xscanvars(vtb, clos->body);
  } else
    switch (x->fno) {
    case CONSOP:
    case PAIROP:
    case APPOP:
      xscanvars(vtb, x->data.args.x1);
      xscanvars(vtb, x->data.args.x2);
      break;
    case VECTOP: {
      int i, n = x->data.vect.n;
      for (i = 0; i < n; i++)
	xscanvars(vtb, x->data.vect.xv[i]);
      break;
    }
    }
}

static inline
xfindvar(xxvartb_t *vtb, int vno)
{
  int i;
  for (i = 0; i < vtb->sz; i++)
    if (vtb->item[i] == vno)
      return 1;
  return 0;
}

typedef struct _xxenv {
  unsigned vno;
  EXPR **ref;
  struct _xxenv *next;
} xxenv_t;

static inline char *make_varname(char *buf, xxvartb_t *vtb,
				 xxenv_t **env, EXPR **ref)
{
  xxenv_t *e = *env;
  unsigned vno = e?e->vno:0;
  int fno;
  for (; e && e->ref != ref; e = e->next) ;
  if (e) {
    sprintf(buf, "X%u", e->vno);
    return buf;
  }
  sprintf(buf, "X%u", ++vno);
  while ((fno = getsym(buf, mainno)) != NONE && xfindvar(vtb, fno))
    sprintf(buf, "X%u", ++vno);
  if (!(e = malloc(sizeof(xxenv_t))))
    return NULL;
  e->vno = vno;
  e->ref = ref;
  e->next = *env;
  *env = e;
  return buf;
}

static void pop_env(xxenv_t **env, xxenv_t *e)
{
  while (*env != e) {
    xxenv_t *next = (*env)->next;
    free(*env);
    *env = next;
  }
}

static EXPR *
xclosview(THREAD *thr, EXPR *x, xxvartb_t *vtb, xxenv_t **env)
{
  if (x->fno == USRVALOP && x->type == FUNCTIONTYPE) {
    /* compiled closure */
    clos_t *clos = (clos_t*)x->data.vp;
    xxenv_t *e = *env;
    EXPR *x1 = clos->pat, *x2 = clos->body,
      *f = funexpr(thr, LAMBDAOP), *y1 = NULL, *y2 = NULL;
    if (f && (y1 = xclosview(thr, x1, vtb, env)))
      y2 = xclosview(thr, x2, vtb, env);
    pop_env(env, e);
    if (!f || !y1 || !y2) {
      if (f) qmfree(thr, qmnew(f));
      if (y1 && y1 != x1) qmfree(thr, qmnew(y1));
      if (y2 && y2 != x2) qmfree(thr, qmnew(y2));
      return NULL;
    }
    if (y1 = consexpr(thr, APPOP, f, y1))
      return consexpr(thr, APPOP, y1, y2);
    else {
      if (f) qmfree(thr, qmnew(f));
      if (y1 && y1 != x1) qmfree(thr, qmnew(y1));
      if (y2 && y2 != x2) qmfree(thr, qmnew(y2));
      return NULL;
    }
  } else if (is_xvar(x)) {
    /* lambda variable */
    EXPR **ref = (EXPR**)x->data.vp;
    char pname[100];
    if (make_varname(pname, vtb, env, ref))
      return funexpr(thr, mksym(pname));
    else
      return NULL;
  }
  switch (x->fno) {
  case CONSOP: case PAIROP: case APPOP: {
    EXPR *x1 = x->data.args.x1, *x2 = x->data.args.x2,
      *y1 = NULL, *y2 = NULL;
    if (y1 = xclosview(thr, x1, vtb, env))
      y2 = xclosview(thr, x2, vtb, env);
    if (!y1 || !y2) {
      if (y1 && y1 != x1) qmfree(thr, qmnew(y1));
      if (y2 && y2 != x2) qmfree(thr, qmnew(y2));
      return NULL;
    } else if (y1 == x1 && y2 == x2)
      return x;
    else
      return consexpr(thr, x->fno, y1, y2);
  }
  case VECTOP: {
    int i, n = x->data.vect.n;
    EXPR **xv = x->data.vect.xv, **yv = NULL;
    for (i = 0; i < n; i++) {
      EXPR *y = xclosview(thr, xv[i], vtb, env);
      if (!y) {
	if (yv) {
	  int j;
	  for (j = 0; j < i; j++) qmfree(thr, yv[j]);
	  free(yv);
	  return NULL;
	}
      } else {
	if (y != xv[i] && !yv) {
	  int j;
	  yv = malloc(n*sizeof(EXPR*));
	  if (!yv) {
	    qmfree(thr, qmnew(y));
	    return NULL;
	  }
	  for (j = 0; j < i; j++) yv[j] = qmnew(xv[j]);
	}
	if (yv) yv[i] = qmnew(y);
      }
    }
    if (yv)
      return vectexpr(thr, n, yv);
    else
      return x;
  }
  default:
    return x;
  }
}

static qmview(THREAD *thr)
{
  EXPR *x = thr->args[0], *y;
  xxvartb_t vartb;
  xxenv_t *env = NULL;
  if (!(x->fno == USRVALOP && x->type == FUNCTIONTYPE))
    return 0;
  xxinitvartb(&vartb);
  xscanvars(&vartb, x);
  y = xclosview(thr, x, &vartb, &env);
  xxclearvartb(&vartb);
  if (y) {
    EXPR *f = funexpr(thr, QUOTEOP), *z;
    if (f && (z = consexpr(thr, APPOP, f, y)))
      return push(thr, z);
    else {
      if (f) qmfree(thr, qmnew(f));
      if (y != x) qmfree(thr, qmnew(y));
      return 0;
    }
  } else
    return 0;
}

static
strconcat(THREAD *thr, char *s1, char *s2)
 /* concatenate two strings and push the result on the stack. */
{
  char           *s;
  int		l1 = strlen(s1), l2 = strlen(s2);
  
  if (l1 >= INT_MAX - l2 || (s = malloc(l1+l2+1)) == NULL) {
    thr->qmstat = MEM_OVF;
    return (0);
  } else
    return (pushstr(thr, strcat(strcpy(s, s1), s2)));
}

static
listconcat(THREAD *thr, EXPR *x1, EXPR *x2)
 /* concatenate two lists and push the result on the stack. */
{
  int             n;

  for (n = 0; x1->fno == CONSOP; x1 = x1->data.args.x2) {
    n++;
    if (!push(thr, x1->data.args.x1))
      return (0);
  }
  if (x1->fno == NILOP) {
    if (!push(thr, x2))
      return (0);
    for (; n > 0; n--)
      if (!pushfun(thr, CONSOP))
	return (0);
    return (1);
  } else {
    for (; n > 0; n--)
      qmfree(thr, *--thr->xsp);
    return (0);
  }
}

static
vectconcat(THREAD *thr, int n1, EXPR **xv1, int n2, EXPR **xv2)
 /* concatenate two vectors and push the result on the stack. */
{
  EXPR          **xv = NULL;

  if (n1 >= INT_MAX - n2 || n1+n2>0 && (xv = malloc((n1+n2)*sizeof(EXPR*)))
      == NULL) {
    thr->qmstat = MEM_OVF;
    return 0;
  } else {
    int i;
    for (i = 0; i < n1; i++)
      xv[i] = qmnew(xv1[i]);
    for (i = 0; i < n2; i++)
      xv[n1+i] = qmnew(xv2[i]);
    return pushvect(thr, n1+n2, xv);
  }
}

static
tupleconcat(THREAD *thr, EXPR *x1, EXPR *x2)
 /* concatenate two tuples and push the result on the stack. */
{
  int             n;

  if (x1->fno == VECTOP)
    if (x2->fno == VECTOP)
      return vectconcat(thr, x1->data.vect.n, x1->data.vect.xv,
			x2->data.vect.n, x2->data.vect.xv);
    else {
      int i;
      for (i = 0; i < x1->data.vect.n; i++)
	if (!push(thr, x1->data.vect.xv[i]))
	  return (0);
      n = x1->data.vect.n;
    }
  else
    for (n = 0; x1->fno == PAIROP; x1 = x1->data.args.x2) {
      n++;
      if (!push(thr, x1->data.args.x1))
	return (0);
    }
  if (x1->fno == VECTOP || x1->fno == VOIDOP) {
    if (!push(thr, x2))
      return (0);
    for (; n > 0; n--)
      if (!pushfun(thr, PAIROP))
	return (0);
    return (1);
  } else {
    for (; n > 0; n--)
      qmfree(thr, *--thr->xsp);
    return (0);
  }
}

static qmconcat(THREAD* thr)
{
  EXPR **args = thr->args;
  switch (args[0]->fno) {
  case STRVALOP:
    if (args[1]->fno == STRVALOP)
      return (strconcat(thr, args[0]->data.s,
			args[1]->data.s));
    else
      return (0);
  case NILOP:
  case VOIDOP:
    return (push(thr, args[1]));
  case CONSOP:
    return (listconcat(thr, args[0], args[1]));
  case PAIROP:
  case VECTOP:
    return (tupleconcat(thr, args[0], args[1]));
  default:
    return (0);
  }
}

/* NOTE: for maximum performance, we should make qmadd/min/mul/div/mod/umin
   reuse an argument where possible (i.e., where refc=1) */

/* inline code for the time-critical stuff in the bigint ops */

#define __max(x,y) (((x)>=(y))?(x):(y))
#define __abs(x) (((x)>=0)?(x):-(x))
#define __sz(z) __abs((z)->_mp_size)

#define __mpzop2(f,sz,x,y) \
{ mpz_t __z; \
mpz_init(__z); \
if (__z->_mp_d && my_mpz_realloc(__z, sz)) { \
  int __sz; \
  f(__z, x, y); \
  if (!__z->_mp_d) { \
    thr->qmstat = MEM_OVF; \
    return 0; \
  } \
  __sz = mpz_size(__z); \
  if (__sz < sz && !my_mpz_realloc(__z, __sz)) { \
    thr->qmstat = MEM_OVF; \
    return 0; \
  } \
  return pushmpz(thr, __z); \
} else { \
  thr->qmstat = MEM_OVF; \
  return 0; \
}}

#define __mpzop1(f,sz,x) \
{ mpz_t __z; \
mpz_init(__z); \
if (__z->_mp_d && my_mpz_realloc(__z, sz)) { \
  int __sz; \
  f(__z, x); \
  if (!__z->_mp_d) { \
    thr->qmstat = MEM_OVF; \
    return 0; \
  } \
  __sz = mpz_size(__z); \
  if (__sz < sz && !my_mpz_realloc(__z, __sz)) { \
    thr->qmstat = MEM_OVF; \
    return 0; \
  } \
  return pushmpz(thr, __z); \
} else { \
  thr->qmstat = MEM_OVF; \
  return 0; \
}}

static qmadd(THREAD* thr)
{
  EXPR **args = thr->args;
  switch (args[0]->fno) {
  case INTVALOP:
    switch (args[1]->fno) {
    case INTVALOP: {
      /* needed: max size of arg 0 and arg 1, plus possible carry */
      int sz = __max(__sz(args[0]->data.z), __sz(args[1]->data.z))+1;
      /* aargh, this is terrible, but we have to check for possible
	 overflows here */
      if (sz < 0) return 0;
      __mpzop2(mpz_add, sz, args[0]->data.z, args[1]->data.z);
    }
    case FLOATVALOP:
      return (pushfloat(thr, mpz_get_d(args[0]->data.z) +
			args[1]->data.f));
    default:
      return (0);
    }
  case FLOATVALOP:
    switch (args[1]->fno) {
    case INTVALOP:
      return (pushfloat(thr, args[0]->data.f +
			mpz_get_d(args[1]->data.z)));
    case FLOATVALOP:
      return (pushfloat(thr, args[0]->data.f +
			args[1]->data.f));
    default:
      return (0);
    }
  default:
    {
      long c, i;
      if (args[1]->fno == INTVALOP && my_mpz_fits_slong_p(args[1]->data.z))
	i = mpz_get_si(args[1]->data.z);
      else
	return 0;
      if (args[0]->type == CHARTYPE &&
#ifdef HAVE_UNICODE
	  (c = u8decode(args[0]->data.s))+i >= 0 && c+i <= 0x10ffff) {
	char t[5], *s = strdup(u8encode(t, c+i));
#else
          (c = (unsigned char)args[0]->data.s[0])+i >= 0 && c+i <= 255) {
	char t[2], *s = strdup(charstr(t, c+i));
#endif
	if (s)
	  return pushstr(thr, s);
	else {
	  thr->qmstat = MEM_OVF;
	  return 0;
	}
      } else if (args[0]->fno >= BINARY && symtb[args[0]->fno].type &&
		 symtb[symtb[args[0]->fno].type].fno_min) {
	i += args[0]->fno;
	if (i >= symtb[args[0]->type].fno_min &&
	    i <= symtb[args[0]->type].fno_max)
	  return pushfun(thr, i);
	else
	  return 0;
      } else
	return 0;
    }
  }
}

static qmmin(THREAD* thr)
{
  EXPR **args = thr->args;
  switch (args[0]->fno) {
  case INTVALOP:
    switch (args[1]->fno) {
    case INTVALOP: {
      /* needed: max size of arg 0 and arg 1, plus possible carry */
      int sz = __max(__sz(args[0]->data.z), __sz(args[1]->data.z))+1;
      if (sz < 0) return 0;
      __mpzop2(mpz_sub, sz, args[0]->data.z, args[1]->data.z);
    }
    case FLOATVALOP:
      return (pushfloat(thr, mpz_get_d(args[0]->data.z) -
			args[1]->data.f));
    default:
      return (0);
    }
  case FLOATVALOP:
    switch (args[1]->fno) {
    case INTVALOP:
      return (pushfloat(thr, args[0]->data.f -
			mpz_get_d(args[1]->data.z)));
    case FLOATVALOP:
      return (pushfloat(thr, args[0]->data.f -
			args[1]->data.f));
    default:
      return (0);
    }
  default:
    if (args[0]->type == CHARTYPE && args[1]->type == CHARTYPE) {
#ifdef HAVE_UNICODE
      long c1 = u8decode(args[0]->data.s), c2 = u8decode(args[1]->data.s);
#else
      long c1 = (unsigned char)args[0]->data.s[0],
	c2 = (unsigned char)args[1]->data.s[0];
#endif
      return pushint(thr, c1-c2);
    } else if (args[0]->fno >= BINARY && args[1]->fno >= BINARY &&
	symtb[args[0]->fno].type &&
	symtb[args[0]->fno].type == symtb[args[1]->fno].type &&
	symtb[symtb[args[0]->fno].type].fno_min)
      return (pushint(thr, args[0]->fno-args[1]->fno));
    else {
      long c, i;
      if (args[1]->fno == INTVALOP && my_mpz_fits_slong_p(args[1]->data.z))
	i = -mpz_get_si(args[1]->data.z);
      else
	return 0;
      if (args[0]->type == CHARTYPE &&
#ifdef HAVE_UNICODE
	  (c = u8decode(args[0]->data.s))+i >= 0 && c+i <= 0x10ffff) {
	char t[5], *s = strdup(u8encode(t, c+i));
#else
          (c = (unsigned char)args[0]->data.s[0])+i >= 0 && c+i <= 255) {
	char t[2], *s = strdup(charstr(t, c+i));
#endif
	if (s)
	  return pushstr(thr, s);
	else {
	  thr->qmstat = MEM_OVF;
	  return 0;
	}
      } else if (args[0]->fno >= BINARY && symtb[args[0]->fno].type &&
		 symtb[symtb[args[0]->fno].type].fno_min) {
	i += args[0]->fno;
	if (i >= symtb[args[0]->type].fno_min &&
	    i <= symtb[args[0]->type].fno_max)
	  return pushfun(thr, i);
	else
	  return 0;
      } else
	return 0;
    }
  }
}

static qmmul(THREAD* thr)
{
  EXPR **args = thr->args;
  switch (args[0]->fno) {
  case INTVALOP:
    switch (args[1]->fno) {
    case INTVALOP: {
      /* needed: total size of arg 0 and arg 1 */
      int sz = __sz(args[0]->data.z)+__sz(args[1]->data.z);
      if (sz < 0) return 0;
      __mpzop2(mpz_mul, sz, args[0]->data.z, args[1]->data.z);
    }
    case FLOATVALOP:
      return (pushfloat(thr, mpz_get_d(args[0]->data.z) *
			args[1]->data.f));
    default:
      return (0);
    }
  case FLOATVALOP:
    switch (args[1]->fno) {
    case INTVALOP:
      return (pushfloat(thr, args[0]->data.f *
			mpz_get_d(args[1]->data.z)));
    case FLOATVALOP:
      return (pushfloat(thr, args[0]->data.f *
			args[1]->data.f));
    default:
      return (0);
    }
  default:
    return (0);
  }
}

static qmfdiv(THREAD* thr)
{
  EXPR **args = thr->args;
  /* NOTE: we assume IEEE floats here, otherwise this may raise SIGFPE */
  switch (args[0]->fno) {
  case INTVALOP:
    switch (args[1]->fno) {
    case INTVALOP:
      return (pushfloat(thr, mpz_get_d(args[0]->data.z) /
			mpz_get_d(args[1]->data.z)));
    case FLOATVALOP:
      return (pushfloat(thr, mpz_get_d(args[0]->data.z) /
			args[1]->data.f));
    default:
      return (0);
    }
  case FLOATVALOP:
    switch (args[1]->fno) {
    case INTVALOP:
      return (pushfloat(thr, args[0]->data.f /
			mpz_get_d(args[1]->data.z)));
    case FLOATVALOP:
      return (pushfloat(thr, args[0]->data.f /
			args[1]->data.f));
    default:
      return (0);
    }
  default:
    return (0);
  }
}

static qmdiv(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == INTVALOP && args[1]->fno == INTVALOP &&
      mpz_sgn(args[1]->data.z) != 0) {
    /* needed: size of arg 0 minus size of arg 1 plus 1 */
    int sz = __sz(args[0]->data.z)-__sz(args[1]->data.z)+1;
    if (sz < 0) sz = 0;
    __mpzop2(mpz_tdiv_q, sz, args[0]->data.z, args[1]->data.z);
  } else
    return (0);
}

static qmmod(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == INTVALOP && args[1]->fno == INTVALOP &&
      mpz_sgn(args[1]->data.z) != 0) {
    /* needed: size of arg 1 */
    int sz = __sz(args[1]->data.z);
    __mpzop2(mpz_tdiv_r, sz, args[0]->data.z, args[1]->data.z);
  } else
    return (0);
}

static qmpow(THREAD* thr)
{
  EXPR **args = thr->args;
  double ip;
  switch (args[0]->fno) {
  case INTVALOP:
    switch (args[1]->fno) {
    case INTVALOP:
      if (mpz_sgn(args[0]->data.z) != 0 ||
	  mpz_sgn(args[1]->data.z) != 0)
	return (pushfloat(thr, pow(mpz_get_d(args[0]->data.z),
			      mpz_get_d(args[1]->data.z))));
      else
	return (0);
    case FLOATVALOP:
      if (is_nan(args[1]->data.f) ||
	  (mpz_sgn(args[0]->data.z) >= 0 ||
	   modf(args[1]->data.f, &ip) == 0.0) &&
	  (mpz_sgn(args[0]->data.z) != 0 ||
	   args[1]->data.f != 0.0))
	return (pushfloat(thr, pow(mpz_get_d(args[0]->data.z),
			      args[1]->data.f)));
      else
	return (0);
    default:
      return (0);
    }
  case FLOATVALOP:
    switch (args[1]->fno) {
    case INTVALOP:
      if (is_nan(args[0]->data.f) ||
	  args[0]->data.f != 0.0 ||
	  mpz_sgn(args[1]->data.z) != 0)
	return (pushfloat(thr, pow(args[0]->data.f,
			      mpz_get_d(args[1]->data.z))));
      else
	return (0);
    case FLOATVALOP:
      if (is_nan(args[0]->data.f) ||
	  is_nan(args[1]->data.f) ||
	  (args[0]->data.f >= 0.0 ||
	   modf(args[1]->data.f, &ip) == 0.0) &&
	  (args[0]->data.f != 0.0 ||
	   args[1]->data.f != 0.0))
	return (pushfloat(thr, pow(args[0]->data.f,
			      args[1]->data.f)));
      else
	return (0);
    default:
      return (0);
    }
  default:
    return (0);
  }
}

/* NOTE: qmidx, qmhash, qmsub, qmsubstr and qmpos should probably use mpz's for
   indexing. At least as soon as handling of 2GB strings and lists becomes
   everyday business. But in that future long values might have 128 bits
   anyway. ;-) */

static qmidx(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[1]->fno == INTVALOP && my_mpz_fits_slong_p(args[1]->data.z)) {
    long		i = mpz_get_si(args[1]->data.z);
    switch (args[0]->fno) {
    case STRVALOP: {
      char           *s, *s1 = args[0]->data.s,	s2[5];
      if (i >= 0) {
#ifdef HAVE_UNICODE
	char *t1 = u8strind(s1, i), *t2 = u8strind(t1, 1);
#else
	size_t l = strlen(s1);
	char *t1 = (i<l)?s1+i:s1+l, *t2 = (i+1<l)?s1+i+1:s1+l;
#endif
	if (*t1) {
	  strncpy(s2, t1, t2-t1);
	  s2[t2-t1] = 0;
	  if ((s = strdup(s2)) == NULL) {
	    thr->qmstat = MEM_OVF;
	    return (0);
	  } else
	    return (pushstr(thr, s));
	} else
	  return 0;
      } else
	return 0;
    }
    case CONSOP: {
      EXPR           *x = args[0];
      for (; i > 0 && x->fno == CONSOP; i--)
	x = x->data.args.x2;
      if (x->fno == CONSOP && i >= 0)
	return (push(thr, x->data.args.x1));
      else
	return (0);
    }
    case PAIROP: {
      EXPR           *x = args[0];
      for (; i > 0 && x->fno == PAIROP; i--)
	x = x->data.args.x2;
      if (x->fno == PAIROP && i >= 0)
	return (push(thr, x->data.args.x1));
      else
	return (0);
    }
    case VECTOP:
      if (i >= 0 && i < args[0]->data.vect.n)
	return (push(thr, args[0]->data.vect.xv[i]));
      else
	return (0);
    default:
      return (0);
    }
  } else
    return (0);
}

static qmcomp(THREAD* thr)
{
  EXPR **args = thr->args;
  int _mode = thr->mode;
  thr->mode = 1;
  if (push(thr, args[0]) && push(thr, args[1]) && push(thr, args[2]) &&
      pushfun(thr, APPOP) && pushfun(thr, APPOP)) {
    thr->mode = _mode;
    return 1;
  } else {
    thr->mode = _mode;
    return 0;
  }
}

static qmumin(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == INTVALOP) {
    /* needed: size of arg 0 */
    int sz = __sz(args[0]->data.z);
    __mpzop1(mpz_neg, sz, args[0]->data.z);
  } else if (args[0]->fno == FLOATVALOP)
    return (pushfloat(thr, -args[0]->data.f));
  else
    return (0);
}

static qmhash(THREAD* thr)
{
  EXPR **args = thr->args;
  EXPR           *x = args[0];
  long		l;

  switch (x->fno) {
  case STRVALOP:
#ifdef HAVE_UNICODE
    l = (long) u8strlen(x->data.s);
#else
    l = (long) strlen(x->data.s);
#endif
    if (l >= 0)
      return (pushint(thr, l));
    else
      return (0);
  case NILOP:
  case CONSOP:
    for (l = 0; x->fno == CONSOP; l++)
      x = x->data.args.x2;
    if (x->fno == NILOP && l >= 0)
      return (pushint(thr, l));
    else
      return (0);
  case VOIDOP:
  case PAIROP:
    for (l = 0; x->fno == PAIROP; l++)
      x = x->data.args.x2;
    if (x->fno == VOIDOP && l >= 0)
      return (pushint(thr, l));
    else
      return (0);
  case VECTOP:
    return (pushint(thr, x->data.vect.n));
  default:
    return (0);
  }
}

static qmunquote(THREAD* thr)
{
  EXPR **args = thr->args;
  if (eval(thr, args[0])) {
    EXPR *x = thr->xsp[-1];
    if (x->fno == APPOP && x->data.args.x1->fno == QUOTEOP) {
      x = qmnew(x->data.args.x2);
      qmfree(thr, thr->xsp[-1]);
      thr->xsp[-1] = x;
      if (!thr->mode) {
	int res = eval(thr, (x = *--thr->xsp));
	qmfree(thr, x);
	return res;
      }
      return 1;
    } else
      return 1;
  } else
    return 0;
}

static qmforce(THREAD* thr)
{
  EXPR **args = thr->args;
  return eval(thr, args[0]);
}

static qmmem(THREAD* thr)
{
  EXPR **args = thr->args;
  /* This will be only invoked for non-special args where (&) is simply the
     identity. */
  return push(thr, args[0]);
}

static qmor(THREAD* thr)
{
  EXPR **args = thr->args;
  EXPR *x = args[0], *y = args[1];
  if (x->fno == INTVALOP && y->fno == INTVALOP) {
    int sz = __max(__sz(x->data.z), __sz(y->data.z)) + 1;
    if (sz < 0) return 0;
    __mpzop2(mpz_ior, sz, x->data.z, y->data.z);
  } else if (y->type != BOOLTYPE)
    return (0);
  else if (x->fno == FALSEOP)
    return (push(thr, y));
  else if (x->fno == TRUEOP)
    return (push(thr, x));
  else
    return (0);
}

static qmorelse(THREAD* thr)
{
  EXPR **args = thr->args;
  EXPR	       *x = args[0], *y = args[1];
  if (x->fno == FALSEOP)
    return (push(thr, y));
  else if (x->fno == TRUEOP)
    return (push(thr, x));
  else
    return (0);
}

static qmand(THREAD* thr)
{
  EXPR **args = thr->args;
  EXPR *x = args[0], *y = args[1];
  if (x->fno == INTVALOP && y->fno == INTVALOP) {
    int sz = __max(__sz(x->data.z), __sz(y->data.z)) + 1;
    if (sz < 0) return 0;
    __mpzop2(mpz_and, sz, x->data.z, y->data.z);
  } else if (y->type != BOOLTYPE)
    return (0);
  else if (x->fno == TRUEOP)
    return (push(thr, y));
  else if (x->fno == FALSEOP)
    return (push(thr, x));
  else
    return (0);
}

static qmandthen(THREAD* thr)
{
  EXPR **args = thr->args;
  EXPR	       *x = args[0], *y = args[1];
  if (x->fno == TRUEOP)
    return (push(thr, y));
  else if (x->fno == FALSEOP)
    return (push(thr, x));
  else
    return (0);
}

static qmnot(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == INTVALOP) {
    int sz = __sz(args[0]->data.z) + 1;
    if (sz < 0) return 0;
    __mpzop1(mpz_com, sz, args[0]->data.z);
  } else if (args[0]->fno == TRUEOP)
    return (pushfun(thr, FALSEOP));
  else if (args[0]->fno == FALSEOP)
    return (pushfun(thr, TRUEOP));
  else
    return (0);
}

static
xcmp(EXPR *x, EXPR *y, int *result)
 /* compare two terms */
{
  if (x->fno >= BINARY && y->fno >= BINARY &&
      symtb[x->fno].type && symtb[x->fno].type == symtb[y->fno].type &&
      symtb[symtb[x->fno].type].fno_min) {
    *result = x->fno-y->fno;
    return 1;
  } else if (x->fno != y->fno)
    switch (x->fno) {
    case INTVALOP:
      if (y->fno == FLOATVALOP) {
	double xf = mpz_get_d(x->data.z);
	if (xf < y->data.f)
	  *result = -1;
	else if (xf > y->data.f)
	  *result = 1;
	else if (xf == y->data.f)
	  *result = 0;
	else
	  return (0);
	return (1);
      } else
	return (0);
    case FLOATVALOP:
      if (y->fno == INTVALOP) {
	double yf = mpz_get_d(y->data.z);
	if (x->data.f < yf)
	  *result = -1;
	else if (x->data.f > yf)
	  *result = 1;
	else if (x->data.f == yf)
	  *result = 0;
	else
	  return (0);
	return (1);
      } else
	return (0);
    default:
      return (0);
    }
  else
    switch (x->fno) {
    case INTVALOP:
      *result = mpz_cmp(x->data.z, y->data.z);
      return (1);
    case FLOATVALOP:
      if (x->data.f < y->data.f)
	*result = -1;
      else if (x->data.f > y->data.f)
	*result = 1;
      else if (x->data.f == y->data.f)
	*result = 0;
      else
        return (0);
      return (1);
    case STRVALOP:
      *result = strcmp(x->data.s, y->data.s);
      return (1);
    default:
      return (0);
    }
}

static inline int check_nan(EXPR *x)
{
  return x->fno == FLOATVALOP && is_nan(x->data.f);
}

static qmle(THREAD* thr)
{
  EXPR **args = thr->args;
  int             result;

  if (check_nan(args[0]) || check_nan(args[1]))
    return (pushfun(thr, FALSEOP));
  else if (xcmp(args[0], args[1], &result))
    if (result < 0)
      return (pushfun(thr, TRUEOP));
    else
      return (pushfun(thr, FALSEOP));
  else
    return (0);
}

static qmgr(THREAD* thr)
{
  EXPR **args = thr->args;
  int             result;

  if (check_nan(args[0]) || check_nan(args[1]))
    return (pushfun(thr, FALSEOP));
  else if (xcmp(args[0], args[1], &result))
    if (result > 0)
      return (pushfun(thr, TRUEOP));
    else
      return (pushfun(thr, FALSEOP));
  else
    return (0);
}

static qmeq(THREAD* thr)
{
  EXPR **args = thr->args;
  int             result;

  if (check_nan(args[0]) || check_nan(args[1]))
    return (pushfun(thr, FALSEOP));
  else if (xcmp(args[0], args[1], &result))
    if (result == 0)
      return (pushfun(thr, TRUEOP));
    else
      return (pushfun(thr, FALSEOP));
  else
    return (0);
}

static qmleq(THREAD* thr)
{
  EXPR **args = thr->args;
  int             result;

  if (check_nan(args[0]) || check_nan(args[1]))
    return (pushfun(thr, FALSEOP));
  else if (xcmp(args[0], args[1], &result))
    if (result <= 0)
      return (pushfun(thr, TRUEOP));
    else
      return (pushfun(thr, FALSEOP));
  else
    return (0);
}

static qmgeq(THREAD* thr)
{
  EXPR **args = thr->args;
  int             result;

  if (check_nan(args[0]) || check_nan(args[1]))
    return (pushfun(thr, FALSEOP));
  else if (xcmp(args[0], args[1], &result))
    if (result >= 0)
      return (pushfun(thr, TRUEOP));
    else
      return (pushfun(thr, FALSEOP));
  else
    return (0);
}

static qmneq(THREAD* thr)
{
  EXPR **args = thr->args;
  int             result;

  if (check_nan(args[0]) || check_nan(args[1]))
    return (pushfun(thr, FALSEOP));
  else if (xcmp(args[0], args[1], &result))
    if (result != 0)
      return (pushfun(thr, TRUEOP));
    else
      return (pushfun(thr, FALSEOP));
  else
    return (0);
}

static qmid(THREAD* thr)
{
  EXPR **args = thr->args;
  if (xeqchk(thr, args[0], args[1]))
    return (pushfun(thr, TRUEOP));
  else
    return (pushfun(thr, FALSEOP));
}

static qmrapp(THREAD* thr)
{
  EXPR **args = thr->args;
  int _mode = thr->mode;
  thr->mode = 1;
  if (push(thr, args[0]) && push(thr, args[1]) &&
      pushfun(thr, APPOP)) {
    thr->mode = _mode;
    return 1;
  } else {
    thr->mode = _mode;
    return 0;
  }
}

static qmseq(THREAD* thr)
{
  EXPR **args = thr->args;
  return (push(thr, args[1]));
}

static shl(THREAD *thr, mpz_t z, int i)
{
  int n = sizeof(mp_limb_t)*8, m = __sz(z), r = i/n, k = i%n, s;
  mpz_t u;
  if (m > INT_MAX-r-1) return 0;
  mpz_init(u);
  if (m == 0) return pushmpz(thr, u);
  if (!u->_mp_d || !my_mpz_realloc(u, m+r+1)) {
    thr->qmstat = MEM_OVF;
    return 0;
  }
  mpz_set(u, z);
  if (k) {
    mp_limb_t x = mpn_lshift(u->_mp_d, u->_mp_d, m, k);
    u->_mp_d[m] = x;
  } else
    u->_mp_d[m] = 0;
  for (s = m; s >= 0; s--) u->_mp_d[s+r] = u->_mp_d[s];
  for (s = 0; s < r; s++) u->_mp_d[s] = 0;
  m += r;
  if (u->_mp_d[m]) m++;
  if (z->_mp_size < 0)
    u->_mp_size = -m;
  else
    u->_mp_size = m;
  if (__sz(z)+r+1 != m && !my_mpz_realloc(u, m)) {
    thr->qmstat = MEM_OVF;
    return 0;
  }
  return pushmpz(thr, u);
}

static shr(THREAD *thr, mpz_t z, int i)
{
  int n = sizeof(mp_limb_t)*8, m = __sz(z), r = i/n, k = i%n, s;
  mpz_t u;
  mpz_init(u);
  if (r >= m) return pushmpz(thr, u);
  if (!u->_mp_d || !my_mpz_realloc(u, m)) {
    thr->qmstat = MEM_OVF;
    return 0;
  }
  for (s = r; s < m; s++) u->_mp_d[s-r] = z->_mp_d[s];
  m -= r;
  if (k) mpn_rshift(u->_mp_d, u->_mp_d, m, k);
  if (!u->_mp_d[m-1]) m--;
  if (z->_mp_size < 0)
    u->_mp_size = -m;
  else
    u->_mp_size = m;
  if (!my_mpz_realloc(u, m)) {
    thr->qmstat = MEM_OVF;
    return 0;
  }
  return pushmpz(thr, u);
}

static qmshl(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == INTVALOP && args[1]->fno == INTVALOP &&
      my_mpz_fits_slong_p(args[1]->data.z)) {
    long i = mpz_get_si(args[1]->data.z);
    if (i > 0)
      return shl(thr, args[0]->data.z, i);
    else if (i == INT_MIN)
      /* calculating -i will overflow */
      return 0;
    else if (i < 0)
      return shr(thr, args[0]->data.z, -i);
    else {
      mpz_t u;
      mpz_init(u);
      if (!u->_mp_d || !my_mpz_realloc(u, __sz(args[0]->data.z))) {
	thr->qmstat = MEM_OVF;
	return 0;
      }
      mpz_set(u, args[0]->data.z);
      return pushmpz(thr, u);
    }
  } else
    return 0;
}

static qmshr(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == INTVALOP && args[1]->fno == INTVALOP &&
      my_mpz_fits_slong_p(args[1]->data.z)) {
    long i = mpz_get_si(args[1]->data.z);
    if (i > 0)
      return shr(thr, args[0]->data.z, i);
    else if (i == INT_MIN)
      /* calculating -i will overflow */
      return 0;
    else if (i < 0)
      return shl(thr, args[0]->data.z, -i);
    else {
      mpz_t u;
      mpz_init(u);
      if (!u->_mp_d || !my_mpz_realloc(u, __sz(args[0]->data.z))) {
	thr->qmstat = MEM_OVF;
	return 0;
      }
      mpz_set(u, args[0]->data.z);
      return pushmpz(thr, u);
    }
  } else
    return 0;
}

static qmpred(THREAD* thr)
{
  EXPR **args = thr->args;
  long c;
  if (args[0]->type == CHARTYPE &&
#ifdef HAVE_UNICODE
      (c = u8decode(args[0]->data.s)) > 0) {
    char t[5], *s = strdup(u8encode(t, c-1));
#else
    (c = (unsigned char)args[0]->data.s[0]) > 0) {
    char t[2], *s = strdup(charstr(t, c-1));
#endif
    if (s)
      return pushstr(thr, s);
    else {
      thr->qmstat = MEM_OVF;
      return 0;
    }
  } else if (args[0]->type && symtb[args[0]->type].fno_min &&
	     args[0]->fno > symtb[args[0]->type].fno_min)
    return pushfun(thr, args[0]->fno-1);
  else
    return 0;
}

static qmsucc(THREAD* thr)
{
  EXPR **args = thr->args;
  long c;
  if (args[0]->type == CHARTYPE &&
#ifdef HAVE_UNICODE
      (c = u8decode(args[0]->data.s)) < 0x10ffff) {
    char t[5], *s = strdup(u8encode(t, c+1));
#else
    (c = (unsigned char)args[0]->data.s[0]) < 255) {
    char t[2], *s = strdup(charstr(t, c+1));
#endif
    if (s)
      return pushstr(thr, s);
    else {
      thr->qmstat = MEM_OVF;
      return 0;
    }
  } else if (args[0]->type && symtb[args[0]->type].fno_min &&
	     args[0]->fno < symtb[args[0]->type].fno_max)
    return pushfun(thr, args[0]->fno+1);
  else
    return 0;
}

static qmenum(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[1]->type == CHARTYPE) {
#ifdef HAVE_UNICODE
    long x0, x1, x2 = u8decode(args[1]->data.s);
    long inc, n;
    if (args[0]->type == CHARTYPE) {
      x0 = u8decode(args[0]->data.s);
      inc = 1;
    } else {
      EXPR *x = args[0];
      n = 0;
      while (x->fno == CONSOP && x->data.args.x1->type == CHARTYPE && n < 2) {
	if (n==0)
	  x0 = u8decode(x->data.args.x1->data.s);
	else
	  x1 = u8decode(x->data.args.x1->data.s);
	n++;
	x = x->data.args.x2;
      }
      if (x->fno != NILOP || n == 0)
	return 0;
      else if (n == 2)
	inc = x1-x0;
      else
	inc = 1;
    }
    if (inc == 0) return 0;
    n = 0;
    if (inc < 0) {
      while (x0 >= x2) {
	char t[5], *s = strdup(u8encode(t, x0));
	if (!s) {
	  thr->qmstat = MEM_OVF;
	  return 0;
	}
	if (!pushstr(thr, s)) return 0;
	x0 += inc;
	n++;
      }
    } else {
      while (x0 <= x2) {
	char t[5], *s = strdup(u8encode(t, x0));
	if (!s) {
	  thr->qmstat = MEM_OVF;
	  return 0;
	}
	if (!pushstr(thr, s)) return 0;
	x0 += inc;
	n++;
      }
    }
#else
    unsigned char x0, x1, x2 = (unsigned char)*args[1]->data.s;
    int inc, n;
    if (args[0]->type == CHARTYPE) {
      x0 = (unsigned char)*args[0]->data.s;
      inc = 1;
    } else {
      EXPR *x = args[0];
      n = 0;
      while (x->fno == CONSOP && x->data.args.x1->type == CHARTYPE && n < 2) {
	if (n==0)
	  x0 = (unsigned char)*x->data.args.x1->data.s;
	else
	  x1 = (unsigned char)*x->data.args.x1->data.s;
	n++;
	x = x->data.args.x2;
      }
      if (x->fno != NILOP || n == 0)
	return 0;
      else if (n == 2)
	inc = x1-x0;
      else
	inc = 1;
    }
    if (inc == 0) return 0;
    n = 0;
    if (inc < 0) {
      while (x0 >= x2) {
	char *s = malloc(2);
	if (!s) {
	  thr->qmstat = MEM_OVF;
	  return 0;
	}
	s[0] = (char)x0;
	s[1] = 0;
	if (!pushstr(thr, s)) return 0;
	x0 += inc;
	n++;
      }
    } else {
      while (x0 <= x2) {
	char *s = malloc(2);
	if (!s) {
	  thr->qmstat = MEM_OVF;
	  return 0;
	}
	s[0] = (char)x0;
	s[1] = 0;
	if (!pushstr(thr, s)) return 0;
	x0 += inc;
	n++;
      }
    }
#endif
    if (!pushfun(thr, NILOP)) return 0;
    while (n-- > 0)
      if (!pushfun(thr, CONSOP)) return 0;
    return 1;
  } else if (args[1]->type && symtb[args[1]->type].fno_min) {
    int x0, x1, x2 = args[1]->fno;
    int inc, n;
    if (args[0]->type == args[1]->type) {
      x0 = args[0]->fno;
      inc = 1;
    } else {
      EXPR *x = args[0];
      n = 0;
      while (x->fno == CONSOP && x->data.args.x1->type == args[1]->type &&
	     n < 2) {
	if (n==0)
	  x0 = x->data.args.x1->fno;
	else
	  x1 = x->data.args.x1->fno;
	n++;
	x = x->data.args.x2;
      }
      if (x->fno != NILOP || n == 0)
	return 0;
      else if (n == 2)
	inc = x1-x0;
      else
	inc = 1;
    }
    if (inc == 0) return 0;
    n = 0;
    if (inc < 0) {
      while (x0 >= x2) {
	if (!pushfun(thr, x0)) return 0;
	x0 += inc;
	n++;
      }
    } else {
      while (x0 <= x2) {
	if (!pushfun(thr, x0)) return 0;
	x0 += inc;
	n++;
      }
    }
    if (!pushfun(thr, NILOP)) return 0;
    while (n-- > 0)
      if (!pushfun(thr, CONSOP)) return 0;
    return 1;
  } else
    return 0;
}

static qmenum1(THREAD* thr)
{
  EXPR **args = thr->args;
  int x0, x1, x2, t0, t1;
  int inc, n;
  if (args[0]->type && symtb[args[0]->type].fno_min) {
    x0 = args[0]->fno;
    t0 = args[0]->type;
    inc = 1;
  } else {
    EXPR *x = args[0];
    n = 0;
    while (x->fno == CONSOP && n < 2) {
      if (n==0) {
	x0 = x->data.args.x1->fno;
	t0 = x->data.args.x1->type;
      } else {
	x1 = x->data.args.x1->fno;
	t1 = x->data.args.x1->type;
      }
      n++;
      x = x->data.args.x2;
    }
    if (x->fno != NILOP || n == 0 || !t0 || !symtb[t0].fno_min ||
	n > 1 && t0 != t1)
      return 0;
    else if (n == 2)
      inc = x1-x0;
    else
      inc = 1;
  }
  if (inc == 0) return 0;
  n = 0;
  if (inc < 0) {
    x2 = symtb[t0].fno_min;
    while (x0 >= x2) {
      if (!pushfun(thr, x0)) return 0;
      x0 += inc;
      n++;
    }
  } else {
    x2 = symtb[t0].fno_max;
    while (x0 <= x2) {
      if (!pushfun(thr, x0)) return 0;
      x0 += inc;
      n++;
    }
  }
  if (!pushfun(thr, NILOP)) return 0;
  while (n-- > 0)
    if (!pushfun(thr, CONSOP)) return 0;
  return 1;
}

static qmexp(THREAD* thr)
{
  EXPR **args = thr->args;
  switch (args[0]->fno) {
  case INTVALOP:
    return (pushfloat(thr, exp(mpz_get_d(args[0]->data.z))));
  case FLOATVALOP:
    return (pushfloat(thr, exp(args[0]->data.f)));
  default:
    return (0);
  }
}

static qmln(THREAD* thr)
{
  EXPR **args = thr->args;
  /* NOTE: we assume IEEE floats here, otherwise this may raise SIGFPE */
  switch (args[0]->fno) {
  case INTVALOP:
    if (mpz_sgn(args[0]->data.z) >= 0)
      return (pushfloat(thr, log(mpz_get_d(args[0]->data.z))));
    else
      return (0);
  case FLOATVALOP:
    if (is_nan(args[0]->data.f) || args[0]->data.f >= 0)
      return (pushfloat(thr, log(args[0]->data.f)));
    else
      return (0);
  default:
    return (0);
  }
}

static qmsqrt(THREAD* thr)
{
  EXPR **args = thr->args;
  switch (args[0]->fno) {
  case INTVALOP:
    if (mpz_sgn(args[0]->data.z) >= 0)
      return (pushfloat(thr, sqrt(mpz_get_d(args[0]->data.z))));
    else
      return (0);
  case FLOATVALOP:
    if (is_nan(args[0]->data.f) || args[0]->data.f >= 0)
      return (pushfloat(thr, sqrt(args[0]->data.f)));
    else
      return (0);
  default:
    return (0);
  }
}

static qmsin(THREAD* thr)
{
  EXPR **args = thr->args;
  switch (args[0]->fno) {
  case INTVALOP:
    return (pushfloat(thr, sin(mpz_get_d(args[0]->data.z))));
  case FLOATVALOP:
    return (pushfloat(thr, sin(args[0]->data.f)));
  default:
    return (0);
  }
}

static qmcos(THREAD* thr)
{
  EXPR **args = thr->args;
  switch (args[0]->fno) {
  case INTVALOP:
    return (pushfloat(thr, cos(mpz_get_d(args[0]->data.z))));
  case FLOATVALOP:
    return (pushfloat(thr, cos(args[0]->data.f)));
  default:
    return (0);
  }
}

static qmatan(THREAD* thr)
{
  EXPR **args = thr->args;
  switch (args[0]->fno) {
  case INTVALOP:
    return (pushfloat(thr, atan(mpz_get_d(args[0]->data.z))));
  case FLOATVALOP:
    return (pushfloat(thr, atan(args[0]->data.f)));
  default:
    return (0);
  }
}

static qmatan2(THREAD* thr)
{
  EXPR **args = thr->args;
  switch (args[0]->fno) {
  case INTVALOP:
    switch (args[1]->fno) {
    case INTVALOP:
      return (pushfloat(thr, atan2(mpz_get_d(args[0]->data.z),
			      mpz_get_d(args[1]->data.z))));
    case FLOATVALOP:
      return (pushfloat(thr, atan2(mpz_get_d(args[0]->data.z),
			      args[1]->data.f)));
    default:
      return (0);
    }
  case FLOATVALOP:
    switch (args[1]->fno) {
    case INTVALOP:
      return (pushfloat(thr, atan2(args[0]->data.f,
			      mpz_get_d(args[1]->data.z))));
    case FLOATVALOP:
      return (pushfloat(thr, atan2(args[0]->data.f,
			      args[1]->data.f)));
    default:
      return (0);
    }
  default:
    return (0);
  }
}

static qmrandom(THREAD* thr)
{
  EXPR **args = thr->args;
  mpz_t z;
  mpz_init(z);
  if (z->_mp_d) {
    mpz_set_ui(z, randomMT());
    return pushmpz(thr, z);
  } else {
    thr->qmstat = MEM_OVF;
    return 0;
  }
}

static qmseed(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == INTVALOP && mpz_sgn(args[0]->data.z) >= 0) {
    seedMT(mpz_get_ui(args[0]->data.z) << 1 | 1);
    return (pushfun(thr, VOIDOP));
  } else
    return (0);
}

static qmsub(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[1]->fno == INTVALOP && my_mpz_fits_slong_p(args[1]->data.z) &&
      args[2]->fno == INTVALOP && my_mpz_fits_slong_p(args[2]->data.z)) {
    long		i = mpz_get_si(args[1]->data.z);
    long		j = mpz_get_si(args[2]->data.z);
    long		c, l;
    if (i < 0) i = 0; c = j-i+1;
    switch (args[0]->fno) {
    case STRVALOP: {
#ifdef HAVE_UNICODE
      char           *s1 = args[0]->data.s, *s2, *t1, *t2;
      if (++j < i ) j = i;
      t1 = u8strind(s1, i);
      t2 = u8strind(t1, j-i);
      if ((l = t2-t1) < 0) l = 0;
      if ((s2 = malloc(l+1)) == NULL) {
	thr->qmstat = MEM_OVF;
	return (0);
      }
      if (l > 0)
	substr(s2, t1, l);
      else
	*s2 = 0;
      return (pushstr(thr, s2));
#else
      char           *s1 = args[0]->data.s, *s2;
      l = strlen(s1);
      if (i >= l || j < i)
	l = 0;
      else if ((l -= i) > c)
	l = c;
      if (l < 0)
	l = 0;
      if ((s2 = malloc(l+1)) == NULL) {
	thr->qmstat = MEM_OVF;
	return (0);
      }
      if (l > 0)
	substr(s2, s1+i, l);
      else
	*s2 = 0;
      return (pushstr(thr, s2));
#endif
    }
    case NILOP:
    case CONSOP: {
      EXPR *x = args[0], *x2;
      if (j < i)
	return pushfun(thr, NILOP);
      while (x->fno == CONSOP && i > 0)
	x = x->data.args.x2, i--;
      x2 = x; l = c;
      while (x2->fno == CONSOP && c > 0)
	x2 = x2->data.args.x2, c--;
      if (x2->fno == NILOP)
	return push(thr, x);
      else if (c > 0)
	return 0;
      for (x2 = x, c = 0; c < l; x2 = x2->data.args.x2, c++)
	if (!push(thr, x2->data.args.x1)) return 0;
      if (!pushfun(thr, NILOP)) return 0;
      for (c = 0; c < l; c++)
	if (!pushfun(thr, CONSOP)) return 0;
      return 1;
    }
    case VOIDOP:
    case PAIROP: {
      EXPR *x = args[0], *x2, **xv;
      if (j < i)
	return pushfun(thr, VOIDOP);
      while (x->fno == PAIROP && i > 0)
	x = x->data.args.x2, i--;
      x2 = x; l = c;
      while (x2->fno == PAIROP && c > 0)
	x2 = x2->data.args.x2, c--;
      if (x2->fno == VOIDOP)
	/* actually this case shouldn't arise (tuple is a legal vector and
	   hence should be handled in the default case) */
	return push(thr, x);
      else if (c > 0)
	return 0;
      /* build a vector from the collected tuple members */
      if ((xv = (EXPR**)malloc(l*sizeof(EXPR*))) == NULL) {
	thr->qmstat = MEM_OVF;
	return (0);
      }
      for (x2 = x, c = 0; c < l; x2 = x2->data.args.x2, c++)
	xv[c] = qmnew(x2->data.args.x1);
      return pushvect(thr, l, xv);
    }
    case VECTOP: {
      EXPR **xv, **xv1 = args[0]->data.vect.xv;
      l = args[0]->data.vect.n;
      if (i >= l || j < i)
	l = 0;
      else if ((l -= i) > c)
	l = c;
      if (l <= 0)
	return pushfun(thr, VOIDOP);
      else if ((xv = (EXPR**)malloc(l*sizeof(EXPR*))) == NULL) {
	thr->qmstat = MEM_OVF;
	return (0);
      }
      for (c = 0; c < l; c++)
	xv[c] = qmnew(xv1[i+c]);
      return pushvect(thr, l, xv);
    }
    default:
      return (0);
    }
  } else
    return 0;
}

static qmsubstr(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == STRVALOP &&
      args[1]->fno == INTVALOP && my_mpz_fits_slong_p(args[1]->data.z) &&
      args[2]->fno == INTVALOP && my_mpz_fits_slong_p(args[2]->data.z)) {
    char           *s1 = args[0]->data.s, *s2, *t1, *t2;
    long		i = mpz_get_si(args[1]->data.z);
    long		c = mpz_get_si(args[2]->data.z);
    long		l;

#ifdef HAVE_UNICODE
    if (i < 0) i = 0;
    if (c<0) c = 0;
    t1 = u8strind(s1, i);
    t2 = u8strind(t1, c);
    if ((l = t2-t1) < 0) l = 0;
    if ((s2 = malloc(l+1)) == NULL) {
      thr->qmstat = MEM_OVF;
      return (0);
    }
    if (l > 0)
      substr(s2, t1, l);
    else
      *s2 = 0;
    return (pushstr(thr, s2));
#else
    if (i < 0) i = 0;
    l = strlen(s1);
    if (i >= l || c <= 0)
      l = 0;
    else if ((l -= i) > c)
      l = c;
    if (l < 0)
      l = 0;
    if ((s2 = malloc(l+1)) == NULL) {
      thr->qmstat = MEM_OVF;
      return (0);
    }
    if (l > 0)
      substr(s2, s1+i, l);
    else
      *s2 = 0;
    return (pushstr(thr, s2));
#endif
  } else
    return (0);
}

static qmpos(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == STRVALOP && args[1]->fno == STRVALOP) {
    char           *s1 = args[0]->data.s;
    char           *s2 = args[1]->data.s;
    char           *s;
    
    if ((s = strstr(s2, s1)) != NULL)
#ifdef HAVE_UNICODE
      return (pushint(thr, (long) u8strpos(s2, s-s2)));
#else
      return (pushint(thr, (long) (s - s2)));
#endif
    else
      return (pushint(thr, (long) -1));
  } else
    return (0);
}

static qmint(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == FLOATVALOP) {
    double ip, fp;
    fp = modf(args[0]->data.f, &ip);
    return pushfloat(thr, ip);
  } else if (args[0]->fno == INTVALOP)
    return pushfloat(thr, mpz_get_d(args[0]->data.z));
  else
    return 0;
}

static qmfrac(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == FLOATVALOP) {
    double ip, fp;
    fp = modf(args[0]->data.f, &ip);
    return pushfloat(thr, fp);
  } else if (args[0]->fno == INTVALOP)
    return pushfloat(thr, 0.0);
  else
    return 0;
}

static qmtrunc(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == FLOATVALOP) {
    double ip, fp, dsz;
    int sz;
    fp = modf(args[0]->data.f, &ip);
    /* estimate the number of limbs required */
    dsz = log(__abs(ip))/log(2)/((double)CHAR_BIT*sizeof(mp_limb_t))+1.0;
    if (dsz < 1.0) dsz = 1.0; /* this can't happen?? */
    /* add an extra limb to be safe */
    sz = ((int)dsz)+1;
    /* this shouldn't happen but ... ;-) */
    if (((double)INT_MAX) <= dsz || sz < 0) return 0;
    __mpzop1(mpz_set_d, sz, ip);
  } else if (args[0]->fno == INTVALOP)
    return push(thr, args[0]);
  else
    return 0;
}

static qmround(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == FLOATVALOP) {
    double ip, fp, dsz;
    int sz;
    fp = modf(args[0]->data.f, &ip);
    ip += (fp>=0.5)?1:(fp<=-0.5)?-1:0;
    dsz = log(__abs(ip))/log(2)/((double)CHAR_BIT*sizeof(mp_limb_t))+1.0;
    if (dsz < 1.0) dsz = 1.0;
    sz = ((int)dsz)+1;
    if (((double)INT_MAX) <= dsz || sz < 0) return 0;
    __mpzop1(mpz_set_d, sz, ip);
  } else if (args[0]->fno == INTVALOP)
    return push(thr, args[0]);
  else
    return 0;
}

static qmfloat(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == INTVALOP)
    return pushfloat(thr, mpz_get_d(args[0]->data.z));
  else if (args[0]->fno == FLOATVALOP)
    return push(thr, args[0]);
  else
    return 0;
}

static unsigned mpz_hash(mpz_t z)
{
  unsigned h = 0;
  int i, len = z->_mp_size;
  if (len < 0) len = -len;
  for (i=0; i<len; i++)
    h ^= z->_mp_d[i];
  if (z->_mp_size < 0)
    h = -h;
  return h;
}

static unsigned float_hash(double d)
{
  unsigned h;
  char *c;
  int i;
  c = (char*)&d;
  for (h=0, i=0; i<sizeof(double); i++) {
    h += c[i] * 971;
  }
  return h;
}

static unsigned str_hash(char *s)
{
  unsigned h = 0, g;
  while (*s) {
    h = (h<<4)+*(s++);
    if ((g = (h & 0xf0000000)))	{
      h = h^(g>>24);
      h = h^g;
    }
  }
  return h;
}

static unsigned ptr_hash(void *p)
{
  return (unsigned)p;
}

static unsigned expr_hash(EXPR *x)
{
  switch (x->fno) {
  case INTVALOP:
    return mpz_hash(x->data.z);
  case FLOATVALOP:
    return float_hash(x->data.f);
  case STRVALOP:
    return str_hash(x->data.s);
  case FILEVALOP: case BADFILEVALOP:
    return ptr_hash(x->data.fp);
  case VECTOP: {
    int i, h;
    h = x->data.vect.n;
    for (i = 0; i < x->data.vect.n; i++) {
	h = (h<<1) | (h<0 ? 1 : 0);
	h ^= expr_hash(x->data.vect.xv[i]);
    }
    return (unsigned)h;
  }
  case USRVALOP:
    return ptr_hash(x->data.vp);
  case CONSOP: case PAIROP: case APPOP: {
    int h;
    h = expr_hash(x->data.args.x1);
    h = (h<<1) | (h<0 ? 1 : 0);
    h ^= expr_hash(x->data.args.x2);
    return (unsigned)h;
  }
  default:
    return (unsigned)x->fno;
  }
}

static qmhashnum(THREAD* thr)
{
  EXPR **args = thr->args;
  return pushuint(thr, expr_hash(args[0]));
}

static qmord(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->type == CHARTYPE)
#ifdef HAVE_UNICODE
    return (pushint(thr, u8decode(args[0]->data.s)));
#else
    return (pushint(thr, (long) (unsigned char) args[0]->data.s[0]));
#endif
  else if (args[0]->type && symtb[args[0]->type].fno_min)
    return pushint(thr, args[0]->fno - symtb[args[0]->type].fno_min);
  else
    return (0);
}

static qmchr(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == INTVALOP &&
      my_mpz_fits_slong_p(args[0]->data.z)) {
    long i = mpz_get_si(args[0]->data.z);
#ifdef HAVE_UNICODE
    if (i >= 0 && i < 0x110000) {
      char            s[5], *t;
      if ((t = strdup(u8encode(s, i)))
	  == NULL) {
#else
    if (i >= 0 && i < 256) {
      char            s[2], *t;
      if ((t = strdup(charstr(s, (char) i)))
	  == NULL) {
#endif
	thr->qmstat = MEM_OVF;
	return (0);
      } else
	return (pushstr(thr, t));
    } else
      return (0);
  } else
    return (0);
}

static qmlist(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == VECTOP) {
    int i, n = args[0]->data.vect.n;
    EXPR **xv = args[0]->data.vect.xv;
    EXPR *x = funexpr(thr, NILOP);
    for (i = n-1; x && i >= 0; i--) {
      EXPR *y = consexpr(thr, CONSOP, xv[i], x);
      if (!y) qmfree(thr, x);
      x = y;
    }
    return push(thr, x);
  } else if (args[0]->fno == VOIDOP)
    return pushfun(thr, NILOP);
  else
    return 0;
}

static qmtuple(THREAD* thr)
{
  EXPR **args = thr->args;
  EXPR *x = args[0];
  int n = 0;
  while (x->fno == CONSOP) {
    n++;
    x = x->data.args.x2;
  }
  if (x->fno == NILOP) {
    EXPR **xv = (n>0)?malloc(n*sizeof(EXPR*)):NULL;
    int i = 0;
    x = args[0];
    while (x->fno == CONSOP) {
      xv[i++] = qmnew(x->data.args.x1);
      x = x->data.args.x2;
    }
    return pushvect(thr, n, xv);
  } else
    return 0;
}

static qmstr(THREAD* thr)
{
  EXPR **args = thr->args;
  char            *s;
  if ((s = sprintx(args[0])) == NULL) {
    thr->qmstat = MEM_OVF;
    return (0);
  } else
    return (pushstr(thr, s));
}

static qmval(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == STRVALOP)
    if (sparsex(args[0]->data.s))
      return 1;
    else {
      if (thr->qmstat == SYNTAX_ERR || thr->qmstat == STR_ERR ||
	  thr->qmstat == BAD_ESC ||
	  thr->qmstat == BAD_SYM || thr->qmstat == BAD_REF)
	thr->qmstat = OK;
      return (0);
    }
  else
    return (0);
}

#define isquote(x,y) (((x)->fno==APPOP&&(x)->data.args.x1->fno==QUOTEOP)?\
		      (((y)=(x)->data.args.x2),1):0)

static qmstrq(THREAD* thr)
{
  EXPR **args = thr->args;
  char            *s;
  EXPR		*x;
  if (isquote(args[0], x))
    if ((s = sprintx(x)) == NULL) {
      thr->qmstat = MEM_OVF;
      return (0);
    } else
      return (pushstr(thr, s));
  else
    return (0);
}

static qmvalq(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == STRVALOP)
    if (pushfun(thr, QUOTEOP))
      if (sparsex(args[0]->data.s))
	return (pushfun(thr, APPOP));
      else {
	qmfree(thr, *--thr->xsp);
	if (thr->qmstat == SYNTAX_ERR || thr->qmstat == STR_ERR ||
	    thr->qmstat == BAD_ESC ||
	    thr->qmstat == BAD_SYM || thr->qmstat == BAD_REF)
	  thr->qmstat = OK;
	return (0);
      }
    else
      return (0);
  else
    return (0);
}

static qmisspecial(THREAD* thr)
{
  EXPR **args = thr->args;
  unsigned long argv = (args[0]->fno==APPOP)?args[0]->data.args.argv:
    symtb[args[0]->fno].argv;
  if (argv & 1)
    return (pushfun(thr, TRUEOP));
  else
    return (pushfun(thr, FALSEOP));
}

static qmisconst(THREAD* thr)
{
  EXPR **args = thr->args;
  EXPR *x = args[0];
  while (x->fno == APPOP) x = x->data.args.x1;
  if (x->fno < BINARY || (symtb[x->fno].flags & CST))
    return (pushfun(thr, TRUEOP));
  else
    return (pushfun(thr, FALSEOP));
}

static qmisfun(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno >= BINARY && !(symtb[args[0]->fno].flags & VSYM))
    return (pushfun(thr, TRUEOP));
  else
    return (pushfun(thr, FALSEOP));
}

static qmisvar(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno >= BINARY && (symtb[args[0]->fno].flags & VSYM))
    return (pushfun(thr, TRUEOP));
  else
    return (pushfun(thr, FALSEOP));
}

static qmisdef(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno >= BINARY && (symtb[args[0]->fno].flags & VSYM))
    return (pushfun(thr, symtb[args[0]->fno].x?TRUEOP:FALSEOP));
  else
    return (0);
}

static qmflip(THREAD* thr)
{
  EXPR **args = thr->args;
  int _mode = thr->mode;
  thr->mode = 1;
  if (push(thr, args[0]) && push(thr, args[2]) &&
      pushfun(thr, APPOP) && push(thr, args[1]) &&
      pushfun(thr, APPOP)) {
    thr->mode = _mode;
    return 1;
  } else {
    thr->mode = _mode;
    return 0;
  }
}

static qmfread(THREAD* thr);

static qmread(THREAD* thr)
{
  EXPR **args = thr->args;
  if (symtb[INPUTOP].x) {
    args[0] = (EXPR*) symtb[INPUTOP].x;
    return (qmfread(thr));
  } else
    return (0);
}

static qmfreadq(THREAD* thr);

static qmreadq(THREAD* thr)
{
  EXPR **args = thr->args;
  if (symtb[INPUTOP].x) {
    args[0] = (EXPR*) symtb[INPUTOP].x;
    return (qmfreadq(thr));
  } else
    return (0);
}

static qmfreadc(THREAD* thr);

static qmreadc(THREAD* thr)
{
  EXPR **args = thr->args;
  if (symtb[INPUTOP].x) {
    args[0] = (EXPR*) symtb[INPUTOP].x;
    return (qmfreadc(thr));
  } else
    return (0);
}

static qmfreads(THREAD* thr);

static qmreads(THREAD* thr)
{
  EXPR **args = thr->args;
  if (symtb[INPUTOP].x) {
    args[0] = (EXPR*) symtb[INPUTOP].x;
    return (qmfreads(thr));
  } else
    return (0);
}

static qmfwrite(THREAD* thr);

static qmwrite(THREAD* thr)
{
  EXPR **args = thr->args;
  if (symtb[OUTPUTOP].x) {
    args[1] = args[0];
    args[0] = (EXPR*) symtb[OUTPUTOP].x;
    return (qmfwrite(thr));
  } else
    return (0);
}

static qmfwriteq(THREAD* thr);

static qmwriteq(THREAD* thr)
{
  EXPR **args = thr->args;
  if (symtb[OUTPUTOP].x) {
    args[1] = args[0];
    args[0] = (EXPR*) symtb[OUTPUTOP].x;
    return (qmfwriteq(thr));
  } else
    return (0);
}

static qmfwritec(THREAD* thr);

static qmwritec(THREAD* thr)
{
  EXPR **args = thr->args;
  if (symtb[OUTPUTOP].x) {
    args[1] = args[0];
    args[0] = (EXPR*) symtb[OUTPUTOP].x;
    return (qmfwritec(thr));
  } else
    return (0);
}

static qmfwrites(THREAD* thr);

static qmwrites(THREAD* thr)
{
  EXPR **args = thr->args;
  if (symtb[OUTPUTOP].x) {
    args[1] = args[0];
    args[0] = (EXPR*) symtb[OUTPUTOP].x;
    return (qmfwrites(thr));
  } else
    return (0);
}

static qmfread(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == FILEVALOP) {
    FILE           *fp;
    int             ret;
    fp = args[0]->data.fp;
#ifdef USE_THREADS
    release_lock();
#endif
#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
    ret = fparsex(fp, args[0]->data.fargs.ic);
#else
    ret = fparsex(fp);
#endif
#ifdef USE_THREADS
    acquire_lock();
#endif
    if (ret)
      return 1;
    else {
      if (ferror(fp)) clearerr(fp);
      if (thr->qmstat == SYNTAX_ERR || thr->qmstat == STR_ERR ||
	  thr->qmstat == BAD_ESC ||
	  thr->qmstat == BAD_SYM || thr->qmstat == BAD_REF)
	thr->qmstat = OK;
      return (0);
    }
  } else
    return (0);
}

static qmfreadq(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == FILEVALOP) {
    FILE           *fp;
    fp = args[0]->data.fp;
    if (pushfun(thr, QUOTEOP)) {
      int ret;
#ifdef USE_THREADS
      release_lock();
#endif
#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
      ret = fparsex(fp, args[0]->data.fargs.ic);
#else
      ret = fparsex(fp);
#endif
#ifdef USE_THREADS
      acquire_lock();
#endif
      if (ret)
	return (pushfun(thr, APPOP));
      else {
	if (ferror(fp)) clearerr(fp);
	qmfree(thr, *--thr->xsp);
	if (thr->qmstat == SYNTAX_ERR || thr->qmstat == STR_ERR ||
	    thr->qmstat == BAD_ESC ||
	    thr->qmstat == BAD_SYM || thr->qmstat == BAD_REF)
	  thr->qmstat = OK;
	return (0);
      }
    } else
      return (0);
  } else
    return (0);
}

static qmfreadc(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == FILEVALOP) {
    FILE           *fp;
    int		c;
#ifdef HAVE_UNICODE
    char            s[5], *t;
#else
    char            s[2], *t;
#endif
    fp = args[0]->data.fp;
#ifdef USE_THREADS
    release_lock();
    if (fp == stdin) acquire_tty();
#endif
#ifdef HAVE_UNICODE
#ifdef HAVE_ICONV
    t = utf8_getc(args[0]->data.fargs.ic, fp, s);
#else
    t = u8getc(fp, s);
#endif
    if (t == NULL) {
#else
    c = getc(fp);
    if (c == EOF) {
#endif
      if (ferror(fp)) clearerr(fp);
#ifdef USE_THREADS
      if (fp == stdin) release_tty();
      acquire_lock();
#endif
      return (0);
    } else if (thr == thr0 && checkbrk) {
      while (c != '\n' && c != EOF)
	c = getc(fp);
#ifdef USE_THREADS
      if (fp == stdin) release_tty();
      acquire_lock();
#endif
      return (0);
    }
#ifdef USE_THREADS
    if (fp == stdin) release_tty();
    acquire_lock();
#endif
#ifdef HAVE_UNICODE
    if ((t = strdup(s)) == NULL) {
#else
    if ((t = strdup(charstr(s, c))) == NULL) {
#endif
      thr->qmstat = MEM_OVF;
      return (0);
    } else
      return (pushstr(thr, t));
  } else
    return (0);
}

static qmfreads(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == FILEVALOP) {
    FILE           *fp;
    char            *s = malloc(MAXSTRLEN*sizeof(char)), *t = s, *r;
    int             a = MAXSTRLEN, l;
    if (!s) {
      thr->qmstat = MEM_OVF;
      return (0);
    }
    *s = 0;
    fp = args[0]->data.fp;
#ifdef USE_THREADS
    release_lock();
    if (fp == stdin) acquire_tty();
#if 0
    pthread_mutex_lock(&reads_mutex);
#endif
#endif
    while((r = fgets(t, MAXSTRLEN, fp)) && *t &&
	  t[(l = strlen(t))-1] != '\n') {
      /* try to enlarge the buffer: */
      int k = t-s+l;
      char *s1;
      if (s1 = (char*) arealloc(s, a, MAXSTRLEN,
				sizeof(char))) {
	s = s1;
	t = s+k;
	a += MAXSTRLEN;
      } else {
	free(s);
#ifdef USE_THREADS
#if 0
	pthread_mutex_unlock(&reads_mutex);
#endif
	if (fp == stdin) release_tty();
	acquire_lock();
#endif
	thr->qmstat = MEM_OVF;
	return (0);
      }
    }
    if (*t && t[(l = strlen(t))-1] == '\n')
      t[l-1] = 0;
#ifdef USE_THREADS
#if 0
    pthread_mutex_unlock(&reads_mutex);
#endif
    if (fp == stdin) release_tty();
    acquire_lock();
#endif
    if (ferror(fp)) {
      clearerr(fp);
      free(s);
      return (0);
    }
    if (!r && !*s || thr == thr0 && checkbrk) {
      free(s);
      return (0);
#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
    } else if (!(t = ictoutf8(args[0]->data.fargs.ic, s))) {
      free(s);
      thr->qmstat = MEM_OVF;
      return (0);
    } else {
      free(s);
      return (pushstr(thr, t));
    }
#else
    } else if (!(t = realloc(s, strlen(s)+1))) {
      free(s);
      thr->qmstat = MEM_OVF;
      return (0);
    } else
      return (pushstr(thr, t));
#endif
  } else
    return (0);
}

/* Define this to have the standard output streams flushed on each write
   operation. This has a high performance penalty, but might be required on
   some systems to get correct output in interactive prompt/input
   situations. */

/* #define FLUSH_STDIO */

#ifdef FLUSH_STDIO
static chkflush(FILE *fp)
{
  if (iflag && (fp == stdout || fp == stderr)) fflush(fp);
}
#endif

static qmfwrite(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == FILEVALOP) {
    FILE           *fp;
    fp = args[0]->data.fp;
#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
    if (!fprintx(fp, args[0]->data.fargs.ic, args[1])) {
#else
    if (!fprintx(fp, args[1])) {
#endif
      if (ferror(fp)) clearerr(fp);
      return (0);
    } else {
#ifdef FLUSH_STDIO
      chkflush(fp);
#endif
      return (pushfun(thr, VOIDOP));
    }
  } else
    return (0);
}

static qmfwriteq(THREAD* thr)
{
  EXPR **args = thr->args;
  EXPR		*x;
  if (args[0]->fno == FILEVALOP && isquote(args[1], x)) {
    FILE           *fp;
    fp = args[0]->data.fp;
#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
    if (!fprintx(fp, args[0]->data.fargs.ic, x)) {
#else
    if (!fprintx(fp, x)) {
#endif
      if (ferror(fp)) clearerr(fp);
      return (0);
    } else {
#ifdef FLUSH_STDIO
      chkflush(fp);
#endif
      return (pushfun(thr, VOIDOP));
    }
  } else
    return (0);
}

static qmfwritec(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == FILEVALOP) {
    if (args[1]->type == CHARTYPE) {
      FILE           *fp;
      char           *s = args[1]->data.s;
#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
      char *t = icfromutf8(args[0]->data.fargs.ic, s);
      if (!t) return 0;
      s = t;
#endif
      fp = args[0]->data.fp;
      while (*s)
	if (putc(*s++, fp) == EOF) {
	  clearerr(fp);
#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
	  free(t);
#endif
	  return (0);
	}
#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
      free(t);
#endif
#ifdef FLUSH_STDIO
      chkflush(fp);
#endif
      return (pushfun(thr, VOIDOP));
    } else
      return (0);
  } else
    return (0);
}

static qmfwrites(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == FILEVALOP) {
    if (args[1]->fno == STRVALOP) {
      FILE           *fp;
      char           *s = args[1]->data.s;
#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
      char *t = icfromutf8(args[0]->data.fargs.ic, s);
      if (!t) return 0;
      s = t;
#endif
      fp = args[0]->data.fp;
      while (*s)
	if (putc(*s++, fp) == EOF) {
	  clearerr(fp);
#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
	  free(t);
#endif
	  return (0);
	}
#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
      free(t);
#endif
#ifdef FLUSH_STDIO
      chkflush(fp);
#endif
      return (pushfun(thr, VOIDOP));
    } else
      return (0);
  } else
    return (0);
}

static char modestr[3];

static qmfopen(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == STRVALOP && args[1]->fno == STRVALOP) {
    FILE           *fp;
    char           *name = args[0]->data.s;
    char           *mode = args[1]->data.s;

    if ((mode[0] == 'r' || mode[0] == 'w' || mode[0] == 'a') &&
	(mode[1] == '\0' || mode[1] == 'b' && mode[2] == '\0')) {
      strcpy(modestr, mode);
#ifndef MSDOS
      modestr[1] = '\0';
#endif
#ifdef USE_THREADS
      release_lock();
#endif
#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
      name = fromutf8(name, NULL);
#endif
      fp = fopen(name, modestr);
#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
      free(name);
#endif
#ifdef USE_THREADS
      acquire_lock();
#endif
      if (fp == NULL)
	return (0);
      else
	return (pushfile(thr, fp));
    } else
      return (0);
  } else
    return (0);
}

static qmpopen(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == STRVALOP && args[1]->fno == STRVALOP) {
    FILE           *fp;
    char           *cmd = args[0]->data.s;
    char           *mode = args[1]->data.s;

    if ((mode[0] == 'r' || mode[0] == 'w') &&
	(mode[1] == '\0' || mode[1] == 'b' && mode[2] == '\0')) {
      strcpy(modestr, mode);
#ifndef MSDOS
      modestr[1] = '\0';
#endif
#ifdef USE_THREADS
      release_lock();
#endif
#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
      cmd = fromutf8(cmd, NULL);
#endif
      fp = popen(cmd, modestr);
#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
      free(cmd);
#endif
#ifdef USE_THREADS
      acquire_lock();
#endif
      if (fp == NULL)
	return (0);
      else {
	setlinebuf(fp);
	return (pushpipe(thr, fp));
      }
    } else
      return (0);
  } else
    return (0);
}

static qmfclose(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == FILEVALOP) {
    int res;
#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
    if (args[0]->data.fargs.ic[0] != (iconv_t)-2 &&
	args[0]->data.fargs.ic[0] != (iconv_t)-1) {
      iconv_close(args[0]->data.fargs.ic[0]);
      args[0]->data.fargs.ic[0] = (iconv_t)-1;
    }
    if (args[0]->data.fargs.ic[1] != (iconv_t)-2 &&
	args[0]->data.fargs.ic[1] != (iconv_t)-1) {
      /* In a stateful encoding we might have to emit a terminating shift
	 sequence. */
      char *s = icfromutf8(args[0]->data.fargs.ic, NULL), *t = s;
      if (t) {
	while (*s) putc(*s++, args[0]->data.fp);
	free(t);
      }
      iconv_close(args[0]->data.fargs.ic[1]);
      args[0]->data.fargs.ic[1] = (iconv_t)-1;
    }
#endif
    res = fclose(args[0]->data.fp);
    args[0]->data.fp = NULL;
    args[0]->fno = BADFILEVALOP;
    if (res)
      return 0;
    else
      return pushfun(thr, VOIDOP);
  } else
    return 0;
}

static qmfeof(THREAD* thr);

static qmeof(THREAD* thr)
{
  EXPR **args = thr->args;
  if (symtb[INPUTOP].x) {
    args[0] = (EXPR*) symtb[INPUTOP].x;
    return (qmfeof(thr));
  } else
    return (0);
}

static qmfeof(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == FILEVALOP) {
    FILE           *fp;
    int             c;
    fp = args[0]->data.fp;
#ifdef USE_THREADS
    release_lock();
    if (fp == stdin) acquire_tty();
#endif
    c = getc(fp);
    if (c == EOF)
      if (ferror(fp)) {
	clearerr(fp);
#ifdef USE_THREADS
	if (fp == stdin) release_tty();
	acquire_lock();
#endif
	return 0;
      } else {
#ifdef USE_THREADS
	if (fp == stdin) release_tty();
	acquire_lock();
#endif
	return (pushfun(thr, TRUEOP));
      }
    else {
      ungetc(c, fp);
#ifdef USE_THREADS
      if (fp == stdin) release_tty();
      acquire_lock();
#endif
      return (pushfun(thr, FALSEOP));
    }
  } else
    return (0);
}

static qmfflush(THREAD* thr);

static qmflush(THREAD* thr)
{
  EXPR **args = thr->args;
  if (symtb[OUTPUTOP].x) {
    args[0] = (EXPR*) symtb[OUTPUTOP].x;
    return (qmfflush(thr));
  } else
    return (0);
}

static qmfflush(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == FILEVALOP) {
    FILE           *fp;
    fp = args[0]->data.fp;
    if (fflush(fp) == EOF) {
      clearerr(fp);
      return 0;
    } else
      return (pushfun(thr, VOIDOP));
  } else
    return (0);
}

static qmversion(THREAD* thr)
{
  return pushstr(thr, toutf8(version, NULL));
}

static qmsysinfo(THREAD* thr)
{
  return pushstr(thr, toutf8(sysinfo, NULL));
}

static qmwhich(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == STRVALOP) {
    char *s = args[0]->data.s;
    if ((s = fromutf8(s, NULL))) {
      int l0 = strlen(s), l = strlen(qpath)+l0;
      char *buf = malloc((l+3)*sizeof(char));
      char *name = malloc((l+MAXSTRLEN+3)*sizeof(char));
      if (buf && name) {
	absname(name, searchlib(buf, s));
	if (!chkfile(name)) {
	  char *t = malloc((l0+3)*sizeof(char));
	  if (!t) goto errexit;
	  strcat(strcpy(t, s), ".q");
	  absname(name, searchlib(buf, t));
	  free(t);
	}
	free(buf);
	if (chkfile(name)) {
	  free(s);
	  s = toutf8(name, NULL);
	  free(name);
	  return pushstr(thr, s);
	} else {
	  free(s);
	  free(name);
	  return 0;
	}
      } else {
      errexit:
	thr->qmstat = MEM_OVF;
	free(s);
	if (buf) free(buf);
	if (name) free(name);
	return 0;
      }
    } else {
      thr->qmstat = MEM_OVF;
      return 0;
    }
  } else if (args[0]->fno == VOIDOP)
    return pushstr(thr, toutf8(which, NULL));
  else
    return 0;
}

static qmhalt(THREAD* thr)
{
  thr->qmstat = HALT;
  return (0);
}

static qmquit(THREAD* thr)
{
  thr->qmstat = QUIT;
  return (0);
}

static qmbreak(THREAD* thr)
{
  if (thr->brkdbg) thr->brkflag = 1;
  return (pushfun(thr, VOIDOP));
}

static qmcatch(THREAD* thr)
{
  EXPR **args = thr->args;
  EXPR *x = args[0], *y = args[1];
  long ap = thr->asp-thr->ast;
  if (push_mark(thr, x) && eval(thr, y) && (thr > thr0 || !checkbrk)) {
    pop_mark(thr);
    return 1;
  } else if (thr == thr0 && checkbrk ||
	     thr->qmstat >= BREAK && thr->qmstat <= XCEPT) {
    if (thr->qmstat == OK)
      if (thr == thr0 && quitflag)
	thr->qmstat_save = QUIT;
      else
	thr->qmstat_save = BREAK;
    else
      thr->qmstat_save = thr->qmstat;
    thr->qmstat = XCEPT_CATCH;
    /* rewind the activation stack */
    while (thr->asp-thr->ast > ap) free(*--thr->asp);
    return 0;
  } else {
    /* other error */
    pop_mark(thr);
    return 0;
  }
}

static qmthrow(THREAD* thr)
{
  EXPR *x = *--thr->xsp;
  qmfree(thr, *--thr->xsp);
  *(thr->xsp++) = x;
  thr->qmstat = XCEPT;
  return (0);
}

static bool initsig[NSIGNALS];
static sighandler_t old_handler[NSIGNALS];
#ifdef HAVE_POSIX_SIGNALS
static struct sigaction new_action, old_action;
#endif

static qmtrap(THREAD* thr)
{
  EXPR **args = thr->args;
  long prev_action = 0;
#ifdef HAVE_POSIX_SIGNALS
  static bool init = 0;
  int i;
  if (!init) {
    sigemptyset(&new_action.sa_mask);
    new_action.sa_flags = 0;
    init = 1;
  }
#else
  sighandler_t res;
#endif
  if (args[0]->fno == INTVALOP && args[1]->fno == INTVALOP &&
      my_mpz_fits_slong_p(args[1]->data.z)) {
    int flag = mpz_sgn(args[0]->data.z);
    long sig = mpz_get_si(args[1]->data.z);
    if (sig <= 0 || sig > NSIGNALS) return 0;
#ifdef HAVE_POSIX_SIGNALS
    if (flag <= 0) {
      if (flag == 0)
	/* revert to old action */
	if (initsig[sig])
	  new_action.sa_handler = old_handler[sig];
	else
	  goto skip;
      else
	/* new action: ignore */
	new_action.sa_handler = SIG_IGN;
      if (sigaction(sig, &new_action, &old_action))
	return 0;
      sigdelset(&new_action.sa_mask, sig);
    } else {
      /* new action: sig_handler */
      new_action.sa_handler = sig_handler;
      if (sigaction(sig, &new_action, &old_action))
	return 0;
      sigaddset(&new_action.sa_mask, sig);
    }
    /* update the signal mask for the other signals */
    for (i = 1; i <= NSIGNALS; i++)
      if (i != sig && sigismember(&new_action.sa_mask, i)) {
	new_action.sa_handler = sig_handler;
	sigaction(i, &new_action, NULL);
      }
    /* check previous action */
    if (!initsig[sig])
      prev_action = 0;
    else if (old_action.sa_handler == sig_handler)
      prev_action = 1;
    else
      prev_action = -1;
    /* remember the previous handler */
    if (flag == 0)
      initsig[sig] = 0;
    else if (!initsig[sig]) {
      old_handler[sig] = old_action.sa_handler;
      initsig[sig] = 1;
    }
#else
    /* we don't have POSIX signals, so we'll just have to put up with bad ol'
       signal() */
    if (flag == 0)
      if (initsig[sig])
	res = signal(sig, old_handler[sig]);
      else
	goto skip;
    else if (flag < 0)
      res = signal(sig, SIG_IGN);
    else
      res = signal(sig, sig_handler);
    if (res == SIG_ERR) return 0;
    if (!initsig[sig])
      prev_action = 0;
    else if (res == sig_handler)
      prev_action = 1;
    else
      prev_action = -1;
    if (flag == 0)
      initsig[sig] = 0;
    else if (!initsig[sig]) {
      old_handler[sig] = res;
      initsig[sig] = 1;
    }
#endif
  skip:
    return pushint(thr, prev_action);
  } else
    return 0;
}

static qmfail(THREAD* thr)
{
  thr->qmstat = XCEPT_FAIL;
  return (0);
}

static qmfail2(THREAD* thr)
{
  thr->qmstat = XCEPT_FAIL2;
  return (0);
}

static qmtime(THREAD* thr)
{
  return pushfloat(thr, systime());
}

static qmsleep(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == INTVALOP &&
      mpz_sgn(args[0]->data.z) >= 0) {
    double d = mpz_get_d(args[0]->data.z);
#ifdef USE_THREADS
    release_lock();
#endif
    syssleep(d);
#ifdef USE_THREADS
    acquire_lock();
#endif
    return (pushfun(thr, VOIDOP));
  } else if (args[0]->fno == FLOATVALOP &&
	     args[0]->data.f >= 0.0) {
#ifdef USE_THREADS
    release_lock();
#endif
    syssleep(args[0]->data.f);
#ifdef USE_THREADS
    acquire_lock();
#endif
    return (pushfun(thr, VOIDOP));
  } else
    return (0);
}

/* function table ("specials" are NULLed out): */

int             (*funtb[BUILTIN]) () = {

  NULL, NULL, NULL, NULL, qmdef, qmundef, NULL, NULL,
  NULL, NULL, NULL, NULL, NULL, NULL, NULL,
  NULL, NULL, NULL,
  NULL, NULL, NULL, NULL, NULL, NULL,
  qmconcat, qmadd, qmmin, qmmul, qmfdiv, qmdiv, qmmod, qmpow, qmidx, qmcomp,
  qmumin, qmhash, NULL, qmunquote, qmforce, qmmem,
  qmor, qmand, qmorelse, qmandthen, qmnot,
  qmle, qmgr, qmeq, qmleq, qmgeq, qmneq, qmid,
  qmrapp, qmseq,
  qmshl, qmshr, qmpred, qmsucc, qmenum, qmenum1, NULL, NULL, NULL, NULL,
  qmexp, qmln, qmsqrt, qmsin, qmcos, qmatan, qmatan2, qmrandom, qmseed,
  qmsub, qmsubstr, qmpos,
  qmint, qmfrac, qmtrunc, qmround, qmfloat, qmhashnum, qmord, qmchr,
  qmlist, qmtuple, qmstr, qmval, qmstrq, qmvalq,
  qmisspecial, qmisconst, qmisfun, qmisvar, qmisdef, qmflip,
  qmread, qmreadq, qmreadc, qmreads,
  qmwrite, qmwriteq, qmwritec, qmwrites,
  qmfread, qmfreadq, qmfreadc, qmfreads,
  qmfwrite, qmfwriteq, qmfwritec, qmfwrites,
  qmfopen, qmpopen, qmfclose, qmeof, qmfeof, qmflush, qmfflush,
  qmversion, qmsysinfo, qmwhich, qmhalt, qmquit, qmbreak,
  qmcatch, qmthrow, qmtrap, qmfail, qmfail2, NULL, qmlambda, NULL,
  qmtime, qmsleep, qmview,
  NULL, NULL, NULL, NULL, NULL,
  NULL, NULL, NULL, NULL, NULL, NULL, NULL,
  NULL, NULL, NULL, NULL,
  NULL, NULL, qmlambda_app, NULL, NULL


};

int             nargs[BUILTIN] = {

  0, 0, 0, 0, 2, 1, 0, 0,
  0, 0, 0, 0, 0, 0, 0,
  0, 0, 0,
  0, 0, 0, 0, 2, 0,
  2, 2, 2, 2, 2, 2, 2, 2, 2, 3,
  1, 1, 1, 1, 1, 1,
  2, 2, 2, 2, 1,
  2, 2, 2, 2, 2, 2, 2,
  2, 2,
  2, 2, 1, 1, 2, 1, 2, 1, 2, 1,
  1, 1, 1, 1, 1, 1, 2, 0, 1,
  3, 3, 2,
  1, 1, 1, 1, 1, 1, 1, 1,
  1, 1, 1, 1, 1, 1,
  1, 1, 1, 1, 1, 3,
  0, 0, 0, 0,
  1, 1, 1, 1,
  1, 1, 1, 1,
  2, 2, 2, 2,
  2, 2, 1, 0, 1, 0, 1,
  0, 0, 1, 0, 0, 0,
  2, 1, 2, 0, 0, 0, 2, 0,
  0, 1, 1,
  0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0,
  0, 0, 1, 0, 0

};
