/* Copyright 2018 The TensorFlow Authors. All Rights Reserved.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
==============================================================================*/

#include "tensorflow/compiler/xla/service/batch_dot_simplification.h"
#include "tensorflow/compiler/xla/service/hlo_matchers.h"
#include "tensorflow/compiler/xla/test.h"
#include "tensorflow/compiler/xla/tests/hlo_test_base.h"
#include "tensorflow/compiler/xla/tests/hlo_verified_test_base.h"

namespace xla {
namespace {

namespace op = xla::testing::opcode_matchers;

class BatchDotSimplificationTest : public HloVerifiedTestBase {};

TEST_F(BatchDotSimplificationTest,
       ElideSingleDegenerateBatchDotDim_VectorVector) {
  const string hlo_text = R"(
HloModule BatchDot

main {
  a = f32[1,3] parameter(0)
  b = f32[1,3] parameter(1)
  ROOT dot = f32[1] dot(a, b), lhs_batch_dims={0}, rhs_batch_dims={0}, lhs_contracting_dims={1}, rhs_contracting_dims={1}
}
)";

  ParseAndVerifyModule(hlo_text);
  BatchDotSimplification pass;
  ASSERT_TRUE(pass.Run(&module()).ValueOrDie());

  HloInstruction* root = module().entry_computation()->root_instruction();
  EXPECT_THAT(root,
              op::Reshape(op::Dot(
                  op::Reshape(op::Parameter(0)), op::Reshape(op::Parameter(1)),
                  /*lhs_contracting_dim=*/0, /*rhs_contracting_dim=*/0)));
}

TEST_F(BatchDotSimplificationTest,
       ElideSingleDegenerateBatchDotDim_MatrixVector) {
  const string hlo_text = R"(
HloModule BatchDot

main {
  a = f32[1,9,3] parameter(0)
  b = f32[1,3] parameter(1)
  ROOT dot = f32[1,9] dot(a, b), lhs_batch_dims={0}, rhs_batch_dims={0}, lhs_contracting_dims={2}, rhs_contracting_dims={1}
}
)";

  ParseAndVerifyModule(hlo_text);
  BatchDotSimplification pass;
  ASSERT_TRUE(pass.Run(&module()).ValueOrDie());

  HloInstruction* root = module().entry_computation()->root_instruction();
  EXPECT_THAT(root,
              op::Reshape(op::Dot(
                  op::Reshape(op::Parameter(0)), op::Reshape(op::Parameter(1)),
                  /*lhs_contracting_dim=*/1, /*rhs_contracting_dim=*/0)));
}

TEST_F(BatchDotSimplificationTest,
       ElideSingleDegenerateBatchDotDim_MatrixMatrix) {
  const string hlo_text = R"(
HloModule BatchDot

main {
  a = f32[1,9,3] parameter(0)
  b = f32[1,3,7] parameter(1)
  ROOT dot = f32[1,9,7] dot(a, b), lhs_batch_dims={0}, rhs_batch_dims={0}, lhs_contracting_dims={2}, rhs_contracting_dims={1}
}
)";

  ParseAndVerifyModule(hlo_text);
  BatchDotSimplification pass;
  ASSERT_TRUE(pass.Run(&module()).ValueOrDie());

  HloInstruction* root = module().entry_computation()->root_instruction();
  EXPECT_THAT(root,
              op::Reshape(op::Dot(
                  op::Reshape(op::Parameter(0)), op::Reshape(op::Parameter(1)),
                  /*lhs_contracting_dim=*/1, /*rhs_contracting_dim=*/0)));
}

TEST_F(BatchDotSimplificationTest,
       ElideMultipleDegenerateBatchDotDims_VectorVector) {
  const string hlo_text = R"(
HloModule BatchDot

main {
  a = f32[9,1,7,1,3] parameter(0)
  b = f32[9,1,7,1,3] parameter(1)
  ROOT dot = f32[9,1,7,1] dot(a, b), lhs_batch_dims={0,1,2,3}, rhs_batch_dims={0,1,2,3}, lhs_contracting_dims={4}, rhs_contracting_dims={4}
}
)";

  ParseAndVerifyModule(hlo_text);
  BatchDotSimplification pass;
  ASSERT_TRUE(pass.Run(&module()).ValueOrDie());

  HloInstruction* root = module().entry_computation()->root_instruction();
  EXPECT_THAT(root,
              op::Reshape(op::Dot(
                  op::Reshape(op::Parameter(0)), op::Reshape(op::Parameter(1)),
                  /*lhs_contracting_dim=*/2, /*rhs_contracting_dim=*/2)));
}

TEST_F(BatchDotSimplificationTest,
       ElideMultipleDegenerateBatchDotDims_VectorMatrix) {
  const string hlo_text = R"(
HloModule BatchDot

main {
  a = f32[9,1,7,1,3] parameter(0)
  b = f32[9,1,7,1,20,3] parameter(1)
  ROOT dot = f32[9,1,7,1,20] dot(a, b), lhs_batch_dims={0,1,2,3}, rhs_batch_dims={0,1,2,3}, lhs_contracting_dims={4}, rhs_contracting_dims={5}
}
)";

  ParseAndVerifyModule(hlo_text);
  BatchDotSimplification pass;
  ASSERT_TRUE(pass.Run(&module()).ValueOrDie());

  HloInstruction* root = module().entry_computation()->root_instruction();
  EXPECT_THAT(root,
              op::Reshape(op::Dot(
                  op::Reshape(op::Parameter(0)), op::Reshape(op::Parameter(1)),
                  /*lhs_contracting_dim=*/2, /*rhs_contracting_dim=*/3)));
}

TEST_F(BatchDotSimplificationTest,
       ElideMultipleDegenerateBatchDotDims_MatrixMatrix) {
  const string hlo_text = R"(
HloModule BatchDot

main {
  a = f32[9,1,7,1,19,3] parameter(0)
  b = f32[9,1,7,1,3,20] parameter(1)
  ROOT dot = f32[9,1,7,1,19,20] dot(a, b), lhs_batch_dims={0,1,2,3}, rhs_batch_dims={0,1,2,3}, lhs_contracting_dims={5}, rhs_contracting_dims={4}
}
)";

  ParseAndVerifyModule(hlo_text);
  BatchDotSimplification pass;
  ASSERT_TRUE(pass.Run(&module()).ValueOrDie());

  HloInstruction* root = module().entry_computation()->root_instruction();
  EXPECT_THAT(root,
              op::Reshape(op::Dot(
                  op::Reshape(op::Parameter(0)), op::Reshape(op::Parameter(1)),
                  /*lhs_contracting_dim=*/3, /*rhs_contracting_dim=*/2)));
}

}  // namespace
}  // namespace xla
