"use strict";
// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConfigWithDefaults = exports.ConfigSection = void 0;
const coreutils_1 = require("@jupyterlab/coreutils");
const __1 = require("..");
/**
 * The url for the config service.
 */
const SERVICE_CONFIG_URL = 'api/config';
/**
 * The namespace for ConfigSection statics.
 */
var ConfigSection;
(function (ConfigSection) {
    /**
     * Create a config section.
     *
     * @returns A Promise that is fulfilled with the config section is loaded.
     */
    function create(options) {
        const section = new DefaultConfigSection(options);
        return section.load().then(() => {
            return section;
        });
    }
    ConfigSection.create = create;
})(ConfigSection = exports.ConfigSection || (exports.ConfigSection = {}));
/**
 * Implementation of the Configurable data section.
 */
class DefaultConfigSection {
    /**
     * Construct a new config section.
     */
    constructor(options) {
        var _a;
        this._url = 'unknown';
        const settings = (this.serverSettings =
            (_a = options.serverSettings) !== null && _a !== void 0 ? _a : __1.ServerConnection.makeSettings());
        this._url = coreutils_1.URLExt.join(settings.baseUrl, SERVICE_CONFIG_URL, encodeURIComponent(options.name));
    }
    /**
     * Get the data for this section.
     */
    get data() {
        return this._data;
    }
    /**
     * Load the initial data for this section.
     *
     * #### Notes
     * Uses the [Jupyter Notebook API](https://petstore.swagger.io/?url=https://raw.githubusercontent.com/jupyter-server/jupyter_server/main/jupyter_server/services/api/api.yaml#!/config).
     *
     * The promise is fulfilled on a valid response and rejected otherwise.
     */
    async load() {
        const response = await __1.ServerConnection.makeRequest(this._url, {}, this.serverSettings);
        if (response.status !== 200) {
            const err = await __1.ServerConnection.ResponseError.create(response);
            throw err;
        }
        this._data = await response.json();
    }
    /**
     * Modify the stored config values.
     *
     * #### Notes
     * Uses the [Jupyter Notebook API](https://petstore.swagger.io/?url=https://raw.githubusercontent.com/jupyter-server/jupyter_server/main/jupyter_server/services/api/api.yaml#!/config).
     *
     * The promise is fulfilled on a valid response and rejected otherwise.
     *
     * Updates the local data immediately, sends the change to the server,
     * and updates the local data with the response, and fulfils the promise
     * with that data.
     */
    async update(newdata) {
        this._data = { ...this._data, ...newdata };
        const init = {
            method: 'PATCH',
            body: JSON.stringify(newdata)
        };
        const response = await __1.ServerConnection.makeRequest(this._url, init, this.serverSettings);
        if (response.status !== 200) {
            const err = await __1.ServerConnection.ResponseError.create(response);
            throw err;
        }
        this._data = await response.json();
        return this._data;
    }
}
/**
 * Configurable object with defaults.
 */
class ConfigWithDefaults {
    /**
     * Create a new config with defaults.
     */
    constructor(options) {
        var _a, _b;
        this._className = '';
        this._section = options.section;
        this._defaults = (_a = options.defaults) !== null && _a !== void 0 ? _a : {};
        this._className = (_b = options.className) !== null && _b !== void 0 ? _b : '';
    }
    /**
     * Get data from the config section or fall back to defaults.
     */
    get(key) {
        const data = this._classData();
        return key in data ? data[key] : this._defaults[key];
    }
    /**
     * Set a config value.
     *
     * #### Notes
     * Uses the [Jupyter Notebook API](https://petstore.swagger.io/?url=https://raw.githubusercontent.com/jupyter-server/jupyter_server/main/jupyter_server/services/api/api.yaml#!/config).
     *
     * The promise is fulfilled on a valid response and rejected otherwise.
     *
     * Sends the update to the server, and changes our local copy of the data
     * immediately.
     */
    set(key, value) {
        const d = {};
        d[key] = value;
        if (this._className) {
            const d2 = {};
            d2[this._className] = d;
            return this._section.update(d2);
        }
        else {
            return this._section.update(d);
        }
    }
    /**
     * Get data from the Section with our classname, if available.
     *
     * #### Notes
     * If we have no classname, get all of the data in the Section
     */
    _classData() {
        const data = this._section.data;
        if (this._className && this._className in data) {
            return data[this._className];
        }
        return data;
    }
}
exports.ConfigWithDefaults = ConfigWithDefaults;
//# sourceMappingURL=index.js.map