// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
import { YFile } from '@jupyter/ydoc';
import { ObservableMap } from '@jupyterlab/observables';
import { Signal } from '@lumino/signaling';
/**
 * A namespace for code editors.
 *
 * #### Notes
 * - A code editor is a set of common assumptions which hold for all concrete editors.
 * - Changes in implementations of the code editor should only be caused by changes in concrete editors.
 * - Common JLab services which are based on the code editor should belong to `IEditorServices`.
 */
export var CodeEditor;
(function (CodeEditor) {
    /**
     * The default implementation of the editor model.
     */
    class Model {
        /**
         * Construct a new Model.
         */
        constructor(options = {}) {
            var _a, _b;
            /**
             * Whether the model should disposed the shared model on disposal or not.
             */
            this.standaloneModel = false;
            this._isDisposed = false;
            this._selections = new ObservableMap();
            this._mimeType = 'text/plain';
            this._mimeTypeChanged = new Signal(this);
            // Track if we need to dispose the model or not.
            this.standaloneModel = typeof options.sharedModel === 'undefined';
            this.sharedModel = (_a = options.sharedModel) !== null && _a !== void 0 ? _a : new YFile();
            this._mimeType = (_b = options.mimeType) !== null && _b !== void 0 ? _b : 'text/plain';
        }
        /**
         * A signal emitted when a mimetype changes.
         */
        get mimeTypeChanged() {
            return this._mimeTypeChanged;
        }
        /**
         * Get the selections for the model.
         */
        get selections() {
            return this._selections;
        }
        /**
         * A mime type of the model.
         */
        get mimeType() {
            return this._mimeType;
        }
        set mimeType(newValue) {
            const oldValue = this.mimeType;
            if (oldValue === newValue) {
                return;
            }
            this._mimeType = newValue;
            this._mimeTypeChanged.emit({
                name: 'mimeType',
                oldValue: oldValue,
                newValue: newValue
            });
        }
        /**
         * Whether the model is disposed.
         */
        get isDisposed() {
            return this._isDisposed;
        }
        /**
         * Dispose of the resources used by the model.
         */
        dispose() {
            if (this._isDisposed) {
                return;
            }
            this._isDisposed = true;
            this._selections.dispose();
            if (this.standaloneModel) {
                this.sharedModel.dispose();
            }
            Signal.clearData(this);
        }
    }
    CodeEditor.Model = Model;
})(CodeEditor || (CodeEditor = {}));
//# sourceMappingURL=editor.js.map