import 'dart:io';

import 'package:fvm/src/models/cache_flutter_version_model.dart';
import 'package:fvm/src/models/config_model.dart';
import 'package:fvm/src/models/flutter_version_model.dart';
import 'package:fvm/src/services/flutter_service.dart';
import 'package:fvm/src/utils/context.dart';
import 'package:fvm/src/utils/exceptions.dart';
import 'package:path/path.dart' as p;
import 'package:test/test.dart';

import '../testing_utils.dart';

// Custom exception handler to test error handling
void throwGitError(String message, List<String> args) {
  final e = ProcessException('git', args, message, 128);
  throw e;
}

/// Creates an isolated test context with separate git cache to avoid conflicts
FvmContext createIsolatedTestContext() {
  final tempDir = Directory.systemTemp.createTempSync(
    'fvm_flutter_service_test_',
  );

  return FvmContext.create(
    isTest: true,
    configOverrides: AppConfig(
      cachePath: p.join(tempDir.path, 'cache'),
      gitCachePath: p.join(tempDir.path, 'git_cache'),
      useGitCache: true,
    ),
  );
}

void main() {
  group('FlutterService', () {
    group('install method', () {
      test('handles non-existent fork', () async {
        final context = TestFactory.context();
        final service = FlutterService(context);
        final version = FlutterVersion.parse('nonexistent-fork/stable');

        expect(
          () => service.install(version),
          throwsA(
            isA<AppException>().having(
              (e) => e.toString(),
              'message',
              contains('not found in configuration'),
            ),
          ),
        );
      });
    });

    test('returns expected error for reset to non-existent version', () {
      // This test verifies the error handling when `resetHard` throws an exception
      // for a non-existent version in a repository

      // We verify that the correct error message patterns in our error handler
      // will trigger the correct AppException with the expected error message

      // Test for unknown revision error message
      expect(
        () => throwGitError(
          'fatal: ambiguous argument \'non-existent-tag\': unknown revision',
          ['reset', '--hard', 'non-existent-tag'],
        ),
        throwsA(
          isA<ProcessException>().having(
            (e) => e.message,
            'message',
            contains('unknown revision'),
          ),
        ),
      );

      // Test for ambiguous argument error message
      expect(
        () => throwGitError(
          'fatal: ambiguous argument \'non-existent-branch\'',
          ['reset', '--hard', 'non-existent-branch'],
        ),
        throwsA(
          isA<ProcessException>().having(
            (e) => e.message,
            'message',
            contains('ambiguous argument'),
          ),
        ),
      );

      // Test for not found error message
      expect(
        () => throwGitError(
          'error: pathspec \'non-existent-ref\' did not match any file(s) known to git',
          ['reset', '--hard', 'non-existent-ref'],
        ),
        throwsA(
          isA<ProcessException>().having(
            (e) => e.message,
            'message',
            contains('did not match any file'),
          ),
        ),
      );
    });

    test('returns expected error for clone failures', () {
      // This test verifies the error handling when the clone operation fails

      // Test for repository not found error
      expect(
        () => throwGitError('fatal: remote: Repository not found.', [
          'clone',
          'https://example.com/fork.git',
        ]),
        throwsA(
          isA<ProcessException>().having(
            (e) => e.message,
            'message',
            contains('Repository not found'),
          ),
        ),
      );

      // Test for remote branch not found error
      expect(
        () => throwGitError(
          'fatal: Remote branch branch-name not found in upstream origin',
          ['clone', '-b', 'branch-name', 'https://example.com/repo.git'],
        ),
        throwsA(
          isA<ProcessException>().having(
            (e) => e.message,
            'message',
            (String msg) =>
                msg.contains('Remote branch') &&
                msg.contains('not found in upstream'),
          ),
        ),
      );
    });

    group('isReferenceError method', () {
      test('detects reference repository errors', () {
        final context = createIsolatedTestContext();
        final service = FlutterService(context);

        // Test various reference error patterns
        expect(
          service.isReferenceError('fatal: reference repository not found'),
          isTrue,
        );
        expect(
          service.isReferenceError('error: unable to read reference'),
          isTrue,
        );
        expect(
          service.isReferenceError('fatal: bad object in reference'),
          isTrue,
        );
        expect(
          service.isReferenceError('error: corrupt reference repository'),
          isTrue,
        );
        expect(service.isReferenceError('fatal: reference not found'), isTrue);
      });

      test('does not detect non-reference errors', () {
        final context = createIsolatedTestContext();
        final service = FlutterService(context);

        // Test non-reference error patterns
        expect(
          service.isReferenceError('fatal: repository not found'),
          isFalse,
        );
        expect(
          service.isReferenceError('fatal: remote branch not found'),
          isFalse,
        );
        expect(service.isReferenceError('error: unknown revision'), isFalse);
        expect(service.isReferenceError('fatal: ambiguous argument'), isFalse);
      });
    });

    group('setup method', () {
      test('calls flutter --version command', () async {
        final context = TestFactory.context();
        final service = FlutterService(context);

        final flutterVersion = FlutterVersion.parse('stable');
        final mockCacheVersion = CacheFlutterVersion.fromVersion(
          flutterVersion,
          directory: p.join(context.versionsCachePath, 'stable'),
        );

        // Since this simply passes through to run(), we can just verify
        // that the method returns a Future that completes normally
        expect(service.setup(mockCacheVersion), isA<Future<ProcessResult>>());
      });
    });

    group('runFlutter method', () {
      test('executes flutter command with the specified args', () async {
        final context = TestFactory.context();
        final service = FlutterService(context);

        final flutterVersion = FlutterVersion.parse('stable');
        final mockCacheVersion = CacheFlutterVersion.fromVersion(
          flutterVersion,
          directory: p.join(context.versionsCachePath, 'stable'),
        );

        expect(
          service.runFlutter(['--help'], mockCacheVersion),
          isA<Future<ProcessResult>>(),
        );
      });
    });

    group('pubGet method', () {
      test('executes flutter pub get command', () async {
        final context = TestFactory.context();
        final service = FlutterService(context);

        final flutterVersion = FlutterVersion.parse('stable');
        final mockCacheVersion = CacheFlutterVersion.fromVersion(
          flutterVersion,
          directory: p.join(context.versionsCachePath, 'stable'),
        );

        expect(service.pubGet(mockCacheVersion), isA<Future<ProcessResult>>());
      });

      test('adds --offline flag in offline mode', () async {
        final context = TestFactory.context();
        final service = FlutterService(context);

        final flutterVersion = FlutterVersion.parse('stable');
        final mockCacheVersion = CacheFlutterVersion.fromVersion(
          flutterVersion,
          directory: p.join(context.versionsCachePath, 'stable'),
        );

        expect(
          service.pubGet(mockCacheVersion, offline: true),
          isA<Future<ProcessResult>>(),
        );
      });
    });

    group('VersionRunner', () {
      test('correctly sets up environment variables', () {
        final context = TestFactory.context();

        final flutterVersion = FlutterVersion.parse('stable');
        final mockCacheVersion = CacheFlutterVersion.fromVersion(
          flutterVersion,
          directory: p.join(context.versionsCachePath, 'stable'),
        );

        final versionRunner = VersionRunner(
          context: context,
          version: mockCacheVersion,
        );

        // Testing implementation details:
        // This test confirms the VersionRunner can be constructed correctly
        // For more comprehensive testing, we'd need to verify the environment variables
        // are correctly set, which would require mocking Platform.environment
        expect(versionRunner, isA<VersionRunner>());
      });
    });
  });
}
