/* PSPP - a program for statistical analysis.
   Copyright (C) 2009, 2010, 2011 Free Sonftware Foundation, Inc.

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>. */

#ifndef OUTPUT_TEXT_ITEM_H
#define OUTPUT_TEXT_ITEM_H 1

/* Text items.

   A text item is a subclass of an output item (see
   output/output-item.h).

   A text item is just a text string. */

#include <stdbool.h>
#include "libpspp/compiler.h"
#include "output/output-item.h"

enum text_item_type
  {
    TEXT_ITEM_PAGE_TITLE,       /* TITLE and SUBTITLE commands. */
    TEXT_ITEM_TITLE,            /* Title. */
    TEXT_ITEM_PARAGRAPH,        /* Normal paragraph of text. */

    /* Log items. */
    TEXT_ITEM_SYNTAX,           /* A single line of PSPP syntax. */
    TEXT_ITEM_LOG,              /* Other logging. */

    /* Spacing.  Some output drivers that are not based on lines and pages
       (e.g. CSV, HTML) may ignore these. */
    TEXT_ITEM_BLANK_LINE,       /* Blank line. */
    TEXT_ITEM_EJECT_PAGE        /* Eject page. */
  };

/* A text item. */
struct text_item
  {
    struct output_item output_item;
    char *text;                 /* The content. */
    enum text_item_type type;   /* Type. */

    bool bold, italic, underline, markup;
    char *typeface;
    int size;
  };

struct text_item *text_item_create (enum text_item_type, const char *text);
struct text_item *text_item_create_nocopy (enum text_item_type, char *text);
struct text_item *text_item_create_format (enum text_item_type,
                                           const char *format, ...)
  PRINTF_FORMAT (2, 3);

enum text_item_type text_item_get_type (const struct text_item *);
const char *text_item_get_text (const struct text_item *);

struct table_item *text_item_to_table_item (struct text_item *);

/* This boilerplate for text_item, a subclass of output_item, was
   autogenerated by mk-class-boilerplate. */

#include <assert.h>
#include "libpspp/cast.h"

extern const struct output_item_class text_item_class;

/* Returns true if SUPER is a text_item, otherwise false. */
static inline bool
is_text_item (const struct output_item *super)
{
  return super->class == &text_item_class;
}

/* Returns SUPER converted to text_item.  SUPER must be a text_item, as
   reported by is_text_item. */
static inline struct text_item *
to_text_item (const struct output_item *super)
{
  assert (is_text_item (super));
  return UP_CAST (super, struct text_item, output_item);
}

/* Returns INSTANCE converted to output_item. */
static inline struct output_item *
text_item_super (const struct text_item *instance)
{
  return CONST_CAST (struct output_item *, &instance->output_item);
}

/* Increments INSTANCE's reference count and returns INSTANCE. */
static inline struct text_item *
text_item_ref (const struct text_item *instance)
{
  return to_text_item (output_item_ref (&instance->output_item));
}

/* Decrements INSTANCE's reference count, then destroys INSTANCE if
   the reference count is now zero. */
static inline void
text_item_unref (struct text_item *instance)
{
  output_item_unref (&instance->output_item);
}

/* Returns true if INSTANCE's reference count is greater than 1,
   false otherwise. */
static inline bool
text_item_is_shared (const struct text_item *instance)
{
  return output_item_is_shared (&instance->output_item);
}

void text_item_submit (struct text_item *);

#endif /* output/text-item.h */
