#!/bin/sh
###################################################
#  This cleans up everything in the source code
#  directories, updates all downloaded source code
#  trees, and recompiles everything.
#
#  NOTE: the OS is not upgraded since the system
#        should be re-booted at least if the kernel
#        has been upgraded.
###################################################

cd $HOME

# number of CPUs to use in parallel make
export NPROCS=4


###################################################
#
# clean up everything
#
###################################################
#
make -C $HOME/pihpsdr    clean
make -C $HOME/wdsp       clean
make -C $HOME/hamlib     clean
make -C $HOME/fldigi     clean

rm -rf $HOME/wsjtx/build
rm -rf $HOME/wsjtx/bin
rm -rf $HOME/wsjtx/share

rm -rf $HOME/SoapySDR/build
rm -rf $HOME/libiio/build
rm -rf $HOME/SoapyPlutoSDR/build

rm -rf $HOME/freedv-gui/codec2/build_linux
rm -rf $HOME/freedv-gui/LPCNet/build_linux
rm -rf $HOME/freedv-gui/build_linux

###################################################
#
# Update, compile and install SoapySDR core
#
###################################################

cd $HOME/SoapySDR
git pull
mkdir build
cd build
cmake ..
make -j $NPROCS
sudo make install
sudo ldconfig

###################################################
#
# Update, compile and install libiio
#
###################################################

cd $HOME/libiio
git pull
mkdir build
cd build
cmake ..
make -j $NPROCS
sudo make install
sudo ldconfig

###################################################
#
# Update, compile and install the
# SoapySDR Pluto module
#
###################################################

cd $HOME/SoapyPlutoSDR
git pull
mkdir build
cd build
cmake ..
make -j $NPROCS
sudo make install
sudo ldconfig

###################################################
#
# Update, compile and install WDSP
#
###################################################

cd $HOME/wdsp
git pull
make clean
make -j $NPROCS
sudo make install
sudo ldconfig

###################################################
#
# Update and compile piHPSDR
#
###################################################

cd $HOME/pihpsdr
git pull
make clean
make -j $NPROCS

###################################################
#
# Compile and install hamlib
# (no update / git pull since we are at a fixed version)
# (no git pull since we are at a fixed version)
#
###################################################

cd $HOME/hamlib
autoreconf -i
./configure --without-libusb
make -j $NPROCS
sudo make install
sudo ldconfig

###################################################
#
# Update and compile fldigi
#
# Note: I always have problems with the
#       national language support
#       therefore I switch it off
#
#       "confdialog" needs MUCH memory to compile
#       so use only one CPU otherwise compilation
#       may fail due to memory shortage!
#       (this is the case on 64-bit systems with gcc9)
#
###################################################

cd $HOME/fldigi
git pull
make clean
autoreconf -i
./configure --disable-flarq --disable-nls
make

###################################################
#
# Compile wsjtx
# (no update / git pull since we are at a fixed version)
#
# Skip generation of documentation and man pages 
# since we need TONS of further software
# (asciidoc, asciidoctor, texlive ...) 
# to do so.
#
###################################################

cd $HOME/wsjtx
export CC=gcc
export CXX=g++
export FC=gfortran
mkdir bin
TARGET=$PWD
CMFLG="-DWSJT_GENERATE_DOCS=OFF -DWSJT_SKIP_MANPAGES=ON"
mkdir build
cd build
cmake -DCMAKE_INSTALL_PREFIX=$TARGET $CMFLG ..
cd ..
cmake --build build --target install -j $NPROCS

###################################################
#
# Update and compile FreeDV
#
###################################################

cd $HOME/freedv-gui
git pull
./build_linux.sh
