<?php

/**
 * Class for GUI-based configuration of a php.ini file
 *
 * PHP Version 5
 *
 * This class is based on the PHP-Gtk2 extension and provides a
 * user-friendly interface for editing php.ini files. All the possible
 * configuration options have been defined in a separate XML file .
 * This allows for easy addition of directives as PHP is
 * developed further.
 *
 * The class can parse any existing ini file, or create new ones. Parsing
 * of existing files is currently a little inefficient, suggestions for
 * improvement are welcome. New files and files previously generated by
 * Gtk2_PHPConfig are parsed at a good speed though.
 *
 * The interface displays all available configuration sections on the left
 * pane, and their corresponding options in the top right pane. The bottom
 * right pane displays the description of the option selected and the
 * facility to change the value of that option.
 *
 * @todo    Implement Christian's suggestion of asking for confirmation when a
 *          file is overwritten while doing a "save as"
 * @todo    Add facility to add directives, apart from additions to XML
 * @todo    Clicking Cancel on the File-Open dialog generates a notice
 *
 * @category Gtk2
 * @package  PHPConfig
 * @author   Anant Narayanan <anant@php.net>
 * @license  http://www.gnu.org/copyleft/lesser.html  LGPL License 2.1
 * @link     http://pear.php.net/package/Gtk2_PHPConfig
 *
 */

/**
 * Gtk2_PHPConfig
 *
 * @category Gtk2
 * @package  PHPConfig
 * @author   Anant Narayanan <anant@php.net>
 * @license  http://www.gnu.org/copyleft/lesser.html  LGPL License 2.1
 * @link     http://pear.php.net/package/Gtk2_PHPConfig
 */
class Gtk2_PHPConfig extends GtkWindow
{
    // {{{ properties

    /**
     * Tree that displays the available configuration sections.
     *
     * @var GtkTreeView
     * @access protected
     */
    protected $sectionTree;

    /**
     * Tree that displays the current section's corresponding options.
     *
     * @var GtkTreeView
     * @access protected
     */
    protected $optionTree;

    /**
     * Description of the currently selected option.
     *
     * @var GtkTextBuffer
     * @access protected
     */
    protected $optionDesc;

    /**
     * Comments associated with the currently selected option.
     *
     * @var GtkTextBuffer
     * @access protected
     */
    protected $optionComments;

    /**
     * The entry box that will accept the value of the current option from
     * the user.
     *
     * @var GtkEntry
     * @access protected
     */
    protected $optionEntry;

    /**
     * The file name of the php.ini file currently open. Will be set to
     * null if a new file is to be generated.
     *
     * @var string
     * @access protected
     */
    protected $fileName;

    /**
     * The XML definition file holding descriptions of all possible php.ini
     * directives.
     *
     * @var DOMDocument
     * @access protected
     */
    protected $xmlDefs;

    /**
     * The XPath object to the XML definition file.
     *
     * @var DOMXPath
     * @access protected
     */
    protected $xpath;

    /**
     * A flag that determined whether the current file is saved or not.
     *
     * @var bool
     * @access protected
     */
    protected $saved;

    /**
     * A buffer used to temporarily store the contents of the currently
     * open file when the save as option is used
     *
     * @var Array
     * @access protected
     */
    protected $saveasBuffer;

    /**
     * The "buffer" array that stores the current values of all the
     * options.
     *
     * @var Array
     * @access protected
     */
    protected $optionIndex;

    /**
     * The progress window that displays the percentage by which a file
     * is parsed.
     *
     * @var GtkWindow
     * @access protected
     */
    protected $progress;

    /**
     * The extension modules loaded/unloaded in the current file parsed.
     *
     * @var Array
     * @access protected
     */
    protected $extensions;

    //}}}
    //{{{ constants

    /**
     * Title of the Window.
     */
    const TITLE = "The PHP Configuration GUI";

    /**
     * Array Index - Current Value of a directive
     */
    const DIR_CURRENT = "Current";

    /**
     * Array Index - Default Value of a directive
     */
    const DIR_DEFAULT = "Default";

    /**
     * Array Index - Line number of directive
     */
    const DIR_NUMBER = "LineNumber";
    /**
     * Array Index - Comments in a directive
     */
    const COMMENT_NORMAL = "CommentsN";

    /**
     * Array Index - InLine comments of directive
     */
    const COMMENT_INLINE = "CommentsI";

    /**
     * Extension Option - Extension is enabled
     */
    const EXT_ENABLED = "Enabled";

    /**
     * Extension Option - Extension is disabled
     */
    const EXT_DISABLED = "Disabled";

    //}}}
    //{{{ __construct()

    /**
     * Sets the classes' properties and calls the creation of the GUI
     * interface.
     *
     * @param string $fileName Filename of the ini file to be edited.
     *                         Will be null if a new file is to be created.
     *
     * @return void
     */
    function __construct($fileName)
    {
        parent::__construct();
        include_once "PEAR/Config.php";
        $this->fileName = $fileName;
        $this->saved    = false;
        $this->xmlDefs  = new DomDocument();
        //where data is stored
        $dataDir = PEAR_Config::singleton()->get("data_dir") .
            DIRECTORY_SEPARATOR . "Gtk2_PHPConfig";
        $this->xmlDefs->load($dataDir."/phpinidefs.xml");
        $this->xpath = new DOMXPath($this->xmlDefs);
        //optionIndex
        $this->optionIndex = $this->getValues();
        $this->buildGUI();
    }

    //}}}
    //{{{ buildGUI()

    /**
     * Creates the user interface to facilitate editing of the ini files.
     * Also calls other functions that parse the XML definition file and
     * then displays the configuration options.
     *
     * @return void
     */
    function buildGUI()
    {
        /* Creating the layout boxes */
        $MainVBox = new GtkVBox();
        $TopHPane = new GtkHPaned();
        $TopHPane->set_position(200);
        $rightVPane = new GtkVPaned();

        /* Displaying the configuration sections */
        $sectionList       = $this->buildSectionList();
        $this->sectionTree = $sectionList;
        // add sectionList to a scrolled window
        $leftScroll = new GtkScrolledWindow();
        $leftScroll->set_policy(Gtk::POLICY_NEVER, Gtk::POLICY_AUTOMATIC);
        $leftScroll->add($sectionList);

        /* Creating the tree that will display configuration options */
        $model = new GtkListStore(Gobject::TYPE_STRING,
            Gobject::TYPE_STRING,
            Gobject::TYPE_STRING);
        $tree  = new GtkTreeView($model);
        // set the optionTree
        $this->optionTree = $tree;

        $rightScroll = new GtkScrolledWindow();
        $rightScroll->set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);
        $rightScroll->add($tree);
        $rightScroll->set_size_request(-1, 400);

        $bottomHBox = new GtkHBox();
        $bottomHBox->pack_start($this->buildDescBox(), true, true);
        $bottomHBox->pack_start($this->buildValueBox(), false, false);

        $rightVPane->pack1($rightScroll);
        $rightVPane->pack2($bottomHBox);

        $TopHPane->add1($leftScroll);
        $TopHPane->add2($rightVPane);
        $MainVBox->pack_start($this->buildMenu(), false, false);
        $MainVBox->pack_start($TopHPane, true, true);

        /* Initialise the main window */
        $this->add($MainVBox);
        $this->setTitle();
        $this->show_all();
        $this->set_size_request(800, 600);
        $this->set_position(Gtk::WIN_POS_CENTER);
        $this->connect_simple("delete-event", array($this, "quit"));
    }

    //}}}
    //{{{ buildMenu()

    /**
     * Creates and returns the menu for the GUI
     *
     * @return GtkMenuBar   The Menu bar of the GUI
     */
    function buildMenu()
    {
        $theMenu   = new GtkMenuBar();
        $menuAccel = new GtkAccelGroup();
        $this->add_accel_group($menuAccel);

        /* Creating the menu */
        $fileMenu = new GtkMenu();

        $fileNew = new GtkImageMenuItem(Gtk::STOCK_NEW, $menuAccel);
        $fileNew->connect_simple("activate", array($this, "newFile"));
        $fileMenu->append($fileNew);

        $fileOpen = new GtkImageMenuItem(Gtk::STOCK_OPEN, $menuAccel);
        $fileOpen->connect_simple("activate", array($this, "openFile"));
        $fileMenu->append($fileOpen);

        $fileSep = new GtkSeparatorMenuItem();
        $fileMenu->append($fileSep);

        $fileSave = new GtkImageMenuItem(Gtk::STOCK_SAVE, $menuAccel);
        $fileSave->connect_simple("activate", array($this, "saveFile"));
        $fileMenu->append($fileSave);

        $fileSaveAs = new GtkImageMenuItem(Gtk::STOCK_SAVE_AS, $menuAccel);
        $fileSaveAs->connect_simple("activate", array($this, "onSaveFileAs"));
        $fileMenu->append($fileSaveAs);

        $fileQuit = new GtkImageMenuItem(Gtk::STOCK_QUIT, $menuAccel);
        $fileQuit->connect_simple("activate", array($this, "quit"));
        $fileMenu->append($fileQuit);

        $fileButton = new GtkMenuItem("_File");
        $fileButton->set_submenu($fileMenu);
        $theMenu->append($fileButton);

        return $theMenu;
    }

    //}}}
    //{{{ buildDescBox()

    /**
     * Creates and the description and comment boxes.
     *
     * @return GtkVBox  The descriptiona and comment boxes.
     */
    function buildDescBox()
    {
        /* Layout for the description and value setting boxes */
        $descVBox   = new GtkVBox();
        $descTitle  = new GtkFrame("Description");
        $descWindow = new GtkScrolledWindow();
        $descWindow->set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);
        $description = new GtkTextBuffer();
        $descShow    = new GtkTextView($description);
        $descShow->set_wrap_mode(Gtk::WRAP_WORD);
        $descShow->set_size_request(-1, 150);
        $this->optionDesc = $description;
        $descWindow->add($descShow);
        $commentTitle  = new GtkFrame("Comments");
        $commentWindow = new GtkScrolledWindow();
        $commentWindow->set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);
        $comments    = new GtkTextBuffer();
        $commentShow = new GtkTextView($comments);
        $commentShow->set_wrap_mode(Gtk::WRAP_WORD);
        $this->optionComments = $comments;
        $commentWindow->add($commentShow);
        $descVBox->pack_start($descTitle, false, false);
        $descTitle->add($descWindow);
        $descVBox->pack_start($commentTitle, false, false);
        $commentTitle->add($commentWindow);

        return $descVBox;
    }

    //}}}
    //{{{ buildValueBox()

    /**
     * Creates and returns the value setting box
     *
     * @return  GtkVBox     The Value setting box.
     */
    function buildValueBox()
    {
        $valueVBox  = new GtkVBox();
        $valueLabel = new GtkLabel("Set a value for the option");

        $entryBox   = new GtkHBox();
        $valueEntry = new GtkEntry();
        $valueEntry->connect_simple("activate", array($this, "onValueSubmit"));
        $this->optionEntry = $valueEntry;
        //unset button
        $valueUnset = new GtkButton(null);
        $unsetIcon  = GtkImage::new_from_stock(Gtk::STOCK_CANCEL,
            Gtk::ICON_SIZE_BUTTON);
        $valueUnset->set_image($unsetIcon);
        $entryBox->pack_start($valueEntry, true, true);
        $entryBox->pack_start($valueUnset, false, false);

        $valueSubmit  = new GtkButton("Set Value");
        $commentLabel = new GtkLabel("Comments are also added\n" .
            "Use the red cross to unset.");
        $setIcon      = GtkImage::new_from_stock(Gtk::STOCK_OK,
            Gtk::ICON_SIZE_BUTTON);
        $valueSubmit->set_image($setIcon);

        $valueSubmit->connect_simple("clicked", array($this, "onValueSubmit"));
        $valueUnset->connect_simple("clicked", array($this, "onValueUnset"));

        /* Adding all the widgets and finalising layout */
        $valueVBox->pack_start($valueLabel, false, false);
        $valueVBox->pack_start($entryBox, false, false);
        $valueVBox->pack_start($valueSubmit, false, false);
        $valueVBox->pack_start($commentLabel, false, false);

        return $valueVBox;
    }

    //}}}
    //{{{ buildSectionList()

    /**
     * Generates a list of available configuration sections after parsing
     * the XML definition file.
     *
     * @return GtkTreeView
     */
    function buildSectionList()
    {
        /* Creating a new model to store configuration sections */
        $model     = new GtkListStore(Gobject::TYPE_STRING);
        $listItems = $this->getSections();
        foreach ($listItems as $item) {
            $model->append(array($item));
        }
        $tree     = new GtkTreeView($model);
        $renderer = new GtkCellRendererText();
        $tree->append_column(new GtkTreeViewColumn("Sections",
            $renderer,
            "text",
            0));

        $itemSelect = $tree->get_selection();
        $itemSelect->set_mode(Gtk::SELECTION_SINGLE);
        $itemSelect->connect("changed", array($this,"onSectionSelect"));
        return $tree;
    }

    //}}}
    //{{{ onOptionSelect()

    /**
     * Sets the description of the configuration option currently selected
     * and displays it along with comments, if any.
     *
     * @param object $selected selected object
     *
     * @return void
     */
    function onOptionSelect($selected)
    {
        list($model, $iter)   = $selected->get_selected();
        list($sModel, $sIter) = $this->sectionTree->get_selection()->get_selected();
        if (!$iter || !$sIter) {
            return;
        }
        $option  = $model->get_value($iter, 0);
        $section = $sModel->get_value($sIter, 0);
        /* Querying XML definition file for configuration options
         * corresponding to the currently selected confugration section
         */

        if ($section == "Dynamic Extensions") {
            $desc = $this->xpath->query('/php-config-options/section/option[@id="extension"]/desc')->item(0)->nodeValue;
            $this->optionEntry->set_text(self::EXT_ENABLED);
            $this->optionDesc->set_text($desc);
            $this->optionComments->set_text("");
        } else {
            $desc     = $this->xpath->query('/php-config-options/section/option[@id="'.$option.'"]/desc')->item(0)->nodeValue;
            $defvalue = $this->xpath->query('/php-config-options/section/option[@id="'.$option.'"]/default')->item(0)->nodeValue;
            $this->optionEntry->set_text($defvalue);
            $this->optionDesc->set_text($desc);
            $this->optionComments->set_text($this->optionIndex[$option][self::COMMENT_NORMAL]);
        }
    }

    //}}}
    //{{{ onSectionSelect

    /**
     * Retreives and displays the set of configuration options associated
     * with the currently selected configuration section.
     *
     * @param GtkTreeSelection $selected Selection object corresponding to the
     *                                   currently selected configuration section.
     *
     * @return void
     */
    function onSectionSelect($selected)
    {
        list($model, $iter) = $selected->get_selected();
        if (!$iter) {
            return;
        }
        $section     =  $model->get_value($iter, 0);
        $optionList  = $this->getOptions($section);
        $optionModel = $this->optionTree->get_model();
        $optionModel->clear();

        /* Filling the model with corresponding configuration options */
        foreach ($optionList as $option) {
            $optionModel->append(array($option[0], $option[1], $option[2]));
        }
        for ($i=0;$i<4;$i++) {
            if ($this->optionTree->get_column(0)) {
                $this->optionTree->remove_column($this->optionTree->get_column(0));
            }
        }
        $cell = new GtkCellRendererText();
        //what to name the column?
        if ($section = "Dynamic Extensions") {
            $nameColumn = new GtkTreeViewColumn("Extension", $cell, "text", 0);
        } else {
            $nameColumn = new GtkTreeViewColumn("Option", $cell, "text", 0);
        }

        $currColumn = new GtkTreeViewColumn("Current Value", $cell, "text", 1);
        $defColumn  = new GtkTreeViewColumn("Default Value", $cell, "text", 2);
        $nameColumn->set_sort_column_id(0);
        $nameColumn->set_resizable(true);
        $currColumn->set_sort_column_id(1);
        $currColumn->set_resizable(true);
        $defColumn->set_sort_column_id(2);
        $defColumn->set_resizable(true);

        $this->optionTree->append_column($nameColumn);
        $this->optionTree->append_column($currColumn);
        $this->optionTree->append_column($defColumn);

        $optionSelect = $this->optionTree->get_selection();
        $optionSelect->set_mode(Gtk::SELECTION_SINGLE);
        $optionSelect->connect("changed", array($this, "onOptionSelect"));

        /* Initial description is empty */
        $this->optionDesc->set_text("");
    }

    //}}}
    //{{{ getSections()

    /**
     * Returns an array of possible configuration sections after parsing the
     * XML definition file.
     *
     * @return  Array
     */
    function getSections()
    {
        $sectionNodes = $this->xmlDefs->getElementsByTagName("section");
        foreach ($sectionNodes as $sectionNode) {
            $sectionName = $sectionNode->getAttribute("name");
            $sections[]  = $sectionName;
        }
        return $sections;
    }

    //}}}
    //{{{ getOptions()

    /**
     * Returns an array of possible configuration options corresponding to
     * a particular configuration section.
     *
     * @param string $section Name of the configuration section for
     *                        which the configuration options must be
     *                        returned.
     *
     * @return  Array
     */
    function getOptions($section)
    {
        $optionList = array();

        if ($section=="Dynamic Extensions") {
            foreach ($this->extensions as $extName=>$extData) {
                $optionList[] =
                    array($extName,
                          $extData[self::DIR_CURRENT],
                          self::EXT_DISABLED);
            }
        } else {
            $optionNodes =
                $this->xpath->query('/php-config-options/section[@name="' .
                    $section . '"]/option');
            foreach ($optionNodes as $optionNode) {
                $optionName   = $optionNode->getAttribute("id");
                $optionList[] =
                    array($optionName,
                          $this->optionIndex[$optionName][self::DIR_CURRENT],
                          $this->optionIndex[$optionName][self::DIR_DEFAULT]);
            }
        }
        return $optionList;
    }

    //}}}
    //{{{ saveFile()

    /**
     * Saves the current state of configuration into a file. Filename will
     * be prompted for if a new file is to be generated.
     *
     * @param bool $saveas is this a 'saveas' operation. default to false.
     *
     * @return void
     */
    function saveFile($saveas = false)
    {
        if ($this->fileName != null) {
            $toWrite = $this->optionIndex;
            $newFile = false;
            if (!file_exists($this->fileName)) {
                $fp = fopen($this->fileName, "w+");
                fclose($fp);
                $newFile    = true;
                $lineNumber = 0;
            }

            $theFile = file($this->fileName);
            $entry   = 0;
            foreach ($toWrite as $option=>$value) {
                /* Empty or Disabled configuration options are not written
                 * to the file. */
                if (!$newFile) {
                    $lineNumber = $value[self::DIR_NUMBER];
                } else {
                    $lineNumber += 1;
                }
                if (trim($value[self::COMMENT_INLINE]) == "") {
                    $inline = "\n";
                } else {
                    $inline = " ;".$value[self::COMMENT_INLINE]."\n";
                }
                if ($value[self::DIR_CURRENT] != "Disabled" &&
                    trim($value[self::DIR_CURRENT]) != "") {
                    if (trim($value[self::COMMENT_NORMAL]) != "") {
                        $theFile[$lineNumber] = "\n;/ ".str_replace("\n", "\n;/ ",
                        $value[self::COMMENT_NORMAL])."\n";
                    }
                    $theFile[$lineNumber] = $option . " = " .
                        $value[self::DIR_CURRENT] . $inline;
                } elseif ($value[self::DIR_CURRENT] == "Disabled" ||
                          trim($value[self::DIR_CURRENT]) == "") {
                        $theFile[$lineNumber] = ";"
                            . $option . " = "
                            . $value[self::DIR_DEFAULT] . $inline;
                }
            }

            foreach ($this->extensions as $extName=>$value) {
                if (!$newFile) {
                    $lineNumber = $value[self::DIR_NUMBER];
                } else {
                    $lineNumber += 1;
                }
                if (trim($value[self::COMMENT_INLINE]) == "") {
                    $inline = "\n";
                } else {
                    $inline = " ;".$value[self::COMMENT_INLINE]."\n";
                }
                if ($value[self::DIR_CURRENT] == self::EXT_ENABLED) {
                    $theFile[$lineNumber] = "extension = ".$extName.$inline;
                } else {
                    $theFile[$lineNumber] = ";extension = ".$extName.$inline;
                }
            }
            $this->saved = true;
            file_put_contents($this->fileName, $theFile);

        } else {
            /* New file to be saved, prompt for filename */
            $filePrompt = new GtkFileChooserDialog("Save file as:",
                $this,
                Gtk::FILE_CHOOSER_ACTION_SAVE,
                array(Gtk::STOCK_CANCEL,
                    Gtk::RESPONSE_CANCEL,
                    Gtk::STOCK_SAVE,
                    Gtk::RESPONSE_ACCEPT));

            if ($filePrompt->run() == Gtk::RESPONSE_ACCEPT) {
                $this->onSaveFile($filePrompt, $saveas);
                $this->saved = true;
            }
            $filePrompt->destroy();
        }
        $this->setTitle();
    }

    //}}}
    //{{{ onSaveFile()

    /**
     * Called in case a new file is to be generated after a filename has
     * been prompted for. Sets the filename property to the filename chosen
     * by the user and calls the saveFile() function.
     *
     * @param GtkFileSelection $filePrompt File selection dialog
     *                                     object that called
     *                                     this function.
     * @param bool             $saveas     is this a 'saveas' operation.
     *
     * @return void
     */
    function onSaveFile($filePrompt, $saveas)
    {
        $this->fileName = $filePrompt->get_filename();
        if ($saveas) {
            $saveasFile = fopen($this->fileName, "w+");
            file_put_contents($this->fileName, $this->saveasBuffer);
            fclose($saveasFile);
        }
        $this->saveFile();
    }

    //}}}
    //{{{ onSaveFileAs()

    /**
     * Called in case the "Save As..." option is selected. The function
     * will prompt for a filename to save the file as.
     *
     * @return void
     */
    function onSaveFileAs()
    {
        if ($this->fileName != null) {
            $this->saveasBuffer = file($this->fileName);
            $this->fileName     = null;
            $this->saveFile(true);
        } else {
            $this->saveFile();
        }
    }

    //}}}
    //{{{ onValueSubmit()

    /**
     * Sets the new user-defined value for the configuration option in the
     * buffer.
     *
     * @return void
     */
    function onValueSubmit()
    {
        $selection = $this->optionTree->get_selection();
        //set what's selected
        list($model, $iter)   = $selection->get_selected();
        list($sModel, $sIter) =
            $this->sectionTree->get_selection()->get_selected();
        if (!$iter || !$sIter) {
            return;
        }

        $option  = $model->get_value($iter, 0);
        $section = $sModel->get_value($sIter, 0);
        $value   = $this->optionEntry->get_text();
        $model->set($iter, 1, $value);

        /* Retreive the comments, if any, in the buffer */
        $comments = $this->optionComments;
        $comments = $comments->get_text($comments->get_start_iter(),
                                        $comments->get_end_iter());

        if ($section == "Dynamic Extensions") {
            $this->extensions[$option][self::DIR_CURRENT] = $value;
            if (trim($comments) != "") {
                $this->extensions[$option][self::COMMENT_NORMAL] = $comments;
            }
        } else {
            $this->optionIndex[$option][self::DIR_CURRENT] = $value;
            if (trim($comments) != "") {
                $this->extensions[$option][self::COMMENT_NORMAL] = $comments;
            }
        }

        /* Set saved flag to false */
        $this->saved = false;
        $this->setTitle();
    }

    //}}}
    //{{{ onValueUnset()

    /**
     * Unsets the value of the currently selected configuration directive to
     * disabled so that it does not appear in the configuration file.
     *
     *  @return void
     */
    function onValueUnset()
    {
        $selection            = $this->optionTree->get_selection();
        list($model, $iter)   = $selection->get_selected();
        list($sModel, $sIter) = $this->sectionTree->get_selection()->get_selected();
        if (!$iter || !$sIter) {
            return;
        }

        $option  = $model->get_value($iter, 0);
        $section = $sModel->get_value($sIter, 0);
        $model->set($iter, 1, "Disabled");

        if ($section == "Dynamic Extensions") {
            $this->extensions[$option][self::DIR_CURRENT] = self::EXT_DISABLED;
        } else {
            $this->optionIndex[$option][self::DIR_CURRENT] = "Disabled";
        }

        $this->saved = false;
        $this->setTitle();
    }

    //}}}
    //{{{ getValues()

    /**
     * Parses the XML definition file and returns the initial state of the
     * buffer. Additionally, if an existing file is opened, parses that
     * file and modifies the buffer to reflect the current state of that
     * file.
     *
     * @return  Array
     */
    function getValues()
    {
        $index       = array();
        $extensions  = array();
        $current     = "";
        $comment     = "";
        $optionNodes = $this->xmlDefs->getElementsByTagName("option");
        $options     = array();
        foreach ($optionNodes as $node) {
            $options[] = array($node->getAttribute("id"),
                $node->getElementsByTagName("default")->item(0)->nodeValue);
        }
        $this->showProgress();

        foreach ($options as $no => $optionData) {
            $total = count($options);
            if ($this->fileName != null) {
                $optionName = $optionData[0];
                $optionDef  = $optionData[1];
                /* Set all configuration options to disabled. These values
                 * will be rewritten after parsing the INI file
                 */
                $index[$optionName] = array(self::DIR_CURRENT=>"Disabled",
                    self::DIR_DEFAULT => $optionDef,
                    self::COMMENT_NORMAL => "",
                    self::COMMENT_INLINE => "",
                    self::DIR_NUMBER => "");
                $this->updateProgress(($no/$total)/10);
            } else {
                /* Set all configuration options to default values since a
                 * new file is to be generated
                 */
                $optionName = $optionData[0];
                $optionDef  = $optionData[1];
                // set option name
                $index[$optionName] = array(self::DIR_CURRENT => $optionDef,
                    self::DIR_DEFAULT => $optionDef,
                    self::COMMENT_NORMAL => "",
                    self::COMMENT_INLINE => "",
                    self::DIR_NUMBER => "");
                $this->updateProgress($no/$total);
            }
        }
        if ($this->fileName != null) {
            $status = file($this->fileName);
            $total  = count($status);
            foreach ($status as $no=>$line) {
                $this->updateProgress(0.1 + (0.9 * ($no / $total)));
                foreach ($options as $opNo => $data) {
                    if (strpos($line, "extension") !== false) {
                        if (substr($line, 0, 9) == "extension") {
                            $extStatus = self::EXT_ENABLED;
                        } elseif (substr($line, 1, 9) == "extension") {
                            $extStatus = self::EXT_DISABLED;
                        }
                        if (isset($extStatus)) {
                            $check = explode("=", $line);
                            if (trim($check[0]) == "extension" ||
                                trim($check[0]) == ";extension") {
                                $extValue    = $check[1];
                                $remComments = explode(";", $extValue);
                                $extName     = trim($remComments[0]);
                                if (array_key_exists(1, $remComments)) {
                                    $extensions[$extName][self::COMMENT_INLINE] =
                                        $remComments[1];
                                } else {
                                    $extensions[$extName][self::COMMENT_INLINE] = "";
                                }
                                $extensions[$extName][self::DIR_CURRENT] =
                                    $extStatus;
                                $extensions[$extName][self::DIR_NUMBER]  = $no;
                            }
                        }
                    }
                    if (strpos($line, $data[0])!==false) {
                        $delFlag      = false;
                        $optionName   = trim($data[0]);
                        $optionLength = strlen(trim($optionName));
                        $optionShown  = substr($line, 0, $optionLength);
                        $optionHidden = substr($line, 1, $optionLength);
                        if ($optionShown == $optionName) {
                            $whereis   = strpos($line, "=");
                            $directive = substr($line, 0, $whereis);
                            $values    = substr($line,
                                $whereis + 1,
                                strlen($line) - $whereis);
                            $values    = explode(";", $values);
                            /* Check if this is not part of a bigger extension */
                            if (strlen(trim($directive)) ==
                                strlen(trim($optionName))) {
                                $index[$optionName][self::DIR_CURRENT]    =
                                    trim($values[0]);
                                $index[$optionName][self::COMMENT_NORMAL] =
                                    $comment;
                                $index[$optionName][self::DIR_NUMBER]     =
                                    $no;
                                if (array_key_exists(1, $values)) {
                                    $index[$optionName][self::COMMENT_INLINE] =
                                        trim($values[1]);
                                }
                            }
                            $comment = "";
                            $delFlag = true;
                        } elseif ($optionHidden == $optionName
                               && $optionHidden != "extension") {
                            $index[$optionName][self::DIR_CURRENT] == "Disabled";
                            $index[$optionName][self::COMMENT_NORMAL] = "";
                            $index[$optionName][self::DIR_NUMBER]     = $no;
                            //reset flag
                            $delFlag = true;
                        } elseif (substr($line, 0, 2) == ";/" && $optionHi) {
                            if ($line != $current) {
                                $current = $line;
                                if (substr($line, 2, 1) == " ") {
                                    $comment .= substr($line, 3, strlen($line)-3);
                                } else {
                                    $comment .= substr($line, 2, strlen($line)-2);
                                }
                            }
                        }
                        /* Performance enhancing trick */
                        if ($delFlag) {
                            unset($options[$opNo]);
                        }
                    }
                }
            }
        }
        $this->progress->destroy();
        $this->extensions = $extensions;
        return $index;
    }

    //}}}
    //{{{ showProgress()

    /**
     * Creates and displays a progress bar that indicates
     * by what amount a file has been parsed.
     *
     * @return void
     */
    function showProgress()
    {
        while (Gtk::events_pending()) {
                Gtk::main_iteration();
        }
        $progressWin = new GtkWindow();
        $progressWin->set_modal(true);
        $progressWin->set_transient_for($this);
        $progressWin->set_title("Parsing File...");
        $progressBar = new GtkProgressBar();
        $progressBar->set_orientation(Gtk::PROGRESS_LEFT_TO_RIGHT);
        $progressWin->add($progressBar);
        $this->progress = $progressWin;
        $this->progress->set_size_request(300, 50);
        $this->progress->set_position(Gtk::WIN_POS_CENTER);
        $this->progress->show_all();

        return true;
    }

    //}}}
    //{{{ updateProgress()

    /**
     * Updates the progress by the fraction received as an argument by
     * the function.
     *
     * @param int $fraction The fraction by which the progress bar must
     *                      be updated.
     *
     * @return true
     */
    function updateProgress($fraction)
    {
        while (Gtk::events_pending()) {
                Gtk::main_iteration();
        }
        $theBar = $this->progress->get_child();
        $theBar->set_fraction($fraction);
        $percent = round(($fraction*100))."%";
        $theBar->set_text($percent);
        return true;
    }

    //}}}
    //{{{ openFile()

    /**
     * Prompts for saving of file that is currently opened and opens a
     * dialog that prompts for the next file the user wishes to open.
     *
     * @return void
     */
    function openFile()
    {
        if ($this->savePrompt()) {
            $filePrompt = new GtkFileChooserDialog("Open Existing file:", $this,
            Gtk::FILE_CHOOSER_ACTION_OPEN, array(Gtk::STOCK_CANCEL,
                Gtk::RESPONSE_CANCEL,
                Gtk::STOCK_OPEN,
                Gtk::RESPONSE_ACCEPT));
        }

        if ($filePrompt->run() == Gtk::RESPONSE_ACCEPT) {
            $this->fileName    = $filePrompt->get_filename();
            $this->optionIndex = $this->getValues();
        } else {
            if (is_object($filePrompt)) {
                $filePrompt->destroy();
            }
        }

        $itemSelect = $this->sectionTree->get_selection();
        $this->onSectionSelect($itemSelect);
        $this->setTitle();
    }

    //}}}
    //{{{ newFile()

    /**
     * Creates a new file to be edited after prompting for saving of the
     * existing file that is open.
     *
     * @return void
     */
    function newFile()
    {
        $this->savePrompt();
        $this->fileName    = null;
        $this->optionIndex = $this->getValues();
        $this->saved       = false;
        $this->setTitle();
    }

    //}}}
    //{{{ savePrompt()

    /**
     * Checks whether the existing open file is saved or not. If not,
     * prompts the user to save it.
     *
     * @return  bool    Returns true if users wants to quit, false
     *                  otherwise.
     */
    function savePrompt()
    {
        if ($this->saved) {
            return true;
        } else {
            /* Display confirmation dialog since buffer is not saved */
            $saveDialog = new GtkMessageDialog($this,
                1,
                Gtk::MESSAGE_QUESTION,
                Gtk::BUTTONS_NONE,
                "Do you want to save your existing INI file?");
            $saveDialog->add_button(Gtk::STOCK_CANCEL, Gtk::RESPONSE_CANCEL);
            $saveDialog->add_button(Gtk::STOCK_NO, Gtk::RESPONSE_NO);
            $saveDialog->add_button(Gtk::STOCK_YES, Gtk::RESPONSE_YES);
            $response = $saveDialog->run();
            $saveDialog->destroy();

            if ($response == Gtk::RESPONSE_YES) {
                /* Save file */
                $this->saveFile();
                return true;
            } elseif ($response == Gtk::RESPONSE_NO) {
                /* Don't save file */
                return true;
            } elseif ($response == Gtk::RESPONSE_CANCEL) {
                /* Don't quit */
                return false;
            }
        }
    }

    //}}}
    //{{{ setTitle()

    /**
     * Sets the title of the window as required.
     *
     * @return    string    The title of the window.
     */
    function setTitle()
    {
        $asterisk = "";
        if (!$this->saved) {
            $asterisk = "*";
        }

        if ($this->fileName==null) {
            $title = self::TITLE." - New File (Default Values) ".$asterisk;
        } else {
            $title = self::TITLE." - ".$this->fileName." ".$asterisk;
        }

        $this->set_title($title);
    }

    //}}}
    //{{{ quit()

    /**
     * Quits the main Gtk loop after checking whether the current state of
     * the buffer has been saved to a file or not.
     *
     * @return void
     */
    function quit()
    {
        if ($this->savePrompt()) {
            Gtk::main_quit();
        }
        return true;
    }

    //}}}

}

/* END */

?>
