# -*- python -*-

# This software was produced by NIST, an agency of the U.S. government,
# and by statute is not subject to copyright in the United States.
# Recipients of this software assume all responsibilities associated
# with its operation, modification and maintenance. However, to
# facilitate maintenance we ask that before distributing modified
# versions of this software, you first contact the authors at
# oof_manager@nist.gov. 

## When GUI tests are rerecorded, checkpoints can get out-of-sync with
## the asssert statements in the log file, because the assert
## statements are simply copied to the output when they're encountered
## in the input, whereas checkpoints are regenerated by running the
## OOF2 code (which is the whole point of rerecording a log file in
## the first place).  Generally, the assertions belong *after* the
## checkpoints.  This script takes a rerecorded log file and looks for
## places where assertions are immediately followed by checkpoints,
## and rearranges the lines so that the checkpoints are immediately
## before the assertions.

import os, sys

suffix = 'unfixed'

def fixfile(filename):
    if os.path.isdir(filename):
        filename = os.path.join(filename, 'log.py')
    if not os.path.exists(filename + '.' + suffix):
        backupname = filename + '.' + suffix
    else:
        backupname = None
        count = 2
        while not backupname:
            trialname = "%s.%s%d" % (filename, suffix, count)
            if not os.path.exists(trialname):
                backupname = trialname
            count += 1
    print "Saving original file", filename, "as", backupname
    os.rename(filename, backupname)

    output = file(filename, "w")

    count = 0
    assertions = []
    for line in file(backupname, "r"):
        line = line.rstrip()
        if len(line) == 0 or line[0] == '#':
            # Blank lines or comments are copied directly to the
            # output, unless they're immediately after assertions, in
            # which case they're kept with the assertions.
            if assertions:
                assertions.append(line)
            else:
                print >> output, line
        elif line.startswith("assert"):
            # Assertions are not written to the output immediately.
            # They're kept in a list and only written after the
            # subsequent checkpoints.
            assertions.append(line)
        elif not line.startswith("checkpoint "):
            # Lines that aren't checkpoints or assertions are copied
            # directly to the output, but only after any stored
            # assertions are written.
            for assertion in assertions:
                print >> output, assertion
            assertions = []
            print >> output, line
        else:
            # Checkpoints are copied directly to the output, thereby
            # putting them above the preceding assertions, if any.
            if assertions:
                count += 1
            print >> output, line
    output.close()
    print >> sys.stderr, "Moved", count, "checkpoints."

if __name__ == '__main__':
    fixfile(sys.argv[1])


