!! Copyright (C) 2002-2006 M. Marques, A. Castro, A. Rubio, G. Bertsch
!!
!! This program is free software; you can redistribute it and/or modify
!! it under the terms of the GNU General Public License as published by
!! the Free Software Foundation; either version 2, or (at your option)
!! any later version.
!!
!! This program is distributed in the hope that it will be useful,
!! but WITHOUT ANY WARRANTY; without even the implied warranty of
!! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!! GNU General Public License for more details.
!!
!! You should have received a copy of the GNU General Public License
!! along with this program; if not, write to the Free Software
!! Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
!! 02110-1301, USA.
!!

#include "global.h"

module vtk_oct_m
  use cube_function_oct_m
  use cube_oct_m
  use geometry_oct_m
  use global_oct_m
  use io_oct_m
  use io_binary_oct_m
  use messages_oct_m
  use namespace_oct_m
  use profiling_oct_m
  use species_oct_m
  use unit_oct_m

  implicit none

  private
  public ::                       &
    dvtk_out_cf,                  &
    zvtk_out_cf,                  &
    dvtk_out_cf_vector,           &
    zvtk_out_cf_vector,           &
    dvtk_out_cf_structured,       &
    zvtk_out_cf_structured,       &    
    vtk_output_geometry

contains

  !< check endianness
  !< Logical output: true is the running architecture uses little endian ordering, false otherwise.
  logical pure function is_little_endian() result(is_little)
    implicit none
    integer, parameter:: I4P  = selected_int_kind(9)  
    integer, parameter:: I1P  = selected_int_kind(2)
    integer(I1P) :: int1(1:4) !< One byte integer array for casting 4 bytes integer.

    int1 = transfer(1_I4P, int1)
    is_little = (int1(1) == 1_I1P)
    return
  end function is_little_endian

  ! -----------------------------------------------------
  
  subroutine vtk_output_geometry(filename, geo, namespace, ascii)
    character(len=*),           intent(in) :: filename
    type(geometry_t),           intent(in) :: geo
    type(namespace_t),          intent(in) :: namespace
    logical,          optional, intent(in) :: ascii

    integer :: iunit, iatom, ierr
    logical :: ascii_
    FLOAT, allocatable :: data(:, :)
    integer, allocatable :: idata(:, :)
    character(len=MAX_PATH_LEN) :: fullname

    ascii_ = optional_default(ascii, .true.)

    fullname = trim(filename)//".vtk"
    
    iunit = io_open(trim(fullname), namespace, action='write')

    write(iunit, '(1a)') '# vtk DataFile Version 3.0 '
    write(iunit, '(6a)') 'Generated by octopus ', trim(conf%version), ' -  git: ', & 
      trim(conf%git_commit), " build: ",  trim(conf%build_time)

    if(ascii_) then
      write(iunit, '(1a)') 'ASCII'
    else
      write(iunit, '(1a)') 'BINARY'
    end if
    
    write(iunit, '(1a)') 'DATASET POLYDATA'
    
    write(iunit, '(a,i9,a)') 'POINTS ', geo%natoms, ' double'

    if(ascii_) then
      do iatom = 1, geo%natoms
        write(iunit, '(3f12.6)') geo%atom(iatom)%x(1:3)
      end do
    else
      call io_close(iunit)
      SAFE_ALLOCATE(data(1:3, 1:geo%natoms))
      do iatom = 1, geo%natoms
        data(1:3, iatom) = geo%atom(iatom)%x(1:3)
      end do
      call io_binary_write(io_workpath(fullname, namespace), 3*geo%natoms, data, &
        ierr, nohead = .true., fendian = is_little_endian())
      SAFE_DEALLOCATE_A(data)
      iunit = io_open(trim(fullname), namespace, action='write', position = 'append')
      write(iunit, '(1a)') ''
    end if
    
    write(iunit, '(a,2i9)') 'VERTICES ', geo%natoms, 2*geo%natoms

    if(ascii_) then
      do iatom = 1, geo%natoms
        write(iunit, '(2i9)') 1, iatom - 1
      end do
    else
      call io_close(iunit)
      SAFE_ALLOCATE(idata(1:2, 1:geo%natoms))
      do iatom = 1, geo%natoms
        idata(1, iatom) = 1
        idata(2, iatom) = iatom - 1
      end do
      call io_binary_write(io_workpath(fullname, namespace), 2*geo%natoms, idata, &
        ierr, nohead = .true., fendian = is_little_endian())
      SAFE_DEALLOCATE_A(idata)
      iunit = io_open(trim(fullname), namespace, action='write', position = 'append')
      write(iunit, '(1a)') ''
    end if
    
    write(iunit, '(a,i9)') 'POINT_DATA', geo%natoms
    write(iunit, '(a)') 'SCALARS element integer'
    write(iunit, '(a)') 'LOOKUP_TABLE default'

    if(ascii_) then
      do iatom = 1, geo%natoms
        write(iunit, '(i9)') nint(species_z(geo%atom(iatom)%species))
      end do
    else
      call io_close(iunit)

      SAFE_ALLOCATE(idata(1:geo%natoms, 1:1))

      do iatom = 1, geo%natoms
        idata(iatom, 1) = nint(species_z(geo%atom(iatom)%species))
      end do
      
      call io_binary_write(io_workpath(fullname, namespace), geo%natoms, idata, &
        ierr, nohead = .true., fendian = is_little_endian())

      SAFE_DEALLOCATE_A(idata)

      iunit = io_open(trim(fullname), namespace, action='write', position = 'append')
      write(iunit, '(1a)') ''
    end if
    
    call io_close(iunit)

  end subroutine vtk_output_geometry

  
#include "undef.F90"
#include "real.F90"
#include "vtk_inc.F90"

#include "undef.F90"
#include "complex.F90"
#include "vtk_inc.F90"
#include "undef.F90"

end module vtk_oct_m

!! Local Variables:
!! mode: f90
!! coding: utf-8
!! End:
