#include "ldastoolsal/MkDir.hh"

extern "C" {
#include <sys/stat.h>
#include <unistd.h>
} // extern "C"

#include <cerrno>
#include <sstream>
#include <stdexcept>

#ifndef ENOERR
#define ENOERR 0
#endif /* ENOERR */

namespace
{
  const char*
  failure( int Error )
  {
    switch( Error )
    {
    case EACCES:
      return "Insufficient permission";
    default:
      break;
    }
    return "Unspecified system error";
  }
}

namespace LDASTools
{
  namespace Cmd
  {
    const char MkDir::PATH_SEP = '/';

    MkDir
    ::MkDir( mode_type Mode, option_type Options )
      : mode( Mode ),
	opt( Options )
    {
    } // MkDir::MkDir

    void MkDir::
    operator()( const std::string& Directory ) const
    {
      errno = ENOERR;

      eval( Directory );

      if ( errno != ENOERR )
      {
	std::ostringstream	msg;

	msg << "Failed to create directory: " << Directory
	    << " (" << failure( errno ) << ")"
	  ;

	throw std::runtime_error( msg.str( ) );
      }
    } // MkDir::operator


    void MkDir::
    eval( const std::string& Directory ) const
    {
      errno = ENOERR;

      if ( ::mkdir( Directory.c_str( ),
		    mode ) != 0 )
      {
	if ( errno == EEXIST )
	{
	  //-------------------------------------------------------------
	  // Stop because the directory alread exists.
	  // Do not consider this a error.
	  //-------------------------------------------------------------
	  errno = ENOERR;
	  return;
	}
	if ( ( errno == ENOENT )
	     && ( mode | OPT_MAKE_PARENT_DIRECTORIES ) )
	{
	  //-------------------------------------------------------------
	  // Recursive call to try to create the parent
	  //-------------------------------------------------------------
	  size_t pos = Directory.rfind( PATH_SEP );

	  if ( ( pos != std::string::npos )
	       && ( pos > 1 ) )
	  {
	    this->operator( )( Directory.substr( 0, pos ) );
	    if ( ( errno == ENOENT ) || ( errno == ENOERR ) )
	    {
	      ::mkdir( Directory.c_str( ), mode );
	    } // if - errno
	  } // if - pos
	} // if - errno
      } // if - mkdir

    } // MkDir::eval
  }
}
