/* -*- mode: c++; c-basic-offset: 2; indent-tabs-mode: nil; -*- */
#ifndef DISK_CACHE__ATOMIC_FILE_HH
#define DISK_CACHE__ATOMIC_FILE_HH

#include <stdio.h>

#include <sstream>
#include <stdexcept>

#include "IO.hh"

#include "Streams/StreamsInterface.hh"

namespace diskCache
{
  template< typename STREAM, typename DCSTREAM >
  class AtomicFile
  {
  public:
    AtomicFile( const std::string& Filename );

    ~AtomicFile( );

    void Read( );

    void Write( Streams::Interface::version_type Version = DCSTREAM::VERSION_DEFAULT );

  private:
    std::string m_filename;
    std::string m_filename_tmp;

    STREAM      m_stream;

    static std::string tmp_filename( const std::string& Filename );
  };

  template< typename STREAM, typename DCSTREAM >
  inline AtomicFile< STREAM, DCSTREAM >::
  AtomicFile( const std::string& Filename )
    : m_filename( Filename )
  {
    
  }

  template< typename STREAM, typename DCSTREAM >
  inline AtomicFile< STREAM, DCSTREAM >::
  ~AtomicFile( )
  {
    m_stream.close( );
  }

  template< typename STREAM, typename DCSTREAM >
  inline void AtomicFile< STREAM, DCSTREAM >::
  Read( )
  {
    m_stream.open( m_filename.c_str( ) );

    {
      DCSTREAM    dc_stream( m_stream );;

      diskCache::Read( dc_stream );
    }
    m_stream.close( );
  }

  template< typename STREAM, typename DCSTREAM >
  inline void AtomicFile< STREAM, DCSTREAM >::
  Write( Streams::Interface::version_type Version )
  {
    static const char backup_ext[] = ".bak";
    std::ostringstream  backup;

    backup << m_filename
           << backup_ext
      ;
    m_filename_tmp = tmp_filename( m_filename );

    {
      m_stream.open( m_filename_tmp.c_str( ) );

      DCSTREAM    dc_stream( m_stream, Version );;

      diskCache::Write( dc_stream );
    }
    m_stream.close( );

    (void)( rename( m_filename.c_str( ), backup.str( ).c_str( ) ) );
    if( rename( m_filename_tmp.c_str(), m_filename.c_str() ) )
    {
         // Creating backup copy failed:
      std::ostringstream     msg;
      msg << "Could not create backup copy of already existing cache file \'"
          << m_filename
          << "\' (errno=" << strerror( errno )
          << "). New cache file is stored as temporary \'"
          << m_filename_tmp
          << '\''
        ;
      
      throw std::runtime_error( msg.str( ) );
    }

  }

  template< typename STREAM, typename DCSTREAM >
  inline std::string AtomicFile< STREAM, DCSTREAM >::
  tmp_filename( const std::string& Filename )
  {
    std::string retval( Filename );
    retval += ".tmp";

    return retval;
  }
} // namespace - GenericAPI

#endif /* DISK_CACHE__ATOMIC_FILE_HH */
