package org.springframework.uaa.client;

import org.springframework.uaa.client.internal.BasicProxyService;
import org.springframework.uaa.client.internal.JdkUrlTransmissionServiceImpl;
import org.springframework.uaa.client.internal.TransmissionAwareUaaServiceImpl;
import org.springframework.uaa.client.internal.UaaDetectedProductsImpl;

/**
 * Singleton factory to obtain the correct {@link UaaService} implementation.
 * <p>
 * Caller can provide an optional {@link TransmissionService} implementation. If none is provided a default
 * {@link JdkUrlTransmissionServiceImpl} will be used to handle UAA's internal data up- and download requirements.
 * <p>
 * Note: Only the first call to {@link #getUaaService(TransmissionService)} per JVM (well to be precise: per classloader
 * hierarchy) will be able to set the {@link TransmissionService} service that UAA will use.
 * 
 * @author Christian Dupuis
 * @since 1.0.1
 * @see TransmissionService
 * @see JdkUrlTransmissionServiceImpl
 */
public abstract class UaaServiceFactory {

	/**
	 * Default {@link ProxyService} instance to use if no special proxy features are required
	 */
	public static final ProxyService DEFAULT_PROXY_SERVICE = new BasicProxyService();
	
	/** Singleton {@link UaaService} instance */
	private static UaaService _uaaService;
	
	/** Singleton {@link UaaDetectedProducts} instance */
	private static UaaDetectedProducts _uaaDetectedProducts;
	
	/** Singleton {@link ProxyService} instance */
	private static ProxyService _proxyService;
	
	/**
	 * @return the singleton {@link UaaService} instance to use
	 */
	public static UaaService getUaaService() {
		return getUaaService(DEFAULT_PROXY_SERVICE);
	}

	/**
	 * Obtain the singleton {@link UaaService}, creating it if necessary. Also stores the
	 * passed services and these become singletons for the remaining methods.
	 * 
	 * <p>
	 * After invoking this method you cannot change the singleton proxy service or UAA service
	 * that will subsequently be returned by this factory.
	 * 
	 * @param proxyService the proxy service to use (required, cannot be null)
	 * @return the singleton {@link UaaService} instance to use
	 */
	public static synchronized UaaService getUaaService(ProxyService proxyService) {
		if (_proxyService == null) {
			_proxyService = proxyService;
		}
		if (_uaaService == null) {
			_uaaService = new TransmissionAwareUaaServiceImpl(new JdkUrlTransmissionServiceImpl(_proxyService));
		}
		return _uaaService;
	}

	/**
	 * Obtain the singleton {@link UaaDetectedProducts}.
	 * 
	 * @return the singleton {@link UaaDetectedProducts} instance to use
	 */	
	public static synchronized UaaDetectedProducts getUaaDetectedProducts() {
		if (_uaaDetectedProducts == null) {
			_uaaDetectedProducts = new UaaDetectedProductsImpl();
		}
		return _uaaDetectedProducts;
	}
	
	/**
	 * Obtain the singleton {@link ProxyService}.
	 * 
	 * @return the singleton {@link ProxyService} instance to use
	 */
	public static synchronized ProxyService getProxyService() {
		if (_proxyService == null) {
			_proxyService = DEFAULT_PROXY_SERVICE;
		}
		return _proxyService;
	}
	
}
