/* -*- c++ -*- */
/*
 * Copyright 2004 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street,
 * Boston, MA 02110-1301, USA.
 */

/*
 * gr-baz by Balint Seeber (http://spench.net/contact)
 * Information, documentation & samples: http://wiki.spench.net/wiki/gr-baz
 */

/*
 * config.h is generated by configure.  It contains the results
 * of probing for features, options etc.  It should be the first
 * file included in your .cc file.
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <baz_pow_cc.h>
#include <gnuradio/io_signature.h>

/*
 * Create a new instance of baz_pow_cc and return
 * a boost shared_ptr.  This is effectively the public constructor.
 */
baz_pow_cc_sptr 
baz_make_pow_cc (float exponent, float div_exp /*= 0.0*/)
{
  return baz_pow_cc_sptr (new baz_pow_cc (exponent, div_exp));
}

/*
 * Specify constraints on number of input and output streams.
 * This info is used to construct the input and output signatures
 * (2nd & 3rd args to gr::block's constructor).  The input and
 * output signatures are used by the runtime system to
 * check that a valid number and type of inputs and outputs
 * are connected to this block.  In this case, we accept
 * only 1 input and 1 output.
 */
static const int MIN_IN = 1;	// mininum number of input streams
static const int MAX_IN = 1;	// maximum number of input streams
static const int MIN_OUT = 1;	// minimum number of output streams
static const int MAX_OUT = 1;	// maximum number of output streams

/*
 * The private constructor
 */
baz_pow_cc::baz_pow_cc (float exponent, float div_exp /*= 0.0*/)
  : gr::sync_block ("pow_cc",
		   gr::io_signature::make (MIN_IN, MAX_IN, sizeof (gr_complex)),
		   gr::io_signature::make (MIN_OUT, MAX_OUT, sizeof (gr_complex)))
  , d_exponent(exponent), d_div_exp(div_exp)
{
}

/*
 * Our virtual destructor.
 */
baz_pow_cc::~baz_pow_cc ()
{
}

void baz_pow_cc::set_exponent(float exponent)
{
  d_exponent = exponent;
}

void baz_pow_cc::set_division_exponent(float div_exp)
{
  d_div_exp = div_exp;
}

int 
baz_pow_cc::work (int noutput_items,
			gr_vector_const_void_star &input_items,
			gr_vector_void_star &output_items)
{
  const gr_complex *in = (const gr_complex *) input_items[0];
  gr_complex *out = (gr_complex *) output_items[0];

  //std::complex</*float*/double> e(d_exponent,0);
  for (int i = 0; i < noutput_items; i++){
	std::complex<double> d(in[i].real(), in[i].imag());
	d = pow(d, d_exponent);
	if (d_div_exp != 0.0f)
	  d /= pow(10.0, (double)d_div_exp);
    //out[i] = pow(in[i], d_exponent);
	out[i] = std::complex<float>((float)d.real(), (float)d.imag());
  }

  return noutput_items;
}
