package tiltfile

import (
	"bytes"
	"embed"
	"fmt"
	"io/fs"
	"os"
	"path/filepath"

	"github.com/tilt-dev/tilt/pkg/model"
)

//go:embed api/*.py api/*/*.py
var api embed.FS

const autoGeneratedHeader = `### AUTO-GENERATED ###
# This file was auto-generated by 'tilt dump api-docs' as part of a Tilt release.
# To make changes, modify the following file in the Tilt repository:
#   internal/tiltfile/%s
# Generated by Tilt version %s
### AUTO-GENERATED ###

`

func ApiStubs() fs.FS {
	f, err := fs.Sub(api, "api")
	if err != nil {
		panic(err)
	}
	return f
}

func WalkApiStubs(fn fs.WalkDirFunc) error {
	return fs.WalkDir(api, "api", fn)
}

func DumpApiStubs(dir string, info model.TiltBuild, callback func(string, error)) error {
	return WalkApiStubs(func(path string, d fs.DirEntry, e error) error {
		if e != nil {
			return e
		}
		var err error
		dest := filepath.Join(dir, path)
		if d.IsDir() {
			err = os.MkdirAll(dest, 0755)
		} else {
			buf := bytes.NewBufferString(fmt.Sprintf(autoGeneratedHeader, path, info.HumanBuildStamp()))
			var bytes []byte
			bytes, err = api.ReadFile(path)
			if err != nil {
				return err
			}
			buf.Write(bytes)
			err = os.WriteFile(dest, buf.Bytes(), 0644)
		}
		callback(path, err)
		return err
	})
}
