// Copyright 2016-2018, Pulumi Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package filestate

import (
	"context"
	"time"

	"github.com/pulumi/pulumi/sdk/v3/go/common/tokens"

	"github.com/pulumi/pulumi/pkg/v3/backend"
	"github.com/pulumi/pulumi/pkg/v3/engine"
	"github.com/pulumi/pulumi/pkg/v3/operations"
	"github.com/pulumi/pulumi/pkg/v3/resource/deploy"
	"github.com/pulumi/pulumi/sdk/v3/go/common/apitype"
	"github.com/pulumi/pulumi/sdk/v3/go/common/util/result"
)

// Stack is a local stack.  This simply adds some local-specific properties atop the standard backend stack interface.
type Stack interface {
	backend.Stack
	Path() string // a path to the stack's checkpoint file on disk.
}

// localStack is a local stack descriptor.
type localStack struct {
	ref      backend.StackReference // the stack's reference (qualified name).
	path     string                 // a path to the stack's checkpoint file on disk.
	snapshot *deploy.Snapshot       // a snapshot representing the latest deployment state.
	b        *localBackend          // a pointer to the backend this stack belongs to.
}

func newStack(ref backend.StackReference, path string, snapshot *deploy.Snapshot, b *localBackend) Stack {
	return &localStack{
		ref:      ref,
		path:     path,
		snapshot: snapshot,
		b:        b,
	}
}

func (s *localStack) Ref() backend.StackReference                            { return s.ref }
func (s *localStack) Snapshot(ctx context.Context) (*deploy.Snapshot, error) { return s.snapshot, nil }
func (s *localStack) Backend() backend.Backend                               { return s.b }
func (s *localStack) Path() string                                           { return s.path }

func (s *localStack) Remove(ctx context.Context, force bool) (bool, error) {
	return backend.RemoveStack(ctx, s, force)
}

func (s *localStack) Rename(ctx context.Context, newName tokens.QName) (backend.StackReference, error) {
	return backend.RenameStack(ctx, s, newName)
}

func (s *localStack) Preview(ctx context.Context, op backend.UpdateOperation) (engine.ResourceChanges, result.Result) {
	return backend.PreviewStack(ctx, s, op)
}

func (s *localStack) Update(ctx context.Context, op backend.UpdateOperation) (engine.ResourceChanges, result.Result) {
	return backend.UpdateStack(ctx, s, op)
}

func (s *localStack) Import(ctx context.Context, op backend.UpdateOperation,
	imports []deploy.Import) (engine.ResourceChanges, result.Result) {
	return backend.ImportStack(ctx, s, op, imports)
}

func (s *localStack) Refresh(ctx context.Context, op backend.UpdateOperation) (engine.ResourceChanges, result.Result) {
	return backend.RefreshStack(ctx, s, op)
}

func (s *localStack) Destroy(ctx context.Context, op backend.UpdateOperation) (engine.ResourceChanges, result.Result) {
	return backend.DestroyStack(ctx, s, op)
}

func (s *localStack) Watch(ctx context.Context, op backend.UpdateOperation, paths []string) result.Result {
	return backend.WatchStack(ctx, s, op, paths)
}

func (s *localStack) GetLogs(ctx context.Context, cfg backend.StackConfiguration,
	query operations.LogQuery) ([]operations.LogEntry, error) {
	return backend.GetStackLogs(ctx, s, cfg, query)
}

func (s *localStack) ExportDeployment(ctx context.Context) (*apitype.UntypedDeployment, error) {
	return backend.ExportStackDeployment(ctx, s)
}

func (s *localStack) ImportDeployment(ctx context.Context, deployment *apitype.UntypedDeployment) error {
	return backend.ImportStackDeployment(ctx, s, deployment)
}

type localStackSummary struct {
	name backend.StackReference
	chk  *apitype.CheckpointV3
}

func newLocalStackSummary(name backend.StackReference, chk *apitype.CheckpointV3) localStackSummary {
	return localStackSummary{name: name, chk: chk}
}

func (lss localStackSummary) Name() backend.StackReference {
	return lss.name
}

func (lss localStackSummary) LastUpdate() *time.Time {
	if lss.chk != nil && lss.chk.Latest != nil {
		if t := lss.chk.Latest.Manifest.Time; !t.IsZero() {
			return &t
		}
	}
	return nil
}

func (lss localStackSummary) ResourceCount() *int {
	if lss.chk != nil && lss.chk.Latest != nil {
		count := len(lss.chk.Latest.Resources)
		return &count
	}
	return nil
}
