package forge

import (
	"fmt"
	"strings"

	"github.com/git-town/git-town/v22/internal/config/configdomain"
	"github.com/git-town/git-town/v22/internal/forge/forgedomain"
	"github.com/git-town/git-town/v22/internal/git/gitdomain"
	. "github.com/git-town/git-town/v22/pkg/prelude"
)

const indentMarker = "-"

type ProposalStackLineageArgs struct {
	Connector                forgedomain.ProposalFinder
	CurrentBranch            gitdomain.LocalBranchName
	Lineage                  configdomain.Lineage
	MainAndPerennialBranches gitdomain.LocalBranchNames
	Order                    configdomain.Order
}

func NewProposalStackLineageBuilder(args ProposalStackLineageArgs, lineageTree OptionalMutable[ProposalStackLineageTree]) Option[ProposalStackLineageBuilder] {
	if args.MainAndPerennialBranches.Contains(args.CurrentBranch) ||
		args.Lineage.Len() == 0 {
		// cannot create proposal stack lineage for main or perennial branch
		return None[ProposalStackLineageBuilder]()
	}

	tree, hasTree := lineageTree.Get()
	if !hasTree {
		var err error
		tree, err = NewProposalStackLineageTree(args)
		if err != nil {
			fmt.Printf("failed to build proposal stack lineage: %s", err.Error())
			return None[ProposalStackLineageBuilder]()
		}
	}

	if tree == nil {
		return None[ProposalStackLineageBuilder]()
	}

	builder := &ProposalStackLineageBuilder{
		mainAndPerennialBranches: args.MainAndPerennialBranches,
		tree:                     tree,
	}

	return Some(*builder)
}

type ProposalStackLineageBuilder struct {
	mainAndPerennialBranches gitdomain.LocalBranchNames
	tree                     *ProposalStackLineageTree
}

// Build returns the proposal stack lineage as a string
func (self *ProposalStackLineageBuilder) Build(args ProposalStackLineageArgs) string {
	var builder strings.Builder
	builder.WriteString("\n-------------------------\n")
	builder.WriteString(self.build(self.tree.Node, args, 0))
	builder.WriteString("\nStack generated by [Git Town](https://github.com/git-town/git-town)\n")
	return builder.String()
}

// UpdateStack updates the underlying tree representation of the proposal stack. If proposal data was
// fetched on a previous call to UpdateStack or during ProposalStackLineageBuilder, the underlying
// data structure fetches that information from a map.
func (self *ProposalStackLineageBuilder) UpdateStack(args ProposalStackLineageArgs) error {
	return self.tree.Rebuild(args)
}

func (self *ProposalStackLineageBuilder) build(node *ProposalStackLineageTreeNode, args ProposalStackLineageArgs, indentLevel int) string {
	var builder strings.Builder
	indent := strings.Repeat(" ", indentLevel*2)
	nextIndentLevel := indentLevel + 1
	if args.MainAndPerennialBranches.Contains(node.branch) {
		builder.WriteString(fmt.Sprintf("%s %s %s\n", indent, indentMarker, node.branch.BranchName()))
		for _, child := range node.childNodes {
			builder.WriteString(self.build(child, args, nextIndentLevel))
		}
		return builder.String()
	}

	proposal, hasProposal := node.proposal.Get()
	if hasProposal {
		builder.WriteString(formattedDisplay(args, indent, proposal))
	} else {
		nextIndentLevel--
	}

	for _, child := range node.childNodes {
		builder.WriteString(self.build(child, args, nextIndentLevel))
	}

	return builder.String()
}

func formattedDisplay(args ProposalStackLineageArgs, currentIndentLevel string, proposal forgedomain.Proposal) string {
	proposalData := proposal.Data
	if args.CurrentBranch == proposalData.Data().Source {
		return fmt.Sprintf("%s %s %s :point_left:\n", currentIndentLevel, indentMarker, proposalData.Data().URL)
	}
	return fmt.Sprintf("%s %s %s\n", currentIndentLevel, indentMarker, proposalData.Data().URL)
}
