package file_test

import (
	"bytes"
	"flag"
	"io/ioutil"
	"path/filepath"
	"testing"

	"github.com/shihanng/gig/internal/file"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

//nolint:gochecknoglobals
var update = flag.Bool("update", false, "update .golden files")

func TestList(t *testing.T) {
	type args struct {
		directory string
	}

	tests := []struct {
		name      string
		args      args
		want      []string
		assertion assert.ErrorAssertionFunc
	}{
		{
			name: "directory not found",
			args: args{
				directory: "unknown",
			},
			want:      nil,
			assertion: assert.Error,
		},
		{
			name: "normal",
			args: args{
				directory: "testdata",
			},
			want: []string{
				"1c",
				"1c-bitrix",
				"a-frame",
				"actionscript",
				"ada",
				"adobe",
				"advancedinstaller",
				"adventuregamestudio",
				"agda",
				"al",
				"alteraquartusii",
				"altium",
				"android",
				"androidstudio",
				"angular",
				"anjuta",
				"ansible",
				"apachecordova",
				"apachehadoop",
				"appbuilder",
				"appceleratortitanium",
				"appcode",
				"appcode+all",
				"appcode+iml",
				"appengine",
				"aptanastudio",
				"arcanist",
				"archive",
				"archives",
				"archlinuxpackages",
				"aspnetcore",
				"assembler",
				"ate",
				"atmelstudio",
				"ats",
				"audio",
				"automationstudio",
				"autotools",
				"autotools+strict",
				"awr",
				"backup",
				"ballerina",
				"basercms",
				"basic",
				"batch",
				"bazaar",
				"bazel",
				"bitrise",
				"bitrix",
				"bittorrent",
				"blackbox",
				"bloop",
				"bluej",
				"bookdown",
				"bower",
				"bricxcc",
				"buck",
				"c",
				"c++",
				"cake",
				"cakephp",
				"cakephp2",
				"cakephp3",
				"calabash",
				"carthage",
				"certificates",
				"ceylon",
				"cfwheels",
				"chefcookbook",
				"chocolatey",
				"clean",
				"clion",
				"clion+all",
				"clion+iml",
				"clojure",
				"cloud9",
				"cmake",
				"cocoapods",
				"cocos2dx",
				"cocoscreator",
				"code",
				"code-java",
				"codeblocks",
				"codecomposerstudio",
				"codeigniter",
				"codeio",
				"codekit",
				"codesniffer",
				"coffeescript",
				"commonlisp",
				"composer",
				"compressed",
				"compressedarchive",
				"compression",
				"concrete5",
				"coq",
				"cordova",
				"craftcms",
				"crashlytics",
				"crbasic",
				"crossbar",
				"crystal",
				"csharp",
				"cuda",
				"cvs",
				"d",
				"dart",
				"darteditor",
				"data",
				"database",
				"datarecovery",
				"dbeaver",
				"defold",
				"delphi",
				"dframe",
				"diff",
				"direnv",
				"diskimage",
				"django",
				"dm",
				"docfx",
				"docpress",
				"docz",
				"dotenv",
				"dotfilessh",
				"dotnetcore",
				"dotsettings",
				"dreamweaver",
				"dropbox",
				"drupal",
				"drupal7",
				"drupal8",
				"e2studio",
				"eagle",
				"easybook",
				"eclipse",
				"eiffelstudio",
				"elasticbeanstalk",
				"elisp",
				"elixir",
				"elm",
				"emacs",
				"ember",
				"ensime",
				"episerver",
				"erlang",
				"espresso",
				"executable",
				"exercism",
				"expressionengine",
				"extjs",
				"fancy",
				"fastlane",
				"finale",
				"firebase",
				"flashbuilder",
				"flask",
				"flex",
				"flexbuilder",
				"floobits",
				"flutter",
				"font",
				"fontforge",
				"forcedotcom",
				"forgegradle",
				"fortran",
				"freepascal",
				"fsharp",
				"fuelphp",
				"fusetools",
				"games",
				"gcov",
				"genero4gl",
				"geth",
				"ggts",
				"gis",
				"git",
				"gitbook",
				"go",
				"godot",
				"goodsync",
				"gpg",
				"gradle",
				"grails",
				"greenfoot",
				"grunt",
				"gwt",
				"haskell",
				"helm",
				"hexo",
				"hol",
				"homeassistant",
				"hsp",
				"hugo",
				"hyperledgercomposer",
				"iar",
				"iar_ewarm",
				"iarembeddedworkbench",
				"idapro",
				"idris",
				"igorpro",
				"images",
				"infer",
				"inforcms",
				"inforcrm",
				"intellij",
				"intellij+all",
				"intellij+iml",
				"ionic3",
				"jabref",
				"java",
				"java-web",
				"jboss",
				"jboss-4-2-3-ga",
				"jboss-6-x",
				"jboss4",
				"jboss6",
				"jdeveloper",
				"jekyll",
				"jenv",
				"jetbrains",
				"jetbrains+all",
				"jetbrains+iml",
				"jgiven",
				"jigsaw",
				"jmeter",
				"joe",
				"joomla",
				"jspm",
				"julia",
				"jupyternotebooks",
				"justcode",
				"kate",
				"kdevelop4",
				"kdiff3",
				"keil",
				"kentico",
				"kicad",
				"kirby2",
				"kobalt",
				"kohana",
				"komodoedit",
				"konyvisualizer",
				"kotlin",
				"labview",
				"labviewnxg",
				"lamp",
				"laravel",
				"latex",
				"lazarus",
				"leiningen",
				"lemonstand",
				"less",
				"liberosoc",
				"librarian-chef",
				"libreoffice",
				"lilypond",
				"linux",
				"lithium",
				"logtalk",
				"lsspice",
				"ltspice",
				"lua",
				"lyx",
				"m2e",
				"macos",
				"magento",
				"magento1",
				"magento2",
				"magic-xpa",
				"matlab",
				"maven",
				"mavensmate",
				"mdbook",
				"mean",
				"mercurial",
				"mercury",
				"metals",
				"metaprogrammingsystem",
				"meteor",
				"meteorjs",
				"microsoftoffice",
				"mikroc",
				"moban",
				"modelsim",
				"modx",
				"momentics",
				"monodevelop",
				"mplabx",
				"mule",
				"nanoc",
				"nativescript",
				"ncrunch",
				"nesc",
				"netbeans",
				"nette",
				"nikola",
				"nim",
				"ninja",
				"node",
				"nodechakratimetraveldebug",
				"notepadpp",
				"nuxt",
				"nuxtjs",
				"nwjs",
				"objective-c",
				"ocaml",
				"octave",
				"octobercms",
				"opa",
				"opencart",
				"opencv",
				"openfoam",
				"openframeworks",
				"openframeworks+visualstudio",
				"oracleforms",
				"orcad",
				"osx",
				"otto",
				"oxideshop",
				"oxygenxmleditor",
				"packer",
				"particle",
				"patch",
				"pawn",
				"perl",
				"perl6",
				"ph7cms",
				"phalcon",
				"phoenix",
				"phpcodesniffer",
				"phpstorm",
				"phpstorm+all",
				"phpstorm+iml",
				"phpunit",
				"pico8",
				"pimcore",
				"pimcore4",
				"pimcore5",
				"pinegrow",
				"platformio",
				"playframework",
				"plone",
				"polymer",
				"powershell",
				"premake-gmake",
				"prepros",
				"prestashop",
				"processing",
				"progressabl",
				"psoccreator",
				"puppet",
				"puppet-librarian",
				"purebasic",
				"purescript",
				"putty",
				"pvs",
				"pycharm",
				"pycharm+all",
				"pycharm+iml",
				"pydev",
				"python",
				"qml",
				"qooxdoo",
				"qt",
				"qtcreator",
				"r",
				"racket",
				"rails",
				"react",
				"reactnative",
				"reasonml",
				"red",
				"redcar",
				"redis",
				"rhodesrhomobile",
				"rider",
				"root",
				"ros",
				"ruby",
				"rubymine",
				"rubymine+all",
				"rubymine+iml",
				"rust",
				"salesforce",
				"salesforcedx",
				"sas",
				"sass",
				"sbt",
				"scala",
				"scheme",
				"scons",
				"scrivener",
				"sdcc",
				"seamgen",
				"senchatouch",
				"serverless",
				"shopware",
				"silverstripe",
				"sketchup",
				"slickedit",
				"smalltalk",
				"snap",
				"snapcraft",
				"solidity",
				"soliditytruffle",
				"sonar",
				"sonarqube",
				"sourcepawn",
				"spark",
				"splunk",
				"spreadsheet",
				"ssh",
				"standardml",
				"stata",
				"stdlib",
				"stella",
				"stellar",
				"stylus",
				"sublimetext",
				"sugarcrm",
				"svn",
				"swift",
				"swiftpackagemanager",
				"swiftpm",
				"symfony",
				"symphonycms",
				"synology",
				"synopsysvcs",
				"tags",
				"tarmainstallmate",
				"terraform",
				"terragrunt",
				"test",
				"testcomplete",
				"testinfra",
				"tex",
				"text",
				"textmate",
				"textpattern",
				"theos-tweak",
				"thinkphp",
				"tortoisegit",
				"tower",
				"turbogears2",
				"twincat",
				"typings",
				"typo3",
				"typo3-composer",
				"umbraco",
				"unity",
				"unrealengine",
				"vaadin",
				"vagrant",
				"valgrind",
				"vapor",
				"venv",
				"vertx",
				"video",
				"vim",
				"virtualenv",
				"virtuoso",
				"visualstudio",
				"visualstudiocode",
				"vivado",
				"vlab",
				"vue",
				"vuejs",
				"vvvv",
				"waf",
				"wakanda",
				"web",
				"webmethods",
				"webstorm",
				"webstorm+all",
				"webstorm+iml",
				"werckercli",
				"windows",
				"wintersmith",
				"wordpress",
				"wyam",
				"xamarinstudio",
				"xcode",
				"xcodeinjection",
				"xilinx",
				"xilinxise",
				"xilinxvivado",
				"xill",
				"xojo",
				"xtext",
				"y86",
				"yeoman",
				"yii",
				"yii2",
				"zendframework",
				"zephir",
				"zig",
				"zsh",
				"zukencr8000",
			},
			assertion: assert.NoError,
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			got, err := file.List(tt.args.directory)
			tt.assertion(t, err)
			assert.Equal(t, tt.want, got)
		})
	}
}

func TestGenerate(t *testing.T) {
	type args struct {
		directory string
		items     []string
	}

	tests := []struct {
		name      string
		args      args
		wantW     string
		assertion assert.ErrorAssertionFunc
	}{
		{
			name: "only ignore",
			args: args{
				directory: "testdata",
				items:     []string{"ELM"},
			},
			wantW:     "only-ignore.golden",
			assertion: assert.NoError,
		},
		{
			name: "only ignore (duplicated items)",
			args: args{
				directory: "testdata",
				items:     []string{"ELM", "elm"},
			},
			wantW:     "only-ignore.golden",
			assertion: assert.NoError,
		},
		{
			name: "with patch",
			args: args{
				directory: "testdata",
				items:     []string{"go", "elm"},
			},
			wantW:     "with-patch.golden",
			assertion: assert.NoError,
		},
		{
			name: "with stack",
			args: args{
				directory: "testdata",
				items:     []string{"lamP"},
			},
			wantW:     "with-stack.golden",
			assertion: assert.NoError,
		},
		{
			name: "with duplicated lines",
			args: args{
				directory: "testdata",
				items:     []string{"go", "c"},
			},
			wantW:     "with-duplicated-lines.golden",
			assertion: assert.NoError,
		},
		{
			name: "with undefined",
			args: args{
				directory: "testdata",
				items:     []string{"go", "go++"},
			},
			wantW:     "with-undefined.golden",
			assertion: assert.Error,
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			w := &bytes.Buffer{}
			err := file.Generate(w, `testdata`, tt.args.items...)
			tt.assertion(t, err)

			goldenPath := filepath.Join(`_golden`, tt.wantW)

			if *update {
				require.NoError(t, ioutil.WriteFile(goldenPath, w.Bytes(), 0600))
			}

			expected, err := ioutil.ReadFile(goldenPath)
			require.NoError(t, err)
			assert.Equal(t, expected, w.Bytes())
		})
	}
}

func TestGenerate_UnknownDirectory(t *testing.T) {
	w := &bytes.Buffer{}
	assert.Error(t, file.Generate(w, `unknown`))
}
