// Copyright © 2022 The Tekton Authors.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package chain

import (
	"fmt"

	"github.com/spf13/cobra"
	"github.com/tektoncd/chains/pkg/chains"
	"github.com/tektoncd/cli/pkg/chain"
	"github.com/tektoncd/cli/pkg/cli"
	"github.com/tektoncd/cli/pkg/taskrun"
	"github.com/tektoncd/pipeline/pkg/apis/pipeline/v1beta1"
	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"
	cliopts "k8s.io/cli-runtime/pkg/genericclioptions"
)

type PayloadOptions struct {
	SkipVerify bool
}

func payloadCommand(p cli.Params) *cobra.Command {
	opts := &PayloadOptions{}
	f := cliopts.NewPrintFlags("chains")

	c := &cobra.Command{
		Use:   "payload",
		Short: "Print Tekton Chains' payload for a specific taskrun",
		Annotations: map[string]string{
			"commandType": "main",
		},
		Args: cobra.ExactArgs(1),
		RunE: func(cmd *cobra.Command, args []string) error {
			skipVerify, _ := cmd.LocalFlags().GetBool("skip-verify")

			// Get the task name.
			taskName := args[0]

			chainsNamespace, err := cmd.Flags().GetString("chains-namespace")
			if err != nil {
				return fmt.Errorf("error: output option not set properly: %v", err)
			}

			// Get the Tekton clients.
			cs, err := p.Clients()
			if err != nil {
				return fmt.Errorf("failed to create tekton client")
			}

			// Retrieve the taskrun.
			tr, err := taskrun.Get(cs, taskName, metav1.GetOptions{}, p.Namespace())
			if err != nil {
				return fmt.Errorf("failed to get TaskRun %s: %v", taskName, err)
			}

			return printPayloads(cs, chainsNamespace, tr, skipVerify)
		},
	}
	f.AddFlags(c)
	c.Flags().BoolVarP(&opts.SkipVerify, "skip-verify", "S", opts.SkipVerify, "Skip verifying the payload'signature")

	return c
}

func printPayloads(cs *cli.Clients, namespace string, tr *v1beta1.TaskRun, skipVerify bool) error {
	// Get the storage backend.
	backends, opts, err := chain.GetTaskRunBackends(cs, namespace, tr)
	if err != nil {
		return fmt.Errorf("failed to retrieve the backend storage: %v", err)
	}
	for _, backend := range backends {
		// Some limitations occur when the backend is OCI.
		if backend.Type() == "oci" {
			// Cannot skip the verification since it is required when fetching the
			// artifacts from the OCI registry.
			if skipVerify {
				return fmt.Errorf("verification is mandatory when the backend is OCI - remove the the `-S` flag from the command line and try again")
			}

			// The key must be fetched from the secrets.
			opts.Key = fmt.Sprintf(x509Keypair, namespace)
		}

		// Fetch the payload.
		payloads, err := backend.RetrievePayloads(opts)
		if err != nil {
			return fmt.Errorf("error retrieving the payloads: %s", err)
		}
		if len(payloads) == 0 {
			fmt.Printf("No payloads found for taskrun %s\n", tr.Name)
			return nil
		}

		// Verify the payload signature.
		if !skipVerify && backend.Type() != "oci" {
			// Retrieve a context with the configuration.
			ctx, err := chain.ConfigMapToContext(cs, namespace)
			if err != nil {
				return err
			}
			trv := chains.TaskRunVerifier{
				KubeClient:        cs.Kube,
				Pipelineclientset: cs.Tekton,
				SecretPath:        "",
			}
			if err := trv.VerifyTaskRun(ctx, tr); err != nil {
				return err
			}
		}

		// Display the payload.
		for _, payload := range payloads {
			fmt.Println(payload)
		}
	}
	return nil
}
