package main

import (
	"fmt"
	"os"
	"text/template"

	"github.com/pelletier/go-toml"
)

// Site represents a site.
type Site struct {
	Title      string            `toml:"title"`
	URLs       []string          `toml:"urls"`
	Tasks      []*Task           `toml:"tasks"`
	Feeds      map[string]string `toml:"feeds"`
	Params     map[string]string `toml:"params"`
	Permalinks map[string]string `toml:"permalinks"`
	permalinks map[string]*template.Template
	templates  Templates
	root       *Dir
}

// Task represents a site build task.
type Task struct {
	Input       []string          `toml:"input"`       // input file suffixes
	OutputExt   string            `toml:"output"`      // output file suffix
	TemplateExt string            `toml:"template"`    // template file suffix
	Preprocess  map[string]string `toml:"preprocess"`  // preprocess commands
	Postprocess string            `toml:"postprocess"` // postprocess command
	StaticDir   string            `toml:"static_dir"`  // static file directory
	OutputDir   string            `toml:"output_dir"`  // output directory
	UglyURLs    bool              `toml:"ugly_urls"`   // whether to use ugly URLs
}

func (t *Task) Match(ext string) bool {
	for i := range t.Input {
		if t.Input[i] == ext {
			return true
		}
	}
	return false
}

// LoadSite loads the site with the given configuration file.
func LoadSite(config string) (*Site, error) {
	f, err := os.Open(config)
	if err != nil {
		return nil, err
	}
	defer f.Close()

	site := &Site{}
	if err := toml.NewDecoder(f).Decode(site); err != nil {
		return nil, err
	}

	funcs := site.funcs()

	// Parse permalinks
	site.permalinks = map[string]*template.Template{}
	for path := range site.Permalinks {
		t := template.New(fmt.Sprintf("permalink %q", path)).Funcs(funcs)
		_, err := t.Parse(site.Permalinks[path])
		if err != nil {
			return nil, err
		}
		site.permalinks[path] = t
	}

	// Load templates
	templateExts := []string{}
	for _, task := range site.Tasks {
		if task.TemplateExt != "" {
			templateExts = append(templateExts, task.TemplateExt)
		}
	}
	site.templates.Funcs(funcs)
	if err := site.templates.Load("templates", templateExts); err != nil {
		return nil, err
	}

	return site, nil
}

func (s *Site) dir(path string) *Dir {
	return s.root.getDir(path)
}

func (s *Site) page(path string) *Page {
	return s.root.getPage(path)
}
