package main

import (
	"embed"
	"flag"
	"fmt"
	"io"
	"io/fs"
	"log"
	"os"
	"path"
	"path/filepath"
	"strings"
)

var (
	Version string
)

func main() {
	if len(os.Args) >= 2 {
		switch os.Args[1] {
		case "build":
			build()
			return
		case "new":
			if len(os.Args) < 3 {
				fmt.Println("usage: kiln new <path>")
				os.Exit(1)
			}
			newSite(os.Args[2])
			return
		case "version":
			fmt.Println("kiln", Version)
			return
		}
	}

	fmt.Println("usage: kiln <build | new | version> args...")
	os.Exit(1)
}

func build() {
	var (
		config string
	)

	flags := flag.NewFlagSet("kiln build", flag.ExitOnError)
	flags.StringVar(&config, "c", "config.toml", "the configuration file to use")
	flags.Parse(os.Args[2:])

	site, err := LoadSite(config)
	if err != nil {
		log.Fatal(err)
	}

	if err := site.run(); err != nil {
		log.Fatal(err)
	}
}

func (site *Site) run() error {
	for _, task := range site.Tasks {
		err := site.runTask(task)
		if err != nil {
			return err
		}
	}
	return nil
}

func (s *Site) runTask(task *Task) error {
	// Read content
	s.root = &Dir{Permalink: "/"}
	if err := s.root.read("content", task, s); err != nil {
		return err
	}
	s.root.sort()
	// Process content
	if err := s.root.process(s, task); err != nil {
		return err
	}
	// Write content
	if err := s.root.write(task.OutputDir, task); err != nil {
		return err
	}
	// Copy static files
	if task.StaticDir != "" {
		err := copyAll(task.StaticDir, task.OutputDir)
		if err != nil {
			return err
		}
	}
	return nil
}

func copyAll(srcDir, dstDir string) error {
	return filepath.Walk(srcDir, func(path string, info fs.FileInfo, err error) error {
		if info.IsDir() {
			// Do nothing
			return nil
		}

		src, err := os.Open(path)
		if err != nil {
			return err
		}
		defer src.Close()

		dstPath := filepath.Join(dstDir, strings.TrimPrefix(path, srcDir))
		os.MkdirAll(filepath.Dir(dstPath), 0755)
		dst, err := os.Create(dstPath)
		if err != nil {
			return err
		}
		defer dst.Close()

		if _, err := io.Copy(dst, src); err != nil {
			return err
		}
		return nil
	})
}

//go:embed templates/_default config.toml
var builtin embed.FS

func newSite(name string) {
	name = path.Clean(name)
	os.Mkdir(name, 0755)
	os.Mkdir(path.Join(name, "content"), 0755)
	os.Mkdir(path.Join(name, "templates"), 0755)
	os.Mkdir(path.Join(name, "templates/_default"), 0755)
	os.Mkdir(path.Join(name, "static"), 0755)
	os.Mkdir(path.Join(name, "public"), 0755)

	config, _ := builtin.ReadFile("config.toml")
	os.WriteFile(path.Join(name, "config.toml"), config, 0644)

	index := []byte("---\ntitle: Hello, world!\n---\n")
	os.WriteFile(path.Join(name, "content/_index.gmi"), index, 0644)

	templates := []string{"atom.xml", "index.gmi", "page.gmi"}
	for _, template := range templates {
		b, _ := builtin.ReadFile(path.Join("templates/_default", template))
		os.WriteFile(path.Join(name, "templates/_default", template), b, 0644)
	}
}
