/*
 * G-ToDo Gkrellm Plugin
 * Copyright 2003-2004 Dimitar Haralanov
 *
 */
/* This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <support.h>
#include <config.h>
#include <time.h>
#include <stdlib.h>
#include <stdarg.h>
#include <debug.h>

/* Get the time specified by the stamp parameter and fill out
 * the gtodo_time_struct */
void gtodo_get_date_from_time (time_t stamp, gtodo_time_struct_t *st) {
     time_t t = (stamp?stamp:time (NULL));
     struct tm *ltm = localtime (&t);

     DPRINTF ("stamp = %d", (int)stamp);
     
     if (stamp < 0) printf ("G-Todo: %s (): stamp negative\n", __FUNCTION__);
     st->month = ltm->tm_mon + 1;
     st->day = ltm->tm_mday;
     st->year = 1900 + ltm->tm_year;
     if ((global_config.time_format == REGULAR) &&
	 ltm->tm_hour > 12) {
	  st->hour = ltm->tm_hour - 12;
	  st->ampm = 1;
     } else
	  st->hour = ltm->tm_hour;
     st->min = ltm->tm_min;
     st->sec = ltm->tm_sec;

     DPRINTF ("st={month=%d,day=%d,year=%d,hour=%d,min=%d,sec=%d}",
	      st->month, st->day, st->year, st->hour, st->min, st->sec);
}

/* Covert the time from the gtodo_time_struct
 * into seconds since the Epoch */
time_t gtodo_get_time_from_date (gtodo_time_struct_t st) {
     struct tm ltm;
     time_t tm;
     ltm.tm_mon = st.month - 1;
     ltm.tm_mday = st.day;
     ltm.tm_year = st.year - 1900;
     ltm.tm_hour = (st.ampm?st.hour+12:st.hour);
     ltm.tm_min = st.min;
     ltm.tm_sec = st.sec;
     tm = mktime (&ltm);
     return tm;
}

/* Get the seconds since the Epoch from the formatted
 * string str. The string is in the format MM/DD/YYYY HH:MM */
time_t gtodo_get_time_from_date_str (gchar *str) {
     gtodo_time_struct_t st = { 0,0,0,0,0,0,0 };
     int m, d, y, h, mm;
     gchar p[3];
     int ret;
     if (global_config.time_format == REGULAR)
	  ret = sscanf (str, "%d/%d/%d %d:%d%s", &m, &d, &y, &h, &mm, p);
     else
	  ret = sscanf (str, "%d/%d/%d %d:%d", &m, &d, &y, &h, &mm);
     DPRINTF ("scanned %d items", ret);
     DPRINTF ("m = %d, d = %d, y = %d, h = %d, mm = %d", m,d,y,h,mm);
     if (ret < 5)
	  return -1;

     if (global_config.time_format == REGULAR && !strcmp (p, "PM")) 
	  h = (h+12==24?0:h+12);

     st.month = m;
     st.day = d;
     st.year = y;
     st.hour = h;
     st.min = mm;
     /* we don't need the seconds */
     /* st.sec = s; */
     
     return gtodo_get_time_from_date (st);
}

/* Covert the time in time to string. The string is in the format
 * MM/DD/YYYY HH:MM */
gchar *gtodo_get_date_from_time_str (time_t time, gchar *str) {
     if (time < 0) printf ("G-Todo: %s (): stamp negative\n", __FUNCTION__);     
     if (!str || strlen (str) < 19) str = calloc (19, sizeof (char));
     if (global_config.time_format == REGULAR) {
	  if (global_config.date_format == EUROPEAN)
	       strftime (str, 19, "%d/%m/%Y %I:%M%p", localtime (&time));
	  else
	       strftime (str, 19, "%m/%d/%Y %I:%M%p", localtime (&time));
     } else {
	  if (global_config.date_format == EUROPEAN)
	       strftime (str, 19, "%d/%m/%Y %H:%M", localtime (&time));
	  else
	       strftime (str, 19, "%m/%d/%Y %H:%M", localtime (&time));
     }
     return str;
}

void display_date_column (GtkTreeViewColumn *column, GtkCellRenderer *renderer,
			  GtkTreeModel *model, GtkTreeIter *iter, gpointer data) {
     char *str;
     glong value;
     if (!strcmp (gtk_tree_view_column_get_title (column), _("Deadline")))
	  gtk_tree_model_get (model, iter, DEADLINE_COLUMN, &value, -1);
     else
	  gtk_tree_model_get (model, iter, REMINDER_COLUMN, &value, -1);

     if (value != 0)
	  str = gtodo_get_date_from_time_str (labs (value), NULL);
     else
	  str = g_strdup (_("Acknowledged"));
     DPRINTF ("date column: '%s'", str);
     g_object_set (renderer, "text", str, NULL);
     g_free (str);
}

#define COMMENT_LENGTH 25

void display_comment_column (GtkTreeViewColumn *column, GtkCellRenderer *renderer,
			     GtkTreeModel *model, GtkTreeIter *iter, gpointer data) {
     gchar *value;
     gint counter;
     
     gtk_tree_model_get (model, iter, COMMENT_COLUMN, &value, -1);
     if (value && strlen (value) > COMMENT_LENGTH) {
	  gchar *rest = value + COMMENT_LENGTH + 2;
	  value[COMMENT_LENGTH + 1] = '\0';
	  value[COMMENT_LENGTH - 2] = value[COMMENT_LENGTH - 1] = value[COMMENT_LENGTH] = '.';
	  free (rest);
     }
     /* check if the line contains any newline characters
      * and replace them with spaces */
     for (counter=0; counter<COMMENT_LENGTH; counter++)
	  if (value && value[counter] == '\n') value[counter] = ' ';
     
     g_object_set (renderer, "text", value, NULL);
     g_free (value);
}

void display_time_column (GtkTreeViewColumn *column, GtkCellRenderer *renderer,
			  GtkTreeModel *model, GtkTreeIter *iter, gpointer data) {
     gchar *str;
     int sec, min, hour, day;
     glong value;
     gtk_tree_model_get (model, iter, TIME_COLUMN, &value, -1);
     sec = value % 60;
     min = (value>=60)?(value / 60)%60:0;
     hour = (value>=3600)?(value / 3600)%24:0;
     day = (value>=86400)?(value/86400):0;
     str = g_strdup_printf ("%3dd %2dh %02dm %02ds", day, hour, min, sec);
     g_object_set (renderer, "text", str, NULL);
     g_free (str);
}

void display_complete_column (GtkTreeViewColumn *column, GtkCellRenderer *renderer,
			      GtkTreeModel *model, GtkTreeIter *iter, gpointer data) {
     gchar *str;
     glong value;
     gtk_tree_model_get (model, iter, COMPLETE_COLUMN, &value, -1);
     str = g_strdup_printf ("%ld%%", value);
     g_object_set (renderer, "text", str, NULL);
     g_free (str);
}

gint gtodo_list_store_sort (GtkTreeModel *model, GtkTreeIter *a, GtkTreeIter *b, gpointer data) {
     gint sort_id = GPOINTER_TO_INT (data);
     gchar *task_a, *task_b;
     glong deadline_a, deadline_b;
     gint value_a, value_b;
     int ret = 0;
     DPRINTF ("sort id %d...", GPOINTER_TO_INT (data));

     switch (sort_id) {
     case SORT_ID_TASK:
	  gtk_tree_model_get (model, a, TITLE_COLUMN, &task_a, -1);
	  gtk_tree_model_get (model, b, TITLE_COLUMN, &task_b, -1);
	  if (task_a == NULL || task_b == NULL) {
	       if (task_a == NULL && task_b == NULL) break;
	       ret = (task_a == NULL) ? -1 : 1;
	  } else
	       ret = g_utf8_collate (task_a, task_b);
	  g_free (task_a);
	  g_free (task_b);
	  break;
     case SORT_ID_COMPLETE:
	  gtk_tree_model_get (model, a, COMPLETE_COLUMN, &value_a, -1);
	  gtk_tree_model_get (model, b, COMPLETE_COLUMN, &value_b, -1);
	  if (value_a == value_b) break;
	  ret = (value_a < value_b) ? -1 : 1;
	  break;
     case SORT_ID_DEADLINE:
	  gtk_tree_model_get (model, a, DEADLINE_COLUMN, &deadline_a, -1);
	  gtk_tree_model_get (model, b, DEADLINE_COLUMN, &deadline_b, -1);
	  if (deadline_a == deadline_b) break;
	  ret = (deadline_a < deadline_b) ? -1 : 1;
	  break;
     case SORT_ID_TIME:
	  gtk_tree_model_get (model, a, TIME_COLUMN, &value_a, -1);
	  gtk_tree_model_get (model, b, TIME_COLUMN, &value_b, -1);
	  if (value_a == value_b) break;
	  ret = (value_a < value_b) ? -1 : 1;
	  break;
     default:
	  DPRINTF ("invalid sort id %d", sort_id);
	  break;
     }
     return ret;
}

/* create and display a simple top-level window that contains
 * a message */
void alert (GtkWidget *window, const char *fmt, ...) {
     GtkWidget *dialog;
     GtkWidget *label;
     va_list args;
     gchar *str;

     /* get and format all the parameters */
     va_start (args, fmt);
     str = g_strdup_vprintf (fmt, args);
     va_end (args);
     DPRINTF ("str is '%s'", str);
     DPRINTF ("window is %p", window);
     
     dialog = gtk_dialog_new_with_buttons (_("ALERT!"), GTK_WINDOW (window),
					   GTK_DIALOG_MODAL|GTK_DIALOG_DESTROY_WITH_PARENT,
					   GTK_STOCK_OK, GTK_RESPONSE_NONE, NULL);
     g_signal_connect_swapped (GTK_OBJECT (dialog), 
			       "response", 
			       G_CALLBACK (gtk_widget_destroy),
			       GTK_OBJECT (dialog));
     
     /* Add the label, and show everything we've added to the dialog. */
     label = gtk_label_new (str);
     gtk_misc_set_padding (GTK_MISC (label), 5, 5);
     gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), label);
     gtk_widget_show_all (dialog);
     g_free (str);
}

