/*
 * G-ToDo Gkrellm Plugin
 * Copyright 2003-2004 Dimitar Haralanov
 *
 */
/* This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <config_tab.h>
#include <support.h>
#include <config.h>
#include <debug.h>

static GtkWidget *gtodo_create_task_page                       (void);
static GtkWidget *gtodo_create_option_page                     (void);
static GtkWidget *gtodo_create_help_page                       (void);
static GtkWidget *gtodo_create_about_page                      (void);

/* Task Page callback */
static void       on_new_entry_am_pm_combo_changed             (GtkEditable *, gpointer);
static void       on_add_task_button_clicked                   (GtkButton *, gpointer);
static void       on_update_task_button_clicked                (GtkButton *, gpointer);
static void       on_remove_task_button_clicked                (GtkButton *, gpointer);
static void       on_task_list_tree_view_selection             (GtkTreeSelection *, gpointer);
static void       gtodo_task_page_clear_fields                 (void);

/* Options Page callbacks */
static void       on_settings_delete_checkbox_toggled          (GtkToggleButton *, gpointer);
static void       on_if_acked_checkbox_toggled                 (GtkToggleButton *, gpointer);
static void       on_settings_remind_complete_checkbox_toggled (GtkToggleButton *, gpointer);
static void       on_settings_remind_old_checkbox_toggled      (GtkToggleButton *, gpointer);
static void       on_settings_time_24_radiobutton_toggled      (GtkToggleButton *, gpointer);
static void       on_settings_date_am_radiobutton_toggled      (GtkToggleButton *, gpointer);
static void       on_settings_date_eu_radiobutton_toggled      (GtkToggleButton *, gpointer);
static void       on_settings_time_12_radiobutton_toggled      (GtkToggleButton *, gpointer);
static void       on_start_reminder_spinbutton_changed         (GtkEditable *, gpointer);
static void       on_repeat_spinbutton_changed                 (GtkEditable *, gpointer);
static void       on_reminder_enable_checkbutton_toggled       (GtkToggleButton *, gpointer);
static void       on_repeated_reminder_checkbutton_toggled     (GtkToggleButton *, gpointer);
static void       on_action_flash_checkbox_toggled             (GtkToggleButton *, gpointer);
static void       on_action_popup_checkbox_toggled             (GtkToggleButton *, gpointer);
static void       on_action_exec_checkbox_toggled              (GtkToggleButton *, gpointer);
static void       on_action_command_entry_changed              (GtkEditable *, gpointer);
static void       on_action_popup_entry_changed                (GtkEditable *, gpointer);

/* Global Widget */
/*     top level */
static GtkWidget *notebook;

/*     task page */
static GtkWidget *new_entry_task_entry;
static GtkWidget *new_entry_comment_textview;
static GtkWidget *new_entry_calendar;
static GtkWidget *new_entry_hour_spinbutton;
static GtkWidget *new_entry_min_spinbutton;
static GtkWidget *new_entry_am_pm_combo;


/*     options page */
static GtkWidget *start_reminder_spinbutton;
static GtkWidget *repeat_spinbutton;
static GtkWidget *reminder_enable_checkbutton;
static GtkWidget *repeated_reminder_checkbutton;
static GtkWidget *action_command_entry;
static GtkWidget *action_popup_entry;

void gtodo_create_config_tab (GtkWidget *widget) {
     GtkWidget *config_label;
     GtkWidget *option_label;
     GtkWidget *help_label;
     GtkWidget *about_label;
     GtkWidget *task_page_widget;
     GtkWidget *option_page_widget;
     GtkWidget *help_page_widget;
     GtkWidget *about_page_widget;
	  
     notebook = gtk_notebook_new ();
     gtk_container_add (GTK_CONTAINER (widget), notebook);

     /* Now, I could put the entire configuration tab in the same
      * function but that would turn into an unbearable nightmare
      * if you decide to read it. This way the nightmare has been
      * ceparated into four unpleasant dreams */
     config_label = gtk_label_new (_("Tasks"));
     task_page_widget = gtodo_create_task_page ();
     gtk_notebook_append_page (GTK_NOTEBOOK (notebook), task_page_widget,
			       config_label);

     option_label = gtk_label_new (_("Options"));
     option_page_widget = gtodo_create_option_page ();
     gtk_notebook_append_page (GTK_NOTEBOOK (notebook), option_page_widget,
			       option_label);

     help_label = gtk_label_new (_("Help"));
     help_page_widget = gtodo_create_help_page ();
     gtk_notebook_append_page (GTK_NOTEBOOK (notebook), help_page_widget,
			       help_label);

     about_label = gtk_label_new (_("About"));
     about_page_widget = gtodo_create_about_page ();
     gtk_notebook_append_page (GTK_NOTEBOOK (notebook), about_page_widget,
			       about_label);

     gtk_widget_show_all (notebook);
}

/* This function creates the tab for the task definitions */
static GtkWidget *gtodo_create_task_page (void) {
     GtkWidget          *task_frame;
     GtkWidget          *task_list_vbox;
     GtkWidget          *task_page_table;
     GtkWidget          *new_entry_frame;
     GtkWidget          *new_entry_task_label;
     GtkWidget          *new_entry_calendar_event_box;
     GtkWidget          *new_entry_calendar_label;
     GtkWidget          *new_entry_time_label;
     GtkObject          *new_entry_hour_spinbutton_adj;
     GtkObject          *new_entry_min_spinbutton_adj;
     GList              *new_entry_am_pm_list = NULL;
     GtkWidget          *new_entry_comment_label;
     GtkWidget          *new_entry_comment_event_box;
     GtkWidget          *new_entry_comment_scrolledwindow;
     GtkWidget          *action_hbutton_box;
     GtkWidget          *add_task_button;
     GtkWidget          *update_task_button;
     GtkWidget          *remove_task_button;
     GtkWidget          *task_list_scrolledwindow;
     GtkWidget          *task_list_tree_view;
     GtkCellRenderer    *renderer;
     GtkTreeViewColumn  *column;
     GtkTreeSortable    *sortmodel;
     GtkTooltips        *tips;
     gtodo_time_struct_t now;

     tips = gtk_tooltips_new ();
     /* this is the page frame which will hold the entire page */
     task_frame = gtk_frame_new (_("Tasks"));
     /* top level vbox */
     task_list_vbox = gtk_vbox_new (FALSE, 3);
     gtk_container_set_border_width (GTK_CONTAINER (task_list_vbox), 3);

     /* this is a frame that holds only the widgets that are used
      * for entering new tasks */
     new_entry_frame = gtk_frame_new (_("New Entry"));
     gtk_box_pack_start (GTK_BOX (task_list_vbox), new_entry_frame, FALSE, TRUE, 0);

     /* setup the new task entry table */
     task_page_table = gtk_table_new (7, 6, FALSE);
     gtk_container_add (GTK_CONTAINER (new_entry_frame), task_page_table);
     gtk_container_set_border_width (GTK_CONTAINER (task_page_table), 3);
     gtk_table_set_row_spacings (GTK_TABLE (task_page_table), 5);
     gtk_table_set_col_spacings (GTK_TABLE (task_page_table), 5);

     new_entry_task_label = gtk_label_new (_("Task"));
     gtk_table_attach (GTK_TABLE (task_page_table), new_entry_task_label, 0, 1, 0, 1,
		       (GtkAttachOptions) (GTK_FILL),
		       (GtkAttachOptions) (0), 0, 0);
     gtk_label_set_justify (GTK_LABEL (new_entry_task_label), GTK_JUSTIFY_LEFT);
     gtk_misc_set_alignment (GTK_MISC (new_entry_task_label), 0, 0.5);
     
     new_entry_task_entry = gtk_entry_new ();
     gtk_table_attach (GTK_TABLE (task_page_table), new_entry_task_entry, 1, 2, 0, 1,
		       (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		       (GtkAttachOptions) (0), 0, 0);
     gtk_tooltips_set_tip (GTK_TOOLTIPS (tips), new_entry_task_entry,
			   _("Set the task title"), NULL);

     new_entry_comment_event_box = gtk_event_box_new ();
     new_entry_comment_scrolledwindow = gtk_scrolled_window_new (NULL, NULL);
     gtk_container_add (GTK_CONTAINER (new_entry_comment_event_box), new_entry_comment_scrolledwindow);
     gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (new_entry_comment_scrolledwindow),
				     GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
     gtk_table_attach (GTK_TABLE (task_page_table), new_entry_comment_event_box, 1, 2, 1, 2,
		       (GtkAttachOptions) (GTK_FILL),
		       (GtkAttachOptions) (GTK_FILL), 0, 0);
     gtk_tooltips_set_tip (GTK_TOOLTIPS (tips), new_entry_comment_event_box,
			   _("Set the task comment. Task comments can be\n"
			     "aritrary long and are used as either better\n"
			     "task descriptions or task log, etc."), NULL);
     
     new_entry_comment_textview = gtk_text_view_new ();
     gtk_container_add (GTK_CONTAINER (new_entry_comment_scrolledwindow),
			new_entry_comment_textview);
     
     new_entry_calendar_label = gtk_label_new (_("Deadline"));
     gtk_table_attach (GTK_TABLE (task_page_table), new_entry_calendar_label, 2, 3, 0, 1,
		       (GtkAttachOptions) (0),
		       (GtkAttachOptions) (0), 0, 0);
     gtk_label_set_justify (GTK_LABEL (new_entry_calendar_label), GTK_JUSTIFY_LEFT);
     gtk_misc_set_alignment (GTK_MISC (new_entry_calendar_label), 0, 0.5);

     /* get the current time so we set the entry box correctly */
     gtodo_get_date_from_time (0, &now);
     
     new_entry_hour_spinbutton_adj = gtk_adjustment_new (now.hour, 0, 23, 1, 4, 4);
     new_entry_hour_spinbutton = gtk_spin_button_new (GTK_ADJUSTMENT (new_entry_hour_spinbutton_adj), 1, 0);
     gtk_table_attach (GTK_TABLE (task_page_table), new_entry_hour_spinbutton, 3, 4, 0, 1,
		       (GtkAttachOptions) (0),
		       (GtkAttachOptions) (0), 0, 0);
     gtk_tooltips_set_tip (GTK_TOOLTIPS (tips), new_entry_hour_spinbutton,
			   _("Set the due hour of the current task"), NULL);

     new_entry_time_label = gtk_label_new (":");
     gtk_table_attach (GTK_TABLE (task_page_table), new_entry_time_label, 4, 5, 0, 1,
		       (GtkAttachOptions) (0),
		       (GtkAttachOptions) (0), 0, 0);
     
     new_entry_min_spinbutton_adj = gtk_adjustment_new (now.min, 0, 59, 1, 10, 10);
     new_entry_min_spinbutton = gtk_spin_button_new (GTK_ADJUSTMENT (new_entry_min_spinbutton_adj), 1, 0);
     gtk_table_attach (GTK_TABLE (task_page_table), new_entry_min_spinbutton, 5, 6, 0, 1,
		       (GtkAttachOptions) (0),
		       (GtkAttachOptions) (0), 0, 0);
     gtk_tooltips_set_tip (GTK_TOOLTIPS (tips), new_entry_min_spinbutton,
			   _("Set the due minute fo the current task"), NULL);

     new_entry_am_pm_list = g_list_append (new_entry_am_pm_list, "AM");
     new_entry_am_pm_list = g_list_append (new_entry_am_pm_list, "PM");
     new_entry_am_pm_combo = gtk_combo_new ();
     gtk_combo_set_popdown_strings (GTK_COMBO (new_entry_am_pm_combo), new_entry_am_pm_list);
     gtk_combo_set_value_in_list (GTK_COMBO (new_entry_am_pm_combo), TRUE, FALSE);
     gtk_entry_set_width_chars (GTK_ENTRY (GTK_COMBO (new_entry_am_pm_combo)->entry), 3);
     gtk_table_attach (GTK_TABLE (task_page_table), new_entry_am_pm_combo, 6, 7, 0, 1,
		       (GtkAttachOptions) (0),
		       (GtkAttachOptions) (0), 0, 0);
     gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (new_entry_am_pm_combo)->entry), (now.ampm?"PM":"AM"));
     gtk_widget_set_sensitive (GTK_WIDGET (new_entry_am_pm_combo), (global_config.time_format==REGULAR?TRUE:FALSE));

     new_entry_calendar_event_box = gtk_event_box_new ();
     new_entry_calendar = gtk_calendar_new ();
     gtk_container_add (GTK_CONTAINER (new_entry_calendar_event_box), new_entry_calendar);
     gtk_table_attach (GTK_TABLE (task_page_table), new_entry_calendar_event_box, 2, 7, 1, 3,
		       (GtkAttachOptions) (0),
		       (GtkAttachOptions) (0), 0, 0);
     gtk_calendar_display_options (GTK_CALENDAR (new_entry_calendar),
				   GTK_CALENDAR_SHOW_HEADING
				   | GTK_CALENDAR_SHOW_DAY_NAMES);
     gtk_tooltips_set_tip (GTK_TOOLTIPS (tips), new_entry_calendar_event_box,
			   _("Set the due date of the current task"), NULL);

     action_hbutton_box = gtk_hbutton_box_new ();
     gtk_hbutton_box_set_layout_default (GTK_BUTTONBOX_END);
     gtk_box_set_spacing (GTK_BOX (action_hbutton_box), 5);
     gtk_table_attach (GTK_TABLE (task_page_table), action_hbutton_box, 0, 2, 2, 3,
		       (GtkAttachOptions) (GTK_FILL),
		       (GtkAttachOptions) (GTK_FILL), 0, 0);
     
     add_task_button = gtk_button_new_with_mnemonic (_("Add Task"));
     gtk_container_add (GTK_CONTAINER (action_hbutton_box), add_task_button);
     
     update_task_button = gtk_button_new_with_mnemonic (_("Update Task"));
     gtk_container_add (GTK_CONTAINER (action_hbutton_box), update_task_button);
     
     remove_task_button = gtk_button_new_with_mnemonic (_("Remove Task"));
     gtk_container_add (GTK_CONTAINER (action_hbutton_box), remove_task_button);
     
     new_entry_comment_label = gtk_label_new (_("Comment"));
     gtk_table_attach (GTK_TABLE (task_page_table), new_entry_comment_label, 0, 1, 1, 2,
		       (GtkAttachOptions) (GTK_FILL),
		       (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), 0, 0);
     gtk_label_set_justify (GTK_LABEL (new_entry_comment_label), GTK_JUSTIFY_LEFT);
     gtk_misc_set_alignment (GTK_MISC (new_entry_comment_label), 0, 0);
     
     task_list_scrolledwindow = gtk_scrolled_window_new (NULL, NULL);
     gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (task_list_scrolledwindow),
				     GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

     /* setup to sorting stuff... */
     sortmodel = GTK_TREE_SORTABLE (task_list_store);
     gtk_tree_sortable_set_sort_func (GTK_TREE_SORTABLE (sortmodel), SORT_ID_TASK,
				      gtodo_list_store_sort, GINT_TO_POINTER (SORT_ID_TASK),
				      NULL);
     gtk_tree_sortable_set_sort_func (GTK_TREE_SORTABLE (sortmodel), SORT_ID_COMPLETE,
				      gtodo_list_store_sort, GINT_TO_POINTER (SORT_ID_COMPLETE),
				      NULL);
     gtk_tree_sortable_set_sort_func (GTK_TREE_SORTABLE (sortmodel), SORT_ID_DEADLINE,
				      gtodo_list_store_sort, GINT_TO_POINTER (SORT_ID_DEADLINE),
				      NULL);
     gtk_tree_sortable_set_sort_func (GTK_TREE_SORTABLE (sortmodel), SORT_ID_TIME,
				      gtodo_list_store_sort, GINT_TO_POINTER (SORT_ID_TIME),
				      NULL);
     
     task_list_tree_view = gtk_tree_view_new_with_model ((GtkTreeModel *)task_list_store);
     gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (task_list_tree_view), TRUE);
     gtk_tree_view_set_headers_clickable (GTK_TREE_VIEW (task_list_tree_view), TRUE);
     
     renderer = gtk_cell_renderer_text_new ();
     gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (task_list_tree_view), -1, _("Task"),
						  renderer, "text", TITLE_COLUMN, NULL);
     renderer = gtk_cell_renderer_text_new ();
     gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (task_list_tree_view), -1, _("Complete"),
						  renderer, "text", COMPLETE_COLUMN, NULL);
     renderer = gtk_cell_renderer_text_new ();
     gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (task_list_tree_view), -1, _("Deadline"),
						  renderer, "text", DEADLINE_COLUMN, NULL);
     column = gtk_tree_view_get_column (GTK_TREE_VIEW (task_list_tree_view), 2);
     gtk_tree_view_column_set_cell_data_func (column, renderer, display_date_column, NULL, NULL);
     
     renderer = gtk_cell_renderer_text_new ();
     gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (task_list_tree_view), -1, _("Time Spent"),
						  renderer, "text", TIME_COLUMN, NULL);
     column = gtk_tree_view_get_column (GTK_TREE_VIEW (task_list_tree_view), 3);
     gtk_tree_view_column_set_cell_data_func (column, renderer, display_time_column, NULL, NULL);

     renderer = gtk_cell_renderer_text_new ();
     gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (task_list_tree_view), -1, _("Comments"),
						  renderer, "text", COMMENT_COLUMN, NULL);
     column = gtk_tree_view_get_column (GTK_TREE_VIEW (task_list_tree_view), 4);
     gtk_tree_view_column_set_cell_data_func (column, renderer, display_comment_column, NULL, NULL);

     /* setup column sort ids */
     column = gtk_tree_view_get_column (GTK_TREE_VIEW (task_list_tree_view), 0);
     gtk_tree_view_column_set_sort_column_id (column, SORT_ID_TASK);
     column = gtk_tree_view_get_column (GTK_TREE_VIEW (task_list_tree_view), 1);
     gtk_tree_view_column_set_sort_column_id (column, SORT_ID_COMPLETE);
     column = gtk_tree_view_get_column (GTK_TREE_VIEW (task_list_tree_view), 2);
     gtk_tree_view_column_set_sort_column_id (column, SORT_ID_DEADLINE);
     column = gtk_tree_view_get_column (GTK_TREE_VIEW (task_list_tree_view), 3);
     gtk_tree_view_column_set_sort_column_id (column, SORT_ID_TIME);
     
     gtk_container_add (GTK_CONTAINER (task_list_scrolledwindow), task_list_tree_view);
     gtk_box_pack_start (GTK_BOX (task_list_vbox), task_list_scrolledwindow, TRUE, TRUE, 3);

     gtk_container_add (GTK_CONTAINER (task_frame), task_list_vbox);

     g_signal_connect (G_OBJECT (gtk_tree_view_get_selection ((GtkTreeView *)task_list_tree_view)),
		       "changed", G_CALLBACK (on_task_list_tree_view_selection),
		       sortmodel);
     g_signal_connect (G_OBJECT (GTK_COMBO (new_entry_am_pm_combo)->entry), "changed",
		       G_CALLBACK (on_new_entry_am_pm_combo_changed), NULL);
     g_signal_connect (G_OBJECT (add_task_button), "clicked",
		       G_CALLBACK (on_add_task_button_clicked), NULL);
     g_signal_connect (G_OBJECT (update_task_button), "clicked",
		       G_CALLBACK (on_update_task_button_clicked),
		       gtk_tree_view_get_selection ((GtkTreeView *)task_list_tree_view));
     g_signal_connect (G_OBJECT (remove_task_button), "clicked",
		       G_CALLBACK (on_remove_task_button_clicked),
		       gtk_tree_view_get_selection ((GtkTreeView *)task_list_tree_view));

     return task_frame;
}

static GtkWidget *gtodo_create_option_page (void) {
     GtkWidget   *option_frame;
     GtkWidget   *option_vbox;
     GtkWidget   *settings_frame;
     GtkWidget   *settings_table;
     GtkWidget   *settings_delete_checkbox;
     GtkWidget   *if_acked_checkbox;
     GtkWidget   *settings_remind_complete_checkbox;
     GtkWidget   *settings_remind_old_checkbox;
     GtkWidget   *settings_time_label;
     GtkWidget   *settings_time_24_radiobutton;
     GSList      *time_group = NULL;
     GtkWidget   *settings_date_am_radiobutton;
     GSList      *date_group = NULL;
     GtkWidget   *settings_date_eu_radiobutton;
     GtkWidget   *settings_date_label;
     GtkWidget   *settings_time_12_radiobutton;
     GtkWidget   *reminder_frame;
     GtkWidget   *reminder_table;
     GtkWidget   *start_before_label;
     GtkWidget   *repeat_label;
     GtkObject   *start_reminder_spinbutton_adj;
     GtkObject   *repeat_spinbutton_adj;
     GtkWidget   *action_frame;
     GtkWidget   *action_table;
     GtkWidget   *action_flash_checkbox;
     GtkWidget   *action_popup_checkbox;
     GtkWidget   *action_exec_checkbox;
     GtkWidget   *action_command_label;
     GtkWidget   *action_popup_label;
     GtkTooltips *tips;

     tips = gtk_tooltips_new ();
     option_frame = gtk_frame_new (_("Options"));
     
     option_vbox = gtk_vbox_new (FALSE, 3);
     gtk_container_add (GTK_CONTAINER (option_frame), option_vbox);
     gtk_container_set_border_width (GTK_CONTAINER (option_vbox), 3);
     
     settings_frame = gtk_frame_new (_("Settings"));
     gtk_box_pack_start (GTK_BOX (option_vbox), settings_frame, TRUE, TRUE, 0);
     
     settings_table = gtk_table_new (3, 4, FALSE);
     gtk_container_add (GTK_CONTAINER (settings_frame), settings_table);
     gtk_container_set_border_width (GTK_CONTAINER (settings_table), 3);
     gtk_table_set_row_spacings (GTK_TABLE (settings_table), 3);
     gtk_table_set_col_spacings (GTK_TABLE (settings_table), 3);
     
     settings_time_label = gtk_label_new (_("Time Format"));
     gtk_table_attach (GTK_TABLE (settings_table), settings_time_label, 0, 1, 0, 1,
		       (GtkAttachOptions) (GTK_FILL),
		       (GtkAttachOptions) (0), 0, 0);
     gtk_label_set_justify (GTK_LABEL (settings_time_label), GTK_JUSTIFY_LEFT);
     gtk_misc_set_alignment (GTK_MISC (settings_time_label), 0, 0.5);
     gtk_tooltips_set_tip (GTK_TOOLTIPS (tips), settings_time_label,
			   _("Choose on how time is displayed through out the plugin"), NULL);
     
     settings_time_24_radiobutton = gtk_radio_button_new_with_mnemonic (NULL, _("24-Hour"));
     gtk_table_attach (GTK_TABLE (settings_table), settings_time_24_radiobutton, 2, 3, 0, 1,
		       (GtkAttachOptions) (GTK_FILL),
		       (GtkAttachOptions) (0), 0, 0);
     gtk_radio_button_set_group (GTK_RADIO_BUTTON (settings_time_24_radiobutton), time_group);
     time_group = gtk_radio_button_get_group (GTK_RADIO_BUTTON (settings_time_24_radiobutton));
     gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (settings_time_24_radiobutton),
				   (global_config.time_format==MILITARY?TRUE:FALSE));
     gtk_tooltips_set_tip (GTK_TOOLTIPS (tips), settings_time_24_radiobutton,
			   _("Set display of time to 24-hour format"), NULL);
     
     settings_time_12_radiobutton = gtk_radio_button_new_with_mnemonic (NULL, _("12-Hour"));
     gtk_table_attach (GTK_TABLE (settings_table), settings_time_12_radiobutton, 1, 2, 0, 1,
		       (GtkAttachOptions) (GTK_FILL),
		       (GtkAttachOptions) (0), 0, 0);
     gtk_radio_button_set_group (GTK_RADIO_BUTTON (settings_time_12_radiobutton), time_group);
     time_group = gtk_radio_button_get_group (GTK_RADIO_BUTTON (settings_time_12_radiobutton));
     gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (settings_time_12_radiobutton),
				   (global_config.time_format==REGULAR?TRUE:FALSE));
     gtk_tooltips_set_tip (GTK_TOOLTIPS (tips), settings_time_12_radiobutton,
			   _("Set display of time to 12-hour format"), NULL);
     
     settings_date_label = gtk_label_new (_("Date Format"));
     gtk_table_attach (GTK_TABLE (settings_table), settings_date_label, 0, 1, 1, 2,
		       (GtkAttachOptions) (GTK_FILL),
		       (GtkAttachOptions) (0), 0, 0);
     gtk_label_set_justify (GTK_LABEL (settings_date_label), GTK_JUSTIFY_LEFT);
     gtk_misc_set_alignment (GTK_MISC (settings_date_label), 0, 0.5);
     gtk_tooltips_set_tip (GTK_TOOLTIPS (tips), settings_date_label,
			   _("Choose on how dates are displayed through out the plugin"), NULL);
     
     settings_date_am_radiobutton = gtk_radio_button_new_with_mnemonic (NULL, _("MM/DD/YYYY"));
     gtk_table_attach (GTK_TABLE (settings_table), settings_date_am_radiobutton, 1, 2, 1, 2,
		       (GtkAttachOptions) (GTK_FILL),
		       (GtkAttachOptions) (0), 0, 0);
     gtk_radio_button_set_group (GTK_RADIO_BUTTON (settings_date_am_radiobutton), date_group);
     date_group = gtk_radio_button_get_group (GTK_RADIO_BUTTON (settings_date_am_radiobutton));
     gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (settings_date_am_radiobutton),
				   (global_config.date_format==AMERICAN?TRUE:FALSE));
     gtk_tooltips_set_tip (GTK_TOOLTIPS (tips), settings_date_am_radiobutton,
			   _("Set display of dates to MM/DD/YYYY format"), NULL);
     
     settings_date_eu_radiobutton = gtk_radio_button_new_with_mnemonic (NULL, _("DD/MM/YYYY"));
     gtk_table_attach (GTK_TABLE (settings_table), settings_date_eu_radiobutton, 2, 3, 1, 2,
		       (GtkAttachOptions) (GTK_FILL),
		       (GtkAttachOptions) (0), 0, 0);
     gtk_radio_button_set_group (GTK_RADIO_BUTTON (settings_date_eu_radiobutton),date_group);
     date_group = gtk_radio_button_get_group (GTK_RADIO_BUTTON (settings_date_eu_radiobutton));
     gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (settings_date_eu_radiobutton),
				   (global_config.date_format==EUROPEAN?TRUE:FALSE));
     gtk_tooltips_set_tip (GTK_TOOLTIPS (tips), settings_date_eu_radiobutton,
			   _("Set display of dates to DD/MM/YYYY"), NULL);

     settings_delete_checkbox = gtk_check_button_new_with_mnemonic (_("Automatically delete events that have expired"));
     gtk_table_attach (GTK_TABLE (settings_table), settings_delete_checkbox, 0, 2, 2, 3,
		       (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		       (GtkAttachOptions) (0), 0, 0);
     gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (settings_delete_checkbox), global_config.auto_delete);
     gtk_tooltips_set_tip (GTK_TOOLTIPS (tips), settings_delete_checkbox,
			   _("If selected all tasks that have expired will\n"
			     "be removed from the list of tasks"), NULL);
     
     if_acked_checkbox = gtk_check_button_new_with_mnemonic (_("Only If Acknowledged"));
     gtk_table_attach (GTK_TABLE (settings_table), if_acked_checkbox, 2, 3, 2, 3,
		       (GtkAttachOptions) (0),
		       (GtkAttachOptions) (0), 0, 0);
     gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (if_acked_checkbox), global_config.if_acked);
     gtk_widget_set_sensitive (GTK_WIDGET (if_acked_checkbox),
			       gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (settings_delete_checkbox)));
     gtk_tooltips_set_tip (GTK_TOOLTIPS (tips), if_acked_checkbox,
			   _("If selected expired tasks will be removed only\n"
			     "if they have been acknowledged"), NULL);
     
     settings_remind_complete_checkbox = gtk_check_button_new_with_mnemonic (_("Execute reminder(s) for complete tasks"));
     gtk_table_attach (GTK_TABLE (settings_table), settings_remind_complete_checkbox, 0, 3, 3, 4,
		       (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		       (GtkAttachOptions) (0), 0, 0);
     gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (settings_remind_complete_checkbox), global_config.remind_complete);
     gtk_tooltips_set_tip (GTK_TOOLTIPS (tips), settings_remind_complete_checkbox,
			   _("If selected task reminders will be executed even\n"
			     "if the tasks due have been marked as 100% complete"), NULL);

     settings_remind_old_checkbox = gtk_check_button_new_with_mnemonic (_("Execute reminder(s) for old tasks"));
     gtk_table_attach (GTK_TABLE (settings_table), settings_remind_old_checkbox, 0, 3, 4, 5,
		       (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		       (GtkAttachOptions) (0), 0, 0);
     gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (settings_remind_old_checkbox), global_config.remind_old);
     gtk_tooltips_set_tip (GTK_TOOLTIPS (tips), settings_remind_old_checkbox,
			   _("If selected reminders will be executed for tasks\n"
			     "that have expired but have not been removed from the list"), NULL);

     reminder_frame = gtk_frame_new (_("Reminders"));
     gtk_box_pack_start (GTK_BOX (option_vbox), reminder_frame, FALSE, TRUE, 0);
     
     reminder_table = gtk_table_new (2, 3, FALSE);
     gtk_container_add (GTK_CONTAINER (reminder_frame), reminder_table);
     gtk_container_set_border_width (GTK_CONTAINER (reminder_table), 3);
     gtk_table_set_row_spacings (GTK_TABLE (reminder_table), 3);
     gtk_table_set_col_spacings (GTK_TABLE (reminder_table), 3);
     
     start_before_label = gtk_label_new (_("Start Reminders"));
     gtk_table_attach (GTK_TABLE (reminder_table), start_before_label, 1, 2, 0, 1,
		       (GtkAttachOptions) (GTK_FILL),
		       (GtkAttachOptions) (0), 0, 0);
     gtk_label_set_justify (GTK_LABEL (start_before_label), GTK_JUSTIFY_LEFT);
     gtk_misc_set_alignment (GTK_MISC (start_before_label), 0, 0.5);
     
     repeat_label = gtk_label_new (_("Every"));
     gtk_table_attach (GTK_TABLE (reminder_table), repeat_label, 1, 2, 1, 2,
		       (GtkAttachOptions) (GTK_FILL),
		       (GtkAttachOptions) (0), 0, 0);
     gtk_label_set_justify (GTK_LABEL (repeat_label), GTK_JUSTIFY_LEFT);
     gtk_misc_set_alignment (GTK_MISC (repeat_label), 0, 0.5);
     
     reminder_enable_checkbutton = gtk_check_button_new_with_mnemonic (_("Enable Reminders"));
     gtk_table_attach (GTK_TABLE (reminder_table), reminder_enable_checkbutton, 0, 1, 0, 1,
		       (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		       (GtkAttachOptions) (0), 0, 0);
     gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (reminder_enable_checkbutton),
				   global_config.enable_reminders);
     gtk_tooltips_set_tip (GTK_TOOLTIPS (tips), reminder_enable_checkbutton,
			   _("If selected task reminders will be executed when taks\n"
			     "are due. For different task reminders see the section\n"
			     "below"), NULL);
     
     start_reminder_spinbutton_adj = gtk_adjustment_new (global_config.start_interval, 1, 120, 1, 10, 10);
     start_reminder_spinbutton = gtk_spin_button_new (GTK_ADJUSTMENT (start_reminder_spinbutton_adj), 1, 0);
     gtk_table_attach (GTK_TABLE (reminder_table), start_reminder_spinbutton, 2, 3, 0, 1,
		       (GtkAttachOptions) (0),
		       (GtkAttachOptions) (0), 0, 0);
     gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (start_reminder_spinbutton), TRUE);
     gtk_tooltips_set_tip (GTK_TOOLTIPS (tips), start_reminder_spinbutton,
			   _("The number of minutes prior to the task's due time\n"
			     "that the reminder will be executed"), NULL);

     repeated_reminder_checkbutton = gtk_check_button_new_with_mnemonic (_("Repeat Reminders"));
     gtk_table_attach (GTK_TABLE (reminder_table), repeated_reminder_checkbutton, 0, 1, 1, 2,
		       (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		       (GtkAttachOptions) (0), 0, 0);
     gtk_widget_set_sensitive (GTK_WIDGET (repeated_reminder_checkbutton),
			       global_config.enable_reminders);
     gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (repeated_reminder_checkbutton),
				    global_config.repeat_reminders);
     gtk_tooltips_set_tip (GTK_TOOLTIPS (tips), repeated_reminder_checkbutton,
			   _("If selected reminders will be repeated every n minutes"), NULL);

     repeat_spinbutton_adj = gtk_adjustment_new (global_config.repeat_interval, 1, 120, 1, 10, 10);
     repeat_spinbutton = gtk_spin_button_new (GTK_ADJUSTMENT (repeat_spinbutton_adj), 1, 0);
     gtk_table_attach (GTK_TABLE (reminder_table), repeat_spinbutton, 2, 3, 1, 2,
		       (GtkAttachOptions) (0),
		       (GtkAttachOptions) (0), 0, 0);
     gtk_widget_set_sensitive (GTK_WIDGET (repeat_spinbutton), global_config.repeat_reminders);
     gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (repeat_spinbutton), TRUE);
     gtk_tooltips_set_tip (GTK_TOOLTIPS (tips), repeat_spinbutton,
			   _("The interval (in minutes) on which task reminders\n"
			     "will be repeated"), NULL);

     action_frame = gtk_frame_new (_("Reminder Actions"));
     gtk_box_pack_start (GTK_BOX (option_vbox), action_frame, FALSE, TRUE, 0);
     
     action_table = gtk_table_new (3, 3, FALSE);
     gtk_container_add (GTK_CONTAINER (action_frame), action_table);
     gtk_container_set_border_width (GTK_CONTAINER (action_table), 3);
     gtk_table_set_row_spacings (GTK_TABLE (action_table), 3);
     gtk_table_set_col_spacings (GTK_TABLE (action_table), 3);
     
     action_flash_checkbox = gtk_check_button_new_with_mnemonic (_("Flash Plugin Icon"));
     gtk_table_attach (GTK_TABLE (action_table), action_flash_checkbox, 0, 1, 0, 1,
		       (GtkAttachOptions) (GTK_FILL),
		       (GtkAttachOptions) (0), 0, 0);
     gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (action_flash_checkbox),
				   global_config.flash);
     gtk_tooltips_set_tip (GTK_TOOLTIPS (tips), action_flash_checkbox,
			   _("If selected the icon on the plugin panel will\n"
			     "start flashing when one or more tasks are due"), NULL);
     
     action_popup_checkbox = gtk_check_button_new_with_mnemonic (_("Popup Reminder"));
     gtk_table_attach (GTK_TABLE (action_table), action_popup_checkbox, 1, 2, 0, 1,
		       (GtkAttachOptions) (GTK_FILL),
		       (GtkAttachOptions) (0), 0, 0);
     gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (action_popup_checkbox),
				   global_config.popup);
     gtk_tooltips_set_tip (GTK_TOOLTIPS (tips), action_popup_checkbox,
			   _("If selected a reminder popup window will be displayed\n"
			     "for each of the tasks that is/are due. If there is text\n"
			     "in the Popup Text entry below, that text will be displayed\n"
			     "in the popup. Otherwise, the task's comment will be displayed"), NULL);

     action_exec_checkbox = gtk_check_button_new_with_mnemonic (_("Execute Command"));
     gtk_table_attach (GTK_TABLE (action_table), action_exec_checkbox, 2, 3, 0, 1,
		       (GtkAttachOptions) (GTK_FILL|GTK_EXPAND),
		       (GtkAttachOptions) (0), 0, 0);
     gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (action_exec_checkbox),
				   global_config.exec);
     gtk_tooltips_set_tip (GTK_TOOLTIPS (tips), action_exec_checkbox,
			   _("If selected a command will be executed when one or\n"
			     "more tasks are due. The command executed is the one\n"
			     "specified in the Command entry. The command will be\n"
			     "executed once per task"), NULL);

     action_command_entry = gtk_entry_new ();
     gtk_table_attach (GTK_TABLE (action_table), action_command_entry, 1, 3, 1, 2,
		       (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		       (GtkAttachOptions) (0), 0, 0);
     gtk_widget_set_sensitive (action_command_entry, global_config.exec);
     gtk_entry_set_text (GTK_ENTRY (action_command_entry), (global_config.command?global_config.command:""));
     gtk_tooltips_set_tip (GTK_TOOLTIPS (tips), action_command_entry,
			   _("The command to execute when one or more taks are due"), NULL);

     action_command_label = gtk_label_new (_("Command"));
     gtk_table_attach (GTK_TABLE (action_table), action_command_label, 0, 1, 1, 2,
		       (GtkAttachOptions) (GTK_FILL),
		       (GtkAttachOptions) (0), 0, 0);
     gtk_label_set_justify (GTK_LABEL (action_command_label), GTK_JUSTIFY_LEFT);
     gtk_misc_set_alignment (GTK_MISC (action_command_label), 0, 0.5);
     gtk_widget_set_sensitive (action_command_label, global_config.exec);
     
     action_popup_label = gtk_label_new (_("Popup Text"));
     gtk_table_attach (GTK_TABLE (action_table), action_popup_label, 0, 1, 2, 3,
		       (GtkAttachOptions) (GTK_FILL),
		       (GtkAttachOptions) (0), 0, 0);
     gtk_label_set_justify (GTK_LABEL (action_popup_label), GTK_JUSTIFY_LEFT);
     gtk_misc_set_alignment (GTK_MISC (action_popup_label), 0, 0.5);
     gtk_widget_set_sensitive (action_popup_label, global_config.popup);
     
     action_popup_entry = gtk_entry_new ();
     gtk_table_attach (GTK_TABLE (action_table), action_popup_entry, 1, 3, 2, 3,
		       (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		       (GtkAttachOptions) (0), 0, 0);
     gtk_widget_set_sensitive (action_popup_entry, global_config.popup);
     gtk_entry_set_text (GTK_ENTRY (action_popup_entry), (global_config.text?global_config.text:""));
     gtk_tooltips_set_tip (GTK_TOOLTIPS (tips), action_popup_entry,
			   _("The text to be displayed in the popup reminder window"), NULL);
	  
     g_signal_connect ((gpointer) settings_delete_checkbox, "toggled",
		       G_CALLBACK (on_settings_delete_checkbox_toggled),
		       if_acked_checkbox);
     g_signal_connect ((gpointer) if_acked_checkbox, "toggled",
		       G_CALLBACK (on_if_acked_checkbox_toggled),
		       NULL);
     g_signal_connect ((gpointer) settings_remind_complete_checkbox, "toggled",
		       G_CALLBACK (on_settings_remind_complete_checkbox_toggled),
		       NULL);
     g_signal_connect ((gpointer) settings_remind_old_checkbox, "toggled",
		       G_CALLBACK (on_settings_remind_old_checkbox_toggled),
		       NULL);
     g_signal_connect ((gpointer) settings_time_24_radiobutton, "toggled",
		       G_CALLBACK (on_settings_time_24_radiobutton_toggled),
		       NULL);
     g_signal_connect ((gpointer) settings_date_am_radiobutton, "toggled",
		       G_CALLBACK (on_settings_date_am_radiobutton_toggled),
		       NULL);
     g_signal_connect ((gpointer) settings_date_eu_radiobutton, "toggled",
		       G_CALLBACK (on_settings_date_eu_radiobutton_toggled),
		       NULL);
     g_signal_connect ((gpointer) settings_time_12_radiobutton, "toggled",
		       G_CALLBACK (on_settings_time_12_radiobutton_toggled),
		       NULL);
     g_signal_connect ((gpointer) start_reminder_spinbutton, "changed",
		       G_CALLBACK (on_start_reminder_spinbutton_changed),
		       NULL);
     g_signal_connect ((gpointer) repeat_spinbutton, "changed",
		       G_CALLBACK (on_repeat_spinbutton_changed),
		       NULL);
     g_signal_connect ((gpointer) reminder_enable_checkbutton, "toggled",
		       G_CALLBACK (on_reminder_enable_checkbutton_toggled),
		       start_before_label);
     g_signal_connect ((gpointer) repeated_reminder_checkbutton, "toggled",
		       G_CALLBACK (on_repeated_reminder_checkbutton_toggled),
		       repeat_label);
     g_signal_connect ((gpointer) action_flash_checkbox, "toggled",
		       G_CALLBACK (on_action_flash_checkbox_toggled),
		       NULL);
     g_signal_connect ((gpointer) action_popup_checkbox, "toggled",
		       G_CALLBACK (on_action_popup_checkbox_toggled),
		       action_popup_label);
     g_signal_connect ((gpointer) action_exec_checkbox, "toggled",
		       G_CALLBACK (on_action_exec_checkbox_toggled),
		       action_command_label);
     g_signal_connect ((gpointer) action_command_entry, "changed",
		       G_CALLBACK (on_action_command_entry_changed),
		       NULL);
     g_signal_connect ((gpointer) action_popup_entry, "changed",
		       G_CALLBACK (on_action_popup_entry_changed),
		       NULL);

     return option_frame;
}

static GtkWidget *gtodo_create_help_page (void) {
     GtkWidget     *help_frame;
     GtkWidget     *help_text_view;
     GtkWidget     *vbox;
     gint 	    i;
     GtkTextBuffer *buffer;
     GtkTextIter    iter;
     static gchar  *help_text[] = {
	  N_("<h>Gkrellm-ToDo Task Plugin\n"),
	  "\n",
	  N_("<b>Introduction\n"),
	  N_("G-ToDo is a GKrellm plugin that lets you organize and track tasks and to-do items.\n" \
	     "Along those lines it has the ability to track time spent on individual tasks, percentage\n" \
	     "of completion of individual tasks, and has the ability to show reminders when tasks\n" \
	     "are due.\n\n"),
	  N_("<b>User Interface\n"),
	  N_("<b>\t-- Plugin Panel\n"),
	  N_("The plugin panel contains an icon and a text decal (which doubles as a button). The\n" \
	     "icon serves as a reminder and has a couple of other functions discussed below. The\n" \
	     "text decal shows, the tasks due today and the total number of tasks defined.\n"),
	  "\n",
	  N_("<b>\t-- Quick View Window\n"),
	  N_("The Quick View Window lets you view and manage your tasks without having to\n" \
	     "open the configuration window every time. It also lets you do a few things that the\n" \
	     "configuration window does not. The Quick View window lets you perform the following\n" \
	     "taks:\n"),
	  "\n",
	  N_("\t- you can select the active task by clicking on any of the radion buttons before\n" \
	  "\t  the task in the "), N_("<i>Active"), N_(" column\n" \
	  "\t- you can change the task-complete percentage by clicking on the "), N_("<i>Complete\n"),
	  N_("\t  column of the task.\n" \
	  "\t- you can change the due date of the taks by clicking on the "), N_("<i>Deadline"), N_(" column\n" \
	  "\t  of the task\n" \
	  "\t- you can view/change the comment of the task by clicking on the "), N_("<i>View"), N_(" button\n" \
	  "\t  from the button bar on the bottom.\n" \
	  "\n" \
	  "It is worth mentioning that you can only change field for the task that is currently\n" \
	  "active. The fields for all the other tasks will be disabled\n"),
	  "\n",
	  N_("All the task listed in the Quick View window can be sorted by clicking on the title of\n"
	     "the column by which you want the sort to happen. The column which allow sorting by are\n" \
	     "the Title, Complete, Deadline, and Time Spent columns.\n"),
	  "\n",
	  N_("<b>\t-- Configuration Window\n"),
	  N_("The configuration window provides three pages that will be of significance to the average\n" \
	  "user:\n" \
	  "\n" \
	  "\t- the "), N_("<i>Tasks "), N_("page lets you define, update, and/or remove tasks from the list\n" \
	  "\t- the "), N_("<i>Options "), N_("page lets you setup how the plugin behaves. One thing to keep in\n" \
	  "\t  mind is that the reminders options are all in minutes\n" \
	  "\t- the "), N_("<i>Help "), N_("page contains this text\n" \
	  "\n" \
	  "The configuration window has tool tip for most of the entries and buttons to help you\n" 
	  "figure out what they mean\n"),
	  "\n",
	  N_("The task list of the Configuration Window is sortable in the exact same manner as the list\n" \
	     "of the Quick View Window. As in the Quick View window the same fields are sortable \n"\
	     "here.\n"),
	  "\n",
	  N_("The command line entry supports some command line formatting, which will allow\n"
	     "you to pass information about the task to the executed command. The supported\n"
	     "formatting is the following:\n"),
	  "\n",
	  N_("<b>\t$a"), N_("\t- Task active state (0/1)\n"),
	  N_("<b>\t$T"), N_("\t- Task title\n"),
	  N_("<b>\t$P"), N_("\t- Completion percentage\n"),
	  N_("<b>\t$D"), N_("\t- Task deadline\n"),
	  N_("<b>\t$t"), N_("\t- Time spent on taks\n"),
	  N_("<b>\t$C"), N_("\t- Task comment\n"),
	  "\n",
	  N_("All strings are escaped before passed to the command. The characters which are \n"
	     "escaped are '\\b', '\\f', '\\n', '\\r', '\\t', '\\' and '\"'.\n"),
	  "\n",
	  N_("<b>\t-- Reminder Window\n"),
	  N_("The reminder window pops up only when \"Popup Reminder\" is selected from the "),
	  N_("<i>Settings\n"),
	  N_("page in the Configuration Window. It offers three options of dealing with the alarm:\n"),
	  "\n",
	  "\t- ", N_("<i>Acknowledge "), N_("the alarm, in which case there will be no more alarm for this task\n" \
	  "\t  and if \"Automatically delete events that have expired\" is selected the task will be\n" \
	  "\t  removed from the list.\n"),
	  "\t- ", N_("<i>Reschedule "), N_("the alarm, in which case the Configuration Window will be opened and\n" \
	  "\t  you will be able to reschedule the due date for the task.\n"),
	  "\t- ", N_("<i>Snooze "), N_("the alarm, in which case the reminder will be silenced for the\n" \
	  "\t period specified in \"Repeat Reminders Every\" section\n"),
	  "\n",
	  N_("<b>\t-- Task View Window\n"),
	  N_("The task view window's primary purpose is to allow the user to edit the comment\n" \
	     "in a better and quicker way then using the configuration window. It also\n" \
	     "displays some task details; currently, only the title and deadline but that could\n" \
	     "be expanded later.\n"),
	  "\n",
	  N_("<b>\t-- Mouse Buttons\n"),
	  N_("The following mouse actions are defined for the plugin panel. Some of the mouse buttons\n" \
	     "have more then one action depending on where in the plugin panel the button has been\n" \
	     "clicked:\n"),
	  N_("<b>\tLeft Click "), N_("\t- left clicking over the text decal will bring up the Quick View\n" \
	  "\t\t\t\t  window\n" \
	  "\t\t\t\t- left clicking over the icon will acknowledge all active\n" \
	  "\t\t\t\t  reminders\n"),
	  N_("<b>\tMiddle Click "), N_("\t- middle click over the icon starts and stops the task timer for\n" \
	  "\t\t\t\t  the active task\n"),
	  N_("<b>\tRight Click "), N_("\t- right click over the panel brings up the configuration window\n" \
	  "\t\t\t\t- right click over the icon resets the timer for the active task\n"),
	  N_("<b>\tMouse Wheel "), N_("\t- the mouse wheel switches the currently active task. Up makes the\n" \
	  "\t\t\t\t  previous task active, down - the next.\n")
     };

     help_frame = gtk_frame_new (_("G-ToDo Help"));

     vbox = gtk_vbox_new (FALSE, 3);
     gtk_container_set_border_width (GTK_CONTAINER (vbox), 3);
     help_text_view = gkrellm_gtk_scrolled_text_view (vbox, NULL,
						      GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
     gtk_text_view_set_cursor_visible (GTK_TEXT_VIEW (help_text_view), FALSE);
     for (i = 0; i < sizeof(help_text)/sizeof(gchar *); ++i)
        gkrellm_gtk_text_view_append(help_text_view, _(help_text[i]));

     /* this section of the code is here just so we can scroll the text view
      * back to the beginning but it doesn't seem to work. from looking at the
      * code here, from the web, and from other plugins, it seems that it is
      * and gtk issue */
     buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (help_text_view));
     gtk_text_buffer_get_start_iter (buffer, &iter);
     gtk_text_view_scroll_to_iter (GTK_TEXT_VIEW (help_text_view), &iter, 0.0, FALSE, 0.0, 0.0);
     /* end scrolling code */
     gtk_container_add (GTK_CONTAINER (help_frame), vbox);
     
     return help_frame;
}

static GtkWidget *gtodo_create_about_page (void) {
     GtkWidget *about_frame;
     GtkWidget *label;
     gchar *about_text = g_strdup_printf ("GKrellM-ToDo %s\n" \
					  "GKrellM Task Management Plugin\n" \
					  "\n" \
					  "Copyright (C) 2003-2004 Dimitar Haralanov\n" \
					  "voidtrance at comcast dot net\n" \
					  "http://voidtrance.home.comcast.net\n" \
					  "\n" \
					  "Released under the GNU Public License",
					  GTODO_VERSION);
     
     about_frame = gtk_frame_new (_("About"));
     label = gtk_label_new (about_text);
     gtk_container_add (GTK_CONTAINER (about_frame), label);
     g_free (about_text);
     return about_frame;
}



/*****************************************************************************/
/*   Task Page Callback                                                      */
static void on_new_entry_am_pm_combo_changed (GtkEditable *editable, gpointer user_data) {

}

static void on_add_task_button_clicked (GtkButton *button, gpointer user_data) {
     GtkTreeIter iter;
     GtkTextBuffer *buffer;
     GtkTextIter start, end;
     gchar *title, *comment;
     glong deadline;
     gtodo_time_struct_t date = { 0,0,0,0,0,0,0 };

     title = g_strdup (gtk_entry_get_text (GTK_ENTRY (new_entry_task_entry)));
     /* make sure that we are not trying to add a task
      * without a title */
     if (!title || strlen (title) == 0) {
	  alert (gtk_widget_get_toplevel (GTK_WIDGET (button)),
		 _("You have to specify a title for the task!"));
	  return;
     }

     /* get the comment from the text buffer */
     buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW(new_entry_comment_textview));
     gtk_text_buffer_get_bounds (GTK_TEXT_BUFFER (buffer), &start, &end);
     comment = gtk_text_buffer_get_text (GTK_TEXT_BUFFER (buffer), &start, &end, FALSE);

     gtk_calendar_get_date (GTK_CALENDAR (new_entry_calendar), &(date.year), &(date.month), &(date.day));
     /* for some reason, the month that gtk_calendar_get_date returns is actually
      * the previous month, odd... */
     date.month++;
     
     date.hour = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (new_entry_hour_spinbutton));
     /* we only care to compute the correct our if the time format is
      * REGULAR. Otherwise, it doesn't matter */
     if (global_config.time_format == REGULAR) {
	  gchar *am_pm = g_strdup (gtk_entry_get_text (GTK_ENTRY (GTK_COMBO (new_entry_am_pm_combo)->entry)));
	  if (!strcmp (am_pm, "PM")) date.hour += 12;
	  g_free (am_pm);
     }
     date.min = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (new_entry_min_spinbutton));
     deadline = gtodo_get_time_from_date (date);

     /* make sure that we are defining a good deadline */
     if (deadline <= time (NULL))
	  alert (gtk_widget_get_toplevel (GTK_WIDGET (button)),
		 _("Adding task that is due in the past!\n(%s)"),
		 gtodo_get_date_from_time_str (deadline, NULL));
     else {
	  gtk_list_store_append (GTK_LIST_STORE (task_list_store), &iter);

	  DPRINTF ("adding task with:\n\tTITLE = %s\n\tDEADLINE = %ld\n\tREMINDER = %ld\n\tCOMMENT = %s",
		   title, deadline, -(deadline - global_config.start_interval*60), comment);
	  gtk_list_store_set (GTK_LIST_STORE (task_list_store), &iter,
			      ACTIVE_COLUMN, FALSE,
			      TITLE_COLUMN, title,
			      COMPLETE_COLUMN, 0,
			      DEADLINE_COLUMN, deadline,
			      REMINDER_COLUMN, -(deadline - global_config.start_interval*60),
			      TIME_COLUMN, 0,
			      COMMENT_COLUMN, comment,
			      -1);
	  global_config.ntasks++;
	  global_config.clean = FALSE;
     }
     gtodo_task_page_clear_fields ();
     g_free (title);
     g_free (comment);
}

static void on_update_task_button_clicked (GtkButton *button, gpointer user_data) {
     GtkTreeIter iter;
     GtkTextBuffer *buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW(new_entry_comment_textview));
     GtkTextIter start, end;
     gchar *title = g_strdup (gtk_entry_get_text (GTK_ENTRY (new_entry_task_entry))),
	  *comment;
     glong deadline;
     gtodo_time_struct_t date = { 0,0,0,0,0,0,0 };

     /* get the new comment
      * The comment could actually be the same but since we fill out the
      * comment window when we make the selection, it is easier this way */
     gtk_text_buffer_get_bounds (GTK_TEXT_BUFFER (buffer), &start, &end);
     comment = gtk_text_buffer_get_text (GTK_TEXT_BUFFER (buffer), &start, &end, FALSE);
     
     gtk_calendar_get_date (GTK_CALENDAR (new_entry_calendar), &(date.year), &(date.month), &(date.day));
     date.month++;
     date.hour = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (new_entry_hour_spinbutton));
     date.min = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (new_entry_min_spinbutton));

     /* make sure we got the right time according to format */
     if (global_config.time_format == REGULAR) {
	  gchar *am = g_strdup (gtk_entry_get_text (GTK_ENTRY (GTK_COMBO (new_entry_am_pm_combo)->entry)));
	  if (!strcmp (am, "PM"))
	       date.hour = (date.hour+12==24?0:date.hour+12);
     }
     deadline = gtodo_get_time_from_date (date);

     /* now that everything is ok, get the selected row and update its values */
     if (gtk_tree_selection_get_selected ((GtkTreeSelection *)user_data, NULL, &iter) == TRUE) {
	  DPRINTF ("updating task with:\n\tTITLE = %s\n\tDEADLINE = %ld\n\tREMINDER = %ld\n\tCOMMENT = %s",
		   title, deadline, -(deadline - global_config.start_interval*60), comment);
	  gtk_list_store_set (GTK_LIST_STORE (task_list_store), &iter,
			      TITLE_COLUMN, title,
			      DEADLINE_COLUMN, deadline,
			      REMINDER_COLUMN, -(deadline - global_config.start_interval*60),
			      COMMENT_COLUMN, comment,
			      -1);
     }
     global_config.clean = FALSE;
     gtodo_task_page_clear_fields ();
     g_free (title);
     g_free (comment);
}

static void on_remove_task_button_clicked (GtkButton *button, gpointer user_data) {
     GtkTreeIter iter;
     gboolean active;

     /* remove the selected task but before we do that check whether it is the
      * active one */
     if (gtk_tree_selection_get_selected ((GtkTreeSelection *)user_data,
					  NULL, &iter) == TRUE) {
	  DPRINTF ("iter (%p) = {stamp=%d,data1=%p,...}", &iter, iter.stamp, iter.user_data);
	  gtk_tree_model_get (GTK_TREE_MODEL (task_list_store), &iter, ACTIVE_COLUMN, &active, -1);
	  gtk_list_store_remove (GTK_LIST_STORE (task_list_store), &iter);
	  global_config.ntasks--;
     }
     /* if the removed task was the active one, make the next task in the list
      * the active one */
     if (active) {
	  if (gtk_tree_model_get_iter_first (GTK_TREE_MODEL (task_list_store), &iter))
	       gtk_list_store_set (GTK_LIST_STORE (task_list_store), &iter,
				   ACTIVE_COLUMN, active, -1);
     }
     if (gtk_tree_model_iter_n_children (GTK_TREE_MODEL (task_list_store), NULL) == 0)
	  global_config.have_active = FALSE;
     global_config.clean = FALSE;
     gtodo_task_page_clear_fields ();
}

static void on_task_list_tree_view_selection (GtkTreeSelection *selection, gpointer data) {
     GtkTreeIter iter;
     GtkTextBuffer *buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW(new_entry_comment_textview));
     gchar *title = NULL, *comment = NULL;
     glong deadline;
     gtodo_time_struct_t now;

     /* find the selected row, get the info from it, and fill out all the value in the
      * entries */
     if (gtk_tree_selection_get_selected (selection, NULL, &iter) == TRUE) {
	  gtk_tree_model_get ((GtkTreeModel *)data, &iter,
			      TITLE_COLUMN, &title,
			      DEADLINE_COLUMN, &deadline,
			      COMMENT_COLUMN, &comment,
			      -1);
	  gtk_entry_set_text (GTK_ENTRY (new_entry_task_entry), title);
	  gtodo_get_date_from_time (deadline, &now);
	  gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (new_entry_am_pm_combo)->entry),
			      ((global_config.time_format==REGULAR && now.ampm)?"PM":"AM"));
	  gtk_spin_button_set_value (GTK_SPIN_BUTTON (new_entry_hour_spinbutton), (float)now.hour);
	  gtk_spin_button_set_value (GTK_SPIN_BUTTON (new_entry_min_spinbutton), (float)now.min);
	  gtk_calendar_select_month (GTK_CALENDAR (new_entry_calendar), now.month-1, now.year);
	  gtk_calendar_select_day (GTK_CALENDAR (new_entry_calendar), now.day);
	  /* we don't want "(null)" to be appearing in the comment if there is none */
	  gtk_text_buffer_set_text (GTK_TEXT_BUFFER (buffer), (comment?comment:""), -1);
	  if (title) g_free (title);
	  if (comment) g_free (comment);

     }
}

static void gtodo_task_page_clear_fields (void) {
    GtkTextBuffer *buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW(new_entry_comment_textview));
    gtodo_time_struct_t now;

    gtk_entry_set_text (GTK_ENTRY (new_entry_task_entry), "");
    gtodo_get_date_from_time (0, &now);
    if (global_config.time_format == REGULAR && now.hour > 12) {
	 now.hour -= 12;
	 gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (new_entry_am_pm_combo)->entry), "PM");
    } else
	 gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (new_entry_am_pm_combo)->entry), "AM");
    gtk_spin_button_set_value (GTK_SPIN_BUTTON (new_entry_hour_spinbutton), (float)now.hour);
    gtk_spin_button_set_value (GTK_SPIN_BUTTON (new_entry_min_spinbutton), (float)now.min);
    gtk_calendar_select_month (GTK_CALENDAR (new_entry_calendar), now.month-1, now.year);
    gtk_calendar_select_day (GTK_CALENDAR (new_entry_calendar), now.day);
    gtk_text_buffer_set_text (GTK_TEXT_BUFFER (buffer), "", -1); 
}

/*****************************************************************************/
/*   Option Page Callback                                                    */

static void on_settings_delete_checkbox_toggled (GtkToggleButton *togglebutton,
						 gpointer user_data) {
     global_config.auto_delete = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (togglebutton));
     gtk_widget_set_sensitive (GTK_WIDGET (user_data), global_config.auto_delete);
}

static void on_if_acked_checkbox_toggled (GtkToggleButton *togglebutton,
					  gpointer user_data) {
     global_config.if_acked = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (togglebutton));
}

static void on_settings_remind_complete_checkbox_toggled (GtkToggleButton *togglebutton,
							  gpointer user_data) {
     global_config.clean = FALSE;
     global_config.remind_complete = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (togglebutton));
}

static void on_settings_remind_old_checkbox_toggled (GtkToggleButton *togglebutton,
							  gpointer user_data) {
     global_config.clean = FALSE;
     global_config.remind_old = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (togglebutton));
}

static void on_settings_time_24_radiobutton_toggled (GtkToggleButton *togglebutton,
						     gpointer user_data) {
     gchar *str;
     gint hour;
     gboolean status = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (togglebutton));
     if (status) {
	  str = g_strdup (gtk_entry_get_text (GTK_ENTRY (GTK_COMBO (new_entry_am_pm_combo)->entry)));
	  hour = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (new_entry_hour_spinbutton));
	  gtk_widget_set_sensitive (GTK_WIDGET (new_entry_am_pm_combo), !status);
	  global_config.time_format = MILITARY;
	  if (!strcmp (str, "PM") && hour < 12) hour += 12;
	  else if (!strcmp (str, "PM") && hour == 12) hour = 0;
	  gtk_spin_button_set_value (GTK_SPIN_BUTTON (new_entry_hour_spinbutton), (float)hour);
     }
     g_free (str);
}

static void on_settings_time_12_radiobutton_toggled (GtkToggleButton *togglebutton,
						     gpointer user_data) {
     gint hour;
     gboolean status = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (togglebutton));
     if (status) {
	  hour = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (new_entry_hour_spinbutton));
	  gtk_widget_set_sensitive (GTK_WIDGET (new_entry_am_pm_combo), status);
	  global_config.time_format = REGULAR;
	  if (hour>12) {
	       hour -= 12;
	       gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (new_entry_am_pm_combo)->entry), "PM");
	  }
	  gtk_spin_button_set_value (GTK_SPIN_BUTTON (new_entry_hour_spinbutton), (float)hour);
     }
}

static void on_settings_date_am_radiobutton_toggled (GtkToggleButton *togglebutton,
						     gpointer user_data) {
     global_config.date_format = (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (togglebutton))?
				  AMERICAN:EUROPEAN);
}

static void on_settings_date_eu_radiobutton_toggled (GtkToggleButton *togglebutton,
						     gpointer user_data) {
     global_config.date_format = (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (togglebutton))?
				  EUROPEAN:AMERICAN);
}

static void on_start_reminder_spinbutton_changed (GtkEditable *editable, gpointer user_data) {
     global_config.start_interval = atoi (gtk_editable_get_chars (GTK_EDITABLE (editable), 0, -1));
     global_config.timer_change = TRUE;
 }

static void on_repeat_spinbutton_changed (GtkEditable *editable, gpointer user_data) {
     global_config.repeat_interval = atoi (gtk_editable_get_chars (GTK_EDITABLE (editable), 0, -1));
     global_config.timer_change = TRUE;
}


static void on_reminder_enable_checkbutton_toggled (GtkToggleButton *togglebutton,
						    gpointer user_data) {
     gboolean state = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (togglebutton));

     global_config.enable_reminders = state;
     gtk_widget_set_sensitive (GTK_WIDGET (repeated_reminder_checkbutton), state);
     gtk_widget_set_sensitive (GTK_WIDGET (start_reminder_spinbutton), state);
     gtk_widget_set_sensitive (GTK_WIDGET (user_data), state);
     
}

static void on_repeated_reminder_checkbutton_toggled (GtkToggleButton *togglebutton,
						      gpointer user_data) {
     gboolean state = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (togglebutton));

     global_config.repeat_reminders = state;
     gtk_widget_set_sensitive (GTK_WIDGET (repeat_spinbutton), state);
     gtk_widget_set_sensitive (GTK_WIDGET (user_data), state);
}

static void on_action_flash_checkbox_toggled (GtkToggleButton *togglebutton,
					      gpointer user_data) {
     global_config.flash = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (togglebutton));
}

static void on_action_popup_checkbox_toggled (GtkToggleButton *togglebutton,
					      gpointer user_data) {
     gboolean state = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (togglebutton));
     
     gtk_widget_set_sensitive (GTK_WIDGET (action_popup_entry), state);
     gtk_widget_set_sensitive (GTK_WIDGET (user_data), state);
     global_config.popup = state;
}

static void on_action_exec_checkbox_toggled (GtkToggleButton *togglebutton,
					     gpointer user_data) {
     gboolean state = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (togglebutton));

     gtk_widget_set_sensitive (GTK_WIDGET (action_command_entry), state);
     gtk_widget_set_sensitive (GTK_WIDGET (user_data), state);
     global_config.exec = state;
}

static void on_action_command_entry_changed (GtkEditable *editable, gpointer user_data) {
     global_config.command = g_strdup (gtk_entry_get_text (GTK_ENTRY (editable)));
     if (global_config.command && strlen (global_config.command) == 0)
	  global_config.command = NULL;
}

static void on_action_popup_entry_changed (GtkEditable *editable, gpointer user_data) {
     global_config.text = g_strdup (gtk_entry_get_text (GTK_ENTRY (editable)));
     if (global_config.text && strlen (global_config.text) == 0)
	  global_config.text = NULL;
}
