/*
 * G-ToDo Gkrellm Plugin
 * Copyright 2003-2004 Dimitar Haralanov
 *
 */
/* This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <config.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <unistd.h>
#include <stdlib.h>
#include <gtodo.h>
#include <debug.h>

/* this function cleans the data direcotry from all
 * the extra comment files */
static void gtodo_clean_config_dir_temp_files (void);

/* Apply the config
 * Currently, the only thing that this function does is
 * recalculate the reminder times for all the tasks */
void gtodo_apply_plugin_config (void) {
     if (global_config.timer_change) {
	  gboolean valid;
	  GtkTreeIter iter;
	  glong deadline, reminder;
	  
	  /* get all the iters and cycle through the list */
	  valid = gtk_tree_model_get_iter_first (GTK_TREE_MODEL (task_list_store), &iter);
	  while (valid) {
	       gtk_tree_model_get (GTK_TREE_MODEL (task_list_store), &iter, DEADLINE_COLUMN, &deadline,
				   REMINDER_COLUMN, &reminder, -1);
	       if (reminder < 0) 
		    gtk_list_store_set (GTK_LIST_STORE (task_list_store), &iter, REMINDER_COLUMN,
					-(deadline - global_config.start_interval*60), -1);
	       valid = gtk_tree_model_iter_next (GTK_TREE_MODEL (task_list_store), &iter);
	  }
	  global_config.timer_change = FALSE;
     }
}

/* Save the pluging configuration */
void gtodo_save_plugin_config (FILE *fp) {
     fprintf (fp, "%s file %s\n", CONFIG_NAME, global_config.file);
     fprintf (fp, "%s time %d\n", CONFIG_NAME, (global_config.time_format==REGULAR?1:0));
     fprintf (fp, "%s date %d\n", CONFIG_NAME, (global_config.date_format==AMERICAN?1:0));
     fprintf (fp, "%s auto %d\n", CONFIG_NAME, global_config.auto_delete);
     fprintf (fp, "%s if_acked %d\n", CONFIG_NAME, global_config.if_acked);
     fprintf (fp, "%s remind_c %d\n", CONFIG_NAME, global_config.remind_complete);
     fprintf (fp, "%s remind_o %d\n", CONFIG_NAME, global_config.remind_old);
     fprintf (fp, "%s reminders %d\n", CONFIG_NAME, global_config.enable_reminders);
     fprintf (fp, "%s sinterval %d\n", CONFIG_NAME, global_config.start_interval);
     fprintf (fp, "%s repeater %d\n", CONFIG_NAME, global_config.repeat_reminders);
     fprintf (fp, "%s rinterval %d\n", CONFIG_NAME, global_config.repeat_interval);
     fprintf (fp, "%s flash %d\n", CONFIG_NAME, global_config.flash);
     fprintf (fp, "%s popup %d\n", CONFIG_NAME, global_config.popup);
     fprintf (fp, "%s exec %d\n", CONFIG_NAME, global_config.exec);
     fprintf (fp, "%s cmd %s\n", CONFIG_NAME, global_config.command);
     fprintf (fp, "%s text %s\n", CONFIG_NAME, global_config.text);
     fprintf (fp, "%s timer %d\n", CONFIG_NAME, global_config.timer_on);

     gtodo_save_task_list (global_config.file);
}

/* Load the plugins configuration.
 * This function is called once per line starting with the
 * plugin CONFIG_NAME */
void gtodo_load_plugin_config (gchar *arg) {
     gchar name[10], value[256];
     if (sscanf (arg, "%s %[^\n]", name, value) != 2) {
	  printf ("G-Todo: %s(): could not read arg\n", __FUNCTION__);
	  return;
     }
     if (!strcmp (name, "file")) global_config.file = g_strdup (value);
     else if (!strcmp (name, "time")) global_config.time_format = (atoi (value)==1?REGULAR:MILITARY);
     else if (!strcmp (name, "date")) global_config.date_format = (atoi (value)==1?AMERICAN:EUROPEAN);
     else if (!strcmp (name, "auto")) global_config.auto_delete = (atoi (value)==1?TRUE:FALSE);
     else if (!strcmp (name, "if_acked")) global_config.if_acked = (atoi (value)==1?TRUE:FALSE);
     else if (!strcmp (name, "remind_c")) global_config.remind_complete = (atoi (value)==1?TRUE:FALSE);
     else if (!strcmp (name, "remind_o")) global_config.remind_old = (atoi (value)==1?TRUE:FALSE);
     else if (!strcmp (name, "reminders")) global_config.enable_reminders = (atoi (value)==1?TRUE:FALSE);
     else if (!strcmp (name, "sinterval")) global_config.start_interval = atoi (value);
     else if (!strcmp (name, "repeater")) global_config.repeat_reminders = (atoi (value)==1?TRUE:FALSE);
     else if (!strcmp (name, "rinterval")) global_config.repeat_interval = atoi (value);
     else if (!strcmp (name, "flash")) global_config.flash = (atoi (value)==1?TRUE:FALSE);
     else if (!strcmp (name, "popup")) global_config.popup = (atoi (value)==1?TRUE:FALSE);
     else if (!strcmp (name, "exec")) global_config.exec = (atoi (value)==1?TRUE:FALSE);
     else if (!strcmp (name, "cmd")) global_config.command = (strcmp(value, "(null)")?g_strdup (value):NULL);
     else if (!strcmp (name, "text")) global_config.text = (strcmp(value, "(null)")?g_strdup (value):NULL);
     else if (!strcmp (name, "timer")) global_config.timer_on = (atoi (value)==1?TRUE:FALSE);
     else printf ("G-Todo: %s(): unknown option %s!\n", __FUNCTION__, name);

     return;
}

/* Set the default value for the variables in the global control
 * block. */
void gtodo_set_default_config (void) {
     global_config.file = gkrellm_make_data_file_name(CONFIG_NAME, "gtodo-tasks");
     global_config.time_format = REGULAR;
     global_config.date_format = AMERICAN;
     global_config.auto_delete = FALSE;
     global_config.if_acked = TRUE;
     global_config.remind_complete = FALSE;
     global_config.enable_reminders = TRUE;
     global_config.start_interval = 15;
     global_config.repeat_reminders = FALSE;
     global_config.repeat_interval = 5;
     global_config.flash = TRUE;
     global_config.popup = FALSE;
     global_config.exec = FALSE;
     global_config.command = NULL;
     global_config.text = NULL;
     global_config.clean = FALSE;
     global_config.acked = TRUE;
     global_config.timer_on = FALSE;
     global_config.timer_change = FALSE;
     global_config.window = FALSE;
     global_config.have_active = FALSE;
     global_config.ntasks = 0;

     /* declare the global task list store
      * this the list store that will be used
      * throughout the entire plugin */
     if (!task_list_store) {
	  task_list_store = gtk_list_store_new (N_COLUMNS, G_TYPE_BOOLEAN, /* working on checkbox */
						G_TYPE_STRING,             /* task title */
						G_TYPE_INT,                /* percentage complete */
						G_TYPE_LONG,               /* deadline */
						G_TYPE_LONG,               /* reminder */
						G_TYPE_INT,                /* time spent */
						G_TYPE_STRING);            /* comment entry */
     }
}

/* Load the tasks from the task list file.
 */
int gtodo_load_task_list (gchar *filename) {
     FILE       *fp;
     GtkTreeIter iter;
     gboolean    active;
     gboolean    clean = TRUE;
     gchar       title[100];
     gchar       cfilename[100];
     gint        percent;
     glong       deadline;
     glong       reminder;
     gint        time;

     /* make sure that we have somewhere to store the
      * tasks in */
     if (!task_list_store) {
	  printf ("G-Todo: %s(): task structure empty!\n", __FUNCTION__);
	  return 0;
     }

     DPRINTF ("opening task file %s", global_config.file);
     if ((fp = fopen (global_config.file, "r")) == NULL) {
	  printf ("G-Todo: %s(): error opening file %s: %s\n", __FUNCTION__, filename, strerror (errno));
	  return 0;
     }
     while (!feof (fp)) {
	  int status = fscanf (fp, "%d|%[^|]|%d|%ld|%ld|%d|%[^\n]", &active, title, &percent, &deadline,
			       &reminder, &time, cfilename); 
	  if (status == EOF) break;
	  else if (status < 7) {
	       /* because of the way fscanf works (it does not forward the files pointer
		* on error), if there is an error while it does its conversions, we will
		* go into an infinite if we don't break here */
	       printf ("G-Todo: %s(): error reading task file!\n", __FUNCTION__);
	       break;
	  } else {
	       /* now we have to read in the comment from the comment file */
	       gchar *comment = NULL;
	       FILE *cfp;

	       DPRINTF ("task %s, reading comment from %s", title, cfilename);
	       if ((cfp = fopen (cfilename, "r")) != NULL) {
		    char buf[4096];
		    while (!feof (cfp)) {
			 if (fread (buf, sizeof (char), 4096, cfp) < 4096)
			      if (ferror (cfp) != 0)
				   clean = FALSE;
			 comment = g_strconcat ((comment?comment:""), buf, NULL);
		    }
	       } else clean = FALSE;

	       /* we only want to remove the file
		* if it has been correctly read */
	       if (clean) unlink (cfilename);
	       
	       gtk_list_store_append (GTK_LIST_STORE (task_list_store), &iter);  
	       gtk_list_store_set (GTK_LIST_STORE (task_list_store), &iter,
				   ACTIVE_COLUMN, (active==1?TRUE:FALSE), TITLE_COLUMN, title,
				   COMPLETE_COLUMN, percent, DEADLINE_COLUMN, deadline,
				   REMINDER_COLUMN, reminder, TIME_COLUMN, time,
				   COMMENT_COLUMN, comment, -1);
	       global_config.ntasks++;
	       if (active) global_config.have_active = TRUE;
	       if (active) gtodo_set_panel_tooltip (title, deadline, percent);
	       if (comment) g_free (comment);
	  }
     }
     return 1;
}

/* Save the tasks to the task list file */
int gtodo_save_task_list (gchar *filename) {
     FILE       *fp;
     GtkTreeIter iter;
     gboolean    active;
     gchar      *title = NULL;
     gint        percent;
     glong       deadline;
     glong       reminder;
     gint        time;
     gchar      *comment = NULL;
     gboolean    valid;
     gchar      *tmp_dir = NULL;
     
     /* anything to save? */
     if (!task_list_store) return 1;

     /* open the file */
     DPRINTF ("saving task file %s", filename);
     if ((fp = fopen (filename, "w")) == NULL) {
	  printf ("G-Todo: %s(): error opening file %s: %s\n", __FUNCTION__, filename, strerror (errno));
	  return 0;
     }

     /* I wish there was a better way to do this but until I figure it
      * out we are doing it the dirty way!! */
     gtodo_clean_config_dir_temp_files ();
      
     /* this is a hack for the tempnam function
      * we need to remove the TMPDIR environment variable
      * otherwise, the filenames end up there */
     tmp_dir = getenv ("TMPDIR");
     unsetenv ("TMPDIR");
     
     /* Get the first iter in the list */
     valid = gtk_tree_model_get_iter_first (GTK_TREE_MODEL (task_list_store), &iter);
     while (valid) {
	  gchar *cfilename = NULL;
	  FILE  *cfp;
	  int chars;
	  
	  gtk_tree_model_get (GTK_TREE_MODEL (task_list_store), &iter, ACTIVE_COLUMN, &active,
			      TITLE_COLUMN, &title, COMPLETE_COLUMN, &percent,
			      DEADLINE_COLUMN, &deadline, REMINDER_COLUMN, &reminder,
			      TIME_COLUMN, &time, COMMENT_COLUMN, &comment, -1);

	  /* save the comment file first */
	  cfilename = tempnam (gkrellm_make_data_file_name (CONFIG_NAME, NULL),
			       ".dat");
	  DPRINTF ("comment file: %s", cfilename);
	  if ((cfp = fopen (cfilename, "w")) != NULL) {
	       chars = fwrite (comment, sizeof (char), (comment?strlen (comment):0), cfp);
	       DPRINTF ("wrote %d chars in %s", chars, cfilename);
	       fclose (cfp);
	  } else
	       fprintf (stderr, "G-Todo: %s(): error opening comment file %s: %s\n", __FUNCTION__,
			cfilename, strerror (errno));

	  /* we only write the task if the entire comment has successfully been
	   * writen out */
	  if (chars == (comment?strlen (comment):0))
	       fprintf (fp, "%d|%s|%d|%ld|%ld|%d|%s\n", active, title, percent, deadline, reminder, time, cfilename);
	  valid = gtk_tree_model_iter_next (GTK_TREE_MODEL (task_list_store), &iter);
	  free (cfilename);
	  if (title) g_free (title);
	  if (comment) g_free (comment);
     }
     /* restoring from above */
     if (tmp_dir) {
	  putenv (g_strdup_printf ("%s=%s", "TMPDIR", tmp_dir));
	  DPRINTF_IF_TRUE (tmp_dir != NULL, "tmp_is not null");
     }
     fclose (fp);
     return 1;
}


static void gtodo_clean_config_dir_temp_files (void) {
     gchar         *path = gkrellm_make_data_file_name (CONFIG_NAME, NULL);
     DIR           *dir;
     struct dirent *e;
     
     if ((dir = opendir (path)) == NULL) return;
     while ((e = readdir (dir)) != NULL)
	  if (!strncmp (e->d_name, ".dat", 4)) {
	       DPRINTF ("found file: %s", e->d_name);
	       unlink (g_strconcat (path, "/", e->d_name, NULL));
	  }

     closedir (dir);
}

	       
     
