!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!
module cubeset_panel_frame_tool
  use cubetools_parameters
  use cubetools_structure
  use cubetools_keywordlist_types
  use cubeset_plot_tool
  use cubeset_messaging
  !
  public :: panel_frame_comm_t,panel_frame_user_t,panel_frame_prog_t
  private
  !
  !-------------------------------------------------------------------
  integer(argu_k), parameter :: nframe = 5
  character(len=argu_l) :: frame_kinds(nframe)
  data frame_kinds /'USER','DEFAULT','PLOT','HEADER','WEDGE'/
  integer(kind=code_k), parameter :: code_frame_user    = 1
  integer(kind=code_k), parameter :: code_frame_default = 2
  integer(kind=code_k), parameter :: code_frame_plot    = 3
  integer(kind=code_k), parameter :: code_frame_header  = 4
  integer(kind=code_k), parameter :: code_frame_wedge   = 5
  !
  ! These definitions assume A4 landscape!
  real(kind=pane_k), parameter :: landscape_default(4) = [0.214,1.322,0.125,0.925]
  real(kind=pane_k), parameter :: landscape_plot(4)    = [0.171,1.081,0.085,0.995]
  real(kind=pane_k), parameter :: landscape_header(4)  = [1.081,1.428,0.085,0.995]
!  real(kind=pane_k), parameter :: landscape_wedge(4)   = [1.190,1.420,0.250,0.480]
  real(kind=pane_k), parameter :: landscape_wedge(4)   = [1.140,1.370,0.300,0.530]
  !
  ! These definitions assume A4 portrait!
  real(kind=pane_k), parameter :: portrait_default(4) = [0.125,0.875,0.100,1.250]
  real(kind=pane_k), parameter :: portrait_plot(4)    = [0.085,0.995,0.171,1.081]
  real(kind=pane_k), parameter :: portrait_header(4)  = [0.085,0.995,1.081,1.428]
  real(kind=pane_k), parameter :: portrait_wedge(4)   = [0.550,0.780,1.140,1.370]
  !
  real(kind=pane_k), parameter :: frame_positions(4,5,2) = reshape([ &
       landscape_default,landscape_default,landscape_plot,landscape_header,landscape_wedge, &
       portrait_default,portrait_default,portrait_plot,portrait_header,portrait_wedge], &
       [4,5,2])
  !-------------------------------------------------------------------
  integer(kind=argu_k), parameter :: nfixed = 5
  character(len=argu_l) :: fixed_corners(nfixed)
  data fixed_corners /'NONE','TLC','TRC','BLC','BRC'/
  integer(kind=code_k), parameter :: code_fixed_none = 1
  integer(kind=code_k), parameter :: code_fixed_tlc  = 2
  integer(kind=code_k), parameter :: code_fixed_trc  = 3
  integer(kind=code_k), parameter :: code_fixed_blc  = 4
  integer(kind=code_k), parameter :: code_fixed_brc  = 5
  !-------------------------------------------------------------------
  !
  type panel_frame_comm_t
     type(option_t),           public,  pointer :: position   !***JP: public is temporary
     type(keywordlist_comm_t), private, pointer :: preset_arg
     type(option_t),           private, pointer :: fixed
     type(keywordlist_comm_t), private, pointer :: corner_arg
   contains
     procedure, public :: register => cubeset_panel_frame_comm_register
     procedure, public :: parse    => cubeset_panel_frame_comm_parse
  end type panel_frame_comm_t
  !
  type panel_frame_user_t
     integer(kind=code_k), public :: fixed_corner = code_fixed_none ! By default, center inside the frame position
     integer(kind=code_k), public :: kind = code_frame_default      !
     real(kind=pane_k),    public :: pos(4) = landscape_default     ! Frame x and y coordinates
   contains
     procedure, public :: toprog     => cubeset_panel_frame_user_toprog
     procedure, public :: fix_corner => cubeset_panel_frame_user_toprog_fix_corner
     procedure, public :: list       => cubeset_panel_frame_user_list
     procedure, public :: uservar    => cubeset_panel_frame_user_uservar
  end type panel_frame_user_t
  !
  type panel_frame_prog_t
     real(kind=pane_k) :: pos(2,2) ! Position
     real(kind=pane_k) :: mar(2)   ! Symmetrical margin
     real(kind=pane_k) :: blc(2)   ! pos(1,:)+mar
     real(kind=pane_k) :: size(2)
     ! VVV temporary workaround for cube procedures
     logical           :: landscape =.true.
   contains
     procedure, public :: uservar => cubeset_panel_frame_prog_uservar
  end type panel_frame_prog_t
  !
contains
  !
  subroutine cubeset_panel_frame_comm_register(comm,error)
    !-------------------------------------------------------------------
    !
    !-------------------------------------------------------------------
    class(panel_frame_comm_t), intent(inout) :: comm
    logical,                   intent(inout) :: error
    !
    type(standard_arg_t) :: stdarg
    type(keywordlist_comm_t)  :: keyarg
    character(len=*), parameter :: rname='PANEL>FRAME>COMM>REGISTER'
    !
    call cubeset_message(setseve%trace,rname,'Welcome')
    !
    call cubetools_register_option(&
         'POSITION','preset | px1 px2 py1 py2',&
         'Set the area of the plot page to be used',&
         'Unit is in fraction of the total GREG plot page smallest&
         & side, e.g. 0.1 0.9 0.1 1.4. Using the same unit in both&
         & direction ensure that the same value correspond to the&
         & same length on the plot',&
         comm%position,&
         error)
    if (error) return
    call keyarg%register(&
         'px1',&
         'Panel position preset or left side x coordinate',&
         strg_id,&
         code_arg_mandatory,&
         frame_kinds,&
         flexible,&
         comm%preset_arg,&
         error)
    if (error) return
    call stdarg%register(&
         'px2',&
         'right side x coordinate',&
         strg_id,&
         code_arg_optional,&
         error)
    if (error) return
    call stdarg%register(&
         'py1',&
         'bottom side y coordinate',&
         strg_id,&
         code_arg_optional,&
         error)
    if (error) return
    call stdarg%register(&
         'py2',&
         'top side y coordinate',&
         strg_id,&
         code_arg_optional,&
         error)
    if (error) return
    !
    call cubetools_register_option(&
         'FIXED','[corner]',&
         'Fix panels to a corner of the total plot page',&
         'TRC for Top Right corner, TLC for the Top Left Corner, BRC&
         & for the Bottom Right Corner and BLC for the Bottom Left&
         & Corner',&
         comm%fixed,error)
    if (error) return
    call keyarg%register(&
         'corner',&
         'Corner to fix panel to',&
         strg_id,&
         code_arg_optional,&
         fixed_corners,&
         .not.flexible,&
         comm%corner_arg,&
         error)
    if (error) return
  end subroutine cubeset_panel_frame_comm_register
  !
  subroutine cubeset_panel_frame_comm_parse(comm,line,user,error)
    !-------------------------------------------------------------------
    ! /POSITION px1 px2 py1 py2 | header|plot|wedge|default
    ! /FIXED_CORNER [NONE|TLC|TRC|BLC|BRC]
    !-------------------------------------------------------------------
    class(panel_frame_comm_t), intent(in)    :: comm
    character(len=*),          intent(in)    :: line
    type(panel_frame_user_t),  intent(inout) :: user
    logical,                   intent(inout) :: error
    !
    character(len=*), parameter :: rname='PANEL>FRAME>COMM>PARSE'
    !
    call cubeset_message(setseve%trace,rname,'Welcome')
    !
    call parse_corner(line,comm%fixed,comm%corner_arg,user%fixed_corner,error)
    if (error) return
    call parse_position(line,comm%position,user%kind,user%pos,error)
    if (error) return
    !
  contains
    !
    subroutine parse_corner(line,opt,arg,code,error)
      !-----------------------------------------------------------------
      !***JP: Obsolescent parsing style
      !-----------------------------------------------------------------
      character(len=*),         intent(in)    :: line
      type(option_t),           intent(in)    :: opt
      type(keywordlist_comm_t), intent(in)    :: arg
      integer(kind=code_k),     intent(inout) :: code
      logical,                  intent(inout) :: error
      !
      character(len=argu_l) :: argu,resolved
      logical :: present
      character(len=*), parameter :: rname='PARSE>POSITION'
      !
      call cubeset_message(setseve%trace,rname,'Welcome')
      !
      call opt%present(line,present,error)
      if (error) return
      if (present) then
         argu = 'TRC'
         call cubetools_getarg(line,opt,1,argu,.not.mandatory,error)
         if (error) return
         call cubetools_keywordlist_user2prog(arg,argu,code,resolved,error)
         if (error) return
      endif
    end subroutine parse_corner
    !
    subroutine parse_position(line,opt,kind,pos,error)
      use cubetools_disambiguate
      !-------------------------------------------------------------------
      !***JP: Obsolescent parsing style
      !-------------------------------------------------------------------
      character(len=*),     intent(in)    :: line
      type(option_t),       intent(in)    :: opt
      integer(kind=code_k), intent(inout) :: kind
      real(kind=pane_k),    intent(inout) :: pos(4)
      logical,              intent(inout) :: error
      !
      integer(kind=argu_k) :: iarg,narg
      character(len=argu_l) :: arg,key
      logical :: present
      character(len=*), parameter :: rname='PARSE>POSITION'
      !
      call opt%present(line,present,error)
      if (error) return
      if (present) then
         narg = opt%getnarg()
         if (narg.eq.1) then
            call cubetools_getarg(line,opt,1,arg,mandatory,error)
            if (error) return
            call cubetools_disambiguate_strict(arg,frame_kinds,kind,key,error)
            if (error) return
         else if (narg.eq.4) then
            kind = code_frame_user
            do iarg=1,narg
               call cubetools_getarg(line,opt,iarg,pos(iarg),mandatory,error)
               if (error) return
            enddo ! iarg
         else
            call cubeset_message(seve%e,rname,'/POSITION takes 1 or 4 arguments')
            error = .true.
            return
         endif
      endif
    end subroutine parse_position
  end subroutine cubeset_panel_frame_comm_parse
  !
  !----------------------------------------------------------------------
  !
  subroutine cubeset_panel_frame_user_toprog(user,prog,page_geo,error)
    !-------------------------------------------------------------------
    ! 
    !-------------------------------------------------------------------
    class(panel_frame_user_t), intent(in)    :: user
    type(panel_frame_user_t),  intent(inout) :: prog
    integer(kind=code_k),      intent(in)    :: page_geo
    logical,                   intent(inout) :: error
    !
    character(len=*), parameter :: rname='PANEL>FRAME>USER>TOPROG'
    !
    call cubeset_message(setseve%trace,rname,'Welcome')
    !
    prog%fixed_corner = user%fixed_corner
    prog%kind = user%kind
    if (prog%kind.eq.code_frame_user) then
       prog%pos = user%pos
    else
       prog%kind = user%kind
       prog%pos = frame_positions(:,prog%kind,page_geo)
    endif
  end subroutine cubeset_panel_frame_user_toprog
  !
  subroutine cubeset_panel_frame_user_toprog_fix_corner(user,prog,error)
    !-------------------------------------------------------------------
    ! 
    !-------------------------------------------------------------------
    class(panel_frame_user_t), intent(in)    :: user
    type(panel_frame_prog_t),  intent(inout) :: prog
    logical,                   intent(inout) :: error
    !
    character(len=*), parameter :: rname='PANEL>FRAME>USER>TOPROG>FIX>CORNER'
    !
    call cubeset_message(setseve%trace,rname,'Welcome')
    !
    select case (user%fixed_corner)
    case (code_fixed_none)
       ! Does nothing
    case (code_fixed_trc)
       prog%mar = 2.0*prog%mar
    case (code_fixed_tlc)
       prog%mar(code_x) = 0
       prog%mar(code_y) = 2.0*prog%mar(code_y)     
    case (code_fixed_brc)
       prog%mar(code_x) = 2.0*prog%mar(code_x)
       prog%mar(code_y) = 0
    case (code_fixed_blc)
       prog%mar = 0
    case default
       call cubeset_message(seve%e,rname,'Unknown prog%fixed_corner code')
       error = .true.
       return
    end select
    prog%blc = prog%pos(1,:)+prog%mar
  end subroutine cubeset_panel_frame_user_toprog_fix_corner
  !
  subroutine cubeset_panel_frame_user_list(user,error)
    !-------------------------------------------------------------------
    ! 
    !-------------------------------------------------------------------
    class(panel_frame_user_t), intent(in)    :: user
    logical,                   intent(inout) :: error
    !
    character(len=mess_l) :: mess
    character(len=*), parameter :: rname='PANEL>FRAME>USER>LIST'
    !
    call cubeset_message(setseve%trace,rname,'Welcome')
    !
    write(mess,'(a26,1x,a,f5.3,a,f5.3,a,f5.3,a,f5.3,a)') &
         'Global frame:','(',&
         user%pos(1),' - ',user%pos(2),') x (', &
         user%pos(3),' - ',user%pos(4),')'
    call cubeset_message(seve%r,rname,mess)
  end subroutine cubeset_panel_frame_user_list
  !
  subroutine cubeset_panel_frame_user_uservar(user,var,error)
    use gkernel_interfaces
    use cubetools_progstruct_types
    !-------------------------------------------------------------------
    ! Create the associated readonly SIC structure
    !***JP: Obsolescent way of defining user variables
    !-------------------------------------------------------------------
    class(panel_frame_user_t), intent(in)    :: user
    type(progstruct_t),        intent(in)    :: var
    logical,                   intent(inout) :: error
    !
    character(len=*), parameter :: rname='PANEL>FRAME>USER>USERVAR'
    !
    call sic_def_inte(trim(var%name)//'frame_fixed_corner',user%fixed_corner,0,0,readonly,error)
    if (error) return
    call sic_def_inte(trim(var%name)//'frame_kind',user%kind,0,0,readonly,error)
    if (error) return
    call sic_def_real(trim(var%name)//'frame_pos',user%pos,1,4,readonly,error)
    if (error) return
  end subroutine cubeset_panel_frame_user_uservar
  !
  !----------------------------------------------------------------------
  !
  subroutine cubeset_panel_frame_prog_uservar(prog,var,error)
    use gkernel_interfaces
    use cubetools_progstruct_types
    !-------------------------------------------------------------------
    ! Create the associated readonly SIC structure
    !***JP: Obsolescent way of defining user variables
    !-------------------------------------------------------------------
    class(panel_frame_prog_t), intent(in)    :: prog
    type(progstruct_t),        intent(in)    :: var
    logical,                   intent(inout) :: error
    !
    integer(kind=8) :: dims(2)
    character(len=*), parameter :: rname='PANEL>FRAME>PROG>USERVAR'
    !
    dims = 2
    call sic_def_real(trim(var%name)//'pos',prog%pos,2,dims,readonly,error)
    if (error) return
    call sic_def_real(trim(var%name)//'mar',prog%mar,1,2,readonly,error)
    if (error) return
    call sic_def_real(trim(var%name)//'blc',prog%blc,1,2,readonly,error)
    if (error) return
    call sic_def_real(trim(var%name)//'size',prog%size,1,2,readonly,error)
    if (error) return
    call sic_def_logi(trim(var%name)//'landscape',prog%landscape,readonly,error)
    if (error) return
  end subroutine cubeset_panel_frame_prog_uservar
end module cubeset_panel_frame_tool
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!
module cubeset_panel_panel_tool ! The repetition of panel is on-purpose here!
  use cubetools_parameters
  use cubetools_structure
  use cubeset_plot_tool
  use cubeset_messaging
  !
  public :: panel_panel_comm_t,panel_panel_user_t,panel_panel_prog_t
  private
  !
  type panel_panel_comm_t
     type(option_t), public, pointer :: comm   !***JP: public is temporary
     type(option_t), public, pointer :: nxy    !***JP: public is temporary
     type(option_t), public, pointer :: inter  !***JP: public is temporary
     type(option_t), public, pointer :: size   !***JP: public is temporary
     type(option_t), public, pointer :: aspect !***JP: public is temporary
   contains
     procedure, public :: register => cubeset_panel_panel_comm_register
     procedure, public :: parse    => cubeset_panel_panel_comm_parse
  end type panel_panel_comm_t
  !
  type panel_panel_user_t
     real(kind=pane_k), public :: ixy(2) = [1.0,1.0]   ! Panel number along the x and y axes
     real(kind=pane_k), public :: nxy(2) = [1.0,1.0]   ! Number of panels along the x and y axes
     real(kind=pane_k), public :: inter(2) = [0.0,0.0] ! Interval betwen panels
     real(kind=pane_k), public :: size(2)  = [1.0,1.0] ! Panel size along the x and y axes
     real(kind=pane_k), public :: aspect = 0.0         ! Aspect ratio. 0.0 means "Not taken into account"
   contains
     procedure, public :: toprog     => cubeset_panel_panel_user_toprog
     procedure, public :: list       => cubeset_panel_panel_user_list
     procedure, public :: uservar    => cubeset_panel_panel_user_uservar
  end type panel_panel_user_t
  !
  type panel_panel_prog_t
     real(kind=pane_k), private :: min(2)
     real(kind=pane_k), private :: max(2)
     real(kind=pane_k), public :: size(2)
     real(kind=pane_k), public :: aspect
   contains
     procedure, public :: compute_position => cubeset_panel_panel_prog_compute_position
     procedure, public :: resize           => cubeset_panel_panel_prog_resize
     procedure, public :: toviewport       => cubeset_panel_panel_prog_toviewport
     procedure, public :: uservar          => cubeset_panel_panel_prog_uservar
  end type panel_panel_prog_t
  !
contains
  !
  subroutine cubeset_panel_panel_comm_register(comm,run,error)
    !-------------------------------------------------------------------
    !
    !-------------------------------------------------------------------
    class(panel_panel_comm_t), intent(inout) :: comm
    external                                 :: run
    logical,                   intent(inout) :: error
    !
    type(standard_arg_t) :: stdarg
    character(len=*), parameter :: rname='PANEL>PANEL>COMM>REGISTER'
    !
    call cubeset_message(setseve%trace,rname,'Welcome')
    !
    call cubetools_register_command(&
         'PANEL','[I [J]]',&
         'Define and configure plot regions in GREG',&
         'if only I is given panel counting starts at the top left&
         & and ends at the bottom right, if both I and J are given I&
         & stands for the column (from left to right) and J the line&
         & (from bottom to top). If no option is given PANEL uses the&
         & previous plot configurations, in the case of the first&
         & call of PANEL in the CUBE session, CUBE plot defaults are&
         & used',&
         run,&
         comm%comm,error)
    if (error) return
    call stdarg%register(&
         'I',&
         'I-eth panel or Column',&
         strg_id,&
         code_arg_optional,&
         error)
    if (error) return
    call stdarg%register(&
         'J',&
         'J-eth line',&
         strg_id,&
         code_arg_optional,&
         error)
    if (error) return
    !
    call cubetools_register_option(&
         'NXY','nx [ny]',&
         'Define the grid of panels',&
         'If only nx is given PANEL will use a square configuration&
         & to display nx plots. If both nx and ny are given nx stand&
         & for the number of columns of plots and ny for the number&
         & of lines of plots, with nx*ny being the total number of&
         & plots',&
         comm%nxy,error)
    if (error) return
    call stdarg%register(&
         'nx',&
         'Number of panels or number of columns',&
         strg_id,&
         code_arg_mandatory,&
         error)
    if (error) return
    call stdarg%register(&
         'ny',&
         'Number of lines',&
         strg_id,&
         code_arg_optional,&
         error)
    if (error) return
    !
    call cubetools_register_option(&
         'INTER','rx [ry]',& ! *** JP: In english, it should be SPACING...
         'Define the spacing between panels in the grid',&
         'If rx and ry are given rx is the space in the x direction&
         & and ry is the spacing on the y direction.  If only rx is&
         & given rx is the spacing in both directions.',&
         comm%inter,&
         error)
    if (error) return
    call stdarg%register(&
         'rx',&
         'Spacing between panels along the x axis',&
         strg_id,&
         code_arg_mandatory,&
         error)
    if (error) return
    call stdarg%register(&
         'ry',&
         'Spacing between panels along the y axis',&
         strg_id,&
         code_arg_optional,&
         error)
    if (error) return
    !
    call cubetools_register_option(&
         'SIZE','sx [sy]',&
         'Define the panel size relatively to one grid rectangle',&
         'If sx and sy are given sx is the size in the x direction&
         & and sy is the size on the y direction.  If only sx is&
         & given sx is the size in both directions.',&
         comm%size,&
         error)
    if (error) return
    call stdarg%register(&
         'sx',&
         'Panel size along the x axis. The default value is 1',&
         strg_id,&
         code_arg_mandatory,&
         error)
    if (error) return
    call stdarg%register(&
         'sy',&
         'Panel size along the y axis. The default value is 1',&
         strg_id,&
         code_arg_optional,&
         error)
    if (error) return
    !
    call cubetools_register_option(&
         'ASPECT','ratio',&
         'Set the aspect ratio of the panel',&
         'A ratio larger than 1 means that the X axis will be larger&
         & than the Y axis. If ratio is 0 or * the aspect ratio is&
         & chosen automatically to maximise plot area. Default is 0',&
         comm%aspect,error)
    if (error) return
    call stdarg%register(&
         'ratio',&
         'panel aspect ratio',&
         strg_id,&
         code_arg_mandatory,&
         error)
    if (error) return
  end subroutine cubeset_panel_panel_comm_register
  !
  subroutine cubeset_panel_panel_comm_parse(comm,line,user,error)
    !-------------------------------------------------------------------
    ! PANEL [I [J]]
    ! /NXY nx [ny]
    ! /INTERVAL rx [ry]
    ! /SIZE sx [sy]
    ! /ASPECT *|ratio
    !-------------------------------------------------------------------
    class(panel_panel_comm_t), intent(in)    :: comm
    character(len=*),          intent(in)    :: line
    type(panel_panel_user_t),  intent(inout) :: user
    logical,                   intent(inout) :: error
    !
    character(len=*), parameter :: rname='PANEL>PANEL>COMM>PARSE'
    !
    call parse_xy(line,comm%comm,user%ixy,error)
    if (error) return
    call parse_xy(line,comm%nxy,user%nxy,error)
    if (error) return
    call parse_xy(line,comm%inter,user%inter,error)
    if (error) return
    call parse_xy(line,comm%size,user%size,error)
    if (error) return
    call parse_string2value(line,comm%aspect,1,strg_star,0.0,user%aspect,error)
    if (error) return
    !
  contains
    !
    subroutine parse_xy(line,opt,xy,error)
      !-----------------------------------------------------------------
      !***JP: Obsolescent parsing style
      !-----------------------------------------------------------------
      character(len=*),  intent(in)    :: line
      type(option_t),    intent(in)    :: opt
      real(kind=pane_k), intent(inout) :: xy(2)
      logical,           intent(inout) :: error
      !
      integer(kind=4), parameter :: iarg_x = 1
      integer(kind=4), parameter :: iarg_y = 2
      character(len=*), parameter :: rname='PARSE>XY'
      !
      call cubeset_message(setseve%trace,rname,'Welcome')
      !
      if (opt%getnarg().gt.0) then
         call cubetools_getarg(line,opt,iarg_x,xy(code_x),mandatory,error)
         if (error) return
         xy(code_y) = 0.0
         call cubetools_getarg(line,opt,iarg_y,xy(code_y),.not.mandatory,error)
         if (error) return
      endif
    end subroutine parse_xy
    !
    subroutine parse_string2value(line,opt,iarg,defstring,defval,argval,error)
      use gkernel_interfaces, only: sic_math_real
      !-------------------------------------------------------------------
      !***JP: Obsolescent parsing style
      !-------------------------------------------------------------------
      character(len=*),  intent(in)    :: line
      type(option_t),    intent(in)    :: opt
      integer(kind=4),   intent(in)    :: iarg
      character(len=*),  intent(in)    :: defstring
      real(kind=real_k), intent(in)    :: defval
      real(kind=real_k), intent(out)   :: argval
      logical,           intent(inout) :: error
      !
      logical :: do
      character(len=argu_l) :: arg
      character(len=mess_l) :: mess
      !
      call opt%present(line,do,error)
      if (error) return
      if (do) then
         call cubetools_getarg(line,opt,iarg,arg,mandatory,error)
         if (error) return
         if (trim(arg).eq.defstring) then
            argval = defval
         else
            call sic_math_real(arg,argu_l,argval,error)
            if (error) then
               write(mess,'(3a)') 'Cannot convert ',arg,' to REAL' 
               call cubeset_message(seve%e,rname,mess)
               return
            endif
         endif
      endif
    end subroutine parse_string2value
  end subroutine cubeset_panel_panel_comm_parse
  !
  !----------------------------------------------------------------------
  !
  subroutine cubeset_panel_panel_user_toprog(user,prog,error)
    !-------------------------------------------------------------------
    ! 
    !-------------------------------------------------------------------
    class(panel_panel_user_t), intent(in)    :: user
    type(panel_panel_user_t),  intent(inout) :: prog
    logical,                   intent(inout) :: error
    !
    character(len=mess_l) :: mess
    character(len=*), parameter :: rname='PANEL>PANEL>USER>TOPROG'
    !
    call cubeset_message(setseve%trace,rname,'Welcome')
    !
    ! Space between panels
    if (user%inter(code_y).eq.0.0) then
       prog%inter(code_x) = user%inter(code_x)
       prog%inter(code_y) = user%inter(code_x)
    else
       prog%inter = user%inter
    endif
    ! Panel size 
    if (user%size(code_y).eq.0.0) then
       prog%size(code_x) = user%size(code_x)
       prog%size(code_y) = user%size(code_x)
    else
       prog%size = user%size
    endif
    ! Aspect ratio
    prog%aspect = user%aspect !***JP: There is something fishy with prog%aspect
    ! NXY
    if (user%nxy(code_y).eq.0.0) then
       call nboxes2nxy(user%aspect,nint(user%nxy(code_x)),prog%nxy,error)
       if (error) return
    else
       prog%nxy = user%nxy
    endif
    if (any(prog%nxy.le.0.0)) then
       write(mess,'(a,f5.1,a,f5.1,a)') 'Can not create grid of panels of shape: (',&
            prog%nxy(1),',',prog%nxy(2),')'
       call cubeset_message(seve%e,rname,mess)
       error = .true.
       return
    endif
    ! IXY
    if (user%ixy(code_y).eq.0.0) then
       ! user%ixy(1) is the linear window number.
       ! In this case, origin is on the top, left corner.
       prog%ixy(code_y) = int((user%ixy(code_x)-1)/prog%nxy(code_x))+1
       prog%ixy(code_x) = user%ixy(code_x)-(prog%ixy(code_y)-1)*prog%nxy(code_x)
       prog%ixy(code_y) = (prog%nxy(code_y)+1)-prog%ixy(code_y)
    else
       ! user%ixy is the window number on the x and y axes.
       ! In this case, axis origin is on the bottom, left corner.
       prog%ixy = user%ixy
    endif
    !
  contains
    !
    subroutine nboxes2nxy(aspect,nboxes,nxy,error)
      real(kind=pane_k),    intent(in)    :: aspect
      integer(kind=inte_k), intent(in)    :: nboxes
      real(kind=pane_k),    intent(inout) :: nxy(2)
      logical,              intent(inout) :: error
      !
      real(kind=pane_k) :: locaspect
      character(len=*), parameter :: rname='PANEL>NXYBOX'
      !
      call cubeset_message(setseve%trace,rname,'Welcome')
      !
      ! ***JP: There is something fishy with prog%aspect
      if (aspect.eq.0.0) then
         locaspect = 1.0
      else
         locaspect = aspect
      endif
      !
      if (locaspect.gt.1.0) then
         nxy(code_y) = int(sqrt(nboxes*locaspect+1))
         nxy(code_x) = max(1,nint(nxy(code_y)/locaspect))
         if (nxy(code_x)*nxy(code_y).lt.nboxes) then
            nxy(code_y) = nxy(code_y)+1
            if (nxy(code_x)*nxy(code_y).lt.nboxes) then
               nxy(code_x) = nxy(code_x)+1
            endif
         else if ((nxy(code_y)-1)*nxy(code_x).eq.nboxes) then
            nxy(code_y) = nxy(code_y)-1
         endif
      else
         nxy(code_x) = int(sqrt(nboxes/locaspect+1))
         nxy(code_y) = max(1,nint(nxy(code_x)*locaspect))
         if (nxy(code_y)*nxy(code_x).lt.nboxes) then
            nxy(code_x) = nxy(code_x)+1
            if (nxy(code_y)*nxy(code_x).lt.nboxes) then
               nxy(code_y) = nxy(code_y)+1
            endif
         else if ((nxy(code_x)-1)*nxy(code_y).eq.nboxes) then
            nxy(code_x) = nxy(code_x)-1
         endif
      endif
    end subroutine nboxes2nxy    
  end subroutine cubeset_panel_panel_user_toprog
  !
  subroutine cubeset_panel_panel_user_list(user,error)
    !-------------------------------------------------------------------
    ! 
    !-------------------------------------------------------------------
    class(panel_panel_user_t), intent(in)    :: user
    logical,                   intent(inout) :: error
    !
    character(len=mess_l) :: mess
    character(len=*), parameter :: rname='PANEL>PANEL>USER>LIST'
    !
    call cubeset_message(setseve%trace,rname,'Welcome')
    !
    write(mess,'(a26,1x,a,f4.1,a,f4.1,a,f4.1,a,f4.1,a)') &
         'Panel:','(',user%ixy(code_x),' x ',user%ixy(code_y), &
         ') in (',user%nxy(code_x),' x ',user%nxy(code_y),')'
    call cubeset_message(seve%r,rname,mess)
    write(mess,'(a26,1x,a,f4.1,a,f4.1,a)') &
         'Panel size:','(',user%size(code_x),' x ',user%size(code_y),&
         ') relative to size of one grid rectangle'
    call cubeset_message(seve%r,rname,mess)
    write(mess,'(a26,1x,a,f4.1,a,f4.1,a)') &
         'Space between panels:','(',user%inter(code_x),' x ',user%inter(code_y),&
         ') times the size of one grid rectangle'
    call cubeset_message(seve%r,rname,mess)
    write(mess,'(a26,2x,f4.1,9x,a)') &
         'Aspect ratio:',user%aspect,'! 0.0 => Not taken into account'
    call cubeset_message(seve%r,rname,mess)
  end subroutine cubeset_panel_panel_user_list
  !
  subroutine cubeset_panel_panel_user_uservar(user,var,error)
    use gkernel_interfaces
    use cubetools_progstruct_types
    !-------------------------------------------------------------------
    ! Create the associated readonly SIC structure.
    ! The SIZE and ASPECT variables are set in the PROG uservar
    !***JP: Obsolescent way of defining user variables
    !-------------------------------------------------------------------
    class(panel_panel_user_t), intent(in)    :: user
    type(progstruct_t),        intent(in)    :: var
    logical,                   intent(inout) :: error
    !
    character(len=*), parameter :: rname='PANEL>PANEL>USER>USERVAR'
    !
    call sic_def_real(trim(var%name)//'ixy',user%ixy,1,2,readonly,error)
    if (error) return
    call sic_def_real(trim(var%name)//'nxy',user%nxy,1,2,readonly,error)
    if (error) return
    call sic_def_real(trim(var%name)//'inter',user%inter,1,2,readonly,error)
    if (error) return
  end subroutine cubeset_panel_panel_user_uservar
  !
  !----------------------------------------------------------------------
  !
  subroutine cubeset_panel_panel_prog_compute_position(prog,&
       requestedixy,requestedinter,frameblc,error)
    !-------------------------------------------------------------------
    ! 
    !-------------------------------------------------------------------
    class(panel_panel_prog_t), intent(inout) :: prog
    real(kind=pane_k),         intent(in)    :: requestedixy(2)
    real(kind=pane_k),         intent(in)    :: requestedinter(2)
    real(kind=pane_k),         intent(in)    :: frameblc(2)
    logical,                   intent(inout) :: error
    !
    real(kind=pane_k) :: ixy(2)
    character(len=*), parameter :: rname='PANEL>PANEL>PROG>COMPUTE>POSITION'
    !
    call cubeset_message(setseve%trace,rname,'Welcome')
    !
    ! Take care of request%inter
    if (any(requestedinter.ne.0.0)) then
       ! White inter-spaces are asked
       ! => Just need to redefine the panel number
       ixy = (requestedixy*(1.0+requestedinter)-requestedinter)
    else
       ixy = requestedixy
    endif
    ! Compute the panel position in fraction of smallest page size
    prog%min = (frameblc+(ixy-1)*prog%size)
  end subroutine cubeset_panel_panel_prog_compute_position
  !
  subroutine cubeset_panel_panel_prog_resize(prog,size,error)
    !-------------------------------------------------------------------
    ! Resize the prog panel size to the size of one rectangle grid
    ! according to the size factor
    ! -------------------------------------------------------------------
    class(panel_panel_prog_t), intent(inout) :: prog
    real(kind=pane_k),         intent(in)    :: size(2)
    logical,                   intent(inout) :: error
    !
    character(len=*), parameter :: rname='PANEL>PANEL>PROG>RESIZE'
    !
    call cubeset_message(setseve%trace,rname,'Welcome')
    !
    prog%size = prog%size*size
    prog%max = prog%min+prog%size
  end subroutine cubeset_panel_panel_prog_resize
  !
  subroutine cubeset_panel_panel_prog_toviewport(prog,page_aspect,error)
    !-------------------------------------------------------------------
    ! 
    !-------------------------------------------------------------------
    class(panel_panel_prog_t), intent(in)    :: prog
    real(kind=pane_k),         intent(in)    :: page_aspect
    logical,                   intent(inout) :: error
    !
    real(kind=pane_k) :: min(2),max(2)
    character(len=*), parameter :: rname='PANEL>PANEL>PROG>TOVIEWPORT'
    !
    call pagefraction2normal(page_aspect,prog%min,min)
    call pagefraction2normal(page_aspect,prog%max,max)
    call plot%viewport(min,max,error)
    if (error) return
    !
  contains
    !
    subroutine pagefraction2normal(page_aspect,in,ou)
      !-----------------------------------------------------------------
      ! From fraction of smallest page size to normal (ie, between 0 and 1)
      ! coordinates
      !-----------------------------------------------------------------
      real(kind=pane_k), intent(in)  :: page_aspect
      real(kind=pane_k), intent(in)  :: in(2)
      real(kind=pane_k), intent(out) :: ou(2)
      !
      if (page_aspect.gt.1.0) then
         ou(code_x) = in(code_x)/page_aspect
         ou(code_y) = in(code_y)
      else
         ou(code_x) = in(code_x)
         ou(code_y) = in(code_y)*page_aspect
      endif
    end subroutine pagefraction2normal
  end subroutine cubeset_panel_panel_prog_toviewport
  !
  subroutine cubeset_panel_panel_prog_uservar(prog,var,error)
    use gkernel_interfaces
    use cubetools_progstruct_types
    !-------------------------------------------------------------------
    ! Create the associated readonly SIC structure
    !***JP: Obsolescent way of defining user variables
    !-------------------------------------------------------------------
    class(panel_panel_prog_t), intent(in)    :: prog
    type(progstruct_t),        intent(in)    :: var
    logical,                   intent(inout) :: error
    !
    character(len=*), parameter :: rname='PANEL>PANEL>PROG>USERVAR'
    !
    call sic_def_real(trim(var%name)//'min',prog%min,1,2,readonly,error)
    if (error) return
    call sic_def_real(trim(var%name)//'max',prog%max,1,2,readonly,error)
    if (error) return
    call sic_def_real(trim(var%name)//'size',prog%size,1,2,readonly,error)
    if (error) return
    call sic_def_real(trim(var%name)//'aspect',prog%aspect,0,0,readonly,error)
    if (error) return
  end subroutine cubeset_panel_panel_prog_uservar  
end module cubeset_panel_panel_tool
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!
module cubeset_panel
  use cubetools_parameters
  use cubesyntax_key_types
  use cubeset_plot_tool
  use cubeset_plot_page_tool
  use cubeset_panel_frame_tool
  use cubeset_panel_panel_tool
  use cubeset_messaging
  !
  public :: panel
  private
  !
  type panel_request_t
     type(plot_page_geometry_user_t), private :: page
     type(panel_frame_user_t),        private :: frame
     type(panel_panel_user_t),        private :: panel
   contains
     procedure, public :: default => cubeset_panel_request_default
     procedure, public :: toprog  => cubeset_panel_request_user2prog
     procedure, public :: list    => cubeset_panel_request_list
  end type panel_request_t
  !
  type panel_comm_t
     type(key_comm_t),         private :: default
     type(key_comm_t),         private :: box
     type(panel_panel_comm_t), private :: panel
     type(panel_frame_comm_t), private :: frame
   contains
     procedure, public  :: register => cubeset_panel_comm_register
     procedure, private :: parse    => cubeset_panel_comm_parse
     procedure, private :: main     => cubeset_panel_comm_main
  end type panel_comm_t
  type(panel_comm_t) :: panel
  !
  type panel_user_t
     type(key_user_t),      private :: list
     type(key_user_t),      private :: default
     type(key_user_t),      private :: box
     type(panel_request_t), private :: request
   contains
     procedure, private :: toprog => cubeset_panel_user_toprog
  end type panel_user_t
  !
  type panel_prog_t
     type(key_prog_t),         private :: list    ! List current panel parameters
     type(key_prog_t),         private :: default ! Reset defaults
     type(key_prog_t),         private :: box     ! Plot box
     type(panel_request_t),    private :: request ! Explicit user requests
     type(panel_frame_prog_t), private :: frame   ! Frame characteristics
     type(panel_panel_prog_t), private :: current ! Current panel characteristics
   contains
     procedure, private :: header  => cubeset_panel_prog_header
     procedure, private :: data    => cubeset_panel_prog_data
     procedure, private :: uservar => cubeset_panel_prog_uservar
  end type panel_prog_t
  !
  type(panel_prog_t) :: buffer ! Global buffer storing the current state
  !
contains
  !
  subroutine cubeset_panel_request_default(request,error)
    !-------------------------------------------------------------------
    ! Use the type defaults by setting request to intent(out)!
    ! In addition, get the page geometry from the plotting library.
    !-------------------------------------------------------------------
    class(panel_request_t), intent(out)   :: request
    logical,                intent(inout) :: error
    !
    character(len=*), parameter :: rname='PANEL>REQUEST>DEFAULT'
    !
    call cubeset_message(setseve%trace,rname,'Welcome')
    !
    call request%page%get_geometry(error)
    if (error) return
  end subroutine cubeset_panel_request_default
  !
  subroutine cubeset_panel_request_user2prog(user,prog,error)
    !-------------------------------------------------------------------
    ! 
    !-------------------------------------------------------------------
    class(panel_request_t), intent(in)    :: user
    type(panel_request_t),  intent(inout) :: prog
    logical,                intent(inout) :: error
    !
    character(len=*), parameter :: rname='PANEL>REQUEST>USER2PROG'
    !
    call cubeset_message(setseve%trace,rname,'Welcome')
    !
    prog%page = user%page
    call user%frame%toprog(prog%frame,prog%page%geo,error)
    if (error) return
    call user%panel%toprog(prog%panel,error)
    if (error) return
  end subroutine cubeset_panel_request_user2prog
  !
  subroutine cubeset_panel_request_list(request,error)
    !-------------------------------------------------------------------
    ! 
    !-------------------------------------------------------------------
    class(panel_request_t), intent(in)    :: request
    logical,                intent(inout) :: error
    !
    character(len=*), parameter :: rname='PANEL>REQUEST>LIST'
    !
    call cubeset_message(setseve%trace,rname,'Welcome')
    !
    call cubeset_message(seve%r,rname,blankstr)
    call request%panel%list(error)
    if (error) return
    call request%frame%list(error)
    if (error) return
  end subroutine cubeset_panel_request_list
  !
  !----------------------------------------------------------------------
  ! Now the standard command
  !
  subroutine cubeset_panel_command(line,error)
    !-------------------------------------------------------------------
    ! Start from buffer current request state. Don't modify it in place to
    ! ensure an atomic behavior in case there would be a flow interruption
    !-------------------------------------------------------------------
    character(len=*), intent(in)    :: line
    logical,          intent(inout) :: error
    !
    type(panel_user_t) :: user
    type(panel_prog_t) :: prog
    character(len=*), parameter :: rname='PANEL>COMMAND'
    !
    call cubeset_message(setseve%trace,rname,'Welcome')
    !
    user%request = buffer%request
    call panel%parse(line,user,error)
    if (error) return
    call panel%main(user,prog,error)
    if (error) return
    buffer = prog
    !***JP: Overkill. It should be done only once.
    call buffer%uservar(error)
    if (error) return
  end subroutine cubeset_panel_command
  !
  !----------------------------------------------------------------------
  !
  subroutine cubeset_panel_comm_register(comm,error)
    !-------------------------------------------------------------------
    !
    !-------------------------------------------------------------------
    class(panel_comm_t), intent(inout) :: comm
    logical,             intent(inout) :: error
    !
    character(len=*), parameter :: rname='PANEL>COMM>REGISTER'
    !
    call cubeset_message(setseve%trace,rname,'Welcome')
    !
    ! Order matters here!
    call comm%panel%register(cubeset_panel_command,error)
    if (error) return
    call comm%frame%register(error)
    if (error) return
    call comm%default%register('DEFAULT','Reset PANEL defaults',error)
    if (error) return
    call comm%box%register('BOX','Draw a box around the current panel position',error)
    if (error) return
  end subroutine cubeset_panel_comm_register
  !
  subroutine cubeset_panel_comm_parse(comm,line,user,error)
    use cubetools_structure
    !-------------------------------------------------------------------
    !
    !-------------------------------------------------------------------
    class(panel_comm_t), intent(inout) :: comm
    character(len=*),    intent(in)    :: line
    type(panel_user_t),  intent(inout) :: user
    logical,             intent(inout) :: error
    !
    character(len=*), parameter :: rname='PANEL>COMM>PARSE'
    !
    call cubeset_message(setseve%trace,rname,'Welcome')
    !
    ! List when only the command is available on the command line
    user%list%present = (cubetools_nopt().eq.0).and.(comm%panel%comm%getnarg().eq.0)
    ! Other keys
    call comm%default%parse(line,user%default,error)
    if (error) return
    if (user%default%present) then
       ! Start by resetting the defaults
       call user%request%default(error)
       if (error) return
!       print *,"defaults"
    else
       ! Nothing to do
!       print *,"no defaults"
    endif
    ! Get global behavior before parsing the user request
    call user%request%page%get_geometry(error)
    if (error) return
    call comm%panel%parse(line,user%request%panel,error)
    if (error) return
    call comm%frame%parse(line,user%request%frame,error)
    if (error) return
    call comm%box%parse(line,user%box,error)
    if (error) return
  end subroutine cubeset_panel_comm_parse
  !
  subroutine cubeset_panel_comm_main(comm,user,prog,error)
    !-------------------------------------------------------------------
    !
    !-------------------------------------------------------------------
    class(panel_comm_t), intent(in)    :: comm
    type(panel_user_t),  intent(inout) :: user
    type(panel_prog_t),  intent(inout) :: prog
    logical,             intent(inout) :: error
    !
    character(len=*), parameter :: rname='PANEL>COMM>MAIN'
    !
    call cubeset_message(setseve%trace,rname,'Welcome')
    !
    call user%toprog(comm,prog,error)
    if (error) return
    if (prog%list%act) then
       call prog%request%list(error)
       if (error) return
    else
       call prog%header(error)
       if (error) return
       call prog%data(error)
       if (error) return
    endif
  end subroutine cubeset_panel_comm_main
  !
  !----------------------------------------------------------------------
  !
  subroutine cubeset_panel_user_toprog(user,comm,prog,error)
    !-------------------------------------------------------------------
    !
    !-------------------------------------------------------------------
    class(panel_user_t), intent(in)    :: user
    class(panel_comm_t), intent(in)    :: comm
    type(panel_prog_t),  intent(inout) :: prog
    logical,             intent(inout) :: error
    !
    type(key_comm_t) :: commlist
    character(len=*), parameter :: rname='PANEL>USER>TOPROG'
    !
    call cubeset_message(setseve%trace,rname,'Welcome')
    !
    call user%list%toprog(commlist,prog%list,error)
    if (error) return
    call user%default%toprog(comm%default,prog%default,error)
    if (error) return
    call user%request%toprog(prog%request,error)
    if (error) return
    call user%box%toprog(comm%box,prog%box,error)
    if (error) return
!!$    print *,"user"
!!$    call user%request%list(error)
!!$    print *,"prog"
!!$    call prog%request%list(error)
  end subroutine cubeset_panel_user_toprog
  !
  !----------------------------------------------------------------------
  !
  subroutine cubeset_panel_prog_header(prog,error)
    !-------------------------------------------------------------------
    ! 
    !-------------------------------------------------------------------
    class(panel_prog_t), intent(inout) :: prog
    logical,             intent(inout) :: error
    !
    real(kind=pane_k) :: fullsize(2),fullaspect,corrfactor
    real(kind=pane_k) :: mynxy(2)
    character(len=*), parameter :: rname='PANEL>PROG>HEADER'
    !
    call cubeset_message(setseve%trace,rname,'Welcome')
    !
    prog%frame%pos = reshape(prog%request%frame%pos,[2,2])
    prog%frame%size = prog%frame%pos(2,:)-prog%frame%pos(1,:)
    ! Take care of prog%request%inter
    if (any(prog%request%panel%inter.ne.0.0)) then
       ! White inter-spaces are asked
       ! => Just need to redefine the number of panels
       mynxy = prog%request%panel%nxy*(1.0+prog%request%panel%inter)-prog%request%panel%inter
    else
       ! Just use user prog%request
       mynxy = prog%request%panel%nxy
    endif
    ! Take care of aspect ratio
    if (prog%request%panel%aspect.ne.0) then
       ! The margin defining the plotting region are defined through the
       ! prog%frame%pos variable. But when we want to respect the panel aspect
       ! ratio, we have to enlarge either the x or y one depending on this
       ! ratio.
       if (prog%request%panel%aspect.le.0.0) then
          call cubeset_message(seve%e,rname,'Negative prog%request%panel%aspect ratio!')
          error = .true.
          return
       endif
       ! Compute the panel size when using all the available plotting area
       fullsize = prog%frame%size/mynxy
       ! Compute the corresponding aspect ratio
       fullaspect = fullsize(code_x)/fullsize(code_y)
       ! Compute the needed correction factor
       corrfactor = prog%request%panel%aspect*mynxy(code_x)/mynxy(code_y)
       if (fullaspect.gt.prog%request%panel%aspect) then
          ! Increase the x margin
          prog%frame%mar(code_x) = 0.5*(prog%frame%size(code_x)-prog%frame%size(code_y)*corrfactor)
          prog%frame%mar(code_y) = 0.0
       elseif (fullaspect.lt.prog%request%panel%aspect) then
          ! Increase the y margin
          prog%frame%mar(code_x) = 0.0
          prog%frame%mar(code_y) = 0.5*(prog%frame%size(code_y)-prog%frame%size(code_x)/corrfactor)
       endif
    else
       prog%frame%mar = 0.0
    endif
    ! Computes the panel size in fraction of smallest page size
    prog%current%size = ((prog%frame%size-2.0*prog%frame%mar)/mynxy)
    prog%current%aspect = prog%current%size(1)/prog%current%size(2)
    ! Fix one of the four corners when asked
    call prog%request%frame%fix_corner(prog%frame,error)
    if (error) return
  end subroutine cubeset_panel_prog_header
  !
  subroutine cubeset_panel_prog_data(prog,error)
    !-------------------------------------------------------------------
    ! 
    !-------------------------------------------------------------------
    class(panel_prog_t), intent(inout) :: prog
    logical,             intent(inout) :: error
    !
    character(len=*), parameter :: rname='PANEL>PROG>DATA'
    !
    call cubeset_message(setseve%trace,rname,'Welcome')
    !
    call prog%current%compute_position(&
         prog%request%panel%ixy,&
         prog%request%panel%inter,&
         prog%frame%blc,&
         error)
    if (error) return
    call prog%current%resize(prog%request%panel%size,error)
    if (error) return
    call prog%current%toviewport(prog%request%page%aspect,error)
    if (error) return
    if (prog%box%act) then
       call plot%box(error)
       if (error) return
    endif
  end subroutine cubeset_panel_prog_data
  !
  subroutine cubeset_panel_prog_uservar(prog,error)
    use cubetools_progstruct_types
    use cubeset_buffer
    !-------------------------------------------------------------------
    ! Create the CUBESET%PANEL readonly uservar
    !-------------------------------------------------------------------
    class(panel_prog_t), intent(in)    :: prog
    logical,             intent(inout) :: error
    !
    type(progstruct_t) :: panel,frame
    character(len=*), parameter :: rname='PANEL>PROG>USERVAR'
    !
    call cubeset_message(setseve%trace,rname,'Welcome')
    !
    ! panel%
    call cubeset%recreate('panel',panel,error)
    if (error) return
    call prog%current%uservar(panel,error)
    if (error) return
    call prog%request%panel%uservar(panel,error)
    if (error) return
    ! panel%frame
    call panel%recreate('frame',frame,error)
    if (error) return
    call prog%frame%uservar(frame,error)
    if (error) return
  end subroutine cubeset_panel_prog_uservar
end module cubeset_panel
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
