!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!
module cubemain_stack
  use cubetools_structure
  use cubetools_keyword_arg
  use cube_types
  use cubeadm_cubeid_types
  use cubemain_range
  use cubemain_windowing
  use cubemain_messaging
  use cubemain_auxiliary
  !
  public :: stack
  public :: cubemain_stack_command
  private
  !
  type :: stack_comm_t
     type(option_t), pointer :: comm     
     type(range_opt_t)       :: range      
     type(option_t), pointer :: mask       
     type(option_t), pointer :: sum        
     type(option_t), pointer :: mean       
     type(option_t), pointer :: weight     
     type(option_t), pointer :: image      
     type(option_t), pointer :: spectrum
     type(keyword_arg_t), pointer :: wei_arg
   contains
     procedure, public  :: register     => cubemain_stack_register
     procedure, private :: parse        => cubemain_stack_parse
     procedure, private :: parse_weight => cubemain_stack_parse_weight
     procedure, private :: main         => cubemain_stack_main
  end type stack_comm_t
  type(stack_comm_t) :: stack
  !
  integer(kind=4), parameter :: icube = 1
  integer(kind=4), parameter :: inois = 2
  type stack_user_t
     type(cubeid_user_t)    :: cubeids           ! Input Cube
     logical                :: dospe   = .true.  ! Do spectral stack
     type(auxiliary_user_t) :: mask              ! Mask
     type(range_array_t)    :: range             ! range
     character(len=argu_l)  :: wei               ! Weight scheme
     logical                :: domean  = .false. ! Output is a mean spectrum
     logical                :: dosum   = .false. ! Output is a sum spectrum
     logical                :: dowei = .false.   ! Use weighting
   contains
     procedure, private :: toprog => cubemain_stack_user_toprog
  end type stack_user_t
  type stack_prog_t
     type(window_array_t) :: wind               ! Window to be spectrally stacked
     type(cube_t),pointer :: cube               ! Input cube
     type(cube_t),pointer :: stack              ! Output spectrum
     type(cube_t),pointer :: mask               ! Mask
     type(cube_t),pointer :: noise              ! Noise reference
     logical              :: domean = .false.   ! Output is a mean spectrum
     logical              :: domask             ! Use a mask
     logical              :: mask2d = .false.   ! Is the mask 2d?
     logical              :: donoise            ! Use weighting by noise  
     logical              :: dospe = .true.     ! Do spectral stack
  end type stack_prog_t
  !
contains
  !
  subroutine cubemain_stack_command(line,error)
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    character(len=*), intent(in)    :: line
    logical,          intent(inout) :: error
    !
    type(stack_user_t) :: user
    character(len=*), parameter :: rname='STACK>COMMAND'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call stack%parse(line,user,error)
    if (error) return
    call stack%main(user,error)
    if (error) continue
  end subroutine cubemain_stack_command
  !
  !----------------------------------------------------------------------
  !
  subroutine cubemain_stack_register(stack,error)
    use cubedag_allflags
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(stack_comm_t), intent(inout) :: stack
    logical,             intent(inout) :: error
    !
    type(cubeid_arg_t) :: cubearg
    type(keyword_arg_t) :: keyarg
    character(len=*), parameter :: comm_abstract = &
         'Stack a cube spectrally or spatially'
    character(len=*), parameter :: comm_help = &
         'Output unit is determined by the type of stacking being&
         & done. By default a spectral stack is computed'
    !
    integer(kind=4), parameter :: nweights=2                      
    character(len=*), parameter :: weights(nweights) = ['Equal','Noise']
    character(len=*), parameter :: rname='STACK>REGISTER'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubetools_register_command(&
         'STACK','[cube [noise]]',&
         comm_abstract,&
         comm_help,&
         cubemain_stack_command,&
         stack%comm,error)
    if (error) return
    call cubearg%register( &
         'CUBE', &
         'Signal cube',  &
         strg_id,&
         code_arg_optional,  &
         [flag_cube], &
         error)
    if (error) return
    call cubearg%register( &
         'NOISE', &
         'Noise reference',  &
         strg_id,&
         code_arg_optional,  &
         [flag_noise], &
         error)
    if (error) return
    !
    call stack%range%register(&
         'RANGE',&
         'Define the velocity range(s) over which to stack',&
         range_is_multiple,error)
    if (error) return
    !
    call cubemain_auxiliary_register(&
         'MASK',&
         'Use a mask to define pixels or channels to be ignored',&
         strg_id,&
         'Mask',&
         [flag_mask],&
         code_arg_optional, &
         stack%mask,error)
    if (error) return
    !
    call cubetools_register_option(&
         'SUM','',&
         'Stack is a sum',&
         'Output unit is Jy for a spectral stack and <CubUnit>.km/s for a&
         & spatial stack',&
         stack%sum,error)
    if (error) return
    !
    call cubetools_register_option(&
         'MEAN','',&
         'Stack is a mean',&
         'Output unit is K(Tmb) for a spectral stack and <CubUnit>&
         & for a spatial stack',&
         stack%mean,error)
    if (error) return
    !
    call cubetools_register_option(&
         'WEIGHT','scheme',&
         'Define the weighting scheme for the stacking',&
         'Only valid for /SPECTRUM',&
         stack%weight,error)
    if (error) return
    call keyarg%register( &
         'scheme',  &
         'Weighting scheme', &
         strg_id,&
         code_arg_mandatory, &
         weights, &
         .not.flexible, &
         stack%wei_arg, &
         error)
    if (error) return
    !
    call cubetools_register_option(&
         'IMAGE','',&
         'Stack is an image',&
         'Default behaviour is /SUM',&
         stack%image,error)
    if (error) return
    !
    call cubetools_register_option(&
         'SPECTRUM','',&
         'Stack is a spectrum',&
         'Default behaviour is /SUM for Jy/beam spectra and /MEAN for&
         & K(Tmb) spectra',&
         stack%spectrum,error)
    if (error) return
  end subroutine cubemain_stack_register
  !
  subroutine cubemain_stack_parse(stack,line,user,error)
    use gkernel_interfaces
    !----------------------------------------------------------------------
    ! STACK 
    !----------------------------------------------------------------------
    class(stack_comm_t), intent(in)    :: stack
    character(len=*),    intent(in)    :: line
    type(stack_user_t),  intent(out)   :: user
    logical,             intent(inout) :: error
    !
    logical :: dorange,domean,dosum,dospectrum,doimage
    character(len=*), parameter :: rname='STACK>PARSE'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubeadm_cubeid_parse(line,stack%comm,user%cubeids,error)
    if (error) return
    call stack%range%parse(line,dorange,user%range,error)
    if (error) return
    call cubemain_auxiliary_parse(line,stack%mask,user%mask,error)
    if (error) return
    !
    call stack%mean%present(line,domean,error)
    if (error) return
    call stack%sum%present(line,dosum,error)
    if (error) return
    call stack%spectrum%present(line,dospectrum,error)
    if (error) return
    call stack%image%present(line,doimage,error)
    if (error) return
    !
    if (domean.and.dosum)then
       call cubemain_message(seve%e,rname,'Options /MEAN and /SUM are incompatible')
       error = .true.
       return
    else if (dosum) then
       user%domean = .false.
       user%dosum  = .true.
    else if (domean) then
       user%domean = .true.
       user%dosum  = .false.
    else
       user%domean = .false.
       user%dosum  = .false.
    endif
    if (dospectrum.and.doimage) then
       call cubemain_message(seve%e,rname,'Options /IMAGE and /SPECTRUM are incompatible')
       error = .true.
       return
    elseif (dospectrum) then
       user%dospe = .true.
    elseif (doimage) then
       user%dospe = .false.
    else
       user%dospe = .true.
    endif
    !
    call stack%parse_weight(line,user,error)
    if(error) return
  end subroutine cubemain_stack_parse
  !
  subroutine cubemain_stack_parse_weight(stack,line,user,error)
    use cubetools_disambiguate
    !----------------------------------------------------------------------
    ! STACK cubname
    ! /WEIGHT equal|noise
    !----------------------------------------------------------------------
    class(stack_comm_t), intent(in)    :: stack
    character(len=*),    intent(in)    :: line
    type(stack_user_t),  intent(inout) :: user
    logical,             intent(inout) :: error
    !
    character(len=*), parameter :: rname='STACK>PARSE>WEIGHT'
    !
    call stack%weight%present(line,user%dowei,error)
    if (error) return
    if (user%dowei) then
       if (.not.user%dospe) then
          call cubemain_message(seve%w,rname,'Option /WEIGHT not relevant when stacking images')
          return
       endif
       call cubetools_getarg(line,stack%weight,1,user%wei,mandatory,error)
       if (error)  return
    endif
  end subroutine cubemain_stack_parse_weight
  !
  subroutine cubemain_stack_main(stack,user,error)
    use cubemain_stack_spatial
    use cubemain_stack_spectral
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(stack_comm_t), intent(in)    :: stack
    type(stack_user_t),  intent(in)    :: user
    logical,             intent(inout) :: error
    !
    type(stack_prog_t) :: prog
    character(len=*), parameter :: rname='STACK>MAIN'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call user%toprog(prog,error)
    if (error) return
    if (prog%dospe) then
       call cubemain_stack_spectral_noaperture(prog%domean,prog%wind,prog%cube,&
            prog%domask,prog%mask,prog%donoise,prog%noise,prog%stack,error)
       if (error) return
    else
       call cubemain_stack_spatial_do(prog%domean,prog%wind,prog%cube,&
            prog%domask,prog%mask2d,prog%mask,prog%stack,error)
       if (error) return
    endif
  end subroutine cubemain_stack_main
  !
  !----------------------------------------------------------------------
  !
  subroutine cubemain_stack_user_toprog(user,prog,error)
    use cubeadm_consistency
    use cubeadm_get
    use cubemain_stack_spectral
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(stack_user_t), intent(in)    :: user
    type(stack_prog_t),  intent(inout) :: prog
    logical,             intent(inout) :: error
    !
    logical :: prob
    character(len=argu_l) :: key
    integer(kind=code_k) :: ikey,access
    type(consistency_t) :: cons
    character(len=*), parameter :: rname='STACK>USER>TOPROG'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    prob = .false.
    !
    prog%dospe = user%dospe
    if (prog%dospe) then
       access = code_access_imaset
    else
       access = code_access_speset
    end if
    !
    call cubeadm_cubeid_get_header(stack%comm,icube,user%cubeids,access,code_read,prog%cube,error)
    if (error) return
    call stack%range%user2prog(prog%cube,user%range,prog%wind,error)
    if (error) return
    !
    if (.not.user%domean.and..not.user%dosum) then
       if (prog%dospe) then
          call cubemain_stack_spectral_domean(prog%cube,prog%domean,error)
          if (error) return
       else
          prog%domean = .false.
       endif
    else if(user%domean.and.user%dosum) then
       call cubemain_message(seve%e,rname,'User%domean and user%dosum are both true')
       error = .true.
       return
    else
       prog%domean = user%domean
    endif
    !
    prog%domask = user%mask%do
    if (prog%domask) then
       call cubemain_auxiliary_user2prog(stack%mask,access,user%mask,prog%mask,error)
       if (error) return
       if (prog%mask%head%arr%n%c.eq.1) then
          prog%mask2d = .true.
          call cons%spatial('Input cube',prog%cube,'Mask',prog%mask,error)
          if (error) return
       else
          prog%mask2d = .false.
          call cons%grid('Input cube',prog%cube,'Mask',prog%mask,error)
          if (error) return
       endif
    endif
    !
    if (user%dowei) then
       call cubetools_keyword_user2prog(stack%wei_arg,user%wei,ikey,key,error)
       if (error) return
       prog%donoise = key.eq.'NOISE'
    else
       prog%donoise = .false.
    endif
    if (prog%donoise.and.prog%dospe) then
       call cubeadm_cubeid_get_header(stack%comm,inois,user%cubeids,access,code_read,prog%noise,error)
       if (error) return
       if (prog%noise%head%arr%n%c.ne.1) then
          call cubemain_message(seve%e,rname,'Multiple noise definitions not supported')
          error = .true.
          return
       endif 
       call cons%signal_noise('Input cube',prog%cube,'Noise',prog%noise,error)
       if (error) return
    endif
    call cons%proceed(error)
    if (error) return
    !
    prog%dospe = user%dospe
  end subroutine cubemain_stack_user_toprog
end module cubemain_stack
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
