!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!
module cubemain_compare
  use cube_types
  use cubemain_messaging
  use cubetools_structure
  use cubeadm_cubeid_types
  !
  public :: compare
  public :: cubemain_compare_command
  private
  !
  type :: compare_comm_t
     type(option_t), pointer :: comm
     type(option_t), pointer :: significant
   contains
     procedure, public  :: register          => cubemain_compare_register
     procedure, private :: parse             => cubemain_compare_parse
     procedure, private :: parse_significant => cubemain_compare_parse_significant
     procedure, private :: main              => cubemain_compare_main
  end type compare_comm_t
  type(compare_comm_t) :: compare
  !
  integer(kind=4), parameter :: ione = 1 
  integer(kind=4), parameter :: itwo = 2
  type compare_user_t
     real(kind=sign_k)   :: significant ! Significant signal level for the residuals
     type(cubeid_user_t) :: cubeids
   contains
     procedure, private :: toprog => cubemain_compare_user_toprog
  end type compare_user_t
  type compare_prog_t
     real(kind=sign_k)     :: significant ! Significant signal level for the residuals
     type(cube_t), pointer :: one
     type(cube_t), pointer :: two
     type(cube_t), pointer :: residuals
     type(cube_t), pointer :: fidelity
   contains
     procedure, private :: header     => cubemain_compare_prog_header
     procedure, private :: data       => cubemain_compare_prog_data
     procedure, private :: image_loop => cubemain_compare_prog_image_loop
     procedure, private :: doimage    => cubemain_compare_prog_doimage
  end type compare_prog_t
  !
contains
  !
  subroutine cubemain_compare_command(line,error)
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    character(len=*), intent(in)    :: line
    logical,          intent(inout) :: error
    !
    type(compare_user_t) :: user
    character(len=*), parameter :: rname='COMPARE>COMMAND'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call compare%parse(line,user,error)
    if (error) return
    call compare%main(user,error)
    if (error) continue
  end subroutine cubemain_compare_command
  !
  !----------------------------------------------------------------------
  !
  subroutine cubemain_compare_register(compare,error)
    use cubedag_allflags
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(compare_comm_t), intent(inout) :: compare
    logical,               intent(inout) :: error
    !
    type(cubeid_arg_t) :: cubearg
    type(standard_arg_t) :: stdarg
    character(len=*), parameter :: comm_abstract = &
         'Compute cube1-cube2 and cube1/(cube1-cube2)'
    character(len=*), parameter :: comm_help = &
         strg_id
    character(len=*), parameter :: rname='COMPARE>REGISTER'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    ! Command
    call cubetools_register_command(&
         'COMPARE','cube1 cube2',&
         comm_abstract,&
         comm_help,&
         cubemain_compare_command,&
         compare%comm,error)
    call cubearg%register(&
         'CUBE1',&
         'Input cube #1',  &
         strg_id,&
         code_arg_mandatory,  &
         [flag_cube],&
         error)
    call cubearg%register(&
         'CUBE2',&
         'Input cube #2',&
         strg_id,&
         code_arg_mandatory,&
         [flag_cube],&
         error)
    ! Option #1
    call cubetools_register_option(&
         'SIGNIFICANT','level',&
         'Define a level of significance for the residuals',&
         strg_id,&
         compare%significant,error)
    call stdarg%register(&
         'LEVEL',&
         'Significant signal value (typical 3 to 5 sigma)',&
         strg_id,&
         code_arg_mandatory,&
         error)
  end subroutine cubemain_compare_register
  !
  subroutine cubemain_compare_parse(compare,line,user,error)
    !----------------------------------------------------------------------
    ! COMPARE id1 id2 /SIGNIFICANT rms
    !----------------------------------------------------------------------
    class(compare_comm_t), intent(in)    :: compare 
    character(len=*),      intent(in)    :: line
    type(compare_user_t),  intent(out)   :: user
    logical,               intent(inout) :: error
    !
    character(len=*), parameter :: rname='COMPARE>PARSE'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubeadm_cubeid_parse(line,compare%comm,user%cubeids,error)
    if (error) return
    call compare%parse_significant(line,user,error)
    if (error) return
  end subroutine cubemain_compare_parse
  !
  subroutine cubemain_compare_parse_significant(compare,line,user,error)
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(compare_comm_t), intent(in)    :: compare 
    character(len=*),      intent(in)    :: line
    type(compare_user_t),  intent(inout) :: user
    logical,               intent(inout) :: error
    !
    logical :: present
    character(len=*), parameter :: rname='COMPARE>PARSE>SIGNIFICANT'
    !
    call compare%significant%present(line,present,error)
    if (error) return
    if (.not.present) then
       user%significant = tiny(0.0)
    else
       call cubetools_getarg(line,compare%significant,1,user%significant,mandatory,error)
       if (error) return
       !
       if (user%significant.lt.0) then
          call cubemain_message(seve%e,rname,'Significant must be positive')
          error = .true.
          return
       endif
    endif
  end subroutine cubemain_compare_parse_significant
  !
  subroutine cubemain_compare_main(compare,user,error)
    use cubeadm_timing
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(compare_comm_t), intent(in)    :: compare
    type(compare_user_t),  intent(inout) :: user
    logical,               intent(inout) :: error
    !
    type(compare_prog_t) :: prog
    character(len=*), parameter :: rname='COMPARE>MAIN'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call user%toprog(prog,error)
    if (error) return
    call prog%header(error)
    if (error) return
    call cubeadm_timing_prepro2process()
    call prog%data(error)
    if (error) return
    call cubeadm_timing_process2postpro()
  end subroutine cubemain_compare_main
  !
  !----------------------------------------------------------------------
  !
  subroutine cubemain_compare_user_toprog(user,prog,error)
    use cubeadm_get
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(compare_user_t), intent(in)    :: user
    type(compare_prog_t),  intent(out)   :: prog
    logical,               intent(inout) :: error
    !
    character(len=*), parameter :: rname='COMPARE>USER>TOPROG'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    prog%significant = user%significant
    call cubeadm_cubeid_get_header(compare%comm,ione,user%cubeids,code_access_imaset,code_read,prog%one,error)
    if (error) return
    call cubeadm_cubeid_get_header(compare%comm,itwo,user%cubeids,code_access_imaset,code_read,prog%two,error)
    if (error) return
  end subroutine cubemain_compare_user_toprog
  !
  !----------------------------------------------------------------------
  !
  subroutine cubemain_compare_prog_header(prog,error)
    use cubetools_consistency_methods
    use cubetools_header_methods
    use cubedag_allflags
    use cubeadm_clone
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(compare_prog_t), intent(inout) :: prog
    logical,               intent(inout) :: error
    !
    logical :: prob
    character(len=*), parameter :: rname='COMPARE>PROG>HEADER'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    prob = .false.
    call cubetools_consistency_grid('Input cube #1',prog%one%head,'Input cube #2',prog%two%head,prob,error)
    if(error) return
    if (cubetools_consistency_failed(rname,prob,error)) return
    call cubeadm_clone_header(prog%one,[flag_compare,flag_residuals],prog%residuals,error)
    if (error) return
    call cubeadm_clone_header(prog%one,[flag_compare,flag_fidelity],prog%fidelity,error)
    if (error) return
    call cubetools_header_put_array_unit('---', prog%fidelity%head,error)
    if (error) return
  end subroutine cubemain_compare_prog_header
  !
  subroutine cubemain_compare_prog_data(prog,error)
    use cubeadm_opened
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(compare_prog_t), intent(inout) :: prog
    logical,               intent(inout) :: error
    !
    type(cubeadm_iterator_t) :: iter
    character(len=*), parameter :: rname='COMPARE>PROG>DATA'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubeadm_datainit_all(iter,error)
    if (error) return
    !$OMP PARALLEL DEFAULT(none) SHARED(prog,error) FIRSTPRIVATE(iter)
    !$OMP SINGLE
    do while (cubeadm_dataiterate_all(iter,error))
       if (error) exit
       !$OMP TASK SHARED(prog) FIRSTPRIVATE(iter,error)
       if (.not.error) &
         call prog%image_loop(iter%first,iter%last,error)
       !$OMP END TASK
    enddo ! ie
    !$OMP END SINGLE
    !$OMP END PARALLEL
  end subroutine cubemain_compare_prog_data
  !
  subroutine cubemain_compare_prog_image_loop(prog,first,last,error)
    use cubeadm_entryloop
    use cubemain_image_real
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(compare_prog_t), intent(inout) :: prog
    integer(kind=entr_k),  intent(in)    :: first
    integer(kind=entr_k),  intent(in)    :: last
    logical,               intent(inout) :: error
    !
    integer(kind=entr_k) :: ie
    type(image_t) :: one,two,residuals,fidelity
    character(len=*), parameter :: rname='COMPARE>IMAGE>LOOP'
    !
    call one%init(prog%one,error)
    if (error) return
    call two%init(prog%one,error)
    if (error) return
    call residuals%reallocate('residuals',prog%one%head%arr%n%l,prog%one%head%arr%n%m,error)
    if (error) return
    call fidelity%reallocate('fidelity',prog%one%head%arr%n%l,prog%one%head%arr%n%m,error)
    if (error) return
    !
    do ie=first,last
      call cubeadm_entryloop_iterate(ie,error)
      if (error) return
      call prog%doimage(ie,one,two,residuals,fidelity,error)
      if (error) return
    enddo
  end subroutine cubemain_compare_prog_image_loop
  !
  subroutine cubemain_compare_prog_doimage(prog,ie,one,two,residuals,fidelity,error)
    use cubemain_image_real
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(compare_prog_t), intent(inout) :: prog
    integer(kind=entr_k),  intent(in)    :: ie
    type(image_t),         intent(inout) :: one
    type(image_t),         intent(inout) :: two
    type(image_t),         intent(inout) :: residuals
    type(image_t),         intent(inout) :: fidelity
    logical,               intent(inout) :: error
    !
    integer(kind=pixe_k) :: il,im
    character(len=*), parameter :: rname='COMPARE>PROG>DOIMAGE'
    !
    call one%get(prog%one,ie,error)
    if (error) return
    call two%get(prog%two,ie,error)
    if (error) return
    do im=1,prog%one%head%arr%n%m
       do il=1,prog%one%head%arr%n%l
          residuals%z(il,im) = one%z(il,im)-two%z(il,im)
          fidelity%z(il,im) = abs(one%z(il,im))/max(abs(residuals%z(il,im)),prog%significant)
       enddo ! il
    enddo ! im
    call residuals%put(prog%residuals,ie,error)
    if (error) return
    call fidelity%put(prog%fidelity,ie,error)
    if (error) return
  end subroutine cubemain_compare_prog_doimage
end module cubemain_compare
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
